/*
Copyright (c) 2000-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	b2bo.c	Bmeps options functions.
*/




/** Inside the b2bo.c module. */
#define B2BO_C	1
#include "bmepsi.h"




#line 52 "b2bo.ctr"




/**	Name of environment variable containg default configuration.
*/
static char str_epsoutput[] = { "EPSOUTPUT" };

/**	Output types.
*/
static char *str_output_types[] = { "ps", "pdf", "bb", NULL };

/**	PDF levels.
*/
static char *str_pdf_levels[] = { "1.2", "1.3", "1.4", NULL };

/**	Predictor names.
*/
static char *str_predictor_names[] = {
  "tiff",		/*  0 */
  "source",		/*  1 */
  "png-sub",		/*  2 */
  "png-up",		/*  3 */
  "png-average",	/*  4 */
  "png-paeth",		/*  5 */
  NULL
};

/**	Whitespaces.
*/
static char commasep[] = { ", \t\r\n" };

/**	Whitespaces and the colon.
*/
static char colonsep[] = { ": \t\r\n" };

/**	Whitespaces and the dot.
*/
static char dotsep[] = { ". \t\r\n" };

/**	Name of configuration file.
*/
static char cfgfilename[] = { "bmeps.conf" };

/**	Abbreviations for encodings.
*/
static char *str_encodings[] = {
  "a$scii85",
  "r$un-length",
  "l$zw",		/* reserved for future use */
  "f$late",
  "d$ct",
  NULL
};

/**	PDF page fitting methods.
*/
static char *str_fit_methods[] = {
  "w$idth", "h$eight", "p$age",
  NULL
};

/**	Known media sizes.
*/
static MSENTRY media_sizes[] = {
  { "A3",        {0.0, 0.0, 842.0, 1190.0, 0.0, 0.0, 842.0, 1190.0 } },
  { "A4",        {0.0, 0.0, 595.0, 842.0, 0.0, 0.0, 595.0, 842.0 } },
  { "A5",        {0.0, 0.0, 420.0, 595.0, 0.0, 0.0, 420.0, 595.0 } },
  { "B4",        {0.0, 0.0, 729.0, 1032.0, 0.0, 0.0, 729.0, 1032.0 } },
  { "B5",        {0.0, 0.0, 516.0, 729.0, 0.0, 0.0, 516.0, 729.0 } },
  { "Letter",    {0.0, 0.0, 612.0, 792.0, 0.0, 0.0, 612.0, 792.0 } },
  { "Legal",     {0.0, 0.0, 612.0, 1008.0, 0.0, 0.0, 612.0, 1008.0 } },
  { "Tabloid",   {0.0, 0.0, 792.0, 1224.0, 0.0, 0.0, 792.0, 1224.0 } },
  { "Ledger",    {0.0, 0.0, 1224.0, 792.0, 0.0, 0.0, 1224.0, 792.0 } },
  { "Statement", {0.0, 0.0, 396.0, 612.0, 0.0, 0.0, 396.0, 612.0 } },
  { "Executive", {0.0, 0.0, 540.0, 720.0, 0.0, 0.0, 540.0, 720.0 } },
  { "Folio",     {0.0, 0.0, 612.0, 936.0, 0.0, 0.0, 612.0, 936.0 } },
  { "Quarto",    {0.0, 0.0, 610.0, 780.0, 0.0, 0.0, 610.0, 780.0 } },
  { "10x14",     {0.0, 0.0, 720.0, 1008.0, 0.0, 0.0, 720.0, 1008.0 } },
  { NULL,        {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 } }
};

/**	Preference key to search for language.
*/
static char *pk[] = {
  "/language",
  NULL
};

/** Configuration command: output type. */
static char *cmd00[] = { "o$utput", "t$ype", NULL };

/** Configuration command: level. */
static char *cmd01[] = { "l$evel", NULL };

/** Configuration command: encoding. */
static char *cmd02[] = { "e$ncoding", NULL };

/** Configuration command: background. */
static char *cmd03[] = { "b$ackground", NULL };

/** Configuration command: image mask trigger level. */
static char *cmd04[] = { "i$mage", "m$ask", "t$rigger", "l$evel", NULL };

/** Configuration command: media size. */
static char *cmd05[] = { "m$edia", "s$ize", NULL };

/** Configuration command: separated dictionary. */
static char *cmd06[] = { "s$eparated", "d$ictionary", NULL };

/** Configuration command: operator dictionary. */
static char *cmd07[] = { "o$perator", "d$ictionary", NULL };

/** Configuration command: multiple data sources. */
static char *cmd08[] = { "m$ultiple", "d$ata", "s$ources", NULL };

/** Configuration command: always use default background. */
static char *cmd09[] = { "a$lways", "u$se", "d$efault", "b$ackground", NULL };

/** Configuration command: mix. */
static char *cmd10[] = { "m$ix", NULL };

/** Configuration command: create image mask. */
static char *cmd11[] = { "c$reate", "i$mage", "m$ask", NULL };

/** Configuration command: dsc comments. */
static char *cmd12[] = { "dsc", "c$omments", NULL };

/** Configuration command: showpage. */
static char *cmd13[] = { "sh$owpage", NULL };

/** Configuration command: vmreclaim. */
static char *cmd14[] = { "v$mreclaim", NULL };

/** Configuration command: use resolution chunk. */
static char *cmd15[] = { "u$se", "r$esolution", "c$hunk", NULL };

/** Configuration command: transfer alpha channel. */
static char *cmd16[] = { "t$ransfer", "a$lpha", "c$hannel", NULL };

/** Configuration command: paper size. */
static char *cmd17[] = { "p$aper", "s$ize", NULL };

/** Configuration command: draft. */
static char *cmd18[] = { "d$raft", NULL };

/** Configuration command: interpolate. */
static char *cmd19[] = { "i$nterpolate", NULL };

/** Configuration command: color. */
static char *cmd20[] = { "c$olor", NULL };

/** Configuration command: jfif sof set. */
static char *cmd21[] = { "j$fif", "s$of", "s$et", NULL };

/** Configuration command: jpeg interpolate. */
static char *cmd22[] = { "j$peg", "i$nterpolate", NULL };

/** Configuration command: pdf fit. */
static char *cmd23[] = { "p$df", "f$it", NULL };

/** Configuration command: allow pdf page attributes. */
static char *cmd24[] = { "a$llow", "p$df", "p$age", "a$ttributes", NULL };

/** Configuration command: force dct passthrough. */
static char *cmd25[] = { "f$orce", "d$ct", "p$assthrough", NULL };

/** Predictor for flate compression. */
static char *cmd26[] = { "predictor", NULL };

/** All configuration commands. */
static char **cmd[] = {
  cmd00, cmd01, cmd02, cmd03, cmd04, cmd05, cmd06, cmd07, cmd08, cmd09,
  cmd10, cmd11, cmd12, cmd13, cmd14, cmd15, cmd16, cmd17, cmd18, cmd19,
  cmd20, cmd21, cmd22, cmd23, cmd24, cmd25, cmd26,
  NULL
};

/** Command strings for writing configuration file. */
static char *cmd_strings[] = {
	/*  0 */	"output type",
	/*  1 */	"level",
	/*  2 */	"encoding",
	/*  3 */	"background",
	/*  4 */	"image mask trigger level",
	/*  5 */	"media size",
	/*  6 */	"separated dictionary",
	/*  7 */	"operator dictionary",
	/*  8 */	"multiple data sources",
	/*  9 */	"always use default background",
	/* 10 */	"mix",
	/* 11 */	"create image mask",
	/* 12 */	"dsc comments",
	/* 13 */	"showpage",
	/* 14 */	"vmreclaim",
	/* 15 */	"use resolution chunk",
	/* 16 */	"transfer alpha channel",
	/* 17 */	"paper size",
	/* 18 */	"draft",
	/* 19 */	"interpolate",
	/* 20 */	"color",
	/* 21 */	"jfif sof set",
	/* 22 */	"jpeg interpolate",
	/* 23 */	"pdf fit",
	/* 24 */	"allow pdf page attributes",
	/* 25 */	"force dct passthrough",
	NULL
};



/**	File types which can be handled.
*/
static char file_types_string[] = {
"# @FILE-TYPES PNG"
#if DK_HAVE_JPEGLIB_H
" JPEG"
#endif
#if DK_HAVE_TIFF_H
" TIFF"
#endif
"\n"
};

/** Suffixes for the output file types. */
static char *ft_suffixes[]= {
  "pdf", "eps", "ps", "bb",
  NULL
};

/** Default options for PS 1. */
static BO bo_ps1 = {
  BMEPS_OUTPUT_TYPE_EPS,
  BMEPS_PS_LEVEL_1,
  0,
  0,
  (
    BMEPS_OPT_ALPHA_MIX
  ),
  0UL,
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
  BMEPS_PREDICTOR_NONE
};

/** Default options for bounding box. */
static BO bo_bb = {
  BMEPS_OUTPUT_TYPE_BB,
  BMEPS_PS_LEVEL_1,
  0,
  0,
  0UL,
  0UL,
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
  BMEPS_PREDICTOR_NONE
};

/** Default options for PS 2. */
static BO bo_ps2 = {
  BMEPS_OUTPUT_TYPE_EPS,
  BMEPS_PS_LEVEL_2,
  (
    BMEPS_ENCODING_ASCII85
    |BMEPS_ENCODING_RUNLENGTH
    |BMEPS_ENCODING_DCT
  ),
  (DKBIF_JFIF_SOF0 | DKBIF_JFIF_SOF1),
  (
    BMEPS_OPT_ALPHA_MIX
    |BMEPS_OPT_OPERATOR_DICTIONARY
    |BMEPS_OPT_COLOR_OUTPUT
    |BMEPS_OPT_DICTIONARY
    |BMEPS_OPT_INTERPOLATE
    |BMEPS_OPT_SEPARATED_DATA
    |BMEPS_OPT_FORCE_DCT
  ),
  0UL,
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
  BMEPS_PREDICTOR_NONE
};

/** Default options for PS 3. */
static BO bo_ps3 = {
  BMEPS_OUTPUT_TYPE_EPS,
  BMEPS_PS_LEVEL_3,
  (
    BMEPS_ENCODING_ASCII85
    |BMEPS_ENCODING_DCT
    |BMEPS_ENCODING_FLATE
  ),
  (DKBIF_JFIF_SOF0 | DKBIF_JFIF_SOF1),
  (
    BMEPS_OPT_ALPHA_MIX
    |BMEPS_OPT_OPERATOR_DICTIONARY
    |BMEPS_OPT_COLOR_OUTPUT
    |BMEPS_OPT_DICTIONARY
    |BMEPS_OPT_INTERPOLATE
    |BMEPS_OPT_FORCE_DCT
  ),
  0UL,
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
  BMEPS_PREDICTOR_NONE
};

/** Default options for EPS 1. */
static BO bo_eps1 = {
  BMEPS_OUTPUT_TYPE_EPS,
  BMEPS_PS_LEVEL_1,
  0,
  0,
  (
    BMEPS_OPT_ALPHA_MIX
  ),
  0UL,
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
  BMEPS_PREDICTOR_NONE
};


/** Default options for EPS 2. */
static BO bo_eps2 = {
  BMEPS_OUTPUT_TYPE_EPS,
  BMEPS_PS_LEVEL_2,
  (
    BMEPS_ENCODING_ASCII85
    |BMEPS_ENCODING_DCT
    |BMEPS_ENCODING_RUNLENGTH
  ),
  (DKBIF_JFIF_SOF0 | DKBIF_JFIF_SOF1),
  (
    BMEPS_OPT_ALPHA_MIX
    |BMEPS_OPT_OPERATOR_DICTIONARY
    |BMEPS_OPT_COLOR_OUTPUT
    |BMEPS_OPT_DICTIONARY
    |BMEPS_OPT_INTERPOLATE
    |BMEPS_OPT_DSC
    |BMEPS_OPT_SEPARATED_DATA
    |BMEPS_OPT_FORCE_DCT
  ),
  0UL,
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
  BMEPS_PREDICTOR_NONE
};

/** Default options for EPS 3. */
static BO bo_eps3 = {
  BMEPS_OUTPUT_TYPE_EPS,
  BMEPS_PS_LEVEL_3,
  (
    BMEPS_ENCODING_ASCII85
    |BMEPS_ENCODING_DCT
    |BMEPS_ENCODING_FLATE
  ),
  (DKBIF_JFIF_SOF0 | DKBIF_JFIF_SOF1),
  (
    BMEPS_OPT_ALPHA_MIX
    |BMEPS_OPT_OPERATOR_DICTIONARY
    |BMEPS_OPT_COLOR_OUTPUT
    |BMEPS_OPT_DICTIONARY
    |BMEPS_OPT_INTERPOLATE
    |BMEPS_OPT_DSC
    |BMEPS_OPT_FORCE_DCT
  ),
  0UL,
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
  BMEPS_PREDICTOR_NONE
};

/** Default options for PDF 1.2. */
static BO bo_pdf12 = {
  BMEPS_OUTPUT_TYPE_PDF,
  BMEPS_PDF_LEVEL_12,
  (
    BMEPS_ENCODING_ASCII85
    |BMEPS_ENCODING_DCT
    |BMEPS_ENCODING_FLATE
  ),
  (DKBIF_JFIF_SOF0 | DKBIF_JFIF_SOF1),
  (
    BMEPS_OPT_ALPHA_MIX
    |BMEPS_OPT_OPERATOR_DICTIONARY
    |BMEPS_OPT_COLOR_OUTPUT
    |BMEPS_OPT_DICTIONARY
    |BMEPS_OPT_INTERPOLATE
    |BMEPS_OPT_FORCE_DCT
  ),
  0UL,
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
  BMEPS_PREDICTOR_INPUT
};

/** Default options for PDF 1.3. */
static BO bo_pdf13 = {
  BMEPS_OUTPUT_TYPE_PDF,
  BMEPS_PDF_LEVEL_13,
  (
    BMEPS_ENCODING_ASCII85
    |BMEPS_ENCODING_DCT
    |BMEPS_ENCODING_FLATE
  ),
  (DKBIF_JFIF_SOF0 | DKBIF_JFIF_SOF1),
  (
    BMEPS_OPT_ALPHA_MIX
    |BMEPS_OPT_OPERATOR_DICTIONARY
    |BMEPS_OPT_COLOR_OUTPUT
    |BMEPS_OPT_DICTIONARY
    |BMEPS_OPT_INTERPOLATE
    |BMEPS_OPT_FORCE_DCT
  ),
  0UL,
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
  BMEPS_PREDICTOR_INPUT
};

/** Default options for PDF 1.4. */
static BO bo_pdf14 = {
  BMEPS_OUTPUT_TYPE_PDF,
  BMEPS_PDF_LEVEL_14,
  (
    BMEPS_ENCODING_ASCII85
    |BMEPS_ENCODING_DCT
    |BMEPS_ENCODING_FLATE
  ),
  (DKBIF_JFIF_SOF0 | DKBIF_JFIF_SOF1),
  (
    BMEPS_OPT_TRANSFER_ALPHA
    |BMEPS_OPT_OPERATOR_DICTIONARY
    |BMEPS_OPT_COLOR_OUTPUT
    |BMEPS_OPT_DICTIONARY
    |BMEPS_OPT_INTERPOLATE
    |BMEPS_OPT_FORCE_DCT
  ),
  BMEPS_PDFOPT_ALLOW_PDF_PAGE_ATTRIBUTES,
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 },
  BMEPS_PREDICTOR_INPUT
};

/** Default options for Fax to PS2. */
static BO bo_faxps2 = {
  BMEPS_OUTPUT_TYPE_EPS,
  BMEPS_PS_LEVEL_2,
  (
    BMEPS_ENCODING_ASCII85
    |BMEPS_ENCODING_DCT
    |BMEPS_ENCODING_RUNLENGTH
  ),
  (DKBIF_JFIF_SOF0 | DKBIF_JFIF_SOF1),
  (
    BMEPS_OPT_ALPHA_MIX
    |BMEPS_OPT_OPERATOR_DICTIONARY
    |BMEPS_OPT_COLOR_OUTPUT
    |BMEPS_OPT_DICTIONARY
    |BMEPS_OPT_INTERPOLATE
    |BMEPS_OPT_PAPER_SIZE
    |BMEPS_OPT_SHOWPAGE
    |BMEPS_OPT_DSC
    |BMEPS_OPT_SEPARATED_DATA
    |BMEPS_OPT_FORCE_DCT
  ),
  0UL,
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 595.0, 842.0, 0.0, 0.0, 595.0, 842.0 },
  BMEPS_PREDICTOR_NONE
};

/** Default options for Fax to PDF. */
static BO bo_faxpdf = {
  BMEPS_OUTPUT_TYPE_PDF,
  BMEPS_PDF_LEVEL_14,
  (
    BMEPS_ENCODING_ASCII85
    |BMEPS_ENCODING_RUNLENGTH
    |BMEPS_ENCODING_DCT
    |BMEPS_ENCODING_FLATE
  ),
  (DKBIF_JFIF_SOF0 | DKBIF_JFIF_SOF1),
  (
    BMEPS_OPT_TRANSFER_ALPHA
    |BMEPS_OPT_OPERATOR_DICTIONARY
    |BMEPS_OPT_DICTIONARY
    |BMEPS_OPT_INTERPOLATE
    |BMEPS_OPT_PAPER_SIZE
    |BMEPS_OPT_FORCE_DCT
  ),
  (BMEPS_PDFOPT_FIT_WIDTH | BMEPS_PDFOPT_ALLOW_PDF_PAGE_ATTRIBUTES),
  1.0, 1.0, 1.0, 0.000001,
  { 0.0, 0.0, 595.0, 842.0, 0.0, 0.0, 595.0, 842.0 },
  BMEPS_PREDICTOR_INPUT
};

/** Default options table. */
static BOENTRY boentry[] = {
  { "ps1",	&bo_ps1 },
  { "ps2",	&bo_ps2 },
  { "ps3",	&bo_ps3 },
  { "ps",	&bo_ps2 },
  { "eps1",	&bo_eps1 },
  { "eps2",	&bo_eps2 },
  { "eps3",	&bo_eps3 },
  { "eps",	&bo_eps2 },
  { "faxps",	&bo_faxps2 },
  { "pdf12",	&bo_pdf12 },
  { "pdf13",	&bo_pdf13 },
  { "pdf14",	&bo_pdf14 },
  { "pdf",	&bo_pdf14 },
  { "faxpdf",	&bo_faxpdf },
  { "bb",	&bo_bb },
  { NULL, NULL }
};

/**	Asterisk string.
*/
static char str_asterisk[] = { "*" };

/**	String all.
*/
static char str_all[] = { "all" };

/**	Encoding names used to show configuration.
*/
static char *str_encodings_for_output[] = {
  "flate", "dct", "lzw", "run-length", "ascii85", NULL
};

/**	Fit options used to show configuration.
*/
static char *str_fittings_for_output[] = {
  "width", "height", "page", NULL
};

/**	Subdirectory in HOME where to place config file.
*/
static char conf_sub_dir[] = { "/.defaults" };

/**	Configuration file name.
*/
static char conf_file_name[] = { "/bmeps.conf" };

/**	Default home directory.
*/
static char hd[] = { "$(user.home)" };

/**	Text for compressed configuration file.
*/
static char *suffixes_to_test[] = { "", ".gz", ".bz2", NULL };



/**	Check whether the named configuration is a buildin default.
	@param	n	Configuration name.
	@return	Pointer to options set on success, NULL on error.
*/
static
BO *
search_default_configuration DK_P1(char *,n)
{
  BO *back = NULL;	/* Function result. */
  BOENTRY *runner;	/* Pointer to traverse array. */
  runner = boentry;
  while((runner->n) && (back == NULL)) {
    if(dkstr_casecmp(runner->n, n) == 0) {
      back = runner->bo;
    } 
    runner++;
  }
  return back;
}



/**	Compare to default option set table entries.
	@param	l	Left BOENTRY pointer.
	@param	r	Right BOENTRY pointer.
	@param	cr	Comparison criteria.
	@return	Comparison result.
*/
static
int
boe_compare DK_P3(void *,l, void *,r, int,cr)
{
  int back = 0;	/* Function result. */
  BOENTRY *pl;	/* Left pointer. */
  BOENTRY *pr;	/* Right pointer. */
  pl = (BOENTRY *)l; pr = (BOENTRY *)r;
  if(l) {
    if(r) {
      switch(cr) {
        case 1: {
	  if(pl->n) {
	    back = dkstr_casecmp(pl->n, (char *)r);
	  } else {
	    back = -1;
	  }
	} break;
	default: {
	  if(pl->n) {
	    if(pr->n) {
	      back = dkstr_casecmp(pl->n, pr->n);
	    } else {
	      back = 1;
	    }
	  } else {
	    if(pr->n) {
	      back = -1;
	    }
	  }
	} break;
      }
    } else {
      back = 1;
    }
  } else {
    if(r) {
      back = -1;
    }
  }
  return back;
}



/**	Destroy BOENTRY.
	@param	boe	Pointer to entry to destroy.
*/
static
void
boe_delete DK_P1(BOENTRY *,boe)
{
  if(boe) {
    if(boe->n) {
      char *x; x = boe->n; dk_delete(x);
    }
    if(boe->bo) {
      BO *bo; bo = boe->bo; dk_delete(bo);
    }
    boe->n = NULL; boe->bo = NULL;
    dk_delete(boe);
  }
}



/**	Create new BOENTRY.
	@param	n	Entry name.
	@return	Pointer on success, NULL on error.
*/
static
BOENTRY *
boe_new DK_P1(char *,n)
{
  BOENTRY *back = NULL;	/* Function result. */
  back = dk_new(BOENTRY,1);
  if(back) {
    back->n = dkstr_dup(n);
    back->bo = dk_new(BO,1);
    if(!((back->n) && (back->bo))) {
      boe_delete(back); back = NULL;
    }
  }
  return back;
}



/**	Change one option bit in option set.
	@param	ov	Old (original) option set.
	@param	fl	Bit to change.
	@param	s	String containing a boolean.
	@param	d	Default value.
	@param	b	Ignored.
	@return	New option set value.
*/
static
unsigned long
change_opt_set DK_P5(unsigned long,ov,unsigned long,fl,char *,s,int,d,int *,b)
{
  unsigned long back;
  int value;
  back = ov; value = d;
  if(s) {
  if(dkstr_is_bool(s)) {
    if(dkstr_is_on(s)) {
      value = 1;
    } else {
      value = 0;
    }
  }
  }
  if(value) {
    back = ov | fl;
  } else {
    back &= (~(fl));
  }
  return back;
}



/**	Set media size.
	@param	bj	Bmeps job.
	@param	bo	Bmeps option set
	@param	l	Text line containing the values.
	@param	back	Pointer to result variable. This
	variable is set to 1 on success, left unchanged
	on errors.
*/
static
void
set_media_size DK_P4(BJ *,bj, BO *,bo, char *,l, int *,back)
{
  char *p1 = NULL;	/* Start of text part 1. */
  char *p2 = NULL;	/* Start of text part 2. */
  char *p3 = NULL;	/* Start of text part 3. */
  char *p4 = NULL;	/* Start of text part 4. */
  char *p5 = NULL;	/* Start of text part 5. */
  char *p6 = NULL;	/* Start of text part 6. */
  char *p7 = NULL;	/* Start of text part 7. */
  char *p8 = NULL;	/* Start of text part 8. */
  double px0 = 0.0;	/* Lower left x of paper. */
  double py0 = 0.0;	/* Lower left y of paper. */
  double px1 = 0.0;	/* Upper right x of paper. */
  double py1 = 0.0;	/* Upper right y of paper. */
  double bx0 = 0.0;	/* Lower left x of draw area. */
  double by0 = 0.0;	/* Lower left y of draw area. */
  double bx1 = 0.0;	/* Upper right x of draw area. */
  double by1 = 0.0;	/* Upper right y of draw area. */
  MSENTRY *mptr = NULL;	/* Media size table entry. */
  
  p1 = dkstr_start(l, colonsep);
  if(p1) {
    p2 = dkstr_next(p1, colonsep);
    if(p2) {
      p3 = dkstr_next(p2, colonsep);
      if(p3) {
        p4 = dkstr_next(p3, colonsep);
	if(p4) {
	  p5 = dkstr_next(p4, colonsep);
	  if(p5) {
	    p6 = dkstr_next(p5, colonsep);
	    if(p6) {
	      p7 = dkstr_next(p6, colonsep);
	      if(p7) {
	        p8 = dkstr_next(p7, colonsep);
		if(p8) {
		  
	          if(sscanf(p1, "%lf", &px0) == 1) {
	          if(sscanf(p2, "%lf", &py0) == 1) {
	          if(sscanf(p3, "%lf", &px1) == 1) {
	          if(sscanf(p4, "%lf", &py1) == 1) {
	          if(sscanf(p5, "%lf", &bx0) == 1) {
	          if(sscanf(p6, "%lf", &by0) == 1) {
	          if(sscanf(p7, "%lf", &bx1) == 1) {
	          if(sscanf(p8, "%lf", &by1) == 1) {
	            (bo->ms).px0 = px0;
	            (bo->ms).py0 = py0;
	            (bo->ms).px1 = px1;
	            (bo->ms).py1 = py1;
	            (bo->ms).bx0 = bx0;
	            (bo->ms).by0 = by0;
	            (bo->ms).bx1 = bx1;
	            (bo->ms).by1 = by1;
	            bo->opt |= BMEPS_OPT_PAPER_SIZE;
	            *back = 1;
		    
		  }
		  }
		  }
		  }
		  }
		  }
		  }
		  }
		} else {
		}
	      } else {
	      }
	    }
	  } else {
	    
	    if(sscanf(p1, "%lf", &px0) == 1) {
	    if(sscanf(p2, "%lf", &py0) == 1) {
	    if(sscanf(p3, "%lf", &px1) == 1) {
	    if(sscanf(p4, "%lf", &py1) == 1) {
	      (bo->ms).px0 = px0;
	      (bo->ms).py0 = py0;
	      (bo->ms).px1 = px1;
	      (bo->ms).py1 = py1;
	      (bo->ms).bx0 = px0;
	      (bo->ms).by0 = py0;
	      (bo->ms).bx1 = px1;
	      (bo->ms).by1 = py1;
	      bo->opt |= BMEPS_OPT_PAPER_SIZE;
	      *back = 1;
	      
	    }
	    }
	    }
	    }
	  }
	} else {
	}
      } else {
      }
    } else {
      int cc;
      
      mptr = media_sizes; cc = 1;
      while((mptr->name) && (cc)) {
        if(dkstr_casecmp(mptr->name, p1) == 0) {
	  cc = 0;
	  bo->opt |= BMEPS_OPT_PAPER_SIZE;
	  dkmem_cpy(&(bo->ms), &(mptr->ms), sizeof(MEDIASIZE));
	  *back = 1;
	  
	}
        mptr++;
      }
    }
  }
  
}



/**	Retrieve allowed SOFs from a text line.
	@param	line	Text line to inspect.
	@return	SOF set.
*/
static
dk_bif_sof_t
retrieve_jfif_sof_set DK_P1(char *,line)
{
  int i;			/* Current SOF to enable. */
  char *p;			/* Current text pointer. */
  char *n;			/* Next text pointer. */
  dk_bif_sof_t back = 0;	/* Function result. */
  if(line) {
    p = dkstr_start(line, colonsep);
    while(p) {
      n = dkstr_next(p, colonsep);
      if(sscanf(p, "%d", &i) == 1) {
        switch(i) {
	  case 0: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF0); } break;
	  case 1: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF1); } break;
	  case 2: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF2); } break;
	  case 3: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF3); } break;
	  case 5: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF5); } break;
	  case 6: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF6); } break;
	  case 7: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF7); } break;
	  case 8: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF8); } break;
	  case 9: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF9); } break;
	  case 10: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF10); } break;
	  case 11: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF11); } break;
	  case 13: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF13); } break;
	  case 14: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF14); } break;
	  case 15: { back = dkbif_jfif_set_sof(back, DKBIF_I_SOF15); } break;
	}
      } else {
        if(strcmp(p, str_asterisk) == 0) {
	  back = 0xFFFF;
	} else {
	  if(strcmp(p, str_all) == 0) {
	    back = 0xFFFF;
	  }
	}
      }
      p = n;
    }
  }
  return back;
}



/**	Set flate predictor.
	@param	bj	Bmeps job.
	@param	bo	Bmeps options set.
	@param	v	Predictor name.
	@return	1 on success, 0 on error.
*/
static
void
set_predictor DK_P3(BJ *,bj, BO *,bo, char *,v) {
  switch(dkstr_array_index(str_predictor_names, v, 0)) {
    case 0: {
      bo->pred = BMEPS_PREDICTOR_TIFF;
    } break;
    case 1: {
      bo->pred = BMEPS_PREDICTOR_INPUT;
    } break;
    case 2: {
      bo->pred = BMEPS_PREDICTOR_PNG_SUB;
    } break;
    case 3: {
      bo->pred = BMEPS_PREDICTOR_PNG_UP;
    } break;
    case 4: {
      bo->pred = BMEPS_PREDICTOR_PNG_AVERAGE;
    } break;
    case 5: {
      bo->pred = BMEPS_PREDICTOR_PNG_PAETH;
    } break;
    default: {
      bo->pred = BMEPS_PREDICTOR_NONE;
      if(dkstr_is_bool(v)) {
        if(dkstr_is_on(v)) {
	  bo->pred = BMEPS_PREDICTOR_TIFF;
	}
      } else {
        char *msgptr[5];
	msgptr[0] = (bj->msg)[99];
	msgptr[1] = v;
	msgptr[2] = (bj->msg)[100];
	msgptr[3] = (bj->msg)[101];
	if(bj->a) {
	  dkapp_log_msg(bj->a, DK_LOG_LEVEL_ERROR, msgptr, 4);
	}
      }
    } break;
  }
}


/**	Add one configuration line to options set.
	@param	bj	Bmeps job.
	@param	bo	Bmeps options set.
	@param	s	Line from configuration file.
	@return	1 on success, 0 on error.
*/
int
bmeps_bo_add_line DK_P3(BJ *,bj, BO *,bo, char *,s)
{
  int back = 0;				/* Function result. */
  char *p1;				/* Pointer to text part 1. */
  char *p2;				/* Pointer to text part 2. */
  char *p3;				/* Pointer to text part 3. */
  char *p4;				/* Pointer to text part 4. */
  char *parts[16];			/* Pointers to split option name. */
  size_t nparts;			/* Number of option parts. */
  int is_removing = 0;			/* Flag: Currently removing. */
  int fl;				/* Flag. */
  int act;				/* Action to take. */
  double r;				/* Color red. */
  double g;				/* Color green. */
  double b;				/* Color blue. */
  char copy_of_key[CONFIG_LINE_SIZE];	/* Copy of the config key. */
  
  p1 = s;
  p2 = dkstr_chr(p1, '=');
  if(p2) {
    *(p2++) = '\0';
    dkstr_chomp(p1, NULL);
    p2 = dkstr_start(p2, NULL);
    if(p2) {
      dkstr_chomp(p2, NULL);
      if(strlen(p1) < sizeof(copy_of_key)) {
        strcpy(copy_of_key, p1);
      } else {
        strncpy(copy_of_key, p1, (sizeof(copy_of_key)-1));
      }
      nparts = dkstr_explode(parts, 15, p1, dotsep);
      if(nparts > 0) {
        act = dkstr_find_multi_part_abbr(parts, cmd, '$', 0);
	
	switch(act) {
	  case -1: {
	    bmeps_tool_msg_3(bj, DK_LOG_LEVEL_WARNING, 90, 91, copy_of_key);
	  } break;
	  case 0: {	/* output type */
	    if(p2) {
	      switch(dkstr_array_index(str_output_types, p2, 0)) {
	        case 0: {
		  bo->ot = BMEPS_OUTPUT_TYPE_EPS; back = 1;
		} break;
		case 1: {
		  bo->ot = BMEPS_OUTPUT_TYPE_PDF; back = 1;
		} break;
		case 2: {
		  bo->ot = BMEPS_OUTPUT_TYPE_BB;  back = 1;
		} break;
		default: {
	          bmeps_tool_msg_3(bj, DK_LOG_LEVEL_ERROR, 73, 74, p2);
	          bj->exval = 1;
		} break;
	      }
	    }
	  } break;
	  case 1: {	/* ps level */
	    switch(bo->ot) {
	      case BMEPS_OUTPUT_TYPE_PDF: {
	        if(p2) {
	          switch(dkstr_array_index(str_pdf_levels, p2, 0)) {
		    case 0: { bo->l = BMEPS_PDF_LEVEL_12; back = 1; } break;
		    case 1: { bo->l = BMEPS_PDF_LEVEL_13; back = 1; } break;
		    case 2: { bo->l = BMEPS_PDF_LEVEL_14; back = 1; } break;
		    default: {
		     bo->l = BMEPS_PDF_LEVEL_14; 
	             bmeps_tool_msg_3(bj, DK_LOG_LEVEL_ERROR, 73, 74, p2);
		    } break;
		  }
		}
	      } break;
	      default: {
	        int i;
		if(p2) {
		  if(sscanf(p2, "%d", &i) == 1) {
		    if(i < 1) i = 1; if(i > 3) i = 3;
		    switch(i) {
		      case 1: { bo->l = BMEPS_PS_LEVEL_1; back = 1; } break;
		      case 2: { bo->l = BMEPS_PS_LEVEL_2; back = 1; } break;
		      case 3: { bo->l = BMEPS_PS_LEVEL_3; back = 1; } break;
		    }
		  } else {
		    bo->l = BMEPS_PS_LEVEL_2;
	            bmeps_tool_msg_3(bj, DK_LOG_LEVEL_ERROR, 73, 74, p2);
		  }
		}
	      } break;
	    }
	  } break;
	  case 2: {	/* encoding */
	    if(p2) {
	      is_removing = 0;
	      if(*p2 == '-') {
	        is_removing = 1;
		p2++;
	      } else {
	        if(*p2 == '+') {
		  p2++;
		} else {
		  bo->enc = 0;
		}
	      }
	      while(p2) {
	        p3 = dkstr_next(p2, colonsep);
		fl = 0;
		switch(dkstr_array_abbr(str_encodings, p2, '$', 0)) {
		  case 0: { fl = BMEPS_ENCODING_ASCII85; back = 1; } break;
		  case 1: { fl = BMEPS_ENCODING_RUNLENGTH; back = 1; } break;
		  case 2: {
		      bmeps_tool_msg_1(bj, DK_LOG_LEVEL_WARNING, 78);
		  } break;
		  case 3: { fl = BMEPS_ENCODING_FLATE; back = 1; } break;
		  case 4: { fl = BMEPS_ENCODING_DCT; back = 1; } break;
		}
		if(is_removing) {
		  bo->enc &= (~(fl));
		} else {
		  bo->enc |= fl;
		}
		p2 = p3;
	      }
	    }
	  } break;
	  case 3: {	/* background */
	    r = g = b = 0;
	    if(p2) {
	      p3 = dkstr_next(p2, colonsep);
	      if(p3) {
	        p4 = dkstr_next(p3, colonsep);
		if(p4) {
		  if(sscanf(p2, "%lf", &r) == 1) {
		  if(sscanf(p3, "%lf", &g) == 1) {
		  if(sscanf(p4, "%lf", &b) == 1) {
		    bo->dbgr = r;
		    bo->dbgg = g;
		    bo->dbgb = b;
		    back = 1;
		    
		  }
		  }
		  }
		}
	      }
	    }
	  } break;
	  case 4: {	/* image mask trigger level */
	    if(p2) {
	      if(sscanf(p2, "%lf", &r) == 1) {
	        bo->mtl = r; back = 1;
	      }
	    }
	  } break;
	  case 5: {	/* media size */
	    set_media_size(bj, bo, p2, &back);
	  } break;
	  case 6: {	/* separated dictionary */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_DICTIONARY, p2, 1, &back
	    );
	  } break;
	  case 7: {	/* operator dictionary */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_OPERATOR_DICTIONARY, p2, 1, &back
	    );
	  } break;
	  case 8: {	/* separated color streams */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_SEPARATED_DATA, p2, 1, &back
	    );
	  } break;
	  case 9: {	/* always use default background */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_PREFER_SPEC_BG, p2, 1, &back
	    );
	  } break;
	  case 10: {	/* mix */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_ALPHA_MIX, p2, 1, &back
	    );
	  } break;
	  case 11: {	/* create image mask */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_IMAGE_MASK, p2, 1, &back
	    );
	  } break;
	  case 12: {	/* dsc */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_DSC, p2, 1, &back
	    );
	  } break;
	  case 13: {	/* showpage */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_SHOWPAGE, p2, 1, &back
	    );
	  } break;
	  case 14: {	/* vmreclaim */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_VMRECLAIM, p2, 1, &back
	    );
	  } break;
	  case 15: {	/* use resolution chunk */
	    
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_RESOLUTION, p2, 1, &back
	    );
	  } break;
	  case 16: {	/* transfer alpha channel */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_TRANSFER_ALPHA, p2, 1, &back
	    );
	  } break;
	  case 17: {	/* paper size */
	    /* bo->opt = change_opt_set(bo->opt, BMEPS_OPT_DICTIONARY, p2, 1); */
	    set_media_size(bj, bo, p2, &back);
	  } break;
	  case 18: {	/* draft */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_DRAFT, p2, 1, &back
	    );
	  } break;
	  case 19: {	/* interpolate */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_INTERPOLATE, p2, 1, &back
	    );
	  } break;
	  case 20: {	/* color */
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_COLOR_OUTPUT, p2, 1, &back
	    );
	  } break;
	  case 21: {	/* jfif sof set */
	    bo->jfif_s = retrieve_jfif_sof_set(p2);
	  } break;
	  case 22: {
	    bo->opt = change_opt_set(
	      bo->opt, BMEPS_OPT_JPEG_INTERPOLATE, p2, 1, &back
	    );
	  } break;
	  case 23: {
	    if(p2) {
	      switch(dkstr_array_abbr(str_fit_methods, p2, '$', 0)) {
	        case 0: {
		  bo->pdfopt =
		  ((bo->pdfopt & (~(BMEPS_PDFOPT_FIT_MASK)))
		  | BMEPS_PDFOPT_FIT_WIDTH);
		} break;
		case 1: {
		  bo->pdfopt =
		  ((bo->pdfopt & (~(BMEPS_PDFOPT_FIT_MASK)))
		  | BMEPS_PDFOPT_FIT_HEIGHT);
		} break;
		case 2: {
		  bo->pdfopt =
		  ((bo->pdfopt & (~(BMEPS_PDFOPT_FIT_MASK)))
		  | BMEPS_PDFOPT_FIT_PAGE);
		} break;
	      }
	    }
	  } break;
	  case 24: {
	    if(p2) {
	      bo->pdfopt = change_opt_set(
	        bo->pdfopt, BMEPS_PDFOPT_ALLOW_PDF_PAGE_ATTRIBUTES, p2, 1, &back
	      );
	    } else {
	      bo->pdfopt |= BMEPS_PDFOPT_ALLOW_PDF_PAGE_ATTRIBUTES;
	    }
	  } break;
	  case 25: {
	    if(p2) {
	      bo->opt = change_opt_set(
	        bo->opt, BMEPS_OPT_FORCE_DCT, p2, 1, &back
	      );
	    }
	  } break;
	  case 26: {
	    if(p2) {
	      set_predictor(bj, bo, p2);
	    } else {
	      bo->pred = BMEPS_PREDICTOR_NONE;
	    }
	  } break;
	}
      } else {
        /* ERROR: Syntax (empty line) */
      }
    } else {
      bmeps_tool_msg_3(bj, DK_LOG_LEVEL_ERROR, 73, 74, p1);
    }
  } else {
    bmeps_tool_msg_3(bj, DK_LOG_LEVEL_ERROR, 73, 74, p1);
  } 
  return back;
}



/**	Read configuration from file.
	@param	bj	Bmeps job.
	@param	a	Application structure.
	@param	n	Ignored.
	@param	s	Storage to save the file contents.
	@param	si	Iterator through storage.
*/
static
int
read_configuration DK_P5(BJ *,bj, dk_app_t *,a, char *,n, dk_storage_t *,s, dk_storage_iterator_t *,si)
{
  dk_stream_t	*ipf = NULL;		/* Input file */
  char inputline[CONFIG_LINE_SIZE];	/* One configuration file line. */
  char *p1 = NULL;			/* Start of configuration line. */
  char *p2 = NULL;			/* Value part of configuration line. */
  int back = 0;				/* Function result. */
  BOENTRY *cboe = NULL;			/* Current bmeps option set entry. */
  BOENTRY *baseboe = NULL;		/* Base bmeps option set entry. */
  BO *cbo = NULL;			/* Current bmeps option set. */
  BO *basebo = NULL;			/* Base bmeps option set. */
  int cc = 0;				/* Flag: Can continue. */
  
  ipf = dkapp_read_file(a, cfgfilename);
  if(ipf) {
    back = 1;
    cc = 1;
    while(cc) {
      if(dkstream_gets(ipf, inputline, sizeof(inputline))) {
        dkstr_delcomm(inputline, '#');
	p1 = dkstr_start(inputline, NULL);
	if(p1) {
	  dkstr_chomp(p1, NULL);
	  if(*p1 == '[') {			
	    cboe = NULL; cbo = NULL;
	    p1++;
	    p1 = dkstr_start(p1, NULL);
	    if(p1) {
	      p2 = dkstr_chr(p1, ']');
	      if(p2) {
	        *p2 = '\0';
		p2 = dkstr_chr(p1, ':');
		if(p2) {
		  *(p2++) = '\0';
		  dkstr_chomp(p2, NULL);
		}
		dkstr_chomp(p1, NULL);
		cboe = boe_new(p1);
		if(cboe) {
		  if(dksto_add(s, (void *)cboe)) {
		    cbo = cboe->bo;
		    dkmem_cpy(cbo, &bo_ps2, sizeof(BO));
		    if(p2) {
		      basebo = search_default_configuration(p2);
		      if(!basebo) {
		        baseboe = dksto_it_find_like(si, (void *)p2, 1);
			if(baseboe) {
			  basebo = baseboe->bo;
			}
		      }
		      if(basebo) {
			dkmem_cpy(cbo, basebo, sizeof(BO));
		      } else {
	                bmeps_tool_msg_3(bj, DK_LOG_LEVEL_ERROR, 75, 76, p2);
	                bj->exval = 1;
		      }
		    }
		  } else {
		    bmeps_tool_error_memory(bj); bj->exval = 1;
		    boe_delete(cboe);
		    cboe = NULL; cbo = NULL;
		  }
		} else {
		  bmeps_tool_error_memory(bj); bj->exval = 1;
		}
	      } else {
	        bmeps_tool_msg_3(bj, DK_LOG_LEVEL_ERROR, 73, 74, p1);
	      }
	    } else {
	      bmeps_tool_msg_3(bj, DK_LOG_LEVEL_ERROR, 73, 74, inputline);
	    }
	  } else {				
	    if(cbo) {
	      if(!bmeps_bo_add_line(bj, cbo, p1)) {	
	        back = 0;
	      }
	    }
	  }
	}
      } else {
        cc = 0;
      }
    }
    dkstream_close(ipf); ipf = NULL;
  } 
  return back;
}



/**	Search configuration file for language-specific configuration.
	@param	bj	Bmeps job.
	@param	a	Application structure.
	@param	n	Language name (for example: "eps" or "pdf").
	@param	bo	Option set to fill.
*/
static
void
search_config_file DK_P4(BJ *,bj, dk_app_t *,a, char *,n, BO *,bo)
{
  dk_storage_t	*conf;			/* Storage containing entries. */
  dk_storage_iterator_t	*confi;		/* Storage iterator. */
  BOENTRY *boe;				/* Bmeps option set table entry. */
  
  if((a) && (n)) {
    conf = dksto_open(0);
    if(conf) {
      dksto_set_comp(conf, boe_compare, 0);
      confi = dksto_it_open(conf);
      if(confi) {
        if(read_configuration(bj, a,n,conf,confi)) {
	  boe = dksto_it_find_like(confi, n, 1);
	  if(boe) {
	    if(boe->bo) {
	      dkmem_cpy(bo, boe->bo, sizeof(BO));
	    } else {
	      bmeps_tool_msg_3(bj, DK_LOG_LEVEL_ERROR, 75, 76, n);
	      bj->exval = 1;
	    }
	  }
	  dksto_it_reset(confi);
	  while((boe = (BOENTRY *)dksto_it_next(confi)) != NULL) {
	    
	    boe_delete(boe);
	  }
	}
        dksto_it_close(confi); confi = NULL;
      } else {
	bmeps_tool_error_memory(bj); bj->exval = 1;
      }
      dksto_close(conf); conf = NULL;
    } else {
      bmeps_tool_error_memory(bj); bj->exval = 1;
    }
  }
  
}



/**	Retrieve preferred language from preferences system.
	@param	bj	Bmeps job.
	@param	buffer	Buffer for language string.
	@param	sz	Buffer size.
	@return	1 on success, 0 on error.
*/
int
bmeps_bo_get_pref_lang DK_P3(BJ *,bj, char *,buffer,size_t,sz)
{
  int back = 0;	/* Function result. */
  back = dkapp_get_pref(bj->a, pk[0], buffer, sz, 0);
  return back;
}



/**	Load configuration for a language.
	@param	bj	Bmeps job.
	@param	langdef	Language definition.
	@param	v	Flag: verbose.
*/
void
bmeps_bo_get_configuration DK_P3(BJ *,bj, char *,langdef, int, v)
{
  BO bobuffer;			/* Temporary buffer for bmeps option set. */
  BO *src = NULL;		/* One of the build-in configurations. */
  char *l = NULL;		/* Language definition. */
  char *myl = NULL;		/* Private copy of language definition. */
  char *confname = NULL;	/* Configuration name. */
  char *p1;			/* Current part of language definition. */
  char *p2;			/* Place to continue in lang definition. */
  char namebuffer[NAME_BUFFER_SIZE];	/* Buffer to retrieve lang def. */
  
  l = langdef;
  if(!l) {
    char *ptr; int i;
    if(bj->o1) {
      ptr = dksf_get_file_type_dot(bj->o1);
      if(ptr) {
        ptr++;
        i = dkstr_array_index(ft_suffixes, ptr, 0);
	if(i > -1) {
	  l = ft_suffixes[i];
	} else {
	  if(v) {
	    bmeps_tool_msg_3(bj, DK_LOG_LEVEL_WARNING, 83, 84, ptr);
	  }
	}
      } else {
	if(v) { bmeps_tool_msg_1(bj, DK_LOG_LEVEL_WARNING, 82); }
      }
    } else {
      if(!((bj->cmd) & BMEPS_CMD_APP)) {
        if(v) { bmeps_tool_msg_1(bj, DK_LOG_LEVEL_WARNING, 81); }
      }
    }
  }
  if(!l) {		
    l = getenv(str_epsoutput);
  }
  if(!l) {		
    if(bj->a) {
      if(dkapp_get_pref(bj->a, pk[0], namebuffer, sizeof(namebuffer), 0)) {
        l = namebuffer;
      }
    }
  }
  if(l) {		
    if(!langdef) {
      /* Configuration ... is used */
      if(v) { bmeps_tool_msg_3(bj, DK_LOG_LEVEL_DEBUG, 85, 86, l); }
    }
    myl = dkstr_dup(l);
    if(myl) {		
      confname = dkstr_start(myl, NULL);
      if(confname) {	
        p1 = dkstr_next(confname, commasep);
	/* if(p1) { *(p1++) = '\0'; } $? ". p1=\"%s\"", p1 */
	dkmem_cpy(&bobuffer, &bo_ps2, sizeof(BO));
	src = search_default_configuration(confname);
	if(src) {
	  dkmem_cpy(&bobuffer, src, sizeof(BO));
	} else {
	  search_config_file(bj, bj->a, confname, &bobuffer);
	}
	src = &bobuffer;
	if(p1) { 
	  while(p1) {
	    p2 = dkstr_next(p1, commasep);
	    
	    (void)bmeps_bo_add_line(bj, &bobuffer, p1);
	    p1 = p2;
	  }
	}
      } else {
	if(v) {
	  bmeps_tool_msg_3(bj, DK_LOG_LEVEL_ERROR, 73, 74, myl);
	}
      }
      dk_delete(myl);
    } else {
      if(v) { bmeps_tool_error_memory(bj); bj->exval = 1; }
    }
  } else {
    if(!((bj->cmd) & BMEPS_CMD_APP)) {
      if(v) { bmeps_tool_msg_1(bj, DK_LOG_LEVEL_INFO, 77); }
    }
  }
  if(!src) {
    src = &bo_ps2;
  }
  if(src) {
    dkmem_cpy(bj->bo, src, sizeof(BO));
  }
  bmeps_tool_correct_bo(bj->bo);
  
}



/**	Show configuration settings.
	@param	bo	Bmeps option set.
	@param	n	Index of option to show.
*/
static
void
show_setting DK_P2(BO *,bo, int, n)
{
  int i;		/* Flag: Is not first element to print. */
  int have_data;	/* Flag: This option is configured. */
  unsigned long ul;	/* Index in SOF set to check. */
  i = have_data = 0;
  switch(n) {
    case 0: {
      i = bo->ot; if(i < 0) i = 0; if(i > 2) i = 2;
      printf("%s=%s\n", cmd_strings[n], str_output_types[i]);
    } break;
    case 1: {
      switch(bo->ot) {
        case BMEPS_OUTPUT_TYPE_EPS: {
	  printf("%s=%d\n", cmd_strings[n], bo->l);
	} break;
	case BMEPS_OUTPUT_TYPE_PDF: {
	  i = bo->l - 9;
	  printf("%s=1.%d\n", cmd_strings[n], i);
	} break;
      }
    } break;
    case 2: {
      printf("%s=", cmd_strings[n]);
      if((bo->enc) & BMEPS_ENCODING_FLATE) {
        fputs(str_encodings_for_output[0], stdout);
        i = 1;
      }
      if((bo->enc) & BMEPS_ENCODING_DCT) {
        if(i) { fputc(',', stdout); }
	fputs(str_encodings_for_output[1], stdout);
        i = 1;
      }
      if((bo->enc) & BMEPS_ENCODING_RUNLENGTH) {
        if(i) { fputc(',', stdout); }
	fputs(str_encodings_for_output[3], stdout);
        i = 1;
      }
      if((bo->enc) & BMEPS_ENCODING_ASCII85) {
        if(i) { fputc(',', stdout); }
	fputs(str_encodings_for_output[4], stdout);
        i = 1;
      }
      printf("\n");
    } break;
    case 3: {
      printf("%s=%lg,%lg,%lg\n", cmd_strings[n], bo->dbgr, bo->dbgg, bo->dbgb);
    } break;
    case 4: {
      printf("%s=%lg\n", cmd_strings[n], bo->mtl);
    } break;
    case 5: {
      if((bo->opt) & BMEPS_OPT_PAPER_SIZE) {
        printf("%s=%lg,%lg,%lg,%lg,%lg,%lg,%lg,%lg\n", cmd_strings[n], (bo->ms).px0, (bo->ms).py0, (bo->ms).px1, (bo->ms).py1, (bo->ms).bx0, (bo->ms).by0, (bo->ms).bx1, (bo->ms).by1);
      }
    } break;
    case 6: {
       if((bo->opt) & BMEPS_OPT_DICTIONARY) { i = 1; }
       printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 7: {
      if((bo->opt) & BMEPS_OPT_OPERATOR_DICTIONARY) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 8: {
      if((bo->opt) & BMEPS_OPT_SEPARATED_DATA) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 9: {
      if((bo->opt) & BMEPS_OPT_PREFER_SPEC_BG) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 10: {
      if((bo->opt) & BMEPS_OPT_ALPHA_MIX) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 11: {
      if((bo->opt) & BMEPS_OPT_IMAGE_MASK) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 12: {
      if((bo->opt) & BMEPS_OPT_DSC) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 13: {
      if((bo->opt) & BMEPS_OPT_SHOWPAGE) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 14: {
      if((bo->opt) & BMEPS_OPT_VMRECLAIM) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 15: {
      if((bo->opt) & BMEPS_OPT_RESOLUTION) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 16: {
      if((bo->opt) & BMEPS_OPT_TRANSFER_ALPHA) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 18: {
      if((bo->opt) & BMEPS_OPT_DRAFT) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 19: {
      if((bo->opt) & BMEPS_OPT_INTERPOLATE) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 20: {
      if((bo->opt) & BMEPS_OPT_COLOR_OUTPUT) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 21: {
      printf("%s=", cmd_strings[n]);
      ul = 1UL;
      for(i = 0; i < 16; i++) {
        if((bo->jfif_s) & ul) {
	  if(have_data) { fputc(',', stdout); }
	  have_data = 1;
	  printf("%d", i);
	}
	ul = ul * 2UL;
      }
      printf("\n");
    } break;
    case 22: {
      if((bo->opt) & BMEPS_OPT_JPEG_INTERPOLATE) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 23: {
      if((bo->pdfopt) & BMEPS_PDFOPT_FIT_MASK) {
        printf("%s=%s\n", cmd_strings[n], str_fittings_for_output[((bo->pdfopt) & BMEPS_PDFOPT_FIT_MASK) - 1UL]);
      }
    } break;
    case 24: {
      i = 0;
      if((bo->pdfopt) & BMEPS_PDFOPT_ALLOW_PDF_PAGE_ATTRIBUTES) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
    case 25: {
      i = 0;
      if((bo->opt) & BMEPS_OPT_FORCE_DCT) { i = 1; }
      printf("%s=%d\n", cmd_strings[n], i);
    } break;
  }
}



/**	Show settings for bmeps option set.
	@param	bo	Option set to show.
*/
static
void
show_entry DK_P1(BO *,bo)
{
  switch(bo->ot) {
    case BMEPS_OUTPUT_TYPE_EPS: {
      show_setting(bo, 0);
      if((bo->opt) & BMEPS_OPT_DRAFT) {
        show_setting(bo, 18);
      } else {
        show_setting(bo, 1);
	show_setting(bo, 12);
	show_setting(bo, 20);
	if((bo->opt) & BMEPS_OPT_COLOR_OUTPUT) {
	  show_setting(bo, 8);
	}
	show_setting(bo, 7);
	show_setting(bo, 6);
	show_setting(bo, 13);
	if((bo->opt) & BMEPS_OPT_DICTIONARY) {
	  show_setting(bo, 14);
	}
	show_setting(bo, 2);
	if((bo->enc) & BMEPS_ENCODING_DCT) {
	  show_setting(bo, 21);
	}
	show_setting(bo, 19);
        if((bo->opt) & BMEPS_OPT_INTERPOLATE) {
	  show_setting(bo, 22);
	}
	show_setting(bo, 10);
        if((bo->opt) & BMEPS_OPT_ALPHA_MIX) {
	  show_setting(bo, 3);
	  show_setting(bo, 9);
	}
	show_setting(bo, 11);
        if((bo->opt) & BMEPS_OPT_IMAGE_MASK) {
	  show_setting(bo, 4);
	}
	show_setting(bo, 15);
	show_setting(bo, 5);
	show_setting(bo, 24);
	show_setting(bo, 25);
      }
    } break;
    case BMEPS_OUTPUT_TYPE_PDF: {
      show_setting(bo, 0);
      if((bo->opt) & BMEPS_OPT_DRAFT) {
        show_setting(bo, 18);
      } else {
        show_setting(bo, 1);
	show_setting(bo, 21);
	show_setting(bo, 20);
	show_setting(bo, 19);
        if((bo->opt) & BMEPS_OPT_INTERPOLATE) {
	  show_setting(bo, 22);
	}
	show_setting(bo, 16);
	if(!((bo->opt) & BMEPS_OPT_TRANSFER_ALPHA)) {
	  show_setting(bo, 10);
          if((bo->opt) & BMEPS_OPT_ALPHA_MIX) {
	    show_setting(bo, 3);
	    show_setting(bo, 9);
	  }
	  show_setting(bo, 11);
          if((bo->opt) & BMEPS_OPT_IMAGE_MASK) {
	    show_setting(bo, 4);
	  }
	}
	show_setting(bo, 15);
	show_setting(bo, 5);
	show_setting(bo, 23);
	show_setting(bo, 24);
	show_setting(bo, 25);
      }
    } break;
    case BMEPS_OUTPUT_TYPE_BB: {
      show_setting(bo, 0);
    } break;
  }
}




/**	Print one configuration line.
	@param	k	Configuration line key.
	@param	v	Configuration line value.
*/
void
print_config_line DK_P2(char *,k, char *,v)
{
  char *parts[16];	/* Parts to split configuration line. */
  size_t nparts;	/* Number of parts after splitting. */
  int act;		/* Index of key. */
  int i;		/* Flag: Is not first value to show. */

  i = 0;
  nparts = dkstr_explode(parts, 15, k, dotsep);
  if(nparts > 0) {
    act = dkstr_find_multi_part_abbr(parts, cmd, '$', 0);
    if(act >= 0) {
      switch(act) {
        case 0:
	case 1:
	case 2:
	case 3:
	case 4:
	case 5:
	case 17:
	case 21:
	case 23: {
	  printf("%s=%s\n", cmd_strings[act], v);
	} break;
	default: {
	  if(dkstr_is_bool(v)) {
	    if(dkstr_is_on(v)) {
	      i = 1;
	    }
	  }
	  printf("%s=%d\n", cmd_strings[act], i);
	} break;
      }
    }
  }
}




/**	Show configuration file contents.
	@param	bj	Bmeps job.
	@return	1 on success, 0 on error.
*/
int
bmeps_bo_show_configuration_file DK_P1(BJ *,bj)
{
  int back = 0;			/* Function result. */
  long maxs = 0;		/* Maximum path length. */
  char *buffer = NULL;		/* Buffer. */
  dk_stream_t *is = NULL;	/* Input stream to read config file. */
  BOENTRY *boptr = NULL;	/* Bmeps option set. */
  char *k = NULL;		/* Option line key. */
  char *v = NULL;		/* Option line value. */
  int llmax;			/* Maximum log level .*/

  maxs = dksf_get_maxpathlen();
  if(maxs < (long)CONFIG_LINE_SIZE) {
    maxs = (long)CONFIG_LINE_SIZE;
  }
  boptr = boentry;
#if DK_HAVE_FNCASEINS
  fputs("# @PLATFORM filename case-insensitive\n", stdout);
#else
  fputs("# @PLATFORM filename case-sensitive\n", stdout);
#endif
  fputs(file_types_string, stdout);
  fputs("# @BUILT-IN\n", stdout);
  while(boptr->n) {
    printf("[%s]\n", boptr->n);
    show_entry(boptr->bo);
    boptr++;
  }
  buffer = dk_new(char,(size_t)maxs);
  if(buffer) {
    if(bj->a) {
      /* it's not a problem if there is no config file */
      llmax = dkapp_llmax_get(bj->a);
      is = dkapp_read_file(bj->a, cfgfilename);
      if(is) {
        back = 1;
	dkmem_res(buffer, maxs);
	fputs("# @CONFIG-FILE\n", stdout);
        while(dkstream_gets(is, buffer, maxs)) {
	  /* fputs(buffer, stdout); */
	  dkstr_delcomm(buffer, '#');
	  k = dkstr_start(buffer, NULL);
	  if(k) {
	    if(*k == '[') {
	      dkstr_chomp(k, NULL);
	      printf("%s\n", k);
	    } else {
	      v = dkstr_chr(k, '=');
	      if(v) {
	        *(v++) = '\0';
		k = dkstr_start(k, NULL); v = dkstr_start(v, NULL);
		if((k) && (v)) {
		  dkstr_chomp(k, NULL); dkstr_chomp(v, NULL);
		  print_config_line(k, v);
		}
	      }
	    }
	  }
	  dkmem_res(buffer, maxs);
	}
        dkstream_close(is);
      } else {
        /* No configuration file */
      }
      dkapp_llmax_set(bj->a, llmax);
    } else {
      /* Internal error, must not happen */
    }
    dk_delete(buffer); buffer = NULL;
  } else {
    if(bj->a) {
      dkapp_err_memory(bj->a, 1, maxs);
    }
  }
  return back;
}



/**	Save standard input into new configuration file.
	@param	bj	Bmeps job.
	@return	1 on success, 0 on error.
*/
int
bmeps_bo_write_configuration_file DK_P1(BJ *,bj)
{
  int back = 0;				/* Function result. */
  int must_eat_up = 1;			/* Flag: Must read until EOF. */
  char buffer[CONFIG_LINE_SIZE];	/* Input line buffer. */
  char b2[CONFIG_LINE_SIZE];		/* Copy of input line. */
  char *hdp;				/* Home directory. */
  FILE *f;				/* Read input file. */
  char **ptr;				/* Pointer to test all suffixes. */
  dk_stat_t *stp = NULL;		/* File information struct. */

  must_eat_up = 1;
  if(bj->a) {
    if(dkapp_transform_string_ext1(bj->a, buffer, sizeof(buffer), hd, 1)) {
      hdp = dkstr_dup(buffer);
      if(hdp) {
        if(strlen(hdp) + strlen(conf_sub_dir) + strlen(conf_file_name)
	   < sizeof(buffer))
	{
	  strcpy(buffer, hdp);
	  strcat(buffer, conf_sub_dir);
	  strcat(buffer, conf_file_name);
	  dksf_correct_fnsep(buffer);
	  f = dkapp_fopen(bj->a, buffer, "w");
	  if(f) {
	    while(fgets(buffer, sizeof(buffer), stdin)) {
	      fputs(buffer, f);
	    }
	    must_eat_up = 0; back = 1;
	    fclose(f);
	    ptr = suffixes_to_test;
	    while(*ptr) {
	      strcpy(b2, hdp);
	      strcat(b2, conf_file_name);
	      strcat(b2, *ptr);
	      dksf_correct_fnsep(b2);
	      stp = dkstat_open(b2);
	      if(stp) {
	        strcpy(buffer, hdp);
	        strcat(buffer, conf_sub_dir);
	        strcat(buffer, conf_file_name);
	        dksf_correct_fnsep(buffer);
	        bmeps_tool_msg_5(bj, DK_LOG_LEVEL_WARNING, 92, 93, 94, b2, buffer);
	        dkstat_close(stp); stp = NULL;
	      }
	      ptr++;
	    }
	  } else {
	    dkapp_err_fopenw(bj->a, buffer);
	  }
	} else {
	  bmeps_tool_msg_3(bj, DK_LOG_LEVEL_ERROR, 87, 88, hdp);
	}
        dk_delete(hdp);
      } else {
        dkapp_err_memory(bj->a, 1, (1 + strlen(buffer)));
      }
    } else {
      bmeps_tool_msg_1(bj, DK_LOG_LEVEL_ERROR, 89);
    }
  } else {
    /* internal setup error, must not happen */
  }
  if(must_eat_up) {
    while(fgets(buffer, sizeof(buffer), stdin)) {
    }
  }
  return back;
}



