/*
Copyright (c) 2001-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	trana.c	The trana program.
*/



#include "dk.h"
#include "dkstr.h"
#include "dkapp.h"
#include "dklic.h"
#include "dklogc.h"
#include "dkmem.h"
#include "dksf.h"

#include <stdio.h>

#if DK_HAVE_STRING_H
#include <string.h>
#endif
#if DK_HAVE_STRINGS_H
#include <strings.h>
#endif
#if DK_HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if DK_HAVE_UNISTD_H
#include <unistd.h>
#endif
#if DK_HAVE_PROCESS_H
#include <process.h>
#endif

#include "dktools-version.h"




#line 75 "trana.ctr"




/**	Version number.
*/
static char the_version_number[] = { VERSNUMB } ;

/**	System configuration directory.
*/
static char sysconfdir[] = { DK_SYSCONFDIR };

/**	Application group name.
*/
static char packagename[] = { "dktools" };

/**	Application name.
*/
static char programname[] = { "trana" };



/**	Three spaces.
*/
static char three_spaces[] = { "   " };



/** Mode option: Use marker.
*/
#define OPTION_MARKER	1

/** Mode option: Use position.
*/
#define OPTION_POSITION	2

/** Mode option: Require trace keyowrd.
*/
#define OPTION_TRACEONLY 4

/** Mode option: Print help.
*/
#define OPTION_HELP	8

/** Mode option: Print version.
*/
#define OPTION_VERSION	16

/** Mode option: Configure.
*/
#define OPTION_CONFIGURE	32

/** Mode option: Unconfigure.
*/
#define OPTION_UNCONF		64

/** Mode option: Numeric.
*/
#define OPTION_NUMERIC		128

/** Mode option: Error occured.
*/
#define OPTION_ERROR		1024

/** Mode option: Show configuration.
*/
#define OPTION_SHOWCONF		2048



/**	Application.
*/
static dk_app_t *app = NULL;



/**	Input line.
*/
static char inputline[1024];

/**	Copy of input line.
*/
static char copyline[1024];



/**	Keyword "trace".
*/
static char kw_trace[] = { "trace" };



/**	Preference key: Use marker.
*/
static char pk_marker[] = { "/marker" };

/**	Preference key: Position.
*/
static char pk_position[] = { "/position" };

/**	Preference key: Trace keyword needed.
*/
static char pk_traceonly[] = { "/trace-only" };

/**	Preference key: Numeric.
*/
static char pk_numeric[] = { "/numeric" };



/**	Message texts used for output, filled by string finder.
*/
static char *messages[10];

/**	String finder entries.
*/
static dk_string_finder_t trana_msg[] = {
{ (char *)"/e/001", &(messages[0]), (char *)"Unknown option!" },
{ (char *)"/e/002", &(messages[1]), (char *)"Too many filenames!" },
{ (char *)"/m/001", &(messages[2]), (char *)"Current configuration:" },
{ (char *)"/m/002", &(messages[3]), (char *)"Markers for vim folding" },
{ (char *)"/m/003", &(messages[4]), (char *)"Position in source" },
{ (char *)"/m/004", &(messages[5]), (char *)"\"trace\" lines only" },
{ (char *)"/m/005", &(messages[6]), (char *)"Show indent numbers" },
{ (char *)"/m/006", &(messages[7]), (char *)" enabled" },
{ (char *)"/m/007", &(messages[8]), (char *)"disabled" },
{ NULL, NULL, NULL }
};



/** String: opening bracket.
*/
static char str_bropen[] = { "(" };

/** String: closing bracket.
*/
static char str_brclose[] = { ")" };

/** String: space.
*/
static char str_spc[] = { " " };



/**	File open mode: read.
*/
static char str_r[] = { "r" };



/**	File open mode: write.
*/
static char str_w[] = { "w" };



/**	String table name.
*/
static char string_table_name[] = { "trana" };



/**	Run for input file and output file.
	@param	in	Input file.
	@param	out	Output file.
	@param	mode	Mode.
	@return	1 on success, 0 on error.
*/
static int
run_for_files DK_P3(FILE *,in,FILE *,out,int,mode)
{
  int cc;
  int indent, direction, istrace;
  char *p1, *p2, *p3;
  cc = 1; indent = 0;
  while(cc) {
    if(fgets(inputline,sizeof(inputline),in)) {
      strcpy(copyline, inputline);
      dkstr_chomp(copyline, NULL);
      p1 = p2 = p3 = NULL; direction  = 0; istrace = 0;
      p1 = dkstr_start(inputline, NULL);
      if(p1) {
        dkstr_chomp(p1, NULL);
        p2 = dkstr_next(p1, NULL);
	if(p2) {
	  p3 = dkstr_next(p2, NULL);
	}
      }
      if(p2) {
        if(*p2 == '+') { direction = 1; }
	if(*p2 == '-') { direction = -1; }
        if(strcmp(p2, kw_trace) == 0) {
	  istrace = 1;
	  if(p3 && (direction == 0)) {
	    if(*p3 == '+') { direction = 1; }
	    if(*p3 == '-') { direction = -1; }
	  }
	}
      }
      if(direction == 1) { indent++; }
      if(mode & OPTION_NUMERIC) {
        fprintf(out,"%-5d ", indent);
      }
      { int i; for(i = 0; i < indent; i++) { fputs("  ", out); } }
      if(istrace || (!(mode & OPTION_TRACEONLY))) {
        if(istrace) {
	  if(mode & OPTION_POSITION) {
	    fputs(p3, out); fputc(' ', out); fputs(p1, out);
	  } else {
	    fputs(p3, out);
	  }
        } else {
	  if(mode & OPTION_POSITION) {
	    fputs(copyline, out);
	  } else {
	    if(p2) {
	      fputs(p2, out);
	      if(p3) {
	        fputc(' ', out); fputs(p3, out);
	      }
	    }
	  }
        }
      }
      if(mode & OPTION_MARKER) {
        switch(direction) {
          case 1: {
	    fprintf(out, " {{{%d", indent);
	  } break;
	  case -1: {
	    fprintf(out, " }}}%d", indent);
	  } break;
        }
      }
      if(direction == -1) { indent--; if(indent < 0) indent = 0; }
      fputc('\n', out);
    } else {
      cc = 0;
    }
  }
  return 1;
}



/**	Run for one inputfile, write to stdout.
	@param	inname	Name of input file.
	@param	mode	Mode.
	@param	outputfile	Output file.
	@return	1 on success, 0 on error.
*/
static int
run_for_input DK_P3(char *,inname,int,mode,FILE *,outputfile)
{
  int back = 0; char *copy_of_inname; dk_fne_t *fne;
  FILE *in;
  copy_of_inname = dkstr_dup(inname);
  if(copy_of_inname) {
    dksf_correct_fnsep(copy_of_inname);
    if(dksf_must_expand_filename(copy_of_inname)) {
      fne = dkfne_open(copy_of_inname, 1, 0);
      if(fne) {
        if(dkfne_next(fne)) {
	  in = dkapp_fopen(app, dkfne_get_fullname(fne), str_r);
	  if(in) {
	    back = run_for_files(in,outputfile,mode);
	    fclose(in);
	  } else {
	    dkapp_err_fopenr(app, dkfne_get_fullname(fne));
	  }
	} else {
	  dkapp_err_matchfile(app, copy_of_inname);
	}
        dkfne_close(fne);
      } else {
        dkapp_err_matchfile(app, copy_of_inname);
      }
    } else {
      in = dkapp_fopen(app, inname, str_r);
      if(in) {
        back = run_for_files(in,outputfile,mode);
        fclose(in);
      } else {
        dkapp_err_fopenr(app, inname);
      }
    }
    dk_delete(copy_of_inname);
  } else {
    dkapp_err_memory(app,1,strlen(inname));
  }
  return back;
  /*
  in = dkapp_fopen(app, inname, str_r);
  if(in) {
    back = run_for_files(in,outputfile,mode);
    fclose(in);
  } else {
    dkapp_err_fopenr(app, inname);
  }
  return back;
  */
}



/**	Run for a pair of filename.
	@param	inname	Input file name.
	@param	outname	Output file name.
	@param	mode	Mode value.
	@return	1 on success, 0 on error.
*/
static int
run_for_names DK_P3(char *,inname,char *,outname,int,mode)
{
  int back = 0;
  FILE *out; char *copy_of_outputname; dk_fne_t *fne;

  copy_of_outputname = dkstr_dup(outname);
  if(copy_of_outputname) {
    dksf_correct_fnsep(copy_of_outputname);
    if(dksf_must_expand_filename(copy_of_outputname)) {
      fne = dkfne_open(copy_of_outputname, 1, 0);
      if(fne) {
        if(dkfne_next(fne)) {
	  out = dkapp_fopen(app, dkfne_get_fullname(fne), str_w);
	  if(out) {
	    back = run_for_input(inname, mode, out);
	    fclose(out);
	  } else {
	    dkapp_err_fopenw(app, dkfne_get_fullname(fne));
	  }
	} else {
	  dkapp_err_matchfile(app, copy_of_outputname);
	}
        dkfne_close(fne);
      } else {
        dkapp_err_matchfile(app, copy_of_outputname);
      }
    } else {
      out = dkapp_fopen(app, copy_of_outputname, str_w);
      if(out) {
        back = run_for_input(inname, mode, out);
        fclose(out);
      } else {
        dkapp_err_fopenw(app, outname);
      }
    }
    dk_delete(copy_of_outputname);
  } else {
    dkapp_err_memory(app,1,strlen(outname));
  }
  return back;
}



/**	Check for a bool preference turned on
	@param	mode	Mode value.
	@param	pk	Preference key.
	@param	bit	Bit pattern corresponding to preference key.
	@return	Changed mode value.
*/
static int
check_prf_on DK_P3(int,mode,char *,pk,int,bit)
{
  int back;
  char line[32];
  back = mode;
  if(dkapp_get_pref(app, pk, line, sizeof(line), 0)) {
    if(dkstr_is_on(line)) {
      back |= bit;
    } else {
      back &= (~bit);
    }
  }
  return back;
}



/**	Retrieve default configuration.
	@return	Mode value.
*/
static int
get_default_mode DK_P0()
{
  int back = OPTION_POSITION;
  back = check_prf_on(back, pk_marker, OPTION_MARKER);
  back = check_prf_on(back, pk_position, OPTION_POSITION);
  back = check_prf_on(back, pk_traceonly, OPTION_TRACEONLY);
  back = check_prf_on(back, pk_numeric, OPTION_NUMERIC);
  return back;
}



/** Options keywords.
*/
static char *kw_options[] = {
  (char *)"m$arker",
  (char *)"p$osition",
  (char *)"t$race-only",
  (char *)"v$ersion",
  (char *)"h$elp",
  (char *)"c$onfigure",
  (char *)"u$nconfigure",
  (char *)"n$umeric",
  (char *)"sh$owconf",
  (char *)"r$eset",
  NULL
};



/**	Strings to write configuration: yes.
*/
static char str_yes[] = { "yes" };

/**	Strings to write configuration: no.
*/
static char str_no[] = { "no" };



/**	Set bit in mode from option.
	@param	mode	Old mode value.
	@param	bit	Bit to set or reset.
	@param	p	New value (set or reset bit).
	@return	New mode value.
*/
static int
set_bit_from_opt DK_P3(int,mode,int,bit,char *,p)
{
  int back;
  back = mode;
  if(*p) {
    if(*p == '-') {
      back &= (~bit);
    } else {
      if(*p == '+') {
        back |= bit;
      } else {
        if(dkstr_is_bool(p)) {
	  if(dkstr_is_on(p)) {
	    back |= bit;
	  } else {
	    back &= (~bit);
	  }
	}
      }
    }
  } else {
    back |= bit;
  }
  return back;
}



/**	Process command line arguments.
	@param	c	Number of command line arguments.
	@param	v	Command line arguments array.
	@param	m	Pointer to mode result variable.
	@param	i	Address of pointer to input file name.
	@param	o	Address of pointer to output file name.
*/
static void
process_arguments DK_P5(int,c,char **,v,int *,m,char **,i,char **,o)
{
  int mode, lfd; char *p1, *p2, **lfdptr;
  char *msgptr[5];
  
  lfdptr = v; lfdptr++; lfd = 1;
  mode = *m;
  while(lfd < c) {
    p1 = *lfdptr; 
    if(*p1 == '-') { 
      p1++;
      switch(*p1) {
        case '-': {
	  p1++;
	  p2 = strchr(p1, '=');
	  if(p2) {
	    *(p2++) = '\0';
	  }
	  switch(dkstr_array_abbr(kw_options,p1,'$',1)) {
	    case 0: {
	      if(p2) {
	        if(dkstr_is_on(p2)) {
		  mode |= OPTION_MARKER;
		} else {
		  mode &= (~OPTION_MARKER);
		}
	      } else {
	        mode |= OPTION_MARKER;
	      }
	    } break;
	    case 1: {
	      if(p2) {
	        if(dkstr_is_on(p2)) {
		  mode |= OPTION_POSITION;
		} else {
		  mode &= (~OPTION_POSITION);
		}
	      } else {
	        mode |= OPTION_POSITION;
	      }
	    } break;
	    case 2: {
	      if(p2) {
	        if(dkstr_is_on(p2)) {
		  mode |= OPTION_TRACEONLY;
		} else {
		  mode &= (~OPTION_TRACEONLY);
		}
	      } else {
	        mode |= OPTION_TRACEONLY;
	      }
	    } break;
	    case 3: {
	      mode |= OPTION_VERSION;
	    } break;
	    case 4: {
	      mode |= OPTION_HELP;
	    } break;
	    case 5: {
	      mode |= OPTION_CONFIGURE;
	    } break;
	    case 6: {
	      mode |= OPTION_UNCONF;
	    } break;
	    case 7: {
	      if(p2) {
	        if(dkstr_is_on(p2)) {
		  mode |= OPTION_NUMERIC;
		} else {
		  mode &= (~OPTION_NUMERIC);
		}
	      } else {
	        mode |= OPTION_NUMERIC;
	      }
	    } break;
	    case 8: {
	      mode |= OPTION_SHOWCONF;
	    } break;
	    case 9: {
mode &= ~(OPTION_TRACEONLY|OPTION_POSITION|OPTION_MARKER|OPTION_NUMERIC);
	    } break;
	    default: {
	      p1--; p1--;
	      msgptr[0] = messages[0];
	      msgptr[1] = str_spc;
	      msgptr[2] = str_bropen;
	      msgptr[3] = p1;
	      msgptr[4] = str_brclose;
	      dkapp_log_msg(app, DK_LOG_LEVEL_ERROR, msgptr, 5);
	      mode |= OPTION_ERROR;
	    } break;
	  }
	} break;
	case 'm': {
	  /* mode |= OPTION_MARKER; */
	  mode = set_bit_from_opt(mode, OPTION_MARKER, ++p1);
	} break;
	case 'p': {
	  /* mode |= OPTION_POSITION; */
	  mode = set_bit_from_opt(mode, OPTION_POSITION, ++p1);
	} break;
	case 't': {
	  /* mode |= OPTION_TRACEONLY; */
	  mode = set_bit_from_opt(mode, OPTION_TRACEONLY, ++p1);
	} break;
	case 'v': {
	  mode |= OPTION_VERSION;
	} break;
	case 'h': {
	  mode |= OPTION_HELP;
	} break;
	case 'c': {
	  mode |= OPTION_CONFIGURE;
	} break;
	case 'u': {
	  mode |= OPTION_UNCONF;
	} break;
	case 'r': {
mode &= ~(OPTION_TRACEONLY|OPTION_MARKER|OPTION_NUMERIC);
mode |= OPTION_POSITION;
	} break;
	case 'n': {
	  /* mode |= OPTION_NUMERIC; */
	  mode = set_bit_from_opt(mode, OPTION_NUMERIC, ++p1);
	} break;
	case 'C': {
	  mode |= OPTION_SHOWCONF;
	} break;
	default: {
	  p1--;
	  msgptr[0] = messages[0];
	  msgptr[1] = str_spc;
	  msgptr[2] = str_bropen;
	  msgptr[3] = p1;
	  msgptr[4] = str_brclose;
	  dkapp_log_msg(app, DK_LOG_LEVEL_ERROR, msgptr, 5);
	  mode |= OPTION_ERROR;
	} break;
      }
    } else {
      if(*i) {
        
        if(*o) {
	  
	  msgptr[0] = messages[1];
	  msgptr[1] = str_spc;
	  msgptr[2] = str_bropen;
	  msgptr[3] = p1;
	  msgptr[4] = str_brclose;
	  dkapp_log_msg(app, DK_LOG_LEVEL_ERROR, msgptr, 5);
	  mode |= OPTION_ERROR;
	} else {
	  
	  *o = p1;
	}
      } else {
        
        *i = p1;
      }
    }
    lfd++; lfdptr++;
    
  }
  
  
  
  
  
  
  
  
  
  
  
  *m = mode; 
}



/**	Set boolean preference.
	@param	pk	Preference key.
	@param	v	New value.
*/
static void
set_bool DK_P2(char *,pk,int,v)
{
  
  dkapp_set_pref(app,pk,(v ? (str_yes) : (str_no)));
}



/**	Help text.
*/
static char *help_text[] = {
(char *)"The trana program can be used to analyze trace files created by",
(char *)"tracecc.",
(char *)"",
(char *)"Usage:",
(char *)"------",
(char *)"trana [-m] [-p] [-t] [-n] [ <inputfile> [ <outputfile> ] ]",
(char *)"trana -c [-m] [-p] [-t] [-n]",
(char *)"trana -u",
(char *)"",
(char *)"Inputfile specifies the name of the input file to read, outputfile",
(char *)"is the name of the output file to create. Standard input and output",
(char *)"are used for unspecified filenames.",
(char *)"-m --marker=<bool>       create markers for folding in vim",
(char *)"-p --position=<bool>     transfer source position to output file",
(char *)"-t --trace-only=<bool>   transfer only lines containing the trace keyword",
(char *)"-n --numeric=<bool>      Prepend all lines by indent level",
(char *)"-c --configure           keep configuration as default configuration",
(char *)"                   NOTE: Command line options and default configu-",
(char *)"                         ration are merged. Use -u to reset all",
(char *)"                         options.",
(char *)"-u --unconfigure         remove default configuration",
NULL
};



/**	License terms.
*/
static char *license_terms[] = {
"Redistribution and use in source and binary forms, with or without",
"modification, are permitted provided that the following conditions are met:",
"* Redistributions of source code must retain the above copyright notice, this",
"  list of conditions and the following disclaimer.",
"* Redistributions in binary form must reproduce the above copyright notice,",
"  this list of conditions and the following disclaimer in the documentation",
"  and/or other materials provided with the distribution.",
"* Neither the name of the Dirk Krause nor the names of other contributors may",
"  be used to endorse or promote products derived from this software without",
"  specific prior written permission.",
"THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS \"AS IS\"",
"AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE",
"IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE",
"ARE DISCLAIMED.",
"IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY",
"DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES",
"(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;",
"LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND",
"ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT",
"(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS",
"SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.",
NULL
};



/**	The real main function.
	@param	argc	Number of command line arguments.
	@param	argv	Command line arguments array.
	@return	1 on success, 0 on error.
*/
static int 
run_main DK_P2(int,argc,char **,argv)
{
  int back = 0;
  int mode = OPTION_POSITION;
  char *inputfilename, *outputfilename, **ptr;
  
  dkapp_find_multi(app, trana_msg, string_table_name );
  inputfilename = outputfilename = NULL;
  mode = OPTION_POSITION;
  mode = get_default_mode();
  process_arguments(argc, argv, &mode, &inputfilename, &outputfilename);
  if(mode & OPTION_ERROR) {
    mode |= OPTION_HELP;
  }
  if(mode & (OPTION_HELP | OPTION_VERSION)) {
    char **cptr;
    if(!(mode & OPTION_ERROR)) { back = 1; }
    printf(
      "\n%s (part of the dklibs collection, version %s)\n",
      programname, the_version_number
    );
    printf("Tracecc output analyzer\n");
    printf("Copyright (C) 2002-2010 Dipl.-Ing. D. Krause\n");
    printf("http://dktools.sourceforge.net/tracecc.html\n\n");
    cptr = license_terms;
    while(*cptr) {
      printf("%s\n", *(cptr++));
    }
    printf("\nLibraries used:\n\n");
    ptr = dklic_get();
    while(*ptr) {
      printf("%s\n", *(ptr++));
    }
    printf("\n");
    if(mode & OPTION_HELP) {
      ptr = help_text;
      while(*ptr) {
        printf("%s\n", *(ptr++));
      }
    }
  } else {
    if(mode & (OPTION_SHOWCONF | OPTION_CONFIGURE | OPTION_UNCONF)) {
      back = 1;
      if(mode & OPTION_UNCONF) {
	dkapp_unconfigure(app);
      } else {
        if(mode & OPTION_SHOWCONF) {
	  dkapp_stdout(app, messages[2]);
	  fputc('\n', stdout);
	  dkapp_stdout(app, "-m");
	  fputs(three_spaces, stdout);
	  dkapp_stdout(app, messages[(mode & OPTION_MARKER) ? 7 : 8]);
	  fputs(three_spaces, stdout);
	  dkapp_stdout(app, messages[3]);
	  fputc('\n', stdout);
	  dkapp_stdout(app, "-p");
	  fputs(three_spaces, stdout);
	  dkapp_stdout(app, messages[(mode & OPTION_POSITION) ? 7 : 8]);
	  fputs(three_spaces, stdout);
	  dkapp_stdout(app, messages[4]);
	  fputc('\n', stdout);
	  dkapp_stdout(app, "-t");
	  fputs(three_spaces, stdout);
	  dkapp_stdout(app, messages[(mode & OPTION_TRACEONLY) ? 7 : 8]);
	  fputs(three_spaces, stdout);
	  dkapp_stdout(app, messages[5]);
	  fputc('\n', stdout);
	  dkapp_stdout(app, "-n");
	  fputs(three_spaces, stdout);
	  dkapp_stdout(app, messages[(mode & OPTION_NUMERIC) ? 7 : 8]);
	  fputs(three_spaces, stdout);
	  dkapp_stdout(app, messages[6]);
	  fputc('\n', stdout);
	} else {
	  set_bool(pk_marker, (mode & OPTION_MARKER));
	  set_bool(pk_position, (mode & OPTION_POSITION));
	  set_bool(pk_traceonly, (mode & OPTION_TRACEONLY));
	  set_bool(pk_numeric, (mode & OPTION_NUMERIC));
	}
      }
    } else {
      
      if(inputfilename) {
        if(outputfilename) {
	  back = run_for_names(inputfilename, outputfilename, mode);
	} else {
	  back = run_for_input(inputfilename,mode,stdout);
	}
      } else {
        run_for_files(stdin,stdout,mode);
	back = 1;
      }
    }
  } 
  return back;
}



/**	Check whether or not to run silently.
	@param	argc	Number of command line arguments.
	@param	argv	Command line arguments array.
	@param	rs	Pointer to result variable (silently).
	@param	rf	Pointer to result variable (filter).
*/
static void
silence_check DK_P4(int,argc,char **,argv,int *,rs,int *,rf)
{
  int i, filenames;
  char *ptr, **lfdptr;
  
  filenames = 0; lfdptr = argv;
  lfdptr++; i = 1;
  while(i < argc) {
    ptr = *lfdptr;
    if(*ptr == '-') {
      ptr++;
      if(*ptr == 's') {
        *rs = 1;
      }
    } else {
      filenames++;
    }
    i++; lfdptr++;
  }
  if(filenames < 2) {
    *rf = 1;
  }
  
}



/**	The main() function of the trana program.
	We only create the dk_app_t variable here and leave all
	other processing up to the run_main() function.
	@param	argc	Number of command line arguments.
	@param	argv	Command line arguments array.
	@return	0 on success, any other value indicates an error.
*/
#if DK_HAVE_PROTOTYPES
int main(int argc, char *argv[])
#else
int main(argc, argv) int argc; char *argv[];
#endif
{
  int exval;
  int rs, rf;
  
#line 947 "trana.ctr"

  
  exval = rs = rf = 0;
  silence_check(argc, argv, &rs, &rf);
  app = dkapp_open_ext1(argc, argv, packagename, sysconfdir, rs, rf);
  if(app) {
    
    exval = run_main(dkapp_get_argc(app), dkapp_get_argv(app));
    dkapp_close(app);
  } else {
    
    if(!rs) {
      fputs("ERROR: Not enough memory!\n", stderr);
      fflush(stdout);
    }
  }
  exval = (exval ? 0 : 1);
  
  
#line 965 "trana.ctr"

  exit(exval); return exval;
}


