/*
jlayout - A Java code generator for GUI layout
Copyright (c) 2007-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	jlread.c	The jlread module of the jlayout program.
*/



/**	Inside the jlread module.
*/
#define	JLREAD_C	1

#include	"jl.h"




#line 54 "jlread.ctr"




/** File open mode.  */
static char str_r[] = { "r" };


/** Control instructions. */
static char *control_instructions[] = {
  /*   0 */	"default",
  /*   1 */	"option",
  NULL
};


/** Close operations. */
static char *close_operations[] = {
  "nothing", "hide", "dispose", "exit", NULL
};



/** Border sub types. */
static char *border_sub_types[] = {
  "lowered", "raised", NULL
};


/** Border types. */
static char *border_types[] = {
/*  0 */	"empty",
/*  1 */	"etched",
/*  2 */	"bevel",
/*  3 */	"line",
/*  4 */	"matte",
/*  5 */	"titled",
NULL
};

/** Horizontal alignment types. */
static char *h_align_types[] = {
	"leading",
	"left",
	"center",
	"right",
	"trailing",
	NULL
};


/** Vertical alignment types. */
static char *v_align_types[] = {
	"top",
	"center",
	"bottom",
	NULL
};

/** String: BorderFactory. */
static char str_borderfactory[] = { "BorderFactory" };

/** String: WindowConstants. */
static char str_windowconstants[] = { "WindowConstants" };

/** String: enabled. */
static char str_enabled[] = { "enabled" };

/** String: default. */
static char str_default[] = { "default" };

/** String: close. */
static char str_close[] = { "close" };

/** String: operation. */
static char str_operation[] = { "operation" };

/** String: title. */
static char str_title[] = { "title" };

/** String: transfer. */
static char str_transfer[] = { "transfer" };

/** String: handler. */
static char str_handler[] = { "handler" };

/** String: horizontal. */
static char str_horizontal[] = { "horizontal" };

/** String: vertical. */
static char str_vertical[] = { "vertical" };

/** String: alignment. */
static char str_alignment[] = { "alignment" };

/** String: minimum. */
static char str_minimum[] = { "minimum" };

/** String: preferred. */
static char str_preferred[] = { "preferred" };

/** String: maximum. */
static char str_maximum[] = { "maximum" };

/** String: glue. */
static char str_glue[] = { "glue" };

/** String: separator. */
static char str_separator[] = { "separator" };

/** String: use. */
static char str_use[] = { "use" };

/** String: content. */
static char str_content[] = { "content" };

/** String: pane. */
static char str_pane[] = { "pane" };

/** String: contents. */
static char str_contents[] = { "contents" };

/** String: border. */
static char str_border[] = { "border" };

/** String: fill. */
static char str_fill[] = { "fill" };

/** String: anchor. */
static char str_anchor[] = { "anchor" };

/** String: action. */
static char str_action[] = { "action" };

/** String: command. */
static char str_command[] = { "command" };

/** String: window. */
static char str_window[] = { "window" };

/** String: item. */
static char str_item[] = { "item" };

/** String: listener. */
static char str_listener[] = { "listener" };

/** String: layout. */
static char str_layout[] = { "layout" };

/** String: text. */
static char str_text[] = { "text" };

/** String: tool. */
static char str_tool[] = { "tool" };

/** String: tip. */
static char str_tip[] = { "tip" };

/** String: as. */
static char str_as[] = { "as" };

/** String: accessible. */
static char str_accessible[] = { "accessible" };

/** String: description. */
static char str_description[] = { "description" };

/** String: constructor. */
static char str_constructor[] = { "constructor" };

/** String: width. */
static char str_width[] = { "width" };

/** String: size. */
static char str_size[] = { "size" };

/** String: button. */
static char str_button[] = { "button" };

/** String: group. */
static char str_group[] = { "group" };

/** String: caret. */
static char str_caret[] = { "caret" };

/** String: change. */
static char str_change[] = { "change" };

/** String: component. */
static char str_component[] = { "component"};

/** String: container. */
static char str_container[] = { "container" };

/** String: document. */
static char str_document[] = { "document" };

/** String: focus. */
static char str_focus[] = { "focus" };

/** String: internal. */
static char str_internal[] = { "internal" };

/** String: frame. */
static char str_frame[] = { "frame" };

/** String: key. */
static char str_key[] = { "key" };

/** String: data. */
static char str_data[] = { "data" };

/** String: list. */
static char str_list[] = { "list" };

/** String: selection. */
static char str_selection[] = { "selection" };

/** String: mouse. */
static char str_mouse[] = { "mouse" };

/** String: motion. */
static char str_motion[] = { "motion" };

/** String: wheel. */
static char str_wheel[] = { "wheel" };

/** String: property. */
static char str_property[] = { "property" };

/** String: table. */
static char str_table[] = { "table" };

/** String: model. */
static char str_model[] = { "model" };

/** String: tree. */
static char str_tree[] = { "tree" };

/** String: expansion. */
static char str_expansion[] = { "expansion" };

/** String: expand. */
static char str_expand[] = { "expand" };

/** String: will. */
static char str_will[] = { "will" };

/** String: undoable. */
static char str_undoable[] = { "undoable" };

/** String: edit. */
static char str_edit[] = { "edit" };

/** String: icon. */
static char str_icon[] = { "icon" };

/** String: color. */
static char str_color[] = { "color" };

/** String: foreground. */
static char str_foreground[] = { "foreground" };

/** String: background. */
static char str_background[] = { "background" };

/** String: font. */
static char str_font[] = { "font" };

/** String: style. */
static char str_style[] = { "style" };

/** String: file. */
static char str_file[] = { "file" };

/** String: name. */
static char str_name[] = { "name" };

/** String: menu. */
static char str_menu[] = { "menu" };

/** String: bar. */
static char str_bar[] = { "bar" };

/** String: weight. */
static char str_weight[] = { "weight" };

/** String: EtchedBorder. */
static char str_etched_border[] = { "EtchedBorder" };

/** String: TitledBorder. */
static char str_titled_border[] = { "TitledBorder" };

/** String: SwingConstants. */
static char str_swing_constants[] = { "SwingConstants" };

/** String: Font. */
static char str_class_font[] = { "Font" };

/** String: Color. */
static char str_class_color[] = { "Color" };

/** String: Box. */
static char str_class_box[] = { "Box" };

/** String: position. */
static char str_position[] = { "position" };

/** Command: enabled. */
static char *cmd_enabled[] = { str_enabled, NULL };

/** Command: default close operation. */
static char *cmd_default_close_operation[] = {
  str_default, str_close, str_operation, NULL
};

/** Command: border. */
static char *cmd_border[] = { str_border, NULL };

/** Command: fill. */
static char *cmd_fill[] = { str_fill, NULL };

/** Command: anchor. */
static char *cmd_anchor[] = { str_anchor, NULL };

/**	Event listener class names.
*/
static char *listener_classes[] = {
/*  0 */	"ActionListener",
/*  1 */	"WindowListener",
/*  2 */	"ItemListener",
/*  3 */	"CaretListener",
/*  4 */	"ChangeListener",
/*  5 */	"ComponentListener",
/*  6 */	"ContainerListener",
/*  7 */	"DocumentListener",
/*  8 */	"FocusListener",
/*  9 */	"InternalFrameListener",
/* 10 */	"KeyListener",
/* 11 */	"ListDataListener",
/* 12 */	"ListSelectionListener",
/* 13 */	"MouseListener",
/* 14 */	"MouseMotionListener",
/* 15 */	"MouseWheelListener",
/* 16 */	"PropertyChangeListener",
/* 17 */	"TableModelListener",
/* 18 */	"TreeExpansionListener",
/* 19 */	"TreeModelListener",
/* 20 */	"TreeSelectionListener",
/* 21 */	"TreeWillExpandListener",
/* 22 */	"UndoableEditListener",
/* 23 */	"TransferHandlerListener",
NULL
};
/** Maximum index in listener_classes array. */
#define MAX_LISTENER_INDEX	23

/** Class name: ButtonGroup. */
static char str_button_group[] = { "ButtonGroup" };

/** Command: title. */
static char *cmd_title[] = {
  str_title, NULL
};

/** Command: contents border. */
static char *cmd_contents_border[] = {
  str_contents, str_border, NULL
};

/** Command: contents fill. */
static char *cmd_contents_fill[] = {
  str_contents, str_fill, NULL
};

/** Command: contents anchor. */
static char *cmd_contents_anchor[] = {
  str_contents, str_anchor, NULL
};

/** Command: action listener. */
static char *cmd_action_listener[] = {
  str_action, str_listener, NULL
};

/** Command: caret listener. */
static char *cmd_caret_listener[] = {
  str_caret, str_listener, NULL
};

/** Command: change listener. */
static char *cmd_change_listener[] = {
  str_change, str_listener, NULL
};

/** Command: component listener. */
static char *cmd_component_listener[] = {
  str_component, str_listener, NULL
};

/** Command: container listener. */
static char *cmd_container_listener[] = {
  str_container, str_listener, NULL
};

/** Command: document listener. */
static char *cmd_document_listener[] = {
  str_document, str_listener, NULL
};

/** Command: focus listener. */
static char *cmd_focus_listener[] = {
  str_focus, str_listener, NULL
};

/** Command: internal frame listener. */
static char *cmd_internal_frame_listener[] = {
  str_internal, str_frame, str_listener, NULL
};

/** Command: key listener. */
static char *cmd_key_listener[] = {
  str_key, str_listener, NULL
};

/** Command: list data listener. */
static char *cmd_list_data_listener[] = {
  str_list, str_data, str_listener, NULL
};

/** Command: list selection listener. */
static char *cmd_list_selection_listener[] = {
  str_list, str_selection, str_listener, NULL
};

/** Command: mouse listener. */
static char *cmd_mouse_listener[] = {
  str_mouse, str_listener, NULL
};

/** Command: mouse motion listener. */
static char *cmd_mouse_motion_listener[] = {
  str_mouse, str_motion, str_listener, NULL
};

/** Command: mouse wheel listener. */
static char *cmd_mouse_wheel_listener[] = {
  str_mouse, str_wheel, str_listener, NULL
};

/** Command: property change listener. */
static char *cmd_property_change_listener[] = {
  str_property, str_change, str_listener, NULL
};

/** Command: table model listener. */
static char *cmd_table_model_listener[] = {
  str_table, str_model, str_listener, NULL
};

/** Command: tree expansion listener. */
static char *cmd_tree_expansion_listener[] = {
  str_tree, str_expansion, str_listener, NULL
};

/** Command: tree model listener. */
static char *cmd_tree_model_listener[] = {
  str_tree, str_model, str_listener, NULL
};

/** Command: tree selection listener. */
static char *cmd_tree_selection_listener[] = {
  str_tree, str_selection, str_listener, NULL
};

/** Command: tree will expand listener. */
static char *cmd_tree_will_expand_listener[] = {
  str_tree, str_will, str_expand, str_listener, NULL
};

/** Command: undoable edit listener. */
static char *cmd_undoable_edit_listener[] = {
  str_undoable, str_edit, str_listener, NULL
};

/** Command: item listener. */
static char *cmd_item_listener[] = {
  str_item, str_listener, NULL
};

/** Command: window listener. */
static char *cmd_window_listener[] = {
  str_window, str_listener, NULL
};

/** Command: layout. */
static char *cmd_layout[] = {
  str_layout, NULL
};

/** Command: contents. */
static char *cmd_contents[] = {
  str_contents, NULL
};

/** Command: text. */
static char *cmd_text[] = {
  str_text, NULL
};

/** Command: constructor. */
static char *cmd_constructor[] = {
  str_constructor, NULL
};

/** Command: width. */
static char *cmd_width[] = {
  str_width, NULL
};

/** Command: size group. */
static char *cmd_size_group[] = {
  str_size, str_group, NULL
};

/** Command: button group. */
static char *cmd_button_group[] = {
  str_button, str_group, NULL
};

/** Command: icon file name. */
static char *cmd_icon_file_name[] = {
  str_icon, str_file, str_name, NULL
};

/** Command: component border. */
static char *cmd_component_border[] = {
  str_component, str_border, NULL
};

/** Command: icon color. */
static char *cmd_icon_color[] = {
  str_icon, str_color, NULL
};

/** Command: use content pane. */
static char *cmd_use_content_pane[] = {
  str_use, str_content, str_pane, NULL
};

/** Command: menu bar. */
static char *cmd_menubar[] = {
  str_menu, str_bar, NULL
};

/** Command: specials. */
static char *cmd_specials[] = {
	/*  0 */	str_glue,
	/*  1 */	str_separator,
	NULL
};

/** Command: weight. */
static char *cmd_weight[] = {
  str_weight, NULL
};

/** Command: border weight. */
static char *cmd_border_weight[] = {
  str_border, str_weight, NULL
};

/** Command: minimum size. */
static char *cmd_minimum_size[] = {
  str_minimum, str_size, NULL
};

/** Command: preferred size. */
static char *cmd_preferred_size[] = {
  str_preferred, str_size, NULL
};

/** Command: maximum size. */
static char *cmd_maximum_size[] = {
  str_maximum, str_size, NULL
};

/** Command: tool tip text. */
static char *cmd_tool_tip_text[] = {
  str_tool, str_tip, str_text, NULL
};

/** Command: accessible description. */
static char *cmd_accessible_description[] = {
  str_accessible, str_description, NULL
};

/** Command: use tool tip text as accessible description. */
static char *cmd_use_tool_tip_text_as_accessible_description[] = {
  str_use, str_tool, str_tip, str_text, str_as, str_accessible,
  str_description,
  NULL
};

/** Command: action command. */
static char *cmd_action_command[] = {
  str_action, str_command, NULL
};

/** Command: horizontal text alignment. */
static char *cmd_horizontal_alignment[] = {
  str_horizontal, str_alignment, NULL
};

/** Command: horizontal text position. */
static char *cmd_horizontal_text_position[] = {
  str_horizontal, str_text, str_position, NULL
};

/** Command: vertical text position. */
static char *cmd_vertical_text_position[] = {
  str_vertical, str_text, str_position, NULL
};

/** Command: vertical alignment. */
static char *cmd_vertical_alignment[] = {
  str_vertical, str_alignment, NULL
};

/** Command: transfer handler. */
static char *cmd_transfer_handler[] = {
  str_transfer, str_handler, NULL
};

/** Command: font. */
static char *cmd_font[] = {
  str_font, NULL
};

/** Command: font style. */
static char *cmd_font_style[] = {
  str_font, str_style, NULL
};

/** Command : foreground. */
static char *cmd_foreground[] = {
  str_foreground, NULL
};

/** Command: background. */
static char *cmd_background[] = {
  str_background, NULL
};



/**	Commands for default lines.
*/
static char **cmds_for_defaults[] = {
  /*   0 */	cmd_border,	/* border */
  /*   1 */	cmd_fill,	/* fill */
  /*   2 */	cmd_anchor,	/* anchor */
  NULL
};



/**	Option commands.
*/
static char **cmds_for_options[] = {
  /*   0 */	cmd_use_tool_tip_text_as_accessible_description,
  NULL
};



/**	Object attribute names.
*/
static char **cmd_for_objects[] = {
  /*   0 */	cmd_border,	/* border */
  /*   1 */	cmd_fill,	/* fill */
  /*   2 */	cmd_anchor,	/* anchor */
  /*   3 */	cmd_contents_border,
  /*   4 */	cmd_contents_fill,
  /*   5 */	cmd_contents_anchor,
  /*   6 */	cmd_action_listener,
  /*   7 */	cmd_item_listener,
  /*   8 */	cmd_window_listener,
  /*   9 */	cmd_layout,
  /*  10 */	cmd_contents,
  /*  11 */	cmd_text,
  /*  12 */	cmd_constructor,
  /*  13 */	cmd_width,
  /*  14 */	cmd_size_group,
  /*  15 */	cmd_caret_listener,
  /*  16 */	cmd_change_listener,
  /*  17 */	cmd_component_listener,
  /*  18 */	cmd_container_listener,
  /*  19 */	cmd_document_listener,
  /*  20 */	cmd_focus_listener,
  /*  21 */	cmd_internal_frame_listener,
  /*  22 */	cmd_key_listener,
  /*  23 */	cmd_list_data_listener,
  /*  24 */	cmd_list_selection_listener,
  /*  25 */	cmd_mouse_listener,
  /*  26 */	cmd_mouse_motion_listener,
  /*  27 */	cmd_mouse_wheel_listener,
  /*  28 */	cmd_property_change_listener,
  /*  29 */	cmd_table_model_listener,
  /*  30 */	cmd_tree_expansion_listener,
  /*  31 */	cmd_tree_model_listener,
  /*  32 */	cmd_tree_selection_listener,
  /*  33 */	cmd_tree_will_expand_listener,
  /*  34 */	cmd_undoable_edit_listener,
  /*  35 */	cmd_icon_file_name,
  /*  36 */	cmd_button_group,
  /*  37 */	cmd_component_border,
  /*  38 */	cmd_use_content_pane,
  /*  39 */	cmd_menubar,
  /*  40 */	cmd_weight,
  /*  41 */	cmd_minimum_size,
  /*  42 */	cmd_preferred_size,
  /*  43 */	cmd_maximum_size,
  /*  44 */	cmd_tool_tip_text,
  /*  45 */	cmd_accessible_description,
  /*  46 */	cmd_action_command,
  /*  47 */	cmd_horizontal_alignment,
  /*  48 */	cmd_vertical_alignment,
  /*  49 */	cmd_transfer_handler,
  /*  50 */	cmd_title,
  /*  51 */	cmd_default_close_operation,
  /*  52 */	cmd_enabled,
  /*  53 */	cmd_font,
  /*  54 */	cmd_foreground,
  /*  55 */	cmd_background,
  /*  56 */	cmd_font_style,
  /*  57 */	cmd_border_weight,
  /*  58 */	cmd_horizontal_text_position,
  /*  59 */	cmd_vertical_text_position,
  NULL
};



/**	Horizontal text positions.
*/
static char *horizontal_text_positions[] = {
  "LEFT", "CENTER", "RIGHT", "LEADING", "TRAILING", NULL
};



/**	Vertical text positions.
*/
static char *vertical_text_positions[] = {
  "TOP", "CENTER", "BOTTOM", NULL
};



/**	Fill values.
*/
static char *fill_values[] = {
  "NONE", "HORIZONTAL", "VERTICAL", "BOTH",
  NULL
};



/**	Anchor setting names.
*/
static char *anchor_values[] = {
  "CENTER",
  "NORTH",
  "NORTHEAST",
  "EAST",
  "SOUTHEAST",
  "SOUTH",
  "SOUTHWEST",
  "WEST",
  "NORTHWEST",
  "PAGE_START",
  "PAGE_END",
  "LINE_START",
  "LINE_END",
  "FIRST_LINE_START",
  "FIRST_LINE_END",
  "LAST_LINE_START",
  "LAST_LINE_END",
  "BASELINE",
  "BASELINE_LEADING",
  "BASELINE_TRAILING",
  "ABOVE_BASELINE",
  "ABOVE_BASELINE_LEADING",
  "ABOVE_BASELINE_TRAILING",
  "BELOW_BASELINE",
  "BELOW_BASELINE_LEADING",
  "BELOW_BASELINE_TRAILING",
  NULL
};


/**	Layout names.
*/
static char *layout_values[] = {
  "NONE",
  "GridBag",
  "Card",
  "Border",
  "Box.X_AXIS",
  "Box.Y_AXIS",
  "Box.PAGE_AXIS",
  "Box.LINE_AXIS",
  "Flow",
  "Grid",
  "Group",
  "Spring",
  NULL
};


/**	Class names.
*/
static char *layout_classes[] = {
/*  0 */	"GridBagLayout",
/*  1 */	"CardLayout",
/*  2 */	"BorderLayout",
/*  3 */	"BoxLayout",
/*  4 */	"FlowLayout",
/*  5 */	"GridLayout",
/*  6 */	"GroupLayout",
/*  7 */	"SpringLayout",
NULL
};


/** Keyowrd: GridBagConstraints. */
static char str_gridbagconstraints[] = { "GridBagConstraints" };

/** Keyword: ImageIcon. */
static char str_imageicon[] = { "ImageIcon" };

/** Keyword: JComponent. */
static char str_jcomponent[] = { "JComponent" };

/** Keyword: Container. */
static char str_Container[] = { "Container" };



/**	Compare two classes by name.
	@param	l	Left class.
	@param	r	Right class.
	@param	cr	Comparison criteria (ignored).
	@return	Comparison result.
*/
static
int
compare_classes_by_name DK_P3(void *,l, void *,r, int,cr)
{
  int back = 0;
  char *pl, *pr;
  pl = (char *)l; pr = (char *)r;
  if(l) {
    if(r) {
      back = strcmp(pl, pr);
    } else { back = 1; }
  } else {
    if(r) { back = -1; }
  }
  return back;
}



/**	Compare two content nodes by line number of appearance.
	@param	l	Left node.
	@param	r	Right node.
	@param	cr	Comparison criteria (ignored).
	@return	Comparison result.
*/
static
int
compare_contents_nodes DK_P3(void *,l, void *,r, int,cr)
{
  int back = 0;
  JCN *pl = NULL, *pr = NULL;
  pl = (JCN *)l; pr = (JCN *)r;
  if(pl) {
    if(pr) {
      if(pl->n < pr->n) {
        back = -1;
      } else {
        if(pl->n > pr->n) {
	  back = 1;
	}
      }
    } else { back = 1; }
  } else { if(pr) { back = -1; } }
  return back;
}



/**	Compare two listeners by number.
	@param	l	Left event listener.
	@param	r	Right event listener.
	@param	cr	Comparison criteria (ignored).
	@return	Comparison result.
*/
static
int
compare_listeners DK_P3(void *,l, void *,r, int,cr)
{
  int back = 0;
  JLISTENER *pl = NULL, *pr = NULL;
  
  pl = (JLISTENER *)l; pr = (JLISTENER *)r;
  if(l) {
    if(r) {
      if(pl->number > pr->number) {
        back = 1;
      } else {
        if(pl->number < pr->number) {
	  back = -1;
	}
      }
    } else { back = 1; }
  } else {
    if(r) { back = -1; }
  }
  
  return back;
}



/**	Compare event listeners, return inverted result.
	@param	l	Left listener.
	@param	r	Right listener.
	@param	cr	Comparison criteria (ignored).
	@return	Inverted comparison result.
*/
static
int
compare_listeners_inverted DK_P3(void *,l, void *,r, int,cr)
{
  int back = 0;
  JLISTENER *pl = NULL, *pr = NULL;
  
  pl = (JLISTENER *)l; pr = (JLISTENER *)r;
  if(l) {
    if(r) {
      if(pl->number > pr->number) {
        back = -1;
      } else {
        if(pl->number < pr->number) {
	  back = 1;
	}
      }
    } else { back = 1; }
  } else {
    if(r) { back = -1; }
  }
  
  return back;
}


/**	Compare two size groups by name.
	@param	l	Left size group name.
	@param	r	Right size group name.
	@param	cr	Comparison criteria (ignored).
	@return	Comparison result.
*/
static
int
jlread_compare_groups_by_name DK_P3(void *,l, void *,r, int,cr)
{
  int back = 0;
  JSG *pl, *pr;
  
  pl = (JSG *)l;
  switch(cr) {
    case CR_COMPARE_OBJECT_AND_NAME: {
      if(l) {
        if(r) {
	  if(pl->n) {
	    back = strcmp(pl->n, (char *)r);
	  } else { back = -1; }
	} else { back = 1; }
      } else { if(r) { back = -1; } }
    } break;
    default: {
      pr = (JSG *)r;
      if(l) {
        if(r) {
	  if(pl->n) {
	    if(pr->n) {
	      back = strcmp(pl->n, pr->n);
	    } else { back = 1; }
	  } else {
	    if(pr->n) { back = -1; }
	  }
	} else { back = 1; }
      } else { if(r) { back = -1; } }
    } break;
  }
  
  return back;
}



int
jlread_compare_objects_by_name DK_P3(void *,l, void *,r, int,cr)
{
  int back = 0;
  JLO *pl, *pr;
  
  pl = (JLO *)l;
  switch(cr) {
    case CR_COMPARE_OBJECT_AND_NAME: {
      if(l) {
        if(r) {
	  if(pl->on) {
	    back = strcmp(pl->on, (char *)r);
	  } else { back = -1; }
	} else { back = 1; }
      } else { if(r) { back = -1; } }
    } break;
    default: {
      pr = (JLO *)r;
      if(l) {
        if(r) {
	  if(pl->on) {
	    if(pr->on) {
	      back = strcmp(pl->on, pr->on);
	    } else { back = 1; }
	  } else {
	    if(pr->on) { back = -1; }
	  }
	} else { back = 1; }
      } else { if(r) { back = -1; } }
    } break;
  }
  
  return back;
}



void
jlread_cn_delete DK_P1(JCN *,j)
{
  
  if(j) {	
    j->n = 0UL; j->o = NULL; j->t = 0;
    dk_delete(j);
  } 
}


JCN *
jlread_cn_new DK_P2(unsigned long, n, JLO *,o)
{
  JCN *back = NULL;
  
  back = dk_new(JCN,1);
  if(back) {
    back->n = n; back->o = o; back->t = JCN_OBJECT;
  } 
  return back;
}



/**	Set default border.
	@param	j	Jlayout job.
	@param	v	Default border setting.
	@return	1 on success, 0 on error.
*/
static
int
default_border DK_P2(JLJ *,j, char *,v)
{
  int back = 0; char *x;
  
  if(j->def_border) {
    x = j->def_border; dk_delete(x); j->def_border = NULL;
  }
  if(v) {
    j->def_border = dkstr_dup(v);
    if(j->def_border) {
      back = 1;
    } else {
      /* ERROR: Memory */
      jlmsg_error_memory(j);
    }
  }
  
  return back;
}



/**	Set default filling.
	@param	j	Jlayout job.
	@param	v	Anchor setting.
	@return	1 on success, 0 on error,
*/
static
int
default_fill DK_P2(JLJ *,j, char *,v)
{
  int back = 0; int i;
  
  j->def_fill = FILL_NOT_SPECIFIED;
  if(v) {
    i = dkstr_array_index(fill_values, v, 0);
    if(i > -1) {
      j->def_fill = i; back = 1;
    } else {
      /* ERROR: Fill not found */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 21);
    }
  }
  
  return back;
}



/**	Set default anchor.
	@param	j	Jlayout job.
	@param	v	Anchor setting.
	@return	1 on success, 0 on error.
*/
static
int
default_anchor DK_P2(JLJ *,j, char *,v)
{
  int back = 0; int i;
  
  j->def_anchor = ANCHOR_NOT_SPECIFIED;
  if(v) {
    i = dkstr_array_index(anchor_values, v, 0);
    if(i > -1) {
      j->def_anchor = i; back = 1;
    } else {
      /* ERROR: Anchor not found */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 22);
    }
  }
  
  return back;
}



/**	Apply an option line.
	@param	j	Jlayout job.
	@param	l	Option line.
*/
static
int
apply_option DK_P2(JLJ *,j, char *,l)
{
  int back = 0;
  char *p1 = NULL, *p2 = NULL, *parts[16];
  size_t nparts; int act;
  
  p1 = l;
  p2 = dkstr_chr(p1, '=');
  if(p2) {
    *(p2++) = '\0';
    p2 = dkstr_start(p2, NULL);
    dkstr_chomp(p1, NULL);
    nparts = dkstr_explode(parts, 15, p1, NULL);
    if(nparts > 0) {
      act = dkstr_find_multi_part_abbr(parts, cmds_for_options, '$', 0);
      switch(act) {
        case 0: {	
	  if(p2) {
	    if(dkstr_is_bool(p2)) {
	      if(dkstr_is_on(p2)) {
	        j->f1 |= F1_USE_TTT_AS_AD;
	      } else {
	        j->f1 &= (~(F1_USE_TTT_AS_AD));
	      }
	      back = 1;
	    } else {
	      /* ERROR: Not a boolean */
	      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 55);
	    }
	  } else {
	    j->f1 |= F1_USE_TTT_AS_AD;
	    back = 1;
	  }
	} break;
      }
    } else {
      /*  ERROR: Empty key */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 23);
    }
  } else {
    /* ERROR: Not a key/value pair */
    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 24);
  }
  
  return back;
}



/**	Register a used class.
	@param	j	Jlayout job.
	@param	n	Class name.
	@return	1 on success, 0 on error.
*/
static
int
register_used_class DK_P2(JLJ *,j, char *,n)
{
  int back = 0;
  char *x;
  if((j) && (n)) {
    if((j->uc_st) && (j->uc_it)) {
      x = dksto_it_find_like(j->uc_it, (void *)n, 0);
      if(x) {
        back = 1;
      } else {
        x = dkstr_dup(n);
	if(x) {
	  if(dksto_add(j->uc_st, (void *)x)) {
	    back = 1;
	  } else {
	    dk_delete(x);
	  }
	}
      }
    }
  }
  return back;
}



/**	Register default key/value pair.
	@param	j	Jlayout job.
	@param	l	Line containing the setting.
	@return	1 on success, 0 on error.
*/
static
int
apply_default DK_P2(JLJ *,j, char *,l)
{
  int back = 0;
  char *p1 = NULL, *p2 = NULL, *parts[16];
  size_t nparts; int act;
  
  p1 = l;
  p2 = dkstr_chr(p1, '=');
  if(p2) {
    *(p2++) = '\0';
    p2 = dkstr_start(p2, NULL);
    dkstr_chomp(p1, NULL);
    nparts = dkstr_explode(parts, 15, p1, NULL);
    if(nparts > 0) {
      act = dkstr_find_multi_part_abbr(parts, cmds_for_defaults, '$', 0);
      switch(act) {
        case 0: {	
	  back = default_border(j, p2);
	} break;
	case 1: {	
	  back = default_fill(j, p2);
	  if(back) {
	    back = register_used_class(j, str_gridbagconstraints);
	    if(!back) {
	      jlmsg_error_memory(j);
	    }
	  }
	} break;
	case 2: {	
	  back = default_anchor(j, p2);
	  if(back) {
	    back = register_used_class(j, str_gridbagconstraints);
	    if(!back) {
	      jlmsg_error_memory(j);
	    }
	  }
	} break;
      }
    } else {
      /*  ERROR: Empty key */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 23);
    }
  } else {
    /* ERROR: Not a key/value pair */
    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 24);
  }
  
  return back;
}



/**	Process a control instruction.
	@param	j	Jlayout job.
	@param	l	Line to process.
	@return	1 on success, 0 on error.
*/
static
int
apply_control_instruction DK_P2(JLJ *,j, char *,l)
{
  int back = 0;
  char *p1 = NULL, *p2 = NULL;
  
  p1 = l; p1++;
  p1 = dkstr_start(p1, NULL);
  if(p1) {		
    p2 = dkstr_next(p1, NULL);
    switch(dkstr_array_index(control_instructions, p1, 0)) {
      case 0: {
        if(p2) {	
	  back = apply_default(j, p2);
	} else {	
	  /* ERROR: Empty default line */
	  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 25);
	}
      } break;
      case 1: {		
        if(p2) {
	  back = apply_option(j, p2);
	} else {
	  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 53);
	}
      } break;
      default: {	
        /* ERROR: Unknown control instruction */
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 26);
      } break;
    }
  } else {		
    /* ERROR: Empty control instruction */
    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 27);
  }
  
  return back;
}



/**	Find object data structure for a given name.
	Create the object if necessary.
	@param	j	Jlayout job.
	@param	n	object name.
	@param	p	Parent object.
	@param	lcr	Line number of object creation.
	@return	Pointer to object on success, NULL on error.
*/
static
JLO *
find_object_for_name DK_P4(JLJ *,j, char *,n, JLO *,p, unsigned long, lcr)
{
  JLO *back = NULL;
  char *i_border; int i_fill, i_anchor;
  
  back = dksto_it_find_like(j->o_it, n, CR_COMPARE_OBJECT_AND_NAME);
  if(back) {
    if(p) {
      if(back->parent) {	
        /* ERROR: Multiple parents for object */
	jlmsg_log3(j, DK_LOG_LEVEL_ERROR, 10, n, 11);
	back = NULL;
      } else {
        back->parent = p;
      }
    }
  } else {
    back = jlo_new(n);
    if(back) {
      back->l_creation = lcr;
      back->parent = p;
      i_border = NULL;
      i_fill = FILL_NOT_SPECIFIED; i_anchor = ANCHOR_NOT_SPECIFIED;
      i_border = j->def_border;
      i_fill = j->def_fill;
      i_anchor = j->def_anchor;
      if(p) {
        if(p->c_border) { i_border = p->c_border; }
        if(p->c_fill != FILL_NOT_SPECIFIED) { i_fill = p->c_fill; }
        if(p->c_anchor != ANCHOR_NOT_SPECIFIED) { i_anchor = p->c_anchor; }
      } else {
        jlmsg_log3(j, DK_LOG_LEVEL_WARNING, 8, n, 9);
      }
      back->fill = i_fill; back->anchor = i_anchor;
      if(i_border) {
        back->border = dkstr_dup(i_border);
	if(!(back->border)) {
	  jlo_delete(back); back = NULL;
	}
      }
      if(back) {
        if(!dksto_add(j->o_st, (void *)back)) {
          jlo_delete(back); back = NULL;
	  /* ERROR: Memory */
	  jlmsg_error_memory(j);
        }
      }
    } else {
      /* ERROR: Memory */
      jlmsg_error_memory(j);
    }
  }
  
  return back;
}





/**	Process an object definition line
	@param	j	Jlayout job.
	@param	l	Text line containing the object definition.
	@return	1 on success, 0 on error.
*/
static
int
apply_object_definition DK_P2(JLJ *,j, char *,l)
{
  int back = 0;
  char *p1 = NULL, *p2 = NULL;
  
  p1 = l;
  p1++; p1 = dkstr_start(p1, NULL);
  if(p1) {
    p2 = dkstr_chr(p1, ']');
    if(p2) {
      *p2 = '\0';
      p2 = dkstr_next(p1, NULL);
      if(p2) {	
	if(register_used_class(j, p1)) {
          j->o_cur = find_object_for_name(j, p2, NULL, j->errlineno);
	  if(j->o_cur) {
	    if(jlo_get_classname(j->o_cur)) {		
	      /* ERROR: Object definition already exists */
	      jlmsg_log3(j, DK_LOG_LEVEL_ERROR, 6, (j->o_cur)->on, 7);
	    } else {					
	      if(jlo_set_classname(j->o_cur, p1)) {
	        back = 1;
	      } else {
	        /* ERROR: Memory */
	        jlmsg_error_memory(j);
	      }
	    }
	  } else {
	    /* Error already handled by find_object_for_name */
	  }
	} else {
	  jlmsg_error_memory(j);
	}
      } else {
        /* ERROR: No object name */
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 28);
      }
    } else {
      /* ERROR: No closing square bracket */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 29);
    }
  } else {
    /* ERROR: Empty text */
    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
  }
  
  return back;
}



/**	Add an event listener to an object.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	tp	Event type.
	@param	t	The listener to add.
	@return	1 on success, 0 on error.
*/
static
int
add_one_listener DK_P4(JLJ *,j, JLO *,o, int,tp, char *,t)
{
  int back = 0;
  JLISTENER *jl;
  
  if((j) && (o)) {		
    if(t) {			
      if(!(o->l_st)) {
        o->l_st = dksto_open(0);
        if(o->l_st) {
          dksto_set_comp(o->l_st,compare_listeners,0);
        }
      }
      if(!(o->l_it)) {
        if(o->l_st) {
          o->l_it = dksto_it_open(o->l_st);
        }
      }
      if(!(o->li_st)) {
        o->li_st = dksto_open(0);
        if(o->li_st) {
          dksto_set_comp(o->li_st,compare_listeners_inverted,0);
        }
      }
      if(!(o->li_it)) {
        if(o->li_st) {
          o->li_it = dksto_it_open(o->li_st);
        }
      }
      if((o->l_st) && (o->l_it) && (o->li_st) && (o->li_it)) {
        
        jl = jlo_listener_new(tp, t, o->n_listeners);
        o->n_listeners += 1UL;
        if(jl) {	
          if(dksto_add(o->l_st, (void *)jl)) {
	    
	    if(dksto_add(o->li_st, (void *)jl)) {
	      
	      if((tp >= 0) && (tp <= MAX_LISTENER_INDEX)) {
	        if(tp != LISTENER_TRANSFER_HANDLER) {
		  back = register_used_class(j, listener_classes[tp]);
		} else {
		  back = 1;
		}
	      } else {
	        back = 1;
	      }
	      
	    }
	  } else {
	    jlo_listener_delete(jl); jl = NULL;
	  }
        }
      }
    } else {
      /* ERROR: Empty listener */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 31);
    }
  } 
  return back;
}



/**	Apply border setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Border setting.
	@return	1 on success, 0 on error.
*/
static
int
apply_border DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  char *x;
  if((j) && (o)) {
    if(o->border) {
      x = o->border; dk_delete(x);
    } o->border = NULL;
    back = 1;
    if(v) {
      o->border = dkstr_dup(v);
      if(!(o->border)) { back = 0; }
    }
  }
  return back;
}



/**	Add one contents element to an object.
	@param	j	Jlayout job.
	@param	o	Current (parent) object.
	@param	c	Object to add (child).
	@param	n	Number (ID) of the child object.
	@return	1 on success, 0 on error.
*/
static
int
add_one_contents_element DK_P4(JLJ *,j, JLO *,o, JLO *,c, unsigned long, n)
{
  int back = 0; JCN *jcn = NULL;
  
  if(!(o->c_st)) {
    o->c_st = dksto_open(0);
    if(o->c_st) {
      dksto_set_comp(o->c_st, compare_contents_nodes, 0);
    }
  }
  if(!(o->c_it)) {
    if(o->c_st) {
      o->c_it = dksto_it_open(o->c_st);
    }
  }
  if((o->c_st) && (o->c_it)) {
    jcn = jlread_cn_new(n, c);
    if(jcn) {
      if(dksto_add(o->c_st, (void *)jcn)) {
        back = 1;
	c->parent = o;
      } else {
        jlread_cn_delete(jcn);
      }
    }
  }
  if(!back) {
    /* ERROR: Memory */
    jlmsg_error_memory(j);
  }
  
  return back;
}



/**	Get long number for position and/or size.
	The number can be specified in the text as absolute number
	or relative (``.'', ``+x'', ``-x'').
	@param	s	Text containing the number (absolute or relative).
	@param	r	Base for relative numbers.
	@return	The number.
*/
static
long
get_long_number DK_P2(char *,s, long,r)
{
  int back = -1L;
  long l;
  
  switch(*s) {
    case '.' : {
      back = r;
    } break;
    case '+' : {
      if(sscanf(&(s[1]), "%ld", &l) == 1) {
        back = r + l;
      }
    } break;
    case '-' : {
      if(sscanf(&(s[1]), "%ld", &l) == 1) {
        back = r - l;
      }
    } break;
    default: {
      if(sscanf(s, "%ld", &l) == 1) {
        back = l;
      }
    } break;
  } 
  return back;
}



/**	Apply fill setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Fill setting.
	@return	1 on success, 0 on error.
*/
static
int
apply_fill DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  if((j) && (o)) {
    o->fill = FILL_NOT_SPECIFIED; back = 1;
    if(v) {
      o->fill = dkstr_array_index(fill_values, v, 0);
      if(o->fill == -1) {
        /* ERROR: Fill value */
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 21);
	back = 0;
      }
    }
  }
  return back;
}



/**	Apply anchor setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Anchor setting.
	@return	1 on success, 0 on error.
*/
static
int
apply_anchor DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  if((j) && (o)) {
    o->anchor = ANCHOR_NOT_SPECIFIED; back = 1;
    if(v) {
      o->anchor = dkstr_array_index(anchor_values, v, 0);
      if(o->anchor == -1) {
        back = 0;
	/* ERROR: Illegal anchor value */
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 22);
      }
    }
  }
  return back;
}



/**	Add information how to add an object to the container.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	s	Further information for adding the object.
	@return	1 on success, 0 on error.
*/
static
int
add_adding_information DK_P3(JLJ *,j, JLO *,o, char *,s)
{
  int back = 0;
  char *newptr = NULL, *cx = NULL, *ptr = NULL, *ptr1 = NULL, *ptr2 = NULL;
  long x = 0L, y = 0L, w = 1L, h = 1L;
  int anchor, fill;
  
  if(o->parent) {		
    switch((o->parent)->layout) {
      case LAYOUT_GRIDBAG: {
        
        if(s) {
	  ptr1 = dkstr_start(s, NULL);
	  if(ptr1) {		
	    ptr2 = dkstr_next(ptr1, NULL);
	    if(ptr2) {		
	      back = 1;
	      anchor = ANCHOR_NOT_SPECIFIED;
	      fill = FILL_NOT_SPECIFIED;
	      ptr = dkstr_next(ptr2, NULL);
	      x = get_long_number(ptr1, (o->parent)->current_x);
	      
	      y = get_long_number(ptr2, (o->parent)->current_y);
	      
	      w = 1L; h = 1L;
	      if(ptr) {		
		ptr1 = ptr; ptr = NULL;
		ptr2 = dkstr_next(ptr1, NULL);
		if(ptr2) {	
		  ptr = dkstr_next(ptr2, NULL);
		}
		if(ptr1) {	
		  w = get_long_number(ptr1, (o->parent)->current_w);
		}
		if(ptr2) {	
		  h = get_long_number(ptr2, (o->parent)->current_h);
		}
		if(ptr) {
		  ptr1 = ptr; ptr = NULL;
		  ptr2 = dkstr_next(ptr1, NULL);
		  if(ptr1) {	
		    if(!apply_anchor(j, o, ptr1)) {
		      back = 0;
		    }
		  }
		  if(ptr2) {	
		    if(!apply_fill(j, o, ptr2)) {
		      back = 0;
		    }
		  }
		}
	      }
	      if((x != -1L) && (y != -1L) && (back)) {
	        if((w > 0L) && (h > 0L)) {
		  o->pos_x = x; o->pos_y = y;
		  o->pos_w = w; o->pos_h = h;
		}
	        (o->parent)->current_x = x;
		(o->parent)->current_y = y;
		if(w > 0L) { (o->parent)->current_w = w; }
		if(h > 0L) { (o->parent)->current_h = h; }
	      }
	    }
	  }
	}
	if(!back) {
	  
	  /* ERROR: Insufficient placement information */
	  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 32);
	}
      } break;
      case LAYOUT_BORDER: {
        
        back = 1;
	if(s) {
	  ptr = dkstr_start(s, NULL);
	  if(ptr) {
	    back = apply_anchor(j, o, ptr);
	  }
	}
      } break;
      case LAYOUT_CARD: {
        
        back = 1;
	if(s) {
	  ptr = dkstr_start(s, NULL);
	  if(ptr) {
	    newptr = dkstr_dup(ptr);
	    if(newptr) {
	      if(o->card_add_text) {
	        cx = o->card_add_text; dk_delete(cx);
	      }
	      o->card_add_text = newptr;
	    } else {
	      back = 0;
	      /* ERROR: Memory */
	      jlmsg_error_memory(j);
	    }
	  }
	}
      } break;
      default: {
        
	switch((o->parent)->ocn) {
	  case CLASS_JTABBEDPANE : {	
	    back = 1;
	    if(s) {
	      ptr1 = dkstr_start(s, NULL);
	      if(ptr1) {		
		ptr2 = dkstr_dup(ptr1);
		if(ptr2) {
		  if(o->card_add_text) {
		    cx = o->card_add_text; dk_delete(cx);
		  } o->card_add_text = ptr2;
		} else {
		  back = 0;
		  /* ERROR: Memory */
		  jlmsg_error_memory(j);
		}
	      }
	    }
	  } break;
	  default: {			
            back = 1;
	    if(s) {
	      if(dkstr_start(s, NULL)) {
	        /* WARNING: Additional info ignored */
		jlmsg_log1(j, DK_LOG_LEVEL_WARNING, 33);
	      }
	    }
	  } break;
	}
      } break;
    }
  } else {		
  }
  
  return back;
}



/**	Apply contents border setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Border setting as text.
	@return	1 on success, 0 on error.
*/
static
int
apply_contents_border DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  char *x;
  if((j) && (o)) {
    if(o->c_border) {
      x = o->c_border; dk_delete(x);
    } o->c_border = NULL;
    back = 1;
    if(v) {
      o->c_border = dkstr_dup(v);
      if(!(o->c_border)) { back = 0; }
    }
  }
  return back;
}



/**	Apply contents fill setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Fill setting text.
	@return	1 on success, 0 on error.
*/
static
int
apply_contents_fill DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  if((j) && (o)) {
    o->c_fill = FILL_NOT_SPECIFIED; back = 1;
    if(v) {
      o->c_fill = dkstr_array_index(fill_values, v, 0);
      if(o->c_fill == -1) {
        /* ERROR: Fill value */
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 21);
	back = 0;
      }
    }
  }
  return back;
}



/**	Apply contents anchor setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Contents anchor setting text.
	@return	1 on success, 0 on error.
*/
static
int
apply_contents_anchor DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  if((j) && (o)) {
    o->c_anchor = ANCHOR_NOT_SPECIFIED; back = 1;
    if(v) {
      o->c_anchor = dkstr_array_index(anchor_values, v, 0);
      if(o->c_anchor == -1) {
        back = 0;
	/* ERROR: Illegal anchor value */
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 22);
      }
    }
  }
  return back;
}


/**	Apply a layout type.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Layout name.
	@return	1 on success, 0 on error.
*/
static
int
apply_layout DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  char *ptr1, *ptr2;
  long l1;
  int class_to_register;
  
  if((j) && (o)) {
    o->layout = LAYOUT_NOT_SPECIFIED;
    if(v) {
      ptr1 = dkstr_start(v, NULL);
      if(ptr1) {
        back = 1;
        ptr2 = dkstr_next(ptr1, NULL);
        o->layout = dkstr_array_index(layout_values, v, 0);
        if(o->layout == -1) {	
          back = 0;
	  /* ERROR: Illegal layout value */
	  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 34);
        } else {
	  class_to_register = 0;
	  switch(o->layout) {
	    case LAYOUT_GRIDBAG: {
	      j->f1 |= F1_HAVE_GRIDBAG;
	    } break;
	    case LAYOUT_GRID: {
	      class_to_register = 5;
	      o->grid_layout_r = o->grid_layout_c = 0L;
	      if(ptr2) {
	        ptr1 = dkstr_next(ptr2, NULL);
		if(ptr1) {	/* rows columns */
		  if(sscanf(ptr2, "%ld", &l1) == 1) {
		    o->grid_layout_r = l1;
		    if(sscanf(ptr1, "%ld", &l1) == 1) {
		      o->grid_layout_c = l1;
		    } else {
		      /* ERROR: Not a number */
		      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
		      back = 0;
		    }
		  } else {
		    /* ERROR: Not a number */
		    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
		    back = 0;
		  }
		} else {	/* columns */
		  if(sscanf(ptr2, "%ld", &l1) == 1) {
		    o->grid_layout_c = l1;
		  } else {
		    /* ERROR: Not a number */
		    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
		    back = 0;
		  }
		}
	      }
	    } break;
	    case LAYOUT_GROUP: {
	      class_to_register = 6;
	      back = 0;
	      /* ERROR: GroupLayout not supported */
	      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 36);
	    } break;
	    case LAYOUT_SPRING: {
	      class_to_register  = 7;
	      back = 0;
	      /* ERROR: SpringLayout not supported */
	      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 37);
	    } break;
	    case LAYOUT_CARD: { class_to_register = 1; } break;
	    case LAYOUT_BORDER: { class_to_register = 2; } break;
	    case LAYOUT_BOX_X: case LAYOUT_BOX_Y: case LAYOUT_BOX_PAGE: case LAYOUT_BOX_LINE:
	    { class_to_register = 3; } break;
	    case LAYOUT_FLOW: { class_to_register = 4; } break;
	  }
	  if(back) {
	    if(!register_used_class(j, layout_classes[class_to_register])) {
	      back = 0; jlmsg_error_memory(j);
	    }
	  }
        }
      } else {
      }
    }
  } else {	
  }
  
  return back;
}



/**	Add a contents node to the current object.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	jcn	Contents node for one contents object.
	@return	1 on success, 0 on error.
*/
static
int
add_jcn DK_P3(JLJ *,j, JLO *,o, JCN *,jcn)
{
  int back = 0;
  
  if(!(o->c_st)) {
    o->c_st = dksto_open(0);
    if(o->c_st) {
      dksto_set_comp(o->c_st, compare_contents_nodes, 0);
    }
  }
  if(!(o->c_it)) {
    if(o->c_st) {
      o->c_it = dksto_it_open(o->c_st);
    }
  }
  if((o->c_st) && (o->c_it)) {
    if(dksto_add(o->c_st, (void *)jcn)) {
      back = 1;
    }
  } 
  return back;
}



/**	Add remaining text to contents node (IGNORED).
	If there is additional text after contents element name
	and position + size we could store it and use it - but how.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	jcn	Contents node.
	@param	t	Text to add.
	@return	1 on success, 0 on error.
*/
static
int
add_remaining_text_to_jcn DK_P4(JLJ *,j, JLO *,o, JCN *,jcn, char *,t)
{
  int back = 1;
  /* ## no idea yet for remaining text ## */
  return back;
}



/**	Apply contents.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Contents element name, optionally followed by position and size.
*/
static
int
apply_contents DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0; int i;
  JLO *jlo = NULL; JCN *jcn = NULL;
  char *p1, *p2;
  
  if((j) && (o) && (v)) {
    p1 = dkstr_start(v, NULL);
    if(p1) {
      if(*p1 == '$') {	
        p2 = dkstr_next(p1, NULL);
	i = dkstr_array_index(cmd_specials, &(p1[1]), 0);
	if(i > -1) {
	  jcn = dk_new(JCN,1);
	  if(jcn) {
	    back = 1;
	    jcn->n = o->n_c;
	    o->n_c += 1UL;
	    jcn->t = JCN_GLUE;
	    switch(i) {
	      case 0: { 
	        jcn->t = JCN_GLUE;
		back = register_used_class(j, str_class_box);
	      } break;
	      case 1: {
	        jcn->t = JCN_SEPARATOR;
	      } break;
	    }
	    jcn->o = NULL;
	    if(p2) {
	      back = add_remaining_text_to_jcn(j, o, jcn, p2);
	    }
	    if(back) {
	      back = add_jcn(j, o, jcn);
	      if(!back) {
	        dk_delete(jcn); jcn = NULL;
	      }
	    }
	  }
	}
      } else {		
        p2 = dkstr_next(p1, NULL);
        jlo = find_object_for_name(j, p1, o, j->errlineno);
        if(jlo) {
          back = add_one_contents_element(j, o, jlo, o->n_c);
	  o->n_c += 1UL;
	  if(p2) {
	    back = add_adding_information(j, jlo, p2);
	  }
        }
      }
    } else {		
    }
  } 
  return back;
}



/**	Apply size setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Size.
	@param	cr	Size type: 2=maximum, 1=preferred, 0=minimum.
	@return	1 on success, 0 on error.
*/
static
int
apply_size DK_P4(JLJ *,j, JLO *,o, char *,v, int,cr)
{
  int back = 0;
  long l;
  char *p1, *p2;
  if((j) && (o)) {
    if(v) {
      p1 = dkstr_start(v, NULL);
      if(p1) {
        p2 = dkstr_next(p1, NULL);
	if(sscanf(p1, "%ld", &l) == 1) {
	  back = 1;
	  switch(cr) {
	    case 2: {
	      o->s_max_x = l;
	    } break;
	    case 1: {
	      o->s_pref_x = l;
	    } break;
	    default: {
	      o->s_min_x = l;
	    } break;
	  }
	  if(p2) {
	    if(sscanf(p2, "%ld", &l) == 1) {
	      switch(cr) {
	        case 2: {
	          o->s_max_y = l;
	        } break;
	        case 1: {
	          o->s_pref_y = l;
	        } break;
	        default: {
	          o->s_min_y = l;
	        } break;
	      }
	    } else {
	      back = 0;
	      /* ERROR: Not a number */
	      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 56);
	    }
	  } else {
	    switch(cr) {
	      case 2: {
	        o->s_max_y = l;
	      } break;
	      case 1: {
	        o->s_pref_y = l;
	      } break;
	      default: {
	        o->s_min_y = l;
	      } break;
	    }
	  }
	} else {
	  /* ERROR: Not a number */
	  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 56);
	}
      } else {
        /* ERROR: Empty string */
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
      }
    } else {
      /* ERROR: Empty string */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
    }
  }
  return back;
}



/**	Apply weight setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Component weight.
	@return	1 on success, 0 on error.
*/
static
int
apply_weight DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  double x;
  char *p1, *p2;
  if((j) && (o)) {
    if(v) {
    p1 = dkstr_start(v, NULL);
    if(p1) {
      p2 = dkstr_next(p1, NULL);
      if(sscanf(p1, "%lf", &x) == 1) {
        o->weightx = x;
	back = 1;
	if(p2) {
	  if(sscanf(p2, "%lf", &x) == 1) {
	    o->weighty = x;
	  } else {
	    /* ERROR: Not a number */
	    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 56);
	    back = 0;
	  }
	} else {
	  o->weighty = x;
	}
      } else {
        /* ERROR: Not a number */
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 56);
      }
    } else {
      /* ERROR: Empty string */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
    }
  } else {
    /* ERROR: Empty string */
    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
  }
  }
  return back;
}



/**	Apply horizontal text position setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Text position.
	@return	1 on success, 0 on error.
*/
static
int
apply_horizontal_text_position DK_P3(JLJ *,j, JLO *,o, char *,v) {
  int back = 0;
  if((j) && (o)) {
    if(v) {
      o->h_textpos = dkstr_array_index(horizontal_text_positions, v, 0);
      if(o->h_textpos != -1) {
        back = register_used_class(j, str_swing_constants);
	if(back == 0) { jlmsg_error_memory(j); }
      } else {
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 60);
      }
    } else {
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
    }
  }
  return back;
}



/**	Apply vertical text position setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Vertical text position.
	@return	1 on success, 0 on error.
*/
static
int
apply_vertical_text_position DK_P3(JLJ *,j, JLO *,o, char *,v) {
  int back = 0;
  if((j) && (o)) {
    if(v) {
      o->v_textpos = dkstr_array_index(vertical_text_positions, v, 0);
      if(o->v_textpos != -1) {
        back = register_used_class(j, str_swing_constants);
	if(back == 0) { jlmsg_error_memory(j); }
      } else {
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 60);
      }
    } else {
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
    }
  }
  return back;
}



/**	Apply border weight setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Border weights.
	@return	1 on success, 0 on error.
*/
static
int
apply_border_weight DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  double f = 0.0;
  char *p1, *p2;
  if((j) && (o)) {
    if(v) {
      p1 = dkstr_start(v, NULL);
      if(p1) {
        p2 = dkstr_next(p1, NULL);
	if(sscanf(p1, "%lf", &f) == 1) {
	  o->weightt = o->weightl = o->weightr = o->weightb = f;
	  back = 1;
	  if(p2) {
	    p1 = dkstr_next(p2, NULL);
	    if(sscanf(p2, "%lf", &f) == 1) {
	      o->weightl = o->weightr = o->weightb = f;
	      if(p1) {
	        p2 = dkstr_next(p1, NULL);
		if(sscanf(p1, "%lf", &f) == 1) {
		  o->weightr = o->weightb = f;
		  if(p2) {
		    if(sscanf(p2, "%lf", &f) == 1) {
		      o->weightb = f;
		    } else {
		      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 56);
		      back = 0;
		    }
		  }
		} else {
		  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 56);
		  back = 0;
		}
	      }
	    } else {
	      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 56);
	      back = 0;
	    }
	  }
	} else {
	  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 56);
	}
      } else {
        jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
      }
    } else {
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
    }
  }
  return back;
}



/**	Apply menu bar setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Name of the menu bar object.
	@return	1 on success, 0 on error.
*/
static
int
apply_menubar DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  char *p1, *p2;
  JLO *jlo;
  
  if((j) && (o) && (v)) {
    p1 = dkstr_start(v, NULL);
    if(p1) {
      p2 = dkstr_next(p1, NULL);
      jlo = find_object_for_name(j, p1, o, j->errlineno);
      if(jlo) {
        o->menubar = jlo;
	back = 1;
	if(p2) {		
	}
      }
    } else {			
    }
  }
  
  return back;
}


/**	Apply horizontal alignment setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Horizontal alignment setting as string.
	@return	1 on success, 0 on error.
*/
static
int
apply_horizontal_alignment DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  if((j) && (o)) {
    if(v) {
      o->h_align = dkstr_array_index(h_align_types, v, 0);
      if(o->h_align != -1) {
        back = register_used_class(j, str_swing_constants);
	if(back == 0) { jlmsg_error_memory(j); }
      } else {
        /* ERROR: Unknown horizontal alignment */
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 57);
      }
    } else {
      /* ERROR: Empty value */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
    }
  }
  return back;
}



/**	Apply vertical alignment setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Vertical alignment setting as string.
	@return	1 on success, 0 on error.
*/
static
int
apply_vertical_alignment DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  if((j) && (o)) {
    if(v) {
      o->h_align = dkstr_array_index(v_align_types, v, 0);
      if(o->h_align != -1) {
        back = register_used_class(j, str_swing_constants);
	if(back == 0) { jlmsg_error_memory(j); }
      } else {
        /* ERROR: Unknown vertical alignment */
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 57);
      }
    } else {
      /* ERROR: Empty value */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
    }
  }
  return back;
}



/**	Apply default close operation setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Close operation.
	@return	1 on success, 0 on error.
*/
static
int
apply_default_close_operation DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  if((j) && (o)) {
    if(v) {
      o->def_close = dkstr_array_index(close_operations, v, 0);
      if(o->def_close != -1) {
        back = register_used_class(j, str_windowconstants);
	if(!back) {
	  jlmsg_error_memory(j);
	}
      } else {
        /* ERROR: Unknown vertical alignment */
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 57);
      }
    } else {
      /* ERROR: Empty value */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 30);
    }
  }
  return back;
}



/**	Apply action command setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Action command.
	@return	1 on success, 0 on error.
*/
static
int
apply_action_command DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0; char *x;
  if((j) && (o)) {
    if(o->action_cmd) {
      x = o->action_cmd; dk_delete(x);
      /* WARNING: text definition overwrites previous setting */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 38);
    } o->action_cmd = NULL;
    back = 1;
    if(v) {
      o->action_cmd = dkstr_dup(v);
      if(!(o->action_cmd)) {
        back = 0;
	/* ERROR: Memory */
	jlmsg_error_memory(j);
      }
    }
  }
  return back;
}



/**	Apply tool tip text.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Tool tip text.
	@return	1 on success, 0 on error.
*/
static
int
apply_tool_tip_text DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0; char *x;
  if((j) && (o)) {
    if(o->tt_text) {
      x = o->tt_text; dk_delete(x);
      /* WARNING: text definition overwrites previous setting */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 38);
    } o->tt_text = NULL;
    back = 1;
    if(v) {
      o->tt_text = dkstr_dup(v);
      if(!(o->tt_text)) {
        back = 0;
	/* ERROR: Memory */
	jlmsg_error_memory(j);
      }
    }
  }
  return back;
}



/**	Apply title.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Title.
	@return	1 on success, 0 on error.
*/
static
int
add_title DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0; char *x;
  if((j) && (o)) {
    if(o->title) {
      x = o->title; dk_delete(x);
      /* WARNING: text definition overwrites previous setting */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 38);
    } o->title = NULL;
    back = 1;
    if(v) {
      o->title = dkstr_dup(v);
      if(!(o->title)) {
        back = 0;
	/* ERROR: Memory */
	jlmsg_error_memory(j);
      }
    }
  }
  return back;
}



/**	Apply accessible description setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Accessible description setting.
	@return	1 on success, 0 on error.
*/
static
int
apply_accessible_description DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0; char *x;
  if((j) && (o)) {
    if(o->acd_text) {
      x = o->acd_text; dk_delete(x);
      /* WARNING: text definition overwrites previous setting */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 38);
    } o->acd_text = NULL;
    back = 1;
    if(v) {
      o->acd_text = dkstr_dup(v);
      if(!(o->acd_text)) {
        back = 0;
	/* ERROR: Memory */
	jlmsg_error_memory(j);
      }
    }
  }
  return back;
}



/**	Apply text setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Text.
	@return	1 on success, 0 on error.
*/
static
int
apply_text DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0; char *x;
  if((j) && (o)) {
    if(o->text) {
      x = o->text; dk_delete(x);
      /* WARNING: text definition overwrites previous setting */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 38);
    } o->text = NULL;
    back = 1;
    if(v) {
      o->text = dkstr_dup(v);
      if(!(o->text)) {
        back = 0;
	/* ERROR: Memory */
	jlmsg_error_memory(j);
      }
    }
  }
  return back;
}



/**	Apply icon file name setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Icon file name.
	@return	1 on success, 0 on error.
*/
static
int
apply_icon_file_name DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  char *ptr1, *ptr2, *x;
  
  if((j) && (o)) {
    if(v) {
      ptr1 = dkstr_start(v, NULL);
      if(ptr1) {
        ptr2 = dkstr_dup(ptr1);
	if(ptr2) {
	  if(o->icon_file) {
	    x = o->icon_file; dk_delete(x);
	  } o->icon_file = ptr2;
	  back = 1;
	} else {
	  /* ERROR: Memory */
	  jlmsg_error_memory(j);
	}
      }
    }
  }
  
  return back;
}



/**	Apply constructor setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Constructor argument.
	@return	1 on success, 0 on error.
*/	
static
int
apply_constructor DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0; char *x;
  if((j) && (o)) {
    if(o->constructor) {
      x = o->constructor; dk_delete(x);
      /* WARNING: Constructor definition overrides previous setting */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 38);
    } o->constructor = NULL;
    back = 1;
    if(v) {
      o->constructor = dkstr_dup(v);
      if(!(o->constructor)) {
        back = 0;
	/* ERROR: Memory */
	jlmsg_error_memory(j);
      }
    }
  }
  return back;
}



/**	Apply text width setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Width setting.
	@return	1 on success, 0 on error.
*/
static
int
apply_text_width DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0; char *x;
  if((j) && (o)) {
    if(o->tf_width) {
      x = o->tf_width; dk_delete(x);
      /* WARNING: New width setting overrides previous setting */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 38);
    } o->tf_width = NULL;
    back = 1;
    if(v) {
      o->tf_width = dkstr_dup(v);
      if(!(o->tf_width)) {
        back = 0;
	/* ERROR: Memory */
	jlmsg_error_memory(j);
      }
    }
  }
  return back;
}



/**	Apply button group name.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Button group name.
	@return	1 on success, 0 on error.
*/
static
int
apply_button_group DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  JSG *g;
  
  if((j) && (o)) {
    if(v) {
      g = dksto_it_find_like(j->b_it, (void *)v, CR_COMPARE_OBJECT_AND_NAME);
      if(!g) {
        g = jsg_new(v);
	if(g) {
	  if(!dksto_add(j->b_st, (void *)g)) {
	    jsg_delete(g); g = NULL;
	  }
	}
      }
      if(g) {
        if(dksto_add(g->s, (void *)o)) {
	  back = 1;
	  j->f1 |= F1_HAVE_BUTTON_GROUPS;
	}
      }
      if(!back) {
        /* ERROR: Memory */
	jlmsg_error_memory(j);
      }
    } else {
      /* ERROR: Missing group name */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 39);
    }
  }
  
  return back;
}



/**	Apply use content pane setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	String containing boolean value.
	@return	1 on success, 0 on error.
*/
static
int
apply_use_content_pane DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 1;
  if(v) {
    if(dkstr_is_bool(v)) {
      if(dkstr_is_on(v)) {
        o->f2 |= F2_USE_CONTENT_PANE;
	back = register_used_class(j, str_Container);
	if(!back) { jlmsg_error_memory(j); }
      } else {
        o->f2 &= (~(F2_USE_CONTENT_PANE));
      }
    } else {
      back = 0;
      /* ERROR: Not a boolean */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 40);
    }
  } else {
    o->f2 |= F2_USE_CONTENT_PANE;
    back = register_used_class(j, str_Container);
    if(!back) { jlmsg_error_memory(j); }
  }
  return back;
}



/**	Apply component border setting.
	@param	j	Jlayout job
	@param	o	Current object.
	@param	v	Component border description.
	@return	1 on success, 0 on error.
*/
static
int
apply_component_border DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  char *p1, *p2, *p3, *p4, *p5;
  int i;
  
  if((j) && (o) && (v)) {
    if(o->border_type == BORDER_NOT_SPECIFIED) {	
      p1 = dkstr_start(v, NULL);
      if(p1) {
        p2 = dkstr_next(p1, NULL);
        back = 1;
        switch(dkstr_array_index(border_types, p1, 0)) {
          case 0: {	/* empty */
	    o->border_type = BORDER_EMPTY;
	    if(p2) {
	      p3 = dkstr_next(p2, NULL);
	      if(sscanf(p2, "%d", &i) == 0) {
	        o->bo_top = o->bo_left = o->bo_right = o->bo_bottom = i;
		p2 = p3;
		if(p2) {
		  p3 = dkstr_next(p2, NULL);
		  if(sscanf(p2, "%d", &i) == 0) {
		    o->bo_left = i;
		    p2 = p3;
		    if(p2) {
		      if(p2) {
		        p3 = dkstr_next(p2, NULL);
			if(sscanf(p2, "%d", &i) == 0) {
			  o->bo_right = i;
			  p2 = p3;
			  if(p2) {
			    if(sscanf(p2, "%d", &i) == 0) {
			      o->bo_bottom = i;
			    } else {
			      back = 0;
			      /* ERROR: Not a number */
			      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
			    }
			  }
			} else {
			  back = 0;
			  /* ERROR: Not a number */
			  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
			}
		      }
		    }
		  } else {
		    back = 0;
		    /* ERROR: Not a number */
		    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
		  }
		}
	      } else {
	        back = 0;
	        /* ERROR: Not a number */
		jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
	      }
	    }
	  } break;
          case 1: {	/* etched */
	    o->border_type = BORDER_ETCHED_LOWERED;
	    if(p2) {
	      switch(dkstr_array_index(border_sub_types, p2, 0)) {
	        case 0: {
	          o->border_type = BORDER_ETCHED_LOWERED;
		  back = register_used_class(j, str_etched_border);
		  if(back == 0) { jlmsg_error_memory(j); }
	        } break;
	        case 1: {
	          o->border_type = BORDER_ETCHED_RAISED;
		  back = register_used_class(j, str_etched_border);
		  if(back == 0) { jlmsg_error_memory(j); }
	        } break;
	        default: {
	          back = 0;
		  /* ERROR: Unknown sub type */
		  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 41);
	        } break;
	      }
	    }
	  } break;
          case 2: {	/* bevel */
	    o->border_type = BORDER_BEVEL_LOWERED;
	    if(p2) {
	      switch(dkstr_array_index(border_sub_types, p2, 0)) {
	        case 0: {
	          o->border_type = BORDER_BEVEL_LOWERED;
	        } break;
	        case 1: {
	          o->border_type = BORDER_BEVEL_RAISED;
	        } break;
	        default: {
	          back = 0;
		  /* ERROR: Unknown sub type */
		  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 41);
	        } break;
	      }
	    }
	  } break;
          case 3: {	/* line */
	    o->border_type = BORDER_LINE;
	    if(p2) {
	      p3 = dkstr_next(p2, NULL);
              p4 = dkstr_dup(p2);
	      if(p4) {
	        if(o->border_text) {
		  p5 = o->border_text; dk_delete(p5);
		}
		o->border_text = p4;
		if(p3) {
		  if(sscanf(p3, "%d", &i) == 1) {
		    o->bo_top = i;
		  } else {
		    back = 0;
		    /* ERROR: Not a number */
		    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
		  }
		}
	      } else {
	        back = 0;
		/* ERROR: Memory */
		jlmsg_error_memory(j);
	      }
	    } else {
	      back = 0;
	      /* ERROR: Missing color */
	      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 42);
	    }
	  } break;
          case 4: {	/* matte */
	    
	    o->border_type = BORDER_MATTE;
	    back = 0;
	    if(p2) {			
	      p3 = dkstr_next(p2, NULL);
	      if(sscanf(p2, "%d", &i) == 1) {
	        o->bo_top = i;
		p2 = p3;
		if(p2) {		
		  p3 = dkstr_next(p2, NULL);
		  if(sscanf(p2, "%d", &i) == 1) {
		    o->bo_left = i;
		    p2 = p3;
		    if(p2) {		
		      p3 = dkstr_next(p2, NULL);
		      if(sscanf(p2, "%d", &i) == 1) {
		        o->bo_right = i;
			p2 = p3;
			if(p2) {	
			  p3 = dkstr_next(p2, NULL);
			  if(sscanf(p2, "%d", &i) == 1) {
			    o->bo_bottom = i;
			    p2 = p3;
			    if(p2) {	
			      p3 = dkstr_chr(p2, ':');
			      if(p3) {	
			        *(p3++) = '\0';
				p3 = dkstr_start(p3, NULL);
				if(p3) {	
				  o->border_subtype = dkstr_array_index(
				    cmd_icon_color, p2, 0
				  );
				  if(o->border_subtype > -1) {
				    p4 = dkstr_dup(p3);	
				    if(p4) {		
				      if(o->border_text) {
				        p5 = o->border_text; dk_delete(p5);
				      }
				      o->border_text = p4; back = 1;
				    } else {
				      /* ERROR: Memory */
				      jlmsg_error_memory(j);
				    }
				  } else {
				    /* ERROR: Must be icon: or color: */
				    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 43);
				  }
				} else {
				  /* ERROR: Missing icon: or color: */
				  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 43);
				}
			      } else {
			        /* ERROR: must be icon: or color: */
				jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 43);
			      }
			    } else {
			      /* ERROR: Too few arguments */
			      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 44);
			    }
			  } else {
			    /* ERROR: Not a number */
			    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
			  }
			} else {
			  /* ERROR: Too few arguments */
			  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 44);
			}
		      } else {
		        /* ERROR: Not a number */
			jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
		      }
		    } else {
		      /* ERROR: Too few arguments */
		      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 44);
		    }
		  } else {
		    /* ERROR: Not a number */
		    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
		  }
		} else {
		  /* ERROR: Too few arguments */
		  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 44);
		}
	      } else {
	        /* ERROR: Not a number */
		jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 35);
	      }
	    } else {
	      /* ERROR: Too few arguments */
	      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 44);
	    }
	  } break;
          case 5: {	/* titled */
	    o->border_type = BORDER_TITLED;
	    if(p2) {
	      p3 = dkstr_dup(p2);
	      if(p3) {
	        if(o->border_text) {
	          p4 = o->border_text; dk_delete(p4);
	        }
	        o->border_text = p3;
		back = register_used_class(j, str_titled_border);
		if(back == 0) { jlmsg_error_memory(j); }
	      } else {
	        back = 0;
	        /* ERROR: Memory */
		jlmsg_error_memory(j);
	      }
	    } else {
	      back = 0;
	      /* ERROR: No title specified */
	      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 45);
	    }
	  } break;
	  default: {
	    back = 0;
	    /* ERROR: Unknown component border type */
	    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 46);
	  } break;
        }
      }
    } else {
      /* ERROR: Border type already specified */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 47);
    }
    if(back) {
      if(!register_used_class(j, str_borderfactory)) {
        back = 0; jlmsg_error_memory(j);
      }
    }
  } 
  return back;
}



/**	Apply enabled setting.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	String containing a boolean value.
	@return	1 on success, 0 on error.
*/
static
int
apply_enabled DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 1;
  if(v) {
    if(dkstr_is_bool(v)) {
      if(dkstr_is_on(v)) {
        o->enab = ENABLED_YES;
      } else {
        o->enab = ENABLED_NO;
      }
    }
  }
  return back;
}



/**	Assign a size group to an object.
	@param	j	Jlayout job.
	@param	o	Current object.
	@param	v	Size group name.
	@return	1 on success, 0 on error.
*/
static
int
apply_size_group DK_P3(JLJ *,j, JLO *,o, char *,v)
{
  int back = 0;
  JSG *g;
  if((j) && (o)) {
    if(v) {
      g = dksto_it_find_like(j->g_it, (void *)v, CR_COMPARE_OBJECT_AND_NAME);
      if(!g) {
        g = jsg_new(v);
	if(g) {
	  if(!dksto_add(j->g_st, (void *)g)) {
	    jsg_delete(g); g = NULL;
	  }
	}
      }
      if(g) {
        if(dksto_add(g->s, (void *)o)) {
	  j->f1 |= F1_HAVE_SIZE_GROUP;
	  back = register_used_class(j, str_jcomponent);
	  if(!back) {
	    jlmsg_error_memory(j);
	  }
	}
      }
      if(!back) {
        /* ERROR: Memory */
	jlmsg_error_memory(j);
      }
    } else {
      /* ERROR: Missing group name */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 48);
    }
  }
  return back;
}



/**	Process an attribute line (key=value).
	@param	j	Jlayout job.
	@param	l	Input line.
	@return	1 on success, 0 on error.
*/
static
int
apply_object_attribute DK_P2(JLJ *,j, char *,l)
{
  int back = 0;
  char *p1, *p2, *p3, *p4, *xdel, *parts[16]; size_t nparts; int act;
  
  p1 = l;
  p2 = dkstr_chr(p1, '=');
  if(p2) {
    *(p2++) = '\0';
    p2 = dkstr_start(p2, NULL);

    if(p2) {

      dkstr_chomp(p1, NULL);
      nparts = dkstr_explode(parts, 15, p1, NULL);
      if(nparts > 0) {
        act = dkstr_find_multi_part_abbr(parts, cmd_for_objects, '$', 0);
        switch(act) {
          case 0: {	/* border */
	    back = apply_border(j, j->o_cur, p2);
	  } break;
          case 1: {	/* fill */
	    back = apply_fill(j, j->o_cur, p2);
	    if(back) {
	      back = register_used_class(j, str_gridbagconstraints);
	      if(!back) {
	        jlmsg_error_memory(j);
	      }
	    }
	  } break;
          case 2: {	/* anchor */
	    back = apply_anchor(j, j->o_cur, p2);
	    if(back) {
	      back = register_used_class(j, str_gridbagconstraints);
	      if(!back) {
	        jlmsg_error_memory(j);
	      }
	    }
	  } break;
          case 3: {	/* contents border */
	    back = apply_contents_border(j, j->o_cur, p2);
	  } break;
          case 4: {	/* contents fill */
	    back = apply_contents_fill(j, j->o_cur, p2);
	  } break;
          case 5: {	/* contents anchor */
	    back = apply_contents_anchor(j, j->o_cur, p2);
	  } break;
          case 6: {	/* action listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_ACTION, p2);
	  } break;
          case 7: {	/* item listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_ITEM, p2);
	  } break;
          case 8: {	/* window listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_WINDOW, p2);
	  } break;
	  case 9: {
	    back = apply_layout(j, j->o_cur, p2);
	  } break;
	  case 10: {
	    back = apply_contents(j, j->o_cur, p2);
	  } break;
	  case 11: {
	    if((j->o_cur)->ocn == CLASS_JPANEL) {		
	      if(p2) {			
	        if((j->o_cur)->border_type == BORDER_NOT_SPECIFIED) {
	          p3 = dkstr_dup(p2);
		  if(p3) {		
		    if((j->o_cur)->border_text) {
		      /* Warning: Border text already defined */
		      jlmsg_log1(j, DK_LOG_LEVEL_WARNING, 38);
		      p4 = (j->o_cur)->border_text;
		      dk_delete(p4);
		    }
		    (j->o_cur)->border_text = p3;
		    (j->o_cur)->border_type = BORDER_TITLED;
		    back = 1;
		  } else {
		    /* ERROR: Memory */
		    jlmsg_error_memory(j);
		  }
	        } else {
	          /* ERROR: Border already specified */
		  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 47);
	        }
	      } else {
	        /* ERROR: No text */
	        jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 48);
	      }
	    } else {			
	      back = apply_text(j, j->o_cur, p2);
	    }
	  } break;
	  case 12: {
	    back = apply_constructor(j, j->o_cur, p2);
	  } break;
	  case 13: {
	    back = apply_text_width(j, j->o_cur, p2);
	  } break;
	  case 14: {
	    back = apply_size_group(j, j->o_cur, p2);
	  } break;
	  case 15: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_CARET, p2);
	  } break;
	  case 16: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_CHANGE, p2);
	  } break;
	  case 17: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_COMPONENT, p2);
	  } break;
	  case 18: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_CONTAINER, p2);
	  } break;
	  case 19: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_DOCUMENT, p2);
	  } break;
	  case 20: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_FOCUS, p2);
	  } break;
	  case 21: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_INTERNAL_FRAME, p2);
	  } break;
	  case 22: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_KEY, p2);
	  } break;
	  case 23: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_LIST_DATA, p2);
	  } break;
	  case 24: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_LIST_SELECTION, p2);
	  } break;
	  case 25: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_MOUSE, p2);
	  } break;
	  case 26: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_MOUSE_MOTION, p2);
	  } break;
	  case 27: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_MOUSE_WHEEL, p2);
	  } break;
	  case 28: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_PROPERTY_CHANGE, p2);
	  } break;
	  case 29: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_TABLE_MODEL, p2);
	  } break;
	  case 30: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_TREE_EXPANSION, p2);
	  } break;
	  case 31: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_TREE_MODEL, p2);
	  } break;
	  case 32: {	/* listener */
	    back = add_one_listener(j, j->o_cur, LISTENER_TREE_SELECTION, p2);
	  } break;
	  case 33: {
	    back = add_one_listener(j, j->o_cur, LISTENER_TREE_WILL_EXPAND, p2);
	  } break;
	  case 34: {
	    back = add_one_listener(j, j->o_cur, LISTENER_UNDOABLE_EDIT, p2);
	  } break;
	  case 35: {
	    back = apply_icon_file_name(j, j->o_cur, p2);
	    j->f1 |= F1_HAVE_ICON_FILE_NAMES;
	    if(back) {
	      back = register_used_class(j, str_imageicon);
	      if(!back) {
	        jlmsg_error_memory(j);
	      }
	    }
	  } break;
	  case 36: {
	    back = apply_button_group(j, j->o_cur, p2);
	    if(back) {
	      back = register_used_class(j, str_button_group);
	      if(!back) {
	        jlmsg_error_memory(j);
	      }
	    }
	  } break;
	  case 37: {
	    back = apply_component_border(j, j->o_cur, p2);
	  } break;
	  case 38: {
	    back= apply_use_content_pane(j, j->o_cur, p2);
	  } break;
	  case 39: {
	    back = apply_menubar(j, j->o_cur, p2);
	  } break;
	  case 40: {
	    back = apply_weight(j, j->o_cur, p2);
	  } break;
	  case 41: {
	    back = apply_size(j, j->o_cur, p2, 0);
	  } break;
	  case 42: {
	    back = apply_size(j, j->o_cur, p2, 1);
	  } break;
	  case 43: {
	    back = apply_size(j, j->o_cur, p2, 2);
	  } break;
	  case 44: {
	    back = apply_tool_tip_text(j, j->o_cur, p2);
	  } break;
	  case 45: {
	    back = apply_accessible_description(j, j->o_cur, p2);
	  } break;
	  case 46: {
	    back = apply_action_command(j, j->o_cur, p2);
	  } break;
	  case 47: {
	    back = apply_horizontal_alignment(j, j->o_cur, p2);
	  } break;
	  case 48: {
	    back = apply_vertical_alignment(j, j->o_cur, p2);
	  } break;
	  case 49: {
	    back = add_one_listener(j, j->o_cur, LISTENER_TRANSFER_HANDLER, p2);
	  } break;
	  case 50: {
	    back = add_title(j, j->o_cur, p2);
	  } break;
	  case 51: {
	    back = apply_default_close_operation(j, j->o_cur, p2);
	  } break;
	  case 52: {
	    back = apply_enabled(j, j->o_cur, p2);
	  } break;
	  case 53: {	/* font */
	    if(p2) {
	      if((j->o_cur)->font) {
	        xdel = (j->o_cur)->font; dk_delete(xdel);
	      }
	      (j->o_cur)->font = dkstr_dup(p2);
	      if((j->o_cur)->font) {
	        back = register_used_class(j, str_class_font);
	      } else { jlmsg_error_memory(j); }
	    }
	  } break;
	  case 54: {	/* foreground */
	    if(p2) {
	      if((j->o_cur)->foreground) {
	        xdel = (j->o_cur)->foreground; dk_delete(xdel);
	      }
	      (j->o_cur)->foreground = dkstr_dup(p2);
	      if((j->o_cur)->foreground) {
	        back = register_used_class(j, str_class_color);
	      } else { jlmsg_error_memory(j); }
	    }
	  } break;
	  case 55: {	/* background */
	    if(p2) {
	      if((j->o_cur)->background) {
	        xdel = (j->o_cur)->background; dk_delete(xdel);
	      }
	      (j->o_cur)->background = dkstr_dup(p2);
	      if((j->o_cur)->background) {
	        back = register_used_class(j, str_class_color);
	      } else { jlmsg_error_memory(j); }
	    }
	  } break;
	  case 56: {	/* font style */
	    
	    if(p2) {
	      if(strstr(p2, "-bold")) {
	        (j->o_cur)->font_changes |= 0x02;
	      } else {
	        if(strstr(p2, "bold")) {
		  (j->o_cur)->font_changes |= 0x01;
		}
	      }
	      if(strstr(p2, "-italic")) {
	        (j->o_cur)->font_changes |= 0x08;
	      } else {
	        if(strstr(p2, "italic")) {
		  (j->o_cur)->font_changes |= 0x04;
		}
	      }
	      back = 1;
	      if((j->o_cur)->font_changes) {
		j->f1 |= F1_HAVE_FONT_STYLE;
	        if(register_used_class(j, str_class_font)) {
		  back = register_used_class(j, str_jcomponent);
		  if(!back) {
		    jlmsg_error_memory(j);
		  }
		} else {
		  jlmsg_error_memory(j); back = 0;
		}
	      }
	    }
	  } break;
	  case 57: {	/* border weight */
	    back = apply_border_weight(j, j->o_cur, p2);
	  } break;
	  case 58: {
	    back = apply_horizontal_text_position(j, j->o_cur, p2);
	  } break;
	  case 59: {
	    back = apply_vertical_text_position(j, j->o_cur, p2);
	  } break;
	  default: {
	    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 13);
	  } break;
        }
      } else {
        /* ERROR: empty key */
        jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 23);
      }
    } else {
      /* ERROR: Empty value */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 50);
    }
  } else {
    /* ERROR: Empty value */
    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 50);
  }
  
  return back;
}



/**	Read entire input file.
	@param	j	Jlayout job.
	@param	inf	Input file.
	@return	1 on success, 0 on error.
*/
static
int
apply_input DK_P2(JLJ *,j, FILE *,inf)
{
  int back = 1, last_back = 1;
  char *p1;
  
  j->errfilename = j->ifname; j->errlineno = 0UL;
  while(fgets(j->input_buffer, j->sz_input_buffer, inf)) {
    j->errlineno += 1UL;
    if(back == 1) {
      dkstr_delcomm(j->input_buffer, '#');
      p1 = dkstr_start(j->input_buffer, NULL);
      if(p1) {		
        dkstr_chomp(p1, NULL);	
	switch(*p1) {
	  case '.' : {	
	    if(!apply_control_instruction(j, p1)) { back = 0; }
	  } break;
	  case '[' : {	
	    if(!apply_object_definition(j, p1)) { back = 0; }
	  } break;
	  default: {	
	    if(!apply_object_attribute(j, p1)) { back = 0; }
	  } break;
	}
      } else {		
      }
    }
    if(back != 1) {
      if(last_back) {
        /* ERROR: Processing aborted due to errors */
	
	jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 12);
      }
    }
    last_back = back;
  }
  j->errfilename = NULL; j->errlineno = 0UL;
  
  return back;
}



/**	Prepare JLJ structure to read input (create storages
	and iterators).
	@param	j	Jlayout job.
	@return	1 on success, 0 on error.
*/
static
int
prepare_to_read_input DK_P1(JLJ *,j)
{
  int back = 0;
  
  j->f1 = 0UL;
  j->o_st = dksto_open(0);
  if(j->o_st) {
    dksto_set_comp(j->o_st, jlread_compare_objects_by_name, CR_COMPARE_OBJECTS);
    j->o_main = NULL; j->o_cur = NULL;
    j->o_main = jlo_new(NULL);
    (j->o_main)->f2 |= F2_USE_CONTENT_PANE;
    j->o_cur = j->o_main;
    if(j->o_main) {
      j->o_it = dksto_it_open(j->o_st);
      if(j->o_it) {
        j->g_st = dksto_open(0);
        if(j->g_st) {
          dksto_set_comp(
            j->g_st, jlread_compare_groups_by_name, CR_COMPARE_OBJECTS
          );
          j->g_it = dksto_it_open(j->g_st);
          if(j->g_it) {
            j->b_st = dksto_open(0);
            if(j->b_st) {
              dksto_set_comp(
                j->b_st, jlread_compare_groups_by_name, CR_COMPARE_OBJECTS
              );
              j->b_it = dksto_it_open(j->b_st);
              if(j->b_it) {
      	        j->uc_st = dksto_open(0);
      	        if(j->uc_st) {
      	          dksto_set_comp(j->uc_st, compare_classes_by_name, 0);
      	          j->uc_it = dksto_it_open(j->uc_st);
      	          if(j->uc_it) {
      	            back = register_used_class(j, str_Container);
		    if(!back) {
		      jlmsg_error_memory(j);
		    }
      	          } else {
      	            jlmsg_error_memory(j);
      	          }
      	        } else {
      	          jlmsg_error_memory(j);
      	        }
              } else {
                /* ERROR: Memory */
      	        jlmsg_error_memory(j);
              }
            } else {
              /* ERROR: Memory */
              jlmsg_error_memory(j);
            }
          } else {
            /* ERROR: Memory */
            jlmsg_error_memory(j);
          }
        } else {
          /* ERROR: Memory */
          jlmsg_error_memory(j);
        }
      } else {
        /* ERROR: Memory */
	jlmsg_error_memory(j);
      }
    } else {
      /* ERROR: Memory */
      jlmsg_error_memory(j);
    }
  } else {
    /* ERROR: Memory */
    jlmsg_error_memory(j);
  }
  
  return back;
}




int
jlread DK_P1(JLJ *,j)
{
  int back = 0;
  FILE *inf;
  
  if(j) {
    if(j->ifname) {
      if(prepare_to_read_input(j)) {
        if(j->a) {
          inf = dkapp_fopen(j->a, j->ifname, str_r);
	  if(!inf) {
	    dkapp_err_fopenr(j->a, j->ifname);
	  }
        } else {
          inf = dksf_fopen(j->ifname, str_r);
	  if(!inf) {
	    /* ERROR: Failed to read file */
	    jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 51);
	  }
        }
        if(inf) {
          back = apply_input(j, inf);
          fclose(inf);
        }
      }
    }
  }
  
  return back;
}

