/*
Copyright (c) 2004-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

/**	@file	dkfig.h	Include file for all fig2vect source modules.
*/

#ifndef DKFIG_H_INCLUDED
/** Protection against multiple inclusions. */
#define DKFIG_H_INCLUDED 1

#include <dkconfig.h>

#if DK_HAVE_STDIO_H
#include <stdio.h>
#endif
#if DK_HAVE_IO_H
#include <io.h>
#endif
#if DK_HAVE_PROCESS_H
#include <process.h>
#endif
#if DK_HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if DK_HAVE_UNISTD_H
#include <unistd.h>
#endif
#if DK_HAVE_MATH_H
#include <math.h>
#endif
#if DK_TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if DK_HAVE_TIME_H
#include <time.h>
#else
#if DK_HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#endif
#endif

#include <dk.h>
#include <dkerror.h>
#include <dklic.h>
#include <dksf.h>
#include <dkma.h>
#include <dkmem.h>
#include <dksf.h>
#include <dkstr.h>
#include <dksto.h>
#include <dkstream.h>
#include <dkof.h>
#include <dkapp.h>
#include <dklogc.h>
#include <dkl2l.h>
#include <dkfont.h>
#include <dkle.h>
#include <dkenc.h>

#include "dkxsp.h"
#include "dkbsp.h"

#if defined(EXTERN)
#undef EXTERN
#endif

#if DK_HAVE_ZLIB_H
#if DK_HAVE_PNG_H
#include <png.h>
#endif
#endif

#if defined(__cplusplus)
extern "C" {
#endif
#if DK_HAVE_JPEGLIB_H
#include <jpeglib.h>
#endif
#if DK_HAVE_PNM_H
#include <pnm.h>
#endif
#if DK_HAVE_NETPBM_PNM_H
#include <netpbm/pnm.h>
#endif
#if defined(__cplusplus)
}
#endif



/**	Bitmap header information.
*/
typedef struct {
  long width;		/**< Width. */
  long height;		/**< Height. */
  int  colored;		/**< Flag: Color image. */
  int  seprgb;		/**< Flag: Use separated data streams. */
  unsigned long number;	/**< Image number. */
} dkfig_bitmap_width;



/**	Bounding box in Fig coordinates.
*/
typedef struct {
  char flags;	/**< Flag set. */
  double xmin;	/**< Left x. */
  double xmax;	/**< Right x. */
  double ymin;	/**< Smallest y. */
  double ymax;	/**< Largest y. */
} dk_fig_bb;



/**	Structure used to store options and file names.
*/
typedef struct {
  unsigned long	number;	/**< Number (in order of appearance). */
  char		*name;	/**< Option contents. */
  unsigned char used;	/**< Flag: Option was understood. */
} dk_fig_opt;



/**	Point coordinates.
*/
typedef struct {
  double x;		/**< X value. */
  double y;		/**< Y value. */
} dk_fig_point;



/**	Bezier point data.
*/
typedef struct {
  dk_fig_point	value;		/**< Point data itselft. */
  dk_fig_point	lcontrol;	/**< Control point "on the left side". */
  dk_fig_point	rcontrol;	/**< Control point "on the right side". */
} dk_fig_bezier_point;



/**	Arrowhead data.
*/
typedef struct {
  int    type;			/**< Fig file: Arrowhead type. */
  int    style;			/**< Fig file: Arrowhead style. */
  double thickness;		/**< Fig File: Thickness. */
  double w;			/**< Fig file: Width. */
  double h;			/**< Fig file: Height. */
  double       alpha;		/**< Calculated: Radians. */
  dk_fig_point p1;		/**< Calculated: Polygon point 1. */
  dk_fig_point p2;		/**< Calculated: Polygon point 2. */
  dk_fig_point p3;		/**< Calculated: Polygon point 3. */
  dk_fig_point p4;		/**< Calculated: Polygon point 4. */
  int          numpoints;	/**< Calculated: Number of polygon points. */
  double decrease;		/**< Calculated: Amount to decrease. */
} dk_fig_arrow;



/** Filling, pattern and drawing.
*/
typedef struct {
  int  ot;		/**< Object type. */
  int  st;		/**< Object sub type. */
  int  cl;		/**< Flag: Closed path. */
  int  ls;		/**< Line style. */
  long lt;		/**< Line thickness. */
  int  pc;		/**< Pen color. */
  int  fc;		/**< Fill color. */
  int  af;		/**< Area fill. */
  int  js;		/**< Join style. */
  int  cs;		/**< Cap style. */
  int  ar;		/**< Arrows. */
  int  ps;		/**< Pen style (unused). */
  double sv;		/**< Style val. */
  dk_fig_arrow ahf;	/**< Forward arrow data. */
  dk_fig_arrow ahb;	/**< Backward arrow data. */
} dk_fig_fpd;



/** Fig file object.
*/
typedef struct _dk_fig_object_ {
  struct _dk_fig_object_ *parent;	/**< Parent object. */
  unsigned long lineno;			/**< Line number of object in file. */
  long		layer;			/**< Depth of object (layer number). */
  int 		objtype;		/**< Object type DK_FIG_OBJ_xxx. */
  int		subtype;		/**< Object subtype. */
  dk_fig_fpd	fpd;			/**< Fill, pattern and draw. */
  void		*data;			/**< Object details. */
  void		*drve;			/**< Driver-specific extension data. */
  dk_storage_t	*osc;			/**< Object special comments. */
  dk_storage_iterator_t *osci;		/**< Object special c iterator. */
  dk_fig_bb	dbb;			/**< Object bounding box. */
} dk_fig_object;



/** Fig drawing.
*/
typedef struct {
  /*
    Drawing data.
  */
  dk_storage_t  	*objects;	/**< Storage: contents as read. */
  dk_storage_iterator_t	*objit;		/**< Iterator: contents as read. */
  dk_storage_t		*ccells;	/**< Storage: Color cells. */
  dk_storage_iterator_t *ccit;		/**< Iterator: Color cells. */
  dk_storage_t		*dlsc;		/**< Storage: Doc level special c. */
  dk_storage_iterator_t	*dlsci;		/**< Iterator: Doc level special c. */
  dk_storage_t		*fonth;		/**< Storage: Font handling. */
  dk_storage_iterator_t *fonthi;	/**< Iterator: Font handling. */
  int			 orientation;	/**< 0=Landscape, 1=Portrait. */
  int			 justification;	/**< 0=Center, 1=FlushLeft. */
  int			 units;		/**< 0=Metric, 1=Inches. */
  int			 paper;		/**< A B C D E A4 A3 A2 A1 A0 B5. */
  int			 multi;		/**< 0=Single, 1=Multiple. */
  int			 transparent;	/**< Index of transparent color. */
  int			 ahlj;		/**< Arrowhead line join. */
  int			 cstype;	/**< Coordinates origin  1=ll, 2=ul. */
  size_t		 spline_segs;	/**< Number of sub segments. */
  size_t		 numlatalpha;	/**< String length to encode number. */
  long			 resolution;	/**< Dpi. */
  unsigned long		 opt1;		/**< Options field 1. */
  unsigned long		 opt2;		/**< Options field 2. */
  unsigned long		 numlatfonts;	/**< Number of LaTeX fonts. */
  unsigned long		 maxitsteps;	/**< Maximum no of iteration steps. */
  unsigned long		 minitsteps;	/**< Minimum no of iteration steps. */
  double		 magnification;	/**< Print magnification. */
  double		 fres;		/**< Dpi as float. */
  dk_fig_bb		 dbb;		/**< Bounding box in Fig coordinates. */
  /*
    Used while reading input.
  */
  int			state;		/**< Current input state. */
  int			errc;		/**< Error code. */
  int			xory;		/**< Flag: read x or y: 0=x, 1=y. */
  size_t		currentpnt;	/**< Number of the current point. */
  unsigned long		lineno;		/**< Line number. */
  unsigned long		errl;		/**< Error line number. */
  char			*inputfilename;	/**< Input file name. */
  dk_storage_t		*nosc;		/**< Next objects special comments. */
  dk_storage_iterator_t *nosci;		/**< Iterator for nosc. */
  dk_fig_object		*currentobj;	/**< Current object. */
  dk_fig_object		*currentcomp;	/**< Current compound. */
  void			*currentdet;	/**< Details of current object. */
} dk_fig_drawing;



/**	Compound object (group).
*/
typedef struct {
  long                   ulx;		/**< BB: Upper left x. */
  long                   uly;		/**< BB: Upper left y. */
  long                   lrx;		/**< BB: Lower right x. */
  long                   lry;		/**< BB: Lower right y. */
  dk_storage_t  	*objects;	/**< Objects storage. */
  dk_storage_iterator_t	*objit;		/**< objects iterator. */
} dk_fig_compound;



/**	Arc calculations struct.
*/
typedef struct {
  /* input */
  double xa;		/**< Point 1 x. */
  double ya;		/**< Point 1 y. */
  double xb;		/**< Point 2 x. */
  double yb;		/**< Point 2 y. */
  double xc;		/**< Point 3 x. */
  double yc;		/**< Point 3 y. */
  /* output */
  double xm;		/**< Center x. */
  double ym;		/**< Center y. */
  double ra;		/**< Radius. */
  double aa;		/**< Alpha for point A. */
  double ab;		/**< Alpha for point B. */
  double ac;		/**< Alpha for point C. */
  double xleft;		/**< BB: left x. */
  double xright;	/**< BB: right x. */
  double ytop;		/**< BB: Upper y. */
  double ybottom;	/**< BB: Lower y. */
  double astart;	/**< Start of arc (rotation), radians. */
  double alength;	/**< Radians difference between a and c. */
  int    revert;	/**< Flag: Must use reverse path for arrowheads. */
  int    mathok;	/**< Flag: Math error. */
} dk_fig_arc_calc;



/**	Arc object.
 */
typedef struct {
  int          direction;	/**< Drawing orientation: 0=clockwise */
  long         x1;		/**< Point 1 x. */
  long         y1;		/**< Point 1 y. */
  long         x2;		/**< Point 2 x. */
  long         y2;		/**< Point 2 y. */
  long         x3;		/**< Point 3 x. */
  long         y3;		/**< Point 3 y. */
  double       lba;		/**< Start alpha in radians. */
  double       rba;		/**< End alpha in radians. */
  double       centerx;		/**< Center x. */
  double       centery;		/**< Center y. */
  dk_fig_arc_calc	calc;	/**< Calculation structure. */
} dk_fig_arc;



/**	Ellipse object.
*/
typedef struct {
  int    direction;		/**< Drawing direction. */
  long   centerx;		/**< Center x. */
  long   centery;		/**< Center y. */
  long   radiusx;		/**< Radius x. */
  long   radiusy;		/**< Radius y. */
  long   startx;		/**< Start x. */
  long   starty;		/**< Start y. */
  long   endx;			/**< End x. */
  long   endy;			/**< End y. */
  double angle;			/**< Rotation angle. */
} dk_fig_ellipse;



/**	Polyline object.
*/
typedef struct {
  int           flipped;	/**< Flag: Flipped. */
  size_t        npoints;	/**< Number of points. */
  long          radius;		/**< Radius for rounded rectangle corner. */
  long         *xvalues;	/**< Points (x-values). */
  long         *yvalues;	/**< Points (y.-values). */
  char         *imagename;	/**< Name of image file. */
  dk_fig_point	pa;		/**< Starting point. */
  dk_fig_point	pe;		/**< End point. */
} dk_fig_polyline;



/**	X-spline object.
*/
typedef struct {
  /* from XFig file */
  int           npoints;		/**< Number of points. */
  size_t	segs;		/**< Bezier segments per X-spline segment. */
  long         *xvalues;		/**< x coordinates. */
  long         *yvalues;		/**< y coordinates. */
  double       *svalues;		/**< s values. */
  /* calculated internally */
  size_t	nbpoints;		/**< Number of bezier points. */
  dk_fig_bezier_point	*bpoints;	/**< Bezier points. */
  /* start and end point for arrowhead */
  size_t	normals;	/**< Start of normal spline after arrowhead. */
  size_t	normale;	/**< End of normal spline before arrowhead. */
  double	ta;		/**< Start t value. */
  dk_fig_bezier_point	pa;	/**< Start point. */
  dk_fig_bezier_point	pa2;	/**< Control point at start. */
  double	te;		/**< End t value. */
  dk_fig_bezier_point	pe;	/**< End point. */
  dk_fig_bezier_point	pe2;	/**< Control point at end. */
  int		flags;		/**< Flags: bit 1=only one partial segment. */
} dk_fig_spline;



/**	Font handling.
	handling:
	* 0 -> label "Text" infont "ptmr"       scaled ...
	* 1 -> label "Text" infont "TimesRoman" scaled ...
	* 2 -> label btex Text etex
	* 3 -> \font\fntAA=ptmr at 12pt;
	* 4 -> \newcommand\font[0]{...size+features...}
	* 5 -> label btext \textbf{Text} etex
*/
typedef struct {
  int  handling;		/**< Handling type. */
  int  fontno;			/**< PS font number or features collection */
  int  ofontno;			/**< Original font number. */
  int  oflags;			/**< Original flags. */
  unsigned long fonthno;	/**< Font handling number. */
  double fontsize;		/**< Font size. */
} dk_fig_fonth_t;



/**	Text object.
*/
typedef struct {
  int    font;				/**< Font number. */
  int    font_flags;			/**< Text flags. */
  long   x;				/**< x position. */
  long   y;				/**< y position. */
  double font_size;			/**< Text size. */
  double angle;				/**< Rotation angle. */
  double height;			/**< Text height. */
  double length;			/**< Text length. */
  char  *text;				/**< Text to show. */
  dk_fig_fonth_t	*font_handling;	/**< Font handling. */
} dk_fig_text;



/**	Color cell object.
*/
typedef struct {
  int number;		/**< Color number. */
  int r;		/**< R component. */
  int g;		/**< G component. */
  int b;		/**< B component. */
  void *drve;		/**< Driver specific data. */
} dk_fig_colorcell;



/**	Color cell.
*/
typedef struct {
  double red;		/**< R component. */
  double green;		/**< G component. */
  double blue;		/**< B component. */
  int    ired;		/**< R component. */
  int    igreen;	/**< G component. */
  int    iblue;		/**< B component. */
} dk_fig_dcc;



/**	Font.
*/
typedef struct _dkfig_font_ {
  char *texname;	/**< TeX font name. */
  char *psname;		/**< PS font name. */
  int  features;	/**< Font features. */
} dk_fig_font;



/**	Conversion job structure.
*/
typedef struct {
  int  	normal_text;	/**< How to handle normal text. */
  int	special_text;	/**< How to handle special text. */
  int 		ahlj;	/**< Arrowhead linejoin. */
  int		 latfs;	/**< LaTeX font setup. */
  int		 bdnum;	/**< Base driver number. */
  int		 svgv;	/**< SVG version. */
  int		 svgs;	/**< SVG viewport specification. */
  unsigned	 psl;	/**< PS level (1, 2 or 3). */
  unsigned	 dscl;	/**< DSC level (1, 2 or 3). */
  size_t	spseg;	/**< Spline sub segments. */
  size_t	 lcfge;	/**< Size of largest configuration entry. */
  long		patlw;	/**< Pattern line width. */
  long		patrp;	/**< Pattern period. */
  unsigned long  optn;	/**< Number of opt entries. */
  unsigned long  opt1;	/**< Conversion options (1). */
  unsigned long  opt2;	/**< Conversion options (2). */
  unsigned	 nodcoord;	/**< Number of digits for coordinates. */
  unsigned	 nodtrigo;	/**< Number of digits for trigon calc res. */
  unsigned	 nodcolor;	/**< Number of digits for colors. */
  unsigned	 circlesteps;	/**< Bezier segments for circle quadrant. */
  unsigned long  maxitsteps;	/**< Maximum number of iteration steps. */
  unsigned long	 minitsteps;	/**< Minimum number of iteration steps. */
  double	 fsf;	/**< Font scale factor. */
  dk_app_t	*app;	/**< Application (s). */
  dk_storage_t	*opt;	/**< Conversion options (d). */
  dk_storage_iterator_t	*opti;	/**< Options iterator (d). */
  char		*drn;	/**< Driver name (d). */
  char		*defdr;	/**< Default driver name (d). */
  char		*basdr;	/**< Base driver name (d).  */
  char		*ifn1;	/**< Input file name as specified (s). */
  char		*ofn1;	/**< Output file name as specified (s). */
  char		*ifn2;	/**< Input file name after expansion (s). */
  char		*ofn2;	/**< Output file name after expansion (s). */
  char          **msg1;	/**< Message texts. */
  char		*texn;	/**< Tex file name. */
  char		*dvin;	/**< Dvi file name. */
  char		*psn;	/**< PS file name. */
  char		*logn;	/**< TeX log file name */
  char		*auxn;	/**< TeX aux file */
  char		*tpfn;	/**< Tex preamble file name */
  char		*incg;	/**< Includegraphics file name */
  char		*fcfg;	/**< Font configuration file name */
  /* filled by dkfig_co_load_configuration */
  dk_storage_t *optg;	/**< Global options from config file (d). */
  dk_storage_iterator_t *optgi; /**< Global options conf file (d) iterator */
  dk_storage_t *optb;	/**< Base driver options from config file (d). */
  dk_storage_iterator_t *optbi; /**< Base driver options (d) iterator */
  dk_storage_t *optd;	/**< Driver specific options from config file (d). */
  dk_storage_iterator_t *optdi; /**< Driver specific options (d) iterator */
  char		*cfgfn;	/**< Configuration file name (d). */
  dk_stream_t	*istrm;	/**< Input stream. */
  dk_stream_t	*ostrm;	/**< Output stream. */
  dk_fig_object *drwng;	/**< Drawing object. */
  /* filled by the outut driver functions */
  void		*outds;	/**< Output driver specific data. */
  dk_le_t	*uc2lat;	/**< Text converter. */
  char		*uc2ldir;	/**< Text converter base directory. */
  unsigned char	af;	/**< Flag: Find base driver name automatically. */
  int	paper_width;	/**< Paper width. */
  int	paper_height;	/**< Paper height. */
  int	paper_xleft;	/**< Left edge of draw range. */
  int	paper_xright;	/**< Right edge of draw range. */
  int	paper_ytop;	/**< Top edge of draw range. */
  int	paper_ybottom;	/**< Bottom edge of draw range. */
  int	align_h;	/**< Horizontal alignment. */
  int	align_v;	/**< Vertical alignment. */
  int	image_align;	/**< Image alignment in rectangle. */
} dk_fig_conversion;



/**	MetaPost output structure.
*/
typedef struct {
  int	multi;			/**< Flag: MultiMetaPost mp (0) or mmp (1). */
  int		 haveflags;	/**< Information the driver collected. */
  int		 linecap;	/**< Linecap style. */
  int		 linejoin;	/**< Linjoin style. */
  int		 havecd;	/**< Flag: Have current depth. */
  long		 currentdepth;	/**< Current layer number. */
  unsigned long	 currentimage;	/**< Number of the next image. */
  double bbx;			/**< BB x range. */
  double bby;			/**< BB y range. */
  double mx;			/**< Coordinates transformation. */
  double nx;			/**< Coordinates transformation. */
  double my;			/**< Coordinates transformation. */
  double ny;			/**< Coordinates transformation. */
  double sf;			/**< Scale factor for widths. */
  double	 linewidth;	/**< Line width. */
  double	 ahlength;	/**< Arrow head length. */
  double	 ahangle;	/**< Arrow head angle in degree. */
  dk_storage_t	*flatlist;	/**< Flat list of primitives. */
  dk_storage_iterator_t	*it;	/**< Iterator for flat list. */
} dk_fig_output_mp;



/**	Rotated ellipse.
*/
typedef struct {
  /* input data */
  double rx;		/**< x radius. */
  double ry;		/**< y radius. */
  double phi0;		/**< Rotation in radians. */
  /* output data */
  double h;		/**< Single height. */
  double w;		/**< Single width. */
} dk_fig_rot_ellipse;



/**	MetaPost output structure.
*/
typedef struct {
  int		     me;	/**< Flag: Math error. */
  double	     patrp;	/**< Delta for pattern repeat. */
  dk_fig_bb	     obb;	/**< Output bounding box. */
  dk_fig_conversion *c;		/**< Conversion job structure. */
  dk_fig_drawing    *d;		/**< Drawing. */
  dk_fig_output_mp  *m;		/**< MetaPost structure. */
  dk_fig_object     *o;		/**< Fig object. */
  dk_stream_t       *s;		/**< Output stream. */
  dk_fig_arrow	    *a;		/**< Arrowhead calculation structure. */
} dkfig_mp_output_instruction;



/**	EPS output structure.
*/
typedef struct {
  int			 flags;		/**< Flag set. */
  int			 me;		/**< Flag: Math error occured. */
  int			 have;		/**< Informations configured. */
  int			 lc;		/**< Linecap style. */
  int			 lj;		/**< Linejoin style. */
  int			 ls;		/**< Line style. */
  int			 spcpass;	/**< Flag: In special comments pass. */
  int			 errprinted;	/**< Flag: Error message printed. */
  unsigned		 psfont;	/**< PS font number. */
  size_t		 allimw;	/**< String length to encode number. */
  unsigned long		 numimw;	/**< Number of image widths. */
  int			 skip_images;	/**< Flag: Skip image handling. */
  double		 xfactor;	/**< Coordinates transformation. */
  double		 xshift;	/**< Coordinates transformation. */
  double		 yfactor;	/**< Coordinates transformation. */
  double		 yshift;	/**< Coordinates transformation. */
  double		 xleft;		/**< BB: left x. */
  double		 xright;	/**< BB: right x. */
  double		 ybottom;	/**< BB: bottom y. */
  double		 ytop;		/**< BB: top y. */
  double		 lw;		/**< Line width. */
  double		 sv;		/**< Element length of dashed lines. */
  double		 psfsize;	/**< PS font size. */
  dk_fig_dcc		 dcc;		/**< Current color cell. */
  dk_fig_conversion	*c;		/**< Conversion job structure. */
  dk_fig_drawing	*d;		/**< Fig drawing (from c). */
  dk_stream_t		*s;		/**< Output stream (c->ostrm). */
  dk_fig_object		*dro;		/**< Root drawing object (c->drwng). */
  dk_storage_t		*fl;		/**< Flattened objs container (dyn) . */
  dk_storage_iterator_t	*fli;		/**< Flattened objs iterator (dyn). */
  dk_storage_t		*iw;		/**< Image widths container (dyn). */
  dk_storage_iterator_t *iwi;		/**< Image widths iterator (dyn). */
  FILE			*fdvi;		/**< DVI file, must close. */
  FILE			*fps;		/**< PS file, must close. */
  int			*fnused;	/**< Flags, from dkfig_output_eps(). */
  int			*patused;	/**< flags, from dkfig_output_eps(). */
  dk_fig_object		*o;		/**< Current object. */
  dk_fig_arrow		*a;		/**< Application structure. */
  double		translate_x;	/**< X translation. */
  double		translate_y;	/**< Y translation. */
  double		scaleval;	/**< Scale factor. */
  int			must_scale;	/**< Flag: Must scale image. */
} dkfig_eps_output_instruction;



/**	SVG output structure.
*/
typedef struct {
  int			 me;		/**< Math error. */
  int			 spcp;		/**< Flag: Special comments pass. */
  int			 errprinted;	/**< Flag: Error message printed. */
  int			 prep_mods;	/**< Flag: Expect modifications. */
  int			 spcpass;	/**< Flag: Special comments pass. */
  size_t		 lpat;	/**< String length for number (patterns). */
  size_t		 lsty;	/**< String length for number (styles). */
  long                   xmin;		/**< Viewport left x in PS points. */
  long                   xmax;		/**< Viewport right x in PS points. */
  long                   ymin;		/**< Viewport upper y in PS points. */
  long                   ymax;		/**< Viewport lower y in PS points. */
  unsigned long          njsl;		/**< Number of JS libraries. */
  unsigned long		 npat;		/**< Number of patterns. */
  unsigned long		 nsty;		/**< Number of styles. */
  double		 mx;		/**< Coordinates transformation. */
  double		 nx;		/**< Coordinates transformation. */
  double		 my;		/**< Coordinates transformation. */
  double		 ny;		/**< Coordinates transformation. */
  dk_fig_conversion	*c;		/**< Conversion job structure. */
  dk_fig_drawing	*d;		/**< Fig drawing (from c). */
  dk_stream_t		*s;		/**< Output stream (c->ostrm). */
  dk_fig_object		*dro;		/**< Fig drawing root object. */
  dk_storage_t		*fl;		/**< Flattened object list. */
  dk_storage_iterator_t	*fli;		/**< Flattened object list. */
  dk_storage_t		*pat;		/**< Patterns. */
  dk_storage_iterator_t	*pati;		/**< Patterns. */
  dk_storage_t		*sty;		/**< Styles. */
  dk_storage_iterator_t	*styi;		/**< Styles. */
  /* during output pass */
  dk_fig_object		*o;		/**< Current object. */
  dk_storage_t		*jsl;		/**< JavaScript libraries. */
  dk_storage_iterator_t	*jsli;		/**< JavaScript libraries. */
  char 			*urw_dir;	/**< URW font directory. */
  char			*fnused;	/**< Fonts used, array(35). */
  dk_font_mapping_t	*fontmap;	/**< Font mapping. */
} dkfig_svg_output_instruction;



/**	SVG fill pattern.
*/
typedef struct _dk_fig_svg_pat_ {
  int		pattp;		/**< Pattern type. */
  int		sred;		/**< Stroke R component. */
  int		sgreen;		/**< Stroke G component. */
  int		sblue;		/**< Stroke B component. */
  int		fred;		/**< Fill R component. */
  int		fgreen;		/**< Fill G component. */
  int		fblue;		/**< Fill B component. */
  long		patrp;		/**< Pattern repeat distance. */
  unsigned long patno;		/**< Pattern ID. */
  unsigned long flags;		/**< DKFIG_SVG_FL_xxx. */
  /* unsigned long lw; */	/**< Line width. */
} dk_fig_svg_pat;



/**	SVG fill style.
*/
typedef struct _dk_fig_svg_style_ {
  int		ls;		/**< Line style. */
  int		lc;		/**< Linecap style. */
  int		lj;		/**< Linejoin style. */
  int		sred;		/**< Stroke color R. */
  int		sgreen;		/**< Stroke color G. */
  int		sblue;		/**< Stroke color B. */
  int		fred;		/**< Fill color R. */
  int		fgreen;		/**< Fill color G. */
  int		fblue;		/**< Fill color B. */
  int		talign;		/**< Text align. */
  long		lw;		/**< Line width. */
  unsigned long	classno;	/**< Class ID number. */
  unsigned long	flags;		/**< Flags which information is set. */
  double	sv;		/**< Style value. */
  dk_fig_svg_pat	*pat;	/**< Fill pattern. */
  dk_fig_fonth_t	*fonth;	/**< Font handling. */
} dk_fig_svg_style;



/**	SVG attribute.
*/
typedef struct _dk_fig_svg_attr_ {
  int		classid;	/**< Attribute type. */
  char		*value;		/**< Attribute value. */
} dk_fig_svg_attr;



/**	SVG-driver-specific data for an object.
*/
typedef struct _dk_fig_svg_drve_ {
  dk_fig_svg_style	*st1;	/**< Object style. */
  dk_fig_svg_style      *st2;	/**< Arrowhead forward style. */
  dk_fig_svg_style      *st3;	/**< Arrowhead backward style. */
  dk_storage_t		*attr;	/**< Attributes storage. */
  dk_storage_iterator_t	*attri;	/**< Attributes storage iterator. */
  dk_storage_t		*xl;	/**< Xlink attributes. */
  dk_storage_iterator_t	*xli;	/**< Xlink attributes storage iterator. */
} dk_fig_svg_drve;



/**	Image used in EPS output driver.
*/
typedef struct {
  int  type;		/**< Type: 0=PS/EPS, 1=PNG, 2=JPEG, 3=NetPBM. */
  int  colored;		/**< Flag: Colored (1) or grayscaled (0). */
  int  binary;		/**< Flag: PS contains binary data. */
  int  fl;		/**< Flag: Image flipped, 0=none, 1=horiz, 2=diag. */
  long xmin;		/**< Origin x. */
  long width;		/**< Image width in pixels / PS points. */
  long ymin;		/**< Origin y. */
  long height;		/**< Image height in pixels / PS points. */
  char *filename;	/**< Name of temporary file. */
} dkfig_eps_image_info;



/**	Image used by PDF output driver.
*/
typedef struct {
  int  flipped;		/**< Flag: flipped, 0=no, 1=horiz, 2=diag. */
  char *inputfilename;	/**< Input file name. */
  char *ofn;		/**< Output file name. */
  char *afn;		/**< Alpha file name. */
  unsigned long ol;	/**< Output length. */
  unsigned long al;	/**< Alpha length. */
  unsigned long w;	/**< Image width. */
  unsigned long h;	/**< Image height. */
  int bpp;		/**< Bits per pixel. */
  int ch;		/**< Number of channels. */
  unsigned long objno;	/**< Object number. */
  unsigned long lineno;	/**< Line number. */
} dkfig_pdf_image;



/**	Pattern description for PDF output driver.
*/
typedef struct {
  int patno;		/**< Pattern number. */
  int bgcol;		/**< Background color. */
  int fgcol;		/**< Foreground color. */
  unsigned long objno;	/**< Object number. */
  /* long lt; */	/**< Line thickness. */
  int tile;		/**< Flag: Tiled. */
  unsigned long lineno;	/**< Line number. */
} dkfig_pdf_pattern;



/**	PDF-driver-specific object extension.
*/
typedef struct {
  int fn;		/**< Font number. */
  dkfig_pdf_pattern *p;	/**< Fill pattern. */
  dkfig_pdf_image *i;	/**< Fill image. */
} dkfig_pdf_drve_text;



/**	PDF output instruction.
*/
typedef struct _dkfig_pdf_output_instruction_ {
  dk_fig_bb		 imgbb;	/**< Bounding box for images. */
  dk_fig_conversion 	*c;	/**< Conversion job structure. */
  dk_fig_drawing 	*d;	/**< Drawing. */
  dk_stream_t 		*s;	/**< Output stream. */
  dk_stream_t		*gcs;	/**< Graphics contents stream. */
  dk_storage_t		*pdfo;	/**< PDF object position storage. */
  dk_storage_iterator_t	*pdfoi;	/**< PDF object position iterator. */
  dk_storage_t 		*fl;	/**< Flattened object container storage. */
  dk_storage_iterator_t *fli;	/**< Flattened object container iterator. */
  dk_storage_t		*patl;	/**< Pattern storage. */
  dk_storage_iterator_t	*patli;	/**< Pattern iterator. */
  dk_storage_t		*imgl;	/**< Images storage. */
  dk_storage_iterator_t *imgli;	/**< Images iterator. */
  dk_fig_object		*co;	/**< Current object. */
  dkfig_pdf_pattern	*cp;	/**< Current pattern. */
  dkfig_pdf_image	*ci;	/**< Current image. */
  unsigned long		*fu;	/**< Flags: Fonts used (14). */
  unsigned long		 no;	/**< Next object number. */
  size_t		 nostr;	/**< Curent obj no encoded as string. */
  int			 ec;	/**< Error code. */
  int			 me;	/**< Flag: Math error. */
  int			 procs;	/**< Flag bit set: Procsets in use. */
  int			 flags;	/**< Flags: Which information do I have. */
  int			 dash;	/**< Dash type. */
  int			 lc;	/**< Linecap style. */
  int			 lj;	/**< Linejoin style. */
  int			 spcpass;	/**< Flag: In pass for spec comm. */
  int			 errprinted;	/**< Flag: Error message issued. */
  double		 xfactor;	/**< Scale factor for x. */
  double		 yfactor;	/**< Scale factor for y. */
  double		 xadd;		/**< Shift value for x. */
  double		 yadd;		/**< Shift value for y. */
  double		 xleft;		/**< BB: Left x. */
  double		 xright;	/**< BB: Right x. */
  double		 ybottom;	/**< BB: Bottom y. */
  double		 ytop;		/**< BB: Top y. */
  dk_fig_dcc		 strok;		/**< Stroke color. */
  dk_fig_dcc		 nonst;		/**< Non-stroke (fill+text) color. */
  double		 lw;		/**< Line width. */
  double		 dasv;		/**< Dash length. */
} dkfig_pdf_output_instruction;


/**	Image for Java output driver. */
typedef struct {
  unsigned long	imgno;		/**< Unique image number. */
  char		*filename;	/**< File name. */
} dkfig_java_image;

/**	Color for Java output driver.  */
typedef struct {
  unsigned long colno;	/**< Color number. */
  double	r;	/**< Red. */
  double	g;	/**< Green. */
  double	b;	/**< Blue. */
} dkfig_java_color;


/**	Font for Java output driver. */
typedef struct {
  unsigned long	fontno;	/**< Font number. */
  int		font;	/**< PS font index (0-35). */
  double	size;	/**< Font size. */
} dkfig_java_font;


/**	Stroke for Java output driver. */
typedef struct {
  unsigned long	strokeno;	/**< Unique stroke number. */
  double	linewidth;	/**< Line width. */
  double	sv;		/**< Style value. */
  int		cap;		/**< Line cap. */
  int		join;		/**< Line join. */
  int		dash;		/**< Dash pattern index. */
} dkfig_java_stroke;


/**	Driver specific object extension for Java output driver. */
typedef struct {
  dkfig_java_stroke		*st;	/**< Stroke. */
  dkfig_java_color		*draw;	/**< Draw color. */
  dkfig_java_color		*fill;	/**< Fill color. */
  dkfig_java_image		*img;	/**< Image. */
  dkfig_java_font		*font;	/**< Font (to show texts). */
  unsigned long			 tn;	/**< Text number. */
} dkfig_java_drve;

/**	Layer information. */
typedef struct {
  long		layer;	/**< Layer number. */
  dk_fig_bb	bb;	/**< Bounding box. */
} dkfig_java_layer;

/**	Java conversion structure.  */
typedef struct {
  dk_fig_conversion	*c;	/**< Conversion job structure. */
  dk_fig_drawing	*d;	/**< Fig drawing. */
  dk_stream_t		*s;	/**< Output stream. */
  dk_fig_object		*dro;	/**< Fig drawing root object. */
  dk_storage_t		*fl;	/**< Flat list. */
  dk_storage_iterator_t	*fli;	/**< Flat list iterator. */
  dk_storage_t		*s_c;	/**< Colors. */
  dk_storage_iterator_t	*i_c;	/**< Colors iterator. */
  dk_storage_t		*s_f;	/**< Fonts. */
  dk_storage_iterator_t	*i_f;	/**< Fonts iterator. */
  dk_storage_t		*s_s;	/**< Strokes. */
  dk_storage_iterator_t	*i_s;	/**< Strokes iterator. */
  dk_storage_t		*s_i;	/**< Images. */
  dk_storage_iterator_t	*i_i;	/**< Images iterator. */
  dk_storage_t		*s_l;	/**< Layers storage. */
  dk_storage_iterator_t	*i_l;	/**< Layers iterator. */
  dk_fig_object		*o;	/**< Current object to process. */
  dk_font_mapping_t	*fontmap;	/**< Font mapping. */
  double		 xfactor;	/**< Coordinates transformation. */
  double		 xshift;	/**< Coordinates transformation. */
  double		 yfactor;	/**< Coordinates transformation. */
  double		 yshift;	/**< Coordinates transformation. */
  double		 xleft;		/**< BB: left x. */
  double		 xright;	/**< BB: right x. */
  double		 ybottom;	/**< BB: bottom y. */
  double		 ytop;		/**< BB: top y. */
  unsigned long		 n_c;		/**< Number of colors. */
  unsigned long		 n_f;		/**< Number of fonts. */
  unsigned long		 n_s;		/**< Number of strokes. */
  unsigned long		 n_i;		/**< Number of images. */
  unsigned long		 n_t;		/**< Number of texts. */
  unsigned long		 c_c;		/**< Current color. */
  unsigned long		 c_f;		/**< Current font. */
  unsigned long		 c_s;		/**< Current stroke. */
  size_t		 l_c;		/**< Digits in number of colors. */
  size_t		 l_f;		/**< Digits in number of fonts. */
  size_t		 l_s;		/**< Digits in number of strokes. */
  size_t		 l_i;		/**< Digits in number of images. */
  size_t		 l_t;		/**< Digits in number of texts. */
  int			 ec;	/**< Error code (last error). */
  int			 me;	/**< Math error. */
  int			 spcpass;	/**< Pass of special comment. */
  int			 wr_utf8;	/**< Flag: Write texts in UTF-8. */
  unsigned char	text_right;	/**< Flag: right aligned text. */
  unsigned char	text_centered;	/**< Flag: centered text. */
  unsigned char	h_c;		/**< Flag: Have current color. */
  unsigned char h_f;		/**< Flag: Have current font. */
  unsigned long h_s;		/**< Flag: Have current stroke. */
} dkfig_java_output_instruction;


/**	PDF object position.
*/
typedef struct {
  unsigned long objno;	/**< Object number. */
  unsigned long pos;	/**< Position in PDF file. */
} dkfig_pdf_object_position;



/*
  Fig object types
*/

/** Fig object: Color. */
#define DK_FIG_OBJ_COLOR	0

/** Fig object: Ellipse. */
#define DK_FIG_OBJ_ELLIPSE	1

/** Fig object: Polyline. */
#define DK_FIG_OBJ_POLYLINE	2

/** Fig object: Spline. */
#define DK_FIG_OBJ_SPLINE	3

/** Fig object: Text. */
#define DK_FIG_OBJ_TEXT		4

/** Fig object: Arc. */
#define DK_FIG_OBJ_ARC		5

/** Fig object: Compound (group of objects). */
#define DK_FIG_OBJ_COMPOUND	6

/** Fig object: Drawing. */
#define DK_FIG_OBJ_DRAWING	255



/* dkfigopt.c */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigopt. */
#ifndef DKFIGOPT_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	Compare two stored options by line number.
	@param	p1	Pointer to left option structure.
	@param	p2	Pointer to right option structure.
	@param	cr	Comparison criteria, 1=pointer/lineno, default=pointer/pointer.
	@return	The comparison result.
*/
EXTERN int
dkfig_opt_compare DK_PR((void *p1, void *p2,int cr));

/**	Destroy option structure, release memory.
	@param	o	Option structure to destroy.
*/
EXTERN void
dkfig_opt_delete DK_PR((dk_fig_opt *o));

/**	Create new option structure.
	Option structures created by this function must be
	destroyed using dkfig_opt_delete().
	@param	number	Line number of option in config file.
	@param	text	Text line (key = value).
	@return	Pointer to new option structure or NULL.
*/
EXTERN dk_fig_opt *
dkfig_opt_new DK_PR((unsigned long number, char *text));

/**	Retrieve text (key = value) from option structure.
	@param	o	Option structure.
	@return	The option text.
*/
EXTERN char *
dkfig_opt_get_text DK_PR((dk_fig_opt *o));

/**	Process all stored options.
	Walk through all 4 option storages
	(general defaults, base driver options, driver
	options and command line arguments) and apply
	the settings to the conversion job structure.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on errors.
*/
EXTERN int
dkfig_opt_process_all DK_PR((dk_fig_conversion *c));

/**	Process one special comment (either for the entire
	drawing or before processing an object).
	@param	c	Conversion job structure.
	@param	line	Special comment.
	@param	dr	Driver name.
	@param	atdoclevel	Flag: at document level.
	@return	1 on success, error indicator on error
	(0=key not found, -1=ignored by the driver,
	-2=only allowed at document level, -3=errors
	while processing, -4=driver name not in the list).
*/
EXTERN int
dkfig_opt_process_special_comment DK_PR((dk_fig_conversion *c, char *line, char *dr, int atdoclevel));

/**	Retrieve special comment text (key = value) from
	special comment line (driverlist: key = value).
	@param	line	Text line to inspect.
	@param	dr	Name of the current driver.
	@return	Pointer to the text on success (Current driver
	is in the driver list and the line syntax is correct),
	NULL on error.
*/
EXTERN char *
dkfig_opt_special_comment_contents DK_PR((char *line, char *dr));

/**	Show configuration file contents.
	@param	c	Conversion job structure.
	@param	fn	File name.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_opt_show_config_file DK_PR((dk_fig_conversion *c, char *fn));

/**	Write standard input to new configuration file.
	@param	c	Conversion job structure.
	@param	fn	File name
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_opt_write_config_file DK_PR((dk_fig_conversion *c, char *fn));

/* only from within the library */
#if DKFIGCO_C
#endif

#if defined(__cplusplus)
}
#endif





/* dkfigco.c */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigco. */
#ifndef DKFIGCO_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	Clean up conversion job structure before
	releasing the memory. This function is invoked
	from dkfig_co_delete() or dkfig_co_new().
	@param	c	Conversion job structure.
*/
EXTERN void
dkfig_co_cleanup DK_PR((dk_fig_conversion *c));

/**	Cleanup up and release conversion job structure.
	@param	c	Conversion job structure.
*/
EXTERN void
dkfig_co_delete DK_PR((dk_fig_conversion *c));

/**	Initialize newly created conversion job structure,
	use default values and preferences obtained from \arg a.
	@param	c	New conversion job structure.
	@param	a	Application structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_co_init DK_PR((dk_fig_conversion *c, dk_app_t *a));

/**	Allocate new dk_fig_conversion structure and initialize it.
	@param	a	Application structure.
	@return	Pointer to new conversion job structure on success,
	NULL on error.
*/
EXTERN dk_fig_conversion *
dkfig_co_new DK_PR((dk_app_t *a));

/**	Apply command line arguments to conversion job.
	@param	c	Conversion job structure.
	@param	argc	Number of command line arguments.
	@param	argv	Command line arguments array.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_co_apply_args DK_PR((dk_fig_conversion *c, int argc, char **argv));

/**	Show current configuration.
	@param	c	Configuration job structure.
*/
EXTERN void
dkfig_co_showconf_app DK_PR((dk_fig_conversion *c));

/**	Save current configuration.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_co_configure_app DK_PR((dk_fig_conversion *c));

/**	Add input stream contents to drawing.
	@param	c	Conversion job structure
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_co_add_stream DK_PR((dk_fig_conversion *c));

/**	Load configuration file, process options
	and run conversion.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_co_run DK_PR((dk_fig_conversion *c));

/**	Show the contents of the used configuration file
	on standard output.
	A GUI wrapped around fig2vect may run
	"fig2vect --configuration-file" to see the driver
	configurations and the configuration settings.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_co_show_config_file DK_PR((dk_fig_conversion *c));

/**	Save a configuration file read from standard input as
	user-specific configuration file.
	A GUI wrapped around fig2vect may allow the user to
	create driver configurations and passes the configuration
	file contents as standard input to
	"fig2vect --write-configuration-file".
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_co_write_config_file DK_PR((dk_fig_conversion *c));

#if defined(__cplusplus)
}
#endif





/* dkfigrd.c */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigrd. */
#ifndef DKFIGRD_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	Compare two Fig objects.
	Comparison by layer number (largest layer number is
	lowest layer), object type, line style and source
	file line number.
	@param	l	Left object.
	@param	r	Right object.
	@param	cr	Comparison criteria.
	@return	The comparison result.
*/
EXTERN int
dkfig_object_compare DK_PR((void *l, void *r, int cr));

/**	Compare two color cells.
	@param	l	Left color cell.
	@param	r	Right color cell or color number.
	@param	cr	Comparison criteria (1=color/number,
	default=color/color).
	@return	The comparison result.
*/
EXTERN int
color_cell_compare DK_PR((void *l, void *r, int cr));

/**	Delete color cell, release memory.
	@param	cc	Color cell.
*/
EXTERN void
dkfig_delete_color_cell DK_PR((dk_fig_colorcell *cc));

/**	Create Fig object structure.
	@param	l	Line number in source file.
	@param	p	Parent object (drawing or compound).
	@param	st	Storage for special comments.
	@param	it	Iterator for special comments storage.
	@return	Pointer to new object on success, NULL on error.
*/
EXTERN dk_fig_object *
dkfig_object_new DK_PR((unsigned long l,dk_fig_object *p, dk_storage_t *st, dk_storage_iterator_t *it));

/**	Create new color cell structure.
	@param	n	Color number.
	@param	r	Red component.
	@param	g	Green component.
	@param	b	Blue component.
	@return	Pointer to new structure on success, NULL on error.
*/
EXTERN dk_fig_colorcell *
dkfig_new_color_cell DK_PR((int n,int r,int g,int b));

/**	Destroy drawing object including all contents.
	@param	o	Drawing root object.
*/
EXTERN void
dkfig_delete DK_PR((dk_fig_object *o));

/**	Create Fig object for entire drawing.
	@param	uwp	Flag: Use web-optimized color palette.
	@return	Pointer to new object on success, NULL on error.
*/
EXTERN dk_fig_object *
dkfig_new DK_PR((int uwp));

/**	Add one line to drawing structure.
	@param	o	Fig drawing root object.
	@param	b	Buffer containing the line to add.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_read_add_line DK_PR((dk_fig_object *o,char *b, dk_fig_conversion *c));

/**	Handle end of input.
	Check parser state (last object must be finished) and prepare
	some structures for text handling.
	@param	o	Fig drawing root object.
	@param	n	Text handling type for normal texts.
	@param	s	Text handling type for special texts.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_read_input_finished DK_PR((dk_fig_object *o, int n, int s, dk_fig_conversion *c));

/**	Retrieve object line number (the line number in which the
	object was started).
	@param	o	Fig object structure.
	@return	The line number.
*/
EXTERN unsigned long
dkfig_rd_get_lineno DK_PR((dk_fig_object *o));

/**	Retrieve line number of last error.
	@param	o	Drawing root object.
	@return	The line number.
*/
EXTERN unsigned long
dkfig_rd_get_errl DK_PR((dk_fig_object *o));

/**	Retrieve error code of last error.
	@param	o	Drawing root object.
	@return	The error code.
*/
EXTERN int
dkfig_rd_get_errc DK_PR((dk_fig_object *o));

/**	Save input file name for later use.
	@param	o	Drawing root object.
	@param	n	File name.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_set_input_filename DK_PR((dk_fig_object *o, char *n));

/**	Initialize root object to parse input.
	@param	o	Drawing root object.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_read_prepare_for_input DK_PR((dk_fig_object *o));

/**	Set number of Bezier spline segments per X-spline segment.
	@param	o	Drawing root object.
	@param	n	Number of segments.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_set_spline_segments DK_PR((dk_fig_object *o, size_t n));

/**	Set options to drawing.
	@param	d	Fig drawing structure.
	@param	o	Options as obtained from conversion job structure.
*/
EXTERN void
dkfig_rd_set_opts DK_PR((dk_fig_drawing *d, unsigned long o));

/**	Set arrowhead line join.
	@param	d	Fig drawing structure.
	@param	ahlj	The arrowhead line join.
*/
EXTERN void
dkfig_rd_set_ahlj DK_PR((dk_fig_drawing *d, int ahlj));

/**	Create flattened container of Fig objects in order of drawing.
	@param	c	Conversion job structure.
	@param	o	Drawing root object.
*/
EXTERN dk_storage_t *
dkfig_flat_list DK_PR((dk_fig_conversion *c, dk_fig_object *o));

/**	Set minimum and maximum number of iteration steps
	for arrowhead length on splines calculation.
	@param	d	Fig drawing structure.
	@param	min	Mininum number of steps.
	@param	max	Maximum number of steps.
*/
EXTERN void
dkfig_rd_set_it_steps DK_PR((dk_fig_drawing *d, unsigned long min, unsigned long max));


#if defined(__cplusplus)
}
#endif

/* dkfigeps.c */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigeps. */
#ifndef DKFIGEPS_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	EPS output driver function.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN
int dkfig_output_eps DK_PR((dk_fig_conversion *c));

/**	PS output driver function.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN
int dkfig_output_ps DK_PR((dk_fig_conversion *c));

/**	TeX output driver function.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN
int dkfig_output_tex DK_PR((dk_fig_conversion *c));

/**	Bounding box output driver function.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN
int dkfig_output_bb DK_PR((dk_fig_conversion *c));

/**	Correct a color cell for using a web optimized
	color palette.
	@param	dcc	Color cell to correct.
*/
EXTERN void
dkfig_eps_correct_for_palette DK_PR((dk_fig_dcc *dcc));

#if defined(__cplusplus)
};
#endif



/* dkfigpdf.c */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigpdf. */
#ifndef DKFIGPDF_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif


#if defined(__cplusplus)
extern "C" {
#endif

/**	PDF output driver function.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN
int dkfig_output_pdf DK_PR((dk_fig_conversion *c));

#if defined(__cplusplus)
}
#endif


/* dkfigmp.c */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigmp. */
#ifndef DKFIGMP_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	MMP (Multi-MetaPost) output driver function.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_output_mmp DK_PR((dk_fig_conversion *c));

/**	MetaPost output driver function.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_output_mp DK_PR((dk_fig_conversion *c));

#if defined(__cplusplus)
}
#endif



/* dkfigsvg.c */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigsvg. */
#ifndef DKFIGSVG_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	SVG output driver function.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_output_svg DK_PR((dk_fig_conversion *c));

#if defined(__cplusplus)
}
#endif



/* dkfigfnt.c */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigfnt. */
#ifndef DKFIGFNT_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

#if VERSION_BEFORE_2004_08_19

/**	Get LaTeX font name for a given index.
	@param	number	Font index.
	@return	The LaTeX name of the font.
*/
EXTERN char *
dkfig_fnt_get_tex_name DK_PR((size_t number));

/**	Get PS font name for a given index.
	@param	number	Font index.
	@return	The LaTeX name of the font.
*/
EXTERN char *
dkfig_fnt_get_ps_name DK_PR((size_t number));

#endif

/**	Compare two font handling structure.
	Comparison by handling type, font number,
	font size, original font number and
	font flags.
	@param	pl	Left font handling structure.
	@param	pr	Right font handling structure.
	@param	cr	Comparison criteria (ignored).
	@return	The comparison result.
*/
EXTERN int
dkfig_fnt_comp_fonth DK_PR((void *pl, void *pr, int cr));

/**	Fill font handling structure to match requirements
	of a Fig text object.
	@param	f	Destination font handling structure.
	@param	n	Text handling type for normal texts.
	@param	s	Text handling type for special texts.
	@param	t	Fig text object.
*/
EXTERN void
dkfig_fnt_fonth_for_text DK_PR((dk_fig_fonth_t *f, int n, int s, dk_fig_text *t));

/**	Create new font handling structure.
	@return	Pointer to new structure on success, NULL on error.
*/
EXTERN dk_fig_fonth_t *
dkfig_fnt_new_fonth DK_PR((void));

/**	Create a copy of a font handling structure.
	@param	f	Source font handling structure.
	@return	Pointer to new copy on success, NULL on error.
*/
EXTERN dk_fig_fonth_t *
dkfig_fnt_copy_fonth DK_PR((dk_fig_fonth_t *f));

/**	Delete font handling structure, release memory.
	@param	f	Font handling structure to delete.
*/
EXTERN void
dkfig_fnt_del_fonth DK_PR((dk_fig_fonth_t *f));


#if defined(__cplusplus)
}
#endif





/* dkfigdt.c */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigdt. */
#ifndef DKFIGDT_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	Write LaTeX preamble to a stream.
	@param	c	Conversion job structure.
	@param	os	Output stream.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_dt_tex_preamble DK_PR((dk_fig_conversion *c, dk_stream_t *os));

/**	Check whether or not the font handling requires LaTeX/TeX.
	@param	fhptr	Font handling structure.
	@return	1 if LaTeX or TeX is needed, 0 if not.
*/
EXTERN int
dkfig_dt_is_latex_text DK_PR((dk_fig_fonth_t *fhptr));

/**	Find number of characters needed to string encode
	a number.
	@param	num	Number to encode.
	@return	Needed string length.
*/
EXTERN size_t
dkfig_dt_needed_alpha DK_PR((unsigned long num));

/**	Write a font name to output stream. The font number
	is encoded as character sequence.
	@param	os	Output stream.
	@param	drw	Fig drawing structure.
	@param	fhptr	Font handling structure.
*/
EXTERN void
dkfig_dt_write_fontname DK_PR((dk_stream_t *os, dk_fig_drawing *drw, dk_fig_fonth_t *fhptr));

/**	Write LaTeX commands for font definitions to
	output stream.
	@param	c	Conversion job structure.
	@param	os	Output stream.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_dt_font_defs DK_PR((dk_fig_conversion *c, dk_stream_t *os));

/**	Write begin{document} to output stream.
	@param	c	Conversion job structure.
	@param	os	Output stream.
*/
EXTERN void
dkfig_dt_begin_document DK_PR((dk_fig_conversion *c, dk_stream_t *os));

/**	Check whether or not there is any text in the drawing
	requiring the use of LaTeX/TeX.
	@param	c	Conversion job structure.
	@return	Test result (1=LaTeX/TeX needed, 0=not needed).
*/
EXTERN int
dkfig_dt_need_tex_file DK_PR((dk_fig_conversion *c));

/**	Write LaTeX/TeX file and run LaTeX/TeX.
	@param	c	Conversion job structure.
	@param	d	Fig drawing structure.
	@param	it	Iterator to flattened Fig objects container.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_dt_run_tex DK_PR((dk_fig_conversion *c, dk_fig_drawing *d, dk_storage_iterator_t *it));

/**	Write end{document} sequence to output stream.
	@param	c	Conversion job structure.
	@param	os	Output stream.
*/
EXTERN void
dkfig_dt_end_document DK_PR((dk_fig_conversion *c, dk_stream_t *os));

/**	Write LaTeX commands for font definitions to
	output stream.
	@param	c	Conversion job structure.
	@param	os	Output stream.
	@param	r	Flag: redefinition (1) or definition (0).
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_dt_font_redef DK_PR((dk_fig_conversion *c, dk_stream_t *os, int r));

#if defined(__cplusplus)
}
#endif




#if VERSION_BEFORE_2009_04_23

/* lat2lat */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except lat2lat. */
#ifndef LAT2LAT_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

EXTERN char *
dkfig_l2l_encoding DK_PR((char c));

#if defined(__cplusplus)
}
#endif

#endif
/* VERSION_BEFORE_2009_04_23 */



/* dkfigtoo */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigtoo. */
#ifndef DKFIGTOO_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	Check whether or not to invert y coordinates.
	@param	d	Fig drawing structure.
	@return	Test result (1=invert, 0=do not invert).
*/
EXTERN int
dkfig_tool_invert_y DK_PR((dk_fig_drawing *d));

/**	Initialize arrowhead structure.
	@param	a	Arrowhead structure.
*/
EXTERN void
dkfig_tool_null_arrow DK_PR((dk_fig_arrow *a));

/**	Calculate arrowhead polygon points.
	@param	a	Arrowhead structure (destination).
	@param	p	Arrowhead end point.
	@param	phi	Direction at the end of line.
	@param	f	Objects filling, pattern and drawing.
	@param	d	Fig drawing structure.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_tool_calc_arrow DK_PR((dk_fig_arrow *a, dk_fig_point *p, double phi, dk_fig_fpd *f, dk_fig_drawing *d, dk_fig_conversion *c));

/**	Initialize a fill, pattern and drawing structure.
	@param	p	Structure to initialize.
*/
EXTERN void
dkfig_tool_null_fpd DK_PR((dk_fig_fpd *p));

/**	Correct all font handling structures for use with plain TeX.
	@param	c	Conversion job structure.
*/
EXTERN void
dkfig_tool_correct_for_plain_tex DK_PR((dk_fig_conversion *c));

/**	Initialize a bounding box structure.
	@param	bb	Structure to initialize.
*/
EXTERN void
dkfig_tool_bb_reset DK_PR((dk_fig_bb *bb));

/**	Add x value to bounding box.
	@param	bb	Bounding box structure.
	@param	x	X value to add.
*/
EXTERN void
dkfig_tool_bb_add_x DK_PR((dk_fig_bb *bb, double x));

/**	Add y value to bounding box.
	@param	bb	Bounding box structure.
	@param	y	Y value to add.
*/
EXTERN void
dkfig_tool_bb_add_y DK_PR((dk_fig_bb *bb, double y));

/**	Get minimum x value from bounding box structure.
	@param	bb	Bounding box structure.
	@return	Minimum x value.
*/
EXTERN double
dkfig_tool_bb_get_xmin DK_PR((dk_fig_bb *bb));

/**	Get maximum x value from bounding box structure.
	@param	bb	Bounding box structure.
	@return	Maximum x value.
*/
EXTERN double
dkfig_tool_bb_get_xmax DK_PR((dk_fig_bb *bb));

/**	Get minimum y value from bounding box structure.
	@param	bb	Bounding box structure.
	@return	Minimum y value.
*/
EXTERN double
dkfig_tool_bb_get_ymin DK_PR((dk_fig_bb *bb));

/**	Get maximum y value from bounding box structure.
	@param	bb	Bounding box structure.
	@return	Maximum y value.
*/
EXTERN double
dkfig_tool_bb_get_ymax DK_PR((dk_fig_bb *bb));

/**	Check: X values available in bounding box structure.
	@param	bb	Bounding box structure.
	@return	Check result (1=values available, 0=no values).
*/
EXTERN int
dkfig_tool_bb_have_x DK_PR((dk_fig_bb *bb));

/**	Check: Y values available in bounding box structure.
	@param	bb	Bounding box structure.
	@return	Check result (1=values available, 0=no values).
*/
EXTERN int
dkfig_tool_bb_have_y DK_PR((dk_fig_bb *bb));

/**	Add object bounding box to drawing bounding box.
	@param	dest	Destination bounding box (drawing).
	@param	src	Source bounding box (object).
*/
EXTERN void
dkfig_tool_bb_add DK_PR((dk_fig_bb *dest, dk_fig_bb *src));

/**	Calculate bounding box for an arc in mathematical
	coordinates (positive y grow upwards).
	@param	arc	Arc calculation structure.
*/
EXTERN void
dkfig_tool_arc_calc DK_PR((dk_fig_arc_calc *arc));

/**	Calculate bounding box for an arc in mathematical
	coordinates (positive y grow upwards).
	@param	o	Arc object.
	@param	d	Fig drawing structure.
*/
EXTERN void
dkfig_tool_bb_arc DK_PR((dk_fig_object *o, dk_fig_drawing *d));

/**	Calculate bounding box for polyline object.
	@param	o	Polyline object.
	@param	d	Fig drawing structure.
*/
EXTERN void
dkfig_tool_bb_polyline DK_PR((dk_fig_object *o, dk_fig_drawing *d));

/**	Calculate bounding box for an ellipse object.
	@param	o	Ellipse object.
	@param	d	Fig drawing structure.
*/
EXTERN void
dkfig_tool_bb_ellipse DK_PR((dk_fig_object *o, dk_fig_drawing *d));

/**	Calculate bounding box for a spline object.
	@param	o	Spline object.
	@param	d	Fig drawing structure.
*/
EXTERN void
dkfig_tool_bb_spline DK_PR((dk_fig_object *o, dk_fig_drawing *d));

/**	Distance between point (xp,yp) and line (0,0)--(x0,y0).
	@param	xp	Point x value.
	@param	yp	Point y value.
	@param	x0	Line endpoint x value.
	@param	y0	Line endpoint y value.
	@return	Distance between point and line.
*/
EXTERN double
dkfig_tool_dist_pt_to_line0 DK_PR((double xp,double yp,double x0,double y0));

/**	Calculate width and height of a rotated ellipse.
	The ellipse parameters rx, ry and phi0 (radians) are
	used as calculation input, output is stored in w and
	h (center to right, center to top).
	@param	e	Ellipse object.
*/
EXTERN void
dkfig_tool_rotated_ellipse DK_PR((dk_fig_rot_ellipse *e));

/**	Final step in spline processing: Shorten spline
	if necessary for arrowheads.
	@param	o	Spline object.
	@param	d	Fig drawing object.
	@param	c	Conversion job structure.
*/
EXTERN int
dkfig_tool_build_one_spline DK_PR((dk_fig_object *o, dk_fig_drawing *d, dk_fig_conversion *c));

/**	Initialize arc calculation structure.
	@param	c	Arc calculation structure.
*/
EXTERN void
dkfig_tool_null_arc_calc DK_PR((dk_fig_arc_calc *c));

/**	Invert the y values of the points.
	@param	c	Arc calculation structure.
*/
EXTERN void
dkfig_tool_invert_arc_calc DK_PR((dk_fig_arc_calc *c));

/**	Swap arrowheads on an arc.
	MetaPost requires to draw arcs in positive
	(counterclockwise) direction. If an arc is specified
	in clockwise direction to use
	"fill arrowhead p withcolor ...".
	@param	fpd	Fill, pattern and draw structure.
*/
EXTERN void
dkfig_tool_swap_arrows DK_PR((dk_fig_fpd *fpd));

/**	Final processing step on arcs: Shorten arc for arrowheads
	if necessary.
	@param	o	Arc object.
	@param	d	Fig drawing structure.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_tool_arc_complete DK_PR((dk_fig_object *o, dk_fig_drawing *d, dk_fig_conversion *c));

/**	Final processing step on polylines: Shorten line for
	arrowheads if necessary.
	@param	o	Polyline object.
	@param	d	Fig drawing structure.
	@param	c	Conversion job structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_tool_polyline_complete DK_PR((dk_fig_object *o, dk_fig_drawing *d, dk_fig_conversion *c));

/**	Fill color cell structure by retrieving data from
	Fig drawing color cell.
	@param	d	Fig drawing structure.
	@param	dcc	Destination color cell.
	@param	n	Color cell number.
*/
EXTERN void
dkfig_tool_fill_dcc DK_PR((dk_fig_drawing *d, dk_fig_dcc *dcc, int n));

/**	Correct a color cell structure for an object's
	area fill style (brightness, patterns...)
	@param	dcc	Color cell.
	@param	colno	Color number.
	@param	af	Object's area fill style.
*/
EXTERN void
dkfig_tool_correct_dcc DK_PR((dk_fig_dcc *dcc, int colno, int af));

/**	Check whether an object must be filled.
	@param	area_fill	Object's area fill style.
	@param	opt		Object options, checked for DKFIG_OPT_FILL_PATTERNS.
	@return	Test result (1=must fill, 0=no fill).
*/
EXTERN int
dkfig_tool_must_fill DK_PR((int area_fill, unsigned long opt));

/**	Check whether or not we need to draw fill patterns.
	@param	area_fill	Object's area fill style.
	@param	opt		Object options, checked for DKFIG_OPT_FILL_PATTERNS.
	@return	Test result (1=must pattern fill, 0=no fill).
*/
EXTERN int
dkfig_tool_must_pattern DK_PR((int area_fill, unsigned long opt));

/**	Write number encoded as string.
	@param	os	Output stream to write to.
	@param	no	Number value.
	@param	lgt	String length.
	@return	1 on success, 0 on errors.
*/
EXTERN int
dkfig_tool_num_as_string DK_PR((dk_stream_t *os, unsigned long no, size_t lgt));

/**	Remove temporary files used for special text processing.
	@param	c	Conversion job structure.
*/
EXTERN void
dkfig_tool_delete_tex_files DK_PR((dk_fig_conversion *c));

/**	Initialize NetPBM library.
	@param	c	Conversion job structure.
*/
EXTERN void
dkfig_tool_init_netpbm DK_PR((dk_fig_conversion *c));

#if defined(__cplusplus)
}
#endif






/* dkfigto2 */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigto2. */
#ifndef DKFIGTO2_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	Create storage container to store image width data.
	@return	Pointer to new storage on success, NULL on error.
*/
EXTERN dk_storage_t *
dkfig_tool2_image_width_storage DK_PR((void));

/**	Find width structure for given image attributes.
	@param	it	Image width data storage iterator.
	@param	width	Image width.
	@param	height	Image height.
	@param	colored	Flag: Colored.
	@param	seprgb	Flag: Separated color channels.
	@param	fl	Flip information.
	@return	Pointer to image width structure on success, NULL on error.
*/
EXTERN dkfig_bitmap_width *
dkfig_tool2_find_entry_for_width DK_PR((dk_storage_iterator_t *it, long width, long height, int colored, int seprgb, int fl));

/**	Add image width structure to storage.
	@param	st	Image width data storage.
	@param	it	Image width data storage iterator.
	@param	width	Image width.
	@param	height	Image height.
	@param	colored	Flag: Colored.
	@param	seprgb	Flag: Separated color channels.
	@param	fl	Flip information.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_tool2_add_image_width DK_PR((dk_storage_t *st, dk_storage_iterator_t *it, long width, long height, int colored, int seprgb, int fl));

/**	Error message: constant text, variable text and constant text.
	@param	c	Conversion job structure.
	@param	i1	First constant text index.
	@param	i2	Second constant text index.
	@param	fn	Variable text.
*/
EXTERN void
dkfig_tool2_combined_error_message DK_PR((dk_fig_conversion *c, size_t i1, size_t i2, char *fn));

/**	Error message: One constant text.
	@param	c	Conversion job structure.
	@param	i	Text index.
*/
EXTERN void
dkfig_tool2_simple_error_message DK_PR((dk_fig_conversion *c, size_t i));

/**	Report why the special comment can not be used.
	@param	c	Conversion job structure.
	@param	speccom	Special comment/option structure.
	@param	i	Failure reason.
*/
EXTERN void
dkfig_tool2_report_special_comment DK_PR((dk_fig_conversion *c, dk_fig_opt *speccom, int i));

/**	Report unused options.
	@param	c	Conversion job structure.
*/
EXTERN void
dkfig_tool2_report_unused_options DK_PR((dk_fig_conversion *c));

/**	Retrieve image size.
	@param	c	Conversion job structure.
	@param	fn	Image file name.
	@param	w	Pointer to width result variable.
	@param	h	Pointer to height result variable.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_tool2_get_image_size DK_PR((dk_fig_conversion *c, char *fn, unsigned long *w, unsigned long *h));

/**	Check whether or not an object is a background rectangle.
	@param	o	Object to inspect.
	@param	wbgr	Flag: Background rectangle uses color white instead of default.
	@return	1=true, 0=false.
*/
EXTERN int
dkfig_tool2_obj_is_bg_rect DK_PR((dk_fig_object *o, int wbgr));

/**	Error output function for the EPS driver.
	@param	oi	EPS output instruction structure.
	@param	i	Index of error message.
*/
EXTERN void
dkfig_tool2_eps_error_message DK_PR((dkfig_eps_output_instruction *oi, size_t i));

/**	Error output function for the SVG driver.
	@param	oi	SVG output instruction structure.
	@param	i	Index of error message.
*/
EXTERN void
dkfig_tool2_svg_error_message DK_PR((dkfig_svg_output_instruction *oi, size_t i));

/**	Progress message.
	@param	c	Conversion job structure.
	@param	i	Message index.
*/
EXTERN void
dkfig_tool2_simple_progress_message DK_PR((dk_fig_conversion *c, size_t i));

/**	Double round with specified number of digits.
	@param	v	Value to round.
	@param	c	Conversion job structure.
	@param	w	Rounding type (1=coordinate, 2=coordinate obtained from trigonometric calculation, default=color).
*/
EXTERN double dkfig_tool2_drd DK_PR((double v, dk_fig_conversion *c, int w));

/**	Retrieve UTF-8 enabled/disabled setting from LANG environment variable.
	@param	c	Conversion job structure.
	@return	1 on success (LANG environment var found), 0 on error.
*/
EXTERN int
dkfig_tool2_utf8_auto DK_PR((dk_fig_conversion *c));

/**	Enable/disable UTF-8.
	@param	c	Conversion job structure.
	@param	v	Text containing either a boolean or "auto".
	@param	allow_auto	Flag: Allow automatic detection of UTF-8 setting.
	@return	1 on success (tests found a result), 0 on error.
*/
EXTERN int
dkfig_tool2_set_utf8 DK_PR((dk_fig_conversion *c, char *v, int allow_auto));

/**	Write LaTeX representation of UTF-8 encoded string to
	output stream.
	@param	os	Output stream.
	@param	c	Conversion job structure.
	@param	t	UTF-8 encoded string.
*/
EXTERN void
dkfig_tool2_utf8_to_latex DK_PR((dk_stream_t *os, dk_fig_conversion *c, char *t));

/**	Log message consisting of 2 constant parts and a variable one.
	@param	c	Conversion job structure.
	@param	l	Log level (DK_LOG_LEVEL_xxx).
	@param	s1	Index of first (constant) part.
	@param	s2	Index of third (constant) part.
	@param	t	Second (variable) part of the message.
*/
EXTERN void
dkfig_tool2_msg3 DK_PR((dk_fig_conversion *c, int l, size_t s1, size_t s2, char *t));

/**	Log message.
	@param	c	Conversion job structure.
	@param	l	Log level (DK_LOG_LEVEL_xxx).
	@param	s1	Index of message text.
*/
EXTERN void
dkfig_tool2_msg1 DK_PR((dk_fig_conversion *c, int l, size_t s1));

/**	Create font mapping from file.
	@param	c	Conversion structure.
	@return	Pointer to font mapping on success, NULL on error.
*/
EXTERN dk_font_mapping_t *
dkfig_tool2_read_font_mapping DK_PR((dk_fig_conversion *c));

#if defined(__cplusplus)
}
#endif



/* dkfigei */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigei. */
#ifndef DKFIGEI_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	Delete image information used by EPS driver.
	@param	i	Image information.
*/
EXTERN void
dkfig_eps_image_info_delete DK_PR((dkfig_eps_image_info *i));

/**	Create image information used by EPS driver.
	@param	fn	Name of image file.
	@return	Pointer to new image information on success, NULL on error.
*/
EXTERN dkfig_eps_image_info *
dkfig_eps_image_info_new DK_PR((char *fn));

/**	Attach an information to an object.
	For each object showing an image we have to store
	width/height/colors/bits per components.
	@param	oi	EPS output instruction structure.
	@param	o	Current object.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfig_eps_attach_image_info DK_PR((dkfig_eps_output_instruction *oi, dk_fig_object *o));

/**	Check whether or not we can use separated color streams for
	an image.
	@param	w	Image width.
	@param	h	Image height.
	@return	1=Yes, we can 0=No we can't.
*/
EXTERN int
dkfig_ei_check_separated_strings DK_PR((unsigned long w, unsigned long h));

/**	Get image type.
	@param	fn	Image file name.
	@return	0=ps/eps, 1=png, 2=jpg, 3=NetPBM.
*/
EXTERN int
dkfig_ei_get_image_type DK_PR((char *fn));

#if defined(__cplusplus)
}
#endif





/* dkfigpi */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigpi. */
#ifndef DKFIGPI_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/**	Prepare to show image.
	@param	oi	PDF output instruction structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfigpi_prepare_image DK_PR((dkfig_pdf_output_instruction *oi));

#if defined(__cplusplus)
}
#endif



/* dkfigjav */

#if defined(EXTERN)
#undef EXTERN
#endif
/** Keyword "extern" for all modules except dkfigjav. */
#ifndef DKFIGJAV_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif

EXTERN
int dkfig_output_java DK_PR((dk_fig_conversion *c));

#if defined(__cplusplus)
}
#endif


/** Command: Run conversion
*/
#define	DKFIG_WHAT_RUN			0

/** Command: Show version.
*/
#define DKFIG_WHAT_VERSION		1

/** Command: Show help text.
*/
#define DKFIG_WHAT_HELP			2

/** Command: Save configuration settings.
*/
#define DKFIG_WHAT_CONFIGURE		4

/** Command: Delete saved configuration settings.
*/
#define DKFIG_WHAT_UNCONFIGURE		8

/** Command: Show current configuration.
*/
#define DKFIG_WHAT_SHOWCONF		16

/** Command: Error occured, show error message and help text.
*/
#define DKFIG_WHAT_ERROR		32

/** Command: Show configuration file contents.
*/
#define DKFIG_WHAT_CONFIG_FILE		64

/** Command: Save new configuration file contents.
*/
#define DKFIG_WHAT_WRITE_CONFIG_FILE	128

/** Command group: All application commands.
*/
#define DKFIG_WHAT_APP (\
DKFIG_WHAT_VERSION \
| DKFIG_WHAT_HELP \
| DKFIG_WHAT_CONFIGURE \
| DKFIG_WHAT_UNCONFIGURE \
| DKFIG_WHAT_SHOWCONF \
| DKFIG_WHAT_CONFIG_FILE \
| DKFIG_WHAT_WRITE_CONFIG_FILE \
)



/*
  The opt1 field.
*/

/** Options (1): Reset all options to built-in defaults.
*/
#define DKFIG_OPT_RESET			0x00000001UL

/** Options (1): Make mode.
*/
#define DKFIG_OPT_MAKE			0x00000002UL

/** Options (1): Use TeX, not LaTeX.
*/
#define DKFIG_OPT_OLD_TEX		0x00000004UL

/** Options (1): Do not embed fonts (MetaPost driver).
*/
#define DKFIG_OPT_NO_EMBEDDED_FONTS	0x00000008UL

/** Options (1): Use web-optimized color palette.
*/
#define DKFIG_OPT_WEB_PALETTE		0x00000010UL

/** Options (1): use coordiantes system settings from Fig file.
*/
#define DKFIG_OPT_USE_CS_SETTING	0x00000020UL

/** Options (1): Enlighten look (half line size only).
*/
#define DKFIG_OPT_ENLIGHTEN_LOOK	0x00000040UL

/** Options (1): Write additional comments to output file.
*/
#define DKFIG_OPT_VERBOSE_OUTPUT        0x00000080UL

/** Options (1): Remove zero linewidth borders.
*/
#define DKFIG_OPT_REMOVE_THIN_BORDERS	0x00000100UL

/** Options (1): Use MetaPost arrowheads (MetaPost driver).
*/
#define DKFIG_OPT_METAPOST_ARROWHEADS	0x00000200UL

/** Options (1): Allow use of fill patterns.
*/
#define DKFIG_OPT_FILL_PATTERNS		0x00000400UL

/** Options (1): Fill patterns of neighboured objects are contiguous.
*/
#define DKFIG_OPT_FILL_CONTIGOUS	0x00000800UL

/** Options (1): When running LaTeX, specify enable/disable write18 (unused).
*/
#define DKFIG_OPT_SPECIFY_WRITE18	0x00001000UL

/** Options (1): When running LaTeX, enable write18 (unused).
*/
#define DKFIG_OPT_ENABLE_WRITE18	0x00002000UL

/** Options (1): PS showpage operator allowed.
*/
#define DKFIG_OPT_USE_PS_SHOWPAGE	0x00004000UL

/** Options (1): After running TeX/LaTeX do not delete *.tex files (unused).
*/
#define DKFIG_OPT_KEEP_TEX_FILES	0x00008000UL

/** Options (1): Flag "NetPBM initialized".
*/
#define DKFIG_OPT_NETPBM_INITIALIZED	0x00010000UL

/** Options (1): Allow PS run-length compression.
*/
#define DKFIG_OPT_ALLOW_PSRL		0x00020000UL

/** Options (1): use separated channels for RGB.
*/
#define DKFIG_OPT_SEPARATED_RGB		0x00040000UL

/** Options (1): Remove border around bitmap images.
*/
#define DKFIG_OPT_REMOVE_BITMAP_BORDER	0x00080000UL

/** Options (1): Keep aspect ration when scaling bitmap images.
*/
#define DKFIG_OPT_KEEP_BITMAP_WH_RATIO	0x00100000UL

/** Options (1): Fill bitmap area using the fill color.
*/
#define DKFIG_OPT_FILL_BITMAP_AREA	0x00200000UL

/** Options (1): Use separated PostScript dictionary for image.
*/
#define DKFIG_OPT_BITMAP_DICTIONARY	0x00400000UL

/** Options (1): In PostScript suggest garbage collection after image.
*/
#define DKFIG_OPT_GARBAGE_COLLECTION	0x00800000UL

/** Options (1): Use CSS for styling SVG objects.
*/
#define DKFIG_OPT_USE_CSS		0x01000000UL

/** Options (1): For SVG produce partial file for embedding to XML files.
*/
#define DKFIG_OPT_SVG_EMBEDDED		0x02000000UL

/** Options (1): Show warning about special text multiple times
   (for each occurance).
*/
#define DKFIG_OPT_REPORT_MULTIPLE	0x04000000UL

/** Options (1): Ignore unknown paper size.
*/
#define DKFIG_OPT_IGNORE_UNKNOWN_PAPER	0x08000000UL

/** Options (1): Prepare SVG output for modifications (do not use CSS).
*/
#define DKFIG_OPT_PREPARE_FOR_MODS	0x10000000UL

/** Options (1): For dashed lines, a dot has linewidth length instead of 0.
*/
#define DKFIG_OPT_DP_DOT_LW		0x20000000UL

/** Options (1): Write TeX/LaTeX command to start of output (MetaPost driver).
*/
#define DKFIG_OPT_WR_TEX_COMMAND	0x40000000UL

/** Options (1): Do not draw the background rectangle.
*/
#define DKFIG_OPT_REMOVE_BG_RECTANGLE	0x80000000UL



/*
  The opt2 field.
*/
/** Options (2): Skip all texts (let TeX/LaTeX handle them).
*/
#define DKFIG_OPT_SKIP_ALL_TEXTS	0x00000001UL

/** Options (2): Use carriage/newline instead of simple newline.
*/
#define DKFIG_OPT_CRNL			0x00000002UL

/** Options (2): Do not compress graphics streams (PDF driver).
*/
#define DKFIG_OPT_PLAIN_TEXT_STREAMS	0x00000004UL

/** Options (2): Open output in full-screen mode (PDF driver).
*/
#define DKFIG_OPT_FULL_SCREEN		0x00000008UL

/** Options (2): Use nonzero fill method, not even-odd (PDF driver).
*/
#define DKFIG_OPT_FILL_NONZERO		0x00000010UL

/** Options (2): Used tiled patterns (PDF driver).
*/
#define DKFIG_OPT_PDF_PATTERN_TILE	0x00000020UL

/** Options (2): Flip objects diagonally.
*/
#define DKFIG_OPT_FLIP_DIAGONAL		0x00000040UL

/** Options (2): Set interpolation flag for bitmap images.
*/
#define DKFIG_OPT_INTERPOLATE_IMAGES	0x00000080UL

/** Options (2): Write full LaTeX source instead of fragment (TeX driver).
*/
#define DKFIG_OPT_FULL_TEX_FILE		0x00000100UL

/** Options (2): Background rectangle uses white color, not default.
*/
#define DKFIG_OPT_WHITE_BGRECT		0x00000200UL

/** Options (2): Use tex command, not latex (TeX driver).
*/
#define DKFIG_OPT_TEX_DRIVER		0x00000400UL

/** Options (2): Use setpagedevice instruction (PS driver).
*/
#define DKFIG_OPT_SETPAGEDEVICE		0x00000800UL

/** Options (2): A unit for font sizes was specified, use it (SVG driver).
*/
#define DKFIG_OPT_SVG_FONTSIZE_UNIT	0x00001000UL

/** Options (2): Text is UTF-8 encoded.
*/
#define DKFIG_OPT_UTF_8			0x00002000UL

/** Options (2): Check LANG environment variable for UTF-8 setting.
*/
#define DKFIG_OPT_UTF_AUTO		0x00004000UL

/** Options (2): Image contains alpha channel data.
*/
#define DKFIG_OPT_HAVE_ALPHA_CHANNEL	0x00008000UL

/** Options (2): Suppress info message about alpha channel in bitmap image.
*/
#define DKFIG_OPT_SUPPRESS_ALPHA_INFO	0x00010000UL

/** Options (2): Write PDF page attributes instruction (TeX driver).
*/
#define DKFIG_OPT_PDF_PAGE_ATTRIBUTES	0x00020000UL

/** Options (2): Fill bounding box white before drawing.
*/
#define DKFIG_OPT_FILL_BOUNDING_BOX	0x00040000UL

/** Options (2): Colored output, not grayscale.
*/
#define DKFIG_OPT_COLOR			0x00080000UL

/** Options (2): Align graphics on paper.
 */
#define DKFIG_OPT_ALIGN_ON_PAPER	0x00100000UL

/** Options (2): Use GS fonts.
 */
#define DKFIG_OPT_USE_GS_FONTS		0x00200000UL

/** Options (2): Use dklibsj Java libraries.
 */
#define DKFIG_OPT_DKLIBSJ_LIBRARIES	0x00400000UL

/** Options (2): Use dklibsj Java libraries for webstart.
 */
#define DKFIG_OPT_DKLIBSJ_WEBSTART	0x00800000UL

/** Options (2): Correct open splines.
 */
#define DKFIG_OPT_CORRECT_OPEN_SPLINES	0x01000000UL

/** Options (2): Write escape sequences.
*/
#define DKFIG_OPT_JAVA_UNICODE		0x02000000UL

/** Horizontal alignment: left. */
#define DKFIG_ALIGN_H_LEFT		0

/** Horizontal alignment: centered. */
#define DKFIG_ALIGN_H_CENTERED		1

/** Horizontal alignment: right. */
#define DKFIG_ALIGN_H_RIGHT		2

/** Vertical alignment: top. */
#define DKFIG_ALIGN_V_TOP		0
/** Vertical alignment: centered. */
#define DKFIG_ALIGN_V_CENTERED		1

/** Vertical alignment: bottom. */
#define DKFIG_ALIGN_V_BOTTOM		2

/** Text handling: Use font as specified in Fig file.
*/
#define DKFIG_TH_FONT_FIG		 1

/** Text handling: Use font with same features as font in Fig file
    (roman/sans-serif/typewriter, normal/bold, upright/italic).
*/
#define DKFIG_TH_FONT_SIMILAR		 2

/** Text handling: Leave all decisions about the font up to LaTeX/TeX.
*/
#define DKFIG_TH_FONT_TEX		 4

/** Text handling: Mask to find DKFIG_TH_FONT_FIG, DKFIG_TH_FONT_SIMILAR
    or DKFIG_TH_FONT_TEX.
*/
#define DKFIG_TH_FONT_MASK		 7

/** Text handling: Leave font size decision up to LaTeX/TeX.
*/
#define DKFIG_TH_SIZE_FIG		 8

/** Text handling: LaTeX/TeX font name, not PS ... font name.
*/
#define DKFIG_TH_HANDLING		16

/** Text handling: Enclose text label in \mbox{} instruction.
*/
#define DKFIG_TH_MBOX			32



/** Base driver: MetaPost.
*/
#define DKFIG_BD_MP	0

/** Base driver: MPS (unused).
*/
#define DKFIG_BD_MPS	1

/** Base driver: EPS.
*/
#define DKFIG_BD_EPS	2

/** Base driver: SVG.
*/
#define DKFIG_BD_SVG	3

/** Base driver: PDF.
*/
#define DKFIG_BD_PDF	4



/* two last significant bits=0: use default font, no explicit switching */

/** Font feature: Roman.
*/
#define DKFIG_FONT_RM	1

/** Font feature: Sans-serif.
*/
#define DKFIG_FONT_SF	2

/** Font feature: Typewriter.
*/
#define DKFIG_FONT_TT	3

/** Font feature: Bold.
*/
#define DKFIG_FONT_BD	4

/** Font feature: Italic.
*/
#define DKFIG_FONT_IT	8

/** Font feature mask: find roman/sans-serif/italic.
*/
#define DKFIG_FONT_FAMILY_MASK (DKFIG_FONT_TT|DKFIG_FONT_SF|DKFIG_FONT_RM)




/** Error handling: No error occured.
*/
#define DKFIG_ERROR_NONE	0

/** Error handling: Not enough memory.
*/
#define DKFIG_ERROR_MEMORY	1

/** Error handling: Syntax error.
*/
#define DKFIG_ERROR_SYNTAX	2

/** Error handling: Numeric error occured.
*/
#define DKFIG_ERROR_MATH	3



/** Very special case: An X-Spline is represented by one Bezier spline
    segment which must be shortened for arrowheads.
*/
#define DKFIG_SPLINE_FLAG_ONE_PARTIAL	1



/** Number of different patterns used by the Fig file format.
*/
#define DKFIG_NUMBER_OF_PATTERNS 22



/** Epsilon for numeric calculations.
*/
#define DKFIG_EPSILON		0.0001



/** Default DSC level.
*/
#define DKFIG_DSC_OLD		2



/** SVG format version 1.0
*/
#define DKFIG_SVG_VERS_10	0

/** SVG format version 1.1
*/
#define DKFIG_SVG_VERS_11	1

/** SVG format version 1.2
*/
#define DKFIG_SVG_VERS_12	2



/** SVG viewport unit: points.
*/
#define DKFIG_SVG_VP_POINTS	0

/** SVG viewport unit: pixels.
*/
#define DKFIG_SVG_VP_PIXELS	1

/** SVG viewport unit: inches.
*/
#define DKFIG_SVG_VP_INCHES	2



/** SVG style flag: Stroke setup defined.
*/
#define DKFIG_SVG_FL_STROKE	0x00000001UL

/** SVG style flag: Fill setup defined.
*/
#define DKFIG_SVG_FL_FILL	0x00000002UL

/** SVG style flag: Text setup defined.
*/
#define DKFIG_SVG_FL_TEXT	0x00000004UL

/** SVG style flag: Enlighten look.
*/
#define DKFIG_SVG_FL_ENLIGHTEN	0x00000008UL




/** PDF procsets: Text.
*/
#define DKFIG_PDF_PROCSET_TEXT	1

/** PDF procsets: Grayscaled images.
*/
#define DKFIG_PDF_PROCSET_IMGG	2

/** PDF procsets: Colored images.
*/
#define DKFIG_PDF_PROCSET_IMGC	4



/** PS driver: Always re-encode font.
*/
#define DKFIG_ALWAYS_RE_ENCODE  0



/** Program name.
*/
#define DKFIG_PROGRAM_NAME	"fig2vect"



/** Drivers: MetaPost.
*/
#define DKFIG_DRIVER_MP		0

/** Drivers: Multi-MetaPost.
*/
#define DKFIG_DRIVER_MMP	1

/** Drivers: EPS.
*/
#define DKFIG_DRIVER_EPS	2

/** Drivers: SVG.
*/
#define DKFIG_DRIVER_SVG	3

/** Drivers: LaTeX/TeX.
*/
#define DKFIG_DRIVER_TEX	4

/** Drivers: Bounding box.
*/
#define DKFIG_DRIVER_BB		5

/** Drivers: PDF.
*/
#define DKFIG_DRIVER_PDF	6

/** Drivers: PS.
*/
#define DKFIG_DRIVER_PS		7

/** Drivers: Java
 */
#define DKFIG_DRIVER_JAVA	8



/** Unit length: pt per inch.
*/
#define DKFIG_PT_PER_INCH	72.72

/** Unit length: bp per inch.
*/
#define DKFIG_BP_PER_INCH	72.0

#endif

