/*
Copyright (c) 2004-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	dkxsp.h	X-spline module.

To calculate x, y, dx/dt and dy/dt for a given t between the control
points B and C we also need values from the control points A and D
next to B and C. See the section "X-Splines" in the chapter
"Technical Details" in fig2vect.pdf for calculation details.

First use dkxsp_reset() to reset the structure.

Next use dkxsp_setA(), dkxsp_setB(), dkxsp_setC(), dkxsp_setD()
and dkxsp_set_pos() to specify the control points and the position
of the segment in the spline.

Use dkxsp_step1() to find the control function types and calculate
the control function coefficients.

Now you can use dkxsp_step2() to calculate x, y, dx/dt and dy/dt
values for a given t in the range \f$0\leq{}t\leq{}1\f$.
The dkxsp_step2() function can be used multiple times if
necessary.
*/



#ifndef DK_XSP_INC



/** Protection from multiple inclusions. */
#define DK_XSP_INC 1



#include <dk.h>



/**	X-spline calculation structure.
*/
typedef struct {
  int		useA;	/**< Flag: Use function from control point A. */
  int		useB;	/**< Flag: Use function from control point B. */
  int		useC;	/**< Flag: Use function from control point C. */
  int		useD;	/**< Flag: Use function from control point D. */
  int		ghA;	/**< Flag: Use g-h-blend from point A. */
  int		ghB;	/**< Flag: Use g-h-blend from point B. */
  int		ghC;	/**< Flag: Use g-h-blend from point C. */
  int		ghD;	/**< Flag: Use g-h-blend from point D. */
  double	sA;	/**< s-value at A. */
  double	sB;	/**< s-value at B. */
  double	sC;	/**< s-value at C. */
  double	sD;	/**< s-value at D. */
  double	tA;	/**< \f$T_A^+\f$ */
  double	tB;	/**< \f$T_B^+\f$ */
  double	tC;	/**< \f$T_C^-\f$ */
  double	tD;	/**< \f$T_D^-\f$ */
  double	pA;	/**< p-value at A. */
  double	pB;	/**< p-value at B. */
  double	pC;	/**< p-value at C. */
  double	pD;	/**< p-value at D. */
  double	qA;	/**< q-value at A. */
  double	qB;	/**< q-value at B. */
  double	qC;	/**< q-value at C. */
  double	qD;	/**< q-value at D. */
  double	xA;	/**< x-value at A. */
  double	xB;	/**< x-value at B. */
  double	xC;	/**< x-value at C. */
  double	xD;	/**< x-value at D. */
  double	yA;	/**< y-value at A. */
  double	yB;	/**< y-value at B. */
  double	yC;	/**< y-value at C. */
  double	yD;	/**< y-value at D. */
  double	dudtA;	/**< du_A/dt. */
  double	dudtB;	/**< du_B/dt. */
  double	dudtC;	/**< du_C/dt. */
  double	dudtD;	/**< du_D/dt. */
  double	dduA;	/**< d^2u_A/dt. */
  double	dduB;	/**< d^2u_B/dt. */
  double	dduC;	/**< d^2u_C/dt. */
  double	dduD;	/**< d^2u_D/dt. */
  double	t;	/**< t for calculation.*/
  double	x;	/**< x value. */
  double	y;	/**< y value. */
  double	ddtx;	/**< dx/dt. */
  double	ddty;	/**< dy/dt. */
  double	fA;	/**< Result from control function A. */
  double	fB;	/**< Result from control function B. */
  double	fC;	/**< Result from control function C. */
  double	fD;	/**< Result from control function D. */
  double	uA;	/**< u for control function A. */
  double	uB;	/**< u for control function B. */
  double	uC;	/**< u for control function C. */
  double	uD;	/**< u for control function D. */
  double	ddtA;	/**< First derivative of control function A. */
  double	ddtB;	/**< First derivative of control function B. */
  double	ddtC;	/**< First derivative of control function C. */
  double	ddtD;	/**< First derivative of control function D. */
  size_t	current;	/**< Index of current control point. */
  size_t	max;		/**< Number of control points. */
  int		isclosed;	/**< Flag: Closed spline. */
  int		docor;		/**< Flag: Do corrections on open splines. */
} dk_xspline_t;

#if defined(EXTERN)
#undef EXTERN
#endif
/** Access function in external module. */
#if defined(DK_XSP_C)
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif



#if defined(__cplusplus)
extern "C" {
#endif



/**	Reset all components.
	@param	s	X-spline calculation structure to reset.
*/
EXTERN void
dkxsp_reset DK_PR((dk_xspline_t *s));

/**	Set values for point A.
	@param	s	X-spline calculation structure.
	@param	sk	s-value.
	@param	x	x-value.
	@param	y	y-value.
*/
EXTERN void
dkxsp_setA DK_PR((dk_xspline_t *s, double sk, double x, double y));

/**	Set values for point B.
	@param	s	X-spline calculation structure.
	@param	sk	s-value.
	@param	x	x-value.
	@param	y	y-value.
*/
EXTERN void
dkxsp_setB DK_PR((dk_xspline_t *s, double sk, double x, double y));

/**	Set values for point C.
	@param	s	X-spline calculation structure.
	@param	sk	s-value.
	@param	x	x-value.
	@param	y	y-value.
*/
EXTERN void
dkxsp_setC DK_PR((dk_xspline_t *s, double sk, double x, double y));

/**	Set values for point D.
	@param	s	X-spline calculation structure.
	@param	sk	s-value.
	@param	x	x-value.
	@param	y	y-value.
*/
EXTERN void
dkxsp_setD DK_PR((dk_xspline_t *s, double sk, double x, double y));

/**	Set spline position.
	@param	s	X-spline calculation structure.
	@param	c	Index of current segment (0 is first, m-2 is last).
	@param	m	Number of control points.
	@param	isc	Flag: Closed spline (1) or open spline (0).
*/
EXTERN void
dkxsp_set_pos DK_PR((dk_xspline_t *s, size_t c, size_t m, int isc));

/**	Step 1: Calculations with no t involved.
	- Check where to interpolate or to approximate.
	- Calculate coefficients for control functions.
	@param	s	X-spline calculation structure.
*/
EXTERN void
dkxsp_step1 DK_PR((dk_xspline_t *s));

/**	Step 2: Calculate x, y, dx/dt and dy/dt for a
	given t.
	@param	s	X-spline calculation structure.
	@param	t	t-value (0<=t<=1).
	@return	1 on success, 0 on error (mathmematical error).
*/
EXTERN int
dkxsp_step2 DK_PR((dk_xspline_t *s, double t));
#if defined(__cplusplus)
}
#endif

#endif

