/* 
 * ThatIP-J Copyright 2003 Algenta Technologies All rights reserved.
 * Use is subject to license terms.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * $Id: CrossIP.java,v 1.3 2004/02/09 19:10:30 dan Exp $
 * 
 * $Log: CrossIP.java,v $
 * Revision 1.3  2004/02/09 19:10:30  dan
 * Protocol v 1.1
 *
 * Revision 1.2  2004/01/25 01:34:20  dan
 * jdk 1.2 and 1.3 compat
 *
 * Revision 1.1  2003/12/08 11:48:05  dan
 * CrossIP initial commit
 *
 */
package com.algenta.crossip;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.Socket;
import java.net.UnknownHostException;
import java.security.NoSuchAlgorithmException;

import com.algenta.thatip.Record;
import com.algenta.thatip.ThatIP;
import com.algenta.thatip.ThatIPConstants;
import com.algenta.thatip.ThatIPServer;
import com.algenta.thatip.ThatIPServerException;
import com.algenta.thatip.ThatIPServerUpdateException;
import com.algenta.util.MD5Crypt3;
import com.algenta.util.StringUtils;

/**
 * @author Dan Smith (<a href="mailto:dan (is at) algenta (dot) com">dan (is at) algenta (dot) com</a>)
 *
 */
public class CrossIP implements CrossIPConstants, ThatIPConstants {
	
	private static Config conf = new Config();
	private static Actions actions = new Actions();
	
	public static void main(String[] args) {
		// load or create configuration file
		if(args != null && args.length > 0) {
			try {
				conf = parseConfigFile(args[0], conf);
			} catch (ConfigFileException e) {
				System.err.println("Problem loading config file: " + args[0] );
				System.err.println(e.getMessage());
				System.exit(1);
			}
		} else {
			try {
				conf = launchWizard(conf);
			} catch (CrossIPException e) {
				System.err.println("Problem running wizard: ");
				System.err.println(e.getMessage());
				System.exit(1);
			}
		}
		System.out.println("Loaded Config File: " + conf.getConfigFile());
		
		// load host update file
		try {
			actions = parseUpdateFile(conf.getUpdateFile(),actions);
		} catch (ConfigFileException e) {
			System.err.println("Problem loading update file: " + conf.getUpdateFile() );
			System.err.println(e.getMessage());
			System.exit(1);
		}
		System.out.println("Loaded Update File: " + conf.getUpdateFile());
		
		Record[] records = conf.getRecords();
		String[] groups = ThatIP.getGroups(records,true);
		
		// check ip addresses
		String ip = "";
		if (conf.getNat() != null && conf.getNat().compareToIgnoreCase("no") == 0) {
			ip = getCurrentIP();
			System.out.println("Found local IP Address: " + ip);
			System.out.println("If this IP is incorrect, please specify you are behind");
			System.out.println("a NAT or router to do advanced tests.");
		}
		
		// update groups if needed
		String[] updates = actions.clientIP(groups,ip);
		try {
			performUpdates(updates,ip);
		} catch (CrossIPException e1) {
			System.err.println("Problem updating client determined hostnames" );
			System.err.println(e1.getMessage());
			System.exit(1);
		}
		
		try {
			performUpdates(actions.autoIP(groups));
		} catch (CrossIPException e2) {
			System.err.println("Problem updating server determined hostnames" );
			System.err.println(e2.getMessage());
			System.exit(1);
		}
		
		// write host update file
		try {
			commitActions();
		} catch (ConfigFileException e3) {
			System.err.println("Problem writing update file" );
			System.err.println(e3.getMessage());
		}
		// write config file if needed
		if (conf.isUpdated()){
			try {
				commitConfig();
			} catch (ConfigFileException e4) {
				System.err.println("Problem writing update file" );
				System.err.println(e4.getMessage());
			}
		}
		System.out.println("Wrote configurations");
		
		
	}

	
	/**
	 * @return
	 */
	private static String getCurrentIP() {
		InetAddress i;
		try {
			i = new Socket("www.internic.net", 80).getLocalAddress();
		} catch (UnknownHostException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return "";
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return "";
		}
		return i.getHostAddress();
	}


	private static Config launchWizard(Config c)  throws CrossIPException {
		
		// launch swing gui or text mode
		
		
		return launchTextWizard(c);
	}
	
	private static Config launchTextWizard(Config c) throws CrossIPException {
		
		System.out.println("CrossIP v." + CROSSIP_VERSION);
		System.out.println("CrossIP uses ThatIP-J v." + THATIP_VERSION +" p." + PROTOCOL_VERSION);
		System.out.println("Copyright (C) 2003-2004 Algenta Technologies, LLC.");
		System.out.println("Written by Dan Smith and Jeremy Iverson");
		System.out.println("http://www.algenta.com - http://www.thatip.com");
		System.out.println("==============================================");
		System.out.println("Welcome to the CrossIP console setup wizard!");
		System.out.println("It will guide you through the setup for CrossIP");
		System.out.println("for use with a ThatIP compatible server.");
		c = textQuestions(c);
		System.out.print("Hashing password...");
		try {
			c.setPassword(MD5Crypt3.MD5Hash(c.getPassword().getBytes()));
		} catch (NoSuchAlgorithmException e) {
			System.out.println("Failed, MD5 not present");
			throw new CrossIPException("Your version of java does not include MD5",e);
		}
		System.out.println("Done!");
		conf = c;
		System.out.println("Logging into ThatIP server at " + conf.getServer());
		performUpdates(new String[0]); // just get the host records
		System.out.println("Retrieved domain records.");
		conf.setUpdated(true);
		actions = textGroups(actions);
		System.out.println("Writing config file.");
		try {
			commitConfig();
			System.out.println("Writing update file.");
			commitActions();
		} catch (ConfigFileException e1) {
			throw new CrossIPException("Unable to write file",e1);
		}		
		return c;
	}
	
	private static void commitConfig() throws ConfigFileException {
		writeFile(conf.getConfigFile(),conf.toString());
	}
	
	private static void commitActions() throws ConfigFileException {
		writeFile(conf.getUpdateFile(),actions.toString());
		
	}
	
	static public void writeFile(String path, String contents)throws ConfigFileException {
		File f = new File(path);
		BufferedWriter output = null;
		try {
			if (!f.exists()) {
				f.createNewFile();	
			}
			if (!f.isFile()) {
				throw new ConfigFileException("This is a directory: " + f);
			}
			if (!f.canWrite()) {
				throw new ConfigFileException("Cannot write to the file: " + f);
			}
			output = new BufferedWriter(new FileWriter(f));
			output.write(contents);
			output.flush();
			output.close();
		} catch (IOException e) {			
			throw new ConfigFileException("Cannot create file: " + f,e);
		} finally {
			if (output != null)
			try {
				output.close();
			} catch (IOException e1) {
				throw new ConfigFileException("Cannot close file: " + f,e1);
			}
		}

	}

	
	private static void performUpdates(String[] groups) throws CrossIPException {
		performUpdates(groups,null);	
	}
	private static void performUpdates(String[] groups, String ipAddr) throws CrossIPException {
		
		try {
			ThatIP.setClientName("CrossIP");
			ThatIP.setClientVersion(CROSSIP_VERSION);
			ThatIPServer server = ThatIP.getServer();
			server.connect(conf.getServer(),conf.getPort());
			int serial = server.loginMD5(conf.getUsername(),"thatip.com",conf.getPassword());
			if (serial > conf.getSerial()) {
				Record[] records = server.getRecords();
				conf.clearRecords();
				conf.addRecord(records);
				conf.setSerial(serial);
				conf.setUpdated(true);
			}
			if(ipAddr == null) {
				System.out.print("Updating groups from client ip: " );
			} else {
				System.out.print("Updating groups from server ip: " );
			}
			for(int i = 0; i < groups.length; ++i) {
				if(i > 0) {
					System.out.print(", ");
				}
				System.out.print(groups[i]);
			}
			System.out.println();
			
			if (ipAddr == null) {
				actions.clientUpdate(groups,server.update(groups));
			} else {
				actions.autoUpdate(groups,server.update(groups,ipAddr));
			}
			server.quit();
			
		} catch (UnknownHostException e1) {
			throw new CrossIPException("Unable to find host " + conf.getServer(),e1);
		} catch (IOException e1) {
			throw new CrossIPException("A network error occurred",e1);
		} catch (ThatIPServerException e1) {
			throw new CrossIPException(e1.getMessage(),e1);
		} catch (NoSuchAlgorithmException e) {
			throw new CrossIPException("Your version of java does not include MD5",e);
		} catch (ThatIPServerUpdateException e) {
			throw new CrossIPException(e.getErrorMessage(),e);
		}
	}
	private static Actions textGroups(Actions a) throws CrossIPException {	
		System.out.println("\nThatIP uses a concept called groups, when you update");
		System.out.println("a group all domains you have placed in it are updated to");
		System.out.println("the same ip address.  You can now select which groups");
		System.out.println("you would like updated from this computer");
		System.out.println("* Press n if you do not want to update the group");
		System.out.println("* Type auto if you want the thatip server to determine the ip");
		System.out.println("* Type client if you want the client to determine the ip");
		Record[] records = conf.getRecords();
		String[] groups = ThatIP.getGroups(records);
		
		System.out.print("Press <Enter> to begin.");
		BufferedReader r = new BufferedReader(new InputStreamReader(System.in));
		String line = "";
		try {
			line = r.readLine();
			for(int i = 0; i < groups.length; ++i) {
				System.out.println("Group: " + groups[i]);	
				for(int j = 0; j < records.length; ++j) {
					if (records[j].getGroup().equalsIgnoreCase(groups[i])) {
						System.out.println(" + " + records[j].toString());
					}
				}	
				boolean correct = false;
				while (!correct) {
					System.out.print("Update this group from this machine? (n,auto,client) :");
					line = r.readLine();
					if (line.length() > 0) {
						if (line.equalsIgnoreCase("n")) {
							correct = true;
						}else if( line.equalsIgnoreCase("auto")) {
							correct = true;
							a.autoUpdate(groups[i],"");
						}else if( line.equalsIgnoreCase("client")) {
							correct = true;
							a.clientUpdate(groups[i],"");
						} else {
							System.out.println("Please enter n, auto, or client");
						}
					}
				}
			}
		} catch (IOException e) {
			throw new CrossIPException("Unable to read console input",e);
		}
		return a;
	}
	private static Config textQuestions(Config c) throws CrossIPException {
		System.out.print("Press <Enter> to begin.");
		BufferedReader r = new BufferedReader(new InputStreamReader(System.in));
		try {
			String line = r.readLine();
			
			System.out.println("Please enter your...");
			System.out.print("ThatIP account name " + textDefault(c.getUsername()));
			line = r.readLine();
			if (line.length() > 0) {
				c.setUsername(line);
			}

			System.out.print("ThatIP account password " + textDefault(c.getPassword()));
			line = r.readLine();
			if (line.length() > 0) {
				c.setPassword(line);
			}

			System.out.print("Config file pathname " + textDefault(c.getConfigFile()));
			line = r.readLine();
			if (line.length() > 0) {
				c.setConfigFile(line);
			}

			System.out.print("Update file pathname " + textDefault(c.getUpdateFile()));
			line = r.readLine();
			if (line.length() > 0) {
				c.setUpdateFile(line);
			}

			System.out.print("ThatIP server hostname " + textDefault(c.getServer()));
			line = r.readLine();
			if (line.length() > 0) {
				c.setServer(line);
			}
			
			boolean number = false;
			while (!number) {
				System.out.print("ThatIP server port " + textDefault(c.getPort()));
				line = r.readLine();
				if (line.length() > 0) {
					try {
						c.setPort(Integer.parseInt(line));
						number = true;
					} catch (NumberFormatException e1) {
						System.out.println("Please enter an integer");
					}
				} else {
					number = true;
				}
			}

			boolean nat = false;
			while (!nat) {
				System.out.print("Are you behind a NAT or router (y/n) " + textDefault(c.getNat()));
				line = r.readLine();
				if (line.length() > 0) {
					if (line.equalsIgnoreCase("n") || line.equalsIgnoreCase("y")) {
						c.setNat(line);
						nat = true;
					} else {
						System.out.println("Please enter y or n");
					}
				} else {
					nat = true;
				}
			}
			

			System.out.println("==============================================");
			System.out.println(CONF_USERNAME + " " + c.getUsername());
			System.out.println(CONF_PASSWORD + " " + c.getPassword()) ;
			System.out.println(CONF_UPDATEFILE + " " + c.getUpdateFile());
			System.out.println(CONF_SERVER + " " + c.getServer());
			System.out.println(CONF_PORT + " " + c.getPort());
			System.out.println(CONF_NAT + " " + c.getNat());			
			System.out.println("==============================================");
			
			boolean correct = false;
			while (!correct) {
				System.out.print("Are these values correct? (y/n) :");
				line = r.readLine();
				if (line.length() > 0) {
					if (line.equalsIgnoreCase("n")) {
						correct = true;
						c = textQuestions(c);
					}else if( line.equalsIgnoreCase("y")) {
						correct = true;
					} else {
						System.out.println("Please enter y or n");
					}
				}
			}
														
		} catch (IOException e) {
			throw new CrossIPException("Unable to read console input",e);
		}
		return c;	
	}
	
	private static String textDefault(String s) {
		if (s == null || s.length() == 0) {
			return "[] :";	
		} else {
			return "[" + s + "] :";
		}
	}

	private static String textDefault(int i) {
		if (i == 0) {
			return "[] :";	
		} else {
			return "[" + i + "] :";
		}
	}
	
	private static Actions parseUpdateFile(String pathname, Actions a) throws ConfigFileException {
		File f= new File(pathname);
		if (f == null ) {
			throw new ConfigFileException("Update file is not set");
		}
		else if(!f.exists()) {
			throw new ConfigFileException("Update file does not exist, check pathname?");
		}
		else if(!f.isFile()){
			throw new ConfigFileException("Update file cannot be a directory");
		} 
		else if(!f.canRead()) {
			throw new ConfigFileException("Update file cannot be read, check permissions?");
		}
		BufferedReader input = null;
		try {
			input =  new BufferedReader( new FileReader(f) );
			String line = "";
			while (( line = input.readLine()) != null){
				if(line.startsWith("#")) {
					continue;
				} else {
					a.load(line);
				}
			}
		} catch (FileNotFoundException e) {
			throw new ConfigFileException("Update file not found",e);
		} catch (IOException e) {
			throw new ConfigFileException("Update file IO problem",e);
		} finally {
			if (input != null) {
				try {
					input.close();
				} catch (IOException e1) {
					throw new ConfigFileException("Could not close update file",e1);
				}
			}	
		
		}
		return a;
	}
	
	/**
	 * parse the configuration file
	 * @param string path to the config file
	 */
	private static Config parseConfigFile(String pathname, Config c) throws ConfigFileException {
		File config = new File(pathname);
		if (config == null ) {
			throw new ConfigFileException("Config file is not set");
		}
		else if(!config.exists()) {
			throw new ConfigFileException("Config file does not exist, check pathname?");
		}
		else if(!config.isFile()){
			throw new ConfigFileException("Config file cannot be a directory");
		} 
		else if(!config.canRead()) {
			throw new ConfigFileException("Config file cannot be read, check permissions?");
		}
		BufferedReader input = null;
		try {
			input =  new BufferedReader( new FileReader(config) );
			String line = null;
			int count = 1;
			while (( line = input.readLine()) != null){
				if(line.startsWith("#")) {
					continue;
				}
				if(line.startsWith("200")) {
					Record record = new Record(line);
					c.addRecord(record);
					continue;
				}
				// name value pairs
				String pair[] = StringUtils.Split(line," ");
				if(pair.length != 2){
					throw new ConfigFileException("Config file error on line " + count);
				}
				if (pair[0].equalsIgnoreCase(CONF_SERIAL)) {
					try {
						c.setSerial(Integer.parseInt(pair[1]));
					} catch (NumberFormatException e1) {
						throw new ConfigFileException("Config file error parsing serial on line " + count);
					}
				} else if(pair[0].equalsIgnoreCase(CONF_USERNAME)) {
					c.setUsername(pair[1]);
				} else if(pair[0].equalsIgnoreCase(CONF_PASSWORD)) {
					c.setPassword(pair[1]);
				} else if(pair[0].equalsIgnoreCase(CONF_UPDATEFILE)) {
					c.setUpdateFile(pair[1]);
				} else if(pair[0].equalsIgnoreCase(CONF_SERVER)) {
					c.setServer(pair[1]);
				} else if(pair[0].equalsIgnoreCase(CONF_NAT)) {
					c.setNat(pair[1]);
				} else if(pair[0].equalsIgnoreCase(CONF_PORT)) {
					try {
						c.setPort(Integer.parseInt(pair[1]));
					} catch (NumberFormatException e1) {
						throw new ConfigFileException("Config file error parsing port on line " + count);
					}	
				}
				count++;
			}
		} catch (FileNotFoundException e) {
			throw new ConfigFileException("Config file not found",e);
		} catch (IOException e) {
			throw new ConfigFileException("Config file IO problem",e);
		} finally {
			if (input != null) {
				try {
					input.close();
				} catch (IOException e1) {
					throw new ConfigFileException("Could not close config file",e1);
				}
			}	
		
		}
		return c;
	}
}
