package net.psammead.util.ui;

import java.awt.*;

/**
 * a simple interpreter for a small DSL to configure GridBagConstraints objects.
 * 
 * the DSL consists of
 * 
 *	at		x/y						(int cells or "relative")
 *	size	width/height			(int cells, "relative" or "remainder")
 *	weight	horizontal/vertical		(float 0..1, "minimum", "medium" or "maximum")
 *	ipad	width/height			(int pixels)
 *	insets	top/left/bottom/right	(int pixels)
 *	anchor	anchoring				absolute values:
 *										center|
 *										north|south|east|west|
 *										northeast|northwest|southeast|southwest|
 *									relative values:
 *										page-start|page-end|line-start|line-end|
 *										first-line-start|first-line-end|
 *										last-line-start|last-line-end
 *	fill	filling					none|horizontal|vertical|both
 *
 */
public class GridBagUtil {
	/** a Syntax error occured while parsing */
	public static class SyntaxError extends RuntimeException {
		SyntaxError(String message) { super(message); }
		SyntaxError(String message, Throwable cause) { super(message, cause); }
	}
	
	/** resets GridBagConstraints to their default value */
	public static void defaults(GridBagConstraints c) {
		c.gridx			= GridBagConstraints.RELATIVE;
		c.gridy			= GridBagConstraints.RELATIVE;
		c.gridwidth		= 1;
		c.gridheight	= 1;
		c.weightx		= 0;
		c.weighty		= 0;
		c.anchor		= GridBagConstraints.CENTER;	
		c.fill			= GridBagConstraints.NONE;
		c.insets		= new Insets(0,0,0,0);
		c.ipadx			= 0;
		c.ipady			= 0;
	}
	
	/** creates new GridBagConstraints (with default values!) and modifies them with a call to the configure method */
	public static GridBagConstraints constraints(String code) {
		GridBagConstraints c	= new GridBagConstraints();
		constrain(c, code);
		return c;
	}
	
	/** <code>constrain(gridBagConstraints, "at 0/0 size 1/1 weight 1/1 anchor north fill both insets 5/5/5/5")</code> */
	public static void constrain(GridBagConstraints c, String code) {
		// TODO: allow "default" as value for every parameter

		String[]	tokens	= code.trim().split("\\s+");
		if (tokens.length % 2 != 0) 
				throw new SyntaxError("illegal number of tokens: " + code + ", expected an even number of tokens");
		
		for (int i=0; i<tokens.length; i+=2) {
			String	selector	= tokens[i];
			String	value		= tokens[i+1];	
				 if ("at".equals(selector))		at(c, value);
			else if ("size".equals(selector))	size(c, value);
			else if ("weight".equals(selector)) weight(c, value);
			else if("ipad".equals(selector))	ipad(c, value);
			else if ("insets".equals(selector)) insets(c, value);
			else if ("anchor".equals(selector)) anchor(c, value);
			else if ("fill".equals(selector))	fill(c, value);
			else throw new SyntaxError("illegal selector: " + selector + ", expected at|size|weight|ipad|insets|anchor|fill");
		}
	}

	//-------------------------------------------------------------------------
	//## property setter
	
	/** sets gridx and gridy in cells with an int number or "relative" */
	private static void at(GridBagConstraints c, String value) {
		try {
			String[]	tuple	= tuple(value, 2);
			c.gridx = tuple[0].equals("relative")	? GridBagConstraints.RELATIVE 
					: Integer.parseInt(tuple[0]);
			c.gridy = tuple[1].equals("relative")	? GridBagConstraints.RELATIVE 
					: Integer.parseInt(tuple[1]);
		}
		catch (NumberFormatException e) {
			throw new SyntaxError("illegal at value: " + value + ", expected relative|<int>", e);
		}
	}

	/** sets gridwidth and gridheight in cells with an int number, "relative" or "remainder" */
	private static void size(GridBagConstraints c, String value) {
		try {
			String[]	tuple	= tuple(value, 2);
			c.gridwidth		= tuple[0].equals("relative")	? GridBagConstraints.RELATIVE
							: tuple[0].equals("remainder")	? GridBagConstraints.REMAINDER
							: Integer.parseInt(tuple[0]);
			c.gridheight	= tuple[1].equals("relative")	? GridBagConstraints.RELATIVE 
							: tuple[1].equals("remainder")	? GridBagConstraints.REMAINDER 
							: Integer.parseInt(tuple[1]);
		}
		catch (NumberFormatException e) {
			throw new SyntaxError("illegal size value: " + value + ", expected relative|remainder|<int>", e);
		}
	}

	/** sets weightx and weighty in doubles within the range 0..1, or "minimum", "medium" or "maximum" */
	private static void weight(GridBagConstraints c, String value) {
		try {
			String[]	tuple	= tuple(value, 2);
			c.weightx	= tuple[0].equals("minimum")	? 0f
						: tuple[0].equals("medium")		? 0.5f
						: tuple[0].equals("maximum")	? 1f
						: Double.parseDouble(tuple[0]);
			c.weighty	= tuple[1].equals("minimum")	? 0f
						: tuple[1].equals("medium")		? 0.5f
						: tuple[1].equals("maximum")	? 1f
						: Double.parseDouble(tuple[1]);
		}
		catch (NumberFormatException e) {
			throw new SyntaxError("illegal weight value: " + value + ", expected minimum|medium|maximum|<double>/minimum|medium|maximum|<double>", e);
		}
	}

	/** sets ipadx and ipady in pixels with an int number */
	private static void ipad(GridBagConstraints c, String value) {
		try {
			String[]	tuple	= tuple(value, 2);
			c.ipadx = Integer.parseInt(tuple[0]);
			c.ipady = Integer.parseInt(tuple[1]);
		}
		catch (NumberFormatException e) {
			throw new SyntaxError("illegal ipad value: " + value + ", expected <int>/<int>", e);
		}
	}
	
	/** sets insets in pixels with an int number */
	private static void insets(GridBagConstraints c, String value) {
		try {
			String[]	tuple	= tuple(value, 4);
			c.insets = new Insets(
				Integer.parseInt(tuple[0]), // top
				Integer.parseInt(tuple[1]), // left
				Integer.parseInt(tuple[2]), // bottom
				Integer.parseInt(tuple[3])	// right
			);
		}
		catch (NumberFormatException e) {
			throw new SyntaxError("illegal insets value: " + value + ", expected <int>/<int>/<int>/<int>", e);
		}
	}
	
	/** sets anchor from human readable names */
	private static void anchor(GridBagConstraints c, String value) {
			 if ("center".equals(value))			c.anchor	= GridBagConstraints.CENTER;
		else if ("north".equals(value))				c.anchor	= GridBagConstraints.NORTH;
		else if ("south".equals(value))				c.anchor	= GridBagConstraints.SOUTH;
		else if ("east".equals(value))				c.anchor	= GridBagConstraints.EAST;
		else if ("west".equals(value))				c.anchor	= GridBagConstraints.WEST;
		else if ("northeast".equals(value))			c.anchor	= GridBagConstraints.NORTHEAST;
		else if ("northwest".equals(value))			c.anchor	= GridBagConstraints.NORTHWEST;
		else if ("southeast".equals(value))			c.anchor	= GridBagConstraints.SOUTHEAST;
		else if ("southwest".equals(value))			c.anchor	= GridBagConstraints.SOUTHWEST;
		else if ("page-start".equals(value))		c.anchor	= GridBagConstraints.PAGE_START;
		else if ("page-end".equals(value))			c.anchor	= GridBagConstraints.PAGE_END;
		else if ("line-start".equals(value))		c.anchor	= GridBagConstraints.LINE_START;
		else if ("line-end".equals(value))			c.anchor	= GridBagConstraints.LINE_END;
		else if ("first-line-start".equals(value))	c.anchor	= GridBagConstraints.FIRST_LINE_START;
		else if ("first-line-end".equals(value))	c.anchor	= GridBagConstraints.FIRST_LINE_END;
		else if ("last-line-start".equals(value))	c.anchor	= GridBagConstraints.LAST_LINE_START;
		else if ("last-line-end".equals(value))		c.anchor	= GridBagConstraints.LAST_LINE_END;
		else throw new SyntaxError("illegal anchor value: " + value + ", expected center|north|south|east|west|northeast|northwest|southeast|southwest|page-start|page-end|line-start|line-end|first-line-start|first-line-end|last-line-start|last-line-end");
	}
	
	/** sets fill from human readable names */
	private static void fill(GridBagConstraints c, String value) {
			 if ("none".equals(value))			c.fill	= GridBagConstraints.NONE;
		else if ("horizontal".equals(value))	c.fill	= GridBagConstraints.HORIZONTAL;
		else if ("vertical".equals(value))		c.fill	= GridBagConstraints.VERTICAL;
		else if ("both".equals(value))			c.fill	= GridBagConstraints.BOTH;
		else throw new SyntaxError("illegal fill value: " + value + ", expected none|horizontal|vertical|both");
	}
	
	//-------------------------------------------------------------------------
	//## simple parser
	
	private static String[] tuple(String in, int length) {
		String[]	out = in.split("/");
		if (out.length != length)	
				throw new SyntaxError("illegal " + length + "-tuple: " + in + ", expected <value>/<value>/...");
		return out;
	}
}
