#include "cake.h"
#include "system.h"
#include "logfile.h"
#include "types.h"
#include <time.h>
#include <stdio.h>
#include <math.h>
#ifdef WIN32
  #include <windows.h>
  #include <mmsystem.h>
#else
  #include <stdlib.h>
#endif

/**
 * Get vendor ID string from processor up to 48 bytes
 */
unsigned char* GetVendorString(void)
{
  static unsigned char szVendor[49] = { '\0' };
  #ifdef WIN32
    unsigned char buffer[49] = { '\0' };
    __asm
    {
      lea   esi,buffer

      mov   eax,0x80000002    // FIRST 16 CHRS   
      xor   ebx,ebx
      xor   ecx,ecx
      xor   edx,edx

      _emit 0x0f
      _emit 0xa2

      mov   [esi],eax
      mov   [esi+4],ebx
      mov   [esi+8],ecx
      mov   [esi+12],edx

      add   esi,16
      mov   eax,0x80000003    // NEXT 16 CHRS 
      xor   ebx,ebx
      xor   ecx,ecx
      xor   edx,edx

      _emit 0x0f
      _emit 0xa2

      mov   [esi],eax
      mov   [esi+4],ebx
      mov   [esi+8],ecx
      mov   [esi+12],edx


      add   esi,16
      mov   eax,0x80000004    // LAST 16 CHRS 
      xor   ebx,ebx
      xor   ecx,ecx
      xor   edx,edx

      _emit 0x0f
      _emit 0xa2

      mov   [esi],eax
      mov   [esi+4],ebx
      mov   [esi+8],ecx
      mov   [esi+12],edx

      mov   eax,0
      mov   [esi+48],al
    }

    strcpy((char*) szVendor, (char*) buffer);
  #endif

  return szVendor;
}

/*
 * CPU INFO - L1 DATA CACHE AMD ONLY
 */
int GetCacheInfo(void)
{
  int L1DCache = 0;

#ifdef WIN32
  __asm
  {
    mov   eax,0x80000005      // AMD CACHE
    mov   ebx,0
    mov   ecx,0
    mov   edx,0

    _emit 0x0f
    _emit 0xa2

    shr   ecx,24
    and   ecx,0xff
    mov   L1DCache,ecx
  }
#endif

  return L1DCache;
}

/********************************************************
PROCESSOR SPEED TEST 
********************************************************/
unsigned int GetCpuSpeed(void)
{
  unsigned long   cpuSpeed  = 0;

#ifdef WIN32
  int       timeStart = 0;
  int       timeStop  = 0;
  unsigned long   StartTicks  = 0;
  unsigned long   EndTicks  = 0;
  unsigned long   TotalTicks  = 0;

  timeStart = timeGetTime();        // GET TICK EDGE
  for(;;)
  {
    timeStop = timeGetTime();   
    if ((timeStop-timeStart) > 1)   // ROLLOVER PAST 1
    {
      __asm{
         xor    eax, eax
         xor    ebx, ebx
         xor    ecx, ecx
         xor    edx, edx
         _emit  0x0f        // CPUID
         _emit  0xa2
         _emit  0x0f        // RDTSC
         _emit  0x31
         mov    [StartTicks], eax
        }
      break;  
    } 
  }

  timeStart = timeStop;       

  for(;;)
  {
    timeStop = timeGetTime();     
    if ((timeStop-timeStart) > 1000)  // ONE SECOND
    {
      __asm{
         xor    eax, eax
         xor    ebx, ebx
         xor    ecx, ecx
         xor    edx, edx
         _emit  0x0f        // CPUID
         _emit  0xa2
         _emit  0x0f        // RDTSC
         _emit  0x31
         mov    [EndTicks], eax
        }

      break;  
    } 
  }

  TotalTicks = EndTicks-StartTicks;   // TOTAL 

  cpuSpeed = TotalTicks/1000000;      // SPEED

#endif
  return (UINT)cpuSpeed;
}


/******************************************************************************
 Routine:   GetCPUCaps
 Input:     Which capability to query (see enum CPUCAPS for an exhaustive list)
 Returns:   Depends on input, either a boolean or an enumeration value.
            CPU_TYPE - enum CPU_TYPES
            CPU_MFG  - enum CPU_MFGS
 Comment:   This function returns information about the capabilies of the
            CPU on which it is called.  The input enumeration covers both
            processor feature bits (the HAS_* values) and "synthesized"
            information.
            
            THE HAS_* QUERIES SHOULD ALWAYS BE USED IN PREFERENCE TO DIRECTLY 
            CHECKING THE CPU TYPE WHEN LOOKING FOR FEATURES.  For instance,
            it is *always* better to check for HAS_3DNOW directly, rather
            than rely on checking for a K6_2, K6_3, or K7.  Likewise,
            HAS_MMX should always be used in preference to other methods
            of checking for MMX instructions.

            The features bits are checked against either the base feature
            bits (CPUID function 1, edx) or the extended feature bits
            (CPUID function 0x80000001, edx), as appropriate.  The return
            value is 1 for feature present or 0 for feature not present,

            The synthesized information is created by interpreting the CPUID
            results in some way.

            The CPUCAPS currently implemented are not exhaustive by any means,
            but they cover the basic features and 3DNow! extensions, which
            are the majority of the used bits.  In future revisions, more
            feature bits and synthesized values will be added.

            Note that this routine caches the feature bits when first called,
            so checking multiple features is relatively efficient after the
            first invocation.  However, tt is not recommended practice to
            use GetCPUCaps() inside time-critical code.

******************************************************************************/
DWORD GetCPUCaps(CPUCAPS cap)
{
    DWORD res = 0;

  #ifdef WIN32
    static DWORD features       = 0;
    static DWORD ext_features   = 0;
    static DWORD processor      = 0;
    static int   init           = 0;

    // Detect CPUID presence once, since all other requests depend on it
    if (init == 0)
    {
      __asm {
        pushfd                  // save EFLAGS to stack.
        pop     eax             // store EFLAGS in EAX.
        mov     edx, eax        // save in EBX for testing later.
        xor     eax, 0x200000   // switch bit 21.
        push    eax             // copy "changed" value to stack.
        popfd                   // save "changed" EAX to EFLAGS.
        pushfd
        pop     eax
        xor     eax, edx        // See if bit changeable.
        jnz     short foundit   // if so, mark 
        mov     eax,-1
        jmp     short around

        ALIGN   4
      foundit:
        // Load up the features and (where appropriate) extended features flags
        mov     eax,1               // Check for processor features
        CPUID
        mov     [features],edx      // Store features bits
        mov     eax,0x80000000      // Check for support of extended functions.
        CPUID
        cmp     eax,0x80000001      // Make sure function 0x80000001 supported.
        jb      short around
        mov     eax,0x80000001      // Select function 0x80000001
        CPUID
        mov     [processor],eax     // Store processor family/model/step
        mov     [ext_features],edx  // Store extende features bits
        mov     eax,1               // Set "Has CPUID" flag to true

      around:
        mov     [init],eax
      }
    }
    if (init == -1)
    {
      // No CPUID, so no CPUID functions are supported
      return 0;
    }

    // Otherwise, perform the requested tests
    switch (cap)
    {
      // Synthesized Capabilities
      case HAS_CPUID:
        // Always true if this code gets executed
        res = 1;
        break;

      case CPU_MFG:
        __asm {
          // Query manufacturer string
          mov     eax,0           // function 0 = manufacturer string
          CPUID

          // These tests could probably just check the 'ebx' part of the string,
          // but the entire string is checked for completeness.  Plus, this function
          // should not be used in time-critical code, because the CPUID instruction
          // serializes the processor. (That is, it flushes out the instruction pipeline.)

          // Test for 'AuthenticAMD'
          cmp     ebx,'htuA'
          jne     short not_amd
          cmp     edx,'itne'
          jne     short not_amd
          cmp     ecx,'DMAc'
          jne     short not_amd
          mov     eax,MFG_AMD
          jmp     short next_test
     
          // Test for 'GenuineIntel'
        not_amd:
          cmp     ebx,'uneG'
          jne     short not_intel
          cmp     edx,'Ieni'
          jne     short not_intel
          cmp     ecx,'letn'
          jne     short not_intel
          mov     eax,MFG_INTEL
          jmp     short next_test
     
          // Test for 'CyrixInstead'
        not_intel:
          cmp     ebx,'iryC'
          jne     short not_cyrix
          cmp     edx,'snIx'
          jne     short not_cyrix
          cmp     ecx,'deat'
          jne     short not_cyrix
          mov     eax,MFG_CYRIX
          jmp     short next_test
     
          // Test for 'CentaurHauls'
        not_cyrix:
          cmp     ebx,'tneC'
          jne     short not_centaur
          cmp     edx,'Hrua'
          jne     short not_centaur
          cmp     ecx,'slua'
          jne     short not_centaur
          mov     eax,MFG_CENTAUR
          jmp     short next_test
     
        not_centaur:
          mov     eax,MFG_UNKNOWN
     
        next_test:
          mov     [res],eax       // store result of previous tests
        }
        break;

      case CPU_TYPE:
        // Return a member of the CPUTYPES enumeration
        // Note: do NOT use this for determining presence of chip features, such
        // as MMX and 3DNow!  Instead, use GetCPUCaps (HAS_MMX) and GetCPUCaps (HAS_3DNOW),
        // which will accurately detect the presence of these features on all chips which
        // support them.
        switch (GetCPUCaps(CPU_MFG))
        {
          case MFG_AMD:
            switch ((processor >> 8) & 0xf) // extract family code
            {
              case 4: // Am486/AM5x86
                res = AMD_Am486;
                break;
              case 5: // K6
                switch ((processor >> 4) & 0xf) // extract model code
                {
                  case 0: res = AMD_K5;       break;
                  case 1: res = AMD_K5;       break;
                  case 2: res = AMD_K5;       break;
                  case 3: res = AMD_K5;       break;
                  case 4: res = AMD_K6_MMX;   break;
                  case 5: res = AMD_K6_MMX;   break;
                  case 6: res = AMD_K6_MMX;   break;
                  case 7: res = AMD_K6_MMX;   break;
                  case 8: res = AMD_K6_2;     break;
                  case 9: res = AMD_K6_3;     break;
                  default:          break;
                }
                break;
              case 6: // K7 Athlon
                res = AMD_K7;
                break;
              default:
                break;
            }
            break;

          case MFG_INTEL:
            switch ((processor >> 8) & 0xf) // extract family code
            {
              case 4:
                switch ((processor >> 4) & 0xf) // extract model code
                {
                  case 0: res = INTEL_486DX;  break;
                  case 1: res = INTEL_486DX;  break;
                  case 2: res = INTEL_486SX;  break;
                  case 3: res = INTEL_486DX2; break;
                  case 4: res = INTEL_486SL;  break;
                  case 5: res = INTEL_486SX2; break;
                  case 7: res = INTEL_486DX2E;break;
                  case 8: res = INTEL_486DX4; break;
                  default:          break;
                }
                break;

              case 5:
                switch ((processor >> 4) & 0xf) // extract model code
                {
                  case 1: res = INTEL_Pentium;    break;
                  case 2: res = INTEL_Pentium;    break;
                  case 3: res = INTEL_Pentium;    break;
                  case 4: res = INTEL_Pentium_MMX;break;
                  default:            break;
                }
                break;

              case 6:
                switch ((processor >> 4) & 0xf) // extract model code
                {
                  case 1: res = INTEL_Pentium_Pro;break;
                  case 3: res = INTEL_Pentium_II; break;
                  case 5: res = INTEL_Pentium_II; break;  // actual differentiation depends on cache settings
                  case 6: res = INTEL_Celeron;    break;
                  case 7: res = INTEL_Pentium_III;break;  // actual differentiation depends on cache settings
                  default:            break;
                }
                break;
            }
            break;

          case MFG_CYRIX:
            res = UNKNOWN;
            break;

          case MFG_CENTAUR:
            res = UNKNOWN;
            break;
          default:
            break;
        }
        break;

      // Feature Bit Test Capabilities
      case HAS_FPU:       res = (features >> 0) & 1;      break;  // bit 0 = FPU
      case HAS_VME:       res = (features >> 1) & 1;      break;  // bit 1 = VME
      case HAS_DEBUG:     res = (features >> 2) & 1;      break;  // bit 2 = Debugger extensions
      case HAS_PSE:       res = (features >> 3) & 1;      break;  // bit 3 = Page Size Extensions
      case HAS_TSC:       res = (features >> 4) & 1;      break;  // bit 4 = Time Stamp Counter
      case HAS_MSR:       res = (features >> 5) & 1;      break;  // bit 5 = Model Specific Registers
      case HAS_MCE:       res = (features >> 6) & 1;      break;  // bit 6 = Machine Check Extensions
      case HAS_CMPXCHG8:  res = (features >> 7) & 1;      break;  // bit 7 = CMPXCHG8 instruction
      case HAS_MMX:       res = (features >> 23) & 1;     break;  // bit 23 = MMX
      case HAS_3DNOW:     res = (ext_features >> 31) & 1; break;  // bit 31 (ext) = 3DNow!
      default:                      break;
    }
  #endif

    return res;
}

// Get an argument in a string
void GetArg(const char* command, int i, char *dest)
{
  int b = 0, j = 0, k = i;
  
  while (command[b] == 32 && b < (int) strlen(command)) ++b;    // Skip blanc chars
  j = b;
  while (k > 0 && j < (int) strlen(command))            // Search for start of arg i
  {
    if (command[j] == 32)
    {
      --k;
      while (command[j] == 32 && j < (int) strlen(command)) ++j;
    }
    else ++j;
  }
  if (j >= (int) strlen(command)) return;
  
  // Search for end of arg i
  k = j+1;
  while (command[k] != 32 && k < (int) strlen(command)) ++k;

  strncpy(dest, &command[j], k-j);
}

int GetNArgs(const char* command)
{
  int c = 0, ret = 0;
  int l = (int) strlen(command);
  
  while (c < l && command[c] == ' ') ++c;   // skip spaces

  while (c < l)
  {
    while (c < l && command[c] != ' ') ++c;
    while (c < l && command[c] == ' ') ++c;   // skip spaces
    ++ret;
  }

  return ret;
}

const char *getExceptionDescription(CakeException val)
{
  switch (val)
  {
    case NO_EXCEPTION:      return "No exception";
    case DEFAULT_EXCEPTION:   return "Default exception";
    case ALLOCATION_ERROR:    return "Allocation error";
    case OVERFLOW_ERROR:    return "Overflow error";
    case NULL_POINTER:      return "Null pointer exception";
    case DIVISION_BY_ZERO:    return "Division by zero";
    case ERROR_OPENING_FILE:  return "Error while opening a file";
    case FATAL_ERROR:     return "Fatal error";
    default:          return "";
  }
}

void ThrowException(CakeException val, const char *msg)
{
  gLogFile->Insert("^1The system has encountered the following exception :\n");
  gLogFile->Insert("^1\tException : ^0%s (%d)\n", getExceptionDescription(val), val);
  if (msg) gLogFile->Insert("^1\tInformation message: ^0%s\n", msg);
  gLogFile->Insert("^1The application will stop.\n");
  
  #ifdef WIN32
    char errmsg[1024] = { '\0' };
    sprintf(errmsg, "%s :\n\n%s", getExceptionDescription(val), msg);
    MessageBox(HWND_DESKTOP, errmsg, "Abnormal program termination", MB_OK | MB_ICONERROR);
  #else
    printf(msg);
  #endif

  exit(val);
}
