//-----------------------------------------------------------------------------
// SkyBox
// FIXME : cloud_height should be used here, during sphere generation
//-----------------------------------------------------------------------------

#include "skybox.h"
#include "console.h"
#include "vars.h"
#include "logfile.h"
#include "camera.h"
#include "math.h"
#include "mem.h"

Var sky_side_size("sky_side_size", 9, VF_LATCH);
Var sky_debug("sky_debug", 0, VF_DEBUG);

#define SPHERE_RAD  10.f
#define EYE_RAD      9.f

/**
 * Real size of the cube
 * This value is the real half size of the cube. Using 1 is a good value
 * because it simplify the calculs when trying to optimize the sky rendering.
 */
#define CUBE_SIZE  1.f

#define SCALE_S 6.0   /**< Arbitrary (?) texture scaling factors */
#define SCALE_T 6.0 

SkyBox::SkyBox()
{
  gVars->RegisterVar(sky_side_size);
  gVars->RegisterVar(sky_debug);
}

SkyBox::~SkyBox()
{
  gVars->UnregisterVar(sky_side_size);
  gVars->UnregisterVar(sky_debug);
}

void SkyBox::Init(int height)
{
  int i;

  side_size = sky_side_size.ivalue;
  if (side_size <= 1) side_size = 2;
  //if (!(side_size % 2)) ++side_size;

  points_len = side_size*side_size;
  elem_len = (side_size-1)*(side_size-1)*6;

  slope = (float)(side_size-1)/(2.f*(float)CUBE_SIZE);
  h = (float)(side_size-1)/2.f;

  // FIXME : Couldn't we use Layers here !!
  // Alloc space for skybox verts, etc.
  skyface       = new Surface[6];
  clouds[0]     = (vertex_t*) cake_malloc(6*points_len*sizeof(vertex_t), "SkyBox::Init.clouds");
  box[0]        = (vertex_t*) cake_malloc(6*4*sizeof(vertex_t), "SkyBox::Init.box");
  clouds_elems  = (int*)      cake_malloc(elem_len*sizeof(int), "SkyBox::Init.clouds_elems");
  box_elems     = (int*)      cake_malloc(6*sizeof(int), "SkyBox::Init.box_elems");
  framenum[0]   = (int*)      cake_malloc(6*elem_len*sizeof(int), "SkyBox::Init.framenum");
  box_shader[0] = (Shader*)   cake_malloc(6*sizeof(Shader), "SkyBox::Init.box_shader");

  // Initialize surfaces
  skyface[0].Init(2);
  skyface[1].Init(2);
  skyface[2].Init(2);
  skyface[3].Init(2);
  skyface[4].Init(2);
  skyface[5].Init(2);

  surf_array  = NULL;
  nsurfs      = 0;
  
  num_elems   = elem_len;

  clouds_shader = -1;   // no shader
  box_shader[0]->num_layers = 0;

  clouds[1] = clouds[0] + points_len;
  clouds[2] = clouds[1] + points_len;
  clouds[3] = clouds[2] + points_len;
  clouds[4] = clouds[3] + points_len;
  clouds[5] = clouds[4] + points_len;
  box[1] = box[0] + 4;
  box[2] = box[1] + 4;
  box[3] = box[2] + 4;
  box[4] = box[3] + 4;
  box[5] = box[4] + 4;
  framenum[1] = framenum[0] + elem_len;
  framenum[2] = framenum[1] + elem_len;
  framenum[3] = framenum[2] + elem_len;
  framenum[4] = framenum[3] + elem_len;
  framenum[5] = framenum[4] + elem_len;
  for (i = 1; i < 6; ++i)
  {
    box_shader[i] = box_shader[i-1] + 1;
    box_shader[i]->num_layers = 0;
    box_shader[i]->content_flags = 0;
    box_shader[i]->cull = 0;
    box_shader[i]->flags = 0;
    box_shader[i]->sortvalue = SORT_NORMAL;
    box_shader[i]->surface_flags = 0;
  }

  // Initialization
  for (i = 0; i < 6; ++i)
  {
    skyface[i].firstvert[0] = clouds[i];
    skyface[i].numverts[0]  = points_len;
    skyface[i].numelems[0]  = elem_len;
    skyface[i].firstelem[0] = clouds_elems;
    skyface[i].firstvert[1] = box[i];
    skyface[i].numverts[1]  = 4;
    skyface[i].numelems[1]  = 6;
    skyface[i].firstelem[1] = box_elems;
    skyface[i].shader = -1;
  }

  memset(framenum[0], 0, 6*elem_len*sizeof(int));
  minmax[0][SKYMINMAX_MINX] = minmax[0][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[0][SKYMINMAX_MAXX] = minmax[0][SKYMINMAX_MAXY]=-CUBE_SIZE;
  minmax[1][SKYMINMAX_MINX] = minmax[1][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[1][SKYMINMAX_MAXX] = minmax[1][SKYMINMAX_MAXY]=-CUBE_SIZE;
  minmax[2][SKYMINMAX_MINX] = minmax[2][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[2][SKYMINMAX_MAXX] = minmax[2][SKYMINMAX_MAXY]=-CUBE_SIZE;
  minmax[3][SKYMINMAX_MINX] = minmax[3][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[3][SKYMINMAX_MAXX] = minmax[3][SKYMINMAX_MAXY]=-CUBE_SIZE;
  minmax[4][SKYMINMAX_MINX] = minmax[4][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[4][SKYMINMAX_MAXX] = minmax[4][SKYMINMAX_MAXY]=-CUBE_SIZE;
  minmax[5][SKYMINMAX_MINX] = minmax[5][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[5][SKYMINMAX_MAXX] = minmax[5][SKYMINMAX_MAXY]=-CUBE_SIZE;

  cloudsheight = height;

  GenBox();
  GenElems();

  skyboxsize = 128.f;
}

void SkyBox::Shut()
{
  int i;
  
  if (skyface)
  {
    skyface[0].Shut();
    skyface[1].Shut();
    skyface[2].Shut();
    skyface[3].Shut();
    skyface[4].Shut();
    skyface[5].Shut();
    delete [] skyface;
  }
  skyface = NULL;
  cake_free(clouds[0]);
  clouds_shader = 0;
  
  cake_free(box[0]);

  for (i = 0; i < 6; ++i)
  {
    for (int j = 0; j < box_shader[i]->num_layers; ++j)
    {
      box_shader[i]->layer[j] = NULL;
    }
  }
  cake_free(box_shader[0]);
  cake_free(clouds_elems);
  cake_free(box_elems);
  cake_free(framenum[0]);

  if (surf_array)
  {
    for (i = 0; i < nsurfs; ++i) surf_array[i] = NULL;
    cake_free(surf_array);
    surf_array = NULL;
    nsurfs = 0;
  }
}

void SkyBox::GenElems(void)
{
    // Box elems in tristrip order (distribution of faces on the surface)
    int *e = clouds_elems;
    for (int v = 0; v < side_size-1; ++v)
    {
    for (int u = 0; u < side_size-1; ++u)
    {
      *e++ = v    *side_size+u;
      *e++ = (v+1)*side_size+u;
      *e++ = v    *side_size+u+1;
      *e++ = v    *side_size+u+1;
      *e++ = (v+1)*side_size+u;
      *e++ = (v+1)*side_size+u+1;     
    }
  }

  e = box_elems;
  *e++ = 0;
  *e++ = 2;
  *e++ = 1;
  *e++ = 1;
  *e++ = 2;
  *e++ = 3;     
}
    
void SkyBox::GenBox(void)
{
  vec3_t orig, drow, dcol;

  // Front
  orig[0] =  CUBE_SIZE;
  orig[1] = -CUBE_SIZE;
  orig[2] =  CUBE_SIZE;
  drow[0] =  0.0;
  drow[1] =  0.0;
  drow[2] = -1;
  dcol[0] =  0.0;
  dcol[1] =  1;
  dcol[2] =  0.0;
  GenBoxSide(SKYBOX_FRONT, orig, drow, dcol);

  // Left
  orig[0] =  CUBE_SIZE;
  orig[1] =  CUBE_SIZE;
  orig[2] =  CUBE_SIZE;
  drow[0] =  0.0;
  drow[1] =  0.0;
  drow[2] = -1;
  dcol[0] = -1;
  dcol[1] =  0.0;
  dcol[2] =  0.0;
  GenBoxSide(SKYBOX_LEFT, orig, drow, dcol);

  // Top
  orig[0] = -CUBE_SIZE;
  orig[1] =  CUBE_SIZE;
  orig[2] =  CUBE_SIZE;
  drow[0] =  0.0;
  drow[1] = -1;
  drow[2] =  0.0;
  dcol[0] =  1;
  dcol[1] =  0.0;
  dcol[2] =  0.0;
  GenBoxSide(SKYBOX_TOP, orig, drow, dcol);

  // Back
  orig[0] = -CUBE_SIZE;
  orig[1] =  CUBE_SIZE;
  orig[2] =  CUBE_SIZE;
  drow[0] =  0.0;
  drow[1] =  0.0;
  drow[2] = -1;
  dcol[0] =  0.0;
  dcol[1] = -1;
  dcol[2] =  0.0;
  GenBoxSide(SKYBOX_BACK, orig, drow, dcol);

  // Right
  orig[0] = -CUBE_SIZE;
  orig[1] = -CUBE_SIZE;
  orig[2] =  CUBE_SIZE;
  drow[0] =  0.0;
  drow[1] =  0.0;
  drow[2] = -1;
  dcol[0] =  1;
  dcol[1] =  0.0;
  dcol[2] =  0.0;
  GenBoxSide(SKYBOX_RIGHT, orig, drow, dcol);

  // Bottom
  orig[0] = -CUBE_SIZE;
  orig[1] = -CUBE_SIZE;
  orig[2] = -CUBE_SIZE;
  drow[0] =  0.0;
  drow[1] =  1;
  drow[2] =  0.0;
  dcol[0] =  1;
  dcol[1] =  0.0;
  dcol[2] =  0.0;
  GenBoxSide(SKYBOX_BOTTOM, orig, drow, dcol);
}

void SkyBox::GenBoxSide(int side, vec3_t orig, vec3_t drow, vec3_t dcol)
{
  vec3_t ndrow, ndcol;
  float step;
  vec3_t pos, w, row;
  vertex_t *v;
  int r, c;
  
    // I don't know exactly what Q3A does for skybox texturing, but this is
    // at least fairly close.  We tile the texture onto the inside of
    // a large sphere, and put the camera near the top of the sphere.
    // We place the box around the camera, and cast rays through the
    // box verts to the sphere to find the texture coordinates.

  float d = EYE_RAD;     // Sphere center to camera distance
  float b = SPHERE_RAD;  // Sphere radius
  float p, t;

  step = 2*CUBE_SIZE/(side_size-1);
  VectorScale(drow, step, ndrow);
  VectorScale(dcol, step, ndcol);

  v = &(clouds[side][0]);
  VectorCopy(orig, row);
  for (r = 0; r < side_size; ++r)
  {
    VectorCopy(row, pos);
    for (c = 0; c < side_size; ++c)
    {
      // pos points from eye to vertex on box
      VectorCopy(pos, v->v_point);
      VectorCopy(pos, w);

      // Normalize pos -> w
      p = InverseSqrt(VectorDot(w, w));
      w[0] *= p;
      w[1] *= p;
      w[2] *= p;

      // Find distance along w to sphere
      t = FastSqrt(d*d*(w[2]*w[2]-1.f) + b*b) - d*w[2];
      w[0] *= t;
      w[1] *= t;

      // Use x and y on sphere as s and t
      v->tex_st[0] = w[0]/(2.f * (float) SCALE_S);
      v->tex_st[1] = w[1]/(2.f * (float) SCALE_T);
      
      VectorAdd(pos, ndcol, pos);
      ++v;
    }
    VectorAdd(row, ndrow, row);
  }

  step = 2*CUBE_SIZE;
  VectorScale(drow, step, ndrow);
  VectorScale(dcol, step, ndcol);

  v = &(box[side][0]);
  VectorCopy(orig, row);
  for (r = 0; r < 2; ++r)
  {
    VectorCopy(row, pos);
    for (c = 0; c < 2; ++c)
    {
      // pos points from eye to vertex on box
      VectorCopy(pos, v->v_point);

      // texture coordinates
      // don't ask me why coordinates are computed like this !
      if (side == SKYBOX_TOP)
      {
        v->tex_st[0] = (float)r;
        v->tex_st[1] = (float)c;
      }
      else if (side == SKYBOX_BOTTOM)
      {
        v->tex_st[0] = 1-(float)r;
        v->tex_st[1] = 1-(float)c;
      }
      else
      {
        v->tex_st[0] = 1-(float)c;
        v->tex_st[1] =   (float)r;
      }

      VectorAdd(pos, ndcol, pos);
      ++v;
    }
    VectorAdd(row, ndrow, row);
  }
}

void SkyBox::MakeSkyList(int n)
{
  if (surf_array)
  {
    for (int i = 0; i < nsurfs; ++i) surf_array[i] = NULL;
    cake_free(surf_array);
    surf_array = NULL;
    nsurfs = 0;
  }

  ncurr = 0;
  nsurfs = n;
  surf_array = (Surface**) cake_malloc(n*sizeof(Surface*), "SkyBox::makeSkyList.surf_array");
}

void SkyBox::AddToList(Surface *surf)
{
  surf_array[ncurr++] = surf;
}

void SkyBox::NewFrame(int v)
{
  memset(framenum[0], v, 6*elem_len*sizeof(int));
  minmax[0][SKYMINMAX_MINX] = minmax[0][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[0][SKYMINMAX_MAXX] = minmax[0][SKYMINMAX_MAXY]=-CUBE_SIZE;
  minmax[1][SKYMINMAX_MINX] = minmax[1][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[1][SKYMINMAX_MAXX] = minmax[1][SKYMINMAX_MAXY]=-CUBE_SIZE;
  minmax[2][SKYMINMAX_MINX] = minmax[2][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[2][SKYMINMAX_MAXX] = minmax[2][SKYMINMAX_MAXY]=-CUBE_SIZE;
  minmax[3][SKYMINMAX_MINX] = minmax[3][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[3][SKYMINMAX_MAXX] = minmax[3][SKYMINMAX_MAXY]=-CUBE_SIZE;
  minmax[4][SKYMINMAX_MINX] = minmax[4][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[4][SKYMINMAX_MAXX] = minmax[4][SKYMINMAX_MAXY]=-CUBE_SIZE;
  minmax[5][SKYMINMAX_MINX] = minmax[5][SKYMINMAX_MINY]= CUBE_SIZE;
  minmax[5][SKYMINMAX_MAXX] = minmax[5][SKYMINMAX_MAXY]=-CUBE_SIZE;
  usedskysides[0] = usedskysides[1] = usedskysides[2] = 
  usedskysides[3] = usedskysides[4] = usedskysides[5] = false;
}

// this is actually taken from Q2 source, it clips the "skypolygons" in a way
// they that dont span two or more skysides
#define MAX_NUM_SKYVERTS   600
#define MAX_NUM_CLIPSKYVERTS 32
#define ON_PLANE  0
#define FRONTSIDE 1
#define BACKSIDE  2
#define SPANNING  3

void SkyBox::DetermineMinMax(vec3_t *verts, int num_verts)
{
  vec3_t absfacedir;
  vec3_t facedir;
  vec3_t p;
  float x, y;
  VectorCopy(*verts, facedir);
  int v, side, skyside;

  for (v = 1; v < num_verts; ++v) // only allow triangles
  {
    VectorAdd(facedir, verts[v], facedir);
  }

  absfacedir[0]=fabsf(facedir[0]);
  absfacedir[1]=fabsf(facedir[1]);
  absfacedir[2]=fabsf(facedir[2]);
  if ((absfacedir[1]>absfacedir[0])&&(absfacedir[1]>absfacedir[2])) side = SKYBOX_LEFT;
  else if((absfacedir[0]>absfacedir[1])&&(absfacedir[0]>absfacedir[2])) side = SKYBOX_FRONT;
  else side = SKYBOX_TOP;

  skyside = facedir[side]>=0.0f ? side : side+3;
  usedskysides[skyside] = true;

  for (v = 0; v < num_verts; ++v)
  {
    VectorCopy(verts[v], p);
    switch (skyside)
    {
      case SKYBOX_FRONT:
        if (p[0] <= 0) continue;
        x =  (vec_t)CUBE_SIZE*p[1]/p[0];
        y = -(vec_t)CUBE_SIZE*p[2]/p[0];
        break;
      case SKYBOX_LEFT:
        if (p[1] <= 0) continue;
        y = -(vec_t)CUBE_SIZE*p[2]/p[1];
        x = -(vec_t)CUBE_SIZE*p[0]/p[1];
        break;
      case SKYBOX_TOP:
        if (p[2] <= 0) continue;
        x =  (vec_t)CUBE_SIZE*p[0]/p[2];
        y = -(vec_t)CUBE_SIZE*p[1]/p[2];
        break;
      case SKYBOX_BACK:
        if (p[0] >= 0) continue;
        x =  (vec_t)CUBE_SIZE*p[1]/p[0];
        y =  (vec_t)CUBE_SIZE*p[2]/p[0];
        break;
      case SKYBOX_RIGHT:
        if (p[1] >= 0) continue;
        y =  (vec_t)CUBE_SIZE*p[2]/p[1];
        x = -(vec_t)CUBE_SIZE*p[0]/p[1];
        break;
      case SKYBOX_BOTTOM:
        if (p[2] >= 0) continue;
        x = -(vec_t)CUBE_SIZE*p[0]/p[2];
        y = -(vec_t)CUBE_SIZE*p[1]/p[2];
        break;
      default:
        continue;
    }

    AddMinMax(x, y, skyside);
  }
}

  const vec3_t clipplanes[]= {{1,1,0},{1,-1,0},{0,-1,1},{0,1,1},{1,0,1},{-1,0,1}};
//const vec3_t clipplanes[]= {{-1,0,1},{0,1,1},{1,-1,0},{0,-1,1},{1,1,0},{1,0,1}};
void SkyBox::ClipSkyPolygon(vec3_t *verts, int num_verts, int planenumber)
{
  if (planenumber == 6)
  {
    DetermineMinMax(verts, num_verts);
    return;
  }

  if (num_verts >= MAX_NUM_CLIPSKYVERTS) return;  // buffer full (@todo generate error)

  int   sides[MAX_NUM_CLIPSKYVERTS+1];
  float dists[MAX_NUM_CLIPSKYVERTS+1];
  int num_backside = 0, num_frontside = 0;
  int v;

  vec3_t plane;
  VectorCopy(clipplanes[planenumber], plane);

  for (v = 0; v < num_verts; ++v)
  {
    dists[v] = DotProduct(verts[v], plane);
    if (dists[v] > 0.1f)
    {
      sides[v] = FRONTSIDE;
      num_frontside = 1;  // abuse as boolean :-)
    }
    else if (dists[v] < -0.1f)
    {
      sides[v] = BACKSIDE;
      num_backside = 1;
    }
    else 
    {
      sides[v] = ON_PLANE;
    }
  }
  if (num_backside^num_frontside)
  {
    ClipSkyPolygon(verts, num_verts, planenumber + 1);
    return;
  }

  num_frontside=num_backside=0;

  vec3_t newvert;
  vec3_t backside[MAX_NUM_CLIPSKYVERTS+2];
  vec3_t frontside[MAX_NUM_CLIPSKYVERTS+2];
  sides[v] = sides[0]; // clone vertex so we always have a valid v+1 vertex
  dists[v] = dists[0];
  VectorCopy(verts[0], verts[v]);

  for (v = 0; v < num_verts; ++v)
  {
    switch (sides[v])
    {
      case FRONTSIDE:
        VectorCopy(verts[v], frontside[num_frontside]); ++num_frontside;
        break;
      case BACKSIDE:
        VectorCopy(verts[v], backside[num_backside]); ++num_backside;
        break;
      case ON_PLANE:
        VectorCopy(verts[v], frontside[num_frontside]); ++num_frontside;
        VectorCopy(verts[v], backside[num_backside]); ++num_backside;
        continue;
      default:
        break;
    }
    if ((sides[v] | sides[v+1]) != SPANNING) continue;

    float t=dists[v]/(dists[v]-dists[v+1]);
    VectorScale(verts[v+1], t, newvert);
    t = 1.0f - t;
    VectorMA(newvert, t, verts[v], newvert);
    VectorCopy(newvert, frontside[num_frontside]); ++num_frontside;
    VectorCopy(newvert, backside[num_backside]); ++num_backside;
  }

  ClipSkyPolygon(frontside, num_frontside, planenumber + 1);
  ClipSkyPolygon(backside, num_backside, planenumber + 1);
}

#define D_EPSILON 0.001f;
void SkyBox::AutoFill(Camera* camera, int viscount)
{
  if (clouds_shader < 0) return;

  int i, j;
  Surface *surf;
  vec3_t skypolygons[MAX_NUM_CLIPSKYVERTS];

  if (sky_debug.ivalue)
  {
    VectorCopy(camera->forward, skypolygons[0]);
    DetermineMinMax(skypolygons, 0);
  }
  else
  {
    vec3_t skyboxrays[MAX_NUM_SKYVERTS];

    for (i = 0; i < nsurfs; ++i)
    {
      surf = surf_array[i];
      if (!surf || surf->shader < 0 ||
        (viscount != -1 && surf->framenum != viscount)) continue; // PVS optimisation

      for (j = 0; j < surf->numverts[0]; ++j)
      {
        VectorSub(surf->firstvert[0][j].v_point, camera->pos, skyboxrays[j]);
      }
      
      for (j = 0; j < surf->numelems[0]; j += 3)
      {
        VectorCopy(skyboxrays[surf->firstelem[0][j]], skypolygons[0]);
        VectorCopy(skyboxrays[surf->firstelem[0][j+1]], skypolygons[1]);
        VectorCopy(skyboxrays[surf->firstelem[0][j+2]], skypolygons[2]);
        ClipSkyPolygon(skypolygons, 3, 0);
      }
    }
  }

  // Fill in the minmax bounds
  int minmaxelems[2];
  int width, nx, ny;
  for (i = 0; i < 6; ++i)
  {
    if (!usedskysides[i]) continue;

    if (minmax[i][SKYMINMAX_MINX] <= -CUBE_SIZE) minmax[i][SKYMINMAX_MINX] = -CUBE_SIZE+D_EPSILON;
    if (minmax[i][SKYMINMAX_MAXX] >=  CUBE_SIZE) minmax[i][SKYMINMAX_MAXX] =  CUBE_SIZE-D_EPSILON;
    if (minmax[i][SKYMINMAX_MINY] <= -CUBE_SIZE) minmax[i][SKYMINMAX_MINY] = -CUBE_SIZE+D_EPSILON;
    if (minmax[i][SKYMINMAX_MAXY] >=  CUBE_SIZE) minmax[i][SKYMINMAX_MAXY] =  CUBE_SIZE-D_EPSILON;

    nx = 6*((int)(minmax[i][SKYMINMAX_MINX]*slope+h));
    ny = 6*((int)(minmax[i][SKYMINMAX_MINY]*slope+h));
    minmaxelems[0] = (side_size-1)*ny+nx;

    ny = 6*((int)(minmax[i][SKYMINMAX_MAXY]*slope+h));
    minmaxelems[1] = (side_size-1)*ny+nx;

    nx = 6*((int)(minmax[i][SKYMINMAX_MAXX]*slope+h));
    ny = 6*((int)(minmax[i][SKYMINMAX_MINY]*slope+h));
    width = ((side_size-1)*ny+nx-minmaxelems[0]+6)*sizeof(int);

    // Fills the quads
    for (j = minmaxelems[0]; j <= minmaxelems[1]; j += 6*(side_size-1))
      memset(&framenum[i][j], 1, width);
  }
}

void SkyBox::FillSide(int side)
{
  memset(framenum[side], 1, elem_len*sizeof(int));
}

void SkyBox::AddMinMax(float x, float y, int p)
{
  if (x < minmax[p][SKYMINMAX_MINX]) minmax[p][SKYMINMAX_MINX] = x;
  if (x > minmax[p][SKYMINMAX_MAXX]) minmax[p][SKYMINMAX_MAXX] = x;
  if (y < minmax[p][SKYMINMAX_MINY]) minmax[p][SKYMINMAX_MINY] = y;
  if (y > minmax[p][SKYMINMAX_MAXY]) minmax[p][SKYMINMAX_MAXY] = y;
}

// Convert cube-relative coordinates to side-relative coordinates and mark
// corresponding elements.
// TODO: Solve border errors
void SkyBox::MarkElem(float x, float y, int p)
{
  int nx = 6*((int)(x*slope+h));
  int ny = 6*((int)(y*slope+h));

  int i = (side_size-1)*ny+nx;

#if 1
  memset(&framenum[p][i], 1, 6*sizeof(int));
#else
  framenum[p][i] =
  framenum[p][i+1] =
  framenum[p][i+2] = 
  framenum[p][i+3] =
  framenum[p][i+4] =
  framenum[p][i+5] = 1;
#endif
}
