//-----------------------------------------------------------------------------
// Overlay
//-----------------------------------------------------------------------------

#include "overlay.h"
#include "console.h"
#include "logfile.h"
#include "shader.h"
#include "render.h"
#include "math.h"
#include "mem.h"
#include "system.h"
#include <string.h>

void Overlay::Init(void)
{
  quads = NULL;
  elems = NULL;
  verts = NULL;
  num_quads = 0;
  num_elems = 0;
  num_verts = 0;

  CreateQuads();
  CreateElems();
  CreateVerts();
}

void Overlay::Shut(void)
{
  if (quads)
  {
    for (int i = 0; i < MAX_OVERLAY_QUADS; ++i) quads[i].Shut();
    delete [] quads;
  }
  if (elems) cake_free(elems);
  if (verts) cake_free(verts);
  quads = NULL;
  elems = NULL;
  verts = NULL;
}

void Overlay::Render(ShaderManager *shaders)
{
  if (!shaders) return;

  if (num_quads)
  {
    Shader *s;

    int last_shader = -1;

    gRender->InitializeViewPort();        // set up 2d viewport
  
    for (unsigned int i = 0; i < num_quads; ++i)
    {
      if (quads[i].shader < 0) continue;
      if (last_shader != quads[i].shader)
      {
        last_shader = quads[i].shader;
        s = shaders->GetShader(quads[i].shader);
        gRender->SetRenderState(s);
      }
      gRender->PushTriangles(&quads[i]);
    }

    num_quads = 0;
    num_elems = 0;
    num_verts = 0;

    gRender->ForceFlush();
  }
}

void Overlay::Quad(int shader, float x, float y, float w, float h, float repx, float repy)
{
  if (MAX_OVERLAY_QUADS <= num_quads + 0 ||
    MAX_OVERLAY_ELEMS <= num_elems + 6 ||
    MAX_OVERLAY_VERTS <= num_verts + 4)
  {
    //if (gConsole) gConsole->Insertln("^1ERROR: Overlay's buffers are too small.");
    //else if (gLogFile) gLogFile->Insert("^1ERROR: Overlay's buffers are too small\n");
    return;   // should stop ?
  }

  quads[num_quads].firstelem[0] = elems;
  quads[num_quads].numelems[0] = 6;
  quads[num_quads].firstvert[0] = verts + num_verts;
  quads[num_quads].numverts[0] = 4;
  quads[num_quads].shader = shader;
  
  elems[num_elems + 0] = num_verts + 0;
  elems[num_elems + 1] = num_verts + 1;
  elems[num_elems + 2] = num_verts + 2;
  elems[num_elems + 3] = num_verts + 2;
  elems[num_elems + 4] = num_verts + 1;
  elems[num_elems + 5] = num_verts + 3;

  verts[num_verts + 0].tex_st[0] = 0.0f;
  verts[num_verts + 0].tex_st[1] = 0.0f;
  verts[num_verts + 0].colour[0] =
  verts[num_verts + 0].colour[1] = 
  verts[num_verts + 0].colour[2] = 
  verts[num_verts + 0].colour[3] = 255;
  verts[num_verts + 0].v_point[0] = x;
  verts[num_verts + 0].v_point[1] = y;
  verts[num_verts + 0].v_point[2] = 0;
  
  verts[num_verts + 1].tex_st[0] = repx;
  verts[num_verts + 1].tex_st[1] = 0.0f;
  verts[num_verts + 1].colour[0] =
  verts[num_verts + 1].colour[1] = 
  verts[num_verts + 1].colour[2] = 
  verts[num_verts + 1].colour[3] = 255;
  verts[num_verts + 1].v_point[0] = x + w;
  verts[num_verts + 1].v_point[1] = y;
  verts[num_verts + 1].v_point[2] = 0;

  verts[num_verts + 2].tex_st[0] = 0.0f;
  verts[num_verts + 2].tex_st[1] = repy;
  verts[num_verts + 2].colour[0] =
  verts[num_verts + 2].colour[1] = 
  verts[num_verts + 2].colour[2] = 
  verts[num_verts + 2].colour[3] = 255;
  verts[num_verts + 2].v_point[0] = x;
  verts[num_verts + 2].v_point[1] = y + h;
  verts[num_verts + 2].v_point[2] = 0;

  verts[num_verts + 3].tex_st[0] = repx;
  verts[num_verts + 3].tex_st[1] = repy;
  verts[num_verts + 3].colour[0] =
  verts[num_verts + 3].colour[1] = 
  verts[num_verts + 3].colour[2] = 
  verts[num_verts + 3].colour[3] = 255;
  verts[num_verts + 3].v_point[0] = x + w;
  verts[num_verts + 3].v_point[1] = y + h;
  verts[num_verts + 3].v_point[2] = 0;

  num_quads += 1;
  num_elems += 6;
  num_verts += 4;
}

void Overlay::String(const char *str,
           int shader,
           float x, float y,
           float xsize, float ysize,
           float rows, float cols,
           int fontstyle)
{

  if (fontstyle & FONT_SHADOWED)
  {
    char *shadowstr = new char[strlen(str)+3];
    sprintf(shadowstr, "^7%s", str);
    
    for (char *p = shadowstr+2; strlen(p); ++p)
    {
      if (*p == '^' && (*(p+1) >= '0' || *(p+1) <= '9'))
      {
        strcpy(p, p+2);
      }
    }

    String(shadowstr, shader, x+2, y+2, xsize, ysize, rows, cols, 0);
    delete [] shadowstr;
  }

  float xnext, xpos, ypos;  // position in the texture (for texture coordinates)
  int num_chars, color = 0;

  // Buffer is too small - skip the string creation
  // Display will be ugly but program shouldn't crash
  if (MAX_OVERLAY_QUADS <= num_quads) return;
  
  num_chars = (int) strlen(str);
  
  quads[num_quads].firstelem[0] = elems;
  quads[num_quads].firstvert[0] = verts + num_verts;
  quads[num_quads].shader = shader;

  quads[num_quads].numelems[0] = num_chars*6; // 6 elems for a char
  quads[num_quads].numverts[0] = num_chars*4; // 4 verts for a char

  for (int i = 0; i < num_chars; ++i)
  {
    if (str[i] == '^')  // color code
    {
      ++i;
      color = str[i] - '0';

      quads[num_quads].numelems[0] -= 12;
      quads[num_quads].numverts[0] -= 8;
      continue;
    }

    xpos = ((float) ((str[i]&15)<<4)/256.0f)*cols/16.f;
    ypos = ((float) ((str[i]>>4)<<4)/256.0f)*rows/16.f;
    xnext = x + xsize;

    if (str[i] == ' ')
    {
      x = xnext;
      quads[num_quads].numelems[0] -= 6;
      quads[num_quads].numverts[0] -= 4;
      continue;
    }

    if (MAX_OVERLAY_ELEMS <= num_elems + 6 ||
      MAX_OVERLAY_VERTS <= num_verts + 4)
    {
      num_elems -= (6*i);
      num_verts -= (4*i);
      return;
    }

    elems[num_elems + 0] = num_verts + 0;
    elems[num_elems + 1] = num_verts + 1;
    elems[num_elems + 2] = num_verts + 2;
    elems[num_elems + 3] = num_verts + 2;
    elems[num_elems + 4] = num_verts + 1;
    elems[num_elems + 5] = num_verts + 3;

    verts[num_verts + 0].tex_st[0] = xpos;
    verts[num_verts + 0].tex_st[1] = ypos;
    ColorCopy(text_colors[color], verts[num_verts + 0].colour);
    verts[num_verts + 0].v_point[0] = x;
    verts[num_verts + 0].v_point[1] = y;
    verts[num_verts + 0].v_point[2] = 0;
  
    verts[num_verts + 1].tex_st[0] = xpos + (cols-1)/256.0f;
    verts[num_verts + 1].tex_st[1] = ypos;
    ColorCopy(text_colors[color], verts[num_verts + 1].colour);
    verts[num_verts + 1].v_point[0] = xnext;
    verts[num_verts + 1].v_point[1] = y;
    verts[num_verts + 1].v_point[2] = 0;

    verts[num_verts + 2].tex_st[0] = xpos;
    verts[num_verts + 2].tex_st[1] = ypos + rows/256.0f;
    ColorCopy(text_colors[color], verts[num_verts + 2].colour);
    verts[num_verts + 2].v_point[0] = x;
    verts[num_verts + 2].v_point[1] = y + ysize;
    verts[num_verts + 2].v_point[2] = 0;

    verts[num_verts + 3].tex_st[0] = xpos + (cols-1)/256.0f;
    verts[num_verts + 3].tex_st[1] = ypos + rows/256.0f;
    ColorCopy(text_colors[color], verts[num_verts + 3].colour);
    verts[num_verts + 3].v_point[0] = xnext;
    verts[num_verts + 3].v_point[1] = y + ysize;
    verts[num_verts + 3].v_point[2] = 0;
    
    x = xnext;
    num_elems += 6;
    num_verts += 4;
  }
  
  num_quads += 1;
}

void Overlay::CreateVerts(void)
{
  verts = (vertex_t*) cake_malloc(MAX_OVERLAY_VERTS*sizeof(vertex_t), "Overlay::CreateVerts.verts");

  if (!verts) ThrowException(ALLOCATION_ERROR, "Overlay::CreateVerts");

  // Initialize the new vertice
  for (int i = 0; i < MAX_OVERLAY_VERTS; ++i)
  {
    VectorClear(verts[i].v_point);
    VectorClear(verts[i].v_norm);
    ColorClear(verts[i].colour);
    TexcoordClear(verts[i].tex_st);
    TexcoordClear(verts[i].lm_st);
  }
}

void Overlay::CreateQuads(void)
{
  quads = new Surface[MAX_OVERLAY_QUADS];

  if (!quads) ThrowException(ALLOCATION_ERROR, "Overlay::CreateQuads");
  
  // Initialize the new quads
  for (int i = 0; i < MAX_OVERLAY_QUADS; ++i) quads[i].Init();
}

void Overlay::CreateElems(void)
{
  elems = (int*) cake_malloc(MAX_OVERLAY_ELEMS*sizeof(int), "Overlay::CreateElems.elems");

  if (!elems) ThrowException(ALLOCATION_ERROR, "Overlay::CreateElems");

  // Initialize the new elems
  for (int i = 0; i < MAX_OVERLAY_ELEMS; ++i) elems[i] = 0;
}
