#ifndef __MD3_H__
#define __MD3_H__

#include "types.h"
#include "q3bsp.h"

// This holds the header information that is read in at the beginning of the file
typedef struct
{ 
  char  fileID[4];          // This stores the file ID - Must be "IDP3"
  int   version;          // This stores the file version - Must be 15
  char  strFile[68];        // This stores the name of the file
  int   numFrames;          // This stores the number of animation frames
  int   numTags;          // This stores the tag count
  int   numMeshes;          // This stores the number of sub-objects in the mesh
  int   numMaxSkins;        // This stores the number of skins for the mesh
  int   headerSize;         // This stores the mesh header size
  int   tagStart;         // This stores the offset into the file for tags
  int   tagEnd;           // This stores the end offset into the file for tags
  int   fileSize;         // This stores the file size
} MD3HEADER;

// This structure is used to read in the mesh data for the .md3 models
typedef struct
{
  char  meshID[4];          // This stores the mesh ID (We don't care)
  char  strName[68];        // This stores the mesh name (We do care)
  int   numMeshFrames;        // This stores the mesh aniamtion frame count
  int   numSkins;         // This stores the mesh skin count
  int     numVertices;        // This stores the mesh vertex count
  int   numTriangles;       // This stores the mesh face count
  int   triStart;         // This stores the starting offset for the triangles
  int   headerSize;         // This stores the header size for the mesh
  int     uvStart;          // This stores the starting offset for the UV coordinates
  int   vertexStart;        // This stores the starting offset for the vertex indices
  int   meshSize;         // This stores the total mesh size
} MD3MESHINFO;

// This is our tag structure for the .MD3 file format.  These are used link other
// models to and the rotate and transate the child models of that model.
typedef struct
{
  char    strName[64];      // This stores the name of the tag (I.E. "tag_torso")
  vec3_t    vPosition;        // This stores the translation that should be performed
  vec_t   rotation[3][3];     // This stores the 3x3 rotation matrix for this frame
} M3DTAG;

// This stores the bone information (useless as far as I can see...)
typedef struct
{
  bboxf_t bbox;           // This is the (x, y, z) mins and maxs for the bone
  float position[3];        // This supposedly stores the bone position???
  float scale;            // This stores the scale of the bone
  char  creator[16];        // The modeler used to create the model (I.E. "3DS Max")
} MD3BONE;

typedef char MD3SKIN[68];       // shader name
typedef float MD3TEXCOORD[2];     // 2 coords
typedef int MD3FACE[3];         // 3 indices

typedef struct 
{
  signed short vertex[3];       // The vertex for this face
  unsigned char normal[2];      // This stores some crazy normal value (not sure...)
} MD3TRIANGLE;

class MD3Loader
{
  public:
    MD3Loader(Q3BSP *bsp);
    ~MD3Loader();

    /**
     * Load a MD3 file into a model.
     * The MD3 can contains more than one model. All models are linked into the entity
     * and entity is returned at the end.
     * @param filename The MD3 file name.
     * @return The new entity that contains all loaded models.
     */
    Entity* LoadMD3(const char *filename);
    void DumpMD3(const char *filename);

  protected:
    VFile*    MD3file;
    MD3HEADER header;
    Q3BSP*    pBSP;

    /**
     * Import a MD3 modelto in a cmodel structure.
     * @param filename The name of MD3 file to load
     * @return A model pointer if loading could achieve, otherwise a NULL pointer
     */
    cmodel_t* ImportMD3(const char *filename);

    /**
     * Read the MD3 data into the given file.
     * @param model The destination model (should already be created).
     * @param modelnum The index of destination model in the BSP model table.
     */
    void ReadMD3Data(cmodel_t *model, int modelnum);

    int LoadSkin(const char *name);
};
#endif  /* __MD3_H__ */
