//-----------------------------------------------------------------------------
// Files
//-----------------------------------------------------------------------------

#include "vars.h"
#include "files.h"
#include "commands.h"
#include "console.h"
#include "logfile.h"
#include "parser.h"
#include "mem.h"
#include "texture.h"
#include "system.h"
#include "definitions.h"
#include <string.h>

#ifdef WIN32
  #include <direct.h>
#else
  #include <sys/types.h>
  #include <dirent.h>
  #include <errno.h>
  #include <unistd.h>
  #include <sys/types.h>
  #include <ctype.h>

  #include <fnmatch.h>
  #include <sys/stat.h>
  #define _getcwd getcwd

  #pragma pack(push, 1)

  typedef struct tagBITMAPFILEHEADER
  {
    WORD bfType;
    DWORD bfSize;
    WORD bfReserved1;
    WORD bfReserved2;
    DWORD bfOffBits;
  } BITMAPFILEHEADER;

  typedef struct tagBITMAPINFOHEADER
  {
    DWORD biSize;
    LONG biWidth;
    LONG biHeight;
    WORD biPlanes;
    WORD biBitCount;
    DWORD biCompression;
    DWORD biSizeImage;
    LONG biXPelsPerMeter;
    LONG biYPelsPerMeter;
    DWORD biClrUsed;
    DWORD biClrImportant;
  } BITMAPINFOHEADER;

  #pragma pack( pop )

  char *lstrlwr(char *s)
  {
    int i = 0;

    while (s[i] != '\0')
    {
      if (isupper(s[i]))
      s[i] = tolower(s[i]);
      ++i;
    }

    return s;
  }

  #define strlwr lstrlwr
  #define _strlwr lstrlwr

  #define BI_RGB    0L
#endif

search_path *FileEnvironment::Search = NULL;
char FileEnvironment::main_path[];

static Pack *packets  = NULL;
int nPackedFiles    = 0;
int nPAKFiles     = 0;

//-----------------------------------------------------------------------------
// Global functions
//-----------------------------------------------------------------------------

void f_StripName(char *path)
{
  int length = (int) strlen(path)-1;

  while (length > 0 && path[length] != '/' && path[length] != '\\')
  {
    --length;
  }
  if (length)
    path[length+1] = 0;
  else 
    path[0] = 0;
}

void f_StripExtension(char *path)
{
  int length = (int) strlen(path)-1;  // ?
  while (length > 0 && path[length] != '.')
  {
    --length;
    if (path[length] == '/')
      return;   // no extension
  }
  if (length)
    path[length] = 0;
}

char * f_Name(char *path)
{
  int length, l;
  l = length = (int) strlen(path)-1;
  while (length > 0 && path[length] != '\\' && path[length] != '/') --length;
  return &path[length+1];
}

char * f_Extension(char *name)
{
  int length, l;
  l = length = (int) strlen(name)-1;  // ?
  while (length > 0 && name[length] != '.')
  {
    --length;
    if (name[length] == '/')
      return &name[l];    // no extension
  }
  if(length==0) return &name[l];
  return &name[length];

  //while(*name!='.'&&*name!=0) ++name;
  //return name;
}

// CAUTION: string must be well formated ('\0' at the end)
inline void f_CorrectSlashInName(char *name)
{
  int i = 0;
  do
  {
    if(name[i] == '\0') break;
    else if(name[i] == '\\') name[i] = '/';
  } while(++i);
}

//-----------------------------------------------------------------------------

void getPAKContent(char *pak_name, char *file_type, char *grep)
{
  int nfounds = 0;

  // Search for local files
  char strpath[MAX_FILEPATH] = { '\0' };
  char path[MAX_FILEPATH];
  strncpy(path, gVars->StringForKey("r_mapsubdir"), MAX_FILEPATH);
  int l = (int) strlen(path);

  if (path[l-1] == '\\' || path[l-1] == '/')
    sprintf(strpath, "%s*%s", path, file_type);
  else
    sprintf(strpath, "%s/*%s", path, file_type);

  SearchFile files(strpath);

  gLogFile->OpenFile();

  for (char *name = files.FirstFile(); name; name = files.NextFile())
  {
    if (!grep || (grep && strstr(strlwr(name), strlwr(grep))))
    {
      gConsole->Insertln("^2%s", name);
      ++nfounds;
    }
  }

  // Search for packed files
  for (Pack* packlist = packets; packlist; packlist = packlist->next)
  {
    if (pak_name == NULL || strstr(strlwr(packlist->name), strlwr(pak_name)))
    {
      for (int i = 0; i < packlist->nFiles; ++i)
      {
        if (file_type == ".*" || strstr(strlwr(packlist->files[i].name), strlwr(file_type)))
        {
          if (!grep || (grep && strstr(strlwr(packlist->files[i].name), strlwr(grep))))
          {
            gConsole->Insertln("^2%s ^0(%s)", packlist->files[i].name, packlist->name);
            ++nfounds;
          }
        }
      }
    }
  }
  gLogFile->CloseFile();
  gConsole->Insertln("%d file%s found", nfounds, nfounds>1?"s":"");
}

void logPreviews(void)
{
  char levelshot_name[256] = { '\0' };
  int levelshot_found = 0;

  // Search for local files
  char strpath[MAX_FILEPATH] = { '\0' };
  char path[MAX_FILEPATH];
  strncpy(path, gVars->StringForKey("r_mapsubdir"), MAX_FILEPATH);
  int l = (int) strlen(path);

  if (path[l-1] == '\\' || path[l-1] == '/')
    sprintf(strpath, "%s*%s", path, ".bsp");
  else
    sprintf(strpath, "%s/*%s", path, ".bsp");

  SearchFile files(strpath);

  gLogFile->OpenFile();

  for (char *name = files.FirstFile(); name; name = files.NextFile())
  {
    gLogFile->Insert("^2%s", name);
    // searh levelshot having TGA or JPG extension
    bool extistga = false;
    char nname[256] = { '\0' };
    int lname = (int) strlen(name)-1;
    while (lname >= 0 && name[lname] != '/' && name[lname] != '\\') --lname;
    strcpy(nname, &name[lname+1]);
    f_StripExtension(nname);
    sprintf(levelshot_name, "levelshots/%s.jpg", nname);
    levelshot_found = FileExist(levelshot_name, 1, 0);
    if (!levelshot_found)
    {
      sprintf(levelshot_name, "levelshots/%s.tga", nname);
      levelshot_found = FileExist(levelshot_name, 1, 0);
      if (levelshot_found) extistga = true;
    }

    if (!levelshot_found)
      strcpy(levelshot_name, "menu/art/unknownmap.jpg");

    VFile levelshotfile(levelshot_name);
    texinfo info;
    if (levelshotfile.mem)
    {
      if (extistga) TGA_Decode(&levelshotfile, &info);
      else JPG_Decode(&levelshotfile, &info);
      FlipTexture(&info);
    }

    char dstfilename[256] = { '\0' };
    sprintf(dstfilename, "shots/%s.bmp", nname);
    WriteBitmapFile(dstfilename, info.width, info.height, info.bpp, info.mem);
    gLogFile->Insert("<br><img src=\"%s\" width=\"%d\" height=\"%d\"><br><br>", dstfilename, info.width, info.height);
  }

  // Search for packed files
  for (Pack* packlist = packets; packlist; packlist = packlist->next)
  {
    for (int i = 0; i < packlist->nFiles; ++i)
    {
      if (strstr(strlwr(packlist->files[i].name), ".bsp"))
      {
        char nname[256] = { '\0' };
        int lname = (int) strlen(packlist->files[i].name)-1;
        while (lname >= 0 && packlist->files[i].name[lname] != '/' && packlist->files[i].name[lname] != '\\') --lname;
        strcpy(nname, &packlist->files[i].name[lname+1]);
        f_StripExtension(nname);

        gLogFile->Insert("^2%s ^0(%s)", packlist->files[i].name, packlist->name);

        // searh levelshot having TGA or JPG extension
        bool extistga = false;
        sprintf(levelshot_name, "levelshots/%s.jpg", nname);
        levelshot_found = FileExist(levelshot_name, 1, 0);
        if (!levelshot_found)
        {
          sprintf(levelshot_name, "levelshots/%s.tga", nname);
          levelshot_found = FileExist(levelshot_name, 1, 0);
          if (levelshot_found) extistga = true;
        }

        if (!levelshot_found)
          strcpy(levelshot_name, "menu/art/unknownmap.jpg");

        VFile levelshotfile(levelshot_name);
        texinfo info;
        if (levelshotfile.mem)
        {
          if (extistga) TGA_Decode(&levelshotfile, &info);
          else JPG_Decode(&levelshotfile, &info);
          FlipTexture(&info);
        }

        char dstfilename[256] = { '\0' };
        sprintf(dstfilename, "shots/%s.bmp", nname);
        WriteBitmapFile(dstfilename, info.width, info.height, info.bpp, info.mem);
        gLogFile->Insert("<br><img src=\"%s\" width=\"%d\" height=\"%d\"><br><br>", dstfilename, info.width, info.height);
      }
    }
  }

  gLogFile->CloseFile();
}

int FileExist(char *file_name, int pak_search, int strip_extension)
{
  FILE * fp;
  char fname[FILE_NAME], sname[FILE_NAME];

  for (search_path *p = FileEnvironment::Search; p; p = p->next)
  {
    sprintf(fname, "%s%s", p->path, file_name);
    if (strip_extension) f_StripExtension(fname);

    f_CorrectSlashInName(fname);

    fp = fopen(fname, "rb");
    if (fp)
    {
      fclose(fp);
      return 1;
    }
  }

  // Si on ne trouve pas le fichier tel quel, on cherche dans les fichiers pak
  if (pak_search)
  {
    for (Pack *p = packets; p; p = p->next)
    {
      for (int i = 0; i < p->nFiles; ++i)
      {
        strcpy(fname, file_name);
        strcpy(sname, p->files[i].name);
        if (strip_extension)
        {
          f_StripExtension(fname);
          f_StripExtension(sname);
        }
        if (stricmp(fname, sname) == 0)
        {
          return 1;
        }
      }
    }
  }

  return 0;
}

//-----------------------------------------------------------------------------
// VFile
//-----------------------------------------------------------------------------

VFile::VFile(const char *name, bool pak_search, const char *pakname)
{
  FILE * fp;
    mem = NULL;
  error = 0;
  char _name[FILE_NAME];

  int lenpak = 0;
  if (pakname) lenpak = (int) strlen(pakname);

  // change all '\\' to '/'
  strcpy(_name, name);
    
  f_CorrectSlashInName(_name);

  // suppress initial '/' or '\\'
  while ((_name[0] == '/' || _name[0] == '\\') && strlen(_name) > 0) strcpy(_name, &(_name[1]));

  // Let's search in pak files
  if (pak_search)
  {
    Pack *p;
    for (p = packets; p; p = p->next)
    {
      if (!lenpak || (lenpak && !stricmp(p->name, pakname)))
      {
        for (int i = 0; i < p->nFiles; ++i)
        {
          if (!stricmp(_name, p->files[i].name))
          {
            if ((fp = fopen(p->name, "rb")) == NULL)
            {
              gConsole->Insertln("^1ERROR: Cannot open %s", p->name);
              return;
            }
            fseek(fp, p->files[i].offset, SEEK_SET);

            mem = (BYTE *) UnZip(fp);
            size = p->files[i].size;
            fclose(fp);
            return;
          }
        }
      }
    }
    if (p) return;
  }

  // The file couldn't be found in pak files, let's search in local folders
  for (search_path *p = FileEnvironment::Search; p; p = p->next)
  {
    sprintf(fname, "%s%s", p->path, _name);

    f_CorrectSlashInName(fname);

    if ((fp = fopen(fname,"rb")) == NULL) continue; // cannot find the file... continue with another path

    // see the size
    fseek(fp, 0L, SEEK_END);
    size = ftell(fp);
    rewind(fp);

    // allocate memory
    mem = new BYTE[size];
    if (mem == NULL)
    {
      fclose(fp);
      gConsole->Insertln("^1ERROR: VFile: Cannot allocate required size");
      // FIXME: critical error!
      return;
    }

    // copy archive to memory
    fread(mem, size, sizeof(BYTE), fp);
    fclose(fp);

    return;
  }

  // display error!! : file not found!
  //gConsole->Insertln("^1ERROR: Error occured while loading file %s", name);
}

VFile::~VFile(void)
{
  // delete memory archive
  if (mem) delete [] mem;
  mem = NULL;
}

//-----------------------------------------------------------------------------
// LumpFile
//-----------------------------------------------------------------------------

LumpFile::LumpFile(const char *name, const int id, const int ver, const int num_lumps):VFile(name)
{
  head = (header *) mem;

  if (!head)
  {
    gConsole->Insertln("^1ERROR: Couldn't read file header");
    return;
  }

    if (head->id != id)
  {
    gConsole->Insertln("^1ERROR: Bad file id in %s; found: %d, expected: %d",
      name, head->id, id);
    delete [] mem; mem = NULL;
    return;
  }
    if (head->ver != ver)
  {
    gConsole->Insertln("^1ERROR: Bad file version in %s; found: %d, expected: %d",
      name, head->ver, ver);
    delete [] mem; mem = NULL;
    return;
  }
}

LumpFile::~LumpFile(void)
{
  // FIXME: como hacer que ste destructor no llame al destructor padre ?
  //      lo llama?
  // en todo caso se debera eliminar mem si es que an no ha sido eliminada.
}

// TODO: test errors...
int LumpFile::ReadLump(int lump, void** dest, size_t elem)
{
  if (!mem) return 0;
  int len = head->lump[lump].filelen;

  // test if size is valid (should stop loading if not, but try anyway)
  if (len % elem) gConsole->Insertln("^1LumpFile::ReadLump: Funny lump size ! Trying to open anyway :\\");

  int num = len / (int) elem;
  
  *dest = cake_malloc(len, "LumpFile::ReadLump");
  if (!(*dest)) ThrowException(ALLOCATION_ERROR, "LumpFile::ReadLump.(*dest)");

  memcpy(*dest, mem + head->lump[lump].fileofs, num * elem);
  return num;
}

//-----------------------------------------------------------------------------
// FileEnvironment
//-----------------------------------------------------------------------------

void FileEnvironment::Init(void)
{
  char pth[MAX_FILEPATH];

  _getcwd(pth, MAX_FILEPATH); // get current path
  #ifdef WIN32
    if (pth[strlen(pth)-1] != '\\') strcat(pth, "\\");
    #else
    if (pth[strlen(pth)-1] != '/') strcat(pth, "/");
    #endif

  // change all '\\' to '/'
  for (int i = 0; i < (int) strlen(pth); ++i) if (pth[i] == '\\') pth[i] = '/';

  strncpy(FileEnvironment::main_path, pth, MAX_FILEPATH);
}

void FileEnvironment::AddPath(const char *path)
{
  search_path *tmp = new search_path;
  int i = 0;

  if (path[0] == '/' ||                               // linux case
    (strlen(path) > 2 && path[1] == ':' && (path[2] == '\\' || path[2] == '/')))  // windows case
  {
    // path use driver letter
    strcpy(tmp->path, path);
  }
  else
  {
    // path is relative to main path
    strncpy(tmp->path, main_path, MAX_FILEPATH);
    
    // adapt path to "../" and "./" elements
    int l = (int) strlen(path), s = 0;
    while (l > 0 && i < l)
    {
      if (i <= l - 3 && path[i] == '.' && path[i+1] == '.' && (path[i+2] == '/' || path[i+2] == '\\'))
      {
        i += 2;
        // remove last dir of the main_path
        int ml = (int) strlen(tmp->path) - 1;
        if (tmp->path[ml] == '\\' || tmp->path[ml] == '/') --ml;
        while (ml >= 0 && (tmp->path[ml] != '\\' && tmp->path[ml] != '/'))
        {
          tmp->path[ml] = '\0';
          --ml;
        }
        s += 3;
      }
      else if (i <= l - 2 && path[i] == '.' && (path[i+1] == '/' || path[i+1] == '\\'))
      {
        ++i;
        s += 2;
      }
      ++i;
    }
    strcat(tmp->path, &(path[s]));
  }

  f_CorrectSlashInName(tmp->path);
  
  tmp->next = FileEnvironment::Search;
  FileEnvironment::Search = tmp;
}

// Search for all pk3 files and add the filenames contained
// in each pk3 file to the "packets" list
void FileEnvironment::LoadPacks(void)
{
  char *      name = NULL;
  FILE *      fp;
  ZIPEnd      End;
  ZIPCtrlHeader File;
  Pack      *packet;
  int       i;

  nPackedFiles = 0;
  nPAKFiles = 0;

  char fname[512];          // FIXME: static allocation

  // we put the supported extensions files in a table
  #define SUPPORTED_EXT 3
  char exts[SUPPORTED_EXT][10];
  for (i = 0; i < SUPPORTED_EXT; ++i) memset(exts[i], 0, 10);
  
  strcpy(exts[0], "*.pk3");
  strcpy(exts[1], "*.pak");
  strcpy(exts[2], "*.zip");

  gConsole->Insertln("<br/><b>^6----- Reading Packs -------------------------------------------</b>");
  gLogFile->OpenFile();
  
  for (i = 0; i < SUPPORTED_EXT; ++i)
  {
    SearchFile packs(exts[i]);
    
    // for each file with supported extensions, open it and add its files to the index
    for (search_path *p = FileEnvironment::Search; p; p = p->next)
    {
      for (name = packs.FirstFile(p->path); name; name = packs.NextFile(p->path))
      {
        memset(fname, '\0', 512*sizeof(char));
        sprintf(fname, "%s%s", p->path, name);

        if ((fp = fopen(fname, "rb")))
        {
          // Read the final header
          fseek(fp, -22, SEEK_END);
          fread(&End,  sizeof(End), 1, fp);
          if (End.Signature != ZIPEndSig)
          {
            gConsole->Insertln("^5WARNING: Corrupted pak file %s", name);
            fclose(fp);
            continue;
          }

          gConsole->Insertln("...reading \"%s\" (%d file%s)", fname, End.FilesOnDisk, End.FilesOnDisk>1?"s":"");

          ++nPAKFiles;

          packet = new Pack;  // Add the packet to packets list
          if (!packet) ThrowException(ALLOCATION_ERROR, "Files::LoadPacks.packet");

          packet->name = new char[strlen(fname)+1];
          if (!packet->name) ThrowException(ALLOCATION_ERROR, "Files::LoadPacks.packet->name");
          memset(packet->name, '\0', (strlen(fname)+1)*sizeof(char));
          strcpy(packet->name, fname);
          packet->nFiles = End.FilesOnDisk;
          packet->files = new PackedFile[End.FilesOnDisk];
          if (!packet->files) ThrowException(ALLOCATION_ERROR, "Files::LoadPacks.packet->files");
          packet->next = packets;
          packets = packet;

          // Read the header of each file
          fseek(fp, End.Offset, SEEK_SET);
          for (int n = 0; n < End.FilesOnDisk; ++n)
          {
            fread(&File, sizeof(File), 1, fp);
            if (File.Signature != ZIPCtrlHeaderSig)
            {
              gConsole->Insertln("^5WARNING: Corrupted file %s", name);
              fclose(fp);
              break;      // continue with next pak
            }

            packet->files[n].size = File.UnCompressedSize;
            packet->files[n].offset = File.Offset;
            memset(packet->files[n].name, '\0', 1024); // FIXME: hack???
            fread(packet->files[n].name, 1, File.FileNameLength, fp);

            fseek(fp, File.ExtraLength+File.CommentLength, SEEK_CUR);
          }
          nPackedFiles += End.FilesOnDisk;

          fclose(fp);
        }
      }
    }
  }
  gLogFile->CloseFile();
  gConsole->Insertln("Reading finished - %d packed files in %d packages",
    nPackedFiles, nPAKFiles);
}

void FileEnvironment::Shut(void)
{
  search_path *s, *snext = NULL;

  for (s = FileEnvironment::Search; s; s = snext)
  {
    snext = s->next;
    delete s;
  }

  Pack *p, *pnext = NULL;
  for (p = packets; p; p = pnext)
  {
    delete [] p->name;
    delete [] p->files;
    pnext = p->next;
    delete p;
  }
}

void FileEnvironment::dumpEnvironment(void)
{
  gConsole->Insertln("<b>File environment :</b>");
  gConsole->Insertln("\tmain path : %s", FileEnvironment::main_path);
  search_path *s = FileEnvironment::Search;
  if (s) gConsole->Insertln("\tsearch path :");
  gLogFile->OpenFile();
  while (s)
  {
    gConsole->Insertln("\t\t%s", s->path);
    s = s->next;
  }
  gLogFile->CloseFile();
  gConsole->Insertln("\tmap path : %s", gVars->StringForKey("r_mapsubdir"));
}

//-----------------------------------------------------------------------------
// SearchFile
//-----------------------------------------------------------------------------

SearchFile::SearchFile(char *pattern)
{
  this->pattern = pattern;
  last = FileEnvironment::Search;
  temp_pat = NULL;
}

SearchFile::~SearchFile(void)
{
  if (temp_pat) delete temp_pat;
}

char *SearchFile::FirstFile(void)
{
  #ifdef WIN32
    struct _finddata_t fileinfo;
  #else
    struct stat buf;
    struct dirent* fileinfo;
    char* temp = new char[1024];
  #endif

  if (last == NULL)   // end of the list, reset and return
  {
    last = FileEnvironment::Search;
    return NULL;
  }

  if (temp_pat) delete [] temp_pat;
  temp_pat = new char[strlen(last->path) + strlen(pattern) + 1];
  strcpy(temp_pat, last->path);
  strcat(temp_pat, pattern);

  f_CorrectSlashInName(temp_pat);

  #ifdef WIN32
    handle = (int) _findfirst(temp_pat, &fileinfo);
    if (handle == -1)
    {
      _findclose(handle);
      if (temp_pat) delete [] temp_pat;
      temp_pat = NULL;
      last = last->next;
      return FirstFile();
    }

    // skip non-valid filenames
    while ((fileinfo.attrib & _A_SUBDIR) || fileinfo.name[0] == '.')
    {
      if (_findnext(handle, &fileinfo) == -1)
      {
        _findclose(handle);
        return NULL;
      }
    }
  #else
    handle = opendir(last->path);

    if (handle == NULL)
    {
      gConsole->Insertln("^1handle == NULL: %d - %s", errno, strerror(errno));
      if (temp_pat) delete [] temp_pat;
      temp_pat = NULL;
      last = last->next;
      return FirstFile();
    }

    while ((fileinfo = readdir(handle)) != NULL)
    {
      memset(&buf, 0, sizeof(buf));
      memset(temp, 0, sizeof(temp));

      strcpy(temp, last->path );
      strcat(temp, fileinfo->d_name);

      if (stat(temp, &buf ) < 0)
        gConsole->Insertln("^6stat: %d - %s", errno, strerror(errno));

      if (!S_ISDIR(buf.st_mode))
        if (!fnmatch(pattern, fileinfo->d_name, FNM_PERIOD))
          break;
    }

    if (fileinfo == NULL)
    {
      gConsole->Insertln("^1readdir handle == NULL: %d - %s", errno, strerror(errno));
      closedir(handle);
      return NULL;
    }
  #endif

  strcpy(lastpath, pattern);
  f_StripName(lastpath);
  #ifdef WIN32
    strcat(lastpath, fileinfo.name);
  #else
    strcat(lastpath, fileinfo->d_name);
  #endif
  return lastpath;
}

char *SearchFile::FirstFile(const char *path)
{
  #ifdef WIN32
    struct _finddata_t fileinfo;
  #else
    struct stat buf;
    struct dirent* fileinfo;
    char* temp = new char[1024];
  #endif

  if (path == NULL)   // end of the list, reset and return
  {
    return NULL;
  }

  if (temp_pat) delete [] temp_pat;
  temp_pat = new char[strlen(path) + strlen(pattern) + 1];
  strcpy(temp_pat, path);
  strcat(temp_pat, pattern);

  f_CorrectSlashInName(temp_pat);

  #ifdef WIN32
    handle = (int) _findfirst(temp_pat, &fileinfo);
    if (handle == -1)
    {
      _findclose(handle);
      if (temp_pat) delete [] temp_pat;
      temp_pat = NULL;
      return NULL;
    }

    // skip non-valid filenames
    while ((fileinfo.attrib & _A_SUBDIR) || fileinfo.name[0] == '.')
    {
      if (_findnext(handle, &fileinfo) == -1)
      {
        _findclose(handle);
        return NULL;
      }
    }
  #else
    handle = opendir(path);

    if (handle == NULL)
    {
      gConsole->Insertln("^1handle == NULL: %d - %s", errno, strerror(errno));
      if (temp_pat) delete [] temp_pat;
      temp_pat = NULL;
      return NULL;
    }

    while ((fileinfo = readdir(handle)) != NULL)
    {
      memset(&buf, 0, sizeof(buf));
      memset(temp, 0, sizeof(temp));

      strcpy(temp, path);
      strcat(temp, fileinfo->d_name);

      if (stat(temp, &buf) < 0)
        gConsole->Insertln("^6stat: %d - %s", errno, strerror(errno));

      if (!S_ISDIR(buf.st_mode))
        if (!fnmatch( pattern, fileinfo->d_name, FNM_PERIOD))
          break;
    }

    if (fileinfo == NULL)
    {
      gConsole->Insertln("^1readdir handle == NULL: %d - %s", errno, strerror(errno));
      closedir(handle);
      return NULL;
    }
  #endif

  strcpy(lastpath, pattern);
  f_StripName(lastpath);
  #ifdef WIN32
    strcat(lastpath, fileinfo.name);
  #else
    strcat(lastpath, fileinfo->d_name);
  #endif
  return lastpath;
}

char *SearchFile::NextFile(void)
{
  #ifdef WIN32
    struct _finddata_t fileinfo;

    while (_findnext(handle, &fileinfo) != -1)
    {
      if ((fileinfo.attrib & _A_SUBDIR)) continue;
      if (fileinfo.name[0] == '.') continue;
      
      strcpy(lastpath, pattern);
      f_StripName(lastpath);
      strcat(lastpath, fileinfo.name);
      return lastpath;
    }
    _findclose(handle);
  #else
    struct stat buf;
    struct dirent* fileinfo;
    char* temp = new char[1024];

    while ((fileinfo = readdir(handle)) != NULL)
    {
      memset(temp, 0, sizeof(temp));
      strcpy(temp, last->path);
      strcat(temp, fileinfo->d_name);

      if( stat(temp, &buf) < 0 )
        gConsole->Insertln("^6stat: %d - %s", errno, strerror(errno));

      if (S_ISDIR(buf.st_mode) || (fnmatch(pattern, fileinfo->d_name, FNM_PERIOD) == FNM_NOMATCH))
        continue;

      strcpy(lastpath, pattern);
      f_StripName(lastpath);
      strcat(lastpath, fileinfo->d_name);
      return lastpath;
    }
    closedir(handle);
  #endif

  if (temp_pat) delete [] temp_pat;
  temp_pat = NULL;
  last = last->next;
  return FirstFile();
}

char *SearchFile::NextFile(const char *path)
{
  #ifdef WIN32
    struct _finddata_t fileinfo;

    while (_findnext(handle, &fileinfo) != -1)
    {
      if ((fileinfo.attrib & _A_SUBDIR)) continue;
      if (fileinfo.name[0] == '.') continue;

      strcpy(lastpath, pattern);
      f_StripName(lastpath);
      strcat(lastpath, fileinfo.name);
      return lastpath;
    }
    _findclose(handle);
  #else
    struct stat buf;
    struct dirent* fileinfo;
    char* temp = new char[1024];

    while ((fileinfo = readdir(handle)) != NULL)
    {
      memset(temp, 0, sizeof(temp));
      strcpy(temp, last->path);
      strcat(temp, fileinfo->d_name);

      if (stat(temp, &buf) < 0)
        gConsole->Insertln("^6stat: %d - %s", errno, strerror(errno));

      if (S_ISDIR(buf.st_mode) || (fnmatch(pattern, fileinfo->d_name, FNM_PERIOD) == FNM_NOMATCH))
        continue;

      strcpy(lastpath, pattern);
      f_StripName(lastpath);
      strcat(lastpath, fileinfo->d_name);
      return lastpath;
    }
    closedir(handle);
  #endif

  if (temp_pat) delete [] temp_pat;
  temp_pat = NULL;
  return NULL;
}

//-----------------------------------------------------------------------------
// Config File
//-----------------------------------------------------------------------------

ConfigFile::ConfigFile(char *name):VFile(name)
{
  char buffer[256];
  if (mem)
  {
    gCommands->AddToConsole = false;
    
    gLogFile->Insert("<b>Loading config file...</b> (%s)\n", name);

    Parser::StartParseBuffer(mem, size);
  
    gLogFile->OpenFile();
    while (Parser::GetToken(true))
    {
      memset(buffer, 0, 256*sizeof(char));
      strcat(buffer, Parser::token);
      while (Parser::GetToken(false))
      {
        strcat(buffer, " ");
        strcat(buffer, Parser::token);
      }
      gLogFile->Insert("\t%d:\t%s\n", Parser::scriptline, buffer);
      if (!gCommands->ExecuteCommand(buffer))
        gLogFile->Insert("\t\t^5WARNING, Command not found: %s\n", buffer);
    }
    gLogFile->CloseFile();
    Parser::StopParseFile();

    gCommands->AddToConsole = true;
  }
  else
  {
    gConsole->Insertln("^5WARNING: Unable to read configuration file. Using default settings.");
  }
}

//-----------------------------------------------------------------------------
// Shader initialization
//-----------------------------------------------------------------------------

#if SHADER_USE_REFERENCES
  static shader_ref_list *shader_refs = NULL; // Shader references

  int AddShaderToList(ShaderFile *fp, const char *shader, const char *file, const char *pak)
  {
    shader_ref_list *ref;

    #if !SHADER_ALLOWDUPLICATE
      for (ref = shader_refs; ref; ref = ref->next)
        if (!strnicmp(shader, ref->shader_name, SHADER_NAME_MAX_LENGTH)) return 0;
    #endif

    ref = (shader_ref_list*) cake_malloc(sizeof(shader_ref_list), "Files::AddShaderToList.ref");
    if (!ref) ThrowException(ALLOCATION_ERROR, "Files::AddShaderToList.ref");

    strncpy(ref->shader_name, shader, SHADER_NAME_MAX_LENGTH);

    memset(ref->file_name, '\0', SHADER_NAME_MAX_LENGTH);
    if (file) strncpy(ref->file_name, file, SHADER_NAME_MAX_LENGTH);

    memset(ref->pak_name, '\0', SHADER_NAME_MAX_LENGTH);
    if (pak) strncpy(ref->pak_name, pak, SHADER_NAME_MAX_LENGTH);

    ref->offset = fp->GetShaderPos();
    ref->next = shader_refs;
    shader_refs = ref;

    return 1;
  }
#else
  static shaders_file_list *shaders_files = NULL;
  static shaders_file_list *curr= NULL;

  int AddShadersFileToList(const char *file)
  {
    shaders_file_list *ref;

    #if !SHADER_ALLOWDUPLICATE
      for (ref = shaders_files; ref; ref = ref->next)
        if (!strnicmp(file, ref->file_name, SHADER_NAME_MAX_LENGTH)) return 0;
    #endif

    ref = (shaders_file_list*) cake_malloc(sizeof(shaders_file_list), "Files::AddShadersFileToList.ref");
    if (!ref) ThrowException(ALLOCATION_ERROR, "AddShadersFileToList.ref");

    strncpy(ref->file_name, file, SHADER_NAME_MAX_LENGTH);
    ref->next = shaders_files;
    shaders_files = ref;

    return 1;
  }
#endif

void InitializeShaders(void)
{
  gConsole->Insertln("<br/><b>^6----- Initializing shaders ------------------------------------</b>");
  int numshaderrefs = 0, numfiles = 0, numskippedfile = 0;

  DestroyShaderList();

  // PAK files
  int l;
  char *name;
  
  gLogFile->OpenFile();
  for (Pack *p = packets; p; p = p->next)
  {
    for (int i = 0; i < p->nFiles; ++i)
    {
      l = (int) strlen(p->files[i].name);
      if (l > 7 &&
        !strnicmp(&p->files[i].name[l-7], ".shader", 7) &&
        (!strnicmp(p->files[i].name, "scripts/", 8) || !strnicmp(p->files[i].name, "scripts\\", 8)))
      {
        ++numfiles;
        gConsole->Insertln("...loading \"%s\" (from \"%s\")", (char *) p->files[i].name, p->name);
        #if SHADER_USE_REFERENCES
        {
          char *shadername;
          ShaderFile file((char *) p->files[i].name, p->name);

          // parse each shader
          while ((shadername = file.GetShaderName()))
          {
            if (AddShaderToList(&file, shadername, (char *) p->files[i].name, p->name)) ++numshaderrefs;
            file.Skip();
          }
        }
        #else
          if (!AddShadersFileToList((char *) p->files[i].name)) ++numskippedfile;
        #endif
      }
    }
  }

  // Local files
  SearchFile search("scripts/*.shader");
  
  // for each .shader file
  for (name = search.FirstFile(); name; name = search.NextFile(), ++numfiles)
  {
    // opens the file:
    gConsole->Insertln("...loading \"%s\" (local file)", name);
    #if SHADER_USE_REFERENCES
    {
      char *shadername;
      ShaderFile file((char *) name);

      // parse each shader
      while ((shadername = file.GetShaderName()))
      {
        if (AddShaderToList(&file, shadername, name, NULL)) ++numshaderrefs;
        file.Skip();
      }
    }
    #else
      if (!AddShadersFileToList(name)) ++numskippedfile;
    #endif
  }
  gLogFile->CloseFile();

  gConsole->Insert("Initialization finished - ");
  #if SHADER_USE_REFERENCES
    gConsole->Insertln("%d shaders referenced in %d files (%d skipped)", numshaderrefs, numfiles, numskippedfile);
  #else
    gConsole->Insertln("found %d files containing shaders (%d skipped)", numfiles, numskippedfile);
  #endif
}

void DestroyShaderList(void)
{
  #if SHADER_USE_REFERENCES
    // Unallocation of completion list
    if (shader_refs)
    {
      for (; shader_refs;)
      {
        shader_ref_list *back = shader_refs->next;
        shader_refs->next = NULL;
        cake_free(shader_refs);
        shader_refs = back;
        back = NULL;
      }
      shader_refs = NULL;
    }
  #else
    curr = NULL;

    // Unallocation of completion list
    if (shaders_files)
    {
      for (; shaders_files;)
      {
        shaders_file_list *back = shaders_files->next;
        shaders_files->next = NULL;
        cake_free(shaders_files);
        shaders_files = back;
        back = NULL;
      }
      shaders_files = NULL;
    }
  #endif
}

#if SHADER_USE_REFERENCES
  shader_ref_list *RefForShader(const char *shader)
  {
    for (shader_ref_list *ref = shader_refs; ref; ref = ref->next)
    {
      if (!strnicmp(shader, (const char*) ref->shader_name, SHADER_NAME_MAX_LENGTH))
      {
        return ref;
      }
    }

    return NULL;
  }
#else
  char *GetFirstShadersFile(void)
  {
    curr = shaders_files;
    return curr->file_name;
  }

  char *NextShadersFile(void)
  {
    curr = curr->next;
    if (curr) return curr->file_name;
    else return NULL;
  }
#endif

//-----------------------------------------------------------------------------
// Bitmap file writer
//-----------------------------------------------------------------------------
int WriteBitmapFile(char *filename, int width, int height, int bpp, unsigned char *imageData)
{
  FILE       *filePtr;      // file pointer
  BITMAPFILEHEADER bitmapFileHeader;  // bitmap file header
  BITMAPINFOHEADER bitmapInfoHeader;  // bitmap info header
  int        imageIdx;      // used for swapping RGB->BGR
  unsigned char  tempRGB;     // used for swapping

  // open file for writing binary mode
  filePtr = fopen(filename, "wb");
  if (!filePtr) return 0;

  // define the bitmap file header
  bitmapFileHeader.bfSize = sizeof(BITMAPFILEHEADER);
  bitmapFileHeader.bfType = 0x4D42;
  bitmapFileHeader.bfReserved1 = 0;
  bitmapFileHeader.bfReserved2 = 0;
  bitmapFileHeader.bfOffBits = sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER);
  
  // define the bitmap information header
  bitmapInfoHeader.biSize = sizeof(BITMAPINFOHEADER);
  bitmapInfoHeader.biPlanes = 1;
  bitmapInfoHeader.biBitCount = bpp*8;
  bitmapInfoHeader.biCompression = BI_RGB;          // no compression
  bitmapInfoHeader.biSizeImage = width * abs(height) * bpp; // width * height * bpp
  bitmapInfoHeader.biXPelsPerMeter = 0;
  bitmapInfoHeader.biYPelsPerMeter = 0;
  bitmapInfoHeader.biClrUsed = 0;
  bitmapInfoHeader.biClrImportant = 0;
  bitmapInfoHeader.biWidth = width;             // bitmap width
  bitmapInfoHeader.biHeight = height;             // bitmap height

  // switch the image data from RGB to BGR
  for (imageIdx = 0; imageIdx < (int) bitmapInfoHeader.biSizeImage; imageIdx+=bpp)
  {
    tempRGB = imageData[imageIdx];
    imageData[imageIdx] = imageData[imageIdx + 2];
    imageData[imageIdx + 2] = tempRGB;
  }

  // write the bitmap file header
  fwrite(&bitmapFileHeader, 1, sizeof(BITMAPFILEHEADER), filePtr);

  // write the bitmap info header
  fwrite(&bitmapInfoHeader, 1, sizeof(BITMAPINFOHEADER), filePtr);

  // write the image data
  fwrite(imageData, 1, bitmapInfoHeader.biSizeImage, filePtr);

  // close our file
  fclose(filePtr);

  return 1;
}
