//-----------------------------------------------------------------------------
// Entities
//-----------------------------------------------------------------------------

#include "entity.h"
#include "definitions.h"
#include "q3bsp.h"
#include "md3.h"
#include "render.h"
#include "mem.h"
#include "camera.h"
#include "console.h"
#include "logfile.h"
#include "cake.h"
#include "parser.h"
#include "math.h"
#include "timer.h"
#include "system.h"
#include "sound.h"

#include <stdio.h>
#include <string.h>     // for memcpy

#define MAX_ENTITY_KEYLENGTH  64
#define MAX_ENTITY_VALLENGTH  256
#define ENTITY_ALLOCATION_BLOC  8

/**
 * key-value pair.
 */
typedef struct
{
  char *key;
  char *val;
} epair_t;

/**
 * Group of epairs.
 */
typedef struct
{
  // parsing elements
  int firstpair;
  int numpairs;
} parse_entity_t;

int num_epairs;
epair_t * epairs = NULL;
parse_entity_t * parse_entities = NULL;

gitem_t EntityManager::bg_itemlist[] = 
{
  {
    NULL,
    NULL,
    { NULL,
    NULL,
    0, 0} ,
    NULL,       // icon
    NULL,       // pickup
    0,
    0,
    0,
    "",         // precache
    ""            // sounds
  },  // leave index 0 alone

  //
  // ARMOR
  //

/*QUAKED item_armor_shard (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_armor_shard", 
    "sound/misc/ar1_pkup.wav",
    { "models/powerups/armor/shard.md3", 
    0,//"models/powerups/armor/shard_sphere.md3",
    0, 0} ,
/* icon */    "icons/iconr_shard",
/* pickup */  "Armor Shard",
    5,
    IT_ARMOR,
    0,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED item_armor_combat (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_armor_combat", 
    "sound/misc/ar2_pkup.wav",
        { "models/powerups/armor/armor_yel.md3",
    0, 0, 0},
/* icon */    "icons/iconr_yellow",
/* pickup */  "Armor",
    50,
    IT_ARMOR,
    0,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED item_armor_body (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_armor_body", 
    "sound/misc/ar2_pkup.wav",
        { "models/powerups/armor/armor_red.md3",
    0, 0, 0},
/* icon */    "icons/iconr_red",
/* pickup */  "Heavy Armor",
    100,
    IT_ARMOR,
    0,
/* precache */ "",
/* sounds */ ""
  },

  //
  // health
  //
/*QUAKED item_health_small (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_health_small",
    "sound/items/s_health.wav",
        { "models/powerups/health/small_cross.md3", 
    "models/powerups/health/small_sphere.md3", 
    0, 0 },
/* icon */    "icons/iconh_green",
/* pickup */  "5 Health",
    5,
    IT_HEALTH,
    0,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED item_health (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_health",
    "sound/items/n_health.wav",
        { "models/powerups/health/medium_cross.md3", 
    "models/powerups/health/medium_sphere.md3", 
    0, 0 },
/* icon */    "icons/iconh_yellow",
/* pickup */  "25 Health",
    25,
    IT_HEALTH,
    0,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED item_health_large (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_health_large",
    "sound/items/l_health.wav",
        { "models/powerups/health/large_cross.md3", 
    "models/powerups/health/large_sphere.md3", 
    0, 0 },
/* icon */    "icons/iconh_red",
/* pickup */  "50 Health",
    50,
    IT_HEALTH,
    0,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED item_health_mega (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_health_mega",
    "sound/items/m_health.wav",
        { "models/powerups/health/mega_cross.md3", 
    "models/powerups/health/mega_sphere.md3", 
    0, 0 },
/* icon */    "icons/iconh_mega",
/* pickup */  "Mega Health",
    100,
    IT_HEALTH,
    0,
/* precache */ "",
/* sounds */ ""
  },


  //
  // WEAPONS 
  //

/*QUAKED weapon_gauntlet (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_gauntlet", 
    "sound/misc/w_pkup.wav",
        { "models/weapons2/gauntlet/gauntlet.md3",
    0, 0, 0},
/* icon */    "icons/iconw_gauntlet",
/* pickup */  "Gauntlet",
    0,
    IT_WEAPON,
    WP_GAUNTLET,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED weapon_shotgun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_shotgun", 
    "sound/misc/w_pkup.wav",
        { "models/weapons2/shotgun/shotgun.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_shotgun",
/* pickup */  "Shotgun",
    10,
    IT_WEAPON,
    WP_SHOTGUN,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED weapon_machinegun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_machinegun", 
    "sound/misc/w_pkup.wav",
        { "models/weapons2/machinegun/machinegun.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_machinegun",
/* pickup */  "Machinegun",
    40,
    IT_WEAPON,
    WP_MACHINEGUN,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED weapon_grenadelauncher (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_grenadelauncher",
    "sound/misc/w_pkup.wav",
        { "models/weapons2/grenadel/grenadel.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_grenade",
/* pickup */  "Grenade Launcher",
    10,
    IT_WEAPON,
    WP_GRENADE_LAUNCHER,
/* precache */ "",
/* sounds */ "sound/weapons/grenade/hgrenb1a.wav sound/weapons/grenade/hgrenb2a.wav"
  },

/*QUAKED weapon_rocketlauncher (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_rocketlauncher",
    "sound/misc/w_pkup.wav",
        { "models/weapons2/rocketl/rocketl.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_rocket",
/* pickup */  "Rocket Launcher",
    10,
    IT_WEAPON,
    WP_ROCKET_LAUNCHER,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED weapon_lightning (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_lightning", 
    "sound/misc/w_pkup.wav",
        { "models/weapons2/lightning/lightning.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_lightning",
/* pickup */  "Lightning Gun",
    100,
    IT_WEAPON,
    WP_LIGHTNING,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED weapon_railgun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_railgun", 
    "sound/misc/w_pkup.wav",
        { "models/weapons2/railgun/railgun.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_railgun",
/* pickup */  "Railgun",
    10,
    IT_WEAPON,
    WP_RAILGUN,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED weapon_plasmagun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_plasmagun", 
    "sound/misc/w_pkup.wav",
        { "models/weapons2/plasma/plasma.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_plasma",
/* pickup */  "Plasma Gun",
    50,
    IT_WEAPON,
    WP_PLASMAGUN,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED weapon_bfg (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_bfg",
    "sound/misc/w_pkup.wav",
        { "models/weapons2/bfg/bfg.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_bfg",
/* pickup */  "BFG10K",
    20,
    IT_WEAPON,
    WP_BFG,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED weapon_grapplinghook (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_grapplinghook",
    "sound/misc/w_pkup.wav",
        { "models/weapons2/grapple/grapple.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_grapple",
/* pickup */  "Grappling Hook",
    0,
    IT_WEAPON,
    WP_GRAPPLING_HOOK,
/* precache */ "",
/* sounds */ ""
  },

  //
  // AMMO ITEMS
  //

/*QUAKED ammo_shells (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "ammo_shells",
    "sound/misc/am_pkup.wav",
        { "models/powerups/ammo/shotgunam.md3", 
    0, 0, 0},
/* icon */    "icons/icona_shotgun",
/* pickup */  "Shells",
    10,
    IT_AMMO,
    WP_SHOTGUN,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED ammo_bullets (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "ammo_bullets",
    "sound/misc/am_pkup.wav",
        { "models/powerups/ammo/machinegunam.md3", 
    0, 0, 0},
/* icon */    "icons/icona_machinegun",
/* pickup */  "Bullets",
    50,
    IT_AMMO,
    WP_MACHINEGUN,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED ammo_grenades (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "ammo_grenades",
    "sound/misc/am_pkup.wav",
        { "models/powerups/ammo/grenadeam.md3", 
    0, 0, 0},
/* icon */    "icons/icona_grenade",
/* pickup */  "Grenades",
    5,
    IT_AMMO,
    WP_GRENADE_LAUNCHER,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED ammo_cells (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "ammo_cells",
    "sound/misc/am_pkup.wav",
        { "models/powerups/ammo/plasmaam.md3", 
    0, 0, 0},
/* icon */    "icons/icona_plasma",
/* pickup */  "Cells",
    30,
    IT_AMMO,
    WP_PLASMAGUN,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED ammo_lightning (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "ammo_lightning",
    "sound/misc/am_pkup.wav",
        { "models/powerups/ammo/lightningam.md3", 
    0, 0, 0},
/* icon */    "icons/icona_lightning",
/* pickup */  "Lightning",
    60,
    IT_AMMO,
    WP_LIGHTNING,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED ammo_rockets (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "ammo_rockets",
    "sound/misc/am_pkup.wav",
        { "models/powerups/ammo/rocketam.md3", 
    0, 0, 0},
/* icon */    "icons/icona_rocket",
/* pickup */  "Rockets",
    5,
    IT_AMMO,
    WP_ROCKET_LAUNCHER,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED ammo_slugs (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "ammo_slugs",
    "sound/misc/am_pkup.wav",
        { "models/powerups/ammo/railgunam.md3", 
    0, 0, 0},
/* icon */    "icons/icona_railgun",
/* pickup */  "Slugs",
    10,
    IT_AMMO,
    WP_RAILGUN,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED ammo_bfg (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "ammo_bfg",
    "sound/misc/am_pkup.wav",
        { "models/powerups/ammo/bfgam.md3", 
    0, 0, 0},
/* icon */    "icons/icona_bfg",
/* pickup */  "Bfg Ammo",
    15,
    IT_AMMO,
    WP_BFG,
/* precache */ "",
/* sounds */ ""
  },

  //
  // HOLDABLE ITEMS
  //
/*QUAKED holdable_teleporter (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "holdable_teleporter", 
    "sound/items/holdable.wav",
        { "models/powerups/holdable/teleporter.md3", 
    0, 0, 0},
/* icon */    "icons/teleporter",
/* pickup */  "Personal Teleporter",
    60,
    IT_HOLDABLE,
    HI_TELEPORTER,
/* precache */ "",
/* sounds */ ""
  },
/*QUAKED holdable_medkit (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "holdable_medkit", 
    "sound/items/holdable.wav",
        { 
    "models/powerups/holdable/medkit.md3", 
    "models/powerups/holdable/medkit_sphere.md3",
    0, 0},
/* icon */    "icons/medkit",
/* pickup */  "Medkit",
    60,
    IT_HOLDABLE,
    HI_MEDKIT,
/* precache */ "",
/* sounds */ "sound/items/use_medkit.wav"
  },

  //
  // POWERUP ITEMS
  //
/*QUAKED item_quad (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_quad", 
    "sound/items/quaddamage.wav",
        { "models/powerups/instant/quad.md3", 
        "models/powerups/instant/quad_ring.md3",
    0, 0 },
/* icon */    "icons/quad",
/* pickup */  "Quad Damage",
    30,
    IT_POWERUP,
    PW_QUAD,
/* precache */ "",
/* sounds */ "sound/items/damage2.wav sound/items/damage3.wav"
  },

/*QUAKED item_enviro (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_enviro",
    "sound/items/protect.wav",
        { "models/powerups/instant/enviro.md3", 
    "models/powerups/instant/enviro_ring.md3", 
    0, 0 },
/* icon */    "icons/envirosuit",
/* pickup */  "Battle Suit",
    30,
    IT_POWERUP,
    PW_BATTLESUIT,
/* precache */ "",
/* sounds */ "sound/items/airout.wav sound/items/protect3.wav"
  },

/*QUAKED item_haste (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_haste",
    "sound/items/haste.wav",
        { "models/powerups/instant/haste.md3", 
    "models/powerups/instant/haste_ring.md3", 
    0, 0 },
/* icon */    "icons/haste",
/* pickup */  "Speed",
    30,
    IT_POWERUP,
    PW_HASTE,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED item_invis (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_invis",
    "sound/items/invisibility.wav",
        { "models/powerups/instant/invis.md3", 
    "models/powerups/instant/invis_ring.md3", 
    0, 0 },
/* icon */    "icons/invis",
/* pickup */  "Invisibility",
    30,
    IT_POWERUP,
    PW_INVIS,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED item_regen (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_regen",
    "sound/items/regeneration.wav",
        { "models/powerups/instant/regen.md3", 
    "models/powerups/instant/regen_ring.md3", 
    0, 0 },
/* icon */    "icons/regen",
/* pickup */  "Regeneration",
    30,
    IT_POWERUP,
    PW_REGEN,
/* precache */ "",
/* sounds */ "sound/items/regen.wav"
  },

/*QUAKED item_flight (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "item_flight",
    "sound/items/flight.wav",
        { "models/powerups/instant/flight.md3", 
    "models/powerups/instant/flight_ring.md3", 
    0, 0 },
/* icon */    "icons/flight",
/* pickup */  "Flight",
    60,
    IT_POWERUP,
    PW_FLIGHT,
/* precache */ "",
/* sounds */ "sound/items/flight.wav"
  },

/*QUAKED team_CTF_redflag (1 0 0) (-16 -16 -16) (16 16 16)
Only in CTF games
*/
  {
    "team_CTF_redflag",
    NULL,
        { "models/flags/r_flag.md3",
    0, 0, 0 },
/* icon */    "icons/iconf_red1",
/* pickup */  "Red Flag",
    0,
    IT_TEAM,
    PW_REDFLAG,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED team_CTF_blueflag (0 0 1) (-16 -16 -16) (16 16 16)
Only in CTF games
*/
  {
    "team_CTF_blueflag",
    NULL,
        { "models/flags/b_flag.md3",
    0, 0, 0 },
/* icon */    "icons/iconf_blu1",
/* pickup */  "Blue Flag",
    0,
    IT_TEAM,
    PW_BLUEFLAG,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED holdable_kamikaze (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "holdable_kamikaze", 
    "sound/items/holdable.wav",
        { "models/powerups/kamikazi.md3", 
    0, 0, 0},
/* icon */    "icons/kamikaze",
/* pickup */  "Kamikaze",
    60,
    IT_HOLDABLE,
    HI_KAMIKAZE,
/* precache */ "",
/* sounds */ "sound/items/kamikazerespawn.wav"
  },

/*QUAKED holdable_portal (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "holdable_portal", 
    "sound/items/holdable.wav",
        { "models/powerups/holdable/porter.md3",
    0, 0, 0},
/* icon */    "icons/portal",
/* pickup */  "Portal",
    60,
    IT_HOLDABLE,
    HI_PORTAL,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED holdable_invulnerability (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "holdable_invulnerability", 
    "sound/items/holdable.wav",
        { "models/powerups/holdable/invulnerability.md3", 
    0, 0, 0},
/* icon */    "icons/invulnerability",
/* pickup */  "Invulnerability",
    60,
    IT_HOLDABLE,
    HI_INVULNERABILITY,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED ammo_nails (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "ammo_nails",
    "sound/misc/am_pkup.wav",
        { "models/powerups/ammo/nailgunam.md3", 
    0, 0, 0},
/* icon */    "icons/icona_nailgun",
/* pickup */  "Nails",
    20,
    IT_AMMO,
    WP_NAILGUN,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED ammo_mines (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "ammo_mines",
    "sound/misc/am_pkup.wav",
        { "models/powerups/ammo/proxmineam.md3", 
    0, 0, 0},
/* icon */    "icons/icona_proxlauncher",
/* pickup */  "Proximity Mines",
    10,
    IT_AMMO,
    WP_PROX_LAUNCHER,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED ammo_belt (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "ammo_belt",
    "sound/misc/am_pkup.wav",
        { "models/powerups/ammo/chaingunam.md3", 
    0, 0, 0},
/* icon */    "icons/icona_chaingun",
/* pickup */  "Chaingun Belt",
    100,
    IT_AMMO,
    WP_CHAINGUN,
/* precache */ "",
/* sounds */ ""
  },

  //
  // PERSISTANT POWERUP ITEMS
  //
/*QUAKED item_scout (.3 .3 1) (-16 -16 -16) (16 16 16) suspended redTeam blueTeam
*/
  {
    "item_scout",
    "sound/items/scout.wav",
        { "models/powerups/scout.md3", 
    0, 0, 0 },
/* icon */    "icons/scout",
/* pickup */  "Scout",
    30,
    IT_PERSISTANT_POWERUP,
    PW_SCOUT,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED item_guard (.3 .3 1) (-16 -16 -16) (16 16 16) suspended redTeam blueTeam
*/
  {
    "item_guard",
    "sound/items/guard.wav",
        { "models/powerups/guard.md3", 
    0, 0, 0 },
/* icon */    "icons/guard",
/* pickup */  "Guard",
    30,
    IT_PERSISTANT_POWERUP,
    PW_GUARD,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED item_doubler (.3 .3 1) (-16 -16 -16) (16 16 16) suspended redTeam blueTeam
*/
  {
    "item_doubler",
    "sound/items/doubler.wav",
        { "models/powerups/doubler.md3", 
    0, 0, 0 },
/* icon */    "icons/doubler",
/* pickup */  "Doubler",
    30,
    IT_PERSISTANT_POWERUP,
    PW_DOUBLER,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED item_doubler (.3 .3 1) (-16 -16 -16) (16 16 16) suspended redTeam blueTeam
*/
  {
    "item_ammoregen",
    "sound/items/ammoregen.wav",
        { "models/powerups/ammo.md3",
    0, 0, 0 },
/* icon */    "icons/ammo_regen",
/* pickup */  "Ammo Regen",
    30,
    IT_PERSISTANT_POWERUP,
    PW_AMMOREGEN,
/* precache */ "",
/* sounds */ ""
  },

  /*QUAKED team_CTF_neutralflag (0 0 1) (-16 -16 -16) (16 16 16)
Only in One Flag CTF games
*/
  {
    "team_CTF_neutralflag",
    NULL,
        { "models/flags/n_flag.md3",
    0, 0, 0 },
/* icon */    "icons/iconf_neutral1",
/* pickup */  "Neutral Flag",
    0,
    IT_TEAM,
    PW_NEUTRALFLAG,
/* precache */ "",
/* sounds */ ""
  },

  {
    "item_redcube",
    "sound/misc/am_pkup.wav",
        { "models/powerups/orb/r_orb.md3",
    0, 0, 0 },
/* icon */    "icons/iconh_rorb",
/* pickup */  "Red Cube",
    0,
    IT_TEAM,
    0,
/* precache */ "",
/* sounds */ ""
  },

  {
    "item_bluecube",
    "sound/misc/am_pkup.wav",
        { "models/powerups/orb/b_orb.md3",
    0, 0, 0 },
/* icon */    "icons/iconh_borb",
/* pickup */  "Blue Cube",
    0,
    IT_TEAM,
    0,
/* precache */ "",
/* sounds */ ""
  },
/*QUAKED weapon_nailgun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_nailgun", 
    "sound/misc/w_pkup.wav",
        { "models/weapons/nailgun/nailgun.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_nailgun",
/* pickup */  "Nailgun",
    10,
    IT_WEAPON,
    WP_NAILGUN,
/* precache */ "",
/* sounds */ ""
  },

/*QUAKED weapon_prox_launcher (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_prox_launcher", 
    "sound/misc/w_pkup.wav",
        { "models/weapons/proxmine/proxmine.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_proxlauncher",
/* pickup */  "Prox Launcher",
    5,
    IT_WEAPON,
    WP_PROX_LAUNCHER,
/* precache */ "",
/* sounds */ "sound/weapons/proxmine/wstbtick.wav "
      "sound/weapons/proxmine/wstbactv.wav "
      "sound/weapons/proxmine/wstbimpl.wav "
      "sound/weapons/proxmine/wstbimpm.wav "
      "sound/weapons/proxmine/wstbimpd.wav "
      "sound/weapons/proxmine/wstbactv.wav"
  },

/*QUAKED weapon_chaingun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
  {
    "weapon_chaingun", 
    "sound/misc/w_pkup.wav",
        { "models/weapons/vulcan/vulcan.md3", 
    0, 0, 0},
/* icon */    "icons/iconw_chaingun",
/* pickup */  "Chaingun",
    80,
    IT_WEAPON,
    WP_CHAINGUN,
/* precache */ "",
/* sounds */ "sound/weapons/vulcan/wvulwind.wav"
  },

  // end of list marker
  {NULL}
};

EntityManager::EntityManager(void)
{
  entities = NULL;
  num_entities = 0;
}

EntityManager::~EntityManager(void)
{
  if (parse_entities) cake_free(parse_entities);
  parse_entities = NULL;

  if (epairs)
  {
    for (int i = 0; i < num_epairs; ++i)
    {
      cake_free(epairs[i].key);
      cake_free(epairs[i].val);
    }
    cake_free(epairs); epairs = NULL;
  }

  if (entities) delete [] entities;
  entities = NULL;
  num_entities = 0;
}

int EntityManager::Load(char *ent, int len)
{
  int i, newlines;

  /* Count entities and pairs */
  newlines = num_entities = 0;
  for (i = 0; i < len; ++i)
  {
    if (ent[i] == '{') ++num_entities;
    if (ent[i] == '\n') ++newlines;
  }
  num_epairs = newlines - (2 * num_entities);

  // Alloc structures
  epairs = (epair_t*) cake_malloc(num_epairs*sizeof(epair_t), "EntityManager::Init.epairs");
  if (!epairs)
  {
    gConsole->Insertln("^1ERROR: Cannot allocate required memory for epairs.");
    return 0;   // FIXME: generate an exception
  }

  for (i = 0; i < num_epairs; ++i)
  {
    epairs[i].key = (char*) cake_malloc(MAX_ENTITY_KEYLENGTH*sizeof(char), "EntityManager::Init.epairs[i].key");
    if (!epairs[i].key)
    {
      gConsole->Insertln("^1ERROR: Cannot allocate required memory for epair keys.");
      return 0;   // FIXME: generate an exception
    }
    memset(epairs[i].key, '\0', MAX_ENTITY_KEYLENGTH);

    epairs[i].val = (char*) cake_malloc(MAX_ENTITY_VALLENGTH*sizeof(char), "EntityManager::Init.epairs[i].val");
    if (!epairs[i].val)
    {
      gConsole->Insertln("^1ERROR: Cannot allocate required memory for epair vals.");
      return 0;   // FIXME: generate an exception
    }
    memset(epairs[i].val, '\0', MAX_ENTITY_VALLENGTH);
  }

  parse_entities = (parse_entity_t*) cake_malloc(num_entities*sizeof(parse_entity_t), "EntityManager::Init.parse_entities");
  if (!parse_entities)
  {
    gConsole->Insertln("^1ERROR: Cannot allocate required memory for parsed entities.");
    return 0;   // FIXME: generate an exception
  }

  Parser::StartParseString(ent);

  int currentity = 0;
  int pair = 0;

  Parser::GetToken(true);
  while (!strcmp(Parser::token, "{"))
  {
    parse_entities[currentity].firstpair = pair;
    parse_entities[currentity].numpairs = 0;

    Parser::GetToken(true);
    while (strcmp(Parser::token, "}"))
    {
      if (strlen(Parser::token) < MAX_ENTITY_KEYLENGTH)
        strncpy(epairs[pair].key, Parser::token, MAX_ENTITY_KEYLENGTH);
      else
        gConsole->Insertln("^5WARNING: Key is not wide enough (line %d)", Parser::scriptline);    

      Parser::GetToken(true);
      if (strlen(Parser::token) < MAX_ENTITY_VALLENGTH)
        strncpy(epairs[pair].val, Parser::token, MAX_ENTITY_VALLENGTH);
      else
        gConsole->Insertln("^5WARNING: Val is not wide enough (line %d)", Parser::scriptline);

      if (++pair > num_epairs)
      {
        gConsole->Insertln("^1ERROR: Too many pairs!");
        return 0;   // FIXME: generate an exception
      }
      ++parse_entities[currentity].numpairs;

      Parser::GetToken(true);
    }

    if (++currentity > num_entities)
    {
      gConsole->Insertln("^1ERROR: Too many entities !");
      return 0;   // FIXME: generate an exception
    }
    
    Parser::GetToken(true);
  }

  Parser::StopParseFile();

  if (pair != num_epairs) gConsole->Insertln("^5WARNING: pairs table was created with a wrong size");
  if (currentity != num_entities) gConsole->Insertln("^5WARNING: entities table was created with a wrong size");

  return 1;
}

void EntityManager::Init(void)
{
  for (int i = 0; i < num_entities; ++i)
  {
    entities[i].LoadSound();
  }
}

const char* EntityManager::Entity_value(int entity, const char *key)
{
    epair_t *pair = &epairs[parse_entities[entity].firstpair];
    for (int i = 0; i < parse_entities[entity].numpairs; ++i, ++pair)
    if (!stricmp(key, pair->key)) return pair->val;
    return "";
}

char* EntityManager::Entity_value_ex(int entity, const char *key)
{
  epair_t *pair = &epairs[parse_entities[entity].firstpair];
  for (int i = 0; i < parse_entities[entity].numpairs; ++i, ++pair)
  {
    if (!stricmp(key, pair->key))
    {
      int l = (int) strlen(pair->val)+1;
      char *dest = (char*) cake_malloc(l*sizeof(char), "EntityManager::Entity_value.dest");
      if (!dest) ThrowException(ALLOCATION_ERROR, "EntityManager::Entity_value.dest");
      memset(dest, '\0', l);
      strcpy(dest, pair->val);
      return dest;
    }
  }

  return NULL;
}

int EntityManager::Find_entity(ENTTYPE type, int number)
{
  int a = 0;

  for (int i = 0; i < num_entities; ++i)
  {
    if (entities[i].enttype == type)
    {
      if (a++ < number) continue;
      return i;
    }
  }

  return -1;  // not found
}

float EntityManager::Entity_float(int entity, const char *key)
{
  const char *tmp = Entity_value(entity, key);
  if (strlen(tmp)) return (float) atof(tmp);
  return 0;
}

int EntityManager::Entity_int(int entity, const char *key)
{
  const char *tmp = Entity_value(entity, key);
  if (strlen(tmp)) return atoi(tmp);
  return 0;
}

void EntityManager::Entity_vec3(int entity, const char *key, vec3_t vec)
{
    const char *val = Entity_value(entity, key);
    if (strlen(val)) sscanf(val, "%f %f %f", &vec[0], &vec[1], &vec[2]);
  else vec[0] = vec[1] = vec[2] = 0;
}

void EntityManager::Report(void)
{
  epair_t *pair;
  gLogFile->Insert("\tNumber of entities : %d\n", num_entities);
  gLogFile->OpenFile();
  for (int i = 0; i < num_entities; ++i)
  {
    gLogFile->Insert("\t{\n");

    pair = &epairs[parse_entities[i].firstpair];

    for (int j = 0; j < parse_entities[i].numpairs; ++j, ++pair)
    {
      gLogFile->Insert("\t\t\"%s\" \"%s\"\n", pair->key, pair->val);
    }
    gLogFile->Insert("\t}\n");
  }
  gLogFile->CloseFile();
}

void EntityManager::UpdateEntities(Client *client)
{
  for (int i = 0; i < num_entities; ++i)
    entities[i].UpdateEntity(client);
}

// ------- ENTITY PROCESSING --------------------------------------------------

int EntityManager::ProcessEntities(Q3BSP *pBSP)
{
  if (!pBSP) return 0;

  int i, j;

  // Alloc memory
  entities = new Entity[num_entities];
  if (!entities)
  {
    gConsole->Insertln("^1ERROR: Cannot allocate required memory for entities.");
    return 0;   // FIXME: generate an exception
  }

  // Parse all entities
  for (i = 0; i < num_entities; ++i)
  {
    entities[i].pBSP = pBSP;

    // Process entity
    // Default content
    entities[i].spawnflags = 0;
    entities[i].spawnflags = Entity_int(i, "spawnflags");
    Entity_vec3(i, "origin", entities[i].origin);
    entities[i].angle = Entity_float(i, "angle");
    entities[i].my_target = Entity_value_ex(i, "targetname");
    entities[i].dst_target = Entity_value_ex(i, "target");
    entities[i].wait = Entity_float(i, "wait");

    // Particular content
    const char *classname_val = Entity_value(i, "classname"); 
         if (!stricmp(classname_val, "worldspawn")) ProcessWorldspawn(i);
    else if (!stricmp(classname_val, "info_player_start") ||
         !stricmp(classname_val, "info_player_deathmatch")) ProcessInfoPlayerDeathmatch(i);

    else if (!stricmp(classname_val, "target_speaker")) ProcessTargetSpeaker(i);
    else if (!stricmp(classname_val, "target_location")) ProcessTargetLocation(i);
    else if (!stricmp(classname_val, "target_print")) ProcessTargetPrint(i);
    else if (!stricmp(classname_val, "target_teleporter")) ProcessTargetTeleporter(i);

    else if (!stricmp(classname_val, "func_timer")) ProcessFuncTimer(i);
    else if (!stricmp(classname_val, "func_bobbing")) ProcessFuncBobbing(i);
    else if (!stricmp(classname_val, "func_pendulum")) ProcessFuncPendulum(i);
    else if (!stricmp(classname_val, "func_rotating")) ProcessFuncRotating(i);
    else if (!stricmp(classname_val, "func_train")) ProcessFuncTrain(i);
    else if (!stricmp(classname_val, "path_corner")) ProcessPathCorner(i);
    else if (!stricmp(classname_val, "func_door")) ProcessFuncDoor(i);

    else if (!stricmp(classname_val, "trigger_push")) ProcessTriggerPush(i);
    else if (!stricmp(classname_val, "trigger_teleport")) ProcessTriggerTeleport(i);
    else if (!stricmp(classname_val, "trigger_multiple")) ProcessTriggerMultiple(i);

    else if (!stricmp(classname_val, "misc_teleporter_dest")) ProcessMiscTeleporterDest(i);
    else ProcessItem(i);
  }

  // find the target for each entity
  for (i = 0; i < num_entities; ++i)
  {
    if (entities[i].dst_target)
    {
      for (j = 0; j < num_entities; ++j)
      {
        if (!entities[j].my_target) continue;
        if (!stricmp(entities[j].my_target, entities[i].dst_target))
        {
          entities[i].AddTarget(&entities[j]);
        }
      }
    }
  }

  return 1;
}

void EntityManager::ProcessWorldspawn(int i)
{
  entities[i].enttype = ET_WORLDSPAWN;
  entities[i].message = Entity_value_ex(i, "message");
  entities[i].ambient = Entity_int(i, "ambient");
  Entity_vec3(i, "_color", entities[i]._color);

  entities[i].noise = Entity_value_ex(i, "music");

  if (entities[i].pBSP && entities[i].message)
    entities[i].pBSP->SetWorldName(entities[i].message);
}

void EntityManager::ProcessItem(int i)
{
  int c = 1;
  while (bg_itemlist[c].classname)
  {
    if (!stricmp(bg_itemlist[c].classname, Entity_value(i, "classname"))) break;
    ++c;
  }

  if (!bg_itemlist[c].classname) return;
  
  entities[i].enttype = ET_ITEM;
  entities[i].itemtype = bg_itemlist[c].giType;
  
  entities[i].angle = (float)rand()/(float)RAND_MAX;  // start rotation with random angle

  // Load the item model
  MD3Loader loader(entities[i].pBSP);

  for (int m = 0; bg_itemlist[c].world_model[m]; ++m)
  {
    Entity *ent = loader.LoadMD3(bg_itemlist[c].world_model[m]);  // should support multiple models
    if (ent) entities[i].AddChild(ent);
    else gConsole->Insertln("^5WARNING: Couldn't load model \"%s\"", bg_itemlist[c].world_model[0]);
  }
}

void EntityManager::ProcessInfoPlayerDeathmatch(int i)
{
  entities[i].enttype = ET_INFO_PLAYER_DEATHMATCH;
}

void EntityManager::ProcessTargetSpeaker(int i)
{
  entities[i].enttype = ET_TARGET_SPEAKER;
  entities[i].noise = Entity_value_ex(i, "noise");
}

void EntityManager::ProcessTargetPrint(int i)
{
  entities[i].enttype = ET_TARGET_PRINT;
  entities[i].message = Entity_value_ex(i, "message");
}

void EntityManager::ProcessTargetLocation(int i)
{
  entities[i].enttype = ET_TARGET_LOCATION;
  entities[i].message = Entity_value_ex(i, "message");
}

void EntityManager::ProcessTargetTeleporter(int i)
{
  entities[i].enttype = ET_TARGET_TELEPORTER;
}

void EntityManager::ProcessTriggerPush(int i)
{
  entities[i].enttype = ET_TRIGGER_PUSH;
  entities[i].model = atoi(Entity_value(i, "model")+1);
}

void EntityManager::ProcessTriggerTeleport(int i)
{
  entities[i].enttype = ET_TRIGGER_TELEPORT;
  entities[i].model = atoi(Entity_value(i, "model")+1);
}

void EntityManager::ProcessTriggerMultiple(int i)
{
  entities[i].enttype = ET_TRIGGER_MULTIPLE;
  entities[i].random = Entity_float(i, "random");
  entities[i].model = atoi(Entity_value(i, "model")+1);

  // default values
  if (!entities[i].wait) entities[i].wait = 0.2f;
}

void EntityManager::ProcessMiscTeleporterDest(int i)
{
  entities[i].enttype = ET_MISC_TELEPORTER_DEST;
  entities[i].angle = (float) atof(Entity_value(i, "angle"));
}

void EntityManager::ProcessFuncTimer(int i)
{
  entities[i].enttype = ET_FUNC_TIMER;
  entities[i].random = Entity_float(i, "random");

  // default values
  if (!entities[i].random) entities[i].random = 1.f;
  if (!entities[i].wait) entities[i].wait = 1.f;
}

void EntityManager::ProcessFuncBobbing(int i)
{
  entities[i].enttype = ET_FUNC_BOBBING;
  entities[i].model = atoi(Entity_value(i, "model")+1);
  entities[i].height = Entity_float(i, "height");
  entities[i].phase = Entity_float(i, "phase");
  entities[i].speed = Entity_float(i, "speed");
  MD3Loader loader(entities[i].pBSP);
  Entity *ent = loader.LoadMD3(Entity_value_ex(i, "model2")); // should support multiple models
  entities[i].AddChild(ent);

  // default values
  if (!entities[i].speed) entities[i].speed = 4;
  if (!entities[i].phase) entities[i].phase = 0;
  if (!entities[i].height) entities[i].height = 32;
}

void EntityManager::ProcessFuncPendulum(int i)
{
  entities[i].enttype = ET_FUNC_PENDULUM;
  entities[i].model = atoi(Entity_value(i, "model")+1);
  entities[i].phase = Entity_float(i, "phase");
  entities[i].speed = Entity_float(i, "speed");
  entities[i].dmg = Entity_int(i, "dmg");

  // default values
  if (!entities[i].speed) entities[i].speed = 30;
}

void EntityManager::ProcessFuncRotating(int i)
{
  entities[i].enttype = ET_FUNC_ROTATING;
  entities[i].model = atoi(Entity_value(i, "model")+1);
  entities[i].speed = Entity_float(i, "speed");
  MD3Loader loader(entities[i].pBSP);
  Entity *ent = loader.LoadMD3(Entity_value_ex(i, "model2")); // should support multiple models
  entities[i].AddChild(ent);

  // default values
  if (!entities[i].speed) entities[i].speed = 100;
}

void EntityManager::ProcessFuncTrain(int i)
{
  entities[i].enttype = ET_FUNC_TRAIN;
  entities[i].model = atoi(Entity_value(i, "model")+1);
  entities[i].speed = Entity_float(i, "speed");

  // default values
  if (!entities[i].speed) entities[i].speed = 100;
  entities[i].angle = entities[i].speed;
  entities[i].wait = 0;
}

void EntityManager::ProcessPathCorner(int i)
{
  entities[i].enttype = ET_PATH_CORNER;
  entities[i].speed = Entity_float(i, "speed");
}

/*QUAKED func_door (0 .5 .8) ? START_OPEN - CRUSHER
Normal sliding door entity. By default, the door will activate when player walks close to it or when
damage is inflicted to it.
-------- KEYS --------
speed : determines how fast the door moves (default 100).
lip : lip remaining at end of move (default 8)
health : if set to a non-zero value, the door must be damaged by "health" amount of points
         to activate (default 0).
dmg : damage to inflict on player when he blocks operation of door (default 4). Door will
      reverse direction when blocked unless CRUSHER spawnflag is set.
team: assign the same team name to multiple doors that should operate together (see Notes).
light : constantLight radius of .md3 model included with entity. Has no effect on the
        entity's brushes (default 0).
color : constantLight color of .md3 model included with entity. Has no effect on the
        entity's brushes (default 1 1 1).
model2 : path/name of model to include (eg: models/mapobjects/pipe/pipe02.md3).
notfree : when set to 1, entity will not spawn in "Free for all" and "Tournament" modes.
notteam : when set to 1, entity will not spawn in "Teamplay" and "CTF" modes.
notsingle : when set to 1, entity will not spawn in Single Player mode (bot play mode).
-------- SPAWNFLAGS --------
START_OPEN : the door will spawn in the open state and operate in reverse.
CRUSHER : door will not reverse direction when blocked and will keep damaging player
^         until he dies or gets out of the way.
-------- NOTES --------
Unlike in Quake 2, doors that touch are NOT automatically teamed. If you want doors to
operate together, you have to team them manually by assigning the same team name to all
of them. Setting the origin key is simply an alternate method to using an origin brush.
When using the model2 key, the origin point of the model will correspond to the origin
point defined by either the origin brush or the origin coordinate value.*/
void EntityManager::ProcessFuncDoor(int i)
{
  entities[i].enttype = ET_FUNC_DOOR;

  entities[i].speed = Entity_float(i, "speed");
  entities[i].lip = Entity_int(i, "lip");
  entities[i].health = Entity_int(i, "health");
  entities[i].dmg = Entity_int(i, "dmg");
  entities[i].team = Entity_value_ex(i, "team");
  entities[i].light = Entity_float(i, "light");
  Entity_vec3(i, "color", entities[i]._color);

  MD3Loader loader(entities[i].pBSP);
  Entity *ent = loader.LoadMD3(Entity_value_ex(i, "model2")); // should support multiple models
  entities[i].AddChild(ent);

  if (!entities[i].speed) entities[i].speed = 400;
  if (!entities[i].accel) entities[i].accel = entities[i].speed;
  if (!entities[i].decel) entities[i].decel = entities[i].speed;
  if (!entities[i].wait) entities[i].wait = 2;
  if (!entities[i].lip) entities[i].lip = 8;
  if (!entities[i].dmg) entities[i].dmg = 2;
}

//-----------------------------------------------------------------------------

Entity::Entity(void)
{
  // Initialize entity
  pBSP = NULL;

  enttype = ET_UNKNOWN;
  itemtype = IT_UNKNOWN;

  parent = NULL;
  children = NULL;
  numchildren = 0;

  targets = NULL;
  numtargets = 0;

  models = NULL;
  model = -1;
  num_lods = 0;

  noise = NULL;
  noise_index = -1;

  message = NULL;
  dst_target = NULL;
  my_target = NULL;

  team = NULL;

  VectorClear(origin);
  VectorClear(rotation);
  VectorClear(translation);
  VectorSet(scaling, 1.f, 1.f, 1.f);
  VectorClear(_color);
  light = 0.f;

  nexttime = 0;
  curr_client = NULL;

  spawnflags = 0;
  ambient = 0;
  height = phase = speed = accel = decel = angle = wait = random = 0.f;
  health = dmg = 0;
  lip = 0;

  transformation[0] = transformation[5] = transformation[10] = transformation[15] = 1.f;
  transformation[1] = transformation[2] = 
  transformation[3] = transformation[4] = 
  transformation[6] = transformation[7] = 
  transformation[8] = transformation[9] = 
  transformation[11] = transformation[12] = 
  transformation[13] = transformation[14] = 0.f;

  bbox[0] = bbox[1] = bbox[2] = bbox[3] = bbox[4] = bbox[5] = 0.f;
}

Entity::~Entity(void)
{
  if (noise_index >= 0) freeSound(noise_index);
  pBSP = NULL;
  children = NULL;
  models = NULL;
  targets = NULL;
}

void Entity::LoadSound(void)
{
  // Load the sounds
  switch (enttype)
  {
    case ET_WORLDSPAWN:
      // nothing to do
      setBGMusic(noise, true);
      break;
    case ET_TARGET_SPEAKER:
      {
        if (!noise) return;
        bool loop = false, global = false;

        if (spawnflags & 1 || spawnflags & 2) loop = true;
        //  if (spawnflags & 3) global = true;

        noise_index = loadSound(noise, loop, (loop?true:false), true/*, global?NULL:origin*/);

        if (spawnflags & 1) toggleSoundPause(noise_index);
      }
      break;
    case ET_TARGET_TELEPORTER:
    case ET_TRIGGER_TELEPORT:
      noise_index = loadSound("sound/world/telein.wav", false, false);
      break;
    case ET_TRIGGER_PUSH:
      noise_index = loadSound("sound/world/jumppad.wav", false, false);
      break;
    default:
      break;
  }
}
void Entity::AddChild(Entity *child_ent)
{
  if (!child_ent) return;

  if (numchildren)
    children = (Entity**) cake_realloc(children, ++numchildren*sizeof(Entity*), "Entity::AddChild.children");
  else
    children = (Entity**) cake_malloc(++numchildren*sizeof(Entity*), "Entity::AddChild.children");

  if (!children) ThrowException(ALLOCATION_ERROR, "Entity::AddChild.children");
  if (child_ent->parent)
  {
    child_ent->parent->RemoveChild(child_ent);
  }

  children[numchildren-1] = child_ent;
  child_ent->parent = this;
}

void Entity::RemoveChild(Entity *child_ent)
{
  for (int i = 0; i < numchildren; ++i)
  {
    if (children[numchildren-1] == child_ent)
    {
      if (i < numchildren-1)
        children[i] = children[numchildren-1];

      children[--numchildren] = NULL;
      break;
    }
  }
  child_ent->parent = NULL;
}

void Entity::SetModel(cmodel_t *model, int lod)
{
  if (!model || lod < 0 || lod > num_lods) return;  // should warn ?

  if (lod == num_lods)
  {
    if (num_lods)
      models = (cmodel_t**) cake_realloc(models, (++num_lods)*sizeof(cmodel_t*), "Entity::setModel.models");
    else
      models = (cmodel_t**) cake_malloc((++num_lods)*sizeof(cmodel_t*), "Entity::setModel.models");

    if (!models)
    {
      gConsole->Insertln("^1ERROR: Allocation failed for model pointer.");
      return; // FIXME: generate an exception
    }
  }
  model->entity = this;
  models[lod] = model;
}

cmodel_t* Entity::GetModel(int lod)
{
  if (lod >= 0 && lod < num_lods) return models[lod];
  else return NULL;
}

void Entity::AddTarget(Entity *target_ent)
{
  if (!target_ent) return;

  if (numtargets)
    targets = (Entity**) cake_realloc(targets, ++numtargets*sizeof(Entity*), "Entity::addChild.targets");
  else
    targets = (Entity**) cake_malloc(++numtargets*sizeof(Entity*), "Entity::addChild.targets");

  targets[numtargets-1] = target_ent;
}

void Entity::RemoveTarget(Entity *target_ent)
{
  for (int i = 0; i < numtargets; ++i)
  {
    if (targets[numtargets-1] == target_ent)
    {
      if (i < numtargets-1)
        targets[i] = targets[numtargets-1];

      targets[--numtargets] = NULL;
      break;
    }
  }
}

void Entity::UpdateTransformMatrix(void)
{
  int i;

  // This computes the transformation matrix for entity
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
    if (parent) glLoadMatrixf(parent->transformation);
    else glLoadIdentity();
    glTranslatef(translation[0], translation[1], translation[2]);
    glRotatef(rotation[0], 1, 0, 0);
    glRotatef(rotation[1], 0, 1, 0);
    glRotatef(rotation[2], 0, 0, 1);
    glScalef(scaling[0], scaling[1], scaling[2]);
    glGetFloatv(GL_MODELVIEW_MATRIX, transformation);
  glPopMatrix();
  gRender->CheckGLError(10);

  if (models)
  {
    #if 0
      MultVect4x4(transformation,  models[0]->bbox,     bbox);
      MultVect4x4(transformation, &models[0]->bbox[3], &bbox[3]);
    #else
      vec3_t t, v[8] =
      {
        { models[0]->bbox[0], models[0]->bbox[1], models[0]->bbox[2] },
        { models[0]->bbox[3], models[0]->bbox[1], models[0]->bbox[2] },
        { models[0]->bbox[3], models[0]->bbox[1], models[0]->bbox[5] },
        { models[0]->bbox[0], models[0]->bbox[1], models[0]->bbox[5] },
        { models[0]->bbox[0], models[0]->bbox[4], models[0]->bbox[2] },
        { models[0]->bbox[3], models[0]->bbox[4], models[0]->bbox[2] },
        { models[0]->bbox[3], models[0]->bbox[4], models[0]->bbox[5] },
        { models[0]->bbox[0], models[0]->bbox[4], models[0]->bbox[5] }
      };

      t[0] = transformation[0]*v[0][0] + transformation[4]*v[0][1] + transformation[8]*v[0][2]  + transformation[12];
      t[1] = transformation[1]*v[0][0] + transformation[5]*v[0][1] + transformation[9]*v[0][2]  + transformation[13];
      t[2] = transformation[2]*v[0][0] + transformation[6]*v[0][1] + transformation[10]*v[0][2] + transformation[14];

      bbox[0] = bbox[3] = t[0];
      bbox[1] = bbox[4] = t[1];
      bbox[2] = bbox[5] = t[2];

      for (i = 1; i < 8; ++i)
      {
        t[0] = transformation[0]*v[i][0] + transformation[4]*v[i][1] + transformation[8]*v[i][2]  + transformation[12];
        t[1] = transformation[1]*v[i][0] + transformation[5]*v[i][1] + transformation[9]*v[i][2]  + transformation[13];
        t[2] = transformation[2]*v[i][0] + transformation[6]*v[i][1] + transformation[10]*v[i][2] + transformation[14];

        bbox[0] = min(bbox[0], t[0]);
        bbox[1] = min(bbox[1], t[1]);
        bbox[2] = min(bbox[2], t[2]);
        bbox[3] = max(bbox[3], t[0]);
        bbox[4] = max(bbox[4], t[1]);
        bbox[5] = max(bbox[5], t[2]);
      }
    #endif
  }

  for (i = 0; i < numchildren; ++i) children[i]->UpdateTransformMatrix();
}

void Entity::UpdateEntity(Client *client)
{
  curr_client = client;

  switch (enttype)
  {
    case ET_FUNC_PENDULUM:
      UpdateFuncPendulum();
      break;
    case ET_FUNC_BOBBING:
      UpdateFuncBobbing();
      break;
    case ET_FUNC_ROTATING:
      UpdateFuncRotating();
      break;
    case ET_FUNC_TRAIN:
      UpdateFuncTrain();
      break;
    case ET_FUNC_TIMER:
      UpdateFuncTimer();
      break;
    case ET_FUNC_DOOR:
      UpdateFuncDoor();
      break;
    case ET_TRIGGER_PUSH:
      UpdateTriggerPush();
      break;
    case ET_TRIGGER_TELEPORT:
      UpdateTriggerTeleport();
      break;
    case ET_TRIGGER_MULTIPLE:
      UpdateTriggerMultiple();
      break;
    case ET_ITEM:
      UpdateItem();
      break;
    default:
      break;
  }
}

void Entity::UpdateItem(void)
{
  #define DEFBOBHEIGHT    7.f
  #define DEFBOBSPEED     1.f
  #define DEFROT        180.f
  #define DEFSCALE      1.f
  #define DEFVERTICALOFFSET 4.f
  #define DEFWEAPONSCALE    1.5f

  float alpha;
  if (itemtype == IT_HEALTH) alpha = (float)Timer::fTime+angle;
  else alpha = ((float)Timer::fTime+angle)/2.f; // weapons and normal items rotate slower

  alpha = (alpha-floorf(alpha))*M_TWO_PI;
  alpha = RAD2DEG(alpha);

  // the first model of multipart-items should always rotate
  if (children)
  {
    if (children[0])
    {
      VectorSet(children[0]->rotation, 0, 0, alpha);
    }

    // rotate outer rings
    if (itemtype == IT_POWERUP && numchildren > 1)
    {
      alpha = (float)Timer::fTime+angle;
      alpha = (alpha-floorf(alpha))*M_TWO_PI;
      alpha = RAD2DEG(alpha);
      if (children[1])
      {
        VectorSet(children[1]->rotation, 0, 0, -alpha);
      }
    }
  }

  float t = ((float)Timer::fTime+angle)/DEFBOBSPEED;
  t = (t-floorf(t))*M_TWO_PI;

  VectorCopy(origin, translation);
  if (itemtype == IT_WEAPON)
  {
    VectorSet(scaling, DEFWEAPONSCALE, DEFWEAPONSCALE, DEFWEAPONSCALE);
    translation[2] += DEFVERTICALOFFSET;
  }
  translation[2] += DEFVERTICALOFFSET*(1+FastCos(t));

  UpdateTransformMatrix();

  // Enable or disable model depending on its distance to camera
  if (numchildren)
  {
    vec3_t vdist;
    VectorSub(&bbox[3], &bbox[0], vdist);
    VectorMA(vdist, 2, origin, vdist);
    VectorScale(vdist, 0.5f, vdist);
    VectorSub(gRender->current_camera->pos, vdist, vdist);
    float fdist = DotProduct(vdist, vdist);

    int lod = (int) fabsf(fdist/40000.f);
    int child_lod;

    for (int c = 0; c < numchildren; ++c)
    {
      if (lod > children[c]->num_lods-1) child_lod = children[c]->num_lods-1;
      else child_lod = lod;

      for (int i = 0; i < children[c]->num_lods; ++i)
      {
        if (i == child_lod) children[c]->models[i]->render = true;
        else children[c]->models[i]->render = false;
      }
    }
  }

  pBSP->LinkEntity(this);

  #undef DEFBOBHEIGHT
  #undef DEFBOBSPEED
  #undef DEFROT
  #undef DEFSCALE
  #undef DEFVERTICALOFFSET
  #undef DEFWEAPONSCALE
}

void Entity::UpdateFuncBobbing(void)
{
  float t = (float) Timer::fTime;
  if (!speed) speed = 4;
  t = t/speed+phase;
  t -= floorf(t);
  t = sinf(M_TWO_PI*t)*height;
  switch (spawnflags)
  {
    case 1:
      VectorSet(translation, -t, 0, 0);
      break;
    case 2:
      VectorSet(translation, 0, t, 0);
      break;
    default:
      VectorSet(translation, 0, 0, t);
      break;
  }
  VectorAdd(translation, origin, translation);

  UpdateTransformMatrix();

  pBSP->LinkEntity(this);
}

void Entity::UpdateFuncPendulum(void)
{
  float t = 0.15f * (float) Timer::fTime;
  t = t-floorf(t)+phase;
  t = sinf(M_TWO_PI*t)*speed;
  VectorCopy(origin, translation);
  VectorSet(rotation, 0, t, 90);

  UpdateTransformMatrix();

  pBSP->LinkEntity(this);
}

void Entity::UpdateFuncRotating(void)
{
  float t = (float) Timer::fTime;
  t *= speed;

  if (spawnflags & 4)
  {
    VectorSet(rotation, -t, 0, 0);
  }
  else if (spawnflags & 8)
  {
    VectorSet(rotation, 0, t, 0);
  }
  else
  {
    VectorSet(rotation, 0, 0, t);
  }

  VectorCopy(origin, translation);

  UpdateTransformMatrix();

  pBSP->LinkEntity(this);
}

void Entity::UpdateFuncTrain(void)
{
  if ((float)Timer::fTime >= wait)  // arrived at next corner
  {
    if (!numtargets) return;
    VectorCopy(targets[0]->origin, origin);
    if (targets[0]->numtargets)
    {
      targets[0] = targets[0]->targets[0];
      if (targets[0]->speed > 0) speed = targets[0]->speed;
      else speed = angle;
      vec3_t v;
      VectorSub(origin, targets[0]->origin, v);
      wait = (float)Timer::fTime+VectorLength(v)/speed; // time of arrival at new pathcorner
      random = (float)Timer::fTime;           // start time
    }
  }

  vec3_t dist =  { 0 };
  float s = ((float)Timer::fTime - random) * speed;
  if (numtargets)
  {
    VectorSub(targets[0]->origin, origin, dist);
    VectorNormalize(dist);
  }

  VectorMA(origin, s, dist, translation);

  UpdateTransformMatrix();

  pBSP->LinkEntity(this);
}

void Entity::UpdateFuncTimer(void)
{
  if ((float)Timer::fTime >= nexttime)
  {
    // fix next term
    nexttime = (float)Timer::fTime + wait + (float)crandom() * random;
    
    UpdateTarget();
  }
}

void Entity::UpdateFuncDoor(void)
{
  if (health <= 0) return;

  gConsole->Insertln("activating door!");
}

void Entity::UpdateTarget(void)
{
  for (int i = 0; i < numtargets; ++i)
  {
    switch (targets[i]->enttype)
    {
      case ET_TARGET_SPEAKER:
        targets[i]->UpdateTargetSpeaker();
        break;
      case ET_TARGET_PRINT:
        targets[i]->UpdateTargetPrint();
        break;
      case ET_TARGET_LOCATION:
        targets[i]->UpdateTargetLocation();
        break;
      case ET_TARGET_TELEPORTER:
        targets[i]->UpdateTargetTeleporter();
        break;
      default:
        break;
    }
  }
}

void Entity::UpdateTargetSpeaker(void)
{
    if (spawnflags == 0 ||
      spawnflags == 4 ||
      spawnflags == 8) playSound(noise_index);
  else if (spawnflags == 1 ||
       spawnflags == 2) toggleSoundPause(noise_index);
}

void Entity::UpdateTargetPrint(void)
{
  gConsole->Insertln(message);
}

void Entity::UpdateTargetLocation(void)
{
  gConsole->Insertln(message);
}

void Entity::UpdateTargetTeleporter(void)
{
  if (!curr_client || !numtargets) return;

  if (curr_client->TeleportTo(targets[0]->origin, targets[0]->angle, 24.f))
    playSound(noise_index);
}

void Entity::UpdateTriggerPush(void)
{
  if (!models || !numtargets || !curr_client) return;

  // TODO: Use BSP here
  if (curr_client->TouchBounds(models[0]->bbox))
  {
    if (curr_client->JumpTo(targets[0]->origin))
      playSound(noise_index);
  }
}

void Entity::UpdateTriggerTeleport(void)
{
  if (!models || !numtargets || !curr_client) return;

  // TODO: Use BSP here
  if (curr_client->TouchBounds(models[0]->bbox))
  {
    if (curr_client->TeleportTo(targets[0]->origin, targets[0]->angle, 24.f))
      playSound(noise_index);
  }
}

void Entity::UpdateTriggerMultiple(void)
{
  if (models && curr_client &&
    curr_client->TouchBounds(models[0]->bbox) &&
    (float)Timer::fTime >= nexttime)
  {
    // fix next term
    nexttime = (float)Timer::fTime + wait + (float)crandom() * random;
    UpdateTarget();
  }
}
