//-----------------------------------------------------------------------------
// Demo Header
//-----------------------------------------------------------------------------

#ifndef __DEMO_H__
#define __DEMO_H__

#include "client.h"

// Demo file structures -------------------------------------------------------

/**
 * Structure of a demo header.
 */
typedef struct
{
  char map[32];       /**< map to load for demo rendering */
  int num_frames;       /**< number of frames */
  char version[8];      /**< demo version */
  float capture_interval;   /**< time interval used for demo recording */
} DemoHeader;

/**
 * Structure of a demo frame.
 */
typedef struct
{
  float timing;       /**< frame time */
  vec3_t campos;        /**< client camera position in frame */
  vec3_t camrot;        /**< client camera rotation in frame */
} DemoFrame;

typedef enum
{
  AUTO_INTERPOLATION,     /**< cake decides automatically what interpolation mode to use */
  LINEAR_INTERPOLATION,   /**< linear interpolation mode */
  SPLINE_INTERPOLATION    /**< cubic spline interpolaiton mode */
} enum_InterpolationMode;

/**
 * The demo class stores and plays demos in cake.
 * It can be used to save a demo into a file or to play an existing
 * demo from a file.
 */
class Demo
{
  public:

    Demo(void);       /**< Default constructor */
    ~Demo(void);      /**< Default destructor */

    /**
     * Starts a demo recording for a client.
     * The function has no effect if another recording is already running.
     * @param client The client that is recorded.
     * @param mapname The name of map used while recording the demo.
     * @param framesinterval The minimum interval in seconds between 2 frame
     *        captures.
     * @return A non null value if recording could start, 0 if not.
     */
    int StartDemoRecording(Client *client, const char *mapname, float framesinterval = 0);

    /**
     * Stops the recording and write destination file.
     * @param destfile The destination file for demo saving.
     * @return A non null value if saving is successfull, 0 if it fails
     */
    int StopDemoRecording(const char* destfile);

    /**
     * Load a demo in a client view.
     * The function loads the demo file and create frames buffer.
     * @param demofile The file that contains demo.
     * @param mode Mode of interpolation used for camera position
     *        and rotation interpolation values. Linear interpolation
     *        is fast and looks well for demos with lot of keyframes,
     *        but can produce abrupt direction variation at keyframes.
     *        If demos has few keyframes, splnne interpolation mode
     *        should be used to avoid this effect. The auto interpolation
     *        mode chooses the adequate interpolation mode depending on
     *        capture keyframe rate.
     * @return The required map name for demo, or an empty string if demo file
     *         couldn't be loaded.
     */
    char* LoadDemo(const char* demofile,
             enum_InterpolationMode mode = AUTO_INTERPOLATION);

    /**
     * Start playing a demo.
     * The function has no effect if no demo file is loaded.
     * @param client The client that will be used to view demo
     * @param loop Define if the demo must loop automatically
     */
    void StartDemoPlaying(Client *client, bool loop = false);

    /**
     * Stops playing a demo.
     * The function has no effect if no demo is playing.
     */
    void StopDemoPlaying(void);

    /**
     * Inform the demo class that a new frame is trigged
     * This function must be used to inform that a new frame has been done. The
     * function is used both for demo recording and playing. If recording, the
     * function checks if a frame must be recorded (depending on framespersecond
     * value). The function returns without doing anything is no new frame record
     * is needed.
     * If playing, the function interpolate camera position in function of framerate.
     * @return A null value if there is currently no recording or playing, the current
               position in demo (between 0 to 1).
     */
    float NewFrame(void);

    /**
     * Dump demo file content (for debug purpose only)
     * @param demofile The demo file to analyse.
     * @param n Number of frames to show (all = -1)
     */
    void DemoDump(char *demofile, int n = 0);

    bool isRecording;                 /**< A demo is currently being recorded */
    bool isPlaying;                   /**< A demo is currently being played */

  private:
    char currmapname[32];               /**< Map name used for demo (recording) */
    Client *currclient;                 /**< Current client (playing or recording) */
    float frame_interval;               /**< Interval between to captures (recording only) */
    float last_frame_time;                /**< Time of last capture (recording only) */
    float start_demo_time;                /**< Time of demo start (playing or recording) */
    int currframenum;                 /**< Current frame number (playing only) */
    enum_InterpolationMode interpolationmode;     /**< Interpolation mode (playing only) */
    bool isLooping;                   /**< Demo is looping (playing only) */

    int numframes;                    /**< Number of frames */
    int numframesAllocated;               /**< Number of allocated frames */
    DemoFrame *frames;                  /**< Demo frames (recording only) */

    vec3_t *cam_positions;                /**< Camera positions array */
    vec3_t *cam_positions_tangents;           /**< Tangents to camera positions (required by spline interpolation) */
    vec3_t *cam_rotations;                /**< Camera rotations array */
    vec3_t *cam_rotations_tangents;           /**< Tangents to camera rotations (required by spline interpolation) */
    float *timings;                   /**< Timings array */

    bool noclipstate;                 /**< Used to store noclip state */

    void AddFrame(void);                /**< Register the frame */
};

#endif  /* __DEMO_H__ */
