//-----------------------------------------------------------------------------
// Commands
//-----------------------------------------------------------------------------

#ifndef __COMMANDS_H__
#define __COMMANDS_H__

typedef void (*xcommand_t) (int argc, char *argv[]);

/**
 * Command structure.
 */
typedef struct cmd_function_s
{
  struct cmd_function_s *next;      /**< Pointer to next command */
  char          *name;      /**< Command name */
  xcommand_t        function;   /**< Pointer to corresponding function */
  char *          description;  /**< Function description */
} cmd_function_t;

typedef char commline[COMMAND_LINELENGTH];

/**
 * Commands class
 * The commands are used in the console to do some modifications on the
 * configuration of the engine. A command is typically a structure containing
 * a name, an attached function and an eventual description.
 * @see cmd_function_t
 */
class Commands
{
    int cmd_argc;
    char** cmd_argv;

    char last_command[COMMAND_LINELENGTH];

    cmd_function_t *cmd_functions;    /**< Pointer on the first function of the list */

    commline *CommandsHistory;      /**< commands history
                       *   The allocation is made block by block, so the console has more allocated lines than filled lines^. */
    int NbrAllocatedHistLines;      /**< number of allocated lines for history */
    int NbrCommHistLines;       /**< number of history lines */

  public:
    bool AddToConsole;          /**< defines if commands must be added to console */
    bool AddToHistory;          /**< defines if commands must be added to history */

    Commands(void);
    ~Commands(void);

    /**
     * Add new command.
     * Add a new command to commands list. The function stops if new
     * command already exists.
     * @param cmd_name the new command name
     * @param function the command associated function
     * @param description_message a description message for the command
     */
    void AddCommand(char *cmd_name, xcommand_t function, const char* description_message = NULL);

    /**
     * Set a new description for an existing command.
     * @param cmd_name the command name
     * @param description a null ended string containing command description
     */
    void SetDescription(char *cmd_name, const char *description);

    /**
     * Remove an existing command of the list.
     * Commands can be removed from commands list. Remove it by entering
     * the command name. If the command cannot be found, the function has
     * no effect.
     * @param cmd_name the command name
     */
    void RemoveCommand(char *cmd_name);

    /**
     * Check for existing command.
     * You can check if a command is already defined in commands list.
     * @param cmd_name the command name
     * @return a boolean value that has 1 if the command was found, else 0
     */
    bool Exists(char *cmd_name);

    /**
     * Console command autocomplete.
     * The function helps you to find the name of an existing command. The
     * function works with global console (gConsole variable).
     * @param partial the partial command to complete
     * @param display_solutions displays the list of solutions if more than
     *        one solution
     * @param auto_complete_command complete current console command with
     *        common begin of every potential solution
     */
    void CompleteCommand(const char *partial, bool display_solutions = 1, bool auto_complete_command = 1);

    /**
     * Give the first solution for auto-complete.
     * Solutions are stored in a list that can be accessed with current
     * function. The function puts the first solution of the list in the
     * current console command. The function initialize an internal
     * solution counter and is required to access other solutions.
     * @param partial the partial command
     */
    void FirstSolution(const char *partial = NULL);

    /**
     * Give the next solution for auto-complete.
     * Solutions are stored in a list that can be accessed with current
     * function. The function puts the next solution of the list in the
     * current console command. The function FirstSolution should have
     * been used before to initialize internal solution counter.
     */
    void NextSolution(void);

    /**
     * Give the previous solution for auto-complete.
     * Solutions are stored in a list that can be accessed with current
     * function. The function puts the previous solution of the list in the
     * current console command. The function FirstSolution should have
     * been used before to initialize internal solution counter.
     */
    void PrevSolution(void);

    /**
     * Execute a command.
     * The function execute function that is associated with command. It
     * also check for variables and aliases, and execute them if the
     * commandline matches with.
     * @param commandline the command to execute with arguments
     * @return a non zero integer if function exists, 0 if not
     */
    int ExecuteCommand(const char *commandline, ...);

    /**
     * Displays the commands list.
     * The commands list is displayed in the console. You can specify if
     * descpription must be written.
     * @param display_descriptions a value that indicates if description
     *        must be given.
     */
    void DisplayCommands(int display_descriptions = 0);

    /**
     * Repeat last executed command.
     * Last command is stored and you can execute it again.
     */
    void RepeatLastCommand(void);

    /**
     * Get the function associated to a command.
     * @param cmd_name the command name
     * @return a pointer to the associated command
     */
    cmd_function_t* GetFunction(char *cmd_name);

    /**
    * Number of lines in the history.
    * @return The history length.
    */
    int GetNbrCommHistLines(void);

    char* GetHistoryLine(int l);
    void AddToCommHistory(char* s);
};

#endif  /* __COMMANDS_H__ */
