//-----------------------------------------------------------------------------
// Commands
//-----------------------------------------------------------------------------

#include "cake.h"
#include "commands.h"
#include "alias.h"
#include "console.h"
#include "logfile.h"
#include "render.h"
#include "vars.h"
#include "definitions.h"
#include "mem.h"
#include "system.h"

#include <stdio.h>
#ifdef WIN32
  #include <tchar.h>          // for _vsntprintf
#else
  #include <stdlib.h>
  #include <stdarg.h>
  //#include <sys/va_list.h>
  #define _vsntprintf vsnprintf
#endif

// For function completion
typedef struct _charlist
{
  commline name;
  struct _charlist *next;
  struct _charlist *prev;
} charlist;

charlist *llist = NULL;
static charlist *curr = NULL;

void cmd_help (int argc, char *argv[])
{
  if (argc > 1)
  {
    cmd_function_t* cmd = gCommands->GetFunction(argv[1]);
    if (cmd)
      gConsole->Insertln("^5%s^0: %s", argv[1], cmd->description);
    else
      gConsole->Insertln("^5WARNING: %s -> Command not found", argv[1]);
  }
  else gConsole->Insertln("usage: %s <string>", argv[0]);
}

void cmd_commands(int argc, char *argv[])
{
  gCommands->DisplayCommands(argc>1?1:0);
}

void cmd_echo(int argc, char *argv[])
{
  if (argc > 1)
  {
    if (!stricmp(argv[1],"on") || !stricmp(argv[1],"1")) gCommands->AddToConsole = true;
    else if (!stricmp(argv[1],"off") || !stricmp(argv[1],"0")) gCommands->AddToConsole = false;
    else
    {
      gConsole->Insertln("^5Argument not valid");
    }
  }
  else gConsole->Insertln("echo : %s", gCommands->AddToConsole?"on":"off");
}

void cmd_rem(int argc, char *argv[])
{
  if (argc > 1)
  {
    char command[1024] = { '\0' };
    for (int i = 0; i < argc; ++i)
    {
      strcat(command, argv[i]);
      if (i != argc-1) strcat(command, " ");
    }
    if (!stricmp(argv[0],"rem")) gConsole->Insertln(command+4);
    else if (!stricmp(argv[0],"//")) gConsole->Insertln(command+3);
    else if (!stricmp(argv[0],"#")) gConsole->Insertln(command+2);
  }
  else gConsole->Insertln("usage: %s <string>", argv[0]);
}

void cmd_togglehistory(int argc, char *argv[])
{
  gCommands->AddToHistory = !gCommands->AddToHistory;
}

Commands::Commands(void)
{
  cmd_argc = 0;
  cmd_argv = NULL;

  NbrAllocatedHistLines = HISTBLOC;
  NbrCommHistLines    = 0;

  memset(last_command, '\0', COMMAND_LINELENGTH*sizeof(char));

  cmd_functions = NULL;

  CommandsHistory = (commline*) cake_malloc(NbrAllocatedHistLines*sizeof(commline), "Commands::Commands.CommandsHistory");
  for (int i = 0; i < NbrAllocatedHistLines; ++i) memset(CommandsHistory[i], '\0', CONSOLE_LINELENGTH);

  AddCommand("help", cmd_help, "displays help about a command");
  AddCommand("cmdlist", cmd_commands, "display all available commands in the console (if an argumentis present, the functions descriptions are given)");
  AddCommand("echo", cmd_echo, "defines the echo value (if echo is on, the function callsare added to the console, if not, only the function results are written in the console)");
  AddCommand("rem", cmd_rem, "writes a remark in the console");
  AddCommand("//", cmd_rem, "writes a remark in the console");
  AddCommand("#", cmd_rem, "writes a remark in the console");
  AddCommand("togglehistory", cmd_togglehistory, "enables or disables the history for console commands");
}

Commands::~Commands(void)
{
  RemoveCommand("help");
  RemoveCommand("cmdlist");
  RemoveCommand("echo");
  RemoveCommand("rem");
  RemoveCommand("//");
  RemoveCommand("#");
  RemoveCommand("togglehistory");

  cake_free(CommandsHistory); CommandsHistory = NULL;

  if (cmd_argv)
  {
    for (int i = 0; i < cmd_argc; ++i) cake_free(cmd_argv[i]);
    cake_free(cmd_argv);
    cmd_argv = NULL;
  }

  // Unallocation of completion list
  for (; llist;)
  {
    charlist *back = llist->next;
    llist->next = NULL;
    llist->prev = NULL;
    cake_free(llist);
    llist = back;
    back = NULL;
  }
  llist = NULL;
  curr = NULL;

  // Clear functions table
  cmd_function_t *next, *cmd = cmd_functions;
  while (cmd)
  {
    if (cmd->description) cake_free(cmd->description);
    next = cmd->next;
    cmd->next = NULL;
    cake_free(cmd);
    cmd = next;
    next = NULL;
  }
}

void Commands::AddCommand(char *cmd_name, xcommand_t function, const char *description_message)
{
  cmd_function_t *cmd;

  // Error if the command is an alias name
  if (gAlias && gAlias->GetAlias(cmd_name)[0])
  {
    gConsole->Insertln("AddCommand: %s already defined as alias.", cmd_name);
    return;
  }

  // Error if command is a variable name
  if (gVars && gVars->IsKey(cmd_name))
  {
    gConsole->Insertln("AddCommand: %s already defined as variable.", cmd_name);
    return;
  }

  // Error if command already exists
  for (cmd = cmd_functions; cmd; cmd = cmd->next)
  {
    if (!stricmp(cmd_name, cmd->name))
    {
      gConsole->Insertln("AddCommand: %s already defined.", cmd_name);
      return;
    }
  }

  cmd = (cmd_function_t*) cake_malloc(sizeof(cmd_function_t), "Commands::AddCommand.cmd");
  cmd->name = cmd_name;
  cmd->function = function;
  cmd->next = cmd_functions;
  if (description_message)
  {
    cmd->description = (char*) cake_malloc(strlen(description_message)+1, "Commands::AddCommand.cmd->description");
    memset(cmd->description, '\0', strlen(description_message)+1);
    strcpy(cmd->description, description_message);
  }
  else cmd->description = NULL;
  cmd_functions = cmd;
}

void Commands::SetDescription(char *cmd_name, const char *description)
{
  // Check functions
  cmd_function_t *cmd;
  for (cmd = cmd_functions; cmd; cmd = cmd->next)
  {
    if (!stricmp(cmd_name, cmd->name))
    {
      if (cmd->description) cake_free(cmd->description);
      cmd->description = (char*) cake_malloc(strlen(description)+1, "Commands::SetDescription.cmd->description");
      memset(cmd->description, '\0', strlen(description)+1);
      strcpy(cmd->description, description);
      return;
    }
  }
}

void Commands::RemoveCommand(char *cmd_name)
{
  cmd_function_t* cmd, **back;
  
  back = &cmd_functions;
  while (1)
  {
    cmd = *back;
    if (!cmd)
    {
      // The command cannot be found in ithe list
      gConsole->Insertln("RemoveCommand: %s not in command list.", cmd_name);
      return;
    }
    if (!stricmp(cmd_name, cmd->name))
    {
      *back = cmd->next;
      if (cmd->description) cake_free(cmd->description);
      cake_free(cmd);
      return;
    }
    back = &cmd->next;
  }
}

cmd_function_t* Commands::GetFunction(char *cmd_name)
{
  // Check functions
  for (cmd_function_t *cmd = cmd_functions; cmd; cmd = cmd->next)
  {
    if (!stricmp(cmd_name, cmd->name))
    {
      return cmd;
    }
  }

  return NULL;
}

bool Commands::Exists(char *cmd_name)
{
  for (cmd_function_t *cmd = cmd_functions; cmd; cmd = cmd->next)
  {
    if (!stricmp(cmd_name, cmd->name)) return true;
  }
  return false;
}

void Commands::CompleteCommand(const char *partial, bool display_solutions, bool auto_complete_command)
{
  int len = (int) strlen(partial);
  int n_partial_sol = 0, i;

  if (!len) return;

  /**
   * @todo Find why this declaration is indispensable. Isn't is a consequence
   *       of bad construction of WriteLine ?
   */
  char *tmp_string = (char*) cake_malloc((len + 1)*sizeof(char), "Commands::CompleteCommand.tmp_string");
  memset(tmp_string, '\0', len + 1);
  strncpy(tmp_string, partial, len);

  // Unallocation of last completion list
  for (; llist;)
  {
    charlist *back = llist->next;
    llist->next = NULL;
    llist->prev = NULL;
    cake_free(llist);
    llist = back;
    back = NULL;
  }
  llist = NULL;

  for (cmd_function_t *cmd = cmd_functions; cmd; cmd = cmd->next)
  {
    if (!strnicmp(tmp_string, cmd->name, len))
    {
      charlist *tmp = (charlist*) cake_malloc(sizeof(charlist), "Commands::CompleteCommand.tmpA");
      memset(tmp->name, '\0', COMMAND_LINELENGTH);
      strcpy(tmp->name, cmd->name);
      tmp->next = llist;
      tmp->prev = NULL;
      if (llist) llist->prev = tmp;
      llist = tmp;

      ++n_partial_sol;
    }
  }

  // Add all matching variables to list
  int nvars = gVars->GetNumber();
  char varname[128];
  for (i = 0; i < nvars; ++i)
  {
    strncpy(varname, gVars->GetName(i), 128);
    if (!strnicmp(tmp_string, varname, len))
    {
      charlist *tmp = (charlist*) cake_malloc(sizeof(charlist), "Commands::CompleteCommand.tmpB");
      memset(tmp->name, '\0', COMMAND_LINELENGTH);
      strcpy(tmp->name, varname);
      tmp->next = llist;
      tmp->prev = NULL;
      if (llist) llist->prev = tmp;
      llist = tmp;

      ++n_partial_sol;
    }
  }

  // Add all matching aliases to list
  int nalias = gAlias->GetNumber();
  char aliasname[128];
  for (i = 0; i < nalias; ++i)
  {
    strncpy(aliasname, gAlias->GetName(i), 128);
    if (!strnicmp(tmp_string, aliasname, len))
    {
      charlist *tmp = (charlist*) cake_malloc(sizeof(charlist), "Commands::CompleteCommand.tmpC");
      memset(tmp->name, '\0', COMMAND_LINELENGTH);
      strcpy(tmp->name, aliasname);
      tmp->next = llist;
      tmp->prev = NULL;
      if (llist) llist->prev = tmp;
      llist = tmp;

      ++n_partial_sol;
    }
  }

  // Sort list

  // Get common prefix of list
  commline prefix = { '\0' };
  char c;
  int n = 0;
  bool search_next = true;

  while (search_next && n_partial_sol)
  {
    if ((int) strlen(llist->name) > n) c = llist->name[n];
    else break;

    for (charlist *tmp = llist; tmp; tmp = tmp->next)
    {
      if ((int) strlen(tmp->name) > n)
      {
        if (tmp->name[n] != c)
        {
          search_next = false;
          break;
        }
      }
    }

    if (search_next) prefix[n++] = c;
  }

  // Display the list if more than one solution
  if (display_solutions && n_partial_sol > 1)
  {
    gConsole->Insertln("%s%s", gConsole->GetPrompt(), tmp_string);
    gLogFile->OpenFile();
    for (charlist *tmp = llist; tmp; tmp = tmp->next)
    {
      gConsole->Insertln("     %s", tmp->name);
    }
    gLogFile->CloseFile();
  }
  else if (n_partial_sol == 1) prefix[n] = ' ';

  // Complete current console command with prefix
  if (auto_complete_command && n_partial_sol > 0) gConsole->SetCurrentCommand(prefix);

  // Free memory
  cake_free(tmp_string);
}

void Commands::FirstSolution(const char *partial)
{
  if (partial != NULL) CompleteCommand(partial, 0, 0);

  if (llist == NULL) return;
  
  curr = llist;

  gConsole->SetCurrentCommand("%s ", llist->name);
}

void Commands::NextSolution(void)
{
  if (llist == NULL) return;

  if (curr) curr = curr->next;
  if (curr == NULL) curr = llist;

  gConsole->SetCurrentCommand("%s ", curr->name);
}

void Commands::PrevSolution(void)
{
  if (llist == NULL) return;

  if (curr) curr = curr->prev;
  if (curr == NULL)
    for (charlist* tmp = llist; tmp; tmp = tmp->next)
      curr = tmp;

  gConsole->SetCurrentCommand("%s ", curr->name);
}

void Commands::DisplayCommands(int display_descriptions)
{
  gLogFile->OpenFile();
  for (cmd_function_t *cmd = cmd_functions; cmd; cmd = cmd->next)
  {
    if (display_descriptions)
      gConsole->Insertln("^5%s^0 : %s", cmd->name, cmd->description);
    else
      gConsole->Insertln("\t%s", cmd->name);
  }
  gLogFile->CloseFile();
}

int Commands::ExecuteCommand(const char *commandline, ...)
{
  va_list msg;
  char buffer[COMMAND_LINELENGTH] = { '\0' };
  char *command = buffer;
  
  va_start (msg, commandline);
  _vsntprintf (buffer, COMMAND_LINELENGTH - 1, commandline, msg);
  va_end (msg);

  if (strlen(command) <= 0) return 0;

  strcpy(last_command, buffer);           // Backup last command

  int i;
  int res = 0;

  gConsole->addToMessages = false;          // We don't dislay commands in the mini-commands,
                            // but only functions results
  i = (int) strlen(command)-1;
  while (command[i] == ';')             // Skip the ; at the command end
    command[i--] = '\0';

  if (AddToConsole && command[0] != '@')
    gConsole->Insertln("\n%s%s", gConsole->GetPrompt(), command);   // Displays the command with prompt
  gConsole->ReInitCurrentCommand();
  gConsole->addToMessages = true;
  if (AddToHistory) Commands::AddToCommHistory(command);

  // Convertit la commande en minuscule
  char c[COMMAND_LINELENGTH] = { '\0' };
  while (command[0] == '@') ++command;        // The @ don't appear at the line beginning
  strcpy(c, command);
//  _strlwr(c);

  // (Re-)Initialization of last command arguments
  if (cmd_argc)
  {
    for (i = 0; i < cmd_argc; ++i) cake_free(cmd_argv[i]);
    cake_free(cmd_argv);
    cmd_argv = NULL;
  }
  cmd_argc = 0;

  // Separates the command arguments
  // Get the number of aguments
  i = 0;
  while (i < (int) strlen(c))
  {
    while (c[i] == ' ' && i < (int) strlen(c)) ++i;
    if (i < (int) strlen(c))
    {
      ++cmd_argc;
      while (c[i] != ' ' && i < (int) strlen(c)) ++i;
    }
  }
  cmd_argv = (char**) cake_malloc(cmd_argc*sizeof(char*), "Commands::ExecuteCommand.cmd_argv");
  for (i = 0; i < cmd_argc; ++i)
  {
    cmd_argv[i] = (char*) cake_malloc(COMMAND_LINELENGTH*sizeof(char), "Commands::ExecuteCommand.cmd_argv[i]");
    memset(cmd_argv[i], '\0', COMMAND_LINELENGTH - 1);
    GetArg(c, i, cmd_argv[i]);
  }

  if (!cmd_argc) return 0;

  // Check functions
  cmd_function_t *cmd;
  for (cmd = cmd_functions; cmd; cmd = cmd->next)
  {
    if (!stricmp(cmd_argv[0], cmd->name))
    {
      if (cmd->function)
      {
        if (cmd_argc > 1 && !stricmp(STR_ASK, cmd_argv[1]))
        {
          gConsole->Insertln("%s: %s", cmd_argv[0], cmd->description);
          res = 1;
          break;
        }

        cmd->function(cmd_argc, cmd_argv);
        res = 1;
        break;
      }
    }
  }

  // Check variables
  if (!res && gVars)
  {
    Var* v = gVars->GetVar(cmd_argv[0]);
    if (v != NULL)
    {
      char* a = v->svalue;
      if (a[0])
      {
        if (cmd_argc > 1)
        {
          if (v->flags & VF_SYSTEM && gVars->IntForKey("sys_appstate") != INITIALIZING)
            gConsole->Insertln("^4You are not allowed to modify \"%s\".", cmd_argv[0]);
          else
          {
            gVars->SetKeyValue(cmd_argv[0], cmd_argv[1]);
            if (v->flags & VF_LATCH && gVars->IntForKey("sys_appstate") != INITIALIZING)
              gConsole->Insertln("%s will be changed upon restarting.", cmd_argv[0]);
            if (v->flags & VF_DEBUG && gVars->IntForKey("sys_appstate") != INITIALIZING)
              gConsole->Insertln("^1!! DEBUG MODE !! ^0Do enable this variable for debug only.");
          }
        }
        else gConsole->Insertln("\"%s\" : \"%s\" [default: \"%s\"]", cmd_argv[0], a, v->default_value);

        res = 1;
      }
    }
  }

  // Check alias
  if (!res && gAlias)
  {
    char* a = gAlias->GetAlias(cmd_argv[0]);
    if (a[0])
    {
      if (gAlias->depth++ < MAX_ALIAS_DEPTH)
      {
        gConsole->Insertln("alias redirection...");

        char str[COMMAND_LINELENGTH];
        sprintf(str, "%s ", a);
        for (int i = 1; i < cmd_argc; ++i)
        {
          strcat(str, cmd_argv[i]);
          strcat(str, " ");
        }
        res = ExecuteCommand(str);
      }
      else
      {
        gConsole->Insertln("^5WARNING: potential infinite recursive loop detected !");
      }
    }
  }
  gAlias->depth = 0;

  return res;
}

void Commands::RepeatLastCommand()
{
  if (strlen(last_command)) ExecuteCommand(last_command);
}

char* Commands::GetHistoryLine(int l)
{
  if (!NbrCommHistLines || l < 1) return "";
  else if (l > NbrCommHistLines) return CommandsHistory[NbrCommHistLines-1];
  else return CommandsHistory[NbrCommHistLines-l];
}

int Commands::GetNbrCommHistLines(void)
{
  return NbrCommHistLines;
}

void Commands::AddToCommHistory(char* s)
{
  // Rem : On alloue par blocs de taille HISTBLOC, pour viter de trop frquentes allocations
  if (NbrCommHistLines == NbrAllocatedHistLines)
  {
    NbrAllocatedHistLines += HISTBLOC;
    CommandsHistory = (commline*) cake_realloc(CommandsHistory, NbrAllocatedHistLines*sizeof(commline), "Commands::AddToCommHistory.CommandsHistory");
  }
  ++NbrCommHistLines;
  memset(CommandsHistory[NbrCommHistLines-1], '\0', CONSOLE_LINELENGTH);  
  strncpy(CommandsHistory[NbrCommHistLines-1], s, CONSOLE_LINELENGTH);
}
