//-----------------------------------------------------------------------------
// Bezier Patch
//-----------------------------------------------------------------------------

#include "bezierpatch.h"
#include "console.h"
#include "logfile.h"
#include "vars.h"
#include "math.h"
#include "mem.h"
#include "system.h"

int Patch_FlatnessTest(float maxflat, vec4_t point0, vec4_t point1, vec4_t point2);
void Patch_Evaluate_QuadricBezier(float t, vec4_t point0, vec4_t point1, vec3_t point2, vec4_t out);

int Patch_FlatnessTest(float maxflat, vec4_t point0, vec4_t point1, vec4_t point2)
{
  float l, dist;
  vec3_t d, t, w;
  VectorSub(point2, point0, d);

  l = VectorLength(d);
  if (!l) return 0;
  VectorScale(d, (1.f/l), d);

  VectorSub(point1, point0, t);
  l = DotProduct(t, d);
  VectorScale(d, l, w);
  VectorSub(t, w, t);
  dist = VectorNormalize(t);
  if (fabs(dist) < maxflat) return 0;

  vec3_t v1, v2, v3;

  VectorAvrg(point1, point0, v1);
  VectorAvrg(point2, point1, v2);
  VectorAvrg(v1, v2, v3);

  return 1 + max(Patch_FlatnessTest(maxflat, point0, v1, v3), Patch_FlatnessTest(maxflat, v3, v2, point2));
}

void Patch_GetFlatness(float maxflat, vec4_t *points, int *patch_cp, int *flat)
{
  int i, p, u, v;

  flat[0] = flat[1] = 0;
  for (v = 0; v < patch_cp[1] - 1; v += 2)
  {
    for (u = 0; u < patch_cp[0] - 1; u += 2)
    {
      p = v * patch_cp[0] + u;

      i = Patch_FlatnessTest(maxflat, points[p],
                      points[p+1],
                      points[p+2]);
      flat[0] = max(flat[0], i);
      i = Patch_FlatnessTest(maxflat, points[p+patch_cp[0]],
                      points[p+patch_cp[0]+1],
                      points[p+patch_cp[0]+2]);
      flat[0] = max(flat[0], i);
      i = Patch_FlatnessTest(maxflat, points[p+2*patch_cp[0]],
                      points[p+2*patch_cp[0]+1],
                      points[p+2*patch_cp[0]+2]);
      flat[0] = max(flat[0], i);

      i = Patch_FlatnessTest(maxflat, points[p],
                      points[p+patch_cp[0]],
                      points[p+2*patch_cp[0]]);
      flat[1] = max(flat[1], i);
      i = Patch_FlatnessTest(maxflat, points[p+1],
                      points[p+patch_cp[0]+1],
                      points[p+2*patch_cp[0]+1]);
      flat[1] = max(flat[1], i);
      i = Patch_FlatnessTest(maxflat, points[p+2],
                      points[p+patch_cp[0]+2],
                      points[p+2*patch_cp[0]+2]);
      flat[1] = max(flat[1], i);
    }
  }
}

void Patch_Evaluate_QuadricBezier(float t, vec4_t point0, vec4_t point1, vec3_t point2, vec4_t out)
{
  float qt = t * t;
  float dt = 2.0f * t, tt;
  vec4_t tvec4;

  tt = 1.0f - dt + qt;
  Vector4Scale(point0, tt, out);

  tt = dt - 2.0f * qt;
  Vector4Scale(point1, tt, tvec4);
  Vector4Add(out, tvec4, out);

  Vector4Scale(point2, qt, tvec4);
  Vector4Add(out, tvec4, out);
}

void Patch_Evaluate(vec4_t *p, int *numcp, int *tess, vec4_t *dest)
{
  int num_patches[2], num_tess[2];
  int index[3], dstpitch, u, v, x, y;
  float s, t, step[2];
  vec4_t *tvec, pv[3][3], v1, v2, v3;

  num_patches[0] = numcp[0] >> 1;
  num_patches[1] = numcp[1] >> 1;
  dstpitch = num_patches[0] * tess[0] + 1;

  step[0] = 1.0f / (float)tess[0];
  step[1] = 1.0f / (float)tess[1];

  for (v = 0; v < num_patches[1]; ++v)
  {
    // last patch has one more row 
    if (v < num_patches[1] - 1) num_tess[1] = tess[1];
    else num_tess[1] = tess[1] + 1;

    for (u = 0; u < num_patches[0]; ++u)
    {
      // last patch has one more column
      if (u < num_patches[0] - 1) num_tess[0] = tess[0];
      else num_tess[0] = tess[0] + 1;

      index[0] = (v * numcp[0] + u) << 1;
      index[1] = index[0] + numcp[0];
      index[2] = index[1] + numcp[0];

      // current 3x3 patch control points
      Vector4Copy(p[index[0]  ], pv[0][0]);
      Vector4Copy(p[index[1]  ], pv[0][1]);
      Vector4Copy(p[index[2]  ], pv[0][2]);
      Vector4Copy(p[index[0]+1], pv[1][0]);
      Vector4Copy(p[index[1]+1], pv[1][1]);
      Vector4Copy(p[index[2]+1], pv[1][2]);
      Vector4Copy(p[index[0]+2], pv[2][0]);
      Vector4Copy(p[index[1]+2], pv[2][1]);
      Vector4Copy(p[index[2]+2], pv[2][2]);
      
      t = 0.0f;
      tvec = dest + v*tess[1]*dstpitch + u*tess[0];

      for (y = 0; y < num_tess[1]; ++y, t += step[1])
      {
        Patch_Evaluate_QuadricBezier(t, pv[0][0], pv[0][1], pv[0][2], v1);
        Patch_Evaluate_QuadricBezier(t, pv[1][0], pv[1][1], pv[1][2], v2);
        Patch_Evaluate_QuadricBezier(t, pv[2][0], pv[2][1], pv[2][2], v3);

        s = 0.0f;
        for (x = 0; x < num_tess[0]; ++x, s += step[0])
          Patch_Evaluate_QuadricBezier(s, v1, v2, v3, tvec[x]);

        tvec += dstpitch;
      }
    }
  }
}

//-----------------------------------------------------------------------------

PatchesGroup::PatchesGroup(void)
{
  surfidx = NULL;
  nsurf = 0;
  size = 1;
  middle[0] = middle[1] = middle[2] = 0.f;
  bbox[0] = bbox[1] = bbox[2] = bbox[3] = bbox[4] = bbox[5] = 0;
}
PatchesGroup::~PatchesGroup(void)
{
  if (surfidx) cake_free(surfidx);
  surfidx = NULL;
}

void PatchesGroup::Add(int val)
{
  if (surfidx)
  {
    surfidx = (int*) cake_realloc(surfidx, (++nsurf)*sizeof(int), "PatchesGroup::Add.surfidx (realloc)");
  }
  else
  {
    surfidx = (int*) cake_malloc(sizeof(int), "PatchesGroup::Add.surfidx (malloc)");
    nsurf = 1;
  }

  if (!surfidx) ThrowException(ALLOCATION_ERROR, "PatchesGroup::Add.surfidx");

  surfidx[nsurf-1] = val;
}

void PatchesGroup::Update(float val, Surface **patches)
{
  int currlevel;
  if (val < MINLIMIT) currlevel = patches[surfidx[0]]->levels-1;  // maximum tesselation
  else if (val > MAXLIMIT) currlevel = 1;             // minimum tesselation
  else currlevel = 1 + (int) ((val-(float)MAXLIMIT)*((float)(2-patches[surfidx[0]]->levels)/(float)(MAXLIMIT-MINLIMIT)));

  if (currlevel < 1) currlevel = 1;
  if (currlevel >= patches[surfidx[0]]->levels) currlevel = patches[surfidx[0]]->levels-1;

  for (int i = 0; i < nsurf; ++i)
    patches[surfidx[i]]->currentlevel = currlevel;
}

void PatchesGroup::DumpGroup(Surface **patches)
{
  gLogFile->OpenFile();
  for (int i = 0; i < nsurf; ++i)
  {
    gConsole->Insertln("patch %d:", patches[i]->indextype);
    gConsole->Insertln("\tpatch_cp: %f %f",
      patches[i]->patch_cp[0], patches[i]->patch_cp[1]);
    gConsole->Insertln("\tcurrentlevel: %d^/%d",
      patches[i]->currentlevel, patches[i]->levels);
    gConsole->Insertln("\tnumverts: %d", patches[i]->numverts);
    gConsole->Insertln("\tnumelems: %d", patches[i]->numelems);
    gConsole->Insertln("\tbounding box: %f %f %f %f %f %f",
      patches[i]->bbox[0], patches[i]->bbox[1], patches[i]->bbox[2],
      patches[i]->bbox[3], patches[i]->bbox[4], patches[i]->bbox[5]);
    gConsole->Insertln("\tv_orig: %f %f %f",
      patches[i]->v_orig[0], patches[i]->v_orig[1], patches[i]->v_orig[2]);
    gConsole->Insertln("\tv_norm: %f %f %f",
      patches[i]->v_norm[0], patches[i]->v_norm[1], patches[i]->v_norm[2]);
    gConsole->Insertln("\tdist: %f", patches[i]->dist);
    gConsole->Insertln("\tshader: %d", patches[i]->shader);
    gConsole->Insertln("\tlm_texnum: %d", patches[i]->lm_texnum);
    gConsole->Insertln("\teffect: %f", patches[i]->effect);
    gConsole->Insertln("\tmodel: %d", patches[i]->model);
    gConsole->Insertln("\tautosprite2_axis: %d", patches[i]->autosprite2_axis);
  }
  gLogFile->CloseFile();
}
