//-----------------------------------------------------------------------------
// App
//-----------------------------------------------------------------------------

#ifndef __APP_H__
#define __APP_H__

#include "world.h"
#include "demo.h"

/**
 * App class.
 * The app class is the main application class. Every program using the
 * cake library must create this class.
 * @bug When I tried to open a map several times, the FPS were not the same. In
 *      fact there was a factor two in difference. Check why.
 * @bug When path includes missionpack directory, texture colors are not
 *      correct. Lava is green and some walls have a blue tint.
 * @see World
 */
class App
{
  private:
    char app_path[PATH_LENGTH];   /**< current application path */

    World* world;         /**< world map */

    Client*   clients;      /**< clients table */
    Demo*   demos;        /**< demos (same number as clients) */
    int     numclients;     /**< number of clients (size of table) */
    int     curr_client;    /**< current client */

    char curr_map[32];        /**< Currently running map mame*/
    bool reloading;         /**< Is currently reloading a map */

  public:
    App(void);            /**< Create application and global variables */
    ~App(void);

    /**
     * Application initialization.
     * Loads config file, initialize environment files, get system infos,
     * get shaders list, etc.
     */
    void Init(void);        

    void Run(void);
    void Shut(void);

    /**
     * Draws a new frame.
     * This is the main entry to world refresh and redraw.
     */
    void DrawWorld(void);

    /**
     * Draws the interface.
     */
    void DrawInterface(void);

    /**
     * Gets the FPS for current frame.
     * @return the current fps as a double value
     */
    double GetFPS(void);

    /**
     * Get a pointer to current world.
     * Return a NULL pointer of no world is currently defined.
     * @return The current world or a null pointer if no world is defined.
     */
    World* GetWorld(void);

    /**
     * Get the current loaded map name.
     * The returned name is the name of the file, and not the name
     * of the worldspawn entity.
     * If no map is currently loaded, this function return a null-ended
     * empty string.
     * @return The current map name.
     */
    const char* GetCurrentMapName(void);

    /**
     * Loads a map.
     * The function loads the given file (if it can) by creating a new world
     * and new clients. It returns a non zero value if loading could be done
     * successfully.
     * @return An integer value that indicates if map could be loaded (non
     * zero) or not (0).
     */
    int LoadMap(const char* mapname);

    /**
     * Reload the current map.
     * The function deletes the current world and reload the current map
     * in a new world. Clients are not deleted and stay "alive".
     * @return The same value as LoadMap function.
     */
    int ReloadMap(void);

    /**
     * Unloads the current map.
     * The function delete the current world (if defined) but does not
     * destroy existing clients. Those are destroyed and re-created by the
     * LoadMap function.
     */
    void UnloadMap(void);

    /**
     * Start recording a demo.
     * The recording of a demo can only start if almost one client is defined.
     * @param framesinterval The interval between demo keyframes.
     */
    void RecordDemo(float framesinterval = 0.1f);

    /**
     * Stops the current demo.
     * The function checks if demo is playing or recording and stop it. If the
     * demo is recording and the parameter is defined, the demo is saved into the
     * given filename. If demo is recording and no destination file is given, the
     * function generates a default filename and save the demo into it. Default
     * filenames are defined as "cakeXXX.demo" where XXX is the demo number.
     * @param destfile The destination filename for recorded demo saving.
     */
    int StopDemo(const char *destfile = NULL);

    /**
     * Plays an existing demo.
     * The function loads a demo and plays it into current client. If the current
     * client is already playing of recording a demo, the function has no effect.
     * The function can load a new world if needed by demo.
     * @param demofile The demo filename to load.
     * @param loop A boolean value defining if demo must loop when it comes at the
     *        end.
     * @param mode The interpolation mode for demo playback.
     */
    void PlayDemo(const char *demofile, bool loop, enum_InterpolationMode mode);

    /**
     * Creates new clients.
     * This function destroys existing clients and creates new ones. The function
     * has no effect if one of existing clients is currently playing or recording
     * a demo. In this case, the function returns 0.
     * @param num The number of clients to be created.
     * @return A zero if clients couldn't be created and a non-zero if clients
     *         could be created.
     */
    int CreateClients(int num);

    /**
     * Gets the number of existing clients.
     * @return The number of existing clients.
     */
    int GetNumClients(void);

    /**
     * Sets a client as current default client.
     * @param num The number of client to be set as default client.
     */
    void SetCurrentClient(int num);

    /**
     * Get a client.
     * @param num The index of client in the clients array.
     * @return A pointer to wished client or a null pointer if it doesn't
     *         exist.
     */
    Client* GetClient(int num);

    /**
     * Get the current default client.
     * @return A pointer to current default client or a null pointer if it
     *         doesn't exist.
     */
    Client* GetCurrentClient(void);

    /**
     * Get the index of current default client.
     * If no client is available, the function returns -1.
     * @return The index of current default client.
     */
    int GetCurrentClientIndex(void);

    /**
     * Initialize the clients.
     * The function calculates the clients viewport position and size so
     * they are all visible simultaneous. It is automatically called by
     * the CreateClients() function and should be called each time the
     * main window is resized.
     */
    void InitClients(void);
};

#endif  /* __APP_H__ */
