// Copyright (c) 2002-2010 Wieger Wesselink
//
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)

/// \file sat/variable_set.h
/// \brief Contains a data structure that maps bdd variables to strings and vice versa

#ifndef SAT_VARIABLE_SET_H
#define SAT_VARIABLE_SET_H

#include <string>
#include <map>
#include <set>
#include <iomanip>
#include <utility>
#include <iostream>
#include <sstream>

namespace buddy {

inline
std::string format_int(int i, int width = 1, char fill = ' ')
{
  std::ostringstream out;
  out << std::setw(width) << std::setfill(fill) << i;
  return std::string(out.str().c_str());
}

struct less_string
{
  bool operator()(const std::string& p, const std::string& q) const
  {
    if (p.length() != q.length())
      return p.length() < q.length();
    return p < q;
  }
};

struct less_string_01
{
  bool operator()(const std::string& p, const std::string& q) const
  {
    if (p.length() == 0 || q.length() == 0)
      return p < q;

    char plast = p[p.length() - 1];
    char qlast = q[q.length() - 1];
    
    if (plast != qlast)
      return plast < qlast;

    if (p.length() != q.length())
      return p.length() < q.length();

    return p < q;
  }
};

// The class variable_set stores a mapping: string -> int.
class variable_set
{
 public:
    typedef std::map<std::string, int> string_map;

 private:
    string_map m_data;

    // returns the number of digits of n
    unsigned int num_digits(unsigned int n)
    {
      unsigned int x = 1;
      unsigned int result = 0;
      do {
        x = 10*x;
        result++;
      }
      while (x < n);
      return result;
    }

 public:

    // maps the string s to the current size of the set if it isn't available yet
    void insert(const std::string& s, int index = -1)
    {
      string_map::const_iterator i = m_data.find(s);
      if (i == m_data.end())
      {
        if (index == -1)
        {
          index = m_data.size();
        }
        m_data.insert(std::make_pair(s, index));
      }
    }

    variable_set()
    {}
    
    variable_set(const std::set<std::string>& variables)
    {
      for (std::set<std::string>::const_iterator i = variables.begin(); i != variables.end(); ++i)
      insert(*i);
    }

    variable_set(const std::vector<std::string>& variables)
    {
      int index = 0;
      for (std::vector<std::string>::const_iterator i = variables.begin(); i != variables.end(); ++i)
      insert(*i, index++);
    }

    variable_set(unsigned int n)
    {
      unsigned int width = num_digits(n);
      for (unsigned int i = 0; i < n; i++)
        m_data.insert(std::make_pair(format_int(i, width, '0'), i));
    }

    // numbers the strings according to less_string
    void sort()
    {
      int index = 0;
      for(string_map::iterator i = m_data.begin(); i != m_data.end(); ++i)
        i->second = index++;
    }
  
    // numbers the strings according to less_string_01
    void sort_01()
    {
      std::set<std::string, less_string_01> s;
      for (string_map::iterator i = m_data.begin(); i != m_data.end(); ++i)
        s.insert(i->first);
      int index = 0;
      for (std::set<std::string, less_string_01>::iterator i = s.begin(); i != s.end(); ++i)
        m_data[*i] = index++;
    }

    void set_index(const std::string& s, int index)
    {
      m_data[s] = index;
    }
  
    int index(const std::string& s) const
    {
      string_map::const_iterator i = m_data.find(s);
      return i == m_data.end() ? -1 : i->second;
    }
    
    std::string name(unsigned i) const
    {
      string_map::const_iterator j = m_data.begin();
      std::advance(j, i);
      return j->first;
    }

    int index(unsigned i) const
    {
      string_map::const_iterator j = m_data.begin();
      std::advance(j, i);
      return j->second;
    }
    
    unsigned size() const
    {
      return m_data.size();
    }
    
    void clear()
    {
      m_data.clear();
    }
    
    void print()
    {
      for (string_map::iterator i = m_data.begin(); i != m_data.end(); ++i)
      {
        std::cout << i->first << " => " << i->second << std::endl;
      }
    }

    // returns a string representation of the variables:
    // a => 0
    // b => 1
    std::string to_string()
    {
      std::stringstream out;
      for (string_map::iterator i = m_data.begin(); i != m_data.end(); ++i)
      {
        out << i->first << " => " << i->second << std::endl;
      }
      return out.str();
    }
    
    void remove(const std::string& name)
    {
      string_map::iterator i = m_data.find(name);
      if (i != m_data.end())
        m_data.erase(i);
    }
};

} // namespace buddy

#endif // SAT_VARIABLE_SET_H
