// Copyright (c) 2009-2010 Wieger Wesselink
//
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)

/// \file sat/parser.h
/// \brief Contains bdd functions

#ifndef SAT_PARSER_H
#define SAT_PARSER_H

#include <iostream>
#include <stack>
#include <string>
#include <vector>

#include <boost/config/warning_disable.hpp>         
#include <boost/spirit/include/qi.hpp>              
#include <boost/spirit/include/support_multi_pass.hpp>
#include <boost/spirit/include/classic_position_iterator.hpp>
#include <boost/spirit/include/phoenix_operator.hpp>
#include <boost/spirit/include/phoenix_object.hpp>  
#include <boost/bind.hpp>

#include "sat/buddy.h"

namespace sat {

  namespace qi = boost::spirit::qi;
  namespace phoenix = boost::phoenix;
  namespace ascii = boost::spirit::ascii;
  namespace classic = boost::spirit::classic;

  struct bdd_builder
  {
    buddy::variable_set variables;
    buddy::variable_set variables0;
    buddy::variable_set variables1;
  
    bdd_builder()
    {}
  
    bdd_builder(const buddy::variable_set& variables0)
      : variables(variables0)
    {}
    
    bdd true_()
    {
      return bddtrue;
    }
    
    bdd false_()
    {
      return bddfalse;
    }
    
    bdd and_(bdd x, bdd y)
    {
      return x & y;
    }
  
    bdd not_(bdd x)
    {
      return !x;
    }
  
    bdd or_(bdd x, bdd y)
    {
      return x | y;
    }
  
    bdd iff(bdd x, bdd y)
    {
      return bdd_apply(x, y, bddop_biimp);
    }
  
    bdd implies(bdd x, bdd y)
    {
      return bdd_apply(x, y, bddop_imp);
    }
  
    bdd xor_(bdd x, bdd y)
    {
      return bdd_apply(x, y, bddop_xor);
    }
  
    bdd var(const std::string& s)
    {
      if (s == "false")
      {
        return false_();
      }
      else if (s == "true")
      {
        return true_();
      }
      else
      {
        int index = variables.index(s);
        if (index == -1)
        {
          int m = s.size() - 1;
          if (s[m] == '0')
          {
            index = variables0.index(s.substr(0, s.size() - 1));
          }
          if (s[m] == '1')
          {
            index = variables1.index(s.substr(0, s.size() - 1));
          }
        }
        return bdd_ithvar(index);
      }
    }
  };

  struct parse_result
  {
    std::stack<bdd> bdd_stack;
    bdd_builder builder;

    std::vector<std::string> variables;
    std::string bench_name;
    std::string logic_name;
    std::string source;
    std::string notes;
    bdd formula;
    bdd initial_state;
    bdd transition_relation;
    bdd final_state;

    bool formula_defined;
    bool initial_state_defined;
    bool transition_relation_defined;
    bool final_state_defined;

    parse_result()
    : formula_defined            (false),
      initial_state_defined      (false),
      transition_relation_defined(false),
      final_state_defined        (false)
    {}

    bool check() const
    {
      if (logic_name == "SAT")
      {
        if (!formula_defined)
        {
          throw std::runtime_error("no formula defined!");
        }
      }
      else if (logic_name == "REACH")
      {
        if (!initial_state_defined)
        {
          throw std::runtime_error("no initial state defined!");
        }
        if (!transition_relation_defined)
        {
          throw std::runtime_error("no transition relation defined!");
        }
        if (!final_state_defined)
        {
          throw std::runtime_error("no final state defined!");
        }
      }
    }

    void print() {
      std::cout << "  benchmark: " << bench_name << std::endl;
      std::cout << "  logic:     " << logic_name << std::endl;
      std::cout << "  source:    " << source << std::endl;
      //std::cout << "  variables: ";
      //for (std::vector<std::string>::const_iterator i = variables.begin(); i != variables.end(); ++i) {
      //  std::cout << *i << " ";
      //}
      //std::cout << std::endl;
      std::cout << "  notes:     " << notes << std::endl;
      //std::cout << "formula:             " << formula.s << std::endl;
      //std::cout << "initial_state:       " << initial_state.s << std::endl;
      //std::cout << "transition_relation: " << transition_relation.s << std::endl;
      //std::cout << "final_state:         " << final_state.s << std::endl;
    }

    void add_bdd(const std::string& s) {
      bdd_stack.push(builder.var(s));
    }

    void add_variable(const std::string& s) {
      variables.push_back(s);
    }

    void set_variables() {
    	if (logic_name == "SAT")
      {
        builder.variables = buddy::variable_set(variables);
        bdd_setvarnum(variables.size());
      }
      else // REACH
      {
      	buddy::variable_set v;
      	buddy::variable_set v0;  // contains 0 variables
      	buddy::variable_set v1;  // contains 1 variables
      	buddy::variable_set v01; // contains 0 + 1 variables
      	int index = 0;
        for (std::vector<std::string>::const_iterator i = variables.begin(); i != variables.end(); ++i)
      	{
      		v.insert(*i, index);
      		v0.insert(*i, index++);
      		v1.insert(*i, index++);
      	}
      	builder.variables = v;
      	builder.variables0 = v0;
      	builder.variables1 = v1;
        bdd_setvarnum(2*variables.size() + 1); // one scratch variable
      }
      variables.clear();
    }

    void set_bench_name(const std::string& s) {
      bench_name = s;
    }

    void set_logic_name(const std::string& s) {
      logic_name = s;
    }

    void set_source(const std::string& s) {
      source = s;
    }

    void set_notes(const std::string& s) {
      notes = s;
    }

    void set_formula() {
      formula_defined = true;
      formula = bdd_stack.top();
      bdd_stack.pop();
    }

    void set_initial_state() {
      initial_state_defined = true;
      initial_state = bdd_stack.top();
      bdd_stack.pop();
    }

    void set_transition_relation() {
      transition_relation_defined = true;
      transition_relation = bdd_stack.top();
      bdd_stack.pop();
    }

    void set_final_state() {
      final_state_defined = true;
      final_state = bdd_stack.top();
      bdd_stack.pop();
    }

    void and_() {
    	bdd y = bdd_stack.top();
    	bdd_stack.pop();
    	bdd x = bdd_stack.top();
    	bdd_stack.pop();
    	bdd_stack.push(builder.and_(x, y));
    }

    void or_() {
    	bdd y = bdd_stack.top();
    	bdd_stack.pop();
    	bdd x = bdd_stack.top();
    	bdd_stack.pop();
    	bdd_stack.push(builder.or_(x, y));
    }

    void xor_() {
    	bdd y = bdd_stack.top();
    	bdd_stack.pop();
    	bdd x = bdd_stack.top();
    	bdd_stack.pop();
    	bdd_stack.push(builder.xor_(x, y));
    }

    void implies() {
    	bdd y = bdd_stack.top();
    	bdd_stack.pop();
    	bdd x = bdd_stack.top();
    	bdd_stack.pop();
    	bdd_stack.push(builder.implies(x, y));
    }

    void iff() {
    	bdd y = bdd_stack.top();
    	bdd_stack.pop();
    	bdd x = bdd_stack.top();
    	bdd_stack.pop();
    	bdd_stack.push(builder.iff(x, y));
    }

    void not_() {
    	bdd x = bdd_stack.top();
    	bdd_stack.pop();
    	bdd_stack.push(builder.not_(x));
    }

    void if_then_else() {
    	bdd z = bdd_stack.top();
    	bdd_stack.pop();
    	bdd y = bdd_stack.top();
    	bdd_stack.pop();
    	bdd x = bdd_stack.top();
    	bdd_stack.pop();
    	// TODO
    	throw std::runtime_error("The operation if_then_else is not implemented!");
    	//bdd_stack.push(builder.if_then_else(x, y, z));
    }
  };

  template <typename Iterator, typename Skipper>
  struct formula_grammar : qi::grammar<Iterator, Skipper>
  {
      // for storing the parse results
      parse_result& data;

      formula_grammar(parse_result& data0)
        : formula_grammar::base_type(benchmark),
          data(data0)
      {
          using namespace qi::labels;
          using qi::lit;
          using qi::uint_;
          using qi::char_;
          using qi::alpha;
          using qi::digit;
          using qi::print;
          using qi::on_error;
          using qi::fail;
          using qi::lexeme;
          using qi::debug;
          using qi::eps;

          using phoenix::construct;
          using phoenix::val;

          simple_identifier =
            lexeme[alpha > *(alpha | digit | char_('.') | char_('\'') | char_('_'))]
            ;

          user_value_content =
              +(
                  lit("\\{")
                  | lit("\\}")
                  | (print - '{' - '}')
               )
              ;

          user_value =
              '{'
              > user_value_content                  [_val = qi::_1]
              > '}'
              ;

          numeral =
            char_('0')                               [_val = qi::_1]
            | char_("1-9")                           [_val = qi::_1]
            > *digit                                [_val += qi::_1]
            ;

          rational =
              numeral                                [_val = qi::_1]
              > char_('.')                          [_val += qi::_1]
              > *char_('0')                         [_val += qi::_1]
              > numeral                             [_val += qi::_1]
              ;

          indexed_identifier =
            simple_identifier                        [_val = qi::_1]
            >> char_('[')                            [_val += qi::_1]
            > (numeral                              [_val += qi::_1]
                % char_(':')                         [_val += qi::_1]
                )
            > char_(']')                            [_val += qi::_1]
            ;

          identifier =
            indexed_identifier                       [_val = qi::_1]
            | simple_identifier                      [_val = qi::_1]
            ;

          var =
            char_('?')                               [_val = qi::_1]
            > simple_identifier                     [_val += qi::_1]
            ;

          fvar =
            char_('$')                               [_val = qi::_1]
            > simple_identifier                     [_val += qi::_1]
            ;

          attribute =
              char_(':')                             [_val = qi::_1]
              > simple_identifier                   [_val += qi::_1]
              ;

          arith_symb =
              +char_("=<>&@#*/%|~+-")
              ;

          fun_symb =
              identifier                             [_val = qi::_1]
              | arith_symb                           [_val = qi::_1]
              ;

          pred_symb =
              identifier                             [_val = qi::_1]
              | arith_symb                           [_val = qi::_1]
              | lit("distinct")                      [_val = "distinct"]
              ;

          sort_symb =
              identifier
              ;

          annotation =
              attribute                              [_val = qi::_1]
              | attribute                            [_val = qi::_1]
              > user_value                          [_val += qi::_1]
              ;

          base_term =
              var                                    [boost::bind(&parse_result::add_bdd, &data, ::_1)]
//              | numeral
//              | rational
              | identifier                           [boost::bind(&parse_result::add_bdd, &data, ::_1)]
              ;

          an_term =
              base_term
              | '('
                > base_term
                > +annotation
                > ')'
//              | '(' > fun_symb > +an_term > *annotation > ')'
              | '('
                > lit("ite")
                > an_formula
                > an_term
                > an_term
                > *annotation
                > ')'
              ;

          //--- Formulas ---//

          prop_atom =
            lit("true")                              [boost::bind(&parse_result::add_bdd, &data, "true")]
            | lit("false")                           [boost::bind(&parse_result::add_bdd, &data, "false")]
            | fvar                                   [boost::bind(&parse_result::add_bdd, &data, ::_1)]
            | identifier                             [boost::bind(&parse_result::add_bdd, &data, ::_1)]
            ;

          an_atom =
              prop_atom
              | '('
                > prop_atom
                > +annotation
                > ')'
//              | '(' > pred_symb > +an_term > +annotation > ')'
              ;

          connective =
              lit("not")                  [_val = 0]
              | lit("implies")
              | lit("if_then_else")
              | lit("and")
              | lit("or")
              | lit("xor")
              | lit("iff")
              ;

          quant_symb =
              lit("exists")
              | lit("forall")
              ;

          quant_var =
              '('
              > var
              > sort_symb
              > ')'
              ;

          not_clause =
              char_('(')
              >> lit("not")
              > an_formula                                  [boost::bind(&parse_result::not_, &data)]
              > *annotation
              > ')'
              ;

          implies_clause =
              '('
              >> lit("implies")
              > an_formula
              > an_formula                                  [boost::bind(&parse_result::implies, &data)]
              > *annotation
              > ')'
              ;

          if_then_else_clause =
              '('
              >> lit("if_then_else")
              > an_formula
              > an_formula
              > an_formula                                  [boost::bind(&parse_result::if_then_else, &data)]
              > *annotation
              > ')'
              ;

          and_clause =
              '('
              >> lit("and")
              > an_formula
              > *an_formula                                [boost::bind(&parse_result::and_, &data)]
              > *annotation
              > ')'
              ;

          or_clause =
              '('
              >> lit("or")
              > an_formula
              > *an_formula                                [boost::bind(&parse_result::or_, &data)]
              > *annotation
              > ')'
              ;

          xor_clause =
              '('
              >> lit("xor")
              > an_formula
              > *an_formula                                [boost::bind(&parse_result::xor_, &data)]
              > *annotation
              > ')'
              ;

          iff_clause =
              '('
              >> lit("iff")
              > an_formula
              > an_formula                                [boost::bind(&parse_result::iff, &data)]
              > *annotation
              > ')'
              ;

          an_formula =
              not_clause
              | implies_clause
              | if_then_else_clause
              | and_clause
              | or_clause
              | xor_clause
              | iff_clause
              | an_atom
              ;

          //--- Theories ---//

          string_content =
              +(
                  lit("\\\"")
                  | (print - '"')
               )
              ;

          string_ =
          	'"'
          	> string_content                       [_val = qi::_1]
          	> '"'
          	;

          fun_symb_decl =
          	  '(' >> fun_symb > +sort_symb > *annotation > ')'
          	| '(' >> numeral > sort_symb > *annotation > ')'
          	| '(' >> rational > sort_symb > *annotation > ')'
          	;

          pred_symb_decl =
          	'(' > pred_symb > *sort_symb > *annotation > ')'
          	;

          theory_name =
          	identifier
          	;

          theory_attribute =
          	lit(":sorts") > '(' > +sort_symb > *annotation > ')'
          	| lit(":funs") > '(' > +fun_symb_decl > ')'
          	| lit(":preds") > '(' > +pred_symb_decl > ')'
          	| lit(":definition") > string_
          	| lit(":axioms") > '(' > +an_formula > ')'
          	| lit(":notes") > string_
          	| annotation
          	;

          theory =
          	'('
          	> lit("theory")
          	> theory_name
          	> +theory_attribute
          	> ')'
          	;

          //--- Logics ---//

          logic_name =
          	identifier
          	;

          logic_attribute =
          	lit(":theory") > theory_name
          	| lit(":language") > string_
          	| lit(":extensions") > string_
          	| lit(":notes") > string_
          	| annotation
          	;

          logic =
          	'('
            > lit("logic")
            > logic_name                                [boost::bind(&parse_result::set_logic_name, &data, ::_1)]
            > +logic_attribute
            > ')'
          	;

          //--- Benchmarks ---//

          status =
          	lit("sat")
          	| lit("unsat")
          	| lit("unknown")
          	;

          bench_name =
          	identifier
          	;

          bench_attribute =
            lit(":logic")
          	  > logic_name         [boost::bind(&parse_result::set_logic_name, &data, ::_1)]
          	//| lit(":assumption") > an_formula
            | lit(":variables")
              > char_('(')
              > +(
                  fvar                                  [boost::bind(&parse_result::add_variable, &data, ::_1)]
                  | identifier                          [boost::bind(&parse_result::add_variable, &data, ::_1)]
                  )
              > char_(')')                             [boost::bind(&parse_result::set_variables, &data)]
          	| lit(":formula")             > an_formula [boost::bind(&parse_result::set_formula, &data)]
            | lit(":initial_state")       > an_formula [boost::bind(&parse_result::set_initial_state, &data)]
            | lit(":transition_relation") > an_formula [boost::bind(&parse_result::set_transition_relation, &data)]
            | lit(":final_state")         > an_formula [boost::bind(&parse_result::set_final_state, &data)]
          	//| lit(":status") > status
          	//| lit(":extrasorts") > '(' > +sort_symb > ')'
          	//| lit(":extrafuns") > '(' > +fun_symb_decl > ')'
          	//| lit(":extrapreds") > '(' > +pred_symb_decl > ')'
          	| lit(":notes") > string_
          	| annotation
          	;

          benchmark =
          	'('
            > lit("benchmark")
            > bench_name            [boost::bind(&parse_result::set_bench_name, &data, ::_1)]
            > +bench_attribute
            > ')'
          	;

          simple_identifier  .name("simple_identifier");
          user_value_content .name("user_value_content");
          user_value         .name("user_value");
          numeral            .name("numeral");
          rational           .name("rational");
          indexed_identifier .name("indexed_identifier");
          identifier         .name("identifier");
          var                .name("var");
          fvar               .name("fvar");
          attribute          .name("attribute");
          arith_symb         .name("arith_symb");
          fun_symb           .name("fun_symb");
          pred_symb          .name("pred_symb");
          sort_symb          .name("sort_symb");
          annotation         .name("annotation");
          connective         .name("connective");
          quant_symb         .name("quant_symb");
          quant_var          .name("quant_var");
          string_content     .name("string_content");
          string_            .name("string_");
          theory_name        .name("theory_name");
          logic_name         .name("logic_name");
          status             .name("status");
          bench_name         .name("bench_name");
          base_term          .name("base_term");
          an_term            .name("an_term");
          prop_atom          .name("prop_atom");
          an_atom            .name("an_atom");
          an_formula         .name("an_formula");
          not_clause         .name("not_clause");
          implies_clause     .name("implies_clause");
          if_then_else_clause.name("if_then_else_clause");
          and_clause         .name("and_clause");
          or_clause          .name("or_clause");
          xor_clause         .name("xor_clause");
          iff_clause         .name("iff_clause");
          theory             .name("theory");
          theory_attribute   .name("theory_attribute");
          fun_symb_decl      .name("fun_symb_decl");
          pred_symb_decl     .name("pred_symb_decl");
          logic_attribute    .name("logic_attribute");
          logic              .name("logic");
          bench_attribute    .name("bench_attribute");
          benchmark          .name("benchmark");
      }

      qi::rule<Iterator, std::string(), Skipper>
          simple_identifier,
          user_value_content,
          user_value,
          numeral,
          rational,
          indexed_identifier,
          identifier,
          var,
          fvar,
          attribute,
          arith_symb,
          fun_symb,
          pred_symb,
          sort_symb,
          annotation,
          connective,
          quant_symb,
          quant_var,
          string_content,
          string_,
          theory_name,
          logic_name,
          status,
          bench_name
          ;

      qi::rule<Iterator, Skipper>
          base_term,
          an_term,
          prop_atom,
          an_atom,
          an_formula,
          not_clause,
          implies_clause,
          if_then_else_clause,
          and_clause,
          or_clause,
          xor_clause,
          iff_clause,
          theory,
          theory_attribute,
          fun_symb_decl,
          pred_symb_decl,
          logic_attribute,
          logic,
          bench_attribute,
          benchmark
          ;
  };

  template <typename Skipper>
  void parse(std::istream& input, const std::string& filename, Skipper const& skipper, parse_result& data)
  {
      // iterate over stream input
      typedef std::istreambuf_iterator<char> base_iterator_type;
      base_iterator_type in_begin(input);
  
      // convert input iterator to forward iterator, usable by spirit parser
      typedef boost::spirit::multi_pass<base_iterator_type> forward_iterator_type;
      forward_iterator_type fwd_begin = boost::spirit::make_default_multi_pass(in_begin);
      forward_iterator_type fwd_end;
  
      // wrap forward iterator with position iterator, to record the position
      typedef classic::position_iterator2<forward_iterator_type> pos_iterator_type;
      pos_iterator_type position_begin(fwd_begin, fwd_end, filename);
      pos_iterator_type position_end;

      formula_grammar<pos_iterator_type, Skipper> smt_grammar(data);
  
      // parse
      try
      {
        qi::phrase_parse(
          position_begin, position_end,
          smt_grammar,
          skipper
          );
      }
      catch(const qi::expectation_failure<pos_iterator_type>& e)
      {
        const classic::file_position_base<std::string>& pos = e.first.get_position();
        std::stringstream msg;
        msg <<
          "parse error at file " << pos.file <<
          " line " << pos.line << " column " << pos.column << std::endl <<
          "'" << e.first.get_currentline() << "'" << std::endl <<
          std::setw(pos.column) << " " << "^- here";
        throw std::runtime_error(msg.str());
      }
  }

} // namespace sat

#endif // SAT_PARSER_H
