var ServicesManagerModule = Class.create(
{
    initialize: function initialize()
    {
		this.siteId				= Portal.Modules.MyCloudPortal.currentSiteId;
		this.servicesList		= {};
		this.templates			= false;
		this.upgrading			= false;
		this.sudoEnabledMessage = 'This feature is not available when root access is enabled.';
		
		// doc root stuff (needs some extra state variables)
		this.newDocRoot			= '';
		this.newDocRootLocation	= '';
		
		// rails support stuff
		this.newSiteType		= '';
		
		// polling data
		this.pollCount			= 0;
		this.pollConfirmMessage	= '';		// the confirm message
		this.pollConfirmTitle	= '';		// the confirm dialog title
		this.pollWorkingMessage	= '';		// the message to show while the job is working
		this.pollFailMessage	= '';		// the message to show if the job fails
		this.pollMessage		= '';		// the actual message to send (xmlData)
		this.pollStartUrl		= '';		// the URL to hit to start the polling job (which will return the job location)
		this.pollUrl			= '';		// the URL to hit to poll for status of the current job
		this.pollCallBack		= false;	// call-back function to run when a job completes successfully
		this.pollStartCallBack	= false;	// call-back function to run when a job is started (after the user confirms the action)
		
		this.typeMap = {
			'rails':	'Ruby on Rails',
			'java': 	'Java',
			'web':		'HTML / Other',
			'php':		'PHP'
		}
		
    },
	
    finishInit: function finishInit()
    {
		this.extendSelf();
		this.registerObservers();
		
		this._init();
    },
	
    dispatchEvent: function dispatchEvent(msg)
    {
		if (msg.channel == '/portal/cloud/services/jobupdate' && msg.data.response == 'commit')
		{
			this.startPollingJob(msg);
		}
		else if (msg.channel == '/portal/cloud/services/jobupdate' && msg.data.response == 'describe')
		{
			this.handlePollingJobUpdate(msg);
		}
    },
	
    extendSelf: function extendSelf()
    {
		Object.extend(this, ServicesManagerControllers);
		Object.extend(this, ServicesManagerViews);
    },
	
    registerObservers: function registerObservers()
    {
		EventManager.subscribe('/portal/cloud/services/jobupdate', { channelHook: 'SiteServices' });
		
        Portal.Data.siteList.observe('siteServicesUpdated', function(siteId)
        {
			if(siteId == this.siteId)
			{
				this._init();
			}
        }.bind(this));
    }
});

var ServicesManagerControllers = 
{
    _init: function _init()
    {
		this.setServicesList();
		this.findNewestVersions();
		this.setServerTypes();
		
		// if the template's already been initialized, set the view data for it, otherwise,
		// run the initViews function to set everything up
		if(!this.setViewData('servicesList', this.servicesList))
		{
			this._initViews();
		}
    },
	
	setServicesList: function setServicesList()
	{
		this.servicesList = Portal.Data.siteList.sites.get(this.siteId).services.servicesList;
		
		// get rid of pulse
		this.servicesList.unset('pulse');
	},
	
	findNewestVersions: function findNewestVersions()
	{
		// interate over the services list
		this.servicesList.each(function(service)
        {
			var newestVersion = "0";
			
			// find the newest version, and store it locally
            service.value.availableVersions.each(function(item)
            {
				if(item > newestVersion)
				{
					newestVersion = item;
				}
            });
			
			// update the newest version in the object
			service.value.newestVersion = newestVersion;
			
			// TODO: Verify the above is a proper reference to the object, negating the need for the following line
			// this.servicesList.set(service.key, service.value);
        }.bind(this));
	},
	
	setServerTypes: function setServerTypes()
	{
        this.servicesList.each(function(service)
        {
			var serverType 	= "";
			var serverImage = "";
			
			switch(service.value.name)
			{
				case 'apache':
				case 'tomcat':
					serverType = 'web server';
					serverImage = 'img_icon_apache.png';
					break;
				case 'jaxer':
					serverType = 'ajax server';
					serverImage = 'img_icon_jaxer.png';
					break;
				case 'mysql':
					serverType = 'database server';
					serverImage = 'img_icon_mysql.png';
					break;
				case 'postfix':
					serverType = 'mail server';
					serverImage = 'img_icon_postfix.png';
					break;
			}
			
			service.value.serverType	= serverType;
			service.value.serverImage	= serverImage;
			service.value.domains		= {};
			
			if(service.key.toLowerCase() == 'apache')
			{
				service.value.domains = Portal.Data.siteList.sites.get(this.siteId).domains;
			}
			
			// TODO: Verify the above is a proper reference to the object, negating the need for the following line
			// this.servicesList.set(service.key, service.value);
        }.bind(this));
	},
	
	manipulateService: function manipulateService(service, action)
	{
		// validate the requested action
		var validActions = this.servicesList.get(service).commands;
		// reject an invalid action		
		if(validActions.indexOf(action) == -1)
		{
			return;
		}
		
		Portal.API.dialogs.confirm(
		{
			message: 'Are you sure you wish to ' + action + ' ' + service.capitalize() + '?',
			title: 'Confirm service ' + action,
			onConfirm: function()
			{
		        var msg = 
		        {
		            url: 'sites/' + this.siteId + '/services/' + service,
		            request: 'performAction',
		            parameters: {},
					actionType: 'POST',
		            returnChannel: Portal.Channels.services
		        }
				
				msg.parameters[action] = action;
				
				var serviceItem = this.servicesList.get(service);
				
				serviceItem.status = 'Restarting';
				this.servicesList.set(service, serviceItem);
				this.setViewData('servicesList', this.servicesList);
				
		        EventManager.publish('/portal/cloud/model', msg);
				
				// change the polling interval to 5 seconds until the requested service action completes
				Portal.Pollers.siteServices[this.siteId.toString()].changeIntervalLength(5);
			}.bind(this)
		});
	},
	
	upgradeService: function upgradeService(service, version)
	{
        Portal.API.dialogs.confirm(
        {
			message: 'Are you sure you wish to upgrade ' + service.capitalize() + ' to version ' + version + '?',
			title: 'Confirm service upgrade',
			onConfirm: function()
			{
				var msg = 
				{
					url: 'sites/' + this.siteId + '/services/' + service,
					request: 'performAction',
					parameters: { upgrade: version },
					actionType: 'POST',
					returnChannel: Portal.Channels.services
				}
				
				var serviceItem = this.servicesList.get(service);
				
				serviceItem.status = 'Upgrading';
				this.servicesList.set(service, serviceItem);
				this.setViewData('servicesList', this.servicesList);
				this.upgrading = true;
				
				EventManager.publish('/portal/cloud/model', msg);
						
				// change the polling interval to 10 seconds until the requested upgrade completes
				Portal.Pollers.siteServices[this.siteId.toString()].changeIntervalLength(10);
			}.bind(this)
		});
	},
	
	confirmPollingJob: function confirmPollingJob(skipConfirm)
	{
		// make sure a state has been set for skipConfirm
		if(typeof(skipConfirm) == 'undefined')
		{
			var skipConfirm = false;
		}
		
		// build the content for the "working" dialog
		var modalContent = '<div class="clean-yellow"><div class="bold line16 center top5"><img src="images_global/img_activity_deploy.gif" class="inline" align="absmiddle" /> ';
		modalContent += this.pollWorkingMessage + '</div><div class="size11 line14 top5 center"><em>This should only take a moment</em></div></div>';
		
		// set up the "working" dialog object, including the message to send once its opened
        this.modalJobDialog = new Control.Modal(modalContent, 
        {
            closeOnClick: 	false,
			className:		'modal',
			width:			300,
			overlayOpacity:	0.75,
			fade:			false,
			height:			null,
			iframeshim:		false,
			afterOpen: function()
			{
				// make sure this is reset
				this.pollCount = 0;
				
				// run the call-back
				if(typeof(this.pollStartCallBack) == 'function')
				{
					this.pollStartCallBack();
					this.pollStartCallBack = false;
				}
				
                EventManager.publish('/portal/cloud/model', 
                {
                    url: 			this.pollStartUrl,
					request: 		'commit',
					returnChannel:	'/portal/cloud/services/jobupdate',
					xmlData:		this.pollMessage
                });
			}.bind(this)
        });

		if(skipConfirm)
		{
			// we need to give the confirm dialog time to close before we open the "working" dialog
			var openModalJobDialog = function()
			{
				this.modalJobDialog.open();
			}.bind(this)
			
			// open that dialog
			setTimeout(openModalJobDialog, 1000);
		}
		else
		{
			// confirm the user's request		
	        Portal.API.dialogs.confirm(
	        {
	            message: '<div class="line16 size12">' + this.pollConfirmMessage + '</div>',
	            title: this.pollConfirmTitle,
	            onConfirm: function()
	            {
					// we need to give the confirm dialog time to close before we open the "working" dialog
					var openModalJobDialog = function()
					{
						this.modalJobDialog.open();
					}.bind(this)
					
					// open that dialog
					setTimeout(openModalJobDialog, 1000);
											
	            }.bind(this)
	        });
		}
	},
	
	startPollingJob: function startPollingJob(msg)
	{
		// make sure we actually got a location
		if(!('location' in msg.data) || msg.data.location == '')
		{
			this.modalJobDialog.close();
			this.modalJobDialog.destroy();
			
			var showError = function()
			{
				Portal.API.dialogs.alert(this.pollFailMessage, 'Error');
			}.bind(this);
			
			setTimeout(showError, 500);
			
			return;
		}
		
		// store the polling location
		this.pollUrl = msg.data.location.replace(Portal.Vars.siteManagerUrl, '');
		
        Portal.Pollers.register('siteServices', this.pollUrl, 
        {
            object: 		Portal.Modules.SiteServices,
			method: 		'handlePollingJobUpdate',
			frequency: 		5,
			args: 			[],
			notifyEvent:	'pollingJobUpdated'
        });
	},
	
	handlePollingJobUpdate: function handlePollingJobUpdate(msg)
	{
		if(!msg)
		{
			// request new info about the job
            EventManager.publish('/portal/cloud/model', 
            {
                url: 			this.pollUrl,
                request:		'describe',
                returnChannel:	'/portal/cloud/services/jobupdate'
            });
		}
		else
		{
			// the job was successful, so we can stop everything
			if(msg.data.xmlData.job.success != '' && msg.data.xmlData.job.success == true)
			{
				this.modalJobDialog.close();
				this.modalJobDialog.destroy();
				
				this.notify('pollingJobUpdated', true);
				
				if(typeof(this.pollCallBack) == 'function')
				{
					this.pollCallBack();
					this.pollCallBack = false;
				}
				
				Portal.Data.siteList.fetchSiteServices(this.siteId);
			}
			// the job explicitly failed
			else if (msg.data.xmlData.job.success == false)
			{
				this.modalJobDialog.close();
				this.modalJobDialog.destroy();
				
				this.notify('pollingJobUpdated', true);
				
				var showErrorDialog = function()
				{
					Portal.API.dialogs.alert(this.pollFailMessage, 'Error');
				}.bind(this)
				
				Portal.Data.siteList.fetchSiteServices(this.siteId);
				
				setTimeout(showErrorDialog, 750);
			}
			// the job is in progress
			else
			{
				this.pollCount ++;
				
				// the job is taking too long, it has most likely failed
				if(this.pollCount > 3)
				{
					this.modalJobDialog.close();
					this.modalJobDialog.destroy();
					
					this.notify('pollingJobUpdated', true);
					
					var showErrorDialog = function()
					{
						Portal.API.dialogs.alert(this.pollFailMessage, 'Error');
					}.bind(this)
					
					Portal.Data.siteList.fetchSiteServices(this.siteId);
					
					setTimeout(showErrorDialog, 750);
				}
				else
				{
					this.notify('pollingJobUpdated');
				}
			}
		}
	},
	
	setJaxerEnabledState: function setJaxerEnabledState(state)
	{
		if(state == 'enabled')
		{
			this.pollConfirmMessage	= 'Are you sure you wish to enable Jaxer?';
			this.pollConfirmTitle	= 'Confirm Jaxer enable';
			this.pollWorkingMessage	= 'Please wait, we\'re enabling Jaxer.';
			this.pollFailMessage	= 'There was an error enabling Jaxer or the process timed out.  Please try again later, or contact support if you continue to receive this error.';
			
			this.pollStartUrl		= 'sites/' + this.siteId + '/services/jaxer?enable=true';
			this.pollMessage		= { message: { enabled: true } };
		}
		else
		{
			this.pollConfirmMessage	= 'Are you sure you wish to disable Jaxer?';
			this.pollConfirmTitle	= 'Confirm Jaxer disabling';
			this.pollWorkingMessage	= 'Please wait, we\'re disabling Jaxer.';
			this.pollFailMessage	= 'There was an error disabling Jaxer or the process timed out.  Please try again later, or contact support if you continue to receive this error.';
			
			this.pollStartUrl		= 'sites/' + this.siteId + '/services/jaxer?disable=true';
			this.pollMessage		= { message: { enabled: false } };
		}
		
		this.confirmPollingJob();
	},
	
	startConfigRestore: function startConfigRestore(config)
	{
		this.pollConfirmMessage	= 'Are you sure you wish to restore ' + config + ' to its original state?';
		this.pollConfirmTitle	= 'Confirm ' + config + ' restore';
		this.pollWorkingMessage	= 'Please wait, we\'re restoring ' + config + '.';
		this.pollFailMessage	= 'There was an error restoring ' + config + ' or the process timed out.  Please try again later, or contact support if you continue to receive this error.';
		
		this.pollStartUrl		= 'sites/' + this.siteId + '/restore';
		this.pollMessage		= { restore: { resource_name: config } };
		
		this.confirmPollingJob();
	},
	
	setRailsEnabledState: function setRailsEnabledState(state)
	{
		if(state == 'enabled')
		{
			this.pollConfirmMessage	= 'Are you sure you wish to enable Ruby on Rails support for this site?';
			this.pollConfirmTitle	= 'Confirm Ruby on Rails enable';
			this.pollWorkingMessage	= 'Please wait, we\'re enabling Ruby on Rails support.';
			this.pollFailMessage	= 'There was an error enabling Ruby on Rails or the process timed out.  Please try again later, or contact support if you continue to receive this error.';
			
			this.pollStartUrl		= 'sites/' + this.siteId 
			this.pollMessage		= { site: { type: 'rails' } };
			
			this.pollCallBack = function() { Portal.Modules.MyCloudPortal.currentSite.type = 'rails' }
		}
		else
		{
			this.pollConfirmMessage	= 'Are you sure you wish to disable Ruby on Rails support for this site?';
			this.pollConfirmTitle	= 'Confirm Ruby on Rails disable';
			this.pollWorkingMessage	= 'Please wait, we\'re disabling Ruby on Rails support.';
			this.pollFailMessage	= 'There was an error disabling Ruby on Rails or the process timed out.  Please try again later, or contact support if you continue to receive this error.';
			
			this.pollStartUrl		= 'sites/' + this.siteId 
			this.pollMessage		= { site: { type: 'web' } };
			
			this.pollCallBack = function() { Portal.Modules.MyCloudPortal.currentSite.type = 'web' }
		}
		
		this.confirmPollingJob();
	},
	
	setSiteType: function setSiteType(type)
	{
		if(type == '---' || type == Portal.Modules.MyCloudPortal.currentSite.type)
		{
			return;
		}
		
		var typeMap = this.typeMap;
		
		this.pollConfirmMessage	= 'Are you sure you wish to set this site\'s platform to "' + typeMap[type] + '"?';
		this.pollConfirmTitle	= 'Confirm site platform change';
		this.pollWorkingMessage	= 'Please wait, we\'re changing the site platform to "' + typeMap[type] + '".';
		this.pollFailMessage	= 'There was an error setting the site platform or the process timed out.  Please try again later, or contact support if you continue to receive this error.';
		
		this.pollStartUrl		= 'sites/' + this.siteId 
		this.pollMessage		= { site: { 'type': type } };
		
		this.pollCallBack = function() { 
			try 
			{
				Portal.Modules.MyCloudPortal.currentSite.type = type;
				
				// update the UI on the overview page, since the deploy / change project menus may need to change depending on the site type
				Portal.Modules.MyCloudPortal.templates.overview.render();
				Portal.Modules.MyCloudPortal.templates.servicesSummary.render();
				Portal.Modules.MyCloudPortal.templates.siteMetaData.render(); 
				
				
				var showComplete = function(){
					Portal.API.dialogs.alert('Your site platform has been changed.  Please allow up to a minute for all changes to be reflected.', 'Site Platform Changed');
					this.templates.main.render();
				}.bind(this);
				
				setTimeout(showComplete, 1000);
				
			} catch (e)
			{
				// nothing to display, we just want to prevent UI errors				
			}
		}.bind(this);
		
		this.confirmPollingJob();
	},
	
	startDocRootChange: function startDocRootChange(domain)
	{
		var newDocRoot 	= '';
		var domainId 	= 0;
		
		switch(domain)
		{
			case 'public':
				newDocRoot 	= $F('docroot_public');
				domainId 	= Portal.Data.siteList.sites.get(this.siteId).domains.publicInfo.get('id');
				
				// store the state vars
				this.newDocRoot 		= newDocRoot;
				this.newDocRootLocation = 'publicInfo';
				
				break;
			case 'staging':
				newDocRoot	= $F('docroot_staging');
				domainId 	= Portal.Data.siteList.sites.get(this.siteId).domains.stagingInfo.get('id');
				
				// store the state vars
				this.newDocRoot 		= newDocRoot;
				this.newDocRootLocation = 'stagingInfo';
				
				break;
		}
		
		// do some basic validation
		if (newDocRoot.indexOf('..') != -1 || newDocRoot.indexOf('//') != -1 || newDocRoot.indexOf('./') != -1) 
		{
			Portal.API.dialogs.alert('Please enter a valid document root', 'Error');
			return;		
		}
		
		// strip off trailing slashes
		if(newDocRoot.endsWith('/'))
		{
			newDocRoot = newDocRoot.substr(0, (newDocRoot.length - 1));
		}
		
		// we store this before we append with the full path
		this.pollConfirmMessage	= 'Are you sure you wish to change the document root for ' + domain + ' to "' + newDocRoot + '"?';
		
		// now, build the full doc root, catching any manipulation errors that occur
		try 
		{
			newDocRoot = Portal.Data.siteList.sites.get(this.siteId).domains[domain + 'Info'].get('documentRootPrefix') + newDocRoot;
		}
		catch (e)
		{
			console.warn(e);
		}
		
		this.pollConfirmTitle	= 'Confirm document root change';
		this.pollWorkingMessage	= 'Please wait, we\'re changing the ' + domain + ' document root.';
		this.pollFailMessage	= 'There was an error changing the document root for ' + domain + ' or the process timed out.  Please try again later, or contact support if you continue to receive this error.';
		
		this.pollStartUrl		= 'domains/' + domainId;
		this.pollMessage		= { message: { domain: { document_root: newDocRoot } } };
		
		// shows the updating message
		this.pollStartCallBack	= function()
		{
			$('docroot_' + domain + '_change_cage').update('<strong>' + domain.capitalize() + ' Document Root</strong><br />Updating doc root...');
		}
		
		// updates the document root in the site model before the next poll interval (so the changes reflect immediately)
		this.pollCallBack = function()
		{
			try 
			{
				Portal.Data.siteList.sites.get(this.siteId).domains[this.newDocRootLocation].set('documentRoot', this.newDocRoot);
			}
			catch (e)
			{
				console.warn(e);
			}
		}.bind(this);
		
		this.confirmPollingJob();
	}
};

var ServicesManagerViews = 
{
    _initViews: function _initViews()
    {
		if(!$('site_services'))
		{
			return;
		}
		
        View.load(
        {
            main: 
            {
                file: 'portlets/cloud/templates/services/services_main.html',
				binding: function()
				{
					return $('site_services');
				},
				behaviors: function()
				{
					this.siteId = Portal.Modules.MyCloudPortal.currentSite.id;
					
					if($('enableJaxerButton'))
					{
						// observe the enable jaxer button
	                    $('enableJaxerButton').observe('click', function(event)
	                    {
							var element = Event.element(event).up();
							
							if(element.hasClassName('enable'))
							{
								this.setJaxerEnabledState('enabled');
							}
							else
							{
								this.setJaxerEnabledState('disabled');
							}
							
	                    }.bind(this));
					}
					
					// if this element isn't present, none of them are
					if($('showPublicDocRootChange'))
					{
						// the change public doc root link
	                    $('showPublicDocRootChange').observe('click', function()
	                    {
							if(Portal.Modules.MyCloudPortal.currentSite.sudoEnabled == true)
							{
								Portal.API.dialogs.alert(Portal.Modules.SiteServices.sudoEnabledMessage);
							}
							else
							{
								$('docroot_public_cage').hide(); 
								$('docroot_public_change_cage').show();
							}
	                    });
						
						// the cancel change public doc root
	                    $('hidePublicDocRootChange').observe('click', function()
	                    {
							$('docroot_public_cage').show();
							$('docroot_public_change_cage').hide();
							$('docroot_public').value=$('hidePublicDocRootChange').rel;
	                    });
						
						// the change staging doc root link
	                    $('showStagingDocRootChange').observe('click', function()
	                    {
							if (Portal.Modules.MyCloudPortal.currentSite.sudoEnabled == true) 
							{
								Portal.API.dialogs.alert(Portal.Modules.SiteServices.sudoEnabledMessage);
							}
							else 
							{
								$('docroot_staging_cage').hide();
								$('docroot_staging_change_cage').show();
							}
	                    });
						
						// the cancel staging doc root change
						$('hideStagingDocRootChange').observe('click', function()
	                    {
							$('docroot_staging_cage').show();
							$('docroot_staging_change_cage').hide();
							$('docroot_staging').value=$('hideStagingDocRootChange').rel;
	                    });
						
						// save doc root change link
	                    $('changePublicDocRoot').observe('click', function()
	                    {
							this.startDocRootChange('public');
	                    }.bind(this));
						
						// save staging doc root change link
	                    $('changeStagingDocRoot').observe('click', function()
	                    {
							this.startDocRootChange('staging');
	                    }.bind(this));
					}
										
					if($('enableRails'))
					{
						// enable / disable rails link
						$('enableRails').observe('click', function()
						{
							if(Portal.Modules.MyCloudPortal.currentSite.sudoEnabled == true)
							{
								Portal.API.dialogs.alert(Portal.Modules.SiteServices.sudoEnabledMessage);
							}
							else
							{
								this.setRailsEnabledState($('enableRails').rel);
							}	
						}.bind(this));
					}
					
				}.bind(this),
				scope:
				{
					servicesList:	this.servicesList,
					siteId:			this.siteId
				}
            },
			serviceItem:
			{
				file: 'portlets/cloud/templates/services/service_item.html'
			}
        }, 
		function(templates)
        {
        	this.templates = templates;
			this.templates.main.set('serviceItem', this.templates.serviceItem);
        }.bind(this));
    },
	
	setViewData: function setViewData(key, value)
	{
		if(typeof(this.templates.main) != 'undefined')
		{
			this.templates.main.set(key, value);
			return true;
		}
		
		return false;
	}
};

Object.Event.extend(ServicesManagerModule);