/*
 -----------------------------------------------------
|          DEPLOYMENT WIZARD STEP PROCESSORS          |
 -----------------------------------------------------
*/
var DeploymentWizardProcessor = 
{
    /**
     * Bootstraps the processing of the current step in the wizard
     *
     * Can also be forced to go to a specific step afterwards, as triggered by clicks on the
     * breadcrumbs.
     *
     * @param {Boolean} forceStep
     */
    processCurrentStep: function(forceStep)
    {
        // make sure we should actually process the current step
        if (this.changingState || this.messagePending || this.validatingSiteName) 
        {
            return;
        }
        
        // store the forced next step
        if (forceStep && !isNaN(forceStep)) 
        {
            this.nextStepOverride = forceStep;
        }
        
        // set up some vars
        var currentStepOk = false;
        this.currentStep = parseInt(this.currentStep);
		
		// since step 5 will hop forward prematurely due to message returns, and 
		// this function is only ever called by user interaction, we can safely
		// set this flag to false here and allow the wizard to proceed forward
		if(this.currentStep == 5 && this.stopAdvance)
		{
			this.stopAdvance = false;
		}
        
        // bootstrap the processing
        switch (this.currentStep)
        {
            case 1:
                currentStepOk = this.processStep1();
                break;
            case 2:
                Object.extend(this.newSiteData.hostingData, Portal.Modules.WizardConfigurator.fetchUsageOptions());
                currentStepOk = true;
                break;
            case 3:
                Object.extend(this.newSiteData.hostingData, Portal.Modules.WizardConfigurator.fetchOptionIds());
                currentStepOk = true;
                break;
            case 4:
                currentStepOk = this.processStep4();
                break;
            case 5:
                currentStepOk = this.processStep5();
                break;
            case 6:
                currentStepOk = this.processStep6();
                break;
            case 7:
                currentStepOk = true;
                break;
            default:
                currentStepOk = false;
        }
		
        // if everything's OK, and we can move on to the next step, do so 
        if ((!forceStep || isNaN(forceStep)) && currentStepOk) 
        {
            this.doNextStep();
        }
        // otherwise move to a specific step
        else if (!isNaN(forceStep) && currentStepOk) 
        {
            this.currentStep = parseInt(forceStep);
			this.sendTrackingData();
            this.buildBreadCrumbs();
            this.loadWizardStepContent();
        }
    },
    
    /**
     * Either moves to the next step or triggers deployment
     *
     * @param {Boolean} stepOverride
     */
    doNextStep: function doNextStep(stepOverride)
    {
        this.changingState = true;
        
        if (stepOverride) 
        {
            this.currentStep = parseInt(stepOverride);
            this.nextStepOverride = '';
        }
        else 
        {
            this.currentStep++;
        }
        
        this.currentStepRunning = (this.currentStepRunning < this.currentStep) ? this.currentStep : this.currentStepRunning;
		
		if(this.isTrial)
		{
			if(this.currentStep > 1 && this.currentStep < 4)
			{
				this.currentStep = 4;
			}
			
			if(this.currentStep > 4 && this.currentStep < 6)
			{
				this.currentStep = 6
			}
			
			if(this.currentStep > 6)
			{
				if (this.processStep6()) 
				{
					this.doDeployment();
				}
				else
				{
					this.changingState = false;
					this.currentStep = 6;
				}
				return;
			}
		}
		else if (this.currentStep > this.totalSteps) 
        {
            this.doDeployment();
			return;
        }
		
		this.sendTrackingData();
        this.buildBreadCrumbs();
        this.loadWizardStepContent();
    },
    
    /**
     * Moves the wizard back 1 step
     *
     */
    doPreviousStep: function doPreviousStep()
    {
        if (this.changingState || this.messagePending) 
        {
            return;
        }
        
        this.currentStep = (this.currentStep - 1 < 1) ? 1 : this.currentStep - 1;
		
		if(this.isTrial)
		{
			if(this.currentStep > 1 && this.currentStep < 4)
			{
				this.validatingSiteName = true;
				this.currentStep = 1;
			}
			
			if(this.currentStep > 4 && this.currentStep < 6)
			{
				this.currentStep = 4
			}
			
		}
        
		this.sendTrackingData();
        this.buildBreadCrumbs();
        this.loadWizardStepContent();
    },
    
    /**
     * Processes the first step of the deployment process
     *
     * There are really two parts of this function:
     *  - The initial validation and error checking
     *  - The message callback for verifying the sitename validity server-side
     *
     * @param {Object} msg
     */
    processStep1: function processStep1(msg)
    {
        if (this.validatingSiteName) 
        {
            return false;
        }
        
        // some utility functions we'll re-use throughout...
        var fixDisplay = function()
        {
            $('site_name_form').setStyle(
            {
                height: $('site_name_form').getHeight() + 0 + 'px'
            });
        }
        
        var showError = function()
        {
            new Effect.Appear('site_name_error', 
            {
                duration: 0.25,
                queue: 'end',
                afterFinish: function()
                {
                    fixDisplay();
                }
            });
        }
        
        var hideError = function()
        {
            new Effect.Fade('site_name_error', 
            {
                duration: 0.25,
                queue: 'front',
                afterFinish: function()
                {
                    fixDisplay();
                }
            });
        }
        
        // hide the error display before we get rolling...
        if ($('site_name_error') && $('site_name_error').style.display != 'none') 
        {
            hideError();
        }
        
        
        // initial validation
        if (!msg) 
        {
            if (this.messagePending) 
            {
                return;
            }
            
            // if the site name was valid from pre-validation, and the user didn't change it, no need to re-validate
            if (this.siteNameValid == true && $F('site_name') == this.newSiteData.siteName) 
            {
                if (this.nextStepOverride != '') 
                {
                    this.doNextStep(this.nextStepOverride);
                    return false;
                }
                
                this.doNextStep();
                return false;
            }
            
            // regex testing of the site name validity
            var siteName = $F('site_name');
            var re = /^[a-z0-9]((([\-a-z0-9]?[a-z0-9])*)|([a-z0-9]*[\-a-z0-9]?[a-z0-9]))$/i;
            if (siteName.length < 2 || !siteName.match(re)) 
            {
                $('site_name_error').update('Please enter a valid site name');
                
                showError();
                
                return false;
            }
            
            // everything's good, let's ask the server if the sitename is valid...
            this.newSiteData.siteName = $F('site_name');
            
            $('site_name').disable();
            
            new Effect.Appear('site_name_activity', 
            {
                duration: 0.25,
                queue: 'end',
                afterFinish: function()
                {
                    fixDisplay();
                    
                    EventManager.publish('/portal/cloud/deploy', 
                    {
                        request: 'checkSiteName',
                        siteName: this.newSiteData.siteName + '.aptanacloud.com'
                    });
                    
                    this.messagePending = true;
                }.bind(this)
            });
            
            return false;
        }
        // callback area of the function
        else 
        {
            this.messagePending = false;
            
            // handle errors
            if (msg.data.siteName == null && $('site_name_error')) 
            {
                $('site_name').enable();
                new Effect.Fade('site_name_activity', 
                {
                    duration: 0.25,
                    queue: 'front',
                    afterFinish: function()
                    {
                        fixDisplay();
						$('site_name').focus();
                    }
                });
                
                this.newSiteData.siteName = '';
                
                if (msg.data.errors.length == 0) 
                {
                    $('site_name_error').update(Portal.API.templates.parseErrors('An unknown error has occurred.  Please try again in a few minutes.  If the problem persists, please <a href="javascript: Portal.API.utils.openUrl(\'http://support.aptana.com/asap\');">contact us</a> for support.', []));
                }
                else
				{
	                $('site_name_error').update(Portal.API.templates.parseErrors('There is a problem with your site name.', msg.data.errors));
				}
				
                $('site_name').focus();
                
                showError();
                
                return false;
            }
            
            // everything's good, move on to the next step
            this.siteNameValid = true;
            
            if (this.nextStepOverride != '') 
            {
                this.doNextStep(this.nextStepOverride);
                return false;
            }
            
            this.doNextStep();
            return false;
        }
    },
    
	/**
	 * Processes the user either logging in or creating an account.
	 * 
	 * Similar to the functionality for step 1, this function functions both as a processor and a 
	 * call-back.  The first pass does some validation, then figures out which helper function to 
	 * call (depending on login / user create).  On the call-back, we marshall the message to the 
	 * appropriate helper function.
	 * 
	 * We also set / unset the "remember me" preferences in this function depending on the user's 
	 * selection
	 * 
	 * @param {Object} msg
	 */
    processStep4: function processStep4(msg)
    {
        if (!msg) 
        {
            if (this.messagePending) 
            {
                return;
            }
            
            $('wizard_username_error').hide();
            $('wizard_password_error').hide();
            $('wizard_email_error').hide();
            
            $('wizard_email_error').update('Please enter a valid email address');
            $('wizard_password_error').update('Please enter a valid password');
			
            new Effect.Fade('wizard_user_error', 
            {
                duration: 0.25,
                queue: 'front'
            });
            
            // existing user
            if (!$('user_current') || ($('user_current').checked == true || Portal.Modules.AptanaIdManager.isLoggedIn)) 
            {
                this.purchaseCredentialsSaved = true;
                
                // skip validation if already verified
                if (this.usernameValid) 
                {
                    if (this.nextStepOverride != '') 
                    {
                        this.doNextStep(this.nextStepOverride);
                        return false;
                    }
                    
                    this.doNextStep();
                    return false;
                }
                
                return this.validateCurrentUser();
            }
            else 
            {
                return this.createNewUser();
            }
        }
        else 
        {
            if (this.usernameAction == 'current') 
            {
                return this.validateCurrentUser(msg);
            }
            else if (this.usernameAction == 'new') 
            {
                return this.createNewUser(msg);
            }
        }
        
        return false;
    },
    
	/**
	 * Processes the billing page.
	 * 
	 * This function is actually pretty straight-forward... all we do here is either
	 * serialize the new account form, or grab the selected billing account id and 
	 * store it in the deployment data.  The tough stuff for billing is in the display
	 * logic in the template and in the view functionality above.
	 * 
	 */
    processStep5: function processStep5(msg)
    {
		// prevent this screen from hopping forward prematurely
		if (this.stopAdvance)
		{
			return;
		}
		
		if(!msg)
		{
			try
			{
		        var formValid = this.validator.validate();
		        $('expiry_error').hide();
				
		        if (!formValid) 
		        {
		            return false;
		        }
		        
		        // collect the billing data...
		        if (Portal.Data.currentUser.billingAccounts.size() > 0) 
		        {
		            var selectedAccount = 0;
		            
		            $$('input.billingAccountSelect').each(function(element)
		            {
		                if ($(element).checked) 
		                {
		                    selectedAccount = element.value;
		                }
		            });
		            
		            this.newSiteData.billingAccountId = selectedAccount;
		            
		            if (selectedAccount == 0) 
		            {
						// check that the date is valid...
						var d = new Date();
						var currentMonth = d.getMonth();
						var currentYear = d.getFullYear();
	
						var selectedMonth = $F('wizard_card_expiry_month');
						
						if(selectedMonth.substr(0, 1) == '0')
						{
							selectedMonth = selectedMonth.substr(1);
						}
						
						if((parseInt(selectedMonth) <= (currentMonth+1) && parseInt(('20' + $F('wizard_card_expiry_year'))) <= currentYear) || parseInt(('20' + $F('wizard_card_expiry_year'))) < currentYear)
						{
							$('expiry_error').show();
							return false;
						}
						
		                var billingInfo = $('wizard_billingInfo').serialize(
		                {
		                    hash: true
		                });
		                this.newSiteData.newBillingData = billingInfo;
		                this.newSiteData.createNewAccount = true;
		            }
		            else 
		            {
		                this.newSiteData.createNewAccount = false;
		            }
		        }
		        else 
		        {
		            var billingInfo = $('wizard_billingInfo').serialize(
		            {
		                hash: true
		            });
		            this.newSiteData.newBillingData = billingInfo;
		            this.newSiteData.createNewAccount = true;
		        }
				
				if(this.messagePending || this.couponCodeApply)
				{
					return false;
				}
				
				
				if($F('coupon_code') != '' && !this.newSiteData.couponValid)
				{
					this.newSiteData.couponCode = $F('coupon_code').escapeHTML();
					
					$('coupon_code_error').hide();
					$('coupon_code_form').hide();
					$('coupon_code_processing').show();
					
					this.validateCouponCode = true;
					
	                this.fetchOrderPreview();
					
					return false;	
				}
		        
		        return true;
			}
			catch(e)
			{
				console.warn(e);
				return false;
			}
		}
		else
		{
			if(this.newSiteData.couponCode != '' && msg.data.xmlData.quote.coupons.length == 0)
			{
				this.couponCodeApply = false;
				this.messagePending = false;
				this.newSiteData.couponCode = '';
				this.newSiteData.couponValid = false;
				
				$('coupon_code_error').show();
				$('coupon_code_error').update('The coupon code you entered is invalid.  Please try again.');
				$('coupon_code_form').show();
				$('coupon_code_processing').hide();
				$('coupon_code').focus();
				
				return false;
			}
			else if (this.newSiteData.couponCode != '' && msg.data.xmlData.quote.coupons[0].coupon.errors.length > 0) 
			{
				this.couponCodeApply = false;
				this.messagePending = false;
				this.newSiteData.couponCode = '';
				this.newSiteData.couponValid = false;
				
				$('coupon_code_error').show();
				$('coupon_code_error').update(msg.data.xmlData.quote.coupons[0].coupon.errors[0].error);
				$('coupon_code_form').show();
				$('coupon_code_processing').hide();
				$('coupon_code').focus();
				
				return false;
			}
			else
			{
				this.newSiteData.couponValid = true;
			}
			
			if (this.nextStepOverride != '') 
            {
                this.doNextStep(this.nextStepOverride);
                return false;
            }
            
            this.doNextStep();
            return false;
		}
    },
    
	/**
	 * Makes sure the user agrees to the terms... easy :)
	 */
    processStep6: function processStep6()
    {
        if ($('agree_yes').checked != true) 
        {
            new Effect.Appear('terms_error', 
            {
                duration: 0.25,
                queue: 'end',
                afterFinish: function()
                {
                    $('terms').setStyle(
                    {
                        height: $('terms').getHeight() + 'px'
                    });
                }
            });
            return false;
        }
        
        new Effect.Fade('terms_error', 
        {
            duration: 0.25,
            queue: 'end',
            afterFinish: function()
            {
                $('terms').setStyle(
                {
                    height: $('terms').getHeight() + 'px'
                });
                
                this.newSiteData.agreeToTerms = true;
            }.bind(this)
        });
        
        return true;
    }
};