var AlertsManagerModule = Class.create(
{
	initialize: function initialize()
	{
		this.templates			= {};
		this.STUDIO_CHANNEL_ID	= 4;
		this.userStudioChannel	= 0;
		this.expandedCategories	= [];
		this.messagePending		= false;
	},
	
	finishInit: function finishInit()
	{
		this.extendSelf();
		this.registerObservers();
		
		this._init();
	},
	
	dispatchEvent: function dispatchEvent(msg)
	{
		this.messagePending = false;
		
		if(msg.data.response == 'commit' && msg.channel == '/portal/cloud/channels')
		{
			this.handleAddChannel(msg);
		}
		else if (msg.data.response == 'delete' && msg.channel == '/portal/cloud/channels')
		{
			Portal.Data.currentUser.fetchUserChannels();
		}
		else if (msg.data.response == 'commit' && msg.channel == '/portal/cloud/alerts')
		{
			Portal.Data.currentUser.fetchChannelSubs();
		}
		else if (msg.data.response == 'delete' && msg.channel == '/portal/cloud/alerts')
		{
			Portal.Data.currentUser.fetchChannelSubs();
		}
	},
	
	extendSelf: function extendSelf()
	{
		Object.extend(this, AlertsManagerControllers);
		Object.extend(this, AlertsManagerViews);
	},
	
	registerObservers: function registerObservers()
	{
		EventManager.subscribe('/portal/cloud/channels', { channelHook: 'AlertsManager' });
		EventManager.subscribe('/portal/cloud/alerts', { channelHook: 'AlertsManager' });
		
        Portal.Data.currentUser.observe(['userChannelsLoaded', 'subscriptionsLoaded'], function()
        {
			this._init();
        }.bind(this));
	}
});

var AlertsManagerControllers = 
{
	_init: function _init()
	{
		this.injectCategoryImages();
		this.findUserStudioChannel();
		this.calcCategoryUsage();
		
		this._initViews();
	},
	
	/**
	 * Looks up the current user's channel id for studio events
	 */
	findUserStudioChannel: function findUserStudioChannel()
	{
        Portal.Data.currentUser.userChannels.each(function(channel)
        {
			if(channel.value.channelTypeId == this.STUDIO_CHANNEL_ID)
			{
				this.userStudioChannel = channel.value.id;
			}
        }.bind(this));
	},
	
	/**
	 * Adds category images to the event categories hash
	 * 
	 */
	injectCategoryImages: function injectCategoryImages()
	{
        Portal.Data.events.categories.each(function(category)
        {
			switch(category.value.name.toLowerCase())
			{
				case 'membership':
					category.value.icon = 'img_icon_membership.png';
					break;
				case 'site':
					category.value.icon = 'img_icon_site.png';
					break;
				case 'billing':
					category.value.icon = 'img_icon_billing_big.png';
					break;
				case 'services':
					category.value.icon = 'img_icon_services_big.png';
					break;
				case 'usage':
					category.value.icon = 'img_icon_usage.png';
					break;
				case 'backups':
					category.value.icon = 'img_icon_backups.png';
					break;
				default:
					category.value.icon = 'img_icon_alert_general.png';
			}
			
			Portal.Data.events.categories.set(category.key, category.value);
        });
	},
	
	/**
	 * Determines how many events in a category have been subscribed to by user channel id
	 * 
	 */
	calcCategoryUsage: function calcCategoryUsage()
	{
		// loop over the categories
        Portal.Data.events.categories.each(function(category)
        {
			var usageCount = {};
			
			// then loop over the events for that category
            Portal.Data.events.fetchByCategory(category.value.id).each(function(event)
            {
				// now, we have to loop ALL the user's subs and figure out whether they have any that match this event...
                Portal.Data.currentUser.subscriptions.each(function(subscription)
                {
					if(subscription.value.eventId == event.value.id)
					{
						if(!usageCount[subscription.value.channelId])
						{
							usageCount[subscription.value.channelId] = 1;
						}
						else
						{
							usageCount[subscription.value.channelId] ++;
						}
					}
                });
            });
			
			category.value.usageCount = usageCount;
			
			Portal.Data.events.categories.set(category.key, category.value);
        });
	},
	
	/**
	 * Shows / hides the individual event rows for a category
	 * 
	 * @param {Integer} categoryId
	 */
	toggleCategoryRows: function toggleCategoryRows(categoryId)
	{
        $$('tr.row_category_' + categoryId).each(function(item)
        {
			if($(item).style.display == 'none')
			{
				$(item).style.display = '';
				$('link_' + categoryId).update('<img src="images_global/button_hidegroup.png" />');
				
				// store this in the list of expanded categories (prevents it from being collapsed on redraw)
				if (this.expandedCategories.indexOf(categoryId) == -1) 
				{
					this.expandedCategories.push(categoryId);
				}
			}
			else
			{
				$(item).style.display = 'none';
				$('link_' + categoryId).update('<img src="images_global/button_expandgroup.png" />');
				
				// remove from the expanded category list
				this.expandedCategories = this.expandedCategories.without(categoryId);
			}
        }.bind(this));
	},
	
	/**
	 * Globally subs / unsubs users from all events in a particular category
	 * 
	 * @param {String} previousState
	 * @param {Integer} categoryId
	 * @param {Integer} userChannelId
	 */
	setCategorySubscriptions: function setCategorySubscriptions(previousState, categoryId, userChannelId)
	{
		var newClassName = '';
		
		// the commit message shell
        var commitMsg = 
        {
			subscription:
			{
				events: []
			}
		};
		// ... and the delete
		var deleteMsg = 
		{
			subscription: 
			{
				subscriptions: []
			}
		}
		// default request type
		var request = 'commit';
		var extra = '';
		
		
		// will set everything to "off"
		if(['mixed', 'on'].indexOf(previousState) != -1)
		{
			newClassName = 'off';
			request = 'delete';
			extra = 'bulk_delete';
		}
		// otherwise, on
		else
		{
			newClassName = 'on';
		}
		
		// get the events for the category
		var events = Portal.Data.events.fetchByCategory(categoryId);
		
		// itereate over all the events
        events.each(function(event)
        {
			// add to the commit message
            commitMsg.subscription.events.push(
            {
                event: 
                {
					channel_id: userChannelId.toString(),
					event_type_id: event.value.id.toString()
				}
            });
			
			if(Portal.Data.currentUser.getUserSubId(event.value.id, userChannelId))
			{
				// and to the delete message
	            deleteMsg.subscription.subscriptions.push(
	            {
	                subscription: 
	                {
						subscription_id: Portal.Data.currentUser.getUserSubId(event.value.id, userChannelId).toString()
					}
	            })
			}
			
			// now do the visual display work...
			$$('a#' + event.value.id).each(function(element)
			{
				var elementVars = element.rel.split('|');
				
				if(elementVars[0] == userChannelId && elementVars[1] == categoryId)
				{
					$(element).removeClassName('on');
					$(element).removeClassName('off');
					$(element).addClassName(newClassName);
				}
			});
        });
		
		this.messagePending = true;
		$('alerts_activity').show();
		
		// finally, send the actual add / delete message (depends on the request type)
		EventManager.publish('/portal/cloud/model', 
        {
			url: 'users/' + Portal.Data.currentUser.userId + '/subscriptions/' + extra,
			request: request,
			xmlData: (request == 'delete') ? deleteMsg : commitMsg,
			returnChannel: '/portal/cloud/alerts'
		});
	},
	
	/**
	 * Individually subs / unsubs from an event
	 * 
	 * @param {String} previousState
	 * @param {Integer} eventId
	 * @param {Integer} userChannelId
	 * @param {Integer} categoryId
	 */
	setUserSubscription: function(previousState, eventId, userChannelId, categoryId)
	{
		// get the category id
		var tempCat = Portal.Data.events.categories.get(categoryId);
		
		// turn on the vent
		if(previousState == 'off')
		{
			// update the counts...
			if (!tempCat.usageCount[userChannelId]) 
			{
				tempCat.usageCount[userChannelId] = 1;
			}
			else
			{
				tempCat.usageCount[userChannelId] ++;
			}
			
			// build the message
	        var msg = 
	        {
				subscription:
				{
					events: 
					[
						{
							event:
							{
								channel_id: userChannelId.toString(),
								event_type_id: eventId.toString()
							}
						}			
					]
				}
				
			};
			
			this.messagePending = true;
			$('alerts_activity').show();
			
			// send the message
			EventManager.publish('/portal/cloud/model', 
	        {
				url: 'users/' + Portal.Data.currentUser.userId + '/subscriptions/',
				request: 'commit',
				xmlData: msg,
				returnChannel: '/portal/cloud/alerts'
			});
			
		}
		else
		{
			// build the message
			var msg = 
			{
				subscriptions:
				[
					{
						subscription:
						{
							subscription_id: Portal.Data.currentUser.getUserSubId(eventId, userChannelId).toString()
						}
					}
				]
			}			

			this.messagePending = true;
			$('alerts_activity').show();

			// send the message
			EventManager.publish('/portal/cloud/model', 
	        {
				url: 'users/' + Portal.Data.currentUser.userId + '/subscriptions/bulk_delete',
				request: 'delete',
				xmlData: msg,
				returnChannel: '/portal/cloud/alerts'
			});
			
			// update the count
			tempCat.usageCount[userChannelId] --;
		}
		
			
		// check button state...
		var newClassName = '';
		if (tempCat.usageCount[userChannelId] > 0) 
		{
			if (tempCat.usageCount[userChannelId] == Portal.Data.events.fetchByCategory(categoryId).size()) 
			{
				newClassName = 'on';				
			}
			else 
			{
				newClassName = 'mixed';
			}
		}
		else
		{
			newClassName = 'off';
		}
		
		// take care of the category image as well...		
        $$('a.alertButton.category').each(function(element)
        {
			if(element.id == categoryId && element.rel == userChannelId)
			{
				$(element).removeClassName('on');
				$(element).removeClassName('off');
				$(element).removeClassName('mixed');
				
				$(element).addClassName(newClassName);
				
				return;
			}
        });
		
	},
	
	checkUserSubscription: function checkUserSubscription(eventId, userChannelId)
	{
		var returnValue = false;
		
        Portal.Data.currentUser.subscriptions.each(function(subscription)
        {
			// found the event... now verify the channel
			if(subscription.value.eventId == eventId && subscription.value.channelId == userChannelId)
			{
				returnValue = true;
			}
        });
		
		return returnValue;
	},
	
	doCreateChannel: function doCreateChannel()
	{
		var fixDisplay = function()
		{
			$('alertsManagerCage').setStyle({ height: $('alertsManagerCage').getHeight() + 'px' });
		}
		
		var email = $F('new_contact');
		var emailRegex = new RegExp(/^[A-Z0-9._%+-]+@[A-Z0-9.-]+\.[A-Z]{2,}$/i);
			
		if(email == '' || !emailRegex.test(email))
		{
			$('add_contact_error').update('Please enter a valid email address.').show();
			fixDisplay();
			
			return;
		}
		
		$('add_contact_error').hide();
		fixDisplay();
		
		var message = 
		{
			contact:
			{
				channel_type_id: '1', // email only for now...
				value: email
			}
		}
		
		$('new_contact').disable();
		$('add_contact_controls').hide();
		$('add_contact_working').show();
		
        EventManager.publish('/portal/cloud/model', 
        {
			request: 'commit',
			url: 'users/' + Portal.Data.currentUser.userId + '/channels',
			xmlData: message,
			returnChannel: '/portal/cloud/channels'
		});
	},
	
	handleAddChannel: function handleAddChannel(msg)
	{
		if(msg.data.status.toString() == '201')
		{
			Portal.Data.currentUser.fetchUserChannels();
			$('show_add_contact').show();
		}
		else
		{
			$('add_contact_working').hide();
			$('add_contact_controls').show();
			$('new_contact').enable();
			
			$('add_contact_error').update(Portal.API.templates.parseErrors('An error ocurred adding the new contact.', msg.data.errors)).show();
			
			$('alertsManagerCage').setStyle({ height: $('alertsManagerCage').getHeight() + 'px' });
		}
	},
	
	deleteUserChannel: function deleteUserChannel(channelId)
	{
        Portal.API.dialogs.confirm(
        {
			message: 'Are you sure you wish to completely remove the selected contact?',
			title: 'Confirm contact removal',
			onConfirm: function()
			{
                EventManager.publish('/portal/cloud/model', 
                {
					url: 'users/' + Portal.Data.currentUser.userId + '/channels/' + channelId,
					request: 'delete',
					returnChannel: '/portal/cloud/channels' 
				});
			}
		});
	}
}

var AlertsManagerViews = 
{
	_initViews: function _initViews()
	{
        View.load(
        {
			main:
			{
				file: 'portlets/cloud/templates/alerts_manager/alerts_manager.html',
				binding: function()
				{
					return $('alertsManagerCage');
				},
				scope:
				{
					categories: Portal.Data.events.categories,
					userChannels: Portal.Data.currentUser.userChannels,
					studioChannel: this.STUDIO_CHANNEL_ID,
					userStudioChannel: this.userStudioChannel,
					expandedCategories: this.expandedCategories
				},
				behaviors: function()
				{
					// category expand / collapse observers
                    $$('a.link_category').each(function(item)
                    {
						$(item).stopObserving('click');
						
                        $(item).observe('click', function(event)
                        {
							this.toggleCategoryRows(Event.element(event).up().id.replace('link_', ''));
                        }.bind(this));
						
                    }.bind(this));
					
					// individual event button observers					
                    $$('a.alertButton').each(function(item)
                    {
						$(item).stopObserving('click');
						
						$(item).observe('click', this.toggleButtonState.bindAsEventListener(this));
                    }.bind(this));
					
					var fixDisplay = function()
					{
						$('alertsManagerCage').setStyle({ height: $('alertsManagerCage').getHeight() + 'px' });
					}
					
					// add contact button observer
					$('show_add_contact').stopObserving('click');
                    $('show_add_contact').observe('click', function(event)
                    {
						if ($('add_contact_form').style.display == 'none')
						{
							$('show_add_contact').hide();
							$('add_contact_form').show();
							$('new_contact').enable();
							$('add_contact_controls').show();
							$('add_contact_working').hide();
							$('add_contact_error').hide();
							$('new_contact').focus();
							fixDisplay();	
						}
						
                    });
					
					// cancel add contact observer
					$('cancel_add_contact').stopObserving('click');
                    $('cancel_add_contact').observe('click', function(event)
                    {
						$('new_contact').value = '';
						$('add_contact_form').hide();
						$('show_add_contact').show();
						
						fixDisplay()
                    });
					
					// add contact button observer (the submit one)
					$('do_add_contact').stopObserving('click');
                    $('do_add_contact').observe('click', this.doCreateChannel.bindAsEventListener(this));
					
				}.bind(this)
			}
		}, 
		function(templates)
        {
			this.templates = templates;
			
			this.templates.main.render();
        }.bind(this));
	},
	
	toggleButtonState: function toggleButtonState(event)
	{
		if (this.messagePending) 
		{
			return;
		}
		
		var element = Event.element(event);
		var previousState = 'on';
		
		// change the css classes first...
		if ($(element).hasClassName('on'))
		{
			$(element).removeClassName('on');
			$(element).addClassName('off');
		}
		else if ($(element).hasClassName('mixed'))
		{
			previousState = 'mixed';
			$(element).removeClassName('mixed');
			$(element).addClassName('off');
		}
		else
		{
			previousState = 'off';
			$(element).removeClassName('off');
			$(element).addClassName('on');
		}
		
		// now, figure out whether this was a category click, or item click, and marshall the
		// appropriate function...
		
		if ($(element).hasClassName('category'))
		{
			this.setCategorySubscriptions(previousState, $(element).id, $(element).rel);
		}
		else
		{
			var elements = $(element).rel.split('|');
			
			this.setUserSubscription(previousState, $(element).id, elements[0], elements[1]);
		}
	}
}
