# Pantera Passive Plugin - Check if page has Session ID
#
# FILENAME      : check_for_parameters.py
# CODER         : Simon Roses Femerling
# DATE          : 07/22/2006
# LAST UPDATE   : 08/21/2006
# ABSTRACT      : Check if page has Session ID
#       
# - Roses Labs Innovations (RL+I)
# Roses Labs
# http://www.roseslabs.com
#
# Copyright (c) 2003-2006 Roses Labs.
#
# You may not distribute, transmit, repost this software for commercial 
# purposes without Roses Labs written permission. 
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, publish,
# distribute the Software, and to permit persons to whom the Software 
# is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
# CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
# TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
# SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

'''
@author:       Simon Roses Femerling
@license:      GNU General Public License 2.0 or later
@contact:      pantera.proxy@gmail.com
@organization: OWASP / Roses Labs
'''

from panteraPlugins import PPM
import re

plugin_info =  {'id':"CheckForSessionID",
                'name':"Session ID Check",
                'author':"Simon Roses Femerling",
                'email':"sroses@roseslabs.com",
                'version':"0.1",
                'desc':"Check if page has Session ID",
                'tag':"session_id"  # defined tags else ignore plugin
                } 

class CheckForSessionID(PPM):
    '''
    CheckForSessionID class.
    '''
    
    def InitAnalyzer(self):
        '''
        Init function.
        '''
        PPM.__init__(self)
        self.session_list = [('ASPSESSIONID.*?(;| )','MS IIS'),                                 # MS IIS
                            ('ASP.NET_SessionId.*?(;| )','ASP.NET'),                            # ASP.NET_SessionId
                            ('PD-S-SESSION-ID.*?(;| )','IBM Tivoli Policy Director WebSeal'),   # IBM Tivoli Policy Director WebSeal 
                            ('PD_STATEFUL.*?(;| )','IBM Tivoli Policy Director WebSeal'),       # IBM Tivoli Policy Director WebSeal 
                            ('WEBTRENDS_ID.*?(;| )','WEBTRENDS'),                               # WEBTRENDS
                            ('sessionid.*?(;| )','IBM WebSphere Application Server'),           # IBM WebSphere Application Server 
                            ('_sn.*?(;| )','IBM WebSphere Application Server'),                 # IBM WebSphere Application Server 
                            ('BCSI-.*?(;| )','BlueCoat Proxy'),                                 # BlueCoat Proxy 
                            ('CFID.*?(;| )','Coldfusion'),                                      # Coldfusion
                            ('CFTOKEN.*?(;| )','Coldfusion'),                                   # Coldfusion
                            ('CFGLOBALS.*?(;| )','Coldfusion'),                                 # Coldfusion
                            ('__utmz.*?(;| )','Urchin Tracking Module'),                        # Urchin Tracking Module
                            ('__utma.*?(;| )','Urchin Tracking Module'),                        # Urchin Tracking Module
                            ('jsessionid.*?(;| )','JRun'),                                      # JRun
                            ('sid.*?(;| )','PHP'),                                              # PHP 
                            ('phpsid.*?(;| )','PHP'),                                           # PHP
                            ('.*id.*?(;| )','Generic Session ID Fingerprint'),                  # generic id fingerprint  
                            ('.*session.*?(;| )','Generic Session ID Fingerprint')]             # generic id fingerprint       
                            
    def BeginAnalyzer(self,obj):
        '''
        Begin function.
        '''
        self.SetObjData(obj)
        # Look for session id in cookies
        cont = False
        for h in obj.serverheader.headerValuesDict.keys():
            t = h.lower()
            if t == 'set-cookie' or t == 'set-cookie2':
                co = obj.serverheader.headerValuesDict[h]
                for dat in self.session_list:
                    id_fin,id_name = dat
                    if cont == True:
                        break
                    for c in co:
                        p = re.compile(id_fin,re.IGNORECASE)
                        r = p.match(c)
                        if r:
                            # add session_id
                            self.SetLevel('info')
                            self.SetResult(1,'session_id')
                            # Add to DDBB
                            self.SetType('session_id')
                            id = r.group()
                            if id[-1] == ';' or id[-1] == ' ':  # strip ; or whitespace
                                l = len(id)
                                id = id[:l-1]
                            self.SetTypeData(id)
                            # Add vuln data
                            s = "Session ID (%s) found on Cookie: %s" % (id_name, id)
                            self.SetResultData(s)
                            cont = True
                            
# RL+I EOF