# Pantera - Web Assessment Studio (WAS)
#
# FILENAME      : panteraUI.py
# CODER         : Simon Roses Femerling
# DATE          : 9/23/2004
# LAST UPDATE   : 06/29/2006
# ABSTRACT      : Python Web Pen-Test Proxy :)
#                 Pantera UI stuff.
#
# - Roses Labs Innovations (RL+I)
# Roses Labs
# http://www.roseslabs.com
#
# Copyright (c) 2003-2006 Roses Labs.
#
# You may not distribute, transmit, repost this software for commercial 
# purposes without Roses Labs written permission. 
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, publish,
# distribute the Software, and to permit persons to whom the Software 
# is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
# CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
# TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
# SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

'''
@author:       Simon Roses Femerling
@license:      GNU General Public License 2.0 or later
@contact:      pantera.proxy@gmail.com
@organization: OWASP / Roses Labs
'''

'''
PanteraProxyUI: Main code file of Pantera.
'''

# Pantera UI imports

import os
import dircache
import cPickle
import urllib
import urlparse
import time
import string

import md5
import sha
import base64
import re

import threading
import Queue
import sets
import sys
import gzip
import StringIO

try:
    import MySQLdb
except:
    print "Error: Pantera needs mysqldb, read documentation please!"

from requestandresponse import RequestAndResponse

import panterautils
import pantera
import panteraHTML
import panteraDB
import panteraAnalyzer
import panteraPlugins
import panteraLib
import panteraFile
import panteraSnitch
#import Dave's VulnXML library
#import VulnXML

__UIVERSION__ = "0.2.1"

#############################################################################################
# Timming
#############################################################################################

t0 = t1 = 0

def start():
    '''
    
    '''
    global t0
    t0 = time.time()

def finish():
    '''
    
    '''
    global t1
    t1 = time.time()

def seconds():
    '''
    
    '''
    return int(t1 - t0)

def milli():
    '''
    
    '''
    return int((t1 - t0) * 1000)

def micro():
    '''
    
    '''
    return int((t1 - t0) * 1000000)


#############################################################################################
# Defines
#############################################################################################

YES = 1
NO  = 0
    
ERROR_STR= """Error removing %(path)s, %(error)s """    
    
#############################################################################################
# Classes
#############################################################################################

###############################################
#
# ThreadNotSaveLinks class
#
###############################################

#############################################################################################
# FUNC     : class ThreadNotSaveLinks
# PARAMS   : threading.Thread
# RETURN   : ...
# ABSTRACT : Thread to save not saved links 
class ThreadNotSaveLinks(threading.Thread):
    '''
    
    '''
    """ ThreadNotSaveLinks class """
    
    #############################################################################################
    # FUNC     : def __init__
    # PARAMS   : conn, value, list(set)
    # RETURN   : ...
    # ABSTRACT : Init internal variables
    def __init__(self, conn, l):
        '''
    
        '''
        """ __init__ func """
        self._db_conn = conn
        self._l = l
        threading.Thread.__init__(self)
    # EOF: def __init__
        
    #############################################################################################    
    # FUNC     : def run
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : Begin Thread. Insert not saved links into DDBB
    def run(self):
        '''
    
        '''
        """ run func """
        for ll in self._l:
            self._db_conn.Insert_New_Not_Save_Link(ll)
    # EOF: def run

###############################################
#
# PanteraProxyUI class
#
###############################################

#############################################################################################
# FUNC     : class PanteraProxyUI
# PARAMS   : ...
# RETURN   : ...
# ABSTRACT : Pantera UI class
class PanteraProxyUI:
    '''
    
    '''
    """Base class for PanteraProxyUI."""

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def __init__(self):
        '''
    
        '''
        """Here we init the internals of PanteraProxyUI."""
        self.triggerhost = "pantera"
        self.basedir = panterautils.pathjoin(os.getcwd(),"SessionData")
        self.http_editor_dir = panterautils.pathjoin(os.getcwd(),"PanteraData","http_editor_profiles")
        self.panteradir = os.getcwd()
        self.dostore = 1
        self.stopallactions = 0
        self.parent = None
        #if the path doesn't exit, make it exist
        panterautils.dmkdir(self.basedir)
        panterautils.dmkdir(self.http_editor_dir)        
        #set up our keywords function table
    #    self.initkeywords()
    #    self.setupfuzzstrings()
        self.requestCache = []
        self.not_save_linksCache = []
        self.requestCacheMaxLength = 5000
        self.not_save_linksCacheMaxLength = 150
        self.logs = []
        self.maxlogs = 1500
   #      self.setupTriggers()
        #stores objects we don't want to talk about
        self.nottolog=[]
        #self.VulnXMLDirectory="VulnXML"
        #self.VulnXMLVariableTestsDirectory="VariableTests"
        #self.VulnXMLDirectoryTestsDirectory="DirectoryTests"
        #self.VulnXMLFileTestsDirectory="FileTests"
        #self.VulnXMLSiteTestDirectory="SiteTests"
        self.log(">>> Pantera UI V %s " % (__UIVERSION__) + "Started")
        self.ntlm=()
        self.proxy=()
        self.parent=None
        self.ConfigData = []
        self.Default404List = []
        self.RestrictedHost = []
        self.RestrictedPages = []
        self.DontSaveExt = []
        self.DontContentType = []
        self.TargetList = []
        self.supress_request_header = []
        self.supress_response_header = []
        self.ppa_plugin_list = []
        self.replacer_list = []
        self.interceptor_list = []
        self.interceptor_urls = []
        self.interceptor_delete_urls = []
        self.target_list = []
        self.data_miner = {}
        self.lst_codes = []
        self.iCounter = 0
        self.http_editor_list = []
        self.http_editor_mybody = pantera.HTTPBody()
        self.http_editor_myheader = pantera.HTTPHeader()

        self.current_project_id = 0
        self.inside_project = NO
        self.current_link_id = 0

        self.log_file = "pantera.log"
        self.log_fp = ""
        
        self.error_log_file = "error.log"
        self.error_log_fp = ""

        # threading pool
        self.pas = []
        self.job_queue = Queue.Queue(0)

        #delete log file when we start
        self.DeleteLog()
        self.LogError("Init Pantera Error Log File")

        self.flag_ddbb = YES
        
        self.pf = panteraFile.PanteraFile()
        return

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def setNTLM(self,ntlm):
        '''
    
        '''
        """Set NTLM."""
        self.ntlm=ntlm

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def setProxy(self,proxy):
        '''
    
        '''
        """Set Proxy."""
        self.proxy=proxy

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def setParent(self,parent):
        '''
    
        '''
        """This function lets the UI  talk to the first panteraproxy instance."""
        self.parent=parent

    #############################################################################################
    def setNoProjectName(self):
        '''
    
        '''
        """Set Project Name to "no name set" when not inside a project."""
        self.ConfigData['project_name'] = "Name Not Set"

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def setConfigDataXML(self, d):
        '''
    
        '''
        """Get XML configuration."""
        self.ConfigData = d

        #
        # Internal inits
        #

        # Convert here str to list for dont save ext
        self.DontSaveExt = panterautils.Str2List(self.ConfigData['dont_save_ext'])

        self.lst_codes = panterautils.Str2List(self.ConfigData['reject_return_codes'],",")

        #set CSS
        css = panteraHTML.PanteraHTML()
        if not self.ConfigData.has_key('css_file') or self.ConfigData['css_file'] == '':
            self.ConfigData['css_file'] = css.ReturnCSSFile()
        else:
            css.SetCSSFile(self.ConfigData['css_file'])

        #set project name
        if not self.ConfigData.has_key('project_name') or self.ConfigData['project_name'] == '':
            self.setNoProjectName()
        else:
            temp = panterautils.FilterBadCar(self.ConfigData['project_name'])
            self.ConfigData['project_name'] = temp
    
        # Init DDBB pool
        self.InitDDBBPool()
        
        # Init Analyzer Threads pool
        self.InitAnalyzerThreadsPool()
                 
        # Plugins
        self.ppa_plugins = panteraPlugins.PassivePluginsManager()
        if self.ConfigData.has_key('ppa_plugins_dir') and self.ConfigData['ppa_plugins_dir'] == '':
            self.ppa_plugins.SetPluginPath(panterautils.pathjoin(os.getcwd(),self.ConfigData['ppa_plugins_dir']))
        else:
            self.ppa_plugins.SetPluginPath(panterautils.pathjoin(os.getcwd(),"ppa_plugins"))

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def InitDDBBPool(self):
        '''
    
        '''
        """ Init DDBB Pool func."""
        #self.pan_db = ''
        #print self.ConfigData['db_login']
        #print self.ConfigData['db_password']        
        try:
            self.pan_db_temp = panteraDB.PanteraDBPool(MySQLdb,5, user=self.ConfigData['db_login'], \
                                                    passwd=self.ConfigData['db_password'],
                                                    host=self.ConfigData['db_host'], \
                                                    db=self.ConfigData['db_name'])
        except: # oouch
            self.LogError("Error DDBB: Unable to connect")
            self.LogError(sys.exc_info()[0])
            self.flag_ddbb = NO
        else: # success
            self.pan_db = panteraDB.PanteraDB(self.pan_db_temp.getConnection())

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def InitAnalyzerThreadsPool(self):
        '''
    
        '''
        """ Init Analyzer Threads Pool func."""
        try:
            self.conn_pool_temp = panteraDB.PanteraDBPool(MySQLdb,int(self.ConfigData['analyzer_threads']), user=self.ConfigData['db_login'], \
                                                    passwd=self.ConfigData['db_password'],
                                                    host=self.ConfigData['db_host'], \
                                                    db=self.ConfigData['db_name'])
        except:
            self.LogError("Error DDBB: Unable to connect")
            self.flag_ddbb = NO
        
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def setDefault404(self, d):
        '''
    
        '''
        """Set the defaults HTTP 404 patterns."""
        self.Default404List = d

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def setRestrictedHost(self, d):
        '''
    
        '''
        """Set restricted hosts."""
        self.RestrictedHost = d

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def setRestrictedPages(self, d):
        '''
    
        '''
        """Set restricted pages."""
        self.RestrictedPages = d

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def getDefault404(self):
        '''
    
        '''
        """Return the defaults HTTP 404 patterns."""
        return self.Default404List

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def getRestrictedHost(self):
        '''
    
        '''
        """Return restricted hosts."""
        self.RestrictedHost.sort()
        return self.RestrictedHost

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def getRestrictedPages(self):
        '''
    
        '''
        """Return restricted pages."""
        self.RestrictedPages.sort()
        return self.RestrictedPages

    #############################################################################################
    def setDontContentType(self, d):
        '''
    
        '''
        self.DontContentType = d
        
    #############################################################################################
    def getDontContentType(self):
        '''
    
        '''
        self.DontContentType.sort()
        return self.DontContentType

    #############################################################################################
    def setTargetList(self, d):
        '''
    
        '''
        self.TargetList = d
        
    #############################################################################################
    def getTargetList(self):
        '''
    
        '''
        self.TargetList.sort()
        return self.TargetList

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT :     
    def GetCurrentTime(self,loginfo):
        '''
    
        '''
        timeoflog=time.asctime()
        logstring= "[%s] : %s\n" % (timeoflog,loginfo)
        return logstring

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def log(self,loginfo):
        '''
    
        '''
        """Log and print internal messages."""
        logstring = self.GetCurrentTime(loginfo)
        #print it out to our running string
        print logstring
        self.logs=[logstring]+self.logs
        if len(self.logs)==self.maxlogs:
            del self.logs[-1]

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def LogLine(self, str):
        '''
    
        '''
        """Log message to internal Pantera log file."""
        file=open(self.log_file,'a')
        file.write(str)
        file.write("--------------------------------------------------------\r\n")
        file.close()

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ReadLog(self):
        '''
    
        '''
        """Read internal Pantera log file."""
        if os.path.isfile(self.log_file):
            file=open(self.log_file,'r')
            r = file.read()
            file.close()
            return r
        else:
            return ""

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DeleteLog(self):
        '''
    
        '''
        """Delete internal Pantera log file."""
        self.DeleteFile(self.log_file)
        
    #############################################################################################
    def DeleteFile(self, f):
        '''
    
        '''
        """Delete a file."""
            
        if os.path.isfile(f):
            os.unlink(f)

    #############################################################################################
    def rmgeneric(self, path, __func__):
        '''
    
        '''
        try:
            __func__(path)
        except OSError, (errno, strerror):
            print ERROR_STR % {'path' : path, 'error': strerror }

    #############################################################################################
    def DeleteDir(self, path):
        '''
    
        '''
        """Delete a directory."""
        
        if not os.path.isdir(path):
            return
    
        files=os.listdir(path)

        for x in files:
            fullpath=os.path.join(path, x)
            if os.path.isfile(fullpath):
                f=os.remove
                self.rmgeneric(fullpath, f)
            elif os.path.isdir(fullpath):
                self.DeleteDir(fullpath)
                f=os.rmdir
                self.rmgeneric(fullpath, f)
    
    #############################################################################################
    def CleanTempCache(self):
        '''
    
        '''
        """ Clean internal Pantera cache."""
        del self.requestCache
        self.requestCache = []
        
    #############################################################################################
    def CleanInterceptorCache(self):
        '''
    
        '''
        """ Clean Interceptor cache. """
        del self.interceptor_delete_urls
        del self.interceptor_urls
        self.interceptor_delete_urls = []
        self.interceptor_urls = []

    #############################################################################################
    def LogError(self, s):
        '''
    
        '''
        file=open(self.error_log_file,'a')
        file.write(self.GetCurrentTime(s))
        file.write("--------------------------------------------------------\r\n")
        file.close()
        
    #############################################################################################
    def ReadErrorLog(self):
        '''
    
        '''
        if os.path.isfile(self.error_log_file):
            file=open(self.error_log_file,'r')
            r = file.read()
            file.close()
            return r
        else:
            return ""
    
    #############################################################################################
    def DeleteErrorLog(self):
        '''
    
        '''
        """Delete internal Pantera error log file."""
        self.DeleteFile(self.error_log_file)

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def wantsRequest(self,myheader):
        '''
    
        '''
        """Cacth if request is related to Pantera internals."""
        
        if myheader.connectHost==self.triggerhost:
            #print "TRIGGERHOST"
            return 1
        
        if myheader.URLargsDict.has_key("PANTERA_TRIGGER"): 
            #print "URLARGSDICT=PANTERA_TRIGGER"
            return 1        
        
        if self.DoInterceptor(myheader):
            #print "DO_INTERCEPTOR"
            return 1
        
        return 0

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def handleRequest(self,myheader,mybody):
        '''
    
        '''
        """Handle request."""
        if not self.wantsRequest(myheader):
            return "Unkown request?"

        extention=myheader.URL.split(".")[-1]

        #print "DEBUG1"
        if extention=="html" and not myheader.URLargsDict.has_key("PANTERA_TRIGGER"):
            filename=myheader.URL.replace("http://pantera/","")
            # Do check here in case of refresh history form
            if filename == '/history_form':
                return self.HistoryHTMLPage()
            return self.serveFile(filename)

        #print "DEBUG2"
    
        #
        # Do here pantera internal commands
        #
        
        #print "URL:" + myheader.URL
        urlfile=myheader.URL.split("/")[-1]
        file=""
        data=""
        haveheader=0

        #print "URLFILE" + urlfile
        #print myheader.URLargsDict
        #print myheader.headerValuesDict
        #print mybody.getArgsDict()
        #print "".join(mybody.data)
        
        if myheader.URLargsDict.has_key("file"):
            file = myheader.URLargsDict["file"]

        # Begin with internal pantera links
        if urlfile =="config_link":
            return self.addHeader(self.ConfigHTMLPage())
        elif urlfile =="session_trace_link":
            param = myheader.URLargsDict
            if param.has_key('order'):
                return self.addHeader(self.SessionTraceHTMLPage(int(param['order'])))
            else:
                return self.addHeader(self.SessionTraceHTMLPage(0))
        elif urlfile =="session_raw_link":
            return self.addHeader(self.SessionTraceRawHTMLPage())
        elif urlfile in ["setconfig","setConfig"]:
            self.setConfig(myheader, mybody)
            return self.addHeader(self.IndexHTMLPage())
        elif urlfile =="pantera_management_link":
            return self.PanteraManagementHTMLPage()
        elif urlfile =="pantera_management_form":
            post_data = mybody.getArgsDict()
            self.DoPanteraManagement(post_data)
            return self.addHeader(self.IndexHTMLPage())
        elif urlfile =="close_project_link":
            self.CloseProject(self.current_project_id)
            return self.addHeader(self.IndexHTMLPage())                    
        elif urlfile =="history_link":
            param = myheader.URLargsDict
            if param.has_key('order'):
                return self.addHeader(self.HistoryHTMLPage(int(param['order'])))
            else:
                return self.addHeader(self.HistoryHTMLPage(0))   
        elif urlfile =="history_form":
            post_data = mybody.getArgsDict()
            if post_data.has_key("notes_page"):
                return self.addHeader(self.LinkNotesHTMLPage(urllib.unquote_plus(post_data["notes_page"])))
        elif urlfile == "direct_link_info":
            return self.addHeader(self.GetInfoHTMLPage(file))
        elif urlfile == "editor_link":
            return self.addHeader(self.EditSendHTMLPage(0,urllib.unquote_plus(file)))
        elif urlfile == "body_page_link":
            return self.addHeader(self.BodyHTMLPage(urllib.unquote_plus(file)))
        elif urlfile == "delete_link":
                f = urllib.unquote_plus(file)
                if self.inside_project == YES:
                    file = self.pan_db.Return_Link_File(self.current_project_id,f)
                    if file == -1:    
                        self.LogError(self.pan_db.GetErrorMsg())
                        return self.addHeader(self.ErrorHTMLPage("Invalid Value"))
                    self.HardDeletePage(f)
                    self.DeleteDir(file)
                else:
                    self.SoftDeletePage(f)
                    self.DeleteDir(f)
                return self.addHeader(self.HistoryHTMLPage())
        elif urlfile == "notes_link":
            return self.addHeader(self.LinkNotesHTMLPage(urllib.unquote_plus(file)))
        elif urlfile == "send_request":
            data=self.sendrequest(myheader,mybody,0)
            haveheader=1
            return self.addHeader(self.IndexHTMLPage())
        elif myheader.URLargsDict.has_key("PANTERA_TRIGGER") and myheader.URLargsDict["PANTERA_TRIGGER"] == "send_rewrite":
            myheader.DelURL()
            #print myheader.URLargsDict
            data=self.sendrequest(myheader,mybody,1)
            haveheader=1
            if not haveheader:
                data=self.addHeader(data)
            return data # return data
        elif urlfile == "utils_link":
            return self.addHeader(self.UtilsHTMLPage())
        elif urlfile =="utils_form":
            post_data = mybody.getArgsDict()
            return self.addHeader(self.UtilsHTMLPage(post_data))
        elif urlfile == "project_link":
            return self.addHeader(self.ProjectHTMLPage())
        elif urlfile in ["setproject","setProject"]:
            w = self.setProject(myheader, mybody)
            if w == 0:    
                return self.addHeader(self.ProjectHTMLPage())
            else:
                return self.addHeader(self.IndexHTMLPage())
        elif urlfile == "report_link":
            return self.addHeader(self.ReportHTMLPage())
        elif urlfile == "statistics_link":
            return self.addHeader(self.StatisticsHTMLPage())
        #elif urlfile == "tree_link":
        #    return self.addHeader(self.TreeHTMLPage())
        elif urlfile == "site_tree_form":
            post_data = mybody.getArgsDict()
            return self.addHeader(self.TreeHTMLPage(post_data))
        elif urlfile == "reconstruct_link":
            return self.addHeader(self.ReconstructHTMLPage())
        elif urlfile == "interceptor_link":
            return self.addHeader(self.InterceptorHTMLPage())
        elif urlfile == "interceptor_form":
            post_data = mybody.getArgsDict()
            if post_data.has_key('cancel'):
                return self.addHeader(self.InterceptorHTMLPage())
            if post_data.has_key('new'):
                return self.addHeader(self.InterceptorHTMLPage(1, post_data))
            if post_data.has_key('new_add'):
                return self.addHeader(self.InterceptorHTMLPage(2, post_data))
            if post_data.has_key('delete'):
                return self.addHeader(self.InterceptorHTMLPage(3, post_data))
            if post_data.has_key('edit'):
                return self.addHeader(self.InterceptorHTMLPage(4, post_data))
        elif urlfile == "replacer_link":
            return self.addHeader(self.ReplacerHTMLPage())
        elif urlfile == "replacer_form":
            post_data = mybody.getArgsDict()
            if post_data.has_key('cancel'):
                return self.addHeader(self.ReplacerHTMLPage())
            if post_data.has_key('new'):
                return self.addHeader(self.ReplacerHTMLPage(1, post_data))
            if post_data.has_key('new_add'):
                return self.addHeader(self.ReplacerHTMLPage(2, post_data))
            if post_data.has_key('delete'):
                return self.addHeader(self.ReplacerHTMLPage(3, post_data))
            if post_data.has_key('edit'):
                return self.addHeader(self.ReplacerHTMLPage(4, post_data))
        elif urlfile == "repeater_link":
            return self.addHeader(self.RepeaterHTMLPage())
        elif urlfile == "querier_link":
            if myheader.URLargsDict.has_key("data_miner"): 
                return self.addHeader(self.QuerierHTMLPage())
            else:
                del self.data_miner
                self.data_miner = {}
                return self.addHeader(self.QuerierHTMLPage())
        elif urlfile == "querier_form":
            del self.data_miner
            self.data_miner = {}
            post_data = mybody.getArgsDict()
            return self.addHeader(self.QuerierHTMLPage(post_data))
        elif urlfile == "supress_header_link":
            return self.addHeader(self.SupressHeaderHTMLPage())
        elif urlfile == "supress_header_form":
            post_data = mybody.getArgsDict()
            if post_data.has_key('do_supress_change') and post_data['do_supress_change'] == "change":
                return self.addHeader(self.SupressHeaderHTMLPage(2, post_data))
            elif post_data.has_key('do_supress_change') and post_data['do_supress_change'] == "new":
                return self.addHeader(self.SupressHeaderHTMLPage(1))
        elif urlfile == "tutorial_link":
            return self.addHeader(self.TutorialHTMLPage())
        elif urlfile == "about_pantera_link":
            return self.addHeader(self.AboutPanteraHTMLPage())
        elif urlfile == "external_doc_link":
            return self.addHeader(self.ExternalDocHTMLPage())
        elif urlfile == "error_console_link":
            return self.addHeader(self.ErrorConsoleHTMLPage())
        elif urlfile == "ppa_management":
            return self.addHeader(self.PPAHTMLPage())
        elif urlfile == "project_notes":
            return self.addHeader(self.ProjectNotesHTMLPage())
        elif urlfile == "project_notes_form":
            post_data = mybody.getArgsDict()
            self.DoInsertProjectNotesData(post_data)
            return self.addHeader(self.IndexHTMLPage())
        elif urlfile == "link_notes_form":
            post_data = mybody.getArgsDict()
            self.DoInsertLinkNotesData(post_data)
            return self.addHeader(self.IndexHTMLPage())
        elif urlfile == "help_link":
            return self.addHeader(self.HelpHTMLPage())
        elif urlfile == "ppa_form":
            post_data = mybody.getArgsDict()
            self.DoPPAManagement(post_data)
            return self.addHeader(self.IndexHTMLPage())
        elif urlfile == "http_editor_link":
            return self.addHeader(self.HTTPEditorHTMLPage())
        elif urlfile == "scan_engine_link":
            return self.addHeader(self.ScanEngineHTMLPage())
        elif urlfile == "analysis_modules_link":
            return self.addHeader(self.AnalysisModHTMLPage())
        elif urlfile == "validate_page":
            return self.addHeader(self.ValidatePageHTMLPage(urllib.unquote_plus(file)))
        elif urlfile == "validate_page_form":
            post_data = mybody.getArgsDict()
            self.SetPageResource(post_data)
            if post_data.has_key('id'):
                return self.addHeader(self.GetInfoHTMLPage(post_data['id']))
            else:
                return self.addHeader(self.HistoryHTMLPage(0))   
        elif urlfile == "ppa_summary_link":
            return self.addHeader(self.PPASummaryHTMLPage())
        elif urlfile == "snitch_link":
            return self.addHeader(self.SnitchHTMLPage())
        elif urlfile == "snitch_form":
            post_data = mybody.getArgsDict()
            return self.addHeader(self.SnitchHTMLPage(post_data))
        elif urlfile == "insert_file_link":
            param = myheader.URLargsDict
            if param.has_key('file'):
                self.InsertFileIntoView(param['file'])
            return self.addHeader(self.IndexHTMLPage())
        elif urlfile == "target_management_link":
            param = myheader.URLargsDict
            if param.has_key('delete') or param.has_key('update'): 
                return self.addHeader(self.TargetHTMLPage(param))
            else: return self.addHeader(self.TargetHTMLPage(''))
        elif urlfile == "target_management_form":
            post_data = mybody.getArgsDict()
            return self.addHeader(self.TargetHTMLPage(post_data))
        elif urlfile == "create_http_editor_profile":
            return self.addHeader(self.HTTPEditorProfileCreatorHTMLPage())
        elif urlfile == "create_http_editor_profile_form":
            post_data = mybody.getArgsDict()            
            return self.addHeader(self.HTTPEditorProfileCreatorHTMLPage(post_data))
        elif urlfile == "http_editor_form":
            post_data = mybody.getArgsDict()            
            if post_data.has_key('http_editor_send_request'):
                arg = {}
                arg['myheader'] = myheader
                arg['mybody'] = mybody
                arg['http_editor_send_request'] = 'yes'
                return self.addHeader(self.HTTPEditorHTMLPage(arg))                
            else:
                return self.addHeader(self.HTTPEditorHTMLPage(post_data))
        
        #elif urlfile == "db_recon_link":
        #    self.InitDDBBPool()   
        #    return self.addHeader(self.IndexHTMLPage())
                                 
        #
        # End pantera internals commands
        #

        if myheader.URL=="/" and not myheader.URLargsDict.has_key("PANTERA_TRIGGER"):
            return self.IndexHTMLPage() # Main page
        elif myheader.URLargsDict.has_key("PANTERA_TRIGGER") and myheader.URLargsDict["PANTERA_TRIGGER"] == "rewrite":
            return self.EditSendRewrite(myheader,mybody)
        else:
            # Do check here in case of refresh history form
            if myheader.URL == '/history_form':
                return self.HistoryHTMLPage()
            return self.serveFile(myheader.URL)
        
        if (data==""):
            byestring=""#notimplementedyet
            return "HTTP/1.1 501 Not implemented!\r\nContent-Length: "+str(len(byestring))+"\r\n\r\n"+byestring
        else:
            if not haveheader:
                data=self.addHeader(data)
            return data

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def registerRequestandResponse(self,clientheader,clientbody,serverheader,serverbody, force=0):
        '''
    
        '''
        """Resgister request and response."""

        if self.dostore==0:
            return 1
        #basically we organize things as first Sites, then pages,
        #then requests+responses

        #we need to check this in case we are crawling or otherwise don't want to store this
        #page
        if clientheader in self.nottolog:
            return 1
        
        # check for rejected return codes, like 404, etc.
        if len(self.lst_codes)>0:
            if serverheader.returncode in self.lst_codes:
                return 1
            
        if serverheader.bodySize>0:
            ce_data = ''
            data = "".join(serverbody.data)
            if serverheader.headerValuesDict.has_key('Content-Encoding'):
                ce_data = serverheader.headerValuesDict['Content-Encoding'][0]
            if ce_data != '' and ce_data.find("gzip")>=0:
                compressedstream = StringIO.StringIO(data)   
                gzipper = gzip.GzipFile(fileobj=compressedstream)      
                data = gzipper.read()
            r = panterautils.SearchPattern(data, self.Default404List)
            del data
            if r == 1: return 1  
    
        # check if domain is allowed
        #
        dom,sav = self.VerifyTarget(clientheader.connectHost)
        if dom is NO and sav is NO: return 1

        # Save targets (domains that pantera sees on proxy mode)
        if self.inside_project == YES and sav is YES:
            if not clientheader.connectHost in self.target_list:
                d = {}
                d['project_id'] = self.current_project_id
                d['type'] = "'target'"   
                d['value'] = clientheader.connectHost
                d['domain'] = "'" + clientheader.connectHost + "'"
                #d['domain'] = clientheader.connectHost
                insert_host_db = panteraDB.PanteraDB(self.pan_db_temp.getConnection())
                r = insert_host_db.Insert_Project_Info(d)
                if r != -1:
                    self.target_list.append(clientheader.connectHost)

        if dom is NO and sav is YES: save_dom_link = NO  # FIX HERE
        else: save_dom_link = YES
                
        # Check Dont Force Save, if "yes" Pantera will not save links!
        if self.ConfigData['force_dont_save'] == "yes":
            cont = False
            ct = False
            if serverheader.headerValuesDict.has_key("Content-Type"):
                content_type = serverheader.headerValuesDict["Content-Type"]
                ct = True
            elif serverheader.headerValuesDict.has_key("Content-type"):
                content_type = serverheader.headerValuesDict["Content-type"]
                ct = True
            if ct:
                for x in content_type:
                    if x in self.DontContentType:
                        cont = True
                    else:
                        cont = False
                        
            if force ==1: cont = False
                        
            if cont is True: return 1        
            
            #ok, now we need to store it in our bucket of things we've just done for the request cache
            ext = clientheader.URL.split(".")[-1]
            ext = ext.lower()
            if ext in self.DontSaveExt and force == 0: return 1

        #do we have this "site" in our store?
        #A site is defined by host,port,isSSL
        site=self.getSiteFromHeader(clientheader)

        if serverheader is None:
            serverheader=pantera.HTTPHeader()
        if serverbody is None:
            serverbody=pantera.HHTPBody()
        
        if not self.haveSiteInStore(site):
            self.createSite(site)

        #need to check for if we have this page or not
        #a page is just: /foo/bar.php or similar
        page=self.getPageH(clientheader)
        if not self.havePageInStore(page):
            self.createPage(page)

        #delete response headers
        
        #we don't want to store duplicates
        if self.duplicateRequestandResponse(clientheader,clientbody,serverheader,serverbody):
            #print "Duplicate request and response"
            return 1

        #print "before store: "+str(clientheader)+" Type: "+str(type(clientheader))
        #otherwise, we need to store this request and response off
        result=self.storeRequestandResponse(clientheader,clientbody,serverheader,serverbody, force, save_dom_link)
        return result

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    #returns 1 if it's a request and response we've seen before, otherwise 0
    def duplicateRequestandResponse(self,clientheader,clientbody,serverheader,serverbody):
        '''
    
        '''
        """Check if request and response is duplicated."""
        #print "inside duplicateRequestandResponse"
        #first get a list of the files in page's directory. the directory
        #is guaranteed to exist
        pagedir=panterautils.pathjoin(self.basedir,self.getPageH(clientheader))
        filelist=dircache.listdir(pagedir)
        #print "pagedir="+pagedir
        #print "filelist="+str(filelist)

        #order N operation here...we iterate over data
        newhash=panterautils.genhash(clientheader,clientbody,serverheader,serverbody)
        #print "Done with hashing in duplicateRequestandReponse"
        
        #we just compare hashes now
        isdir = os.path.isdir
        for afile in filelist:
            #ignore directories
            if isdir(afile):
                continue
            oldhash=afile.split("_")[0]
            if oldhash==newhash:
                return 1

        #print "Unique object: leaving duplicateRequestandResponse"
        return 0
        
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    #stores a request and response into our file structure for later retrival
    def storeRequestandResponse(self, clientheader,clientbody,serverheader,serverbody,force=0,save_dom_link=0):
        '''
    
        '''
        """Store the request and response."""
        #print "instore "+str(clientheader)+" Type: "+str(type(clientheader))
        dir=panterautils.pathjoin(self.basedir,self.getPageH(clientheader))
        #print "Dir: "+dir
        hash=panterautils.genhash(clientheader,clientbody,serverheader,serverbody)
     
        #we encode the directory name (the full page) for easy uniqueness test
        filename=panterautils.pathjoin(dir,hash+"_"+self.strencode(clientheader.connectHost))

        #here we check for any ODBC strings or whatnot, and if we don't
        #see one of those, and we are in the "don't store" list, we just
        #return
        """
        triginfo=self.scanForTriggers(serverheader,serverbody)
        if triginfo=="" and clientheader in self.nottolog:
            return 1
        """

        """
        #print out the warning
        if triginfo!="":
            self.log("Warning: "+filename+" triggered "+triginfo)
        """

        #print "Storing request in filename: "+filename
        
        obj=RequestAndResponse(clientheader,clientbody,serverheader,serverbody)
        
        #print "obj: "+str(obj)
        #obj.printme()
        openfile=open(filename,"wb")
        #print "openfile="+str(openfile)+" object: "+str(obj)
        binary=1
        #cPickle.dump(obj,openfile,binary,protocol = cPickle.HIGHEST_PROTOCOL)
        #cPickle.dump(obj,openfile,cPickle.HIGHEST_PROTOCOL)
        cPickle.dump(obj,openfile,binary)
        openfile.close()
        #print "Done storing request in filename: "+filename
        #print "Now saving in requestCache"
        
        #ok, now we need to store it in our bucket of things we've just done for the request cache
        ext = clientheader.URL.split(".")[-1]
        ext = ext.lower()
            
        # check for content-type
        cont = False
        ct = False
        if serverheader.headerValuesDict.has_key("Content-Type"):
            content_type = serverheader.headerValuesDict["Content-Type"]
            ct = True
        elif serverheader.headerValuesDict.has_key("Content-type"):
            content_type = serverheader.headerValuesDict["Content-type"]
            ct = True
        if ct:
            for x in content_type:
                if x in self.DontContentType:
                    cont = True
                else:
                    cont = False
            
        #
        # Here begin PPA or just save to DDBB (in case not intereting links)
        #
        if ext not in self.DontSaveExt and cont is False and save_dom_link == YES or force == 1:   
        #if ext not in self.DontSaveExt and cont is False or force == 1:   
            #let's not save pictures in the request cache
            if self.inside_project != YES:
                self.saveInRequestCache(filename)
            
            # store in DDBB if inside project 
            if self.inside_project == YES:
                sf = ""
                if clientheader.clientisSSL:
                    sf += "https://"
                else:
                    sf += "http://"
                sf += "%s%s" % (clientheader.connectHost,clientheader.URL)
                pan_db = panteraDB.PanteraDB(self.pan_db_temp.getConnection())
                res = pan_db.Insert_New_Link(self.current_project_id,filename,sf)
                if res == -1:
                    self.LogError(pan_db.GetErrorMsg())
            
                if res == 1:
                    return 1
        
                last_id = pan_db.ReturnLastIDFromTable("links_t")
                if last_id == -1:
                    self.LogError(pan_db.GetErrorMsg())
                
                    # clean PPA list and set plugins
                self.ppa_plugins.DeletePlugins()
                adder = self.ppa_plugins.AddPlugin
                for l in self.ppa_plugin_list:
                    adder(l)
                        
                d = (last_id,obj)
        
                self.job_queue.put(d)
                db_pool = panteraDB.PanteraDB(self.conn_pool_temp.getConnection())
                pa = panteraAnalyzer.PanteraPassiveAnalyzer(self.ppa_plugins, self.job_queue, self.current_project_id, db_pool)
                pa.start()
                #self.conn_pool_temp.returnConnection(db_pool)        
        else:
            if self.inside_project == YES:
                self.data = {}
                self.data['project_id'] = self.current_project_id
                self.data['method'] = "'" + clientheader.verb + "'"
                m = md5.new(serverheader.genhash())
                self.data['page_hash'] = "'" + m.hexdigest() + "'"
                self.data['return_code'] = "'" + serverheader.returncode + "'"
                self.data['return_msg'] = "'" + serverheader.returnmessage + "'"
                self.data['version'] = "'" + serverheader.version + "'"                
                self.data['extension'] = "'" + ext + "'"
                self.data['host'] = "'" + clientheader.connectHost + "'"
                sf = ""
                if clientheader.clientisSSL:
                    sf += "https://"
                else:
                    sf += "http://"
                sf += "%s%s" % (clientheader.connectHost,clientheader.URL)
                d = (sf, self.data, filename)
                self.saveInNotSaveLinksCache(d)
        
        return 1

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    #rips off the arguments and stuff to yield a nice /bob/dave.php
    #takes in a clientheader, not a string!
    #TODO: this fails currently for urls with a asdf.ng/bob=asdf&asdf=asdf
    #syntax
    def getPageH(self,clientheader):
        '''
    
        '''
        """Get page header."""
        #we already have this stored in the client header
        #print "getPageH "+str(clientheader)+" Type: "+str(type(clientheader))
        site=self.sitestrh(clientheader)
        url=panterautils.urlnormalize(clientheader.URL)
        #dunno why this happens, but it does - techinsurance.com does it
        if url[-1]=="/":
            return panterautils.pathjoin(site,url,"_directory_")
        else:
            return panterautils.pathjoin(site,url)

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def getSiteFromHeader(self,clientheader):
        '''
    
        '''
        """Return site from header."""
        return [clientheader.connectHost,str(clientheader.connectPort),str(clientheader.clientisSSL)]

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    #returns a 1 if we have that site in our store
    def haveSiteInStore(self,site):
        '''
    
        '''
        """Check if site is already stored."""
        sitename=self.sitestr(site)
        result= os.path.isdir(panterautils.pathjoin(self.basedir,sitename))
        #print "Do we have "+sitename+" in our store: "+str(result)
        return result

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT :              
    #creates a "site" store on disk
    def createSite(self,site):
        '''
    
        '''
        """Create site on stored."""
        #is this / going to bite us when we go win32? Who cares? :>
        #fixed with panterautils.pathjoin!
        panterautils.dmkdir(panterautils.pathjoin(self.basedir,self.sitestr(site)))
        return 1

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    #converts a site to a string, just a .join call for now
    def sitestr(self,site):
        '''
    
        '''
        """Converts a site to a string."""
        #print "Site="+str(site)
        return self.strencode("_".join(site))

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def sitestrh(self,clientheader):
        '''
    
        '''
        """Return site string."""
        return self.sitestr(self.getSiteFromHeader(clientheader))

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def strencode(self,astring):
        '''
    
        '''
        """Encode string."""
        return panterautils.strencode(astring)

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    #returns boolean value for whether we've seen this page before
    def havePageInStore(self,page):
        '''
    
        '''
        """Return boolean if site is in stored."""
        dir=self.getDir(page)
        wholepath, filename = os.path.split(page)
        return os.path.isdir(panterautils.pathjoin(self.basedir,dir,filename))

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    #strip off the following dave.php and leave /bob/
    def getDir(self,page):
        '''
    
        '''
        """Return directory from page."""
        return "/"+os.path.dirname(page)

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    #creates a directory for our page. It looks like this: ./sitebase/bob/dave.php/
    def createPage(self,page):
        '''
    
        '''
        """Create directory to the page."""
        #used to do some crazy stuff here, but it's really quite simple
        #print "page="+page
        #print "basedir="+self.basedir
        dirtomake=panterautils.pathjoin(self.basedir,page)
        #print "Trying to make dir "+dirtomake
        panterautils.dmkdir(dirtomake)
        return 1

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def saveInRequestCache(self,filename):
        '''
    
        '''
        """Save page into internal Pantera cache."""
        if filename in self.requestCache: # if link already exists pass...
            return
        self.requestCache=[filename]+self.requestCache
        if len(self.requestCache)==self.requestCacheMaxLength:
            del self.requestCache[-1]

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def saveInNotSaveLinksCache(self,d):
        '''
    
        '''
        """Save page into internal Pantera cache."""
        #self.not_save_linksCache=[filename]+self.not_save_linksCache
        self.not_save_linksCache.append(d)
        self.CheckNotSaveLinks()
        
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def CheckNotSaveLinks(self,flush=0):
        '''
    
        '''
        if self.inside_project == YES:
            if len(self.not_save_linksCache)>=1:
                if len(self.not_save_linksCache)==self.not_save_linksCacheMaxLength or flush==1:
                    pan_db = panteraDB.PanteraDB(self.pan_db_temp.getConnection())
                    ThreadNotSaveLinks(pan_db, self.not_save_linksCache).start()
                    del self.not_save_linksCache
                    self.not_save_linksCache = []

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    #serve a file, replacing keywords with something appropriate
    #used for static html files. not spike cgi requests
    def serveFile(self,filename):
        '''
    
        '''
        """Servers a page from stored."""
        debug_serveFile=0
        if debug_serveFile:
            print "serving file "+filename
        #strip this last bit off
        mybase=panterautils.pathjoin(self.basedir,"..")
        realfilename=panterautils.pathjoin(mybase,filename)
        if os.path.isfile(realfilename):
            file=open(realfilename,"rb") # not the best method to read all files, but should work!
            data=file.read()
            file.close()
        else:
            data="Pantera Error: - No file found: %s" % realfilename
                
        header=""
        #adds both the header and the data
        ext=filename.split(".")[-1]
        #print "EXT: " + ext
        if ext == "gif":
            header+=self.addHeader(data, "image")
        elif ext == "png":
            header+=self.addHeader(data, "image")
        elif ext == "js":
            header+=self.addHeader(data, "js")
        elif ext == "css":
            header+=self.addHeader(data, "css")        
        else:
            header+=self.addHeader(data)
        if debug_serveFile:
            print "done serving file %s" % filename
        response=header
        return response

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def addHeader(self,data, type="text"):
        '''
    
        '''
        """Add HTTP header."""
        result="HTTP/1.1 200 OK\r\n"
        result+="Server: Pantera Proxy %s\r\n" % pantera.__version__
        result+="Connection: close\r\n"
    
        if type == "text":
            result+="Content-Type: text/html\r\n"
        elif type == "js":
            result+="Content-Type: text/plain\r\n"
        elif type == "image":
            result+="Content-Type: image\r\n"
        elif type == "css":
            result+="Content-Type: text/css\r\n"
        else:
            result+="Content-Type: text/html\r\n"
        result+="Content-Length: %s\r\n" % str(len(data))
        result+="\r\n"
        #print result
                
        result+=data
        return result

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def setConfig(self, myheader, mybody):
        '''
    
        '''
        """Set internal Pantera configurations variables."""
        post_data = mybody.getArgsDict()
        #print post_data
        if post_data.has_key("css_file"):
            self.ConfigData['css_file'] = urllib.unquote_plus(post_data["css_file"])
        if post_data.has_key("user_agent"):
            self.ConfigData['user_agent'] = urllib.unquote_plus(post_data["user_agent"])
        if post_data.has_key("add_string_404") and not post_data["add_string_404"] == "":
            self.Default404List.append(urllib.unquote_plus(post_data["add_string_404"]))
        if post_data.has_key("del_string_404"):
            temp_str = urllib.unquote_plus(post_data["del_string_404"])
            for i in self.Default404List:
                if i == temp_str:
                    self.Default404List.remove(i)
        if post_data.has_key("add_restricted_host") and not post_data["add_restricted_host"] == "":
            self.RestrictedHost.append(urllib.unquote_plus(post_data["add_restricted_host"]))
        if post_data.has_key("del_restricted_host"):
            temp_str = urllib.unquote_plus(post_data["del_restricted_host"])
            for i in self.RestrictedHost:
                if i == temp_str:
                    self.RestrictedHost.remove(i)
        if post_data.has_key("add_restricted_pages") and not post_data["add_restricted_pages"] == "":
            self.RestrictedPages.append(urllib.unquote_plus(post_data["add_restricted_pages"]))
        if post_data.has_key("del_restricted_pages"):
            temp_str = urllib.unquote_plus(post_data["del_restricted_pages"])
            for i in self.RestrictedPages:
                if i == temp_str:
                    self.RestrictedPages.remove(i)
        if post_data.has_key("dont_save_ext"): 
            self.DontSaveExt = panterautils.Str2List(urllib.unquote_plus(post_data["dont_save_ext"]))
        if post_data.has_key("dont_save_content_type"):
            self.DontContentType = panterautils.Str2List(urllib.unquote_plus(post_data["dont_save_content_type"]))
        if post_data.has_key('force_dont_save') and post_data['force_dont_save'] == "yes":
            self.ConfigData['force_dont_save'] = "yes"
        else:
            self.ConfigData['force_dont_save'] = "no"
        if post_data.has_key("user"):
            self.ConfigData['user'] = urllib.unquote_plus(post_data["user"])
        if post_data.has_key("password"):
            self.ConfigData['password'] = urllib.unquote_plus(post_data["password"])
        if post_data.has_key("domain"):
            self.ConfigData['domain'] = urllib.unquote_plus(post_data["domain"])
        if post_data.has_key("proxy_user"):
            self.ConfigData['proxy_user'] = urllib.unquote_plus(post_data["proxy_user"])
        if post_data.has_key("proxy_password"):
            self.ConfigData['proxy_password'] = urllib.unquote_plus(post_data["proxy_password"])
        if post_data.has_key("proxy_domain"):
            self.ConfigData['proxy_domain'] = urllib.unquote_plus(post_data["proxy_domain"])
        if post_data.has_key('verbose') and post_data['verbose'] == "yes":
            self.ConfigData['verbose'] = "yes"
        else:
            self.ConfigData['verbose'] = "no"
        if post_data.has_key('debug') and post_data['debug'] == "yes":
            self.ConfigData['debug'] = "yes"
        else:
            self.ConfigData['debug'] = "no"
        if post_data.has_key('refresh') and post_data['refresh'] == "yes":
            self.ConfigData['refresh'] = "yes"
        else:
            self.ConfigData['refresh'] = "no"
        if post_data.has_key('refresh_seconds'):
            self.ConfigData['refresh_seconds'] = int(urllib.unquote_plus(post_data["refresh_seconds"]))
        #if post_data.has_key('db_login'):
        #    self.ConfigData['db_login'] = urllib.unquote_plus(post_data["db_login"])
        #if post_data.has_key('db_password'):
        #    self.ConfigData['db_password'] = urllib.unquote_plus(post_data["db_password"])
        #if post_data.has_key('db_host'):
        #    self.ConfigData['db_host'] = urllib.unquote_plus(post_data["db_host"])
        #if post_data.has_key("db_name"):
        #    self.ConfigData['db_name'] = urllib.unquote_plus(post_data["db_name"])
        if post_data.has_key('analyzer_threads'):
            self.ConfigData['analyzer_threads'] = int(urllib.unquote_plus(post_data["analyzer_threads"]))
        if post_data.has_key('replace_user_agent') and post_data['replace_user_agent'] == "yes":
            self.ConfigData['replace_user_agent'] = "yes"
        else:
            self.ConfigData['replace_user_agent'] = "no"
        if post_data.has_key('force_auth') and post_data['force_auth'] == "yes":
            self.ConfigData['force_auth'] = "yes"
        else:
            self.ConfigData['force_auth'] = "no"
        if post_data.has_key('force_proxy_auth') and post_data['force_proxy_auth'] == "yes":
            self.ConfigData['force_proxy_auth'] = "yes"
        else:
            self.ConfigData['force_proxy_auth'] = "no"
        if post_data.has_key('reject_return_codes'):
            self.lst_codes = panterautils.Str2List(urllib.unquote_plus(post_data['reject_return_codes']),",")
            
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ReturnClientHeader(self, obj):
        '''
    
        '''
        """Return client header and parse values."""
        str_h = ""

        str_h += "%s %s%s" % (obj.verb,obj.connectHost,obj.URL) 
        args = ""
        if obj.useRawArguments == 0:
            if len(obj.allURLargs) > 0:
                args += "?%s" % obj.allURLargs
            else:
                if len(obj.URLargsDict) > 0:
                    args += "?"
                    args += panterautils.joinargs(obj.URLargsDict,orderlist=obj.orderlist)
        #str_h += "%s %s\\r\\n\"+\r\n" % (args,obj.version)
        str_h += "%s %s\\r\\n" % (args,obj.version)
        if len(obj.headerValuesDict)>0:
            str_h += "\"+\r\n"
        i = len(obj.headerValuesDict)
        z = 1
        for hkey in obj.headerValuesDict.keys():
            for val in obj.headerValuesDict[hkey]:
                if val.count('"')>0:
                    str_h += "\"%s: %s" % (hkey,panterautils.FixString(val)) 
                else:
                    str_h += "\"%s: %s" % (hkey,val) 
                if z < i:
                    str_h += "\\r\\n\"+\r\n"
                    z += 1
        return str_h
        
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ReturnDataForPage(self, file):
        '''
    
        '''
        """Return page body."""
        ch = self.pf.displayClientHeader(file)
        if ch == "" or ch is None:
            return (None, None, None, None)
        args = ""
        if ch.useRawArguments == 0:
            if len(ch.allURLargs) > 0:
                args += "?%s" % ch.allURLargs
            else:
                if len(ch.URLargsDict) > 0:
                    args += "?"
                    args += panterautils.joinargs(ch.URLargsDict,orderlist=ch.orderlist)
    
        realdir=file.replace(self.basedir,"")

        #print "getOptions realdir="+realdir
        site=panterautils.pathsplit(realdir)[0]
        #print "Site=%s"%site
        try: 
            sitename=site.split("_")[0]
            siteport=site.split("_")[1]
            sitessl=site.split("_")[2]=="1"
        except:
            #something bad happened
            #XXX - Must fix this bug
            return "Error at ReturnDataForPage(%s)" % file
        if sitessl:
            site="https://%s" % sitename
        else:
            site="http://%s" % sitename
            #site always uses / because it is a URL now
            site+="/"
            site+="/".join(panterautils.pathsplit(realdir)[1:-1])
            site=site.replace("/_directory_","")
        return (ch.verb, file, site, args) 

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    #supports rewrite!
    #sends the actual request to the remote server!
    def sendrequest(self,myheader,mybody,mode):
        '''
    
        '''
        """Send request to server."""
        result=""
        del_d = []
        itrigger = False        
        
        #new header and body to fill up
        newh=pantera.HTTPHeader()
        newb=pantera.HTTPBody()
        #now disassemble myheader

        #debug
        keys=myheader.URLargsDict.keys()
        #print "Keys: "+str(keys)
        bodyDict=mybody.getArgsDict()
        #print "BodyDict=%s"%str(bodyDict)
        newh.URL=urllib.unquote_plus(bodyDict["URL"])
        if newh.URL == '':
            newh.URL = '/'
        #print "newh.URL="+newh.URL
        newh.verb=urllib.unquote_plus(bodyDict["Verb"])
        if newh.verb == '':
            newh.verb = 'GET'
        newh.connectHost=urllib.unquote_plus(bodyDict["ConnectHost"])
        if newh.connectHost == '':
            return "Error: Missing host!!"
        newh.connectPort=urllib.unquote_plus(bodyDict["ConnectPort"])
        if newh.connectPort == '':
           newh.connectPort = 80 
        if bodyDict.has_key('version'):
            if bodyDict['version']=='':
                newh.version='HTTP/1.0'
            else:
                newh.version=urllib.unquote_plus(bodyDict["version"])
        else:
            newh.version="HTTP/1.1"
        #checkbox, only exists if it is checked
        newh.clientisSSL= bodyDict.has_key("SSL")

        #handle each other
        did=["PANTERA_TRIGGER","URL","Verb","ConnectHost","ConnectPort","SSL","version"]
        firstbodyarg=1
        for akey in bodyDict.keys():
            #filter the ones we already did
            if akey in did:
                if "PANTERA_TRIGGER" == akey:
                    itrigger = True
                continue
    
            #do delete first
            if akey.count('del_'):
                valuename=akey[4:]
                headername=akey
                del_d.append(valuename)
                did.append(valuename)
                did.append(headername)
            
            #is it a header value?
            if akey.count("Header"):
                #names
                if akey[-1]=="N":
                    valuename=akey[:-1]+"V"
                    headername=akey
                else:
                    valuename=akey
                    headername=akey[:-1]+"N"
                    
                header=bodyDict[headername]
                value=bodyDict[valuename]
                #add this to the did list so we don't do it again
                did.append(valuename)
                did.append(headername)

                if header!="":
                    if not newh.headerValuesDict.has_key(header):
                        newh.headerValuesDict[header]=[]
                    newh.headerValuesDict[header].append(urllib.unquote_plus(value))

            if akey.count("URLArg"):
                #names
                if akey[-1]=="N":
                    valuename=akey[:-1]+"V"
                    argname=akey
                else:
                    valuename=akey
                    argname=akey[:-1]+"N"

                arg=urllib.quote_plus(bodyDict[argname])
                value=bodyDict[valuename]
                #add this to the did list so we don't do it again
                did.append(valuename)
                did.append(argname)
                #store it
                if arg!="":
                    t = urllib.unquote_plus(value)
                    newh.URLargsDict[arg]= t.replace(' ','%20') # take care of whitespaces
                    #newh.URLargsDict[arg]=urllib.unquote_plus(value)
            
            if akey.count("Body"):
                #names
                if akey[-1]=="N":
                    valuename=akey[:-1]+"V"
                    argname=akey
                else:
                    valuename=akey
                    argname=akey[:-1]+"N"

                arg=urllib.quote_plus(bodyDict[argname])
                value=bodyDict[valuename]
                #add this to the did list so we don't do it again
                did.append(valuename)
                did.append(argname)
                #storeit
                #print "arg=%s argname=%s valuename=%s"%(arg,argname,valuename)
                #print "Args:%s"%str(myheader.URLargsDict)
                if arg!="":
                    if not firstbodyarg:
                        newb.data.append("&")
                    #we must put characters into the body, not strings
                    newstring=urllib.unquote_plus(arg)+"="+urllib.unquote_plus(value)
                    for ch in newstring:
                        newb.data.append(ch)
                    firstbodyarg=0
         
        # delete data     
        for f in del_d:
            if newh.headerValuesDict.has_key(f):
                del newh.headerValuesDict[f]
            if newh.URLargsDict.has_key(f):
                del newh.URLargsDict[f]
        
        if itrigger == True and mode == 1:
            #print "INTERCEPTOR SAVE HASH"
            #print newh.genhash()
            self.interceptor_urls.append(newh.genhash())
        
        if bodyDict.has_key('del_this_link') and bodyDict['del_this_link'] == 'yes':
            self.interceptor_delete_urls.append(newh.genhash())
        
        if mode == 2:
            self.http_editor_myheader = newh
            self.http_editor_mybody = newb
        
        #ok, so now we have a new header and body (newh, newb)
        #print "newbody=%s"%str(newb.data)
        result=self.makeRequest(newh,newb)
        return result

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    #makes a request - doesn't fork off a new thread
    #takes in a header and body
    #does handle SSL
    #returns a header and body from the server as a string
    def makeRequest(self,newh,newb):
        '''
    
        '''
        """Makes a request."""
        
        #delete request header
        #newh.headerValuesDict = self.DeleteRequestHeader(newh.headerValuesDict)
        
        #replacer request
        #newh.headerValuesDict = self.DoReplacer(newh.headerValuesDict)
        
        #we send ourselves in as the UI for our child request
        myconnection=pantera.PanteraProxyConnection(None,self,proxy=self.proxy,ntlm=self.ntlm)
        myconnection.clientisSSL=newh.clientisSSL
        if newh.clientisSSL:
            myconnection.sslHost=newh.connectHost
            myconnection.sslPort=newh.connectPort
        result=myconnection.sendRequest(newh,newb)
        return result

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def SoftDeletePage(self, fil):
        '''
    
        '''
        """Delete page from internal Pantera cache."""
        
        try:
            f = self.requestCache[int(fil)]
        except Exception:
            return self.ErrorHTMLPage("Invalid Value")                    

        for t in self.requestCache:
            if t == f:
                self.requestCache.remove(f)
        if not f in self.requestCache: # we make sure there arent more links to delete file. This allow multiples links with  one file!
            self.DeleteFile(f)

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def HardDeletePage(self, id):
        '''
    
        '''
        """Delete page from stored and database."""
        file = self.pan_db.Return_Link_File(self.current_project_id, id)
        if file == -1:
            self.LogError(self.pan_db.GetErrorMsg())        
            return
        res = self.pan_db.Delete_Link(self.current_project_id,id)
        if res == -1:
            self.LogError(self.pan_db.GetErrorMsg())
            return
        links = self.pan_db.Return_All_Links(self.current_project_id)
        if links == -1:
            self.LogError(self.pan_db.GetErrorMsg())
            return
        l = []
        append = l.append
        for li in links:
            append(li[1])
        if not file in l: # we check for multiple links if none delete file
            self.DeleteFile(file)            
        return

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def setProject(self, myheader,mybody):
        '''
    
        '''
        """Creates, opens or deletes projects."""
        w = 0
        post_data = mybody.getArgsDict()
        if post_data.has_key('project_status') and post_data['project_status']=='new': # new project
            temp = urllib.unquote_plus(post_data["project_name"])
            if temp == '':
                return w
            temp = panterautils.FilterBadCar(temp)
            self.ConfigData['project_name'] = temp
            res = self.pan_db.Insert_New_Project(self.ConfigData['project_name'])
            if res == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            self.inside_project = YES
            self.ppa_plugins.DeletePlugins()
            self.ppa_plugins.LoadPlugins() # load PPA plugins
            self.ppa_plugin_list = []
            self.ppa_plugin_list = self.ppa_plugins.GetPlugins()
            self.current_project_id = self.pan_db.ReturnLastProjectID()
            if self.current_project_id == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            res = self.pan_db.Set_ProjectID(self.current_project_id)
            if res == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            if len(self.requestCache)>0:
                for t in self.requestCache:
                    infile=open(t,"rb")
                    obj=cPickle.load(infile)
                    infile.close()
                    self.DeleteFile(t)                     
                    # now, register link
                    self.registerRequestandResponse(obj.clientheader,obj.clientbody,obj.serverheader,obj.serverbody)
                    time.sleep(0.8) # time for the PPA magic to work...
                self.CleanTempCache()
            else:
                self.DeleteLog()
                self.SetCounterZero()
            w = 1
        elif post_data.has_key('action_project') and post_data['action_project']=='open_project': # open project
            pid = urllib.unquote_plus(post_data["do_project"])
            res = self.pan_db.Set_ProjectID(pid)
            if res == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            self.OpenProject(pid)
            w = 1
        elif post_data.has_key('action_project') and post_data['action_project']=='delete_project': # delete project
            pid = urllib.unquote_plus(post_data["do_project"])
            links = self.pan_db.Return_All_Links(pid)
            if links == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            for l in links:
                self.DeleteFile(l[1])
                self.DeleteDir(l[1])
            if int(pid) == int(self.current_project_id):
                self.CloseProject(pid)
            res = self.pan_db.Delete_Project(pid)
            if res == -1:
                self.LogError(self.pan_db.GetErrorMsg())
        return w    

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def CloseProject(self, pid):
        '''
    
        '''
        """Close project."""
        if self.inside_project == YES:
            self.setNoProjectName()
            self.inside_project = NO
            self.current_project_id = 0
            self.CleanTempCache()
        
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT :     
    def OpenProject(self, pid):
        '''
    
        '''
        """Open project."""
        self.inside_project = YES
        self.current_project_id = pid
        self.ppa_plugins.DeletePlugins()
        self.ppa_plugins.LoadPlugins() # load PPA plugins
        self.ppa_plugin_list = []
        self.ppa_plugin_list = self.ppa_plugins.GetPlugins()
        # set links into database
        if len(self.requestCache)>0:
            for t in self.requestCache:
                infile=open(t,"rb")
                obj=cPickle.load(infile)
                infile.close()
                self.DeleteFile(t)                     
                # now, register link
                self.registerRequestandResponse(obj.clientheader,obj.clientbody,obj.serverheader,obj.serverbody)
                time.sleep(0.8) # time for the PPA magic to work...
        else:
            self.DeleteLog()
            self.SetCounterZero()
        c = self.pan_db.Return_ProjectName(self.current_project_id)
        if c == -1:
            self.LogError(self.pan_db.GetErrorMsg())
        c = c[0]
        self.ConfigData['project_name'] = c[0]

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DoPanteraManagement(self, post_data):
        '''
    
        '''
        """Deletes the log file or the cache."""
        if post_data.has_key('del_log_file') and post_data['del_log_file']=='yes':
            self.DeleteLog()
        if post_data.has_key('del_temp_cache') and post_data['del_temp_cache']=='yes':
            self.CleanTempCache()
        if post_data.has_key('del_error_log_file') and post_data['del_error_log_file']=='yes':
            self.DeleteErrorLog()
        if post_data.has_key('del_session_dir') and post_data['del_session_dir']=='yes':
            self.CleanTempCache()
            self.DeleteDir(self.basedir)
        if post_data.has_key('del_inter_cache') and post_data['del_inter_cache'] == 'yes':
            self.CleanInterceptorCache()

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DeleteRequestHeader(self, hd):
        '''
    
        '''
        """Delete request header."""
        # check & delete headers 
        if len(self.supress_request_header) > 0:
            for h in self.supress_request_header:
                if hd.has_key(h):
                    del hd[h]
        return hd

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DeleteResponseHeader(self, hd):
        '''
    
        '''
        """Delete response header."""
        # check & delete headers 
        if len(self.supress_response_header) > 0:
            for h in self.supress_response_header:
                if hd.has_key(h):
                    del hd[h]
        return hd

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ReturnResourceIcons(self, id,a):
        '''
    
        '''
        self.str_page = ""
        dat = {}
        for x in a: 
            if x['link_id'] == id:
                dat = {}
                dat = x
        if dat.has_key('page_date') and dat.has_key('page_time')  :
            self.str_page += "<b>[%s | %s]</b>" % (dat['page_date'],dat['page_time'])                
        if dat.has_key('have_ssl') and dat['have_ssl'] == 1:
            self.str_page += "<img src=\"./img/lock_icon.png\" name=\"SSL\" alt=\"SSL\">"
        if dat.has_key('have_auth') and dat['have_auth'] == 1:        
            self.str_page += "<img src=\"./img/icon_auth.gif\" name=\"Authentication\" alt=\"Authentication\">"
        if dat.has_key('have_email') and dat['have_email'] == 1:    
            self.str_page += "<img src=\"./img/icon_email.png\" name=\"Email\" alt=\"Email\">"
        if dat.has_key('have_script') and dat['have_script'] == 1:
            self.str_page += "<img src=\"./img/icon_script.png\" name=\"Script\" alt=\"Script\">"
        if dat.has_key('have_form') and dat['have_form'] == 1:
            self.str_page += "<img src=\"./img/icon_form.png\" name=\"Form\" alt=\"Form\">"
        if dat.has_key('have_auth_form') and dat['have_auth_form'] == 1:
            self.str_page += "<img src=\"./img/icon_authform.png\" name=\"Auth Form\" alt=\"Auth Form\">"
        if dat.has_key('have_hidden') and dat['have_hidden'] == 1:
            self.str_page += "<img src=\"./img/icon_hidden.png\" name=\"Hidden\" alt=\"Hidden\">"
        if dat.has_key('have_cookie') and dat['have_cookie'] == 1:
            self.str_page += "<img src=\"./img/icon_cookie.png\" name=\"Cookie\" alt=\"Cookie\">"   
        if dat.has_key('have_session_id') and dat['have_session_id'] == 1:          
            self.str_page += "<img src=\"./img/icon_session_id.gif\" name=\"Session ID\" alt=\"Session ID\">"        
        if dat.has_key('have_comment') and dat['have_comment'] == 1:        
            self.str_page += "<img src=\"./img/icon_comment.png\" name=\"Comment\" alt=\"Comment\">"
        if dat.has_key('have_querystr') and dat['have_querystr'] == 1:
            self.str_page += "<img src=\"./img/icon_querystr.png\" name=\"URI Params\" alt=\"URI Params\">"
        if dat.has_key('have_external_link') and dat['have_external_link'] == 1:     
            self.str_page += "<img src=\"./img/url.png\" name=\"External Links\" alt=\"External Links\">"
        if dat.has_key('have_object') and dat['have_object'] == 1:
            self.str_page += "<img src=\"./img/icon_activex.png\" name=\"Object\" alt=\"Object\">"
        if dat.has_key('have_postauth') and dat['have_postauth'] == 1:        
            self.str_page += "<img src=\"./img/icon_postauth.png\" name=\"Post Auth\" alt=\"Post Auth\">"
        if dat.has_key('have_vuln') and dat['have_vuln'] == 1:        
            self.str_page += "<img src=\"./img/icon_target.png\" name=\"Attack Vector\" alt=\"Attack Vector\">"
        return self.str_page

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DoInsertProjectNotesData(self, post_data):
        '''
    
        '''
        if self.inside_project == YES:
            if post_data.has_key('project_notes'):
                ldat = urllib.unquote_plus(post_data['project_notes'])
                ldat = panterautils.FilterBadCar(ldat,1) 
                res = self.pan_db.Update_Project_Notes(self.current_project_id,ldat)                
                if res == -1:
                    self.LogError(self.pan_db.GetErrorMsg())

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DoInsertLinkNotesData(self, post_data):
        '''
    
        '''
        if self.inside_project == YES:
            if post_data.has_key('link_notes') and post_data.has_key('id'):
                ldat = urllib.unquote_plus(post_data['link_notes'])
                ldat = panterautils.FilterBadCar(ldat,1) 
                res = self.pan_db.Update_Link_Notes(self.current_project_id,urllib.unquote_plus(post_data['id']),ldat)                
                if res == -1:
                    self.LogError(self.pan_db.GetErrorMsg())
    
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DoPPAManagement(self, post_data):
        '''
    
        '''
        self.ppa_plugin_list = []
        for mod in self.ppa_plugins.GetPlugins():
            mod_dat = self.ppa_plugins.GetPluginData(mod)
            id_class = self.ppa_plugins.GetPluginClass(mod,mod_dat['id'])
            f,v = string.split(str(id_class),".")
            for k in post_data.keys():
                if k == v: # plugin yes
                    self.ppa_plugin_list.append(mod)
        
        self.ppa_plugins.DeletePlugins()
        for p in self.ppa_plugin_list:
            self.ppa_plugins.AddPlugin(p)

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DoReplacer(self, data, mode=0):
        '''
    
        '''
        pass
        
    def ReturnReplacerList(self):
        '''
    
        '''
        return self.replacer_list

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DoInterceptor(self, myheader):
        '''
    
        '''
        
        #print myheader.URL
        #print myheader.URLargsDict
        #print myheader.headerValuesDict 
        
        urlfile=myheader.URL.split("/")[-1]
        ext = urlfile.split(".")[-1]
        ext = ext.lower()
        if ext in self.DontSaveExt: return 0
        
        hash = myheader.genhash()    
        #print "INTERCEPTOR HASH"
        #print hash
        data = myheader.RecontructHeader(myheader)
            
        for c in self.interceptor_list:
            p = re.compile(c['regex'])
            r = re.search(p,data)
            if myheader.URL =='/send_rewrite':
                return 1
            if r and not myheader.URLargsDict.has_key('PANTERA_TRIGGER') and not hash in self.interceptor_urls:
                myheader.SetRewrite()
                #print "WE GOT MATCH"
                return 1
        return 0

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DeleteInterceptorURL(self, myheader):
        '''
    
        '''
        hash = myheader.genhash()
        if hash in self.interceptor_delete_urls:
            self.interceptor_delete_urls.remove(hash)
            self.interceptor_urls.remove(hash)
        return

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def AddHashHeaderInterceptor(self,myheader):
        '''
    
        '''
        hash = myheader.genhash()
        self.interceptor_urls.append(hash)        

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DataMinerProjectInfoVisor(self, dat_list, name=""):
        '''
    
        '''
        my_str = ""

        my_str += "<form action=\"querier_form\" method=\"POST\">\r\n"
        my_str += "<input type=\"submit\" name=\"do_action\" value=\"Delete\" > | "
        my_str += "<input type=\"submit\" name=\"do_action\" value=\"Update\" > | "
        my_str += "<input type=\"checkbox\" name=\"checkall\" onclick=\"checkUncheckAll(this);\"/>Select/Unselect All |"
        my_str += " [ <a class=\"textlink\" href=\"querier_link\">New Search</a> ]"
        my_str += "<br><br>\r\n"
        my_str += "<table>\r\n"
        
        for i in dat_list:
            my_str += "<tr>\r\n"
            my_str += "<td><input type=\"checkbox\" name=\"action_%s\" value=\"%s\"></td>\r\n" %(i['project_info_id'],i['project_info_id'])
            do = "domain_%s" % i['project_info_id']
            va = "value_%s" % i['project_info_id']
            my_str += "%s" % panterautils.printFormEntry("domain",do,i['domain'],1,20)
            my_str += "%s" % panterautils.printFormEntry(name,va,i['value'],1,75)
            my_str += "</tr>\r\n"

        my_str += "</table>\r\n"
        my_str += "</form>\r\n"
        return my_str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DataMinerSerachLinksWithVisor(self, dat_list):
        '''
    
        '''
        my_str = ""

        my_str += "<form action=\"querier_form\" method=\"POST\">\r\n"
        my_str += "<input type=\"submit\" name=\"do_action_file_delete\" value=\"Delete\" > | "
        my_str += "<input type=\"checkbox\" name=\"checkall\" onclick=\"checkUncheckAll(this);\"/>Select/Unselect All |"
        my_str += " [ <a class=\"textlink\" href=\"querier_link\">New Search</a> ]"
        my_str += "<br><br>\r\n"
        my_str += "<table>\r\n"
    
        a = self.pan_db.Select_All_Page_Info(self.current_project_id)
        if a == -1:
            self.LogError(self.pan_db.GetErrorMsg())
            return ""
        counter = 1
        for l in dat_list:
            (verb, file, site, args) = self.ReturnDataForPage(l[1])
            if verb is None or file is None or site is None or args is None:
                continue
            uri_t = panterautils.TruncateURL(site+args,115)
            my_str += "<tr>"
            my_str += "<td><input type=\"checkbox\" name=\"deletefile_%s\" value=\"%s\"> <b>%s</b></td><td> >> %s <a CLASS=\"textlink\" href=\"direct_link_info?file=%s\">%s</a></td>\r\n" % (l[0],l[0],`counter`,panterautils.FilterBadCar(verb,2),l[0],panterautils.FilterBadCar(uri_t,2))
            my_str += "</tr>"
            my_str += "<tr>"
            my_str += "<td></td><td> << %s</td>\r\n" % panterautils.FilterBadCar(self.pf.ReturnServerFirstLine(file),2)
            my_str += "</tr>\r\n"
            my_str += "<tr>\r\n"
            my_str += "<td></td><td>"
            # Add icon resources
            my_str += self.ReturnResourceIcons(l[0],a)
            my_str += "</td>"
            my_str += "</tr>\r\n"
            my_str += "<tr>"
            my_str += "<td></td><td>"
            my_str += "<a class=\"textlink\" href=\"direct_link_info?file=%s\"><img src=\"./img/info_button.gif\" border=\"0\" alt=\"Info\"></a> | " % l[0]
            my_str += "<a class=\"textlink\" href=\"editor_link?file=%s\"><img src=\"./img/edit_send_button.gif\" border=\"0\" alt=\"Edit&Send\"></a> | " % l[0]
            self.b = ""
            self.b = self.pf.ReturnPageBody(file)
            if self.b != "":
                my_str += "<a class=\"textlink\" href=\"body_page_link?file=%s\"><img src=\"./img/page_body_button.gif\" border=\"0\" alt=\"Page Body\"></a> | " % l[0]
            else:
                my_str += "<img src=\"./img/empty_button.gif\" border=\"0\" alt=\"No body\"></a> | "
            my_str += "<a class=\"textlink\" href=\"delete_link?file=%s\"><img src=\"./img/delete_button.gif\" border=\"0\" alt=\"Delete\"></a> | " % l[0]   
            my_str += "<a class=\"textlink\" href=\"notes_link?file=%s\"><img src=\"./img/notes_button.gif\" border=\"0\" alt=\"Notes\"></a>" % l[0]   
            my_str += "</td>\r\n"
            my_str += "</tr>\r\n"
            counter += 1

        my_str += "</table>\r\n"
        my_str += "</form>\r\n"
        return my_str
    
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DataMinerEntireLinksVisor(self, dat_list, dat_list1):
        '''
    
        '''
        my_str = ""

        my_str += "<form action=\"querier_form\" method=\"POST\">\r\n"
        my_str += "<input type=\"submit\" name=\"do_action_file_delete\" value=\"Delete\" > | "
        my_str += "<input type=\"checkbox\" name=\"checkall\" onclick=\"checkUncheckAll(this);\"/>Select/Unselect All |"
        my_str += " [ <a class=\"textlink\" href=\"querier_link\">New Search</a> ]"
        my_str += "<br><br>\r\n"
        my_str += "<table>\r\n"
        
        for i in dat_list:
            (verb, file, site, args) = self.ReturnDataForPage(i[1])
            my_str += "<tr>\r\n"
            my_str += "<td><input type=\"checkbox\" name=\"deletegood_%s\" value=\"%s\"></td>\r\n" %(i[0],i[0])
            my_str += "%s " % panterautils.printFormEntry("Link",'not',site+args,1,100)
            my_str += "<td><a class=\"textlink\" href=\"direct_link_info?file=%s\">Info</a></td>\r\n" % i[0]
            my_str += "</tr>\r\n"
        for i in dat_list1:
            my_str += "<tr>\r\n"
            my_str += "<td><input type=\"checkbox\" name=\"deletebad_%s\" value=\"%s\"></td>\r\n" %(i[0],i[0])
            my_str += "%s" % panterautils.printFormEntry("Link",'not',i[1],1,100)
            my_str += "<td><a class=\"textlink\" href=\"insert_file_link?file=%s\">Insert into Testing View</a></td>\r\n" % i[0]
            my_str += "</tr>\r\n"

        my_str += "</table>\r\n"
        my_str += "</form>\r\n"
        return my_str
        
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def DataMinerSearchCriteria(self, dat_list, dat_list1):
        '''
    
        '''
        my_str = ""

        my_str += "<form action=\"querier_form\" method=\"POST\">\r\n"
        my_str += "<input type=\"submit\" name=\"do_action_file_delete\" value=\"Delete\" > | "
        my_str += "<input type=\"checkbox\" name=\"checkall\" onclick=\"checkUncheckAll(this);\"/>Select/Unselect All |"
        my_str += " [ <a class=\"textlink\" href=\"querier_link\">New Search</a> ]"
        my_str += "<br><br>\r\n"
        my_str += "<table>\r\n"
        
        if dat_list != -1 and dat_list != '' and len(dat_list)>0:
            for i in dat_list:
                my_str += "<tr>\r\n"
                my_str += "<td><input type=\"checkbox\" name=\"deletegood_%s\" value=\"%s\"></td>\r\n" %(i['id'],i['id'])
                my_str += "%s" % panterautils.printFormEntry("Method:",'not',i['method'],1,7)
                my_str += "%s" % panterautils.printFormEntry("Link:",'not',i['url'],1,70)
                my_str += "%s" % panterautils.printFormEntry("Return Code:",'not',i['return_code'],1,5)
                my_str += "<td><a class=\"textlink\" href=\"direct_link_info?file=%s\">Info</a></td>\r\n" % i['id']
                my_str += "</tr>\r\n"                
                
        if dat_list1 != -1 and dat_list1 != '' and len(dat_list1)>0:                
            for i in dat_list1:
                my_str += "<tr>\r\n"
                my_str += "<td><input type=\"checkbox\" name=\"deletebad_%s\" value=\"%s\"></td>\r\n" %(i['id'],i['id'])
                my_str += "%s" % panterautils.printFormEntry("Method:",'not',i['method'],1,7)
                my_str += "%s" % panterautils.printFormEntry("Link:",'not',i['url'],1,70)
                my_str += "%s" % panterautils.printFormEntry("Return Code:",'not',i['return_code'],1,5)
                my_str += "<td><a class=\"textlink\" href=\"insert_file_link?file=%s\">Insert into Testing View</a></td>\r\n" % i['id']
                my_str += "</tr>\r\n"                
                
        my_str += "</table>\r\n"
        my_str += "</form>\r\n"
        return my_str
        
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def SetPageResource(self, args):
        '''
    
        '''
        """ """
        dat = {}
        
        if args.has_key('id'):
            dat['page_info_id'] = args['id']
        else:        
            return 0
        
        if args.has_key('ssl'):
            if args['ssl'] == '1': 
                dat['have_ssl'] = 1
            elif args['ssl'] == '0':
                dat['have_ssl'] = 0
            else:
                dat['have_ssl'] = 0
        else:
            dat['have_ssl'] = 0
            
        if args.has_key('email'):
            if args['email'] == '1':
                dat['have_email'] = 1
            elif  args['email'] == '0':  
                dat['have_email'] = 0
            else:
                dat['have_email'] = 0
        else:
            dat['have_email'] = 0

        if args.has_key('script'):
            if args['script'] == '1':
                dat['have_script'] = 1
            elif args['script'] == '0':
                dat['have_script'] = 0
            else:
                dat['have_script'] = 0
        else:
            dat['have_script'] = 0

        if args.has_key('form'):
            if args['form'] == '1':
                dat['have_form'] = 1
            elif args['form'] == '0':
                dat['have_form'] = 0
            else:
                dat['have_form'] = 0
        else:
            dat['have_form'] = 0

        if args.has_key('auth_form'):
            if args['auth_form'] == '1':
                dat['have_auth_form'] = 1
            elif args['auth_form'] == '0':
                dat['have_auth_form'] = 0
            else:
                dat['have_auth_form'] = 0
        else:
            dat['have_auth_form'] = 0

        if args.has_key('cookie'):
            if args['cookie'] == '1':
                dat['have_cookie'] = 1
            elif args['cookie'] == '0':
                dat['have_cookie'] = 0
            else:
                dat['have_cookie'] = 0
        else:
            dat['have_cookie'] = 0

        if args.has_key('session_id'):
            if args['session_id'] == '1':
                dat['have_session_id'] = 1
            elif args['session_id'] == '0':
                dat['have_session_id'] = 0
            else:
                dat['have_session_id'] = 0
        else:
            dat['have_session_id'] = 0

        if args.has_key('external_link'):
            if args['external_link'] == '1':
                dat['have_external_link'] = 1
            elif args['external_link'] == '0':
                dat['have_external_link'] = 0
            else:
                dat['have_external_link'] = 0
        else:
            dat['have_external_link'] = 0

        if args.has_key('comment'):
            if args['comment'] == '1':
                dat['have_comment'] = 1
            elif args['comment'] == '0':
                dat['have_comment'] = 0
            else:
                dat['have_comment'] = 0
        else:
            dat['have_comment'] = 0

        if args.has_key('vuln'):
            if args['vuln'] == '1':
                dat['have_vuln'] = 1
            elif args['vuln'] == '0':
                dat['have_vuln'] = 0
            else:
                dat['have_vuln'] = 0
        else:
            dat['have_vuln'] = 0

        if args.has_key('hidden'):
            if args['hidden'] == '1':
                dat['have_hidden'] = 1
            elif args['hidden'] == '0':
                dat['have_hidden'] = 0
            else:
                dat['have_hidden'] = 0
        else:
            dat['have_hidden'] = 0

        if args.has_key('object'):
            if args['object'] == '1':
                dat['have_object'] = 1
            elif args['object'] == '0':
                dat['have_object'] = 0
            else:
                dat['have_object'] = 0
        else:
            dat['have_object'] = 0

        if args.has_key('postauth'):
            if args['postauth'] == '1':
                dat['have_postauth'] = 1
            elif args['postauth'] == '0':
                dat['have_postauth'] = 0
            else:
                dat['have_postauth'] = 0
        else:
            dat['have_postauth'] = 0

        if args.has_key('querystr'):
            if args['querystr'] == '1':
                dat['have_querystr'] = 1
            elif args['querystr'] == '0':
                dat['have_querystr'] = 0
            else:
                dat['have_querystr'] = 0
        else:
            dat['have_querystr'] = 0

        if args.has_key('auth'):
            if args['auth'] == '1':
                dat['have_auth'] = 1
            elif args['auth'] == '0':
                dat['have_auth'] = 0
            else:
                dat['have_auth'] = 0
        else:
            dat['have_auth'] = 0
            
        r = self.pan_db.Update_Project_Info(self.current_project_id, dat)
        if r == -1:
            self.LogError(self.pan_db.GetErrorMsg())

        return
        
    def InsertFileIntoView(self, i):
        '''
    
        '''
        r = self.pan_db.Return_Not_Link_Save_File(self.current_project_id, i)
        if r == -1: return -1
        infile=open(r,"rb")
        obj=cPickle.load(infile)
        infile.close()
        self.DeleteFile(r)              
        r = self.pan_db.Delete_Not_Save_link(self.current_project_id, i)
        if r == -1: return -1
        # now, register link
        self.registerRequestandResponse(obj.clientheader,obj.clientbody,obj.serverheader,obj.serverbody,1)
        time.sleep(0.2)
        return
        
    def VerifyTarget(self, host):
        '''
    
        '''
        d = {}
        do_val = YES
        sav_val = YES
        r = 0
        
        # check domain
        for val in self.TargetList:
            if val['domain'] in host:
                if val['action'] == 'no': do_val = NO
                else: do_val = YES
                if val['save'] == 'no': sav_val = NO
                else: sav_val = YES
                r = 1
        if r == 1: return do_val, sav_val # domain allowed and save
                                
        # check any rule
        for val in self.TargetList:
            if val['domain'] == 'any':
                if val['action'] == 'no': do_val = NO
                else: do_val = YES
                if val['save'] == 'no': sav_val = NO
                else: sav_val = YES
                return do_val, sav_val # domain allowed and save
                
        return do_val, sav_val # domain allowed and save
        
    def GetCounter(self):
        '''
    
        '''
        return self.iCounter 
        
    def AddCounter(self):
        '''
    
        '''
        self.iCounter += 1
        
    def SetCounterZero(self):
        '''
    
        '''
        self.iCounter = 0
        
####################################################################################################3
# on-the-fly html pages
####################################################################################################3

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def IndexHTMLPage(self):
        '''
    
        '''
        """UI main page. (home)"""
        str = ""
        str1 = ""
        html = panteraHTML.PanteraHTML()

        #html.SetDataHeader("<META HTTP-EQUIV=\"Expires\" CONTENT=\"-1\">")

        # Set automatic page refresh
        if self.ConfigData.has_key('refresh') and self.ConfigData['refresh'] == 'yes':
            html.SetDataHeader("<meta http-equiv=\"refresh\" content=%d>" % self.ConfigData['refresh_seconds'])

        str += html.CreateHeader("OWASP / Roses Labs Pantera Web Assessment Studio (WAS)")

        if self.inside_project == YES: # From DDBB
            time.sleep(0.2) # time for the PPA magic to work...
            # Check for not save links
            self.CheckNotSaveLinks(1)
            #start()
            counter_url = 0
            ic = 0
            links = self.pan_db.Return_All_Links(self.current_project_id)
            if links == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            a = self.pan_db.Select_All_Page_Info(self.current_project_id)
            if a == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            link_len = len(links)
            if link_len > 10:
                counter_url = 10
            else:
                counter_url = link_len
            str1 ="<h2>Last %s Requests</h2>\r\n" % `counter_url`
            # FIX THIS!!!!
            links.reverse()
            for l in links:
                if ic >= counter_url:
                    break
                (verb, file, site, args) = self.ReturnDataForPage(l[1])
                uri_t = panterautils.TruncateURL(site+args,80)
                str1 += ">> %s <a class=\"textlink\" href=\"direct_link_info?file=%s\">%s</a><br>\r\n" % (panterautils.FilterBadCar(verb,2),l[0],panterautils.FilterBadCar(uri_t,2))
                str1 += "<< " + panterautils.FilterBadCar(self.pf.ReturnServerFirstLine(file),2)+"<br>\r\n"
                str1 += self.ReturnResourceIcons(l[0],a)
                str1 += "<hr>"
                ic += 1
            #finish()
            #print "time" 
            #print seconds() 
            #print milli()
            #print micro()
        else:                          # from temporal cache
            # it should be more elegant...
            topURL = []
            counter_url = 0
            for t in self.requestCache:
                topURL.append(t)
                counter_url += 1
                if counter_url >= 10:
                    break
            str1 ="<h2>Last %s Requests (Reading from temporal cache)</h2>\r\n" % `counter_url`

            for f in topURL:
                (verb, file,site,args) = self.ReturnDataForPage(f)
                if verb is None or file is None or site is None or args is None:
                    continue
                uri_t = panterautils.TruncateURL(site+args,80)
                str1 += ">> %s <a class=\"textlink\" href=\"direct_link_info?file=%s\">%s</a><br>\r\n" % (panterautils.FilterBadCar(verb,2),self.requestCache.index(f),panterautils.FilterBadCar(uri_t,2))
                str1 += "<< " + panterautils.FilterBadCar(self.pf.ReturnServerFirstLine(file),2)+"\r\n"
                str1 += "<hr>"
        
        html.SetDataRigth(str1)

        str += html.CreateHomePage(self.ConfigData['project_name'])                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ConfigHTMLPage(self):
        '''
    
        '''
        """UI configuration page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Pantera Config",1)                

        # create our config page

        #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br>\r\n"
        str_page += "<form action=\"setconfig\" method=\"POST\">\r\n"
        str_page += "<table>\r\n"
        str_page += "<tr><td><h2>General Options</h2></td></tr>\r\n"
        str_page +="<tr><td></td></tr>\r\n"

        if self.ConfigData['replace_user_agent'] == "yes":
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Replace User Agent","replace_user_agent",1,1)
        else:
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Replace User Agent","replace_user_agent",0,1)

        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("User Agent","user_agent",self.ConfigData['user_agent'],1,50)

        if self.ConfigData['force_dont_save'] == "yes":
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Force Dont Save","force_dont_save",1,1)
        else:
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Force Dont Save","force_dont_save",0,1)

        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Dont Save Extension:","dont_save_ext",panterautils.List2Str(self.DontSaveExt,","),1,50)

        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Dont Save Content-Type:","dont_save_content_type",panterautils.List2Str(self.DontContentType,","),1,50)
        
        sstr = panterautils.List2Str(self.lst_codes,",")
        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Reject Return Codes","reject_return_codes",sstr,1,50)        

        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Add 404 Detection Pattern","add_string_404","",1,50)

        str_page += "<tr><td>Delete 404 Detection Pattern:</td><td></td></tr>\r\n"
        for v in self.Default404List:
            str_page += "<tr><td></td><td><input type=\"radio\" name=\"del_string_404\" value=\"%s\">%s</td></tr>\r\n" % (v,panterautils.FilterBadCar(v,2))

        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Add Restricted Host","add_restricted_host","",1,50)

        str_page += "<tr><td>Delete Restricted Host:</td><td></td></tr>\r\n"
        for v in self.RestrictedHost:
            str_page += "<tr><td></td><td><input type=\"radio\" name=\"del_restricted_host\" value=\"%s\">%s</td></tr>\r\n" % (v,panterautils.FilterBadCar(v,2))

        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Add Restricted Page","add_restricted_pages","",1,50)

        str_page += "<tr><td>Delete Restricted Page:</td><td></td></tr>\r\n"
        for v in self.RestrictedPages:
            str_page += "<tr><td></td><td><input type=\"radio\" name=\"del_restricted_pages\" value=\"%s\">%s</td></tr>\r\n" % (v,panterautils.FilterBadCar(v,2))

        str_page +="<tr><td></td></tr>\r\n"
        str_page += "<tr><td><h3>Authentication</h3></td></tr>\r\n"
        str_page +="<tr><td></td></tr>\r\n"

        if self.ConfigData['force_auth'] == "yes":
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Force Authentication","force_auth",1,1)
        else:
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Force Authentication","force_auth",0,1)

        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("User","user",self.ConfigData['user'],1)
        #str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Password","password",self.ConfigData['password'],1)
        str_page += "<tr><td>%s:</td><td><input type=\"password\" name=\"%s\" value=\"%s\" size=\"15\"></td>\r\n" % ("Password", "password", self.ConfigData['password']) 
        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Domain","domain",self.ConfigData['domain'],1,50)
        
        if self.ConfigData['force_proxy_auth'] == "yes":
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Force Proxy Authentication","force_proxy_auth",1,1)
        else:
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Force Proxy Authentication","force_proxy_auth",0,1)

        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Proxy User","proxy_user",self.ConfigData['proxy_user'],1)
        #str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Proxy Password","proxy_password",self.ConfigData['proxy_password'],1)
        str_page += "<tr><td>%s:</td><td><input type=\"password\" name=\"%s\" value=\"%s\" size=\"15\"></td>\r\n" % ("Proxy Password", "proxy_password", self.ConfigData['proxy_password']) 
        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Proxy Domain","proxy_domain",self.ConfigData['proxy_domain'],1,50)        
        
        str_page +="<tr><td></td></tr>\r\n"
        str_page += "<tr><td><h3>Misc</h3></td></tr>\r\n"
        str_page +="<tr><td></td></tr>\r\n"

        if self.ConfigData['verbose'] == "yes":
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Verbose","verbose",1,1)
        else:
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Verbose","verbose",0,1)

        if self.ConfigData['debug'] == "yes":
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Debug","debug",1,1)
        else:
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Debug","debug",0,1)

        if self.ConfigData['refresh'] == "yes":
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Refresh Page","refresh",1,1)
        else:
            str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Refresh Page","refresh",0,1)

        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Refresh Page Seconds","refresh_seconds",self.ConfigData['refresh_seconds'],1,5)
        
        #str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("DDBB User","db_login",self.ConfigData['db_login'],1,30)
        
        #str_page += "<tr><td>%s:</td><td><input type=\"password\" name=\"%s\" value=\"%s\" size=\"30\"></td>\r\n" % ("DDBB Password", "db_password", self.ConfigData['db_password']) 
        
        #str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("DDBB Host","db_host",self.ConfigData['db_host'],1,30)
            
        #str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("DDBB Name","db_name",self.ConfigData['db_name'],1,30)
        
        str_page +="<tr><td></td></tr>\r\n"
        str_page += "<tr><td><h2>Plugins Options</h2></td></tr>\r\n"
        str_page +="<tr><td></td></tr>\r\n"
        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("PPA Threads","analyzer_threads",self.ConfigData['analyzer_threads'],1,5)
  
        str_page += "<tr>\r\n"

        str_page += "<tr>\r\n"
        str_page +="<tr><td></td></tr>\r\n"
        str_page += "<td><h2>Look&Feel</h2></td>\r\n"
        str_page +="<tr><td></td></tr>\r\n"
        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("CSS File","css_file",self.ConfigData['css_file'],1,30)
        str_page +="<tr><td></td></tr>\r\n"
        str_page += "</tr>\r\n"
        
        str_page += "</table>\r\n"

        str_page += "<table>\r\n"
        #str_page += "<tr><td>[<a href=\"javascript: history.go(-1)\">Back</a>]<br></td></tr><tr>\r\n"
        str_page +="<td><input type=\"submit\"></td>\r\n"
        str_page +="<td><input type=\"reset\"></td>\r\n"
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        str_page +="</form>\r\n"

        str += html.CreatePage("Pantera Configuration", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ProjectHTMLPage(self):
        '''
    
        '''
        """UI project page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        if self.flag_ddbb == NO: # Error: No DDBB
            return self.ErrorHTMLPage('Pantera was unable to connect to DDBB. Make sure your DDBB is up and running...')        
        
        str += html.CreateHeader("Project Management",1) 

        str_page += "<table border=0>\r\n"
        
        str_page += "<tr>\r\n"
        str_page += "<td><h2>New Project</h2></td>\r\n"
        str_page += "</tr>\r\n"

        str_page += "<tr>\r\n"
        str_page += "<form action=\"setproject\" method=\"POST\">\r\n"
        str_page += "<table border=0>\r\n"
        str_page += "<tr>%s</tr>\r\n" % panterautils.printHiddenEntry("project_status","new",1)
        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Project Name","project_name","",1,30)
        str_page += "</table>\r\n"    
        str_page += "<table><tr>\r\n"
        str_page +="<td><input type=\"submit\"></td>\r\n"
        str_page +="<td><input type=\"reset\"></td>\r\n"
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        str_page += "</form>\r\n"
        str_page += "</tr>\r\n"

        r = self.pan_db.Select_All_Project()
        if r == -1:
            self.LogError(self.pan_db.GetErrorMsg())
        if r: # if we got projects, show them
            str_page += "<tr>\r\n"
            str_page += "<td><h2>Available Projects</h2></td>\r\n"
            str_page += "</tr>\r\n"

            str_page += "<tr>\r\n"
            str_page += "<form action=\"setproject\" method=\"POST\">\r\n"
            str_page += "<table border=0>\r\n"
            str_page += "<tr>\r\n"
            str_page += "%s" % panterautils.printFormSelectMenuID("Select Project:","do_project",r,1)
            str_page += "<td><input type=radio name=\"action_project\" value=\"open_project\" checked>Open</td>"
            str_page += "<td><input type=radio name=\"action_project\" value=\"delete_project\">Delete</td>\r\n"
            str_page += "</tr>\r\n"
            str_page += "</table>\r\n"
            str_page += "<table><tr>\r\n"
            str_page +="<td><input type=\"submit\"></td>\r\n"
            str_page +="<td><input type=\"reset\"></td>\r\n"
            str_page += "</tr>\r\n"
            str_page += "</table>\r\n"
            str_page += "</form>\r\n"
    
        str_page += "</table>\r\n"
        
        str += html.CreatePage("Project Management", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def PanteraManagementHTMLPage(self):
        '''
    
        '''
        """UI Management page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Cleaner",1)                

        # create pantera management page

        str_page += "<table border=0>\r\n"
        
        str_page += "<tr>\r\n"
        str_page += "<td><h2>Cleaner</h2></td>\r\n"
        str_page += "</tr>\r\n"
        str_page += "<form action=\"/pantera_management_form\" method=\"POST\">\r\n"
        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Delete Log File","del_log_file",0,1)
        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Delete Error Log File","del_error_log_file",0,1)
        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Delete Session Directory","del_session_dir",0,1)
        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Clean Temporal Cache","del_temp_cache",0,1)
        str_page += "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("Clean Interceptor Cache","del_inter_cache",0,1)
        str_page += "<tr>\r\n"
        str_page +="<td><input type=\"submit\"></td>\r\n"
        str_page +="<td><input type=\"reset\"></td>\r\n"
        str_page += "</tr>\r\n"
        str_page += "</form>\r\n"
        str_page += "</table>\r\n"

        str += html.CreatePage("Cleaner", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ReportHTMLPage(self):
        '''
    
        '''
        """UI report page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Report Generation",1)                

        # create pantera report generation

    
        str += html.CreatePage("Report Generation", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def HistoryHTMLPage(self,order=0): # order = 0 (show first link first ) / order=1 (show last link first)
        '''
    
        '''
        """UI history page."""
        str = ""
        str_page = ""
        counter = 1
        html = panteraHTML.PanteraHTML()

        # Set automatic page refresh
        if self.ConfigData.has_key('refresh') and self.ConfigData['refresh'] == 'yes':
            html.SetDataHeader("<meta http-equiv=\"refresh\" content=%d>" % self.ConfigData['refresh_seconds'])

        str += html.CreateHeader("Session History",1) 

        #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br>\r\n"

        str_page += "<table border=0>\r\n"
        str_page += "<tr>\r\n"
        if self.inside_project == YES:
            str_page += "<td>Interesting Links: <b>%s</b> | </td>" % self.pan_db.Count_Links(self.current_project_id)
            str_page += "<td>Total Links: <b>%s</b></td>" % self.pan_db.Count_All_Links(self.current_project_id)
        else:
            str_page += "<td>Total Links: <b>%s</b></td>" % len(self.requestCache)
        str_page += "</tr>\r\n"
        str_page += "</table><br>\r\n"        
        
        str_page += "<table border=0>\r\n"
        str_page += "<tr>\r\n"
        if order==0:
            str_page += "<td><a href=\"http://pantera/history_link?order=1\"><img style=\"border: none;\" src=\"./img/up-red.gif\" alt=\"Last Link First\"></a><td></td>\r\n"
        else:
            str_page += "<td><a href=\"http://pantera/history_link?order=0\"><img style=\"border: none;\" src=\"./img/down-red.gif\" alt=\"First Link First\"></a><td></td>\r\n"
        str_page += "</tr>\r\n"
        str_page += "<tr>\r\n"

        # create session history
        if self.inside_project == YES:
            links = self.pan_db.Return_All_Links(self.current_project_id)
            if links == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            a = self.pan_db.Select_All_Page_Info(self.current_project_id)
            if a == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            if order==1:
                links.reverse()
            for l in links:
                (verb, file, site, args) = self.ReturnDataForPage(l[1])
                if verb is None or file is None or site is None or args is None:
                    continue
                uri_t = panterautils.TruncateURL(site+args,115)
                str_page += "<tr>"
                str_page += "<td><b>%s</b></td><td> >> %s <a class=\"textlink\" href=\"direct_link_info?file=%s\">%s</a></td>\r\n" % (`counter`,panterautils.FilterBadCar(verb,2),l[0],panterautils.FilterBadCar(uri_t,2))
                str_page += "</tr>"
                str_page += "<tr>"
                str_page += "<td></td><td> << %s</td>\r\n" % panterautils.FilterBadCar(self.pf.ReturnServerFirstLine(file),2)
                str_page += "</tr>\r\n"
                str_page += "<tr>\r\n"
                str_page += "<td></td><td>"
                # Add icon resources
                str_page += self.ReturnResourceIcons(l[0],a)
                str_page += "</td>"
                str_page += "</tr>\r\n"
                str_page += "<tr>"
                str_page += "<td></td><td>"
                str_page += "<a class=\"textlink\" href=\"direct_link_info?file=%s\"><img src=\"./img/info_button.gif\" border=\"0\" alt=\"Info\"></a> | " % l[0]
                str_page += "<a class=\"textlink\" href=\"editor_link?file=%s\"><img src=\"./img/edit_send_button.gif\" border=\"0\" alt=\"Edit&Send\"></a> | " % l[0]
                self.b = ""
                self.b = self.pf.ReturnPageBody(file)
                if self.b != "":
                    str_page += "<a class=\"textlink\" href=\"body_page_link?file=%s\"><img src=\"./img/page_body_button.gif\" border=\"0\" alt=\"Page Body\"></a> | " % l[0]
                else:
                    str_page += "<img src=\"./img/empty_button.gif\" border=\"0\" alt=\"No body\"></a> | "
                str_page += "<a class=\"textlink\" href=\"delete_link?file=%s\"><img src=\"./img/delete_button.gif\" border=\"0\" alt=\"Delete\"></a> | " % l[0]   
                str_page += "<a class=\"textlink\" href=\"notes_link?file=%s\"><img src=\"./img/notes_button.gif\" border=\"0\" alt=\"Notes\"></a>" % l[0]   
                str_page += "</td>\r\n"
                str_page += "</tr>\r\n"
                counter += 1
        else: # No project, reading from temp cache
            urls = []

            for t in self.requestCache:
                urls.append(t)
            if order==1:
                urls.reverse()
            for file in urls:
                (verb, file, site, args) = self.ReturnDataForPage(file)
                if verb is None or file is None or site is None or args is None:
                    continue
                uri_t = panterautils.TruncateURL(site+args,115)
                str_page += "<tr>"
                str_page += "<td><b>%s</b></td><td> >> %s <a class=\"textlink\" href=\"direct_link_info?file=%s\">%s</a></td>\r\n" % (`counter`,panterautils.FilterBadCar(verb,2),self.requestCache.index(file),panterautils.FilterBadCar(uri_t,2))
                str_page += "</tr>"
                str_page += "<tr>"
                str_page += "<td></td><td> << %s</td>\r\n" % panterautils.FilterBadCar(self.pf.ReturnServerFirstLine(file),2)
                str_page += "</tr>"
                str_page += "<tr>"
                str_page += "<td></td><td>"
                str_page += "<a class=\"textlink\" href=\"direct_link_info?file=%s\"><img src=\"./img/info_button.gif\" border=\"0\" alt=\"Info\"></a> | " % (self.requestCache.index(file))
                str_page += "<a class=\"textlink\" href=\"editor_link?file=%s\"><img src=\"./img/edit_send_button.gif\" border=\"0\" alt=\"Edit&Send\"></a> | " % (self.requestCache.index(file))
                self.b = ""
                self.b = self.pf.ReturnPageBody(file)
                if self.b != "":
                    str_page += "<a class=\"textlink\" href=\"body_page_link?file=%s\"><img src=\"./img/page_body_button.gif\" border=\"0\" alt=\"Page Body\"></a> | " % (self.requestCache.index(file))
                else:
                    str_page += "<img src=\"./img/empty_button.gif\" border=\"0\" alt=\"No body\"></a> | "
                str_page += "<a class=\"textlink\" href=\"delete_link?file=%s\"><img src=\"./img/delete_button.gif\" border=\"0\" alt=\"Edit&Send\"></a>" % (self.requestCache.index(file))
                str_page += "</td>\r\n"
                str_page += "</tr>\r\n"
                counter += 1

        str_page += "<tr>\r\n"
        if order==0:
            str_page += "<td><a href=\"http://pantera/history_link?order=1\"><img style=\"border: none;\" src=\"./img/up-red.gif\" alt=\"Last Link First\"></a><td></td>\r\n"
        else:
            str_page += "<td><a href=\"http://pantera/history_link?order=0\"><img style=\"border: none;\" src=\"./img/down-red.gif\" alt=\"First Link First\"></a><td></td>\r\n"
        str_page += "</tr>\r\n"
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]\r\n"    
        
        if self.inside_project == YES:
            str += html.CreatePage("Session History: "+ self.ConfigData['project_name'], "", str_page)                
        else:
            str += html.CreatePage("Session History (Reading from temporal cache)", "", str_page)                            
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def StatisticsHTMLPage(self):
        '''
    
        '''
        """UI statistics page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Project Assessment Statistics",1)                

        # create pantera project statistics
        if self.inside_project == YES:
            
            str_page += "<table border=0>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<th><b>General Information</b></th>\r\n"
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<th>&nbsp;</th>\r\n"
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Interesting Links:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Count_Links(self.current_project_id)
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Total Links:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Count_All_Links(self.current_project_id)
            str_page += "<tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<th>&nbsp;</th>\r\n"
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<th><b>Pages Extension Counter</b></th>\r\n"
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<th>&nbsp;</th>\r\n"
            str_page += "</tr>\r\n"
            
            # Get extensions from XML file
            xmlp = panteraLib.ExtensionParser()
            fil = panterautils.pathjoin(os.getcwd(),"data")
            xmlp.SetFileName(panterautils.pathjoin(fil,xmlp.GetFileName()))
            res = xmlp.BeginParser()
            desc = -1
            if res != 0:
                self.LogError(xmlp.GetError())

            l = xmlp.ReturnAllExt()
            r = self.pan_db.Return_Extension_Count(self.current_project_id,l)
            if r != -1:
                # Extensions table
                for k,v in r.iteritems():
                    if res == 0:
                        desc,_ = xmlp.ReturnExtData(k)
                    if k == "unknow":
                        continue
                    str_page += "<tr>\r\n"
                    str_page += "<td>%s</td><td width=\"10\"><b>%s</b></td>" % (string.upper(k),v)
                    if desc != -1:
                        str_page += "<td>%s</td>\r\n" % panterautils.FilterBadCar(desc,2)
                    str_page += "</tr>\r\n"
                if r.has_key('unknow'):
                    str_page += "<tr>\r\n"
                    str_page += "<td>UNKNOW</td><td width=\"10\"><b>%s</b></td>" % r['unknow']
                    str_page += "<tr>\r\n"
                if len(r)==0:
                    str_page += "<td>No data</td><td>&nbsp;</td>"
            
            str_page += "<tr>\r\n"
            str_page += "<th>&nbsp;</th>\r\n"
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<th><b>Data Gathered from Application</b></th>\r\n"
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<th>&nbsp;</th>\r\n"
            str_page += "</tr>\r\n"
            l = self.pan_db.Return_Unique_Types_Project_Info(self.current_project_id)
            if l == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            if len(l)>0:
                for i in l:
                    str_page += "<tr>\r\n"
                    t = "'%s'" % i 
                    str_page += "<td>Total %s:</td><td width=\"10\"><b>%s</b></td>" % (i,self.pan_db.Return_Count_Project_Info(self.current_project_id,t))
                    str_page += "</tr>\r\n"
            else:
                str_page += "<tr><td>No data</td><td>&nbsp;</td></tr>\r\n"                
            
            # Links info table
            str_page += "<tr>\r\n"
            str_page += "<th>&nbsp;</th>\r\n"
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<th><b>Links Information</b></th>\r\n"
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<th>&nbsp;</th>\r\n"
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Authentication:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'auth') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with SSL:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'ssl') 
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Forms:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'form') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Emails:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'email') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Scripts:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'script') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Authentication Forms:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'auth_form') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Cookies:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'cookie') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Session ID:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'session_id') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with External Links:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'external_link') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Comments:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'comment') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Attack Vectors:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'vuln') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Hidden Tags:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'hidden') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Objects:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'object') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with Post Authetication:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'postauth') 
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td>Links with URL Params:</td><td width=\"10\"><b>%s</b></td>" % self.pan_db.Return_Count_Links_Info(self.current_project_id,'querystr') 
            str_page += "</tr>\r\n"
        
            str_page += "</table>\r\n"        
        else:
            str_page += "We are sorry but you must be inside a project.<br>\r\n"
    
        str += html.CreatePage("Project Assessment Statistics", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def SessionTraceHTMLPage(self, order=0):
        '''
    
        '''
        """UI session trace page."""
        str = ""
        str_page = ""
        counter1 = 0
        counter2 = 0
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Session Trace",1)                

        # create our session trace page
        
        #str_page += "<p>\r\n"
        #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br>\r\n"
        str_page += "<form name=\"select_form\"><table><tr><th valign=\"top\">Select URL:</th>\r\n"
        str_page += "<script type=\"text/javascript\">"
        str_page += "<!--\r\n"            
        str_page += "function FillData(){\r\n"       
        str_page += "var request_array=new Array();\n\r"
        str_page += "var response_array=new Array();\n\r"
        if self.inside_project == YES:
            links = self.pan_db.Return_All_Links(self.current_project_id)
            if links == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            for l in links:
                str_page += "request_array[%s]=\"%s\";\r\n" % (`counter1`, self.ReturnClientHeader(self.pf.displayClientHeader(l[1])))
                str_page += "response_array[%s]=\"%s\";\r\n" % (`counter1`,self.pf.ReturnServerHeader(l[1]))
                counter1 += 1
        else:
            for e in self.requestCache:
                print self.ReturnClientHeader(self.pf.displayClientHeader(e))
                str_page += "request_array[%s]=\"%s\";\r\n" % (`counter1`, self.ReturnClientHeader(self.pf.displayClientHeader(e)))
                str_page += "response_array[%s]=\"%s\";\r\n" % (`counter1`,self.pf.ReturnServerHeader(e))
                counter1 += 1
        str_page += "destination = document.select_form.show_all_links.options[document.select_form.show_all_links.selectedIndex].value;\r\n"
        str_page += "document.select_form.show_request.value = request_array[destination];"
        str_page += "document.select_form.show_response.value = response_array[destination];}"
        str_page += "//-->\r\n"
        str_page += "</script>\r\n"

        str_page += "<td>\r\n"
        str_page += "<select name=\"show_all_links\" size=\"8\" onchange=\"FillData()\">\r\n"
        if self.inside_project == YES:
            links = self.pan_db.Return_All_Links(self.current_project_id)
            if links == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            if order == 1:
                links.reverse()
            for l in links:
                (_, file, site, args) = self.ReturnDataForPage(l[1])
                uri_t = panterautils.TruncateURL(site+args,114)
                str_page += "<option value=\"%s\">%s" % (`counter2`,panterautils.FilterBadCar(uri_t,2))
                counter2 += 1
        else:
            urls = self.requestCache
            if order == 1:
                urls.reverse()
            for e in urls:
                (_, file, site, args) = self.ReturnDataForPage(e)
                uri_t = panterautils.TruncateURL(site+args,114)
                str_page += "<option value=\"%s\">%s" % (`counter2`,panterautils.FilterBadCar(uri_t,2))
                counter2 += 1
        str_page += "</select>\r\n"
        str_page += "</td>\r\n"
        if order==0:
            str_page += "<td><a href=\"http://pantera/session_trace_link?order=1\"><img style=\"border: none;\" src=\"./img/up-red.gif\" alt=\"Last Link First\"></a><td></td>\r\n"
        else:
            str_page += "<td><a href=\"http://pantera/session_trace_link?order=0\"><img style=\"border: none;\" src=\"./img/down-red.gif\" alt=\"First Link First\"></a><td></td>\r\n"    
        str_page += "</table>\r\n"

        str_page += "<table>\r\n"
        str_page += "<tr><td>REQUEST:</td>\r\n"
        str_page += "<td>RESPONSE:</td></tr>\r\n" 
        str_page += "<tr><td><textarea name=\"show_request\" cols=50 rows=16 readonly>"
        str_page += "</textarea></td>\r\n"
        str_page += "<td><textarea name=\"show_response\" cols=50 rows=16 readonly>"
        str_page += "</textarea></td></tr>\r\n"

        str_page += "</table></form>\r\n"
        
        if self.inside_project == YES:
            str += html.CreatePage("Session Trace", "", str_page)                
        else:
            str += html.CreatePage("Session Trace (Reading from temporal cache)", "", str_page)  
        
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def SessionTraceRawHTMLPage(self):
        '''
    
        '''
        """UI raw session trace page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Session Trace (RAW)",1)                

        # create our session trace raw page

        #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br>\r\n"
        str_page += "<textarea name=\"raw_asession\" cols=100 rows=25 readonly>"
        str_page += self.ReadLog()
        str_page += "</textarea>\r\n"

        str += html.CreatePage("Session Trace (RAW)", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def TreeHTMLPage(self, args = ""):
        '''
    
        '''
        """UI tree page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        html.SetDataHeader("<link rel=\"StyleSheet\" href=\"dtree.css\" type=\"text/css\" />\r\n")
        html.SetDataHeader("<script type=\"text/javascript\" src=\"dtree.js\"></script>\r\n")

        str += html.CreateHeader("Site Tree",1)                

        # create pantera site tree page

        # tree form

        links = sets.Set([])

        # get links from cache or DDBB
        if self.inside_project == YES:
            l = self.pan_db.Select_Project_Info(self.current_project_id,"'target'")
            if l == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            if len(l)>0:
                add1 = links.add
                for x in l:
                    add1(x['value'])
        else:
            add1 = links.add
            for i in self.requestCache:
                o = self.pf.displayClientHeader(i)
                s,_,_ = self.getSiteFromHeader(o)
                add1(s)

        # Show form if we got data
        if len(links)>0:
            str_page += "<form action=\"site_tree_form\" method=\"POST\">\r\n"
            str_page += "<table cellpadding=\"10\" cellspacing=\"0\">\r\n"
            str_page += "<caption><b>Site Tree Re-Construction Options</b></caption>\r\n"
            str_page += "<tr>\r\n"
            str_page += "%s" % panterautils.printFormSelectMenu('Domain:','domain',links,1)  
            str_page+= "%s\r\n" % panterautils.printHiddenEntry("domain_recon","yes",1) 
            str_page += "<td><input type=\"submit\"></td>\r\n"
            str_page += "</tr>\r\n"
            str_page += "</table>\r\n"
            str_page += "</form>\r\n"
        else:
            str_page += "<caption>No domain to reconstruct!</caption>\r\n" 
            str += html.CreatePage("Site Tree", "", str_page)                
            return str
            
        #        
        # Re-Construct target site tree
        #
        
        if args != '' and args.has_key('domain'):

            # get all links of the selected domain
            dirs = sets.Set([])
            all_links = sets.Set([])
            all_dir = sets.Set([])
            add1 = all_links.add
            add2 = all_dir.add
            if self.inside_project == YES:
                li = self.pan_db.Return_All_Links(self.current_project_id)
                if li == -1:
                    self.LogError(self.pan_db.GetErrorMsg())
                for z in li:
                    o = self.pf.displayClientHeader(z[1])
                    s,_,_ = self.getSiteFromHeader(o)
                    if s == args['domain']:
                        url = panterautils.urlnormalize(o.URL)
                        t = '' 
                        #while 1:
                        s = url
                        i = url.count('/')
                        #print url
                        if (i  >=  0):
                            while "/" in url:
                                st,url = url.split('/',1)
                                #print st
                                #print url
                                if st != "":
                                    st += '/'
                                t += st
                                add2(t)
                                #print t
                            t += url
                            if url != '':
                                add1(t)   
                        else:
                            #print url
                            if url != '':
                                add1(s)
                        """
                        #c = panterautils.utils_split_dir(url)
                        #print c
                        t = '' 
                        #while 1:
                        i = string.find(url,'/')
                        if i>=0: # dir found
                            st = url[i:]
                            t += st
                            add2(t)
                        else:
                            add1(t+v) # links
                            #break
                        """
                            
            else:
                for i in self.requestCache:
                    o = self.pf.displayClientHeader(i)
                    s,_,_ = self.getSiteFromHeader(o)
                    if s == args['domain']:
                        url = panterautils.urlnormalize(o.URL)
                        t = '' 
                        #while 1:
                        s = url
                        i = url.count('/')
                        #print url
                        if (i  >=  0):
                            while "/" in url:
                                st,url = url.split('/',1)
                                #print st
                                #print url
                                if st != "":
                                    st += '/'
                                t += st
                                add2(t)
                                #print t
                            t += url
                            if url != '':
                                add1(t)   
                        else:
                            #print url
                            if url != '':
                                add1(s)
                            
            str_page += "<hr>\r\n"

            #for x in all_links:
            #    print x
                
            #for x in all_dir:
            #    print x


            str_page += "<div class=\"dtree\">\r\n"

            str_page += "<p><a href=\"javascript: d.openAll();\">open all</a> | <a href=\"javascript: d.closeAll();\">close all</a></p>\r\n"

            str_page += "<script type=\"text/javascript\">\r\n"
            str_page += "<!--\r\n"

            str_page += "d = new dTree('d');\r\n"

            str_page += "d.add(0,-1,'%s');\r\n" % args['domain']
            
            dirs = []
            add = dirs.append
            for x in all_dir:
                add(x)
            dirs.sort()
            
            links = []
            add = links.append
            for z in all_links:
                add(z)
            links.sort()
            
            i  = 1
            for x in dirs:
                str_page += "d.add(%s,0,'%s','%s','','','./img/folder.gif');\r\n" % (i,x,x)
                i += 1
            
            """
            i  = 1
            last = ''
            z = 0   
            c = 1 
            for x in dirs:
                print x
                print last
                str_page += "d.add(%s,0,'%s','%s','','','./img/folder.gif');\r\n" % (i,x,x)
                last = dirs[z]
                if last == '' or x == last: 
                    i += 1
                    z += 1
                    continue
                r = string.find(last,x)                
                print r
                if r>=0: 
                    str_page += "d.add(%s,%s,'%s','%s','','','./img/folder.gif');\r\n" % (i,c,x,x)
                    c += 1
                else:
                    i += 1
                    z += 1
                
                
                #str_page += "d.add(%s,0,'%s','%s','','','./img/folder.gif');\r\n" % (i,x,x
                print x
                if x != '':
                    last = dirs[i-1]
                    print last
                    if last != '':
                        r = string.find(x,last)                
                        print r
                        if r>=0:
                            c = 1
                            str_page += "d.add(%s,%s,'%s','%s','','','./img/folder.gif');\r\n" % (i,c,x,x)
                            c += 1
                        else:
                            str_page += "d.add(%s,0,'%s','%s','','','./img/folder.gif');\r\n" % (i,x,x)
                            i += 1
                else:
                    str_page += "d.add(%s,0,'%s','%s','','','./img/folder.gif');\r\n" % (i,x,x)
                i += 1
                #if x != '':
                #    last = x
                """
            
            """
            d.add(1,0,'Node 1','example01.html');
            d.add(2,0,'Node 2','example01.html');
            d.add(3,1,'Node 1.1','example01.html');
            d.add(4,0,'Node 3','example01.html');
            d.add(5,3,'Node 1.1.1','example01.html');
            d.add(6,5,'Node 1.1.1.1','example01.html');
            d.add(7,0,'Node 4','example01.html');
            d.add(8,1,'Node 1.2','example01.html');
            d.add(9,0,'My Pictures','example01.html','Pictures I\'ve taken over the years','','','img/imgfolder.gif');
            d.add(10,9,'The trip to Iceland','example01.html','Pictures of Gullfoss and Geysir');
            d.add(11,9,'Mom\'s birthday','example01.html');
            d.add(12,0,'Recycle Bin','example01.html','','','img/trash.gif');
            """

            str_page += "document.write(d);\r\n"
            str_page += "//-->\r\n"
            str_page += "</script>\r\n"
            str_page += "</div>\r\n"

        str += html.CreatePage("Site Tree", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ReconstructHTMLPage(self):
        '''
    
        '''
        """UI reconstruct site page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Reconstruct Site",1)                

        # create pantera report generation

    
        str += html.CreatePage("Reconstruct Site", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def UtilsHTMLPage(self, args=""):
        '''
    
        '''
        """UI utils page."""
        sstr = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        sstr += html.CreateHeader("Pantera Utilities",1)                

        #
        # show form results
        #
        if args != "" and args.has_key('string'):
            istr = urllib.unquote_plus(args["string"])
            ids_modes = ""
            
            if istr == "":
                istr = "NO STRING"
            
            str_page += "<table>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td><b>STRING:</b> </td><td>%s</td>\r\n" % panterautils.FilterBadCar(istr,1)
            str_page += "</tr>\r\n"
            if args.has_key("md5") and args["md5"] == "yes":
                m = md5.new(istr)
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("MD5","",m.hexdigest(),1,50)
            if args.has_key("sha1") and args["sha1"] == "yes":
                m = sha.new(istr)
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("SHA1","",m.hexdigest(),1,50)
            if args.has_key("rot13") and args["rot13"] == "yes":
                table = string.maketrans('nopqrstuvwxyzabcdefghijklmNOPQRSTUVWXYZABCDEFGHIJKLM',
                                         'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ')
                m = string.translate(istr, table)
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("ROT13","",m,1,50)
            #if args.has_key("hex") and args["hex"] == "yes":
            #    str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("HEX","",panterautils.encode_hex(istr),1,50)
            if args.has_key("url") and args["url"] == "yes":
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("URL","",urllib.quote(istr),1,50)            
            if args.has_key("random_hex") and args["random_hex"] == "yes":
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("RANDOM HEX","",panterautils.encode_hex_random(istr),1,50)
            #if args.has_key("hex2str") and args["hex2str"] == "yes":
            #    str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("HEX2STR","",panterautils.hex2str(istr),1,50)
            if args.has_key("unicode") and args["unicode"] == "yes":
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("UNICODE","",panterautils.str2unicode(istr),1,50)
            if args.has_key("base64") and args["base64"] == "yes":
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("BASE64","",base64.encodestring(istr),1,50)
            if args.has_key("html") and args["html"] == "yes":
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("HTML ENCODING","",str(panterautils.FixStringForXML(istr)),1,50)
            if args.has_key("double_hex") and args["double_hex"] == "yes":
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("DOUBLE PERCENT HEX","",panterautils.encode_uri_double_percent_hex(istr),1,50)
            if args.has_key("nibble_hex") and args["nibble_hex"] == "yes":
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("DOUBLE NIBBLE HEX","",panterautils.encode_uri_double_nibble_hex(istr),1,50)
            if args.has_key("first_nibble") and args["first_nibble"] == "yes":
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("FIRST NIBBLE HEX","",panterautils.encode_uri_first_nibble_hex(istr),1,50)
            if args.has_key("second_nibble") and args["second_nibble"] == "yes":
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("SECOND NIBBLE HEX","",panterautils.encode_uri_second_nibble_hex(istr),1,50)
            #if args.has_key("deco_hex") and args["deco_hex"] == "yes":
            #    str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("DECODE HEX","","",1,50)
            if args.has_key("deco_url") and args["deco_url"] == "yes":
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("DECODE URL","",urllib.unquote(istr),1,50)
            if args.has_key("deco_base64") and args["deco_base64"] == "yes":
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("DECODE BASE64","",base64.decodestring(istr),1,50)
            str_page += "</table>\r\n"

            if args.has_key("mode_1") and args["mode_1"] == "yes":
                ids_modes += "1"
            if args.has_key("mode_2") and args["mode_2"] == "yes":
                ids_modes += "2"
            if args.has_key("mode_3") and args["mode_3"] == "yes":
                ids_modes += "3"
            if args.has_key("mode_4") and args["mode_4"] == "yes":
                ids_modes += "4"
            if args.has_key("mode_5") and args["mode_5"] == "yes":
                ids_modes += "5"
            if args.has_key("mode_6") and args["mode_6"] == "yes":
                ids_modes += "6"
            if args.has_key("mode_7") and args["mode_7"] == "yes":
                ids_modes += "7"
            if args.has_key("mode_8") and args["mode_8"] == "yes":
                ids_modes += "8"

            if ids_modes:
                str_page += "<table>\r\n"
                str_page += "<tr>\r\n"
                str_page += "<td>Anti IDS Modes: </td><td>%s</td>\r\n" % ids_modes
                str_page += "</tr><tr>\r\n"
                str_page += "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Anti IDS String","",panterautils.encode_anti_ids(istr,ids_modes),1,90)
                str_page += "</tr>\r\n"
                str_page += "</table>\r\n"
                        
            str_page += "<hr>\r\n"

        #
        # display utils form
        #
        str_page += "<form action=\"/utils_form\" method=\"POST\">\r\n"
        str_page += "<table>\r\n"

        str_page +="<tr>\r\n"
        str_page += panterautils.printFormEntry("String","string","",1,30)
        str_page +="</tr>\r\n"
        str_page += "</table>\r\n"

        str_page += "<table>\r\n"
        str_page +="<tr>\r\n"
        str_page +="<td><h2>Encoders</h2></td>\r\n"
        str_page +="</tr>\r\n"
        str_page += "</table>\r\n"
        str_page += "<table>\r\n"
        str_page += "<tr>\r\n"
        str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("MD5","md5",0,1)
        str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("SHA1","sha1",0,1)
        str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("ROT13","rot13",0,1)
        #str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("HEX","hex",0,1)
        str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("URL","url",0,1)
        str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("RANDOM HEX","random_hex",0,1)
        #str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("HEX2STR","hex2str",0,1)
        str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("UNICODE","unicode",0,1)
        str_page += "%s<td></td>\r\n" % panterautils.printFormCheckbox("BASE64","base64",0,1)
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        
        str_page += "<table>\r\n"
        str_page += "<tr>\r\n"
        str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("HTML ENCODING","html",0,1)
        str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("DOUBLE PERCENT HEX","double_hex",0,1)
        str_page += "%s<td></td>\r\n" % panterautils.printFormCheckbox("DOUBLE NIBBLE HEX","nibble_hex",0,1)
        str_page += "</tr>\r\n"
        str_page += "<tr>\r\n"
        str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("FIRST NIBBLE HEX","first_nibble",0,1)
        str_page += "%s<td></td>\r\n" % panterautils.printFormCheckbox("SECOND NIBBLE HEX","second_nibble",0,1)
        str_page += "</tr>\r\n"        
        str_page += "</table>\r\n"

        str_page += "<table>\r\n"
        str_page +="<tr>\r\n"
        str_page +="<td><h2>Decoders</h2></td>\r\n"
        str_page +="</tr>\r\n"
        str_page += "</table>\r\n"
        str_page += "<table>\r\n"
        str_page += "<tr>\r\n"
        #str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("HEX","deco_hex",0,1)
        str_page += "%s<td> |</td>\r\n" % panterautils.printFormCheckbox("URL","deco_url",0,1)
        str_page += "%s<td></td>\r\n" % panterautils.printFormCheckbox("BASE64","deco_base64",0,1)
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        
        str_page += "<table>\r\n"
        str_page += "<tr>\r\n"
        str_page += "<td><h2>Anti IDS</h2></td>\r\n"
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        str_page += "<table>\r\n"
        str_page += "<tr>\r\n"
        str_page += "%s\r\n" % panterautils.printFormCheckbox("1 Random URI (non-UTF8) encoding","mode_1",0,1)
        str_page += "</tr>\r\n"
        str_page += "<tr>\r\n"
        str_page += "%s\r\n" % panterautils.printFormCheckbox("2 Directory self-reference (/./)","mode_2",0,1)
        str_page += "</tr>\r\n"
        str_page += "<tr>\r\n"
        str_page += "%s\r\n" % panterautils.printFormCheckbox("3 Premature URL ending","mode_3",0,1)
        str_page += "</tr>\r\n"
        str_page += "<tr>\r\n"
        str_page += "%s\r\n" % panterautils.printFormCheckbox("4 Prepend long random string","mode_4",0,1)
        str_page += "</tr>\r\n"
        str_page += "<tr>\r\n"
        str_page += "%s\r\n" % panterautils.printFormCheckbox("5 Fake parameter","mode_5",0,1)
        str_page += "</tr>\r\n"
        str_page += "<tr>\r\n"
        str_page += "%s\r\n" % panterautils.printFormCheckbox("6 TAB as request spacer","mode_6",0,1)
        str_page += "</tr>\r\n"
        str_page += "<tr>\r\n"
        str_page += "%s\r\n" % panterautils.printFormCheckbox("7 Windows random case sensitivity","mode_7",0,1)
        str_page += "</tr>\r\n"
        str_page += "<tr>\r\n"
        str_page += "%s\r\n" % panterautils.printFormCheckbox("8 Windows directory separator (\)","mode_8",0,1)
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"

        str_page += "<table>\r\n"
        str_page += "<tr>\r\n"
        str_page += "<td><input type=\"submit\"></td>\r\n"
        str_page += "<td><input type=\"reset\"></td>\r\n"
        str_page += "</tr>\r\n"
        str_page += "</table></form>\r\n"

        sstr += html.CreatePage("Pantera Utilities", "", str_page)                
        return sstr

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def GetInfoHTMLPage(self, id):
        '''
    
        '''
        """UI get info of a page."""
        str_page = ""
        str2 = ""
        str1 = ""
        html = panteraHTML.PanteraHTML()

        if self.inside_project == YES:
            file = self.pan_db.Return_Link_File(self.current_project_id,id)
            if file == -1:    
                self.LogError(self.pan_db.GetErrorMsg())
                return self.ErrorHTMLPage("Invalid Value")
        else:
            try:
                file = self.requestCache[int(id)]
            except Exception:
                return self.ErrorHTMLPage("Invalid Value")

        ch = self.pf.displayClientHeader(file)
        #if ch == "":
        #    return
        str2 += "http"
        if ch.clientisSSL:
            str2 += "s"
        str2 += "://%s%s" % (ch.connectHost,ch.URL) 
        args = ""
        if ch.useRawArguments == 0:
            if len(ch.allURLargs) > 0:
                args += "?%s" % ch.allURLargs
            else:
                if len(ch.URLargsDict) > 0:
                    args += "?"
                    args += panterautils.joinargs(ch.URLargsDict,orderlist=ch.orderlist)
        str2 += args

        str_page += html.CreateHeader("Info Page: " +str2,1)

        # Back & Next
        if self.inside_project == YES:
            links = self.pan_db.Return_All_Links(self.current_project_id)
            l = []
            ids = []
            for f in links:
                l.append(f[1])
                ids.append(f[0])
            i = l.index(file)
            try:
                id_back = ids[i-1]
            except:
                id_back = None
            try:
                id_next = ids[i+1]
            except:
                id_next = ids[0]                
            if id_back is not None:
                str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Back</a> ] " % (id_back)
            str_page += "<b>%s / %s</b> " % (i+1,len(l))
            str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Next</a> ] " % (id_next)
            str_page += "[ <a class=\"textlink\" href=\"history_link\">Up</a> ] | "
            str_page += "[ <a class=\"textlink\" href=\"editor_link?file=%s\">Edit&Send</a> ]" % id
            self.b = ""
            self.b = self.pf.ReturnPageBody(file)
            if self.b != "":
                str_page += "[ <a class=\"textlink\" href=\"body_page_link?file=%s\">Page Body</a> ] " % id
            str_page += "[ <a class=\"textlink\" href=\"delete_link?file=%s\">Delete</a> ]" % id
            k = self.data_miner.keys()
            if len(k)>0:
                str_page += "[ <a class=\"textlink\" href=\"querier_link?data_miner=yes\">Data Miner</a> ]"
        else:
            i = self.requestCache.index(file)
            try:
                back = self.requestCache[i-1]
            except:
                back = None
            try:
                next = self.requestCache[i+1]
            except:
                next = self.requestCache[0]
            if back is not None:
                str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Back</a> ] " % (self.requestCache.index(back))
            str_page += "<b>%s / %s</b> " % (i+1,len(self.requestCache))
            str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Next</a> ] " % (self.requestCache.index(next))
            str_page += "[ <a class=\"textlink\" href=\"history_link\">Up</a> ] | "
            str_page += "[ <a class=\"textlink\" href=\"editor_link?file=%s\">Edit&Send</a> ] " % (i)
            self.b = ""
            self.b = self.pf.ReturnPageBody(file)
            if self.b != "":
                str_page += "[ <a class=\"textlink\" href=\"body_page_link?file=%s\">Page Body</a> ] " % (i)
            str_page += "[ <a class=\"textlink\" href=\"delete_link?file=%s\">Delete</a> ]" % (i)

        # Page Order
        # 
        # 1- TOC
        # 2- General Info
        # 3- Resources Icons
        # 4- Client Header
        # 5- Response Header
        # 6- Vuln
        # 7- Notes
        
        # 1- TOC
        #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br><br>\r\n" 
        str_page += "<h2><u><a name=\"#toc\">Table Of Contents</a></u></h2>\r\n"        
        str_page += "<b>General Info</b><br>\r\n"
        if self.inside_project == YES:
            str_page += "<b>Visual Resource Icons</b><br>\r\n"
        str_page += "<b>Client Request Header</b><br>\r\n"
        str_page += "<b>Server Response Header</b><br>\r\n"
        if self.inside_project == YES:        
            str_page += "<b>Pantera Passive Analysis</b><br>\r\n"
            str_page += "<b>Notes</b><br>\r\n"
        str_page += "<br><hr>\r\n"
        
        # 2- General Info
        #
        # - date/time
        # - extension type
        # - extension comment
        # -
        #
        str_page += "<h3><a name=\"#general\">General Info</a></h3>\r\n"
        
        str_page += "<table>\r\n"
        if self.inside_project == YES:
            dat = {}
            dat = self.pan_db.Select_Page_Info(self.current_project_id,id)
            if dat == -1 or not len(dat)>0:
                self.LogError(self.pan_db.GetErrorMsg())
                return self.ErrorHTMLPage("Error with link")
            str_page += "<tr>\r\n"
            str_page += "<td><b>Time:</b></td><td>%s</td>\r\n" % dat['page_time']
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td><b>Date:</b></td><td>%s</td>\r\n" % dat['page_date']
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td><b>Page Hash (MD5):</b></td><td>%s</td>\r\n" % dat['page_hash']
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td><b>Method:</b></td><td>%s</td>\r\n" % panterautils.FilterBadCar(dat['method'],2)
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td><b>Return Code:</b></td><td>%s</td>\r\n" % panterautils.FilterBadCar(dat['return_code'],2)
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td><b>Return Mesage:</b></td><td>%s</td>\r\n" % panterautils.FilterBadCar(dat['return_msg'],2)
            str_page += "</tr>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td><b>Version:</b></td><td>%s</td>\r\n" % panterautils.FilterBadCar(dat['version'],2)
            str_page += "</tr>\r\n"

        # Add here extension type and desc (XML), outside project 
        urlfile=ch.URL.split("/")[-1]
        ext = urlfile.split(".")[-1]
        ext = ext.lower()
        # Begin parsing our XML file
        xmlp = panteraLib.ExtensionParser()
        fil = panterautils.pathjoin(os.getcwd(),"data")
        xmlp.SetFileName(panterautils.pathjoin(fil,xmlp.GetFileName()))
        res = xmlp.BeginParser()
        desc = -1
        notes = -1
        if res == 0:
            l = xmlp.ReturnAllExt()
            str_page += "<tr>\r\n"
            if ext in l:
                str_page += "<td><b>Extension:</b></td><td>%s</td>\r\n" % ext
            else:
                if ext != '':
                    str_page += "<td><b>Extension:</b></td><td>Unknow (%s)</td>\r\n" % ext
                else:
                    str_page += "<td><b>Extension:</b></td><td>Unknow</td>\r\n"                    
            str_page += "</tr>\r\n"
        
            desc,notes = xmlp.ReturnExtData(ext)
            if desc != -1:
                str_page += "<tr>\r\n"
                str_page += "<td><b>Extension Description:</b></td><td>%s</td>\r\n" % panterautils.FilterBadCar(desc,2)
                str_page += "</tr>\r\n"
        else:
            self.LogError(xmlp.GetError())
        
        str_page += "</table>\r\n"
        
        if notes != -1:
            str_page += "<b>Extension Notes:</b><br>\r\n"
            str_page += "<textarea name=\"ext_notes\" cols=50 rows=8 readonly>\r\n"
            str_page += "%s" % notes
            str_page += "</textarea>\r\n"
        
        str_page += "<br>\r\n"
        
        if self.inside_project == YES:
        
            # 3- Resources Icons
            #
            #
        
            str_page += "<hr>\r\n"
            str_page += "<h3><a name=\"#reso\">Visual Resource Icons</a></h3>\r\n"
        
            str_page += "<table border=\"0\" width=\"50%\">\r\n"

            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/lock_icon.png\" alt=\"SSL\" ></td>\r\n"
            str_page += "<td>SSL:</td>\r\n"
            if dat.has_key('have_ssl') and dat['have_ssl'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"                
            str_page += "<td> | </td>\r\n"
            str_page += "<td><img src=\"./img/icon_auth.gif\" alt=\"Authentication\"></td>\r\n"
            str_page += "<td>Authentication:</td>\r\n"
            if dat.has_key('have_auth') and dat['have_auth'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"            
            str_page += "<td> | </td>\r\n"
            str_page += "<td><img src=\"./img/icon_script.png\" alt=\"Script\" ></td>\r\n"
            str_page += "<td>Script:</td>\r\n"
            if dat.has_key('have_script') and dat['have_script'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "</tr>\r\n"
                     
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_form.png\" alt=\"Form\" ></td>\r\n"
            str_page += "<td>Form:</td>\r\n"
            if dat.has_key('have_form') and dat['have_form'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "<td> | </td>\r\n"
            str_page += "<td><img src=\"./img/icon_authform.png\" alt=\"Auth Form\" ></td>\r\n"
            str_page += "<td>Authentication Form:</td>\r\n"
            if dat.has_key('have_auth_form') and dat['have_auth_form'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "<td> | </td>\r\n"
            str_page += "<td><img src=\"./img/icon_hidden.png\" alt=\"Hidden\" ></td>\r\n"
            str_page += "<td>Hidden</td>\r\n"
            if dat.has_key('have_hidden') and dat['have_hidden'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "</tr>\r\n"
                    
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_cookie.png\" alt=\"Cookie\" ></td>\r\n"
            str_page += "<td>Cookie:</td>\r\n"
            if dat.has_key('have_cookie') and dat['have_cookie'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "<td> | </td>\r\n"
            str_page += "<td><img src=\"./img/icon_session_id.gif\" alt=\"Session ID\" ></td>\r\n"
            str_page += "<td>Session ID:</td>\r\n"
            if dat.has_key('have_session_id') and dat['have_session_id'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "<td> | </td>\r\n"
            str_page += "<td><img src=\"./img/icon_comment.png\" alt=\"Comment\" ></td>\r\n"
            str_page += "<td>Comment:</td>\r\n"
            if dat.has_key('have_comment') and dat['have_comment'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "</tr>\r\n"
                    
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_querystr.png\" alt=\"URL Params\" ></td>\r\n"
            str_page += "<td>Query Params:</td>\r\n"
            if dat.has_key('have_querystr') and dat['have_querystr'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "<td> | </td>\r\n"
            str_page += "<td><img src=\"./img/url.png\" alt=\"External Link\" ></td>\r\n"
            str_page += "<td>External Links:</td>\r\n"
            if dat.has_key('have_external_link') and dat['have_external_link'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "<td> | </td>\r\n"
            str_page += "<td><img src=\"./img/icon_target.png\" alt=\"Vuln\" ></td>\r\n"
            str_page += "<td>Attack Vector:</td>\r\n"
            if dat.has_key('have_vuln') and dat['have_vuln'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "</tr>\r\n"
            
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_activex.png\" alt=\"Object\" ></td>\r\n"
            str_page += "<td>Object:</td>\r\n"
            if dat.has_key('have_object') and dat['have_object'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "<td> | </td>\r\n"
            str_page += "<td><img src=\"./img/icon_postauth.png\" alt=\"Post Auth\" ></td>\r\n"
            str_page += "<td>Post Authentication:</td>\r\n"
            if dat.has_key('have_postauth') and dat['have_postauth'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "<td> | </td>\r\n"
            str_page += "<td><img src=\"./img/icon_email.png\" alt=\"Email\" ></td>\r\n"
            str_page += "<td>Mail:</td>\r\n"
            if dat.has_key('have_email') and dat['have_email'] == 1:
                str_page += "<td><img src=\"./img/icon_check.png\" alt=\"Check\" ></td>\r\n"
            else:
                str_page += "<td>&nbsp;&nbsp;&nbsp;</td>\r\n"
            str_page += "</tr>\r\n"
            
            str_page += "</table>\r\n"
            str_page += "<br>\r\n"
            str_page += "[ <a class=\"textlink\" href=\"validate_page?file=%s\">Edit</a> ] \r\n" % id
            #str_page += "<br>\r\n"
            
        # 4- Client Header
        #
        # Ourtside Project
        str_page += "<hr>\r\n"
        str_page += "<h3><a name=\"#client_request\">Client Request Header</a></h3>\r\n"
        
        str_page += "<textarea name=\"raw_asession\" cols=90 rows=15 readonly>"
        res_t = self.pf.displayClientHeader(file)
        r = self.pf.ReturnClientHeader(res_t)
        if r != "":
            str_page += r
        else:
            self.LogError(self.pf.GetErroMsg())
        str_page += "</textarea>\r\n"
        str_page += "<br><br>\r\n"
            
        # 5- Response Header
        #
        # Ourtside Project
        str_page += "<hr>\r\n"
        str_page += "<h3><a name=\"#response\">Server Response Header</a></h3>\r\n"
        
        str_page += "<textarea name=\"raw_asession\" cols=90 rows=15 readonly>"
        res_t = self.pf.JustResponseHeader(file)
        if res_t != "":
            str_page += res_t
        else:
            self.LogError(self.pf.GetErroMsg())
        str_page += "</textarea>\r\n"
        str_page += "<br><br>\r\n"
        
        if self.inside_project == YES:
            # 6- Passive Analysis Information
            #
            # 
            str_page += "<hr>\r\n"
            str_page += "<h3><a name=\"#vulns\">Pantera Passive Analysis</a></h3>\r\n"
            
            lst = dat['vuln_data']
            if lst is not None:
                str_page += "<table>\r\n"
                lst1 = panterautils.Str2List(lst,'|<>|')
                self.info = []
                self.low = []
                self.med = []
                self.high = []
                self.safe = []
                for ls in lst1:
                    co = string.count(ls,'(<>)')
                    if ls != '' and co>0:
                        lev, str = string.split(ls,'(<>)')
                        if lev == 'info':
                            self.info.append(str)
                        if lev == 'low':
                            self.low.append(str)
                        if lev == 'med':
                            self.med.append(str)
                        if lev == 'high':
                            self.high.append(str)
                        if lev == 'safe':
                            self.safe.append(str)
                
                for i in self.info:        
                    str_page += "<tr>\r\n"
                    str_page += "<td width=\"2%\">\r\n"
                    str_page += "<img src=\"./img/info.png\" alt=\"Info Level\" >\r\n"
                    str_page += "</td>\r\n"
                    str_page += "<td width=\"60%\">\r\n"
                    str_page += "%s\r\n" % panterautils.FilterBadCar(i,2)
                    str_page += "</td>\r\n"
                    str_page += "</tr>\r\n"
                for i in self.low:        
                    str_page += "<tr>\r\n"
                    str_page += "<td width=\"2%\">\r\n"
                    str_page += "<img src=\"./img/low.png\" alt=\"Low Level\" >\r\n"
                    str_page += "</td>\r\n"
                    str_page += "<td width=\"60%\">\r\n"
                    str_page += "%s\r\n" % panterautils.FilterBadCar(i,2)
                    str_page += "</td>\r\n"
                    str_page += "</tr>\r\n"
                for i in self.med:        
                    str_page += "<tr>\r\n"
                    str_page += "<td width=\"2%\">\r\n"
                    str_page += "<img src=\"./img/med.png\" alt=\"Medium Level\" >\r\n"
                    str_page += "</td>\r\n"
                    str_page += "<td width=\"60%\">\r\n"
                    str_page += "%s\r\n" % panterautils.FilterBadCar(i,2)
                    str_page += "</td>\r\n"
                    str_page += "</tr>\r\n"
                for i in self.high:        
                    str_page += "<tr>\r\n"
                    str_page += "<td width=\"2%\">\r\n"
                    str_page += "<img src=\"./img/high.png\" alt=\"High Level\" >\r\n"
                    str_page += "</td>\r\n"
                    str_page += "<td width=\"60%\">\r\n"
                    str_page += "%s\r\n" % panterautils.FilterBadCar(i,2)
                    str_page += "</td>\r\n"
                    str_page += "</tr>\r\n"
                for i in self.safe:        
                    str_page += "<tr>\r\n"
                    str_page += "<td width=\"2%\">\r\n"
                    str_page += "<img src=\"./img/safe.png\" alt=\"Safe Level\" >\r\n"
                    str_page += "</td>\r\n"
                    str_page += "<td width=\"60%\">\r\n"
                    str_page += "%s\r\n" % panterautils.FilterBadCar(i,2)
                    str_page += "</td>\r\n"
                    str_page += "</tr>\r\n"
                
                str_page += "</table>\r\n"        
                
                str_page += "<br>[ <a class=\"textlink\" href=\"ppa_summary_link\">PPA Summary</a> ]\r\n"
                
            # 7- Notes
            #
            # 
            str_page += "<hr>\r\n"
            str_page += "<h3><a name=\"#notes\">Notes</a></h3>\r\n"
            
            str_page += "<form action=\"/history_form\" method=\"POST\">\r\n"
            c = self.pan_db.Return_Link_Notes(self.current_project_id,id)
            if c == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            c = c[0]
            if c[0] is not None and c[0] != "":
                str_page += "<textarea name=\"link_notes\" cols=80 rows=20 readonly>"
                str_page += "%s" % c[0]
                str_page += "</textarea>\r\n"
            str_page += "<br><a class=\"textlink\" href=\"notes_link?file=%s\"><img src=\"./img/notes_button.gif\" border=\"0\" alt=\"Notes\"></a>" % id
            #str_page += "<br><button name=\"notes_page\" value=\"%s\">Notes</button>" % id
            str_page += "</form>\r\n"
                    
        str_page += "<hr>\r\n"
        #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br>\r\n" 

        # Back & Next
        if self.inside_project == YES:
            links = self.pan_db.Return_All_Links(self.current_project_id)
            l = []
            ids = []
            for f in links:
                l.append(f[1])
                ids.append(f[0])
            i = l.index(file)
            try:
                id_back = ids[i-1]
            except:
                id_back = None
            try:
                id_next = ids[i+1]
            except:
                id_next = ids[0]                
            if id_back is not None:
                str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Back</a> ] " % (id_back)
            str_page += "<b>%s / %s</b> " % (i+1,len(l))
            str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Next</a> ] " % (id_next)
            str_page += "[ <a class=\"textlink\" href=\"history_link\">Up</a> ] | "
            str_page += "[ <a class=\"textlink\" href=\"editor_link?file=%s\">Edit&Send</a> ] " % id
            self.b = ""
            self.b = self.pf.ReturnPageBody(file)
            if self.b != "":
                str_page += "[ <a class=\"textlink\" href=\"body_page_link?file=%s\">Page Body</a> ] " % id
            str_page += "[ <a class=\"textlink\" href=\"delete_link?file=%s\">Delete</a> ]" % id
            k = self.data_miner.keys()
            if len(k)>0:
                str_page += "[ <a class=\"textlink\" href=\"querier_link?data_miner=yes\">Data Miner</a> ]"
        else:
            i = self.requestCache.index(file)
            try:
                back = self.requestCache[i-1]
            except:
                back = None
            try:
                next = self.requestCache[i+1]
            except:
                next = self.requestCache[0]
            if back is not None:
                str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Back</a> ] " % (self.requestCache.index(back))
            str_page += "<b>%s / %s</b> " % (i+1,len(self.requestCache))
            str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Next</a> ] " % (self.requestCache.index(next))        
            str_page += "[ <a class=\"textlink\" href=\"history_link\">Up</a> ] | "
            str_page += "[ <a class=\"textlink\" href=\"editor_link?file=%s\">Edit&Send</a> ] " % (i)
            self.b = ""
            self.b = self.pf.ReturnPageBody(file)
            if self.b != "":
                str_page += "[ <a class=\"textlink\" href=\"body_page_link?file=%s\">Page Body</a> ] " % (i)
            str_page += "[ <a class=\"textlink\" href=\"delete_link?file=%s\">Delete</a> ]" % (i)
        
        str1 += html.CreatePage("Info Page:", str2, str_page)                
        return str1

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def EditSendHTMLPage(self, mode=0,f=0,mybody=""):
        '''
    
        '''
        """UI edit & send a page."""
        str1 = ""
        str2 = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        if self.inside_project == YES:
            file = self.pan_db.Return_Link_File(self.current_project_id,f)
            if file == -1:    
                self.LogError(self.pan_db.GetErrorMsg())
                return self.ErrorHTMLPage("Invalid Value")
        else:
            try:
                file = self.requestCache[int(f)]
            except Exception:
                return self.ErrorHTMLPage("Invalid Value")

      # Back & Next & Warp Menu :)
        if self.inside_project == YES:
            i = -1
            links = self.pan_db.Return_All_Links(self.current_project_id)
            x = 0
            for c in links:
                if c[1] == file:
                    i = c[0]
                    break
                x += 1
            try:
                b = links[x-1]
                back = b[0]
            except:
                back = None
            try:
                n = links[x+1]
                next = n[0]
            except:
                n = links[0]
                next = n[0]
                
            if i == -1:
                return self.ErrorHTMLPage("Error with value")                

            if back is not None:
                str_page += "[ <a class=\"textlink\" href=\"editor_link?file=%s\">Back</a> ] " % back
            str_page += "<b>%s / %s </b> " % (i,len(links))
            str_page += "[ <a class=\"textlink\" href=\"editor_link?file=%s\">Next</a> ] " % next
            str_page += "[ <a class=\"textlink\" href=\"history_link\">Up</a> ] | "
            str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Info</a> ] " % f
            self.b = ""
            self.b = self.pf.ReturnPageBody(file)
            if self.b != "":
                str_page += "[ <a class=\"textlink\" href=\"body_page_link?file=%s\">Page Body</a> ] " % f
            str_page += "[ <a class=\"textlink\" href=\"delete_link?file=%s\">Delete</a> ]" % f
            k = self.data_miner.keys()
            if len(k)>0:
                str_page += "[ <a class=\"textlink\" href=\"querier_link?data_miner=yes\">Data Miner</a> ]"
        else:
            i = self.requestCache.index(file)
            try:
                back = self.requestCache[i-1]
            except:
                back = None
            try:
                next = self.requestCache[i+1]
            except:
                next = self.requestCache[0]
            if back is not None:
                str_page += "[ <a class=\"textlink\" href=\"editor_link?file=%s\">Back</a> ] " % (self.requestCache.index(back))
            str_page += "<b>%s / %s</b> " % (i+1,len(self.requestCache))
            str_page += "[ <a class=\"textlink\" href=\"editor_link?file=%s\">Next</a> ] " % (self.requestCache.index(next))
            str_page += "[ <a class=\"textlink\" href=\"history_link\">Up</a> ] | "
            str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Info</a> ] " % i         
            self.b = ""
            self.b = self.pf.ReturnPageBody(file)
            if self.b != "":
                str_page += "[ <a class=\"textlink\" href=\"body_page_link?file=%s\">Page Body</a> ] " % i
            str_page += "[ <a class=\"textlink\" href=\"delete_link?file=%s\">Delete</a> ]" % i

        if mode==0:
            ch = self.pf.displayClientHeader(file)
        else:
            ch = file
        str2 += "http"
        if ch.clientisSSL:
            str2 += "s"
        str2 += "://%s%s" % (ch.connectHost,ch.URL) 
        args = ""
        if ch.useRawArguments == 0:
            if len(ch.allURLargs) > 0:
                args += "?%s" % ch.allURLargs
            else:
                if len(ch.URLargsDict) > 0:
                    args += "?"
                    args += panterautils.joinargs(ch.URLargsDict,orderlist=ch.orderlist)
        str2 += args

        str1 += html.CreateHeader("Edit&Send: " +str2,1)

        #
        # Begin construt request FORM 
        #

        if mode == 0:
            cb=self.pf.displayClientBody(file)
        else:
            cb=mybody

        if mode == 0:
            requestfile=file.replace(self.basedir,"")
        #else:
        #    requestfile=string.replace(self.basedir,"")
            

        #must use the real URL so that loading href "/bob.something" works as if from that server
        our_url="%s_send_request" % ch.URL
        #strip off leading double slashes
        if our_url[:2]=="//":
            our_url=our_url[1:]
        site="http"
        if ch.clientisSSL:
            site+="s"
        site+="://%s:%s" % (ch.connectHost,str(ch.connectPort))
        if mode==0:
            str_page+="<form action=\"/send_request\" method=\"POST\">\r\n"
        else:
            str_page+="<form action=\"/send_rewrite\" method=\"POST\">\r\n"            
        #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br>\r\n"
        str_page += "<table><tr><td></td></tr>\r\n"
        if mode == 0:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printHiddenEntry("PANTERA_TRIGGER","yes",1)
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Verb", "Verb", ch.verb,1)
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Host", "ConnectHost", ch.connectHost,1)
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Port","ConnectPort", str(ch.connectPort),1)
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("URL","URL",ch.URL,1,30)
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("SSL","SSL",ch.clientisSSL,1)

        str_page += "<tr><td><h3>Headers</h3></td></tr>"
        #print out all the headers

        i=0
        for hkey in ch.headerValuesDict.keys():
            for val in ch.headerValuesDict[hkey]:
                str_page += "<tr>\r\n"
                str_page += "%s\r\n" % panterautils.printFormEntryAndValue("Header"+str(i),hkey,val,1,30)
                str_page += "%s\r\n" % panterautils.printFormCheckbox("Delete","del_"+hkey,0,1)
                str_page += "</tr>\r\n"
                i=i+1
                
        #some extra headers if the user wants
        for i in range(i,i+5,1):
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("Header"+str(i),"","",1,30)
            
        str_page +="<tr><td><h2>URL Args</h2></td></tr>\r\n"
        i=0
        if len(ch.URLargsDict) > 0:
            for akey in ch.URLargsDict.keys():
                if akey == "PANTERA_TRIGGER" and mode != 0:
                    del ch.URLargsDict[akey]
                    continue
                str_page += "<tr>\r\n"
                str_page += "%s\r\n" % panterautils.printFormEntryAndValue("URLArg"+str(i),akey,ch.URLargsDict[akey],1,30)
                str_page += "%s\r\n" % panterautils.printFormCheckbox("Delete","del_"+akey,0,1)
                str_page += "</tr>\r\n"
                i=i+1
                
        #some extra URL arguments if the user wants
        for i in range(i,i+5,1):
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("URLArg"+str(i),"","",1,30)

        #now the body arguments
        i=0
        str_page+="<tr><td><h2>Body Args</h2></td></tr>\r\n"
        if len(cb.data)>0:
            bodyargs=panterautils.splitargs("".join(cb.data))
            if bodyargs is not None:
                for akey in bodyargs.keys():
                    str_page += "<tr>\r\n"
                    str_page += "%s\r\n" % panterautils.printFormEntryAndValue("BodyArg"+str(i),akey,bodyargs[akey],1,30)
                    str_page += "%s\r\n" % panterautils.printFormCheckbox("Delete","del_"+akey,0,1)
                    str_page += "</tr>\r\n"
                    i=i+1

        #some extra body arguments if the user wants
        for i in range(i,i+5,1):
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("BodyArg"+str(i),"","",1,30)

        str_page += "<tr>\r\n"    
        str_page += "<td><input type=\"submit\"></td>"
        str_page += "<td><input type=\"reset\"></td>"
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        str_page += "</form>"

        #
        # End construct request FORM
        #

        str1 += html.CreatePage("Edit&Send:", str2, str_page)                
        return str1

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def EditSendRewrite(self, myheader="",mybody=""):
        '''
    
        '''
        """ EditSendRewrite func """
        str1 = ""
        str2 = ""
        str_page = ""

        #print "TAG 1"
        #print myheader.URL
        #print myheader.URLargsDict
        #print myheader.headerValuesDict

        ch = myheader
        str2 += "http"
        if ch.clientisSSL:
            str2 += "s"
        str2 += "://%s%s" % (ch.connectHost,ch.URL) 
        args = ""
        if ch.useRawArguments == 0:
            if len(ch.allURLargs) > 0:
                args += "?%s" % ch.allURLargs
            else:
                if len(ch.URLargsDict) > 0:
                    args += "?"
                    args += panterautils.joinargs(ch.URLargsDict,orderlist=ch.orderlist)
        str2 += args

        str_page += "<html>\r\n"
        str_page += "<head>\r\n"
        str_page += "<title>Pantera Interceptor Editor</title>\r\n" 
        str_page += "<style type=\"text/css\">\r\n"
        str_page += "<!--\r\n"
        str_page += ".tealtable, .tealtable TD, .tealtable TH\r\n"
        str_page += "{\r\n"
        str_page += "background-color:teal;\r\n"
        str_page += "color:white;\r\n"
        str_page += "}\r\n"
        str_page += "-->\r\n"
        str_page += "</style>\r\n"
        str_page += "</head>\r\n"
        str_page += "<body>\r\n"
        str_page += "<table border=\"0\" class=\"tealtable\"><tr><th><h2>Pantera Interceptor Editor</h2></th></tr></table>\r\n"

        #
        # Create Command Center 
        #

        st = "http"
        if ch.clientisSSL:
            st += "s"
        st += "://%s" % ch.connectHost
        str_page+="<form action=\"%s/?PANTERA_TRIGGER=send_rewrite\" method=\"POST\">\r\n" % st
        str_page += "<h4><u>Command Center</u></h4><table border=\"0\">\r\n"
        #if self.inside_project == YES:
        #    au = []
        #    str_page += "<tr>\r\n"
        #    str_page += "<td>Use this Auth</td><td></td>\r\n"
        #    str_page += "</tr>\r\n"
        str_page += "<tr>\r\n"
        str_page += "<td><td>%s\r\n" % panterautils.printFormCheckbox("Allow link to be re-intercepter","del_this_link",1,1)        
        str_page += "</tr>\r\n"
        str_page += "</table><br>\r\n"

        str_page += "<hr><br>\r\n"

        #
        # Begin construt request FORM 
        #

        cb=mybody
        
        #must use the real URL so that loading href "/bob.something" works as if from that server
        our_url="%s_send_request" % ch.URL
        #strip off leading double slashes
        if our_url[:2]=="//":
            our_url=our_url[1:]
        site="http"
        if ch.clientisSSL:
            site+="s"
        site+="://%s:%s" % (ch.connectHost,str(ch.connectPort))
        
        #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br>\r\n"
        str_page += "<table><tr><td></td></tr>\r\n"
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printHiddenEntry("PANTERA_TRIGGER","yes",1)
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Verb", "Verb", ch.verb,1)
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Host", "ConnectHost", ch.connectHost,1)
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Port","ConnectPort", str(ch.connectPort),1)
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("URL","URL",ch.URL,1,30)
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("SSL","SSL",ch.clientisSSL,1)

        str_page += "<tr><td><h3>Headers</h3></td></tr>"
        #print out all the headers

        i=0
        for hkey in ch.headerValuesDict.keys():
            for val in ch.headerValuesDict[hkey]:
                str_page += "<tr>\r\n"
                str_page += "%s\r\n" % panterautils.printFormEntryAndValue("Header"+str(i),hkey,val,1,30)
                str_page += "%s\r\n" % panterautils.printFormCheckbox("Delete","del_"+hkey,0,1)
                str_page += "</tr>\r\n"
                i=i+1
                
        #some extra headers if the user wants
        for i in range(i,i+5,1):
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("Header"+str(i),"","",1,30)
            
        str_page +="<tr><td><h2>URL Args</h2></td></tr>\r\n"
        i=0
        if len(ch.URLargsDict) > 0:
            for akey in ch.URLargsDict.keys():
                if akey == "PANTERA_TRIGGER":
                    #del ch.URLargsDict[akey]
                    continue
                str_page += "<tr>\r\n"
                str_page += "%s\r\n" % panterautils.printFormEntryAndValue("URLArg"+str(i),akey,ch.URLargsDict[akey],1,30)
                str_page += "%s\r\n" % panterautils.printFormCheckbox("Delete","del_"+akey,0,1)
                str_page += "</tr>\r\n"
                i=i+1
                
        #some extra URL arguments if the user wants
        for i in range(i,i+5,1):
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("URLArg"+str(i),"","",1,30)

        #now the body arguments
        i=0
        str_page+="<tr><td><h2>Body Args</h2></td></tr>\r\n"
        if len(cb.data)>0:
            bodyargs=panterautils.splitargs("".join(cb.data))
            if bodyargs is not None:
                for akey in bodyargs.keys():
                    str_page += "<tr>\r\n"
                    str_page += "%s\r\n" % panterautils.printFormEntryAndValue("BodyArg"+str(i),akey,bodyargs[akey],1,30)
                    str_page += "%s\r\n" % panterautils.printFormCheckbox("Delete","del_"+akey,0,1)
                    str_page += "</tr>\r\n"
                    i=i+1

        #some extra body arguments if the user wants
        for i in range(i,i+5,1):
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("BodyArg"+str(i),"","",1,30)

        str_page += "<tr>\r\n"    
        str_page += "<td><input type=\"submit\"></td>"
        str_page += "<td><input type=\"reset\"></td>"
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        str_page += "</form>"
        str_page += "</body>\r\n"

        #
        # End construct request FORM
        #

        return str_page

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def BodyHTMLPage(self,f):
        '''
    
        '''
        """UI display a page's body."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        #html.SetDataHeader(html.ReturnTextSearchScript())

        str += html.CreateHeader("Body Page",1)                

        # create our body page
        
        if self.inside_project == YES:
            file = self.pan_db.Return_Link_File(self.current_project_id,f)
            if file == -1:    
                self.LogError(self.pan_db.GetErrorMsg())
                return self.ErrorHTMLPage("Invalid Value")
        else:
            try:
                file = self.requestCache[int(f)]
            except Exception:
                return self.ErrorHTMLPage("Invalid Value")                    

        # Back & Next & Warp Menu :)
        if self.inside_project == YES:
            i = -1
            links = self.pan_db.Return_All_Links(self.current_project_id)
            l = []
            myappend = l.append
            for fx in links:
                b = self.pf.ReturnPageBody(fx[1])
                if b != "":
                    myappend((fx[0],fx[1]))
            x = 0
            for c in l:
                if c[1] == file:
                    i = c[0]
                    break
                x += 1
            try:
                b = l[x-1]
                back = b[0]
            except:
                back = None
            try:
                n = l[x+1]
                next = n[0]
            except:
                n = l[0]
                next = n[0]
                
            if i == -1:
                return self.ErrorHTMLPage("Error with value")                

            if back is not None:
                str_page += "[ <a class=\"textlink\" href=\"body_page_link?file=%s\">Back</a> ] " % back
            str_page += "<b>%s / %s </b> " % (i,len(l))
            str_page += "[ <a class=\"textlink\" href=\"body_page_link?file=%s\">Next</a> ] " % next
            str_page += "[ <a class=\"textlink\" href=\"history_link\">Up</a> ] | "
            str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Info</a> ] " % f
            str_page += "[ <a class=\"textlink\" href=\"editor_link?file=%s\">Edit&Send</a> ] " % f
            str_page += "[ <a class=\"textlink\" href=\"delete_link?file=%s\">Delete</a> ]" % f
            k = self.data_miner.keys()
            if len(k)>0:
                str_page += "[ <a class=\"textlink\" href=\"querier_link?data_miner=yes\">Data Miner</a> ]"
        else:
            l = []
            for fx in self.requestCache:
                b = self.pf.ReturnPageBody(fx)
                if b != "":
                    l.append(fx)
            i = self.requestCache.index(file)
            c = l.index(file)
            try:
                back = l[c-1]        
            except:
                back = None
            try:
                next = l[c+1]
            except:
                next = l[0]
            if back is not None:
                str_page += "[ <a class=\"textlink\" href=\"body_page_link?file=%s\">Back</a> ] " % (self.requestCache.index(back))
            str_page += "<b>%s / %s</b> " % (i+1,len(self.requestCache))
            str_page += "[ <a class=\"textlink\" href=\"body_page_link?file=%s\">Next</a> ] " % (self.requestCache.index(next))
            str_page += "[ <a class=\"textlink\" href=\"history_link\">Up</a> ] | "
            str_page += "[ <a class=\"textlink\" href=\"direct_link_info?file=%s\">Info</a> ] " % i
            str_page += "[ <a class=\"textlink\" href=\"editor_link?file=%s\">Edit&Send</a> ] " % i
            str_page += "[ <a class=\"textlink\" href=\"delete_link?file=%s\">Delete</a> ]" % i

        #str_page += "<form name=\"f1\" action=\"\"" 
        #str_page += "onSubmit=\"if(this.t1.value!=null && this.t1.value!='')"
        #str_page += "findString(this.t1.value);return false\">\r\n"
        #str_page += "<input type=\"text\" name=t1 value=\"\" size=20>\r\n"
        #str_page += "<input type=\"submit\" name=b1 value=\"Find\">\r\n"
        #str_page += "</form>\r\n"

        #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br>\r\n" 
        str_page += "<textarea name=\"raw_session\" cols=100 rows=25 readonly>"
        str_page += self.pf.ReturnPageBody(file)
        str_page += "</textarea>\r\n"
        (_,_,site,args) = self.ReturnDataForPage(file)

        #str_page += "<form name=\"f1\" action=\"\"" 
        #str_page += "onSubmit=\"if(this.t1.value!=null && this.t1.value!='')"
        #str_page += "findString(this.t1.value);return false\">"
        #str_page += "<td><input type=\"text\" name=t1 value=\"\" size=20></td>"
        #str_page += "<td><input type=\"submit\" name=b1 value=\"Find\"</td>"
        #str_page += "</form>\r\n"
                
        str_page += html.ReturnTextSearchScript()

        str += html.CreatePage("Body Page of:", site+args, str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ErrorHTMLPage(self,msg):
        '''
    
        '''
        """UI error page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Pantera Error",1)                

        # create error page
        str_page += "<textarea name=\"raw_asession\" cols=100 rows=25 readonly>"
        str_page += msg
        str_page += "</textarea>\r\n"

        str += html.CreatePage("Pantera Error", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def InterceptorHTMLPage(self, do_interceptor_change=0, args=""):
        '''
    
        '''
        """UI intercept page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()
        counter = 0

        str += html.CreateHeader("Interceptor",1)                

        # create pantera interceptor
        
        # set new interceptor
        if do_interceptor_change == 2:
            #
            # check for old ID and delete it.
            for c in self.interceptor_list:
                if c['filter_id'] == args['filter_id']:
                    self.interceptor_list.remove(c) 
            dat = {}
            dat['regex'] = urllib.unquote_plus(args['regex'])
            dat['filter_id'] = args['filter_id']
            self.interceptor_list.append(dat)
            do_interceptor_change = 0

        #
        # delete interceptor filter
        #
        if do_interceptor_change == 3:
            for c in self.interceptor_list:
                if c['filter_id'] == args['filter_id']:
                    self.interceptor_list.remove(c) 
            do_interceptor_change = 0

        # 
        # add new interceptor
        #
        if do_interceptor_change == 1:        
            str_page += "<form action=\"/interceptor_form\" method=\"POST\">\r\n"
            str_page += "<table>\r\n"
            str_page += "<tr>%s</tr>" % panterautils.printFormEntry("Regular Expression", "regex","",1,30)
            str_page+= "%s\r\n" % panterautils.printHiddenEntry("filter_id","int."+panterautils.randstr(5)) 
            str_page += "<tr><td><input type=\"submit\" name=\"new_add\" value=\"Add Interceptor\">"
            str_page += " | <input type=\"submit\" name=\"cancel\" value=\"Cancel\"></td></tr>"
            str_page += "</table>\r\n"
            str_page += "</form>"        
        
        #
        # edit interceptor filter 
        #
        if do_interceptor_change == 4:
            for c in self.interceptor_list:
                if c['filter_id'] == args['filter_id']:
                    dat = c
            str_page += "<form action=\"/interceptor_form\" method=\"POST\">\r\n"
            str_page += "<table>\r\n"
            str_page += "<tr>%s</tr>" % panterautils.printFormEntry("Regular Expression", "regex",dat['regex'],1,30)
            str_page+= "%s\r\n" % panterautils.printHiddenEntry("filter_id",dat['filter_id']) 
            str_page += "<tr><td><input type=\"submit\" name=\"new_add\" value=\"Add Interceptor\">"
            str_page += " | <input type=\"submit\" name=\"cancel\" value=\"Cancel\"></td></tr>"
            str_page += "</table>\r\n"
            str_page += "</form>"        
        
        #
        # Create interceptor form
        #
        if do_interceptor_change==0:
            str_page += "<form action=\"/interceptor_form\" method=\"POST\">\r\n"
            str_page += "<table>\r\n"
            str_page += "<tr><td><input type=\"submit\" name=\"new\" value=\"Add Interceptor\"></td><td></td></tr>"
            str_page += "<tr><td></td><td><td></tr>\r\n"
            str_page += "<tr><td></td><td><td></tr>\r\n"            
            str_page += "</table>\r\n"
            str_page += "</form>"
            if len(self.interceptor_list)>0:
                for c in self.interceptor_list:
                    str_page += "<form action=\"/interceptor_form\" method=\"POST\">\r\n"
                    str_page += "<table>\r\n"
                    str_page += "<tr><td><b>Interceptor Filter %s</b></td><td></td></tr>\r\n" % counter
                    str_page += "<tr><td>Regex Expression: </td><td>%s</td></tr>\r\n" % panterautils.FilterBadCar(c['regex'],2)
                    str_page+= "%s\r\n" % panterautils.printHiddenEntry("filter_id",c['filter_id']) 
                    str_page += "<tr><td><input type=\"submit\" name=\"delete\" value=\"Delete\"> | <input type=\"submit\" name=\"edit\" value=\"Edit\"></td><td></td></tr>"
                    str_page += "<tr><td></td><td><td></tr>\r\n"
                    str_page += "<tr><td></td><td><td></tr>\r\n"            
                    str_page += "</table>\r\n"
                    str_page += "</form>"
                    counter += 1
    
        str += html.CreatePage("Interceptor", "", str_page)                
        return str
    
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ReplacerHTMLPage(self, do_replacer_change=0, args=""):
        '''
    
        '''
        """UI replace page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()
        counter = 0

        str += html.CreateHeader("Replacer",1)                

        # create pantera replacer
        if do_replacer_change == 2:
            #
            # check for old ID and delete it.
            for c in self.replacer_list:
                if c['filter_id'] == args['filter_id']:
                    self.replacer_list.remove(c) 
            dat = {}
            dat['regex'] = urllib.unquote_plus(args['regex'])
            dat['repla_str'] = urllib.unquote_plus(args['repla_str'])
            if args.has_key('request'):
                dat['request'] = YES
            else:
                dat['request'] = NO
            if args.has_key('reply_header'):
                dat['reply_header'] = YES
            else:
                dat['reply_header'] = NO
            if args.has_key('reply_body'):
                dat['reply_body'] = YES
            else:
                dat['reply_body'] = NO
            # if user forgot to check any of the options we will do it for him...
            if dat['request'] == NO and dat['reply_header'] == NO and dat['reply_body'] == NO:
                dat['request'] = YES
            dat['filter_id'] = args['filter_id']
            self.replacer_list.append(dat)
            do_replacer_change = 0

        #
        # delete replacer filter
        #
        if do_replacer_change == 3:
            for c in self.replacer_list:
                if c['filter_id'] == args['filter_id']:
                    self.replacer_list.remove(c) 
            do_replacer_change = 0

        # 
        # add new replacer
        #
        if do_replacer_change == 1:        
            str_page += "<form action=\"/replacer_form\" method=\"POST\">\r\n"
            str_page += "<table>\r\n"
            str_page += "<tr>%s</tr>" % panterautils.printFormEntry("Regular Expression", "regex","",1,30)
            str_page += "<tr>%s</tr>" % panterautils.printFormEntry("Replacement String", "repla_str","",1,30)
            str_page += "<tr>%s</tr>" % panterautils.printFormCheckbox("Apply to Request","request",1,1)
            str_page += "<tr>%s</tr>" % panterautils.printFormCheckbox("Apply to Reply Header","reply_header",0,1)
            str_page += "<tr>%s</tr>" % panterautils.printFormCheckbox("Apply to Reply Body","reply_body",0,1)
            str_page+= "%s\r\n" % panterautils.printHiddenEntry("filter_id","rep."+panterautils.randstr(5)) 
            str_page += "<tr><td><input type=\"submit\" name=\"new_add\" value=\"Add Replacer\">"
            str_page += " | <input type=\"submit\" name=\"cancel\" value=\"Cancel\"></td></tr>"
            str_page += "</table>\r\n"
            str_page += "</form>"
        
        #
        # edit replacer filter 
        #
        if do_replacer_change == 4:
            for c in self.replacer_list:
                if c['filter_id'] == args['filter_id']:
                    dat = c
            str_page += "<form action=\"/replacer_form\" method=\"POST\">\r\n"
            str_page += "<table>\r\n"
            str_page += "<tr>%s</tr>" % panterautils.printFormEntry("Regular Expression", "regex",dat['regex'],1,30)
            str_page += "<tr>%s</tr>" % panterautils.printFormEntry("Replacement String", "repla_str",dat['repla_str'],1,30)
            if dat.has_key('request') and dat['request'] == YES:
                str_page += "<tr>%s</tr>" % panterautils.printFormCheckbox("Apply to Request","request",1,1)
            else:
                str_page += "<tr>%s</tr>" % panterautils.printFormCheckbox("Apply to Request","request",0,1)
            if dat.has_key('reply_header') and dat['reply_header'] == YES:
                str_page += "<tr>%s</tr>" % panterautils.printFormCheckbox("Apply to Reply Header","reply_header",1,1)
            else:
                str_page += "<tr>%s</tr>" % panterautils.printFormCheckbox("Apply to Reply Header","reply_header",0,1)
            if dat.has_key('reply_body') and dat['reply_body'] == YES:
                str_page += "<tr>%s</tr>" % panterautils.printFormCheckbox("Apply to Reply Body","reply_body",1,1)
            else:
                str_page += "<tr>%s</tr>" % panterautils.printFormCheckbox("Apply to Reply Body","reply_body",0,1)
            str_page+= "%s\r\n" % panterautils.printHiddenEntry("filter_id",dat['filter_id']) 
            str_page += "<tr><td><input type=\"submit\" name=\"new_add\" value=\"Add Replacer\">"
            str_page += " | <input type=\"submit\" name=\"cancel\" value=\"Cancel\"></td></tr>"
            str_page += "</table>\r\n"
            str_page += "</form>"
            
        #
        # create replacer home form
        #
        if do_replacer_change==0:
            str_page += "<form action=\"/replacer_form\" method=\"POST\">\r\n"
            str_page += "<table>\r\n"
            str_page += "<tr><td><input type=\"submit\" name=\"new\" value=\"Add Replacer\"></td><td></td></tr>"
            str_page += "<tr><td></td><td><td></tr>\r\n"
            str_page += "<tr><td></td><td><td></tr>\r\n"            
            str_page += "</table>\r\n"
            str_page += "</form>"
            if len(self.replacer_list)>0:
                for c in self.replacer_list:
                    str_page += "<form action=\"/replacer_form\" method=\"POST\">\r\n"
                    str_page += "<table>\r\n"
                    str_page += "<tr><td><b>Replacer Filter %s</b></td><td></td></tr>\r\n" % counter
                    str_page += "<tr><td>Regex Expression: </td><td>%s</td></tr>\r\n" % panterautils.FilterBadCar(c['regex'],2)
                    str_page += "<tr><td>Replacement String: </td><td>%s</td></tr>\r\n" % panterautils.FilterBadCar(c['repla_str'],2)
                    if c.has_key('request') and c['request'] == YES:
                        str_page += "<tr><td>Apply to Request: </td><td>True</td></tr>\r\n"
                    else:
                        str_page += "<tr><td>Apply to Request: </td><td>False</td></tr>\r\n"
                    if c.has_key('reply_header') and c['reply_header'] == YES:
                        str_page += "<tr><td>Apply to Reply Header: </td><td>True</td></tr>\r\n"
                    else:
                        str_page += "<tr><td>Apply to Reply Header: </td><td>False</td></tr>\r\n"
                    if c.has_key('reply_body') and c['reply_body'] == YES:
                        str_page += "<tr><td>Apply to Reply Body: </td><td>True</td></tr>\r\n"
                    else:
                        str_page += "<tr><td>Apply to Reply Body: </td><td>False</td></tr>\r\n"
                    str_page+= "%s\r\n" % panterautils.printHiddenEntry("filter_id",c['filter_id']) 
                    str_page += "<tr><td><input type=\"submit\" name=\"delete\" value=\"Delete\"> | <input type=\"submit\" name=\"edit\" value=\"Edit\"></td><td></td></tr>"
                    str_page += "<tr><td></td><td><td></tr>\r\n"
                    str_page += "<tr><td></td><td><td></tr>\r\n"            
                    str_page += "</table>\r\n"
                    str_page += "</form>"
                    counter += 1
    
        str += html.CreatePage("Replacer", "", str_page)                
        return str
    
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def SupressHeaderHTMLPage(self, do_supress_change=0, args=""):
        '''
    
        '''
        """UI suppres header page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()
        did_h = []
        h_done = []

        str += html.CreateHeader("Supress Header",1)                

        # 
        # perform supress headers changes
        #
        if do_supress_change == 2:
            for akey in args:
                if akey.count('header'):
                    if args[akey] == '':
                        h_done.append(akey)
                        pass
                    else:
                        num = akey[6:]
                        req_name = "h"+num+"_request"
                        resp_name = "h"+num+"_response"
                        if args[akey] in self.supress_request_header:
                            self.supress_request_header.remove(args[akey])
                        if args[akey] in self.supress_response_header:
                            self.supress_response_header.remove(args[akey])
                        if args.has_key(req_name) and args[req_name] == "yes":     
                            self.supress_request_header.append(args[akey])
                        if args.has_key(resp_name) and args[resp_name] == "yes":    
                            self.supress_response_header.append(args[akey])
                        h_done.append(akey)
                    do_supress_change = 0
            
        #
        # create pantera supress header
        #
        
        # first show any supress already defined
        
        if do_supress_change==0:
            if len(self.supress_request_header) > 0 or len(self.supress_response_header): 
                str_page += "<form action=\"/supress_header_form\" method=\"POST\">\r\n"
                str_page += "<table>\r\n"
                str_page += "<tr><td><h2>Supressed Headers</h2></td></tr>\r\n" 
                str_page += "</table>\r\n"
                str_page += "<table>\r\n"                       
                for h in self.supress_request_header:
                    if h in self.supress_response_header:
                        str_page += "<tr><td>Header: </td><td><b>%s</b></td><td> | <td>" % panterautils.FilterBadCar(h)
                        str_page += "%s<td> | <td>" % panterautils.printFormCheckbox("Request","",1,1)
                        str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","",1,1)
                        did_h.append(h)
                for h in self.supress_request_header:
                    if h not in did_h:
                        str_page += "<tr><td>Header: </td><td><b>%s</b></td><td> | <td>" % panterautils.FilterBadCar(h)
                        str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Request","",1,1)
                for h in self.supress_response_header:
                    if h not in did_h:
                        str_page += "<tr><td>Header: </td><td><b>%s</b></td><td> | <td>" % panterautils.FilterBadCar(h)
                        str_page += "<td></td>%s</tr>\r\n" % panterautils.printFormCheckbox("Response","",1,1)
                str_page += "</table>\r\n"
                str_page+= "%s\r\n" % panterautils.printHiddenEntry("do_supress_change","new")
                str_page += "<input type=\"submit\">"
                str_page += "</form>"
            else:
                do_supress_change = 1
 
        if do_supress_change==1:

            #
            # create supress header form
            #
            str_page += "<form action=\"/supress_header_form\" method=\"POST\">\r\n"
            str_page += "<table>\r\n"
            str_page += "<tr><td><h2>Change Supressed Headers</h2></td></tr>\r\n" 
            str_page += "</table>\r\n"
            str_page += "<table>\r\n"
            if len(self.supress_request_header) > 0 or len(self.supress_response_header) > 0:
                i = 0
                for h in self.supress_request_header:
                    if h in self.supress_response_header:
                        str_page += "<tr>%s" % panterautils.printFormEntry("Header", "header" +`i`,h,1,30)
                        str_page += "%s" % panterautils.printFormCheckbox("Request","h" + `i` + "_request",1,1)
                        str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","h" + `i` + "_response",1,1)
                        i += 1
                        did_h.append(h)
                for h in self.supress_request_header:
                    if h not in did_h:
                        str_page += "<tr>%s" % panterautils.printFormEntry("Header", "header" + `i`,h,1,30)
                        str_page += "%s\r\n" % panterautils.printFormCheckbox("Request","h" + `i` + "_request",1,1)
                        str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","h" + `i` + "_response",0,1)
                        i += 1
                for h in self.supress_response_header:
                    if h not in did_h:
                        str_page += "<tr>%s" % panterautils.printFormEntry("Header", "header" + `i`,h,1,30)
                        str_page += "%s\r\n" % panterautils.printFormCheckbox("Request","h" + `i` + "_request",0,1)
                        str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","h" + `i` + "_response",1,1)                        
                        i += 1
                str_page += "<tr>%s" % panterautils.printFormEntry("Header", "header" + `i`,"",1,30)
                str_page += "%s" % panterautils.printFormCheckbox("Request","h"+ `i`+ "_request",1,1)
                str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","h"+ `i` + "_response",0,1)
                i += 1
                str_page += "<tr>%s" % panterautils.printFormEntry("Header", "header" + `i`,"",1,30)
                str_page += "%s" % panterautils.printFormCheckbox("Request","h"+ `i`+ "_request",1,1)
                str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","h"+ `i` + "_response",0,1)
                i += 1
                str_page += "<tr>%s" % panterautils.printFormEntry("Header", "header" + `i`,"",1,30)
                str_page += "%s" % panterautils.printFormCheckbox("Request","h"+ `i`+ "_request",1,1)
                str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","h"+ `i` +"_response",0,1)
                i += 1
            else:
                str_page += "<tr>%s" % panterautils.printFormEntry("Header", "header1","",1,30)
                str_page += "%s" % panterautils.printFormCheckbox("Request","h1_request",1,1)
                str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","h1_response",0,1)
                str_page += "<tr>%s" % panterautils.printFormEntry("Header", "header2","",1,30)
                str_page += "%s" % panterautils.printFormCheckbox("Request","h2_request",1,1)
                str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","h2_response",0,1)
                str_page += "<tr>%s" % panterautils.printFormEntry("Header", "header3","",1,30)
                str_page += "%s" % panterautils.printFormCheckbox("Request","h3_request",1,1)
                str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","h3_response",0,1)
                str_page += "<tr>%s" % panterautils.printFormEntry("Header", "header4","",1,30)
                str_page += "%s" % panterautils.printFormCheckbox("Request","h4_request",1,1)
                str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","h4_response",0,1)
                str_page += "<tr>%s" % panterautils.printFormEntry("Header", "header5","",1,30)
                str_page += "%s" % panterautils.printFormCheckbox("Request","h5_request",1,1)
                str_page += "%s</tr>\r\n" % panterautils.printFormCheckbox("Response","h5_response",0,1)
            str_page+= "%s\r\n" % panterautils.printHiddenEntry("do_supress_change","change",1)
            str_page += "<td><input type=\"submit\"></td>"
            str_page += "</table>\r\n"
            str_page += "</form>"
    
        str += html.CreatePage("Supress Header", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def QuerierHTMLPage(self, args=""):
        '''
    
        '''
        """UI querier page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        if args != "":
            html.SetDataHeader("<script language=\"JavaScript\" src=\"public_scripts.js\"></script>")

        str += html.CreateHeader("Data Miner",1)                

        # create pantera querier
        if self.inside_project == YES:
            
            #
            # Adder
            #
            
            str_page += "<form action=\"querier_form\" method=\"POST\">\r\n"
            str_page += "<table cellpadding=\"10\" cellspacing=\"0\">\r\n"
            str_page += "<caption><b>Add to Project Info</b></caption>\r\n"
            str_page += "<tr>\r\n"
            str_page += "%s" % panterautils.printFormEntry("Type", "type","",1,15)
            str_page += "%s" % panterautils.printFormEntry("Value", "value","",1,15)
            l = self.pan_db.Select_Project_Info(self.current_project_id,"'target'")
            if l == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            if len(l)>0:
                z = []
                add = z.append
                for x in l:
                    add(x['value'])
                str_page += "%s" % panterautils.printFormSelectMenu('Domain:','domain',z,1)
            else:
                str_page += "%s" % panterautils.printFormEntry("Domain", "domain","",1,15)
            str_page+= "%s\r\n" % panterautils.printHiddenEntry("add_project_info","yes",1) 
            str_page += "<td><input type=\"submit\"></td>\r\n"
            str_page += "</tr>\r\n"
            str_page += "</table>\r\n"
            str_page += "</form>\r\n"
            
            #
            # Searcher
            #
            
            l = self.pan_db.Return_Unique_Types_Project_Info(self.current_project_id)
            if l == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            if len(l)>0:
                #str_page += "<br>\r\n"
                str_page += "<form action=\"querier_form\" method=\"POST\">\r\n"
                str_page += "<table cellpadding=\"10\" cellspacing=\"0\">\r\n"
                str_page += "<caption><b>Search on Project Info</b></caption>\r\n"
                str_page += "<tr>\r\n"
                str_page += "%s" % panterautils.printFormSelectMenu('Select:','project_info_op',l,1)
                str_page+= "%s\r\n" % panterautils.printHiddenEntry("select_project_info","yes",1) 
                str_page += "<td><input type=\"submit\"></td>\r\n"            
                str_page += "</tr>\r\n"
                str_page += "</table>\r\n"
                str_page += "</form>\r\n"
            
            #str_page += "<br>\r\n"                       
            
            have_list = []
            r = 0
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'auth')
            if r > 0:
                have_list.append('auth')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'ssl')
            if r > 0:
                have_list.append('ssl')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'form')
            if r > 0:
                have_list.append('form')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'email')
            if r > 0:
                have_list.append('email')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'script')
            if r > 0:
                have_list.append('script')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'auth_form')
            if r > 0:
                have_list.append('auth_form')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'cookie')
            if r > 0:
                have_list.append('cookie')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'session_id')
            if r > 0:
                have_list.append('session_id')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'external_link')
            if r > 0:
                have_list.append('external_link')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'comment')
            if r > 0:
                have_list.append('comment')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'vuln')
            if r > 0:
                have_list.append('vuln')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'hidden')
            if r > 0:
                have_list.append('hidden')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'object')
            if r > 0:
                have_list.append('object')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'postauth')
            if r > 0:
                have_list.append('postauth')
            r = self.pan_db.Return_Count_Links_Info(self.current_project_id,'querystr')
            if r > 0:
                have_list.append('querystr')
            if len(have_list)>0:
                str_page += "<form action=\"/querier_form\" method=\"POST\">\r\n"
                str_page += "<table cellpadding=\"10\" cellspacing=\"0\">\r\n"
                str_page += "<caption><b>Search Links With</b></caption>\r\n"
                str_page += "<tr>\r\n"
                str_page += "%s" % panterautils.printFormSelectMenu('Select:','select_have',have_list,1)
                str_page += "%s\r\n" % panterautils.printHiddenEntry("select_links_with","yes",1) 
                str_page += "<td><input type=\"submit\"></td>\r\n"
                str_page += "</tr>\r\n"
                str_page += "</table>\r\n"
                str_page += "</form>\r\n"
            
            #str_page += "<br>\r\n"     
            r = self.pan_db.Count_All_Links(self.current_project_id)
            if r>0:                  
                str_page += "<form action=\"/querier_form\" method=\"POST\">\r\n"
                str_page += "<table cellpadding=\"10\" cellspacing=\"0\">\r\n"
                str_page += "<caption><b>Search Links</b></caption>\r\n"
                str_page += "<tr>\r\n"
                l2 = ['Interesting','Not Interesting','All']
                str_page += "%s" % panterautils.printFormSelectMenu('Select:','select_links',l2,1)
                l3 = ['method','return_code','return_msg','version','extension','host']
                str_page += "%s" % panterautils.printFormSelectMenu('Select:','select_criteria',l3,1)            
                str_page += "%s" % panterautils.printFormEntry("Criteria", "criteria","",1,15)
                str_page+= "%s\r\n" % panterautils.printHiddenEntry("search_links","yes",1) 
                str_page += "<td><input type=\"submit\"></td>\r\n"
                str_page += "</tr>\r\n"
                str_page += "</table>\r\n"
                str_page += "</form>\r\n"
            
            r = self.pan_db.Count_All_Links(self.current_project_id)
            if r>0:                  
                str_page += "<form action=\"/querier_form\" method=\"POST\">\r\n"
                str_page += "<table cellpadding=\"10\" cellspacing=\"0\">\r\n"
                str_page += "<tr>\r\n"
                str_page += "<td><b>Display All Links</b></td>\r\n"
                str_page+= "%s\r\n" % panterautils.printHiddenEntry("all_links","yes",1) 
                str_page += "<td><input type=\"submit\"></td>\r\n"
                str_page += "</tr>\r\n"
                str_page += "</table>\r\n"
                str_page += "</form>\r\n"
         
            #
            # Results
            #
            
            k = self.data_miner.keys()
            if len(k)>0:
                args = self.data_miner
            
            if args != "":
                
                self.data_miner = args
                
                str_page += "<hr>\r\n"
                str_page += "<br>\r\n"
            
                #
                # Add data to project info table
                #
                if args.has_key('add_project_info') and args['add_project_info'] == 'yes' and \
                    args.has_key('type') and not args['type'] == '' and args.has_key('value') \
                    and not args['value'] == '' and args.has_key('domain') and not args['domain'] == '':
                    d = {}
                    d['project_id'] = self.current_project_id
                    d['type'] = "'%s'" % panterautils.FilterBadCar(args['type'])
                    d['value'] = "%s" % panterautils.FilterBadCar(args['value'])
                    d['domain'] = "'%s'" % panterautils.FilterBadCar(args['domain'])
                    r = self.pan_db.Insert_Project_Info(d)
                    if r == -1:
                        self.LogError(self.pan_db.GetErrorMsg())
                    else:
                        del self.data_miner
                        self.data_miner = {}
                        str_page += "<caption><b>Project Info Data Set</b></caption><br>\r\n"
                        str_page += "<a class=\"textlink\" href=\"querier_link\"><img src=\"./img/ok_button.gif\" border=\"0\" alt=\"OK\"></a>"   
                 
                #       
                # Display data from project info table
                #
                elif args.has_key('select_project_info') and args['select_project_info'] == 'yes' and args.has_key('project_info_op'):
                
                    type = "'%s'" % panterautils.FilterBadCar(args['project_info_op'])
                    l = self.pan_db.Select_Project_Info_With_ID(self.current_project_id,type)
                    if l == -1:
                        self.LogError(self.pan_db.GetErrorMsg())
                        str_page += "<caption>Error with data!</caption>\r\n"                        
                    if len(l)>0: # display data
                        str_page += "<caption><b>Project Info Data Results: %s (%s)</b></caption><br><br>\r\n" % (panterautils.FilterBadCar(args['project_info_op'],2),len(l)) 
                        str_page += self.DataMinerProjectInfoVisor(l,args['project_info_op'])
                    else:
                        str_page += "<caption>No values!</caption>\r\n"                        
             
                #
                # Display links with
                #
                elif args.has_key('select_links_with') and args['select_links_with'] == 'yes' and \
                args.has_key('select_have') and args['select_have'] != '':
                
                    have = panterautils.FilterBadCar(args['select_have'])
                    r = self.pan_db.Select_Links_With(self.current_project_id,have)
                    if r == -1:
                        self.LogError(self.pan_db.GetErrorMsg())
                        str_page += "<caption>Error with data!</caption>\r\n"                        
                    if len(r)>0: # display data
                        x = self.pan_db.Count_Links(self.current_project_id)
                        if x == -1:
                            self.LogError(self.pan_db.GetErrorMsg())
                            x = 0
                        str_page += "<caption><b>Links With Results: %s (%s / %s)</b></caption><br><br>\r\n" % (panterautils.FilterBadCar(args['select_have'],2),len(r),x)                       
                        str_page += self.DataMinerSerachLinksWithVisor(r)
                    else:
                        str_page += "<caption>No values!</caption>\r\n"          
                
                #
                # Show all links
                #
                elif args.has_key('all_links') and args['all_links'] == 'yes':
                
                    r,r1 = self.pan_db.Return_Entire_Links(self.current_project_id)
                    if r == -1 and r1 == -1:
                        self.LogError(self.pan_db.GetErrorMsg())
                        str_page += "<caption>Error with data!</caption>\r\n"                        
                    if len(r)>0: # display data
                        str_page += "<caption><b>Display All Links (%s)</b></caption><br><br>\r\n" % self.pan_db.Count_All_Links(self.current_project_id)
                        str_page += self.DataMinerEntireLinksVisor(r,r1)
                    else:
                        str_page += "<caption>No values!</caption>\r\n"
                 
                #       
                # Search links with criteria
                #
                elif args.has_key('search_links') and args['search_links'] == 'yes' and \
                    args.has_key('select_links') and not args['select_links'] == '' and \
                    args.has_key('select_criteria') and not args['select_criteria'] == '' and \
                    args.has_key('criteria') and not args['criteria'] == '':
                    
                    select_links = panterautils.FilterBadCar(args['select_links'])
                    select_criteria = panterautils.FilterBadCar(args['select_criteria'])
                    criteria = panterautils.FilterBadCar(args['criteria'])
                    
                    r,r1 = self.pan_db.Return_Links_Criteria(self.current_project_id, select_links, select_criteria, criteria)
                    if r == -1 or r1 == -1:
                        self.LogError(self.pan_db.GetErrorMsg())
                        str_page += "<caption>Error with data!</caption>\r\n"
                    else:
                        i = len(r)+len(r1)
                        if i > 0:
                            x = self.pan_db.Count_All_Links(self.current_project_id)
                            if x == -1:
                                self.LogError(self.pan_db.GetErrorMsg())
                                x = 0
                            str_page += "<caption><b>Links With Results: %s = %s (%s / %s)</b></caption><br><br>\r\n" % (select_criteria, criteria,i,x)                       
                            str_page += self.DataMinerSearchCriteria(r,r1)
                        else:
                            str_page += "<caption>No values!</caption>\r\n"

                #
                # Delete data from project info table
                #
                elif args.has_key('do_action') and args['do_action'] == 'Delete':
                    r = 0
                    for k in args.keys():
                        if string.find(k,'action_')>=0:
                            x,id = string.split(k,'_')
                            r = self.pan_db.Delete_Project_Info(self.current_project_id,id)
                            if r == -1:
                                self.LogError(self.pan_db.GetErrorMsg())
                    if r == -1:
                        str_page += "<caption><b>Error Deleting Data from Project Info</b></caption><br>\r\n"
                    else:
                        str_page += "<caption><b>Data Deleted from Project Info</b></caption><br>\r\n"                        
                    str_page += "<a class=\"textlink\" href=\"querier_link\"><img src=\"./img/ok_button.gif\" border=\"0\" alt=\"OK\"></a>"   
                 
                #   
                # Update data from project info table
                #
                elif args.has_key('do_action') and args['do_action'] == 'Update':
                    r = 0
                    for k in args.keys():
                        if string.find(k,'action_')>=0:
                            x,id = string.split(k,'_')
                            do = ''
                            va = ''
                            if args.has_key('domain_'+id) and args['domain_'+id] != '' and \
                            args.has_key('value_'+id) and args['value_'+id] != '':
                                do = urllib.unquote_plus(args['domain_'+id])
                                va = urllib.unquote_plus(args['value_'+id])                
                                        
                                r = self.pan_db.Update_Project_Info(self.current_project_id,id,va,do)
                                if r == -1:
                                    self.LogError(self.pan_db.GetErrorMsg())
                    if r == -1:
                        str_page += "<caption><b>Error Updating Data to Project Info</b></caption><br>\r\n"
                    else:
                        str_page += "<caption><b>Data Updated in Project Info</b></caption><br>\r\n"                        
                    str_page += "<a class=\"textlink\" href=\"querier_link\"><img src=\"./img/ok_button.gif\" border=\"0\" alt=\"OK\"></a>"   
                 
                #   
                # Delete data from project info table
                #
                elif args.has_key('do_action_file_delete') and args['do_action_file_delete'] == 'Delete':
                    
                    r = 0
                    for k in args.keys():
                        if string.find(k,'deletefile_')>=0:
                            x,id = string.split(k,'_')
                            r = self.pan_db.Return_Link_File(self.current_project_id,id)
                            if r == -1:    
                                self.LogError(self.pan_db.GetErrorMsg())
                            self.HardDeletePage(id)
                            self.DeleteDir(r)
                        if string.find(k,'deletegood_')>=0:
                            x,id = string.split(k,'_')
                            r = self.pan_db.Return_Link_File(self.current_project_id,id)
                            if r == -1:    
                                self.LogError(self.pan_db.GetErrorMsg())
                            self.HardDeletePage(id)
                            self.DeleteDir(r)
                        if string.find(k,'deletebad_')>=0:
                            x,id = string.split(k,'_')
                            r = self.pan_db.Delete_Not_Save_Link(self.current_project_id,id)
                            if r == -1:    
                                self.LogError(self.pan_db.GetErrorMsg())
                    if r == -1:
                        str_page += "<caption><b>Error Deleting Links</b></caption><br>\r\n"
                    else:
                        str_page += "<caption><b>Links Deleted</b></caption><br>\r\n"                        
                    str_page += "<a class=\"textlink\" href=\"querier_link\"><img src=\"./img/ok_button.gif\" border=\"0\" alt=\"OK\"></a>"   
                    
                #
                # Unkown..
                #
                else:
                    str_page += "<caption><b>Unknow Option: Maybe some error with data? Try again please.</b></caption>\r\n"
                    
        else:
            str_page += "We are sorry but you must be inside a project.<br>\r\n"

        str += html.CreatePage("Data Miner", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def RepeaterHTMLPage(self):
        '''
    
        '''
        """UI repeater page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Repeater",1)                

        # create pantera interceptor

    
        str += html.CreatePage("Repeater", "", str_page)                
        return str
    
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def TutorialHTMLPage(self):
        '''
    
        '''
        """UI help page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Tutorial",1)                

        # create pantera interceptor

    
        str += html.CreatePage("Tutorial", "", str_page)                
        return str
    
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def HelpHTMLPage(self):
        '''
    
        '''
        """UI help page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Help",1)                

        # create pantera help page

    
        str += html.CreatePage("Help", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def AboutPanteraHTMLPage(self):
        '''
    
        '''
        """UI about page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("About OWASP Pantera",1)                

        # create pantera interceptor
        str_page += "<table>"
        str_page += "<td>"
        #str_page += "<tr>"
        str_page += "<img src=\"./img/panther.gif\"/>"
        #str_page += "</tr>"
        #str_page += "<tr>"
        str_page += "</td>"
        str_page += "<td>"
        str_page += "<b>Pantera UI Version:</b> %s<br>\r\n" % __UIVERSION__
        str_page += "<b>Pantera Version:</b> %s<br>\r\n" % pantera.__version__
        str_page += "<b>Lead Developer:</b> Simon Roses Femerling<br>\r\n"
        str_page += "<b>Email:</b> pantera.proxy@gmail.com<br>\r\n"
        str_page += "<p>Visit http://www.owasp.org or http://www.roseslabs.com to contact the developers.</p>\r\n"
        str_page += "<p>We will like to thanks all the betatesters and programs that has "
        str_page += " inspired Pantera development.</p><br>\r\n"
        str_page += "<p>OWASP Pantera GNU (C) 2006</p>\r\n"
        
        str_page += "</td>\r\n"
        str_page += "</table>\r\n"
    
        str += html.CreatePage("About OWASP Pantera", "", str_page)                
        return str
    
    #############################################################################################
    # FUNC     :    
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ExternalDocHTMLPage(self):
        '''
    
        '''
        """External Doc page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("External Documentation",1)                

        # document links
        str_page += "<ul>"
        #str_page += "<li><a href=\"%s\">HTTP State Management Mechanism (RFC 2019)</a></li>" % panterautils.pathjoin(self.panteradir,"pdoc", "rfc2019.html")
        #str_page += "<li><a href=\"./pdoc/rfc2019.htm\">HTTP State Management Mechanism (RFC 2019)</a></li>" #% panterautils.pathjoin(self.panteradir,"pdoc", "rfc2019.html")
        str_page += "</ul>"
    
        str += html.CreatePage("External Documentation", "", str_page)                
        return str    
    
    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ErrorConsoleHTMLPage(self):
        '''
    
        '''
        """Error Console Page"""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Error Console",1)                

        # create our session trace raw page

        #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br>\r\n" 
        str_page += "<textarea name=\"raw_asession\" cols=100 rows=25 readonly>"
        str_page += self.ReadErrorLog()
        str_page += "</textarea>\r\n"
        
        str += html.CreatePage("Error Console", "", str_page)                
        return str    

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def PPAHTMLPage(self):
        '''
    
        '''
        """PPA Management Page"""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        # create PPA management page

        if self.inside_project == YES:

            html.SetDataHeader("<script language=\"JavaScript\" src=\"public_scripts.js\"></script>")

            str += html.CreateHeader("PPA Management",1)                

            self.ssl_title = 0
            self.email_title = 0
            self.script_title = 0
            self.form_title = 0
            self.auth_title = 0
            self.auth_form_title = 0
            self.cookie_title = 0            
            self.session_id_title = 0        
            self.external_link_title = 0        
            self.comment_title = 0        
            self.vuln_title = 0        
            self.hidden_title = 0        
            self.object_title = 0        
            self.postauth_title = 0        
            self.querystr_title = 0      
            self.recon_title = 0  

            str_page += "<h3>Pantera Passive Analyzer Plugins</h3>\r\n"

            str_page += "<form action=\"/ppa_form\" method=\"POST\">\r\n"
            str_page += "<input type=\"submit\"> | "
            str_page += "<input type=\"checkbox\" name=\"checkall\" onclick=\"checkUncheckAll(this);\"/>Select/Unselect All Plugins\n\r"
            str_page += "<br><br>\r\n"
            str_page += "<table border=0>\r\n"

            # NEED FIX AS input and textarea tags are send inside POST form. 

            self.ppa_plugins.DeletePlugins()
            self.ppa_plugins.LoadPlugins() # Make sure that all plugins are displayed
            for mod in self.ppa_plugins.GetPlugins():
                mod_dat = self.ppa_plugins.GetPluginData(mod)
                if mod_dat is None:
                    continue
                check = self.ppa_plugins.CheckTag(mod_dat['tag']) 
                if check == 1 and mod_dat.has_key('tag'):
                    if mod_dat['tag'] == 'ssl':
                        if self.ssl_title == 0:
                            str_page += "<tr><td><h4>SSL</h4></td></tr>\r\n"
                            self.ssl_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled /></td>\r\n" % panterautils.FilterBadCar(mod_dat['name'])
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled /></td>\r\n" % panterautils.FilterBadCar(mod_dat['version'])
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled /></td>\r\n" % panterautils.FilterBadCar(mod_dat['author'])
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled /></td>\r\n" % panterautils.FilterBadCar(mod_dat['email'])
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'email':
                        if self.email_title == 0:
                            str_page += "<tr><td><h4>Email</h4></td></tr>\r\n"
                            self.email_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'script':
                        if self.script_title == 0:
                            str_page += "<tr><td><h4>Script</h4></td></tr>\r\n"
                            self.script_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'form':
                        if self.form_title == 0:
                            str_page += "<tr><td><h4>Form</h4></td></tr>\r\n"
                            self.form_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'auth':
                        if self.auth_title == 0:
                            str_page += "<tr><td><h4>Authentication</h4></td></tr>\r\n"
                            self.auth_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'auth_form':
                        if self.auth_form_title == 0:
                            str_page += "<tr><td><h4>Auth Form</h4></td></tr>\r\n"
                            self.auth_form_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'cookie':
                        if self.cookie_title == 0:
                            str_page += "<tr><td><h4>Cookie</h4></td></tr>\r\n"
                            self.cookie_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'session_id':
                        if self.session_id_title == 0:
                            str_page += "<tr><td><h4>Session ID</h4></td></tr>\r\n"
                            self.session_id_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'external_link':
                        if self.external_link_title == 0:
                            str_page += "<tr><td><h4>External Link</h4></td></tr>\r\n"
                            self.external_link_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'comment':
                        if self.comment_title == 0:
                            str_page += "<tr><td><h4>Comment</h4></td></tr>\r\n"
                            self.comment_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'vuln':
                        if self.vuln_title == 0:
                            str_page += "<tr><td><h4>Vulnerabilities</h4></td></tr>\r\n"
                            self.vuln_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'hidden':
                        if self.hidden_title == 0:
                            str_page += "<tr><td><h4>Hidden</h4></td></tr>\r\n"
                            self.hidden_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"                   
                    if mod_dat['tag'] == 'object':
                        if self.object_title == 0:
                            str_page += "<tr><td><h4>Object</h4></td></tr>\r\n"
                            self.object_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n" 
                    if mod_dat['tag'] == 'postauth':
                        if self.postauth_title == 0:
                            str_page += "<tr><td><h4>PostAuth</h4></td></tr>\r\n"
                            self.postauth_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n" 
                    if mod_dat['tag'] == 'querystr':
                        if self.querystr_title == 0:
                            str_page += "<tr><td><h4>Query Params</h4></td></tr>\r\n"
                            self.querystr_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
                    if mod_dat['tag'] == 'recon':
                        if self.recon_title == 0:
                            str_page += "<tr><td><h4>Recon</h4></td></tr>\r\n"
                            self.recon_title = 1
                        str_page += "<tr>\r\n"    
                        if mod in self.ppa_plugin_list:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"yes\" checked></td>\r\n" % mod_dat['id']
                        else:
                            str_page += "<td><input type=\"checkbox\" name=\"%s\" value=\"no\"></td>\r\n" % mod_dat['id']                            
                        str_page += "<td>Name: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['name']
                        str_page += "<td>Version: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"5\" disabled></td>\r\n" % mod_dat['version']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Author: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['author']
                        str_page += "<td>Email: </td>\r\n"
                        str_page += "<td><input type=\"text\" name=\"\" value=\"%s\" size=\"25\" disabled></td>\r\n" % mod_dat['email']
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>&nbsp;</td>\r\n"
                        str_page += "<td>Description: </td>\r\n"
                        str_page += "<td><textarea name=\"desc\" cols=30 rows=3 disabled>"
                        str_page += "%s" % panterautils.FilterBadCar(mod_dat['desc'])
                        str_page += "</textarea></td>\r\n"
                        str_page += "</tr>\r\n"
            str_page += "</table>\r\n"
            str_page += "<br>\r\n"    
            str_page += "<input type=\"submit\"> | "
            str_page += "<input type=\"checkbox\" name=\"checkall\" onclick=\"checkUncheckAll(this);\"/>Select/Unselect All Plugins\n\r"
            str_page += "</FORM>\r\n"
            
        else:
            str += html.CreateHeader("PPA Management",1)                
            str_page += "We are sorry but you must be inside a project.<br>\r\n"
            
        str += html.CreatePage("PPA Management", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ProjectNotesHTMLPage(self):
        '''
    
        '''
        """UI project notes page."""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Project Notes",1)                

        if self.inside_project == YES:
    
            # create our project notes page

            #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br>\r\n"
            str_page += "<form action=\"/project_notes_form\" method=\"POST\">\r\n"
            str_page += "<textarea name=\"project_notes\" cols=100 rows=25>"
            c = self.pan_db.Return_Project_Notes(self.current_project_id)
            if c == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            c = c[0]
            if c[0] is not None:
                str_page += "%s" % c[0]
            str_page += "</textarea>\r\n"
            str_page += "<td><input type=\"submit\"></td>"
            str_page += "</form>\r\n"

        else:
            str_page += "We are sorry but you must be inside a project.<br>\r\n"
            
        str += html.CreatePage("Project Notes", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def LinkNotesHTMLPage(self, id):
        '''
    
        '''
        """UI link notes page."""
        str = ""
        str_page = ""
        str2 = ""
        html = panteraHTML.PanteraHTML()


        if self.inside_project == YES:

            file = self.pan_db.Return_Link_File(self.current_project_id,id)
            if file == -1:    
                self.LogError(self.pan_db.GetErrorMsg())
                return self.ErrorHTMLPage("Invalid Value")

            ch = self.pf.displayClientHeader(file)
            str2 += "http"
            if ch.clientisSSL:
                str2 += "s"
            str2 += "://%s%s" % (ch.connectHost,ch.URL) 
            args = ""
            if ch.useRawArguments == 0:
                if len(ch.allURLargs) > 0:
                    args += "?%s" % ch.allURLargs
                else:
                    if len(ch.URLargsDict) > 0:
                        args += "?"
                        args += panterautils.joinargs(ch.URLargsDict,orderlist=ch.orderlist)
            str2 += args

            str_page += html.CreateHeader("Link Notes: " +str2,1)
    
            # create our project notes page

            #str_page += "[<a href=\"javascript: history.go(-1)\">Back</a>]<br>\r\n"
            str_page += "<form action=\"/link_notes_form\" method=\"POST\">\r\n"
            str_page += "<textarea NAME=\"link_notes\" COLS=100 ROWS=25>"
            c = self.pan_db.Return_Link_Notes(self.current_project_id,id)
            if c == -1:
                self.LogError(self.pan_db.GetErrorMsg())
            c = c[0]
            if c[0] is not None:
                str_page += "%s" % c[0]
            str_page += "</textarea>\r\n"
            str_page += "<input type=\"hidden\" value=\"%s\" name=\"id\"></td>\r\n" % id
            str_page += "<td><input type=\"submit\"></td>"
            str_page += "</form>\r\n"
        else:
            str_page += html.CreateHeader("Link Notes:",1)
            
            str_page += "We are sorry but you must be inside a project.<br>\r\n"
            
        str += html.CreatePage("Link Notes", str2, str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def HTTPEditorHTMLPage(self, args = ''):
        '''
    
        '''
        """ HTTP Editor Page"""
        sstr = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        sstr += html.CreateHeader("Pantera HTTP Editor ",1)                

        #
        # create our http editor
        #
        
        my_dat = {}
        load_file = 0
        myh = None
        myb = None
        send_flag = 0
        data = ''
        
        port_flag = 0
        host_flag = 0
        if args != '':
            if args.has_key('load_http_editor_profile'):
                
                xmlp = panteraLib.HTTPEditorParser()
                profile = urllib.unquote_plus(args['profile'])
                fil = panterautils.pathjoin(self.http_editor_dir,profile)
                xmlp.SetFileName(fil)
                res = xmlp.BeginParser()
                if res != 0:
                    return self.ErrorHTMLPage(xmlp.GetError())
                
                my_dat = xmlp.ReturnData()
                load_file = 1
                
            if args.has_key('http_editor_send_request'):
                send_flag = 1
                                
                myheader = args['myheader']
                mybody = args['mybody']
                
                data=self.sendrequest(myheader,mybody,2)
                
        #
        # first do http editor profile form
        #
        
        str_page += "<table>\r\n"
        str_page += "<caption>HTTP Editor Profiles</caption>\r\n"
        str_page += "<tr>\r\n"
        str_page += "<form action=\"create_http_editor_profile\" method=\"POST\">\r\n"
        str_page += "<td><input type=\"submit\" value=\"Create & Edit Profile\"></td>"
        str_page += "</form>\r\n"
        str_page += "<td>&nbsp;</td>\r\n"
        str_page += "<td>&nbsp;</td>\r\n"
        str_page += "<td>&nbsp;</td>\r\n"
        files=os.listdir(self.http_editor_dir)
        l = []
        add = l.append
        for i in files:
            fullpath=os.path.join(self.http_editor_dir, i)
            if os.path.isdir(fullpath):
                continue
            else:
                add(i)
        l.sort()
        if len(l)>0:
            str_page += "<form action=\"http_editor_form\" method=\"POST\">\r\n"
            str_page += "%s" % panterautils.printFormSelectMenu('Profile:','profile',l,1)
            str_page+= "%s\r\n" % panterautils.printHiddenEntry("load_http_editor_profile","yes",1)
            str_page += "<td><input type=\"submit\" name=\"action\" value=\"Load Profile\"></td>"
            str_page += "</form>\r\n"
        str_page += "</tr>\r\n"        
        str_page += "</table>\r\n"
        
        #
        # Create editor form
        #

        if load_file == 1:
            str_page += "<h3>HTTP Editor Information (Profile: %s)</h3>\r\n" % panterautils.FilterBadCar(urllib.unquote_plus(args['profile']),1)        
        else:
            str_page += "<h3>HTTP Editor Information</h3>\r\n"        
        str_page+="<form action=\"http_editor_form\" method=\"POST\">\r\n"
        str_page += "<table><tr><td></td></tr>\r\n"
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printHiddenEntry("PANTERA_TRIGGER","yes",1)
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printHiddenEntry("http_editor_send_request","yes",1)
        if load_file == 1:
            if my_dat['verb_style'] == 'raw':
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Verb:", "Verb", my_dat['verb'],1)
            elif my_dat['verb_style'] == 'list':
                a = my_dat['verb']
                try:
                    l = string.split(a,',')
                except:
                    str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Verb:", "Verb", my_dat['verb'],1)
                str_page += "%s" % panterautils.printFormSelectMenu('Verb:','Verb',l,1)
            else:
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Verb:", "Verb", 'GET',1)
        else:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Verb:", "Verb", 'GET',1)
        if load_file == 1:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("URL:","URL",my_dat['url'],1,60)
        else:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("URL:","URL",'/',1,60)
        if load_file == 1:
            if my_dat['version_style'] == 'raw':
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Version:", "version", my_dat['version'],1)
            elif my_dat['version_style'] == 'list':
                a = my_dat['version']
                try:
                    l = string.split(a,',')
                except:
                    str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Version:", "version", my_dat['version'],1)
                str_page += "%s" % panterautils.printFormSelectMenu('Version:','version',l,1)
            else:
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Version:", "version", '',1)
        else:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Version:", "version", '',1)
        if load_file == 1:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Host:", "ConnectHost", my_dat['connecthost'],1)
        else:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Host:", "ConnectHost", '',1)
        if load_file == 1:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Port:","ConnectPort", my_dat['connectport'],1)
        else:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Port:","ConnectPort", '80',1)
        if load_file == 1:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("SSL","SSL",my_dat['ssl'],1)
        else:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("SSL","SSL",0,1)

        str_page += "</table>\r\n"
        str_page += "<table>\r\n"
        str_page += "<tr><td><h3>Headers</h3></td></tr>"
        i = 0
        if load_file == 1:
            l = my_dat['header_params']
            if len(l)>0:
                for c in l:
                    n,v = c
                    if n == '': continue
                    str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("Header"+str(i),n,v,1,30)
                    i += 1
            for i in range(i,i+my_dat['header_num'],1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("Header"+str(i),"","",1,30)
                i += 1
        else:
            for i in range(i,i+5,1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("Header"+str(i),"","",1,30)
                i += 1
        str_page +="<tr><td><h2>URL Args</h2></td></tr>\r\n"
        i=0
        if load_file == 1:
            l = my_dat['url_params']
            if len(l)>0:
                for c in l:
                    n,v = c
                    if n == '': continue
                    str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("URLArg"+str(i),n,v,1,30)
                    i += 1
            for i in range(i,i+my_dat['url_num'],1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("URLArg"+str(i),"","",1,30)
                i += 1
        else:
            for i in range(i,i+5,1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("URLArg"+str(i),"","",1,30)
                i += 1
        i=0
        str_page+="<tr><td><h2>Body Args</h2></td></tr>\r\n"
        if load_file == 1:
            l = my_dat['body_params']
            if len(l)>0:
                for c in l:
                    n,v = c
                    if n == '': continue
                    str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("BodyArg"+str(i),n,v,1,30)
                    i += 1
            for i in range(i,i+my_dat['body_num'],1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("BodyArg"+str(i),"","",1,30)
                i += 1
        else:
            for i in range(i,i+5,1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("BodyArg"+str(i),"","",1,30)
                i += 1
        str_page += "<tr>\r\n"    
        str_page += "<td><input type=\"submit\"></td>"
        str_page += "<td><input type=\"reset\"></td>"
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        str_page += "</form>"

        # Show data
        
        if send_flag == 1:
            
            str_page += "<hr>"
            if data != "Error: Missing host!!":
                str_page += "<h3>Request</h3>\r\n"    
                str_page += "<textarea name=\"editor_request\" cols=90 rows=15>"
                str_page += panterautils.constructRequest(self.http_editor_myheader, self.http_editor_mybody)
                str_page += "</textarea>\r\n"
            str_page += "<h3>Response</h3>\r\n"    
            str_page += "<textarea name=\"editor_response\" cols=90 rows=15>"
            str_page += data
            str_page += "</textarea>\r\n"
        
        sstr += html.CreatePage("Pantera HTTP Editor", "", str_page)                
        return sstr    

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def HTTPEditorProfileCreatorHTMLPage(self, args = ''):
        '''
    
        '''
        """ """
        sstr = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        sstr += html.CreateHeader("Pantera HTTP Editor Profile Creator",1)                

        # create our HTTP editor Profile Creator

        flag = 0
        my_dat = {}
        load_file = 0
        if args != '':
        
            # Load xml file into dict
            if args.has_key('open_http_editor_profile'):
                
                xmlp = panteraLib.HTTPEditorParser()
                profile = urllib.unquote_plus(args['profile'])
                fil = panterautils.pathjoin(self.http_editor_dir,profile)
                xmlp.SetFileName(fil)
                res = xmlp.BeginParser()
                if res != 0:
                    return self.ErrorHTMLPage(xmlp.GetError())
                
                my_dat = xmlp.ReturnData()
                load_file = 1
                
            # Create a new http editor profile file
            if args.has_key('new_http_editor_profile'):
                if not args.has_key('file') or args['file'] == '':
                    flag = 1
                else:
                    did = []
                    page=panterautils.pathjoin(self.http_editor_dir,args['file'])
                    if page.count('.xml')<=0: # if we are mising xml extension then added
                        page += '.xml'
                    
                    fp = panteraFile.PanteraFileManager()
                    fp.SetFilename(page)
                    r = fp.OpenFile()
                    if r == -1:
                        return self.ErrorHTMLPage(fp.GetErroMsg())
                    
                    sdat = ""
                    sdat += "<http_editor_profile>\r\n"
                    if args.has_key('verb_style'):
                        sdat += " <verb_style>%s</verb_style>\r\n" % panterautils.FixStringForXML(urllib.unquote_plus(args['verb_style']))
                    else:
                        sdat += " <verb_style>raw</verb_style>\r\n"
                    if args.has_key('verb'):
                        sdat += " <verb>%s</verb>\r\n" % panterautils.FixStringForXML(urllib.unquote_plus(args['verb']))
                    else:
                        sdat += " <verb></verb>\r\n"                                     
                    if args.has_key('url'):
                        sdat += " <url>%s</url>\r\n" % panterautils.FixStringForXML(urllib.unquote_plus(args['url']))
                    else:
                        sdat += " <url></url>\r\n" 
                    if args.has_key('version_style'):
                        sdat += " <version_style>%s</version_style>\r\n" % panterautils.FixStringForXML(urllib.unquote_plus(args['version_style']))
                    else:
                        sdat += " <version_style></version_style>\r\n" 
                    if args.has_key('version'):
                        sdat += " <version>%s</version>\r\n" % panterautils.FixStringForXML(urllib.unquote_plus(args['version']))
                    else:
                        sdat += " <version></version>\r\n"
                    if args.has_key('connecthost'):
                        sdat += " <connecthost>%s</connecthost>\r\n" % panterautils.FixStringForXML(urllib.unquote_plus(args['connecthost']))
                    else:
                        sdat += " <connecthost></connecthost>\r\n" 
                    if args.has_key('connectport'):
                        sdat += " <connectport>%s</connectport>\r\n" % panterautils.FixStringForXML(urllib.unquote_plus(args['connectport']))
                    else:
                        sdat += " <connectport></connectport>\r\n"
                    if args.has_key('ssl') and args['ssl'] == 'yes':
                        sdat += " <ssl>1</ssl>\r\n" 
                    else:
                        sdat += " <ssl>0</ssl>\r\n"
                    if args.has_key('url_num'):
                        sdat += " <url_params extra=\"%s\">\r\n" % panterautils.FixStringForXML(urllib.unquote_plus(args['url_num']))
                    else:
                        sdat += " <url_params extra=\"3\">\r\n"
                    for akey in args.keys():
                        if akey.count("url_param_"):
                            if akey in did:
                                continue
                            if akey[-1]=="N":
                                valuename=akey[:-1]+"V"
                                headername=akey
                            else:
                                valuename=akey
                                headername=akey[:-1]+"N"
                            header=''
                            value=''
                            header=panterautils.FixStringForXML(urllib.unquote_plus(args[headername]))
                            value=panterautils.FixStringForXML(urllib.unquote_plus(args[valuename]))
                            did.append(valuename)
                            did.append(headername)
                            if header != '':
                                sdat += "  <url_param name=\"%s\" value=\"%s\" />\r\n" % (header, value)
                    sdat += " </url_params>\r\n"
                    if args.has_key('header_num'):
                        sdat += " <header_params extra=\"%s\">\r\n" % panterautils.FixStringForXML(urllib.unquote_plus(args['header_num']))
                    else:
                        sdat += " <header_params extra=\"3\">\r\n"
                    for akey in args.keys():
                        if akey.count("header_param_"):
                            if akey in did:
                                continue
                            if akey[-1]=="N":
                                valuename=akey[:-1]+"V"
                                headername=akey
                            else:
                                valuename=akey
                                headername=akey[:-1]+"N"
                            header=''
                            value=''
                            header=panterautils.FixStringForXML(urllib.unquote_plus(args[headername]))
                            value=panterautils.FixStringForXML(urllib.unquote_plus(args[valuename]))
                            did.append(valuename)
                            did.append(headername)
                            if header != '':                
                                sdat += "  <header_param name=\"%s\" value=\"%s\" />\r\n" % (header, value)
                    sdat += " </header_params>\r\n"
                    if args.has_key('body_num'):
                        sdat += " <body_params extra=\"%s\">\r\n" % panterautils.FixStringForXML(urllib.unquote_plus(args['body_num']))
                    else:
                        sdat += " <body_params extra=\"3\">\r\n"
                    for akey in args.keys():
                        if akey.count("body_param_"):
                            if akey in did:
                                continue
                            if akey[-1]=="N":
                                valuename=akey[:-1]+"V"
                                headername=akey
                            else:
                                valuename=akey
                                headername=akey[:-1]+"N"
                            header=''
                            value=''
                            header=panterautils.FixStringForXML(urllib.unquote_plus(args[headername]))
                            value=panterautils.FixStringForXML(urllib.unquote_plus(args[valuename]))
                            did.append(valuename)
                            did.append(headername)
                            if header != '':                
                                sdat += "  <body_param name=\"%s\" value=\"%s\" />\r\n" % (header, value)
                    sdat += " </body_params>\r\n"
                    sdat += "</http_editor_profile>\r\n"
                    
                    r = fp.WriteData(sdat)
                    if r == -1:
                        return self.ErrorHTMLPage(fp.GetErroMsg())
                        
                    fp.CloseFile()
        
                    str_page += "<caption><b>HTTP Editor Profile Created</b></caption><br>\r\n"
                    str_page += "<a class=\"textlink\" href=\"http_editor_link\"><img src=\"./img/ok_button.gif\" border=\"0\" alt=\"OK\"></a>"   
                    sstr += html.CreatePage("Pantera HTTP Editor Profile Creator", "", str_page)                
                    return sstr

        #
        # HTTP Editor Creator Form
        #
        files=os.listdir(self.http_editor_dir)
        l = []
        add = l.append
        for i in files:
            fullpath=os.path.join(self.http_editor_dir, i)
            if os.path.isdir(fullpath):
                continue
            else:
                add(i)
        l.sort()
        if len(l)>0:
            str_page += "<table>\r\n"
            str_page += "<caption>Open Profile for Edit</caption>\r\n"
            str_page += "<form action=\"create_http_editor_profile_form\" method=\"POST\">\r\n"
            str_page += "%s" % panterautils.printFormSelectMenu('Profile:','profile',l,1)
            str_page+= "%s\r\n" % panterautils.printHiddenEntry("open_http_editor_profile","yes",1) 
            str_page += "<td><input type=\"submit\" name=\"action\" value=\"Edit Profile\"></td>"
            str_page += "</form>\r\n"
            str_page += "</table>\r\n"        
        
        # New Profile Form
        str_page += "<h3>Create HTTP Editor Profile</h3>\r\n"
        str_page+="<form action=\"create_http_editor_profile_form\" method=\"POST\">\r\n"
        str_page += "<table>\r\n"
        if flag == 0:
            if args != '' and load_file == 1:
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Profile File", "file", urllib.unquote_plus(args['profile']),1,20)
            else:
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Profile File", "file", '',1,20)
        else:
            str_page+= "<tr>%s<td><font color=\"red\">* Missing Filename</font></td></tr>\r\n" % panterautils.printFormEntry("Profile File", "file", '',1,20)
        str_page += "</table>\r\n"           
        str_page += "<table>\r\n"
        str_page+= "<tr>%s</tr>\r\n" % panterautils.printHiddenEntry("new_http_editor_profile","yes",1)
        str_page += "<tr>\r\n"
        str_page += "<td>Verb Style</td>\r\n"
        if load_file == 1:
            if my_dat['verb_style'] == 'raw':
                str_page += "<td><input type=\"radio\" name=\"verb_style\" value=\"raw\" checked>Raw</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"verb_style\" value=\"list\">List</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"verb_style\" value=\"raw\" >Raw</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"verb_style\" value=\"list\" checked>List</td>\r\n"                
        else:
            str_page += "<td><input type=\"radio\" name=\"verb_style\" value=\"raw\" checked>Raw</td>\r\n"
            str_page += "<td><input type=\"radio\" name=\"verb_style\" value=\"list\">List</td>\r\n"                
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        str_page += "<table>\r\n"   
        if load_file == 1:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Verb", "verb", my_dat['verb'],1)
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("URL","url",my_dat['url'],1,60)
        else:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Verb", "verb", 'GET',1)
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("URL","url",'/',1,60)
        str_page += "</table>\r\n"
        str_page += "<table>\r\n"   
        str_page += "<tr>\r\n"
        str_page += "<td>Version Style</td>\r\n"
        if load_file == 1:
            if my_dat['version_style'] == 'raw':
                str_page += "<td><input type=\"radio\" name=\"version_style\" value=\"raw\" checked>Raw</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"version_style\" value=\"list\">List</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"version_style\" value=\"raw\" >Raw</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"version_style\" value=\"list\" checked>List</td>\r\n"                
        else:
            str_page += "<td><input type=\"radio\" name=\"version_style\" value=\"raw\" checked>Raw</td>\r\n"
            str_page += "<td><input type=\"radio\" name=\"version_style\" value=\"list\">List</td>\r\n"                
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        str_page += "<table>\r\n"
        if load_file == 1:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Version","version",my_dat['version'],1)
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Host", "connecthost", my_dat['connecthost'],1)
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Port","connectport", my_dat['connectport'],1)
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("SSL","ssl",my_dat['ssl'],1)
        else:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Version","version",'',1)
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Host", "connecthost", '',1)
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Port","connectport", '80',1)
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormCheckbox("SSL","ssl",0,1)
        str_page += "<tr><td><h3>Headers</h3></td></tr>"
        if load_file == 1:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Headers","header_num",my_dat['header_num'],1,5)
        else:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Headers","header_num",'3',1,5)
        i=0
        if load_file == 1:
            l = my_dat['header_params']
            if len(l)>0:
                for c in l:
                    n,v = c
                    if n == '': continue
                    str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("header_param_"+str(i),n,v,1,30)
                    i += 1
            for i in range(i,i+5,1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("header_param_"+str(i),"","",1,30)
                i += 1
        else:
            for i in range(i,i+5,1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("header_param_"+str(i),"","",1,30)
                i += 1
        str_page += "<tr><td><h3>URL Args</h3></td></tr>"
        if load_file == 1:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("URL","url_num",my_dat['url_num'],1,5)
        else:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("URL","url_num",'3',1,5)
        i=0
        if load_file == 1:
            l = my_dat['url_params']
            if len(l)>0:
                for c in l:
                    n,v = c
                    if n == '': continue
                    str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("url_param_"+str(i),n,v,1,30)
                    i += 1
            for i in range(i,i+5,1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("url_param_"+str(i),"","",1,30)
                i += 1
        else:
            for i in range(i,i+5,1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("url_param_"+str(i),"","",1,30)
                i += 1
        str_page += "<tr><td><h3>Body Args</h3></td></tr>"
        if load_file == 1:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Body","body_num",my_dat['body_num'],1,5)
        else:
            str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntry("Body","body_num",'3',1,5)
        i=0
        if load_file == 1:
            l = my_dat['body_params']
            if len(l)>0:
                for c in l:
                    n,v = c
                    if n == '': continue
                    str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("body_param_"+str(i),n,v,1,30)
                    i += 1
            for i in range(i,i+5,1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("body_param_"+str(i),"","",1,30)
                i += 1
        else:
            for i in range(i,i+5,1):
                str_page+= "<tr>%s</tr>\r\n" % panterautils.printFormEntryAndValue("body_param_"+str(i),"","",1,30)
                i += 1
        str_page += "</table>\r\n"
        str_page += "<table>\r\n"
        str_page += "<tr>\r\n"            
        str_page += "<td><input type=\"submit\"></td>"
        str_page += "<td><input type=\"reset\"></td>"
        str_page += "</tr>\r\n"
        str_page += "</table>\r\n"
        str_page += "</form>"
        str_page += "<a class=\"textlink\" href=\"http_editor_link\"><img src=\"./img/back_button.gif\" alt=\"Back\" border=\"0\"></a>\r\n"
        
        sstr += html.CreatePage("Pantera HTTP Editor Profile Creator", "", str_page)                
        return sstr

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ScanEngineHTMLPage(self):
        '''
    
        '''
        """ Scan Engine Page"""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Pantera Scan Engine",1)                

        # create our scan engine home page

        
        str += html.CreatePage("Pantera Scan Engine", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def AnalysisModHTMLPage(self):
        '''
    
        '''
        """ Analysis Modules Page"""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Analysis Modules",1)                

        # create our scan engine home page

        
        str += html.CreatePage("Analysis Modules", "", str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def ValidatePageHTMLPage(self, id):
        '''
    
        '''
        """UI validate page. page."""
        str = ""
        str_page = ""
        str2 = ""
        html = panteraHTML.PanteraHTML()

        if self.inside_project == YES:

            file = self.pan_db.Return_Link_File(self.current_project_id,id)
            if file == -1:    
                self.LogError(self.pan_db.GetErrorMsg())
                return self.ErrorHTMLPage("Invalid Value")

            ch = self.pf.displayClientHeader(file)
            str2 += "http"
            if ch.clientisSSL:
                str2 += "s"
            str2 += "://%s%s" % (ch.connectHost,ch.URL) 
            args = ""
            if ch.useRawArguments == 0:
                if len(ch.allURLargs) > 0:
                    args += "?%s" % ch.allURLargs
                else:
                    if len(ch.URLargsDict) > 0:
                        args += "?"
                        args += panterautils.joinargs(ch.URLargsDict,orderlist=ch.orderlist)
            str2 += args

            str_page += html.CreateHeader("Validate Page: " +str2,1)
    
            # create our valudate page

            dat = {}
            dat = self.pan_db.Select_Page_Info(self.current_project_id,id)
            if dat == -1 or not len(dat)>0:
                self.LogError(self.pan_db.GetErrorMsg())
                return self.ErrorHTMLPage("Error with link")
            
            str_page += "<form action=\"validate_page_form\" method=\"POST\">\r\n"
            str_page += "<table border=\"0\" width=\"10%\">\r\n"

            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/lock_icon.png\" alt=\"SSL\" ></td>\r\n"
            str_page += "<td>SSL:</td>\r\n"
            if dat.has_key('have_ssl') and dat['have_ssl'] == 1:
                str_page += "<td><input type=\"radio\" name=\"ssl\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"ssl\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"ssl\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"ssl\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"
            
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_auth.gif\" alt=\"Authentication\"></td>\r\n"
            str_page += "<td>Authentication:</td>\r\n"
            if dat.has_key('have_auth') and dat['have_auth'] == 1:
                str_page += "<td><input type=\"radio\" name=\"auth\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"auth\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"auth\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"auth\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"
                            
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_script.png\" alt=\"Script\" ></td>\r\n"
            str_page += "<td>Script:</td>\r\n"
            if dat.has_key('have_script') and dat['have_script'] == 1:
                str_page += "<td><input type=\"radio\" name=\"script\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"script\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"script\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"script\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"
                     
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_form.png\" alt=\"Form\" ></td>\r\n"
            str_page += "<td>Form:</td>\r\n"
            if dat.has_key('have_form') and dat['have_form'] == 1:
                str_page += "<td><input type=\"radio\" name=\"form\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"form\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"form\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"form\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"
            
            str_page += "<tr>\r\n"    
            str_page += "<td><img src=\"./img/icon_authform.png\" alt=\"Auth Form\" ></td>\r\n"
            str_page += "<td>Authentication Form:</td>\r\n"
            if dat.has_key('have_auth_form') and dat['have_auth_form'] == 1:
                str_page += "<td><input type=\"radio\" name=\"auth_form\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"auth_form\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"auth_form\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"auth_form\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"
                
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_hidden.png\" alt=\"Hidden\" ></td>\r\n"
            str_page += "<td>Hidden</td>\r\n"
            if dat.has_key('have_hidden') and dat['have_hidden'] == 1:
                str_page += "<td><input type=\"radio\" name=\"hidden\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"hidden\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"hidden\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"hidden\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"
                    
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_cookie.png\" alt=\"Cookie\" ></td>\r\n"
            str_page += "<td>Cookie:</td>\r\n"
            if dat.has_key('have_cookie') and dat['have_cookie'] == 1:
                str_page += "<td><input type=\"radio\" name=\"cookie\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"cookie\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"cookie\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"cookie\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"                
                
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_session_id.gif\" alt=\"Session ID\" ></td>\r\n"
            str_page += "<td>Session ID:</td>\r\n"
            if dat.has_key('have_session_id') and dat['have_session_id'] == 1:
                str_page += "<td><input type=\"radio\" name=\"session_id\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"session_id\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"session_id\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"session_id\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"
            
            str_page += "<tr>\r\n"    
            str_page += "<td><img src=\"./img/icon_comment.png\" alt=\"Comment\" ></td>\r\n"
            str_page += "<td>Comment:</td>\r\n"
            if dat.has_key('have_comment') and dat['have_comment'] == 1:
                str_page += "<td><input type=\"radio\" name=\"session_id\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"session_id\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"session_id\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"session_id\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"
                    
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_querystr.png\" alt=\"URL Params\" ></td>\r\n"
            str_page += "<td>Query Params:</td>\r\n"
            if dat.has_key('have_querystr') and dat['have_querystr'] == 1:
                str_page += "<td><input type=\"radio\" name=\"querystr\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"querystr\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"querystr\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"querystr\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"
                
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/url.png\" alt=\"External Link\" ></td>\r\n"
            str_page += "<td>External Links:</td>\r\n"
            if dat.has_key('have_external_link') and dat['have_external_link'] == 1:
                str_page += "<td><input type=\"radio\" name=\"external_link\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"external_link\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"external_link\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"external_link\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"                
                
            str_page += "<tr>\r\n"                
            str_page += "<td><img src=\"./img/icon_target.png\" alt=\"Vuln\" ></td>\r\n"
            str_page += "<td>Attack Vector:</td>\r\n"
            if dat.has_key('have_vuln') and dat['have_vuln'] == 1:
                str_page += "<td><input type=\"radio\" name=\"vuln\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"vuln\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"vuln\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"vuln\" value=\"0\" checked>No</td>\r\n"                
            str_page += "<tr>\r\n"
            
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_activex.png\" alt=\"Object\" ></td>\r\n"
            str_page += "<td>Object:</td>\r\n"
            if dat.has_key('have_object') and dat['have_object'] == 1:
                str_page += "<td><input type=\"radio\" name=\"object\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"object\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"object\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"object\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"
                
            str_page += "<tr>\r\n"
            str_page += "<td><img src=\"./img/icon_postauth.png\" alt=\"Post Auth\" ></td>\r\n"
            str_page += "<td>Post Authentication:</td>\r\n"
            if dat.has_key('have_postauth') and dat['have_postauth'] == 1:
                str_page += "<td><input type=\"radio\" name=\"postauth\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"postauth\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"postauth\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"postauth\" value=\"0\" checked>No</td>\r\n"                
            str_page += "</tr>\r\n"
                
            str_page += "<tr>\r\n"                
            str_page += "<td><img src=\"./img/icon_email.png\" alt=\"Email\" ></td>\r\n"
            str_page += "<td>Mail:</td>\r\n"
            if dat.has_key('have_email') and dat['have_email'] == 1:
                str_page += "<td><input type=\"radio\" name=\"email\" value=\"1\" checked>Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"email\" value=\"0\">No</td>\r\n"                
            else:
                str_page += "<td><input type=\"radio\" name=\"email\" value=\"1\">Yes</td>\r\n"
                str_page += "<td><input type=\"radio\" name=\"email\" value=\"0\" checked>No</td>\r\n"                
            str_page += "<tr>\r\n"
            
            str_page += "</table>\r\n"
            str_page += "<input TYPE=\"hidden\" value=\"%s\" name=\"id\"></td>\r\n" % id
            str_page += "<input type=\"submit\"> | <a class=\"textlink\" href=\"direct_link_info?file=%s\"><img src=\"./img/back_button.gif\" alt=\"Back\"></a>\r\n" % id
            str_page += "</form>\r\n"
            
        else:
            str_page += html.CreateHeader("Validate Page",1)
            
            str_page += "We are sorry but you must be inside a project.<br>\r\n"
            
        str += html.CreatePage("Validate Page", str2, str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def PPASummaryHTMLPage(self):
        '''
    
        '''
        """UI PPA summary page."""
        str = ""
        str_page = ""
        str2 = ""
        html = panteraHTML.PanteraHTML()

        if self.inside_project == YES:

            str_page += html.CreateHeader("Pantera Passive Analysis Summary",1)
    
            # create our PPA summary page
 
            r = self.pan_db.Select_All_Page_Info(self.current_project_id)
            if r == -1:
                self.LogError(self.pan_db.GetErrorMsg())
                return self.ErrorHTMLPage("Error selecting all pages info.")
           
            flag = 0
            for dat in r:
                lst = dat['vuln_data']
                if lst is not None:
                    r1 = self.pan_db.Return_Link_File(self.current_project_id, dat['link_id'])
                    if r1 == -1:
                        self.LogError(self.pan_db.GetErrorMsg())
                        return self.ErrorHTMLPage("Error selecting a page link")    
                    flag = 1
                    (verb, file, site, args) = self.ReturnDataForPage(r1)
                    uri_t = panterautils.TruncateURL(site+args,100)
                    str_page += "<li><a class=\"textlink\" href=\"direct_link_info?file=%s\">%s</a></li><br>\r\n" % (dat['link_id'],panterautils.FilterBadCar(uri_t,2))
                    str_page += self.ReturnResourceIcons(dat['link_id'],r)
                    str_page += "<br><br>\r\n"
                    
                    lst1 = panterautils.Str2List(lst,'|<>|')
                    self.info = []
                    self.low = []
                    self.med = []
                    self.high = []
                    self.safe = []
                    for ls in lst1:
                        co = string.count(ls,'(<>)')
                        if ls != '' and co>0:
                            lev, s_tr = string.split(ls,'(<>)')
                            if lev == 'info':
                                self.info.append(s_tr)
                            if lev == 'low':
                                self.low.append(s_tr)
                            if lev == 'med':
                                self.med.append(s_tr)
                            if lev == 'high':
                                self.high.append(s_tr)
                            if lev == 'safe':
                                self.safe.append(s_tr)
                                                
                    for i in self.info:        
                        str_page += "<img src=\"./img/info.png\" alt=\"Info Level\">  %s<br>\r\n" % panterautils.FilterBadCar(i,2)
                    for i in self.low:        
                        str_page += "<img src=\"./img/low.png\" alt=\"Low Level\">  %s<br>\r\n" % panterautils.FilterBadCar(i,2)
                    for i in self.med:         
                        str_page += "<img src=\"./img/med.png\" alt=\"Medium Level\">  %s<br>\r\n" % panterautils.FilterBadCar(i,2)
                    for i in self.high:        
                        str_page += "<img src=\"./img/high.png\" alt=\"High Level\">  %s<br>\r\n" % panterautils.FilterBadCar(i,2)
                    for i in self.safe:        
                        str_page += "<img src=\"./img/safe.png\" alt=\"Safe Level\">  %s<br>\r\n" % panterautils.FilterBadCar(i,2)
                                        
                    str_page += "<hr>\r\n"        
                    
            # no data
            if flag == 0:
                str_page += "<caption><b>No PPA data!</b></caption>\r\n"          
                
        else:
            str_page += html.CreateHeader("Pantera Passive Analysis Summary",1)
            
            str_page += "We are sorry but you must be inside a project.<br>\r\n"
            
        str += html.CreatePage("Pantera Passive Analysis Summary", str2, str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def SnitchHTMLPage(self, args=""):
        '''
    
        '''
        """UI The Snitch page."""
        str = ""
        str_page = ""
        str2 = ""
        html = panteraHTML.PanteraHTML()

        if self.inside_project == YES:

            str_page += html.CreateHeader("The Snitch",1)
    
            # create our snitch page
            
            str_page += "<form action=\"snitch_form\" method=\"POST\">\r\n"
            str_page += "<table>\r\n"
            str_page += "<tr>\r\n"
            str_page += "<td><input type=\"radio\" name=\"value\" value=\"urls\" checked>Links</td>\r\n"
            str_page += "<td><input type=\"radio\" name=\"value\" value=\"comment\">Comment</td>\r\n"
            str_page += "<td><input type=\"radio\" name=\"value\" value=\"script\">Script</td>\r\n"
            str_page += "</tr>\r\n"
            str_page += "</table>\r\n"
            str_page += "<br><input type=\"submit\">"
            str_page += "</form>\r\n"

            if args != '':
                str_page += "<hr>\r\n"

                # Show links
                if args.has_key('value') and args['value'] == 'urls':
                    urls = sets.Set([])
                    u = self.pan_db.Return_All_Link_Snitch(self.current_project_id)
                    if u == -1:
                        self.LogError(self.pan_db.GetErrorMsg())
                        return self.ErrorHTMLPage("Snitch Error: getting snitch info")
                    adder = urls.add
                    for i in u:
                        x = cPickle.loads(i)                    
                        ur = x.ReturnURLS()
                        for z in ur:
                            adder(z)

                    str_page += "<h3><u>Links</u></h3>\r\n"
                    if len(urls)>0:
                        for c in urls:
                            str_page += "%s<br>\r\n" % panterautils.printFormEntry("Link", "",panterautils.FilterBadCar(c,2),1,100)
                    else:
                        str_page += "<caption><b>No Data</b></caption>\r\n"          
                        
                # Show comment
                elif args.has_key('value') and args['value'] == 'comment':
                    comment = sets.Set([])
                    
                    u = self.pan_db.Return_All_Link_Snitch(self.current_project_id)
                    if u == -1:
                        self.LogError(self.pan_db.GetErrorMsg())
                        return self.ErrorHTMLPage("Snitch Error: getting snitch info")
                    adder = comment.add
                    for i in u:
                        x = cPickle.loads(i)                    
                        ur = x.ReturnComment()
                        id = x.ReturnID()
                        for z in ur:
                            adder((id,z))
    
                    str_page += "<h3><u>Comment</u></h3>\r\n"
                    if len(comment)>0:
                        for c in comment:
                            id,c = c
                            if c != '':
                                r = self.pan_db.Select_All_Page_Info(self.current_project_id)
                                if r == -1:
                                    self.LogError(self.pan_db.GetErrorMsg())
                                    return self.ErrorHTMLPage("Error selecting all pages info.")
                                r1 = self.pan_db.Return_Link_File(self.current_project_id, id)
                                if r1 == -1:
                                    self.LogError(self.pan_db.GetErrorMsg())
                                    return self.ErrorHTMLPage("Error selecting a page link")    
                                (verb, file, site, args) = self.ReturnDataForPage(r1)
                                uri_t = panterautils.TruncateURL(site+args,100)
                                str_page += "<li><a class=\"textlink\" href=\"direct_link_info?file=%s\">%s</a></li><br>\r\n" % (id,panterautils.FilterBadCar(uri_t,2))
                                str_page += self.ReturnResourceIcons(id,r)
                                #str_page += "<br><br>\r\n"
                                #l = panterautils.SearchSensibleWords(c)
                                #if len(l)>0 and l != -1:
                                #    for i in l:
                                #        str_page += "<img src=\"./img/low.png\" alt=\"Low Level\" >%s<br>\r\n" % panterautils.FilterBadCar(i,2)
                                str_page += "<br><br>\r\n"                                        
                                str_page += "<textarea name=\"comment_data\" cols=90 rows=15>"
                                str_page += unicode(c, 'latin-1')
                                str_page += "</textarea>\r\n"
                                str_page += "<hr>\r\n"
                    else:
                        str_page += "<caption><b>No Data</b></caption>\r\n"          
                        
                # Show script
                elif args.has_key('value') and args['value'] == 'script':
                    script = sets.Set([])
                    
                    u = self.pan_db.Return_All_Link_Snitch(self.current_project_id)
                    if u == -1:
                        self.LogError(self.pan_db.GetErrorMsg())
                        return self.ErrorHTMLPage("Snitch Error: getting snitch info")
                    adder = script.add
                    for i in u:
                        x = cPickle.loads(i)                    
                        ur = x.ReturnScript()
                        for z in ur:
                            z,_ = z
                            id = x.ReturnID()
                            adder((id,z))
    
                    str_page += "<h3><u>Script</u></h3>\r\n"
                    if len(script)>0:
                        for c in script:
                            id,c = c
                            if c != '':
                                r = self.pan_db.Select_All_Page_Info(self.current_project_id)
                                if r == -1:
                                    self.LogError(self.pan_db.GetErrorMsg())
                                    return self.ErrorHTMLPage("Error selecting all pages info.")
                                r1 = self.pan_db.Return_Link_File(self.current_project_id, id)
                                if r1 == -1:
                                    self.LogError(self.pan_db.GetErrorMsg())
                                    return self.ErrorHTMLPage("Error selecting a page link")    
                                (verb, file, site, args) = self.ReturnDataForPage(r1)
                                uri_t = panterautils.TruncateURL(site+args,100)
                                str_page += "<li><a class=\"textlink\" href=\"direct_link_info?file=%s\">%s</a></li><br>\r\n" % (id,panterautils.FilterBadCar(uri_t,2))
                                str_page += self.ReturnResourceIcons(id,r)
                                str_page += "<br>\r\n"
                                str_page += "<textarea name=\"comment_data\" cols=90 rows=15>"
                                str_page += c
                                str_page += "</textarea>\r\n"
                                str_page += "<hr>\r\n"
                    else:
                        str_page += "<caption><b>No Data</b></caption>\r\n"          
                    
                else:
                    str_page += "<CAPTION><b>Unknow Option: Maybe some error with data? Try again please.</b></CAPTION>\r\n"
                    
        else:
            str_page += html.CreateHeader("The Snitch",1)
            
            str_page += "We are sorry but you must be inside a project.<br>\r\n"
            
        str += html.CreatePage("The Snitch", str2, str_page)                
        return str

    #############################################################################################
    # FUNC     : 
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : 
    def TargetHTMLPage(self, args=''):
        '''
    
        '''
        """ TargetHTMLPage Page"""
        str = ""
        str_page = ""
        html = panteraHTML.PanteraHTML()

        str += html.CreateHeader("Domain Management",1)                

        # create our target management home page

        if args != '':
                        
            # New domain entry
            #
            if args.has_key('do_action') and urllib.unquote_plus(args['do_action']) == 'Add Domain':
                
                str_page += "<h3>Insert New Domain Rule</h3>\r\n"                            
                
                # blank entry
                str_page += "<form action=\"target_management_form\" method=\"POST\">\r\n"
                str_page += "<table cellpadding=\"10\" cellspacing=\"0\">\r\n"
                str_page += "<tr>\r\n"
                str_page += "%s\r\n" % panterautils.printFormEntry("Domain:", "domain",'',1,25)
                l = ['yes','no']
                str_page += "%s" % panterautils.printFormSelectMenu('Action:','action',l,1)
                str_page += "%s" % panterautils.printFormSelectMenu('Save:','save',l,1)
                str_page+= "%s\r\n" % panterautils.printHiddenEntry("do_action","insert_new_domain",1) 
                str_page += "<td><input type=\"submit\"></td>\r\n"            
                str_page += "</tr>\r\n"
                str_page += "</table>\r\n"
                str_page += "</form>\r\n"
                
                #Display domains that Pantera has already seen
                if self.inside_project == YES:
                    l = self.pan_db.Select_Project_Info(self.current_project_id,"'target'")
                    if l == -1:
                        self.LogError(self.pan_db.GetErrorMsg())
                    if len(l)>0:
                        str_page += "<h3>Know Domains</h3>\r\n"                            
                        z = []
                        add = z.append
                        for x in l:
                            flag = 0
                            for c in self.TargetList:
                                if c['domain'] in 'any': continue
                                if x['value'] in c['domain']:
                                    flag = 1
                            if flag == 0:
                                add(x['value'])
                            
                        str_page += "<form action=\"target_management_form\" method=\"POST\">\r\n"
                        str_page += "<table cellpadding=\"10\" cellspacing=\"0\">\r\n"
                        str_page += "%s " % panterautils.printFormSelectMenu("Domain",'domain',z,1)
                        l1 = ['yes','no']
                        str_page += "%s" % panterautils.printFormSelectMenu('Action:','action',l1,1)
                        str_page += "%s" % panterautils.printFormSelectMenu('Save:','save',l1,1)
                        str_page+= "%s\r\n" % panterautils.printHiddenEntry("do_action","insert_new_domain",1)
                        str_page += "<tr>\r\n"
                        str_page += "<td><input type=\"submit\"></td>\r\n"            
                        str_page += "</tr>\r\n"
                        str_page += "</table>\r\n"
                        str_page += "</form>"         
                    
                else:
                    uni_do = sets.Set([])
                    
                    for i in self.requestCache:
                        ch = self.pf.displayClientHeader(i)
                        flag = 0
                        for l in self.TargetList:
                            if l['domain'] == 'any': continue
                            if ch.connectHost == l['domain']:
                                flag = 1
                        if flag == 0:
                            uni_do.add(ch.connectHost) 
                    
                    if len(uni_do)>0:
                        str_page += "<h3>Know Domains</h3>\r\n" 
                        str_page += "<form action=\"target_management_form\" method=\"POST\">\r\n"
                        str_page += "<table cellpadding=\"10\" cellspacing=\"0\">\r\n"
                        str_page += "%s " % panterautils.printFormSelectMenu("Domain",'domain',uni_do,1)
                        l1 = ['yes','no']
                        str_page += "%s" % panterautils.printFormSelectMenu('Action:','action',l1,1)
                        str_page += "%s" % panterautils.printFormSelectMenu('Save:','save',l1,1)
                        str_page+= "%s\r\n" % panterautils.printHiddenEntry("do_action","insert_new_domain",1) 
                        str_page += "</tr>\r\n"
                        str_page += "<tr>\r\n"
                        str_page += "<td><input type=\"submit\"></td>\r\n"            
                        str_page += "</tr>\r\n"
                        str_page += "</table>\r\n"
                        str_page += "</form>"         
 
                str_page += "<a class=\"textlink\" href=\"target_management_link\"><img src=\"./img/back_button.gif\" border=\"0\" alt=\"Back\"></a>"   

            # Insert new domain (single)
            #
            elif args.has_key('do_action') and args['do_action'] == 'insert_new_domain' and \
            args.has_key('action') and args.has_key('save') and args.has_key('domain'):
            
                try:
                    dom = args['domain']
                    ac = args['action']
                    sav = args['save']
                except:
                    return self.ErrorHTMLPage("Error with domain rule")
                
                d = {}
                d['domain'] = dom
                flag = 0
                for l in self.TargetList:
                    if d['domain'] == l['domain']:
                        flag = 1
                if flag == 0:
                    d['action'] = ac
                    d['save']  = sav
                    self.TargetList.append(d)
            
                if flag == 0:
                    str_page += "<caption><b>Domain Inserted</b></caption><br>\r\n"
                else:
                    str_page += "<caption><b>Error: Domain was already know</b></caption><br>\r\n"
                str_page += "<a class=\"textlink\" href=\"target_management_link\"><img src=\"./img/ok_button.gif\" border=\"0\" alt=\"OK\"></a>"   
            
            # Update domain
            #
            elif args.has_key('do_action') and args['do_action'] == 'update_domain':
                
                num = -1
                if args.has_key('num'):
                    num = args['num']
                
                try:
                    _ = self.TargetList.pop(int(num))
                except:
                    return self.ErrorHTMLPage("Invalid Value")
                
                try:
                    dom = args['domain']
                    ac = args['action']
                    sav = args['save']
                except:
                    return self.ErrorHTMLPage("Error with domain rule")
                
                d = {}
                d['domain'] = dom
                flag = 0
                for l in self.TargetList:
                    if d['domain'] == l['domain']:
                        flag = 1
                if flag == 0:
                    d['action'] = ac
                    d['save']  = sav
                    self.TargetList.append(d)
            
                if flag == 0:
                    str_page += "<caption><b>Domain Inserted</b></caption><br>\r\n"
                else:
                    str_page += "<caption><b>Error: Domain was already know</b></caption><br>\r\n"
                str_page += "<a class=\"textlink\" href=\"target_management_link\"><img src=\"./img/ok_button.gif\" border=\"0\" alt=\"OK\"></a>"   
            
            # Delete domain
            #
            elif args.has_key('delete'):
            
                try:
                    _ = self.TargetList.pop(int(args['delete']))
                except:
                    return self.ErrorHTMLPage("Invalid Value")
                    
                str_page += "<caption>Domain Rule Deleted!</caption><br>\r\n"
                str_page += "<a class=\"textlink\" href=\"target_management_link\"><img src=\"./img/ok_button.gif\" border=\"0\" alt=\"OK\"></a>"
                
            # Update domain form
            #
            elif args.has_key('update'):
                
                try:
                    val = self.TargetList[int(args['update'])]
                except:
                    return self.ErrorHTMLPage("Invalid Value")
                
                str_page += "<h3>Domain Rule Update</h3>\r\n"            
                # blank entry
                str_page += "<form action=\"target_management_form\" method=\"POST\">\r\n"
                str_page += "<table cellpadding=\"10\" cellspacing=\"0\">\r\n"
                str_page += "<tr>\r\n"
                str_page += "%s\r\n" % panterautils.printFormEntry("Domain:", "domain",val['domain'],1,25)
                if val['action'] == 'yes':
                    l = ['yes','no']
                else:
                    l = ['no','yes']
                str_page += "%s" % panterautils.printFormSelectMenu('Action:','action',l,1)
                if val['save'] == 'yes':
                    l = ['yes','no']
                else:
                    l = ['no','yes']
                str_page += "%s" % panterautils.printFormSelectMenu('Save:','save',l,1)
                str_page+= "%s\r\n" % panterautils.printHiddenEntry("do_action","update_domain",1) 
                str_page+= "%s\r\n" % panterautils.printHiddenEntry("num",args['update'],1) 
                str_page += "<td><input type=\"submit\"></td>\r\n"            
                str_page += "</tr>\r\n"
                str_page += "</table>\r\n"
                str_page += "</form>\r\n"
                str_page += "<a class=\"textlink\" href=\"target_management_link\"><img src=\"./img/back_button.gif\" border=\"0\" alt=\"Back\"></a>"   

            # Unknow option
            #     
            else:
                str_page += "<caption>Unknow Option: Maybe some error with data? Try again please.</caption><br>\r\n"                        
        
        # No args 
        #        
        else:

            str_page += "<form action=\"target_management_form\" method=\"POST\">\r\n"
            str_page += "<table>\r\n"
            str_page += "<tr><td><input type=\"submit\" name=\"do_action\" value=\"Add Domain\">"
            str_page += "</table>\r\n"
            str_page += "</form>"         

            if len(self.TargetList)>0:
                str_page += "<h3>Current Domains</h3>\r\n"            
                str_page += "<table>\r\n"
                i = 0
                for x in self.TargetList:
                    str_page += "<tr>\r\n"
                    str_page += "%s" % panterautils.printFormEntry("Domain", "domain",panterautils.FilterBadCar(x['domain'],2),1,25)
                    if x['action'] == 'yes':
                        str_page += "<td><b>Action:</b></td><td><img src=\"./img/safe.png\" alt=\"Allow\" ></td>\r\n"
                    else:
                        str_page += "<td><b>Action:</b></td><td><img src=\"./img/high.png\" alt=\"Forbiden\" ></td>\r\n"
                    if x['save'] == 'yes':
                        str_page += "<td><b>Save:</b></td><td><img src=\"./img/safe.png\" alt=\"Allow\" ></td>\r\n"
                    else:
                        str_page += "<td><b>Save:</b></td><td><img src=\"./img/high.png\" alt=\"Forbiden\" ></td>\r\n"
                    str_page += "<td> [ <a class=\"textlink\" href=\"target_management_link?delete=%s\">Delete</a> ]</td>\r\n" % i
                    str_page += "<td> [ <a class=\"textlink\" href=\"target_management_link?update=%s\">Update</a> ]</td>\r\n" % i
                    str_page += "</tr>\r\n"
                    i += 1
                str_page += "</table>\r\n"
            else:
                str_page += "<CAPTION><b>No domain rules defined.</b></CAPTION>\r\n"
        
        str += html.CreatePage("Domain Management", "", str_page)                
        return str
    # EOF: def TargetHTMLPage

# RL+I EOF