package org.jboss.deployment;

import javax.management.Notification;

/**
 * This notification is sent by the J2eeDeployer MBean to NotificationListener(s)
 * who have registered with deployer.  It should be noted that a deployment
 * notification can reference a .war or .ear archive which contains other
 * archives.  Look at the Deployment object and its Vector properties to
 * determine the nested archives.
 *
 * @see org.jboss.deployment.J2eeDeployer
 * @see org.jboss.deployment.Deployment
 * @author Frederick N. Brier
 */
public class DeploymentNotification extends Notification
{
   /** Notification Type == class name
    */
   public final static String TYPE_NOTIFICATION = DeploymentNotification.class.getName();

   /**
    * Phase constant indicating that the archive has been unpacked, but the
    * application has not been started.
    */
   public final static int PREDEPLOY = 0;

   /**
    * Phase constant indicating that the archive has been unpacked, and the
    * application started.
    */
   public final static int POSTDEPLOY = 1;

   /**
    * Phase constant indicating that the application is about to be stopped.
    */
   public final static int PREUNDEPLOY = 2;
   
   /**
    * Phase constant indicating that the application has been stopped
    and the
    * temporary unzipped files are about to be cleaned up.
    */
   public final static int POSTUNDEPLOY = 3;
   
   /**
    * Array of strings used to generate the notification message text
    based on
    * the specified phase.
    */
   protected final static String[] phaseMessage = {
      "Unpacked but have not started ",
      "Unpacked and started ",
      "About to stop ",
      "Have stopped and about to cleanup "
   };
   
   private int phase;
   private Deployment deployment;
      
      /**
       * This constructor creates a deployment notification.
       * @param source is the J2eeDeployer
       * @param sequenceNumber is an incrementing sequence number for tracking
       * these event notifications.
       * @param deployment is the deployment object describing the archive being deployed.
       * @param phase is an integer constant that describes whether the archive
       * is being deployed or undeployed and whether the application has started
       * yet.
       * @see org.jboss.deployment.Deployment
       */
      public DeploymentNotification( Object source, long sequenceNumber,
         Deployment deployment, int phase )
      {
         super( TYPE_NOTIFICATION, source, sequenceNumber, System.currentTimeMillis(),
            phaseMessage[ phase ] + deployment.getName() );
         this.phase = phase;
         this.deployment = deployment;
      }

      /**
       * @return the phase constant describing the current state of the deployment.
       */
      public int getPhase()
      {
         return phase;
      }

      /**
       * @param phase is the phase constant describing the current state of
         the deployment.
       */
      public void setPhase( int phase )
      {
         this.phase = phase;
      }
      
      
      /**
       * @return the deployment object associated with this event.
       * @see org.jboss.deployment.Deployment
       */
      public Deployment getDeployment()
      {
         return deployment;
      }
      
      
      /**
       * @param deployment is the deployment associated with this event.
       * @see org.jboss.deployment.Deployment
       */
      public void setDeployment( Deployment deployment )
      {
         this.deployment = deployment;
      }
      
}
