/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#include <QtCore>
#include <QtNetwork>
#include <QtDebug>

#include "serverthread.h"
#include "qgs.h"
#include "math.h"
#define DIR_SEP "/"
#include "mainwindow.h"
#include "qgle_statics.h"

GLEServerThread::GLEServerThread(QObject *parent, quint16 port)
	: QThread(parent)
{
	// Just save the requested port
	this->port = port;

	listening = false;
	updateDPI = false;
}

GLEServerThread::~GLEServerThread()
{
	// Use a locker to get access to shared variables,
	// set abort=true and wait for the server to die
	mutex.lock();
	abort = true;
	mutex.unlock();

	wait();

}

void GLEServerThread::setDPI(double new_dpi)
{
	if (dpi != new_dpi)
	{
		dpi = new_dpi;
		emit dpiChanged(dpi);
	}
}

void GLEServerThread::setAutoScale(bool ascale)
{
	updateDPI = ascale;
}

void GLEServerThread::setUserAutoScale(bool ascale)
{
	userAutoScale = ascale;
}

bool GLEServerThread::shouldAutoScale()
{
	return updateDPI;
}

void GLEServerThread::setDefaultResolution(double new_dpi)
{
	defaultDPI = new_dpi;
}

double GLEServerThread::defaultResolution()
{
	return(defaultDPI);
}

void GLEServerThread::initialise()
{
	// Make sure abort is safe
	abort = false;

	// Start the processing
	start(LowPriority);
}

bool GLEServerThread::isListening()
{
	return(listening);
}

void GLEServerThread::waitForServer() 
{
	serverReadyMutex.lock();
	if (isListening()) {
		serverReadyMutex.unlock();
		return;
	}
	serverReady.wait(&serverReadyMutex);
	serverReadyMutex.unlock();
}

void GLEServerThread::run()
{
	// This is the bit that runs in a different thread
	
	// Initialise variables
	QTcpSocket *clientConnection;
	QString gleCommand;
	char buffer[2000];
	qint64 bytesread;
	QStringList parts;
	QString dirname, filename, fullname, part;
	QString glefname;

	QStringList gsargs;

	QImage gs_image;

	// Set up some handy regexps
	QRegExp rxdir(".*dir: *\"([^\"]*)\".*");
	QRegExp rxfile(".*epsfile: *\"([^\"]*)\".*");
	QRegExp glefile(".*glefile: *\"([^\"]*)\".*");
	QRegExp rxdpi(".*dpi: *\"([^\"]*)\".*");
	// This one checks whether the call was made by qgle
	QRegExp rxqgle(".*qgle: *\"true\".*");

	// Is the server connection from QGLE or from GLE?
	bool calledByQgle;

	// Create a new server and listen on the requested port
	server = new QTcpServer();
	if (!server->listen(QHostAddress::LocalHost, port))
	{
		emit serverMessage(tr("Unable to start the server"));
		serverReady.wakeAll();
		return;
	}
	
	serverReadyMutex.lock();
	listening = true;
	serverReady.wakeAll();
	serverReadyMutex.unlock();
	
	// Notify the user that it's running
	emit serverMessage(tr("Server Listening"));

	// Loop forever
	forever
	{
		// Check whether we should quit
		mutex.lock();
		if (abort) {
			mutex.unlock();
			server->close();
			delete server;
			listening = false;
			return;
		}
		mutex.unlock();


		// Wait for a new connection for 10ms
		if (server->waitForNewConnection(10))
		{
			// Get the connection
			clientConnection = server->nextPendingConnection();

			gleCommand = "";
			calledByQgle = false;

			// While we are connected
			while(clientConnection->state() == QAbstractSocket::ConnectedState)
			{
				// Check whether we should quit
				mutex.lock();
				if (abort)
				{
					listening = false;
					delete server;
					return;
				}
				mutex.unlock();

				// See whether there's anything to read
				if(clientConnection->waitForReadyRead(10))
				{
					// There is, so find out how much
					bytesread = clientConnection->bytesAvailable();
					// and read it
					clientConnection->read(buffer, bytesread);
					buffer[bytesread] = 0;
					// append it to our gleCommand buffer
					gleCommand += buffer;
				}
			}

			// Break it up into separate lines
			parts = gleCommand.split(QRegExp("[\\r\\n]+"));

			// Find the bits we're interested in
			bool hasDPISpecification = false;
			foreach(part, parts)
			{
				qDebug() << "Received: " << part;
				if(rxdir.exactMatch(part))
				{
					dirname = rxdir.capturedTexts()[1];
				}
				if(rxfile.exactMatch(part))
				{
					filename = rxfile.capturedTexts()[1];
				}
				if(rxdpi.exactMatch(part))
				{
					hasDPISpecification = true;
					setDPI(rxdpi.capturedTexts()[1].toDouble());
				}
				if(glefile.exactMatch(part))
				{
					glefname = glefile.capturedTexts()[1];
				}
				if(rxqgle.exactMatch(part))
				{
					calledByQgle = true;
				}
			}

			// derive the full name and notify the user
			fullname = dirname + QDir::separator() + filename;
			mutex.lock();
			
			// if new file, then existing drawn objects should be removed
			// note: this happens if you do "gle -p f1.gle" and then "gle -p f2.gle"
			QString newGleFile = dirname + QDir::separator() + glefname;
			if (((GLEMainWindow*)parent())->getCurrentGleFile() != newGleFile) {
				qDebug() << "New GLE file opened";
				emit newGleFileSet();
				
				// Autoscale new files to size of drawing area
				// (Unless the dpi is specified, e.g. with "gle -dpi x".)
				// (and unless user has specified not to do so)
				if (!hasDPISpecification) 
				{
					setAutoScale(userAutoScale);
					setDPI(defaultResolution());
					qDebug() << "Autoscale: " << userAutoScale <<
						"; Res: " << defaultResolution();
					QGLE::flushIO();
				}
			}

			gleFile.setEpsFile(fullname);
			gleFile.setGleFile(newGleFile);

			mutex.unlock();

			emit serverMessage(((QString) "Open: ") + fullname);

			render(CalledBySocket);

			// Delete the connection and loop round to wait for a new one
			delete(clientConnection);
		}
	}
}

void GLEServerThread::render(int caller)
{
	int b1,b2,b3,b4;
	double width;
	double height;
	
	// Bounding box can have negative numbers
	QRegExp rxbb("^%%BoundingBox: +([\\-\\d]+) +([\\-\\d]+) +([\\-\\d]+) +([\\-\\d]+).*$");

	b1 = b2 = b3 = b4 = 0;

	GSInterpreterLib::Position position;
	GSInterpreterLib *interpreter;


	QFile file(gleFile.epsFile());
	// Check that the file can be opened and scan through for
	// BBOX and file size
	if (file.open( QIODevice::ReadOnly ))
	{
		// Get the file size
		position.first = 0;
		position.second = file.size();

		// Scan through file for BoundingBox and set width and height
		QTextStream in(&file);
		while (!in.atEnd())
		{
			QString line = in.readLine();
			if (rxbb.exactMatch(line))
			{
				b1 = rxbb.capturedTexts()[1].toInt();
				b2 = rxbb.capturedTexts()[2].toInt();
				b3 = rxbb.capturedTexts()[3].toInt();
				b4 = rxbb.capturedTexts()[4].toInt();
				break;
			}
		}

		// Derive the width and height and close the file
		width = b3-b1+1;
		height = b4-b2+1;
		file.close();
		
		// Auto scale new files to size of drawing area
		if (shouldAutoScale()) {
			setAutoScale(false);
			QSize size = ((GLEMainWindow*)parent())->getScrollAreaSize();
			setDPI(QGLE::computeAutoScaleDPIFromPts(size, width, height));
		}

		// Open the file for reading by GS
		FILE *fh = fopen(gleFile.epsFile().toLatin1().constData(), "r");
		
		// Create the interpreter object and set a few parameters
		interpreter = new GSInterpreterLib();
		interpreter->setOrigin(b1, b2);
		interpreter->setProgressive(false);
		interpreter->setMedia("a4");
		interpreter->setMagnify(1);
		interpreter->setOrientation(0);
		interpreter->setDPI(dpi);
		int img_wd = (int)floor((double)dpi/72.0*width+1);
		int img_hi = (int)floor((double)dpi/72.0*height+1);
		interpreter->setSize(img_wd, img_hi);
		interpreter->setAABits(4,4);

		// Check that QImages can be passed through the signal-slot
		// mechanism and connect the interpreter's "Finished" signal
		// to our "imageReady" signal (which is connected to updateDisplay()
		// by mainwindow.cpp
		qRegisterMetaType<GLEFileInfo>("GLEFileInfo");
		qRegisterMetaType<QImage>("QImage");
		if (caller == CalledByQGLE)
			connect(interpreter, SIGNAL(Finished(QImage)),
					this, SIGNAL(readyToGo(QImage)));
		else
			connect(interpreter, SIGNAL(Finished(QImage)),
					this, SLOT(prepareFileInfo(QImage)));
		// Run the interpreter
		interpreter->run(fh, position);
		
		if (interpreter->hasError()) {
			const GSError& e = interpreter->getError();
			QString error = QString("Ghostscript error: %1 %2")
			                        .arg(e.getCode())
			                        .arg(e.getName());
			qDebug() << "Error: " << error;
			emit serverMessage(error);
		}

		// Clean up
		fclose(fh);
		delete interpreter;
	}
}

void GLEServerThread::EPSToImage(QString file, double new_dpi, bool autoScale)
{
	gleFile.clear();
	gleFile.setEpsFile(file);
	dpi = new_dpi;
	setAutoScale(autoScale);
	qDebug() << "Rendering " << file;
	render(CalledByQGLE);
}

void GLEServerThread::prepareFileInfo(QImage image)
{
	mutex.lock();
	gleFile.setTemp(false);
	gleFile.setImage(image,dpi);
	gleFile.setImageWithExtras(image);
	gleFile.updateTimeStamp();
	emit imageReady(gleFile);
	mutex.unlock();


}
