/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#ifndef __GLEMAINWINDOW_H
#define __GLEMAINWINDOW_H

#include <QtGui>
#include "gledrawing.h"
//#include "toolbar.h"
#include "serverthread.h"
#include "qgle_definitions.h"
#include "fileinfo.h"
#include "about.h"

class GLESettings;

//! Class implementing Main QGLE Window
/*!
 * This class is responsible for controlling the graphical user
 * interface, setting up signals & slots, providing menu bars and so
 * on.  Like most of the classes in this project, it is a descendant of
 * QObject, so has access to internationalisation functions etc.
 */
class GLEMainWindow : public QMainWindow
{
	// The QT moc macro
	Q_OBJECT
	
public:
	//! Constructor
	/*! 
	 * Responsible for generation of the Window and initialisation
	 * of children, signals & slots.
	 */
	GLEMainWindow(int argc, char *argv[]);

	//! Return the GLE version
	inline const QString& getGLEVersion() { return GLEVersionNumber; }
	//! Return the GS version
	inline const QString& getGsLibVersion() { return GsLibVersionNumber; }

protected:
	//! Called when QGLE is closing
	void closeEvent(QCloseEvent *event);
	//! Called when a MIME object is dragged onto window
	void dragEnterEvent(QDragEnterEvent *event);
	//! Called when a MIME object is dropped on the window
	void dropEvent(QDropEvent *event);
	//! Called when the user presses a key
	void keyPressEvent(QKeyEvent *event);
	//! Called when the user releases a key
	void keyReleaseEvent(QKeyEvent *event);

private slots:
	//! Display an About Box
	void about();
	//! Display a server status message
	void updateServerStatus(QString message);
	//! Start or stop the server
	void serverToggle(bool switchOn);
	//! Start the server
	void startServer();
	//! Stop the server
	void stopServer();
	//! Display the settings dialogue box
	void openSettingsDialogue();
	//! Open download dialog box
	void openDownloadDialogue();	
	//! Open an existing file
	void openFile(QString fileName = QString());
	//! Create a new file
	void newFile();
	//! Save the file
	void save();
	//! Save the file under a new name
	void saveAs();
	//! Save the file under the given name
	void save(QString fileName);
	//! Switch to preview mode
	void previewModeToggle(bool state);
	//! Switch to edit mode
	void editModeToggle(bool state);

	//! Switch to pointer tool
	void pointerToolToggle(bool state);
	//! Switch to line tool
	void lineToolToggle(bool state);
	//! Switch to circle tool
	void circleToolToggle(bool state);
	//! Switch to move tool
	void moveToolToggle(bool state);
	//! Switch to arc tool
	void arcToolToggle(bool state);

	//! Zoom in
	void zoomIn();
	//! Zoom out
	void zoomOut();
	//! Zoom worker function (called by zoomIn() and zoomOut())
	void zoom(double dpi);
	
	//! Open current file in text editor
	void openInTextEditor();

public slots:
	//! Update the coordinate display
	void updateMousePosition(QPointF gle = QPointF(-1.0,-1.0));
	//! Process newly rendered file (from "gle -p")
	void newFileInfo(GLEFileInfo newFile);
	//! Process newly rendered image (from QGLE)
	void renderComplete(QImage image);
	//! Set the base point for relative coordinates
	void setRelativeBasePoint(QPointF gle);
	//! Display a status bar message
	void statusBarMessage(QString msg);
	//! Start or stop the file monitor
	void updateFileMonitor(bool state);
	//! Check whether the file has changed
	void checkForFileUpdates();
	//! Enable or disable the save menu option
	void setSaveEnable(bool state);
	//! Update the status bar display of resolution
	void updateDPIDisplay(double newDPI);

signals:
	//! A new tool has been selected
	void toolSelected(int newTool);
	//! The image has changed, so update the drawing
	void imageChanged(QImage newImage);
	//! There's a new EPS file to be rendered
	void newEPS(QString file, double dpi, bool autoscale);
	//! The autoscale setting has changed
	void autoScaleChanged(bool new_autoscale);
	//! The default resolution has changed
	void defaultResolutionChanged(double new_dpi);
	//! Page size has changed
	void glePageSizeChanged(QSizeF newSize);

private:
	//! Create the actions for use in menus and toolbars.
	void createActions();
	//! Create the menus.
	void createMenus();
	//! Create a simple status bar at the bottom of the screen.
	void createStatusBar();
	//! Create the tool bars
	void createToolBars();
	//! Add the edit mode toolbars
	void createEditModeToolBars();
	//! Destroy the edit mode toolbars
	void destroyEditModeToolBars();
	//! Show/hide and enable/disable edit mode toolbars
	void enableEditModeToolBars(bool enable);

	//! A box to show the about message
	AboutBox *box;

	//! Remember the last mouse position
	QPointF lastMousePosition;

	//! Check whether GLE works
	bool checkGLE();
	//! Get the GLE version
	QString GLEVersion(QString gleCommand);

	//! Render an EPS file from currentFile.epsFile()
	void renderEPS();
	//! Render an EPS file
	void renderEPS(QString epsFile);
	//! Render an GLE file from currentFile.gleFile()
	void renderGLE();
	//! Render an GLE file
	void renderGLE(QString gleFile, bool combinedOutput = false);
	//! Render GLE Code combined with drawing objects
	void renderGLEandObjects();

	//! Initialise GS
	void initLibGS();
	//! Sometimes one has to ask anyway before reload (auto reload not possible)
	bool askAnywayBeforeReload();
		
	//! Menu/toolbar member variables: file menu
	QMenu *fileMenu;
	//! Menu/toolbar member variables: tools menu
	QMenu *toolsMenu;
	//! Menu/toolbar member variables: help menu
	QMenu *helpMenu;

	//! What form should the coordinate display take?
	int coordView;

	//! Should we restore the mode after rendering
	bool saveMode;


	//! Action causing the application to quit
	QAction *quitAct;
	//! Linked to the slot: about()
	QAction *aboutAct;
	//! Toggle the grid on and off
	QAction *gridAct;
	//! Toggle grid snap on and off
	QAction *gridSnapAct;
	//! Toggle the status of the server on and off
	QAction *serverAct;
	//! Toggle orthosnap
	QAction *orthoSnapAct;
	//! Toggle osnap
	QAction *osnapAct;
	//! Toggle polar snap
	QAction *polarSnapAct;
	//! Display the settings dialogue box
	QAction *settingsAct;
	//! Open an existing file
	QAction *openAct;
	//! Save the file under a new name
	QAction *saveAsAct;
	//! Save the file
	QAction *saveAct;
	//! Create a new file
	QAction *newAct;
	//! Go to preview mode
	QAction *previewModeAct;
	//! Go to edit mode
	QAction *editModeAct;
	//! Zoom In Action
	QAction *zoomInAct;
	//! Zoom Out Action
	QAction *zoomOutAct;
	//! Open file in text editor
	QAction *editorAct;
	//! Download action
	QAction *downloadAct;

	//! Pointer tool
	QAction *pointerToolAct;
	QAction *lineToolAct;
	QAction *circleToolAct;
	QAction *arcToolAct;
	QAction *moveToolAct;

	//! Coordinate Display
	QLabel *mousePosition;
	//! Resolution Display
	QLabel *dpiDisplay;
	//! Grid snap button
	QPushButton *gridSnapButton;

	//! Server thread
	GLEServerThread *serverThread;

	//! Current file information
	GLEFileInfo currentFile;

	//! Scroll Area containing the drawing widget
	QScrollArea *scrollArea;

	//! Size of (new) diagram
	QSizeF newDiagramSize;
	

	//! Is the server running?
	bool serverRunning;
	//! Was it running in preview mode?
	bool serverInPreviewMode;

	//! What mode are we currently in?
	int currentMode;

	//! What was the last mode
	int oldMode;

	//! Is the mode toggle by click or by function
	bool toggleByFunction;

	//! Are we mid-render?
	bool renderInProgress;

	//! Are we adding the extras?
	bool imageWithExtras;

	//! Enumeration of the various coordinate display options
	enum
	{
		CoordinateOff,
		CoordinateCart,
		CoordinatePolar,
		CoordinateRelCart,
		CoordinateRelPolar
	};

	//! Enumeration for a hash of temporary files
	enum
	{
		SimpleEPS,
		CombinedEPS,
		CombinedGLE
	};
	//! Hash of temporary files
	QHash<int,QString> tempFiles;

	//! The last point clicked (for relative coordinates)
	QPointF lastPoint;

	//! File toolbar
	QToolBar *fileToolBar;
	//! Control toolbar
	QToolBar *controlToolBar;
	//! Mode toolbar
	QToolBar *modeToolBar;
	//! Tools toolbar
	QToolBar *toolsToolBar;
	//! View toolbar
	QToolBar *viewToolBar;
	
	QString exeLocation;
	QString GLEGhostScriptLocation;
	QString GsLibVersionNumber;
	QString GLEVersionNumber;

	//! Timer for checking for file changes
	QTimer *fileMonitorTimer;
	int fileMonitorCount;

	//! Flag used to force QGLE to wait for rendering to be complete
	bool waitForRender;
	//! Next time we switch to edit mode, we should render the base image
	bool reRenderOnEdit;

public:
	//! The drawing area
	GLEDrawingArea *drawingArea;

	//! Settings for application
	GLESettings *settings;

	//! Remove all drawn objects
	void resetDrawing();	
	//! Return the size of the drawing area
	QSize getScrollAreaSize();
	//! Return the name of the current file
	QString getCurrentGleFile();
	
	//! Automatically scale the drawing to fit the window
	bool autoScale;
};

#endif
