/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#ifndef _GLEDRAWINGOBJECT_H
#define _GLEDRAWINGOBJECT_H

#include <QtGui>
#include <math.h>
#include "qgle_definitions.h"
#include "qgle_statics.h"

//! GLEDrawingObject class: base class for all drawing objects
class GLEDrawingObject : public QObject
{
	Q_OBJECT

	//! Allow daughters access to private parts (so to speak)
	friend class GLELine;
	friend class GLECircle;
	friend class GLEArc;
	friend class SnapLine;
public:
	//! The class constructor
	GLEDrawingObject(double resolution, int imageHeight, QObject *parent = 0);

	//! A function that returns the GLE Code necessary to draw the object
	QStringList GLECode();
	
	//! A function that returns a painter path that can be used to draw the object
	QPainterPath path();

	//! Set the pen for drawing
	void setPen(QPen newPen) { paintPen = newPen; };
	//! Return the pen used for drawing
	QPen pen() { return(paintPen); };

	//! Make this object selected
	void setSelected(bool sel);

	//! Is this object selected?
	bool isSelected();

	//! Virtual function to draw the arc on a provided painter
	virtual void draw(QPainter *p) = 0;
	//! Virtual function to return the shortest distance between a given point and the object
	virtual double distanceToPoint(QPointF p, QPointF *nearestPoint = 0) = 0;
	//! Virtual function to set a point
	virtual void setPoint(int pointChoice, QPointF p) = 0;
	//! Virtual function to determine whether we're inside an object
	virtual bool isInside(QPointF p) = 0;
	//! Virtual function to find points where a line intersects an object
	virtual QList<QPointF> intersections(QPointF qtp1, QPointF qtp2) = 0;
	virtual QList<QPointF> intersections(QPointF qtp1, double angle) = 0;
	virtual QList<QPointF> intersections(double qtm, double qtc, bool vertical) = 0;
	
	//! Function to draw the standard osnap handles
	void drawOSnaps(QPainter *p);

	//! Function to add osnaps relative to a given point
	virtual void addRelativeOSnaps(QPointF p) = 0;
	void clearRelativeOSnaps();
	
	//! Function to determine the distance to the nearest standard osnap handle
	double nearestOSnap(QPointF p, QPointF *osnap);

	//! Get a point in GLE coordinates
	QPointF getGLEPoint(int pointChoice);
	//! Get a point in Qt coordinates
	QPointF getQtPoint(int pointChoice);
	//! Get a value in GLE coordinates
	double getGLEDouble(int pointChoice);
	//! Get a value in Qt coordinates
	double getQtDouble(int pointChoice);
	//! Check whether a point or value has been set
	bool isSet(int pointChoice);

public slots:
	//! Update the resolution of drawing objects
	void setDPI(double newDPI);
	//! Update the image height used for coordinate transforms
	void setPixmapSize(QSize newPixmapSize);

signals:
	//! Notify the world that a point has changed
	void pointChanged();
	//! Notify subclasses that the dpi or image height had changed
	void imageChanged();

private:
	//! Contains the GLE Code
	QStringList *gleCode;
	//! Contains the painter path
	QPainterPath *paintPath;
	//! Contains the pen used to draw the path
	QPen paintPen;

	//! Contains the standard osnap handles
	QList< QPair<QPointF,int> > osnapHandles;

	//! Contains the relative osnap points
	QList< QPair<QPointF,int> > relativeOSnaps;

	//! The list of points
	QHash <int, QPointF> pointHash;

	//! Is the object selected?
	bool selected;

	//! Image resolution
	double dpi;
	//! Image height
	int pixmapHeight;

	//! Line Width
	double lineWidth;

};

#endif
