//
//
//

#include "all.h"
#include "SourceLine.h"
#include "mem_limits.h"
#include "token.h"

#include "tokens/Tokenizer.h"
#include "glearray.h"
#include "polish.h"
#include "pass.h"
#include "var.h"
#include "sub.h"
#include "op_def.h"
#include "core.h"
#include "cutils.h"
#include "gprint.h"
#include "keyword.h"

GLEParser* g_parser;

extern int *(*gpcode)[];   /* gpcode is a pointer to an array of poiter to int */

void get_cap(TOKENS tk,int *ntok,int *curtok,int *pcode,int *plen);
void get_join(TOKENS tk,int *ntok,int *curtok,int *pcode,int *plen);
void g_marker_def(char *s1, char *s2);

extern int this_line;
extern int gle_debug;

#define tok(n) tk[n]

struct sub_st{
	char name[40];
	int typ;
	int np;
	int ptyp[20];
	char *pname[20];
};
struct sub_st *psub;

#define skip_space
#define dbg if ((gle_debug & 8)>0)

char *mark_name[30];
char *mrk_fname[61];
char *mrk_name[61];
char *mark_sub[30];
int mark_subp[30];
int nmark;
int nmrk;
int std_nmrk;

/* pos=   Offset to find the data			*/
/* idx=   For switches, which can only have one value. 	*/
/* The pos is the order the items will be placed in the pcode */
/*---------------------------------------------------------------------------*/
/* Input is gle command (tokenized) , output is pcode */
static int cur_mode = 0;  /* 0 --> normal, 1 or greater ---> external begin...end */

void get_key_info(OPKEY lkey, int* count, int* width) {
	*width = 0; *count = 0;
	for (int i = 0; lkey[i].typ != typ_end; i++) {
		int p = lkey[i].pos;
		if (p > *width) *width = p;
		(*count)++;
	}
}

void GLEParser::get_block_type(int type, string& result) {
	char block_type_str[20];
	sprintf(block_type_str, "%d", type);
	char* block_type = block_type_str;
	switch (type) {
		case 1: /* path */
			block_type = "path"; break;
		case 2: /* box */
			block_type = "box"; break;
		case 3: /* scale */
			block_type = "scale"; break;
		case 4: /* rotate */
			block_type = "rotate"; break;
		case 5: /* translate */
			block_type = "translate"; break;
		case 6: /* if */
			block_type = "if"; break;
		case 7: /* sub */
			block_type = "sub"; break;
		case 8: /* name */
			block_type = "name"; break;
		case 9: /* text */
			block_type = "text"; break;
		case 10: /* graph */
			block_type = "graph"; break;
		case 11: /* xaxis */
			block_type = "xaxis"; break;
		case 12: /* yaxis */
			block_type = "yaxis"; break;
		case 13: /* x2axis */
			block_type = "x2axis"; break;
		case 14: /* y2axis */
			block_type = "y2axis"; break;
		case 15: /* curve */
			block_type = "curve"; break;
		case 16: /* KEY */
			block_type = "key"; break;
		case 17: /* origin */
			block_type = "origin"; break;
		case 18: /* table */
			block_type = "table"; break;
		case 19: /* clip */
			block_type = "clip"; break;
		case 20: /* until */
			block_type = "until"; break;
		case 21: /* shear */
			block_type = "shear"; break;
		case 22: /* config */
			block_type = "config"; break;
		case 23: /* tex preamble */
			block_type = "tex_preamble"; break;
		case 24: /* surface */
			block_type = "surface"; break;
		case 25: /* letz */
			block_type = "letz"; break;
		case 26: /* fitz */
			block_type = "fitz"; break;
		case 27: /* fit */
			block_type = "fit"; break;
		case 28: /* contour */
			block_type = "contour"; break;
		case 29: /* tex */
			block_type = "tex"; break;
		case OP_BEGIN_OBJECT:
			block_type = "object"; break;
	}
	result = block_type;
}

void GLEParser::checkmode() throw(ParserError) {
	/* Check for text mode block */
	if (cur_mode != 0) {
		string block_type;
		get_block_type(cur_mode, block_type);
		g_throw_parser_error("end of file while in block type '", block_type.c_str(), "'");
	}
	cur_mode = 0;
	/* Check for other type of block */
	GLESourceBlock* block = last_block();
	if (block != NULL) {
		stringstream err;
		err << "end of file while in block type '" << block->getName() << "'";
		err << " starting on line " << block->getFirstLine();
		g_throw_parser_error(err.str());
	}
}

GLEParser::GLEParser(GLEPolish* polish) : m_lang(), m_tokens(&m_lang, false) {
	m_polish = polish;
	m_auto_endif = false;
}

GLEParser::~GLEParser() {
}

void GLEParser::initTokenizer() {
	TokenizerLanguage* lang = getTokens()->get_language();
	lang->setLineCommentTokens("!");
	lang->setSpaceTokens(" \t\r\n");
	lang->enableCComment();
	lang->setSingleCharTokens(",;=@()[]{}");
		// ; -> allows more commands on single line
		// = -> for variable assignments
		// @ -> for user defined subroutine calls
		// (,) -> for calls of the form proc(arg1,...,argn)
	TokenizerLanguageMultiLevel* multi = new TokenizerLanguageMultiLevel();
	multi->setOpenClose('(',')');
	multi->setOpenClose('[',']');
	multi->setOpenClose('{','}');
	multi->setEndToken(' ');
	multi->setEndToken(';');
	multi->setEndToken(',');
	lang->setMulti(multi);
}

void GLEParser::setAllowSpace(bool allow) {
	TokenizerLanguageMultiLevel* multi = getTokens()->get_language()->getMulti();
	if (allow) multi->resetEndToken(' ');
	else multi->setEndToken(' ');
}

void GLEParser::checkValidName(const string& name, const char* type, int pos) throw(ParserError) {
	if (name.length() <= 0) {
		throw getTokens()->error(pos, string("zero length ")+type+" name");
	}
	if (name[0] >= '0' && name[0] <= '9') {
		throw getTokens()->error(pos, string(type)+" name should not start with a digit");
	}
	for (int i = 0; i < name.length(); i++) {
		char ch = name[i];
		if (!((ch >= 'A' && ch <= 'Z') || (ch >= 'a' && ch <= 'z') ||
		      (ch >= '0' && ch <= '9') || (ch == '$') || (ch == '_'))) {
			throw getTokens()->error(pos+i, string("invalid character '")+ch+"' in "+type+" name");
		}
	}
}

void GLEParser::polish(GLEPcode& pcode, int *rtype) throw(ParserError) {
	Tokenizer* tokens = getTokens();
	string& expr = tokens->next_multilevel_token();
	int pos = tokens->token_pos_col();
	// cout << "pos = " << pos << endl;
	try {
		// cout << "Polish: '" << expr << "'" << endl;
		m_polish->polish(expr.c_str(), pcode, rtype);
	} catch (ParserError err) {
		err.incColumn(pos-1);
		throw err;
	}
}

void GLEParser::polish_eol(GLEPcode& pcode, int *rtype) throw(ParserError) {
	setAllowSpace(true);
	polish(pcode, rtype);
	setAllowSpace(false);
}

void GLEParser::polish(const char* str, GLEPcode& pcode, int *rtype) throw(ParserError) {
	try {
		m_polish->polish(str, pcode, rtype);
	} catch (ParserError err) {
		err.setParserString(str);
		throw err;
	}
}

void GLEParser::get_xy(GLEPcode& pcode) throw(ParserError) {
	int vtype = 1;
	polish(pcode, &vtype);
	vtype = 1;
	polish(pcode, &vtype);
}

void GLEParser::get_exp(GLEPcode& pcode) throw(ParserError) {
	int vtype = 1;
	polish(pcode, &vtype);
}

void GLEParser::get_exp_eol(GLEPcode& pcode) throw(ParserError) {
	int vtype = 1;
	polish_eol(pcode, &vtype);
}

void GLEParser::get_strexp(GLEPcode& pcode) throw(ParserError) {
	int vtype = 2;
	polish(pcode, &vtype);
}

int GLEParser::get_anyexp(GLEPcode& pcode) throw(ParserError) {
	int vtype = 0;
	polish(pcode, &vtype);
	return vtype;
}

void GLEParser::get_if(GLEPcode& pcode) throw(ParserError) {
	Tokenizer* tokens = getTokens();
	string expr = tokens->next_multilevel_token();
	int pos = tokens->token_pos_col();
	/* Support spaces between in if expression */
	while (true) {
		string& token = tokens->next_multilevel_token();
		if (str_i_equals(token.c_str(), "THEN")) {
			break;
		} else if (token == "") {
			throw error("'then' expected after if condition");
		}
		expr += " ";
		expr += token;
	}
	try {
		int rtype = 1;
		// cout << "Polish: '" << expr << "'" << endl;
		m_polish->polish(expr.c_str(), pcode, &rtype);
	} catch (ParserError err) {
		err.incColumn(pos-1);
		throw err;
	}
}

void GLEParser::parse_if(int srclin, GLEPcode& pcode) throw(ParserError) {
	get_if(pcode);
	GLESourceBlock* block = add_block(GLE_SRCBLK_MAGIC+GLE_OPBEGIN_IF, srclin);
	block->setOffset2(pcode.size());
	pcode.addInt(0);
	pcode.addInt(0);
}

void GLEParser::get_subroutine_call(string* name, GLEPcode& pcode) throw(ParserError) {
	string fct_name;
	if (name != NULL) {
		fct_name = *name;
	} else {
		fct_name = m_tokens.next_token();
		str_to_uppercase(fct_name);
	}
	GLESub* sub = sub_find((char*)fct_name.c_str());
	if (sub == NULL) {
		throw error("function '"+fct_name+"' not defined");
	}
	int np = sub->getNbParam();
	pcode.addInt(PCODE_EXPR);    /* Expression follows */
	int savelen = pcode.size();  /* Used to set acutal length at end */
	pcode.addInt(0);	     /* Length of expression */
	int nb_param = 0;
	// cout << "reading args for: '" << fct_name << "'" << endl;
	while (not_at_end_command()) {
		if (nb_param >= np) {
			char err_str[100];
			sprintf(err_str, "': found >= %d, expected %d", nb_param+1, np);
			throw error(string("too many parameters in call to '")+fct_name+err_str);
		}
		int vtype = sub->getParamType(nb_param);
		polish(pcode, &vtype);
		nb_param++;
	}
	if (nb_param != np) {
		char err_str[100];
		sprintf(err_str, "': found %d, expected %d", nb_param, np);
		throw error(string("incorrect number of parameters in call to '")+fct_name+err_str);
	}
	pcode.addFunction(sub->getIndex()+LOCAL_START_INDEX);
	pcode.setInt(savelen, pcode.size() - savelen - 1);
}

int GLEParser::get_subroutine_def(GLEPcode& pcode) throw(ParserError) {
	string uc_token;
	string& token = m_tokens.next_token();
	str_to_uppercase(token, uc_token);
	GLESub* sub = sub_def((char*)uc_token.c_str());
	var_set_local_map(sub->getLocalVars());
	for (int np = 0; not_at_end_command(); np++) {
		token = m_tokens.next_token();
		str_to_uppercase(token, uc_token);
		sub_param(sub, (char*)uc_token.c_str());
		if (!valid_var((char*)uc_token.c_str())) {
			throw error("invalid subroutine parameter");
		}
	}
	int sdx = sub->getIndex();
	pcode.addInt(sdx); /* put sub number in pcode */
	return sdx;
}

int GLEParser::get_optional(OPKEY lkey, GLEPcode& pcode) throw(ParserError) {
	// find the largest width
	int count, width;
	get_key_info(lkey, &count, &width);
	// the location in the pcode tells what option it is, must remember
	// zero all the optional parameters.
	// puts zero in for each option wheather it is there or not
	int plen = pcode.size();
	for (int i = 0; i < width+1; i++) {
		pcode.addInt(0);
	}
	int ret = -1;
	while (m_tokens.has_more_tokens()) {
		string& token = m_tokens.next_token();
		if (token == ";") {
			m_tokens.pushback_token();
			return ret;
		}
		bool found = false;
		for (int i = 0; i < count && !found; i++) {
			if (str_i_equals(token.c_str(), lkey[i].name)) {
			 	ret = get_one_option(&lkey[i], pcode, plen);
				found = true;
			}
		}
		if (!found) {
			throw create_option_error(lkey, count, token);
		}
	}
	return ret;
}

ParserError GLEParser::create_option_error(OPKEY lkey, int count, string& token) {
	stringstream strm;
	if (count == 1) {
		strm << "found '" << token << "', but expecting '" << lkey[0].name << "'";
	} else {
		strm << "found '" << token << "', but expecting one of:";
		for (int i = 0; i < count; i++) {
			if (i % 5 == 0) {
				strm << endl << "       ";
			} else {
				strm << " ";
			}
			strm << lkey[i].name;
			if (i < count-1) strm << ",";
		}
	}
	return m_tokens.error(strm.str());
}

void GLEParser::duplicate_error(GLEPcode& pcode, int pos) throw(ParserError) {
	if (pcode.getInt(pos) != 0) throw error("duplicate or illegal combination of qualifiers");
}

int GLEParser::get_one_option(op_key* lkey, GLEPcode& pcode, int plen) throw(ParserError) {
// switches 	int 	placed in directly, 1 present, 0 not present
// expressions 	LONG* 	pointed to, 0 if not present.
// color/fill	LONG* 	Pointer to exp 0 if not present.
// marker	LONG*	Pointer to exp 0 if not present.
// lstyle 	LONG*	Pointer to exp 0 if not present.
// font 	int* 	Pointer to string expression.
// justify 	int
	int pos = plen + lkey->pos - 1;
	duplicate_error(pcode, pos);
	switch (lkey->typ) {
	case typ_val:
		pcode.setInt(pos, pcode.size() - pos);
		get_exp(pcode);
	 	break;
	case typ_val2:
		pcode.setInt(pos, pcode.size() - pos);
		get_exp(pcode);
		pos++;
		duplicate_error(pcode, pos);
		pcode.setInt(pos, pcode.size() - pos);
		get_exp(pcode);
		break;
	case typ_val4:
		pcode.setInt(pos, pcode.size() - pos);
		get_exp(pcode);
		get_exp(pcode);
		get_exp(pcode);
		get_exp(pcode);
		break;
	case typ_str:
		pcode.setInt(pos, pcode.size() - pos);
		get_strexp(pcode);
	 	break;
	case typ_switch:
		pcode.setInt(pos, lkey->idx);
		return lkey->idx;
		break;
	case typ_color:
	case typ_fill:
		pcode.setInt(pos, pcode.size() - pos);
		get_fill(pcode);
		break;
	case typ_marker:
		pcode.setInt(pos, pcode.size() - pos);
		get_marker(pcode);
		break;
	case typ_lstyle:
		pcode.setInt(pos, pcode.size() - pos);
		get_exp(pcode);
		break;
	case typ_justify:
		pcode.setInt(pos, get_first(op_justify));
		break;
	case typ_arrow:
		pcode.setInt(pos, get_first(op_arrow));
		break;
	default :
		gprint("*** error non existent type ***");
		break;
	}
	return -1;
}

int GLEParser::get_first(OPKEY lkey) throw(ParserError) {
	int count, width;
	get_key_info(lkey, &count, &width);
	string& token = m_tokens.next_token();
	for (int i = 0; i < count; i++) {
		if (str_i_equals(token.c_str(), lkey[i].name)) {
			return lkey[i].idx;
		}
	}
	throw create_option_error(lkey, count, token);
}

bool GLEParser::try_get_token(const char* token) throw(ParserError) {
	string& my_token = m_tokens.try_next_token();
	if (str_i_equals(token, my_token.c_str())) {
		return true;
	} else if (my_token != "") {
		m_tokens.pushback_token();
	}
	return false;
}

void GLEParser::get_token(const char* token) throw(ParserError) {
	string& my_token = m_tokens.next_token();
	if (!str_i_equals(token, my_token.c_str())) {
		throw error(string("expected '")+token+"', but found '"+my_token+"' instead");
	}
}

// In future color should be representad as array!
void GLEParser::get_fill(GLEPcode& pcode) throw (ParserError) {
	int vtype = 1;
	string& token = m_tokens.next_token();
	if (str_i_str(token.c_str(), "RGB") != NULL) {
		m_tokens.pushback_token();
		get_exp(pcode);
	} else if (token == "(") {
		string expr = string("cvtgray(")+m_tokens.next_token()+")";
		polish(expr.c_str(), pcode, &vtype);
		m_tokens.ensure_next_token(")");
	} else if (is_float(token)) {
		string expr = string("cvtgray(")+token+")";
		polish(expr.c_str(), pcode, &vtype);
	} else	if (strchr(token.c_str(),'$') != NULL) {
		string expr = string("CVTCOLOR(")+token+")";
		polish(expr.c_str(), pcode, &vtype);
	} else {
		m_tokens.pushback_token();
		pcode.addInt(8);
		pcode.addInt(get_first(op_color_typ));
	}
}

void GLEParser::get_color(GLEPcode& pcode) throw (ParserError) {
	int vtype = 1;
	string& token = m_tokens.next_token();
	if (str_i_str(token.c_str(), "RGB") != NULL) {
		m_tokens.pushback_token();
		get_exp(pcode);
	} else if (token == "(") {
		string expr = string("cvtgray(")+m_tokens.next_token()+")";
		polish(expr.c_str(), pcode, &vtype);
		m_tokens.ensure_next_token(")");
	} else if (is_float(token)) {
		string expr = string("cvtgray(")+token+")";
		polish(expr.c_str(), pcode, &vtype);
	} else	if (strchr(token.c_str(),'$') != NULL) {
		string expr = string("CVTCOLOR(")+token+")";
		polish(expr.c_str(), pcode, &vtype);
	} else {
		m_tokens.pushback_token();
		pcode.addInt(8);
		pcode.addInt(get_first(op_color_typ));
	}
}

void GLEParser::get_marker(GLEPcode& pcode) throw (ParserError) {
	int vtype = 1;
	string& token = m_tokens.next_token();
	if (token == "(" || is_float(token)) {
		string expr = string("CVTINT(")+token+")";
		polish(expr.c_str(), pcode, &vtype);
	} else	if (strchr(token.c_str(),'$') != NULL) {
		string expr = string("CVTMARKER(")+token+")";
		polish(expr.c_str(), pcode, &vtype);
	} else {
		pcode.addInt(8);
		pcode.addInt(pass_marker(token));
	}
}

int GLEParser::pass_marker(const string& marker) throw (ParserError) {
	/* if 0, maybe its a user defined marker, ie a subroutine */
	/* Use -ve to signify subroutine instead of normal marker */
	int mark_idx = 0;
	for (int i = 0; i < nmark; i++) {
		if (str_i_equals(mark_name[i], marker.c_str())) {
			mark_idx = -(++i);
			break;
		}
	}
	if (mark_idx == 0)  {
		for (int i = nmrk-1; i >= 0; i--) {
			if (str_i_equals(mrk_name[i], marker.c_str())) {
				mark_idx = ++i;
				break;
			}
		}
	}
	if (mark_idx == 0) throw error("invalid marker name");
	return mark_idx;
}

void GLEParser::define_marker_1(GLEPcode& pcode) throw (ParserError) {
	string name;
	Tokenizer* tokens = getTokens();
	str_to_uppercase(tokens->next_token(), name);
	string font = tokens->next_token();
	int ccc = tokens->next_integer();
	double sz = tokens->next_double();
	double dx = tokens->next_double();
	double dy = tokens->next_double();
	g_defmarker((char*)name.c_str(), (char*)font.c_str(), ccc, dx, dy, sz, true);
}

void GLEParser::define_marker_2(GLEPcode& pcode) throw (ParserError) {
	string name, sub;
	Tokenizer* tokens = getTokens();
	tokens->ensure_next_token_i("marker");
	str_to_uppercase(tokens->next_token(), name);
	str_to_uppercase(tokens->next_token(), sub);
	g_marker_def((char*)name.c_str(), (char*)sub.c_str());
}

void GLEParser::get_font(GLEPcode& pcode) throw (ParserError) {
	int etype = 1;
	if (get_nb_fonts() == 0) font_load();
	string& token = m_tokens.next_token();
	int token_len = token.length();
	char first_char = token_len > 0 ? token[0] : ' ';
	if (first_char == '"' || token.find("$") != string::npos) {
		string parse = "CVTFONT("+token+")";
		polish(parse.c_str(), pcode, &etype);
		return;
	}
	pcode.addInt(8);
	int count = get_nb_fonts();
	for (int i = 1; i <= count; i++) {
		const char* name = get_font_name(i);
		if (str_i_equals(name, token.c_str())) {
			pcode.addInt(i);
			return;
		}
	}
	stringstream strm;
	strm << "invalid font name {" << token << "}, expecting one of:";
	int idx = 0;
	for (int i = 1; i <= count; i++) {
		if (idx % 5 == 0) {
			strm << endl << "       ";
		} else {
			strm << " ";
		}
		if (get_font_name(i) != NULL) {
			strm << get_font_name(i);
			bool has_more = false;
			for (int j = i+1; j <= count; j++) {
				if (get_font_name(j) != NULL) {
					has_more = true;
					break;
				}
			}
			if (has_more) strm << ",";
			idx++;
		}
	}
	throw m_tokens.error(strm.str());
}

void GLEParser::get_papersize(GLEPcode& pcode) throw (ParserError) {
	const string& token = m_tokens.next_token();
	int type = g_papersize_type(token);
	if (type == GLE_PAPER_UNKNOWN) {
		m_tokens.pushback_token();
		pcode.addInt(0);
		get_xy(pcode);
	} else {
		pcode.addInt(1);
		pcode.addInt(type);
	}
}

void GLEParser::get_justify(GLEPcode& pcode) throw (ParserError) {
	pcode.addInt(get_first(op_justify));
}

void GLEParser::get_join(GLEPcode& pcode) throw (ParserError) {
	pcode.addInt(get_first(op_join));
}

void GLEParser::get_cap(GLEPcode& pcode) throw (ParserError) {
	pcode.addInt(get_first(op_cap));
}

void GLEParser::get_var_add(int *var, int *vtype) throw (ParserError) {
	string uc_token;
	string& token = m_tokens.next_token();
	str_to_uppercase(token, uc_token);
	var_findadd((char*)uc_token.c_str(), var, vtype);
}

void GLEParser::get_var(GLEPcode& pcode) throw (ParserError) {
	int var;
	int vtype = 0;
	get_var_add(&var, &vtype);
	pcode.addInt(var);
}

bool GLEParser::not_at_end_command() {
	string& token = m_tokens.try_next_token();
	if (token == "") return false;
	if (token == ";") {
		m_tokens.pushback_token();
		return false;
	}
	m_tokens.pushback_token();
	return true;
}

bool GLEParser::test_not_at_end_command() {
	string& token = m_tokens.try_next_token();
	if (token == "") return false;
	if (token == ";") return false;
	m_tokens.pushback_token();
	return true;
}

void GLEParser::passt(SourceLine &SLine, GLEPcode& pcode) throw(ParserError) {
	resetSpecial();
	pcode.clear();
	static int sdx;
	static int i,f,vtyp,v,in_sub,vidx;
	int srclin = SLine.line_no;
	this_line = srclin;
	int fctkey;
	int nbcmd = 0;
	int position;
	GLESourceBlock* block;
	string first, temp_str;
	if (cur_mode != 0) {
		do_text_mode(SLine, getTokens(), pcode);
		return;
	}
	setAllowSpace(false);
	bool single_cmd = false;
	Tokenizer* tokens = getTokens();
	if (m_auto_endif) {
		// Allow for "IF test THEN code" blocks	without "END IF"
		block = last_block();
		if (block->getType() == GLE_SRCBLK_MAGIC+GLE_OPBEGIN_IF) {
			string& token = tokens->try_next_token();
			if (str_i_equals(token, "ELSE")) m_auto_endif = false;
			if (token != "") tokens->pushback_token();
		} else if (block->getType() != GLE_SRCBLK_ELSE) {
			m_auto_endif = false;
		}
		if (m_auto_endif) {
			m_auto_endif = false;
			do_endif(srclin);
		}
	}
	while (tokens->has_more_tokens()) {
		int vtype = 0;
		int etype = 1;
		bool allow_extra_tokens = false;
		int pos_endoffs = pcode.size();
		pcode.addInt(0); // save space for end offset
		str_to_uppercase(tokens->next_token(), first);
		int pos_first = tokens->token_pos_col();
		find_mkey((char*)first.c_str(), &fctkey);
		// cout << "first = " << first << " idx = " << fctkey << endl;
		if (fctkey == 0) {
			if (first == "@") {
				pcode.addInt(52);
				get_subroutine_call(NULL, pcode);
			} else if (first == "LOCAL") {
				// Declaration of local variable
				if (!has_local_var_map()) {
					throw error("can't define a local variable outside of sub");
				}
				// Get variable name
				str_to_uppercase(tokens->next_token(), temp_str);
				checkValidName(temp_str, "variable", tokens->token_pos_col());
				pcode.addInt(51);
				var_add_local((char*)temp_str.c_str(),&vidx,&vtype);
				pcode.addInt(vidx);
				// Immediately assign a value to it
				if (tokens->is_next_token("=")) {
					polish_eol(pcode, &vtype);
				} else {
					// Or assign the value of zero to it
					if (vtype == 1) pcode.addDoubleExpression(0);
					else pcode.addStringExpression("");
					// And possibly define more local variables in one go
					while (tokens->is_next_token(",")) {
						str_to_uppercase(tokens->next_token(), temp_str);
						checkValidName(temp_str, "variable", tokens->token_pos_col());
						pcode.addInt(51);
						var_add_local((char*)temp_str.c_str(),&vidx,&vtype);
						pcode.addInt(vidx);
						if (vtype == 1) pcode.addDoubleExpression(0);
						else pcode.addStringExpression("");
					}
				}
			} else if (tokens->is_next_token("=")) {
				// Variable assignment
				checkValidName(first, "variable", pos_first);
				pcode.addInt(51);
				var_findadd((char*)first.c_str(),&vidx,&vtype);
				pcode.addInt(vidx);
				polish_eol(pcode, &vtype);
			} else {
				/* call subroutine without @ sign */
				pcode.addInt(52);
				get_subroutine_call(&first, pcode);
			}
		} else {
			pcode.addInt(fctkey);
			switch (fctkey) {
			case 65: /* PSCOMMENT */
				temp_str = m_tokens.read_line();
				str_remove_quote(temp_str);
				pcode.addStringNoID(temp_str);
				break;
			case 66: /* BB_TWEAK */
				break;
			  case 1:  /* ALINE */
				get_xy(pcode);
				get_optional(op_line, pcode);
				break;
			  case 2: /* AMOVE */
				get_xy(pcode);
				break;
			  case 73: /* ASETPOS */
			  case 81: /* RSETPOS */
				get_xy(pcode);
				break;
			  case 3: /* ARC r a1 a2  */
				get_exp(pcode);
				get_xy(pcode);
				get_optional(op_arc, pcode);
				break;
			  case 4: /* ARCTO x1 y1 x2 y2 r */
				get_xy(pcode);
				get_xy(pcode);
				get_exp(pcode);
				break;
			  case 5: /* BEGIN  "PATH"  "BOX"  "SCALE"  "ROTATE"  "TRANSLATE" "SHEAR" "GRAPH" ... */
				single_cmd = true;
				f = get_first(op_begin);
				pcode.addInt(f);
				/*------------------------------------------*/
				/*       Check if begin variable matches    */
				/*------------------------------------------*/
				switch (f) {
				  case 1: /* path */
					get_optional(op_begin_path, pcode);
					break;
				  case 2: /* box */
					get_optional(op_begin_box, pcode);
					break;
				  case 3: /* scale */
					get_xy(pcode);
					get_optional(op_begin_scale, pcode);
					break;
				  case 21: /* shear */
					get_xy(pcode);
					get_optional(op_begin_scale, pcode);
					break;
				  case 4: /* rotate */
					get_exp(pcode);
					get_optional(op_begin_scale, pcode);
					break;
				  case 5: /* translate */
					get_xy(pcode);
					get_optional(op_begin_scale, pcode);
					break;
				  case 19: /* clip */
				  case 17: /* origin */
					break;
				  case 6: /* if */
				  case 7: /* sub */
					throw error("not a valid begin option");
					break;
				  case 8: /* name joe */
					get_strexp(pcode);
					break;
				  case 9: /* text */
					cur_mode = 9;
					get_optional(op_begin_text, pcode);
					break;
				  case 10: /* graph */
					cur_mode = 10;
					break;
				  case 11: /* xaxis */
					cur_mode = 11;
					break;
				  case 12: /* yaxis */
					cur_mode = 12;
					break;
				  case 13: /* x2axis */
					cur_mode = 13;
					break;
				  case 14: /* y2axis */
					cur_mode = 14;
					break;
				  case 15: /* curve */
					break;
				  case 16: /* KEY */
					cur_mode = 16;
					break;
				  case 18: /* table */
					cur_mode = 18;
					break;
				  case 22: /* config */
					cur_mode = 22;
					get_strexp(pcode);
					break;
				  case 23: /* tex preamble */
					cur_mode = 23;
					break;
				  case 24: /* surface */
					cur_mode = 24;
					break;
				  case 25: /* letz */
					cur_mode = 25;
					break;
				  case 26: /* fitz */
					cur_mode = 26;
					break;
				  case 27: /* fit */
					cur_mode = 27;
					break;
				  case 28: /* contour */
					cur_mode = 28;
					break;
				  case 29: /* tex */
					cur_mode = 29;
					get_optional(op_tex, pcode);
					break;
				  case OP_BEGIN_OBJECT:
					get_strexp(pcode); /* name for object */
					break;
				}
				/* here should copy source line across for "begin width 3 */
				if (cur_mode>0)	pcode.addInt(0);
				else add_block(GLE_SRCBLK_MAGIC+f, srclin);
				break;
			  case 6: /* BEZIER x1 y1 x2 y2 x3 y3 */
				get_xy(pcode);
				get_xy(pcode);
				get_xy(pcode);
				break;
			  case 7 :	/* box x y  [left | center | right]  [fill xxx] name*/
				get_xy(pcode);
				get_optional(op_box, pcode);
				break;
			  case 52: /* call subroutine */
				get_subroutine_call(NULL, pcode);
				break;
			  case 8 :	/* circle rad fill */
				get_exp(pcode);
				get_optional(op_circle, pcode);
				break;
			  case 70 :	/* ellipse major minor fill */
				get_xy(pcode);
				get_optional(op_circle, pcode);
				break;
			  case 71 :	/* elliptical_arc major minor a1 a2 fill */
				get_xy(pcode);
				get_xy(pcode);
				get_optional(op_arc, pcode);
				break;
			  case 72 :	/* elliptical_narc major minor a1 a2 fill */
				get_xy(pcode);
				get_xy(pcode);
				get_optional(op_arc, pcode);
				break;
			  case 9 : /* closepath */
				break;
			  case 53: /* comment !  or blank line */
				break;
			  case 10 : /* curve sx sy x y x y x y ... ex ey */
				while (not_at_end_command()) {
					pcode.addInt(111);
					get_xy(pcode);
				}
				pcode.addInt(999);
				break;
			  case 60 : /* defmarker xyz rm 33 1 -.4 -.2 */
				define_marker_1(pcode);
				break;
			  case 11 :  /* define marker jj subname */
				define_marker_2(pcode);
				break;
			  case 12 :
				pcode.addStringNoID(SLine.text);
				break;
			  case 13 :/* ELSE ... */
				if (nbcmd > 0) {
					throw error(pos_first, "command must be the first command on the line");
				}
				block = check_block_type(pos_first, GLE_SRCBLK_ELSE, GLE_SRCBLK_MAGIC+GLE_OPBEGIN_IF, -1);
				if (try_get_token("IF")) {
					add_else_block_update(srclin, pcode, true);
					pcode.setInt(pos_endoffs, pcode.size());
					pos_endoffs = pcode.size();
					pcode.addInt(0);
					pcode.addInt(22);
					parse_if(srclin, pcode);
					allow_extra_tokens = true;
				} else if (tokens->has_more_tokens()) {
					add_else_block_update(srclin, pcode, false);
					m_auto_endif = true;
					allow_extra_tokens = true;
				} else {
					(*gpcode)[block->getFirstLine()][block->getOffset2()] = srclin+1;
					add_else_block(srclin, pcode, false);
				}
				break;
			  case 14 : /* END if, sub, path, box, scale, translate, rotate */
				single_cmd = true;
				i = get_optional(op_begin, pcode);
				if (i == 0) throw error("type of 'end' missing, e.g. end if, end sub");
				if (i == GLE_OPBEGIN_IF) {
			  		check_block_type(pos_first, GLE_SRCBLK_MAGIC+i, GLE_SRCBLK_MAGIC+i, GLE_SRCBLK_ELSE);
					do_endif(srclin+1);
				} else {
			  		block = check_block_type(pos_first, GLE_SRCBLK_MAGIC+i, GLE_SRCBLK_MAGIC+i, -1);
					if (i == 7) {
						do_endsub(srclin);
						sub_set_startend(sdx, block->getFirstLine(), srclin);
						var_clear_local();
						in_sub = false;
					} else if (i == GLE_OPBEGIN_BOX || i == OP_BEGIN_OBJECT) {
						pcode.addInt(block->getFirstLine());
					}
					remove_last_block();
				}
				break;
			  case 16 : /* FILL (fillpath) */
				break;
			  case 15 : /* FCLOSE inchan */
				get_exp(pcode);
				break;
			  case 17: /* fopen "a.a" inchan read|write */
				get_strexp(pcode);
				get_var_add(&v,&vtyp);
				pcode.addInt(v);
				if (try_get_token("WRITE")) {
					pcode.addInt(1);
				} else {
					get_token("READ");
					pcode.addInt(0);
				}
				break;
			  case 61 : /* fread CHAN a$ x   */
			  case 62 : /* freadln */
				while (not_at_end_command()) {
					get_var_add(&v, &vtyp);
					pcode.addInt(49);
					pcode.addInt(v);
					pcode.addInt(vtyp);
				}
				break;
			  case 63 : /* fwrite */
			  case 64 : /* fwriteln */
				while (not_at_end_command()) {
					pcode.addInt(49);
					position = pcode.size();
					pcode.addInt(0);
					pcode.setInt(position, get_anyexp(pcode));
				}
				break;
			  case 75 : /* fgetline */
				get_exp(pcode);
				get_var_add(&v, &vtyp);
				pcode.addInt(v);
				break;
			  case 76 : /* ftokenizer chan commenttoks spacetoks singlechartoks */
				get_exp(pcode);
				get_strexp(pcode);
				get_strexp(pcode);
				get_strexp(pcode);
				break;
			  case 77: /* papersize */
			  	get_papersize(pcode);
			  	break;
			  case 78: /* margins */
				get_xy(pcode);
				get_xy(pcode);
			  	break;
			  case 79: /* orientation */
		  		pcode.addInt(get_first(op_orientation));
			  	break;
			  case 18 :  /* for var = exp1 to exp2 [step exp3] */
			  	/* create new for block */
				block = add_block(GLE_SRCBLK_FOR, srclin);
				get_var_add(&v, &vtyp);
				block->setVariable(v);
				/* translate first part of FOR into variable assignment */
				pcode.setInt(pcode.size()-1, 51);
				pcode.addInt(v);
				get_token("=");
				get_exp(pcode);
				pcode.setInt(pos_endoffs, pcode.size());
				pos_endoffs = pcode.size();
				/* now add real FOR, jump by next should go to this location */
				block->setOffset1(pos_endoffs);
				pcode.addInt(0);
				pcode.addInt(18); /* opcode FOR */
				pcode.addInt(v);
				block->setOffset2(pcode.size());
				pcode.addInt(0);
				get_token("TO");
				get_exp(pcode);
				get_optional(op_for_step, pcode);
				break;
			  case 19 :/* goto */
				throw error("GOTO is considered bad programming practice :-)");
				break;
			  case 20 : /* gsave */
				break;
			  case 54 : /* grestore */
				break;
			  case 21 : /* icon x y */
				get_xy(pcode);
				break;
			  case 22 :  /* IF exp THEN ...  */
				if (nbcmd > 0) {
					// Do not move to parse_if (the latter is also used for "else if")
					throw error(pos_first, "command must be the first command on the line");
				}
				parse_if(srclin, pcode);
				if (tokens->has_more_tokens()) {
					m_auto_endif = true;
					allow_extra_tokens = true;
				}
				break;
			  case 55 : /* POSTCRIPT file$  width  height */
				get_strexp(pcode);
				get_xy(pcode);
				break;
				//
				// -- no need to ifdef these as
				//
			  case 67 : /* TIFF file$  width  height */
				printf("\nWarning: TIFF is deprecated: use BITMAP instead\n");
				get_strexp(pcode);
				get_xy(pcode);
				get_optional(op_bitmap, pcode);
				break;
			  case 68 : /* BITMAP file width height [type colors compress dpi greyscale resize] */
				get_strexp(pcode);
				get_xy(pcode);
				get_optional(op_bitmap, pcode);
				break;
			  case 69 : /* BITMAP_INFO file width, height [type] */
				get_strexp(pcode);
				get_var(pcode);
				get_var(pcode);
				get_optional(op_bitmap_info, pcode);
				break;
			  case GLE_KW_COLORMAP:
			  	get_strexp(pcode); // function
				get_xy(pcode);     // xrange
				get_xy(pcode);     // yrange
				get_xy(pcode);     // bitmap size
				get_xy(pcode);     // screen size
				get_optional(op_colormap, pcode);
			  	break;
			  case 56 : /* draw a previously defined object */
				get_strexp(pcode);
				get_optional(op_draw, pcode);
				break;
			  case 57 : /* plotter fonts */
				get_token("FONTS");
				break;
			  case 23 : /* include "string" */
				setSpecial(GLE_PARSER_INCLUDE);
				temp_str = tokens->next_token();
				str_remove_quote(temp_str);
				setInclude(temp_str);
				break;
			  case 58 : /* bigfile "string" This waits until 'run' to read the file*/
				get_strexp(pcode);
				break;
			  case 24 : /* input 1 a$=20,fill$=10,j=6 prompt "Age and name " */
				    /* input 1 a$,yval prompt "Age and name " */
			  case 25 : /* join a.tl->b.br   ,   string, arrows&line, string */
				get_strexp(pcode);
				pcode.addInt(get_first(op_joinname));
				get_strexp(pcode);
				get_optional(op_curve, pcode);
				break;
			  case 26 : /* marker square [2.2] */
				get_marker(pcode);
				if (not_at_end_command()) {
					get_exp(pcode);
				} else {
					pcode.addInt(0);
				}
				break;
			  case 27 : /* MOVE name */
				get_strexp(pcode);
				break;
			  case 28 : /* narc, Arc in clockwise direction */
				get_exp(pcode);
				get_xy(pcode);
				get_optional(op_arc, pcode);
				break;
			  case 29 : /* newpath */
				break;
			  case 30 : /* next */
			  	single_cmd = true;
				if (not_at_end_command()) {
			  		block = check_block_type(pos_first, GLE_SRCBLK_NEXT, GLE_SRCBLK_FOR, -1);
					get_var_add(&v, &vtyp);
					check_loop_variable(v);
				} else {
			  		block = check_block_type(pos_first, GLE_SRCBLK_NEXT, GLE_SRCBLK_WHILE, GLE_SRCBLK_UNTIL);
				}
				/* Set jump address */
				pcode.addInt(block->getFirstLine());
				pcode.addInt(block->getOffset1());
				/* Update address in first line @ getOffset2() */
				(*gpcode)[block->getFirstLine()][block->getOffset2()] = srclin;
				/* And remove block from stack */
				remove_last_block();
				break;
			  case 31 : /* pie r a1 a2 fill pattern */
				get_exp(pcode);
				get_xy(pcode);
				get_optional(op_fill, pcode);
				break;
			  case 33 :
				get_xy(pcode);
				get_xy(pcode);
				get_xy(pcode);
				break;
			  case 34 : /* region */
				throw error("REGION is not yet implemented");
				break;
			  case 50 : /* Return EXP */
				if (not_at_end_command()) {
					get_exp_eol(pcode);
				} else {
					etype=1;
					polish("0", pcode, &etype);
				}
				if (tokens->has_more_tokens()) {
					throw error(pos_first, "return should be the last command on a line");
				}
				block = find_block(GLE_SRCBLK_MAGIC+GLE_OPBEGIN_SUB);
				if (block == NULL) {
					throw error(pos_first, "return outside of subroutine");
				}
				block = block->addDependendBlock(GLE_SRCBLK_RETURN, srclin);
				block->setOffset2(pcode.size());
				pcode.addInt(0);
				break;
			  case 35 : /* Reverse the current path */
				break;
			  case 36 : /* rline */
				get_xy(pcode);
				get_optional(op_line, pcode);
				break;
			  case 37 : /* rmove */
				get_xy(pcode);
				break;
			  case 38 : /* rotate */
				get_exp(pcode);
				break;
			  case 39 : /* save joe */
				get_strexp(pcode);
				break;
			  case 40: /* scale x y */
				get_xy(pcode);
				break;
			  case 41: /* SET color font hei just lwidth lstyle ldist */
				while (not_at_end_command()) {
				 f = get_first(op_set);
				 pcode.addInt(500+f);
				 switch (f) {
				  case 1: /* height */
					get_exp(pcode);
					break;
				  case 2: /* font */
					get_font(pcode);
					break;
				  case 3: /* justify */
					get_justify(pcode);
					break;
				  case 4: /* color */
					get_color(pcode);
					break;
				  case 5: /* dashlen */
					get_exp(pcode);
					break;
				  case 6: /* dash */
					get_exp(pcode);
					break;
				  case 7: /* lwidth */
					get_exp(pcode);
					break;
				  case 8: /* join */
					/* get_join(); */
					get_join(pcode);
					break;
				  case 9: /* cap */
					/* get_cap(); */
					get_cap(pcode);
					break;
				  case 10: /* fontlwidth */
					get_exp(pcode);
					break;
				  case OP_SET_ARROW_STYLE:
				  case OP_SET_IMAGE_FORMAT:
				  case OP_SET_TEX_SCALE:
				  	get_strexp(pcode);
					break;
				  case OP_SET_TEX_LABELS:
				  	get_exp(pcode);
					break;
				  case OP_SET_ARROW_SIZE:
				  case OP_SET_ARROW_ANGLE:
				  case OP_SET_TITLE_SCALE:
				  case OP_SET_ATITLE_SCALE:
				  case OP_SET_ALABEL_SCALE:
				  case OP_SET_TICKS_SCALE:
				  case OP_SET_ATITLE_DIST:
				  case OP_SET_ALABEL_DIST:
				  	get_exp(pcode);
					break;
				  }
				}
				break;
			  case 42 : /* size */
				get_xy(pcode);
				get_optional(op_size, pcode);
				break;
			  case 43 : /* STROKE */
				break;
			  case 44 : /* SUB JOE X Y$ Z   ... END SUB  */
				single_cmd = true;
				if (in_sub) {
					throw error("can't define a subroutine within a sub ");
				}
				in_sub = true;
				sdx = get_subroutine_def(pcode);
				add_block(GLE_SRCBLK_MAGIC+GLE_OPBEGIN_SUB, srclin);
				break;
			  case 45 : /* text */
				temp_str = m_tokens.read_line();
				pcode.addStringNoID(temp_str);
				break;
			  case 59 : /* textdef */
				temp_str = m_tokens.read_line();
				pcode.addStringNoID(temp_str);
				break;
			  case 46 : /* translate x y */
				get_xy(pcode);
				break;
			  case 47 : /* until */
			  	single_cmd = true;
				get_exp_eol(pcode);
				block = add_block(GLE_SRCBLK_UNTIL, srclin);
				block->setOffset2(pcode.size());
				pcode.addInt(0);
				break;
			  case 48 : /* while */
			  	single_cmd = true;
				get_exp_eol(pcode);
				block = add_block(GLE_SRCBLK_WHILE, srclin);
				block->setOffset2(pcode.size());
				pcode.addInt(0);
				break;
			  case 32 : /* print numexp,strexp,strexp */
			  case 49 : /* write numexp,strexp,strexp */
				while (not_at_end_command()) {
					pcode.addInt(fctkey);
					position = pcode.size();
					pcode.addInt(0);
					pcode.setInt(position, get_anyexp(pcode));
				}
				break;
			  case 74 : /* tex */
				get_strexp(pcode);
				get_optional(op_tex, pcode);
				break;
			  case GLE_KW_RESTOREDEFAULTS:
				break;
			  default:
				throw error("unrecognised command {"+first+"}");
			}
		}
		if (!allow_extra_tokens && test_not_at_end_command()) {
			temp_str = tokens->read_line();
			throw error(string("extra tokens after command '")+first+"': '"+temp_str+"'");
		}
		pcode.setInt(pos_endoffs, pcode.size());
		nbcmd++;
		if (nbcmd > 1 && single_cmd) {
			throw error(pos_first, "command must occur on a separate line");
		}
	}
}

void GLEParser::do_text_mode(SourceLine &SLine, Tokenizer* tokens, GLEPcode& pcode) throw (ParserError) {
	int pos_endoffs = pcode.size();
	// Save space for end offset
	pcode.addInt(0);
	pcode.addInt(5);
	string str = tokens->read_line();
	// Handle comment symbol '!' at start of line
	if (str.length() > 0 && str[0] == '!') {
		str = "";
	}
	str_replace_start(str, "\\!", "!");
	// Check for end of block
	int pos = str_starts_with_trim(str, "END");
	if (pos != -1) {
		int len = str.length();
		string second_token = str.substr(pos, len-pos);
		str_trim_both(second_token);
		int idx = gt_index(op_begin, (char*)second_token.c_str());
		if (idx == cur_mode) {
			pcode.addInt(0);
			cur_mode = 0;
			return;
		}
	}
	// Add string to the block
	pcode.addInt(cur_mode);
	pcode.addStringNoID(str);
	pcode.setInt(pos_endoffs, pcode.size());
}

GLESourceBlock* GLEParser::add_else_block(int srclin, GLEPcode& pcode, bool dangling) {
	remove_last_block();
	GLESourceBlock* block = add_block(GLE_SRCBLK_ELSE, srclin);
	block->setOffset2(pcode.size());
	block->setDangling(dangling);
	pcode.addInt(0);
	pcode.addInt(0);
	return block;
}

GLESourceBlock* GLEParser::add_else_block_update(int srclin, GLEPcode& pcode, bool dangling) {
	GLESourceBlock* if_block = last_block();
	int line = if_block->getFirstLine();
	int offs = if_block->getOffset2();
	GLESourceBlock* else_block = add_else_block(srclin, pcode, dangling);
	(*gpcode)[line][offs] = srclin;
	(*gpcode)[line][offs+1] = pcode.size();
	return else_block;
}

void GLEParser::do_endif(int srclin) {
	GLESourceBlock* block = last_block();
	(*gpcode)[block->getFirstLine()][block->getOffset2()] = srclin;
	remove_last_block();
	block = last_block();
	while (block != NULL && block->isDanglingElse()) {
		(*gpcode)[block->getFirstLine()][block->getOffset2()] = srclin;
		remove_last_block();
		block = last_block();
	}
}

void GLEParser::do_endsub(int srclin) {
	GLESourceBlock* block = last_block();
	int nb = block->getNbDependendingBlocks();
	for (int i = 0; i < nb; i++) {
		GLESourceBlock* dep = block->getDependingBlock(i);
		(*gpcode)[dep->getFirstLine()][dep->getOffset2()] = srclin;
	}
}

GLESourceBlock* GLEParser::find_block(int type) {
	int last = m_blocks.size()-1;
	while (last >= 0 && m_blocks[last].getType() != type) {
		last--;
	}
	return last >= 0 ? &m_blocks[last] : NULL;
}

GLESourceBlock* GLEParser::add_block(int type, int first_line) {
	m_blocks.push_back(GLESourceBlock(type, first_line));
	return &m_blocks.back();
}

GLESourceBlock* GLEParser::last_block() {
	return m_blocks.size() > 0 ? &m_blocks.back() : NULL;
}

void GLEParser::remove_last_block() {
	m_blocks.pop_back();
}

void GLEParser::check_loop_variable(int var) throw (ParserError) {
	GLESourceBlock* block = last_block();
	if (block == NULL || var != block->getVariable()) {
		stringstream err;
		err << "illegal variable '" << var_get_name(var);
		err << "': loop variable is '" << var_get_name(block->getVariable()) << "'";
		throw error(err.str());
	}
}

GLESourceBlock* GLEParser::check_block_type(int pos, int t0, int t1, int t2) throw (ParserError) {
	GLESourceBlock* block = last_block();
	if (block == NULL) {
		stringstream err;
		const char* end_t0 = GLESourceBlockEndName(t0);
		if (end_t0 != NULL) err << end_t0 << " ";
		err << "'" << GLESourceBlockName(t0) << "' without corresponding ";
		const char* begin_t1 = GLESourceBlockBeginName(t1);
		if (begin_t1 != NULL) err << begin_t1 << " ";
		err << "'" << GLESourceBlockName(t1) << "'";
		if (t2 != -1) {
			err << " or ";
			const char* begin_t2 = GLESourceBlockBeginName(t2);
			if (begin_t2 != NULL) err << begin_t2 << " ";
			err << "'" << GLESourceBlockName(t2) << "'";
		}
		throw error(pos, err.str());
	}
	if (block->getType() != t1 && block->getType() != t2) {
		stringstream err;
		err << "unterminated '" << block->getName() << "'";
		err << " " << block->getKindName();
		err << " (starting on line " << block->getFirstLine() << ") before ";
		const char* end_name = GLESourceBlockEndName(t0);
		if (end_name != NULL) err << end_name << " ";
		err << "'" << GLESourceBlockName(t0) << "'";
		throw error(pos, err.str());
	}
	return block;
}

GLESourceBlock::GLESourceBlock(int type, int first_line) {
	m_block_type = type;
	m_first_line = first_line;
	m_variable = -1;
	m_pcode_offs1 = 0;
	m_pcode_offs2 = 0;
	m_dangling = false;
	m_deps = NULL;
}

GLESourceBlock::GLESourceBlock(const GLESourceBlock& block) {
	m_block_type = block.m_block_type;
	m_first_line = block.m_first_line;
	m_variable = block.m_variable;
	m_pcode_offs1 = block.m_pcode_offs1;
	m_pcode_offs2 = block.m_pcode_offs2;
	m_dangling = block.m_dangling;
	m_deps = NULL;
	if (block.m_deps != NULL) {
		int size = block.m_deps->size();
		m_deps = new vector<GLESourceBlock>();
		for (int i = 0; i < size; i++) {
			m_deps->push_back((*block.m_deps)[i]);
		}
	}
}

GLESourceBlock::~GLESourceBlock() {
	if (m_deps != NULL) delete m_deps;
}

GLESourceBlock* GLESourceBlock::addDependendBlock(int type, int first_line) {
	if (m_deps == NULL) m_deps = new vector<GLESourceBlock>();
	m_deps->push_back(GLESourceBlock(type, first_line));
	return &m_deps->back();
}

int GLESourceBlock::getNbDependendingBlocks() {
	return m_deps != NULL ? m_deps->size() : 0;
}

const char* GLESourceBlock::getName() {
	return GLESourceBlockName(m_block_type);
}

const char* GLESourceBlock::getKindName() {
	switch (m_block_type) {
		case GLE_SRCBLK_UNTIL:
		case GLE_SRCBLK_WHILE:
		case GLE_SRCBLK_FOR:
			return "loop";
		default:
			return "block";
	}
}

const char* GLESourceBlockEndName(int type) {
	switch (type) {
		case GLE_SRCBLK_NEXT:
		case GLE_SRCBLK_ELSE:
			return NULL;
		default:
			return "end";
	}
}

const char* GLESourceBlockBeginName(int type) {
	switch (type) {
		case GLE_SRCBLK_FOR:
		case GLE_SRCBLK_UNTIL:
		case GLE_SRCBLK_WHILE:
		case GLE_SRCBLK_MAGIC+GLE_OPBEGIN_IF:
			return NULL;
		default:
			return "begin";
	}
}

const char* GLESourceBlockName(int type) {
	if (type > GLE_SRCBLK_MAGIC) {
		int count, width;
		get_key_info(op_begin, &count, &width);
		for (int i = 0; i < count; i++) {
			if (op_begin[i].idx == type - GLE_SRCBLK_MAGIC) return op_begin[i].name;
		}
	}
	switch (type) {
		case GLE_SRCBLK_UNTIL: return "UNTIL";
		case GLE_SRCBLK_WHILE: return "WHILE";
		case GLE_SRCBLK_FOR:   return "FOR";
		case GLE_SRCBLK_NEXT:  return "NEXT";
		case GLE_SRCBLK_ELSE:  return "ELSE";
		default: return "unknown";
	}
}

void gt_xy(int *curtok, TOKENS tk, int *ntok, int *pcode, int *plen) {
	int etype;
	etype = 1;
	if (*ntok < *curtok) {
		gprint("Expecting x expression on end of line\n");
	}
	polish(tok((*curtok)++),(char *) pcode,plen,&etype);
	etype = 1;
	if (*ntok < *curtok) {
		gprint("Expecting y expression on end of line\n");
	}
	polish(tok((*curtok)++),(char *) pcode,plen,&etype);
}

void gt_find_error(const char* found, OPKEY lkey, int nk) {
	gprint("Found {%s} expecting one of: \n", found);
	printf("		");
	for (int i=0; i<nk; i++) {
		printf("%s",lkey[i].name);
		if (i != nk-1) printf(", ");
		if ((i+1) % 3 == 0) printf("\n		");
	}
	if (nk % 3 != 0) printf("\n");
}

int gt_first(OPKEY lkey, int *curtok, TOKENS tk, int *ntok, int *pcode, int *plen) {
	int nk,i,width=0,p;
	for (i=0; lkey[i].typ!=typ_end; i++) {
		p = lkey[i].pos;
		if (p>width) width = p ;
	}
	nk = i;
	for (i=0; i<nk; i++) {
		if (str_i_equals(lkey[i].name,tok(*curtok))) {
			(*curtok)++;
			return lkey[i].idx;
		}
	}
	gt_find_error(tok(*curtok), lkey, nk);
	(*curtok)++;
	return 0;
}

int gt_firstval(OPKEY lkey,char *s) {
	int nk,i,width=0,p;
	for (i=0; lkey[i].typ!=typ_end; i++) {
		p = lkey[i].pos;
		if (p>width) width = p ;
	}
	nk = i;
	for (i=0; i<nk; i++) {
		if (str_i_equals(lkey[i].name,s)) {
			dbg gprint("Got match {%s} \n",s);
			return lkey[i].idx;
		}
	}
	gt_find_error(s, lkey, nk);
	return 0;
}

int gt_index(OPKEY lkey,char *s) {
	for (int i = 0; lkey[i].typ!=typ_end; i++) {
		if (str_i_equals(lkey[i].name,s)) {
			return lkey[i].idx;
		}
	}
	return 0;
}

#undef get_first
#define get_first(key) gt_first(key,curtok,tk,ntok,pcode,plen)

void mystrcpy(char **d,char *s)
{
	if (*d!=0) myfree(*d);
	*d = 0;
	*d = (char*) myallocz(strlen(s)+1);
	strcpy(*d,s);
}
#undef get_exp
#define get_exp() polish(tok((*curtok)++),(char *) pcode,plen,&etype)

/* pos=   Offset to find the data			*/
/* idx=   For switches, which can only have one value. 	*/

int pass_justify(char *s) {
	return gt_firstval(op_justify,s);
}

int pass_color(const char *s) {
	double xx;
	int j;
	int i;
	char vv[80];
	if (str_i_str(s,"RGB")!=NULL) {
		polish_eval((char*)s,&xx);
	} else if (*s=='.' || *s=='(' || isdigit(*s)) {
		strcpy(vv,"cvtgray(");
		strcat(vv,s); strcat(vv,")");
		polish_eval(vv,&xx);
	} else if (strchr(s,'$') != NULL) {
		strcpy(vv,"cvtcolor(");
		strcat(vv,s); strcat(vv,")");
		polish_eval(vv,&xx);
	} else {
		if (s[0] == 0) {
			gprint("Expecting color name, but found empty string");
		} else {
			return gt_firstval(op_color_typ,(char*)s);
		}
	}
	memcpy(&j,&xx,sizeof(int));
	return j;
}

int pass_color_var(const char *s) {
	if (strchr(s,'$') != NULL) {
		int idx, typ;
		char tmpcol[100];
		string var_str = s;
		str_to_uppercase(var_str);
		var_find((char*)var_str.c_str(), &idx, &typ);
		if (idx >= 0) {
			var_getstr(idx, tmpcol);
			return pass_color(tmpcol);
		} else {
			gprint("Color '%s' not defined", s);
			return 0;
		}
	} else {
		return pass_color(s);
	}
}

#define get_exps(ss) polish(ss,(char *) pcode,plen,&etype)

void get_color(TOKENS tk,int *ntok,int *curtok,int *pcode,int *plen) {
	int etype = 1;
	char vv[80];

	if (str_i_str(tok(*curtok),"RGB")!=NULL) {
		get_exps(tok(*curtok));
	} else if (*tok(*curtok)=='(' || isdigit(*tok(*curtok))) {
		strcpy(vv,"cvtgray(");
		strcat(vv,tok(*curtok)); strcat(vv,")");
		get_exps(vv);
	} else	if (strchr(tok(*curtok),'$') != NULL) {
		strcpy(vv,"cvtcolor(");
		strcat(vv,tok(*curtok)); strcat(vv,")");
		get_exps(vv);
	} else {
		*(pcode+(*plen)++) = 8;
		*(pcode+(*plen)++) = get_first(op_color_typ);
		return;
	}
	(*curtok)++;
}

void get_fill(TOKENS tk,int *ntok,int *curtok,int *pcode,int *plen) {
	int etype = 1;
	char vv[80];

	if (str_i_str(tok(*curtok),"RGB")!=NULL) {
		get_exps(tok(*curtok));
	} else if (*tok(*curtok)=='(' || isdigit(*tok(*curtok))) {
		strcpy(vv,"cvtgray(");
		strcat(vv,tok(*curtok)); strcat(vv,")");
		get_exps(vv);
	} else	if (strchr(tok(*curtok),'$') != NULL) {
		strcpy(vv,"cvtcolor(");
		strcat(vv,tok(*curtok)); strcat(vv,")");
		get_exps(vv);
	} else {
		*(pcode+(*plen)++) = 8;
		*(pcode+(*plen)++) = get_first(op_color_typ);
		return;
	}
	(*curtok)++;
}

void get_justify(TOKENS tk,int *ntok,int *curtok,int *pcode,int *plen) {
	*(pcode+(*plen)++) = get_first(op_justify);
}

void get_join(TOKENS tk,int *ntok,int *curtok,int *pcode,int *plen) {
	*(pcode+(*plen)++) = get_first(op_join);
}

void get_cap(TOKENS tk,int *ntok,int *curtok,int *pcode,int *plen) {
	*(pcode+(*plen)++) = get_first(op_cap);
}

struct mark_struct { char *name; char *font; int cc; double rx; double ry; double scl;};
struct mark_struct stdmark[] = {
		"DOT","RM",46,-.125,-.0435,3.0,		/* dot */
		"CROSS","TEXSY",2,-.375,-.24,1.0,	/* cross */
		"FCIRCLE","GLEMARK",4,-.5,-.5,0.7,	/* fcircle */
		"FSQUARE","GLEMARK",6,-.5,-.5,0.7,	/* fsquare */
		"FTRIANGLE","GLEMARK",5,-.5,-.433,0.7,	/* ftriangle */
		"FDIAMOND","GLEMARK",7,-.5,-.7,0.7,	/* fdiamond */
		"CIRCLE","GLEMARK",12,-.5,-.5,0.7,	/* circle */
		"SQUARE","GLEMARK",2,-.5,-.5,0.7,	/* square */
		"TRIANGLE","GLEMARK",1,-.5,-.433,0.7,	/* triangle */
		"DIAMOND","GLEMARK",3,-.5,-.7,0.7,	/* diamond */
		"PLUS","TEXCMR",43,-.375,-.24,1.0,	/* plus, fixed */
		"CLUB","TEXSY",124,-.38,-.3,1.0,	/* club */
		"HEART","TEXSY",126,-.38,-.34,1.0,	/* heart */
		"DIAMONDZ","TEXSY",125,-.38,-.26,1.0,	/* diamondz */
		"SPADE","TEXSY",127,-.375,-.24,1.0,	/* spade (needs fixing) */
		"STAR","TEXMI",63,-.25,-.21,1.0,	/* star */
		"SNAKE","TEXSY",120,-.21,-.22,1.0,	/* snake */
		"DAG","TEXSY",121,-.21,-.22,1.0,	/* dag */
		"DDAG","TEXSY",122,-.21,-.22,1.0,	/* dagg */
		"ASTERIX","TEXSY",3,-.25,-.24,1.0,	/* asterix */
		"ASTERISK","TEXSY",3,-.25,-.24,1.0,	/* asterix */
		"OPLUS","TEXSY",8,-.40,-.24,1.0,	/* oplus */
		"OMINUS","TEXSY",9,-.40,-.24,1.0,	/* ominus */
		"OTIMES","TEXSY",10,-.40,-.24,1.0,	/* otimes */
		"ODOT","TEXSY",12,-.40,-.24,1.0,	/* odot */
		"TRIANGLEZ","TEXSY",52,-.44,-.26,1.0,	/* trianglez */
		"DIAMONDZ","TEXSY",125,-.38,-.26,1.0,	/* diamondz */
		"WCIRCLE","GLEMARK",8,-.5,-.5,0.7,	/* wcircle */
		"WTRIANGLE","GLEMARK",9,-.5,-.433,0.7,	/* wtriangle */
		"WSQUARE","GLEMARK",10,-.5,-.5,0.7,	/* wsquare */
		"WDIAMOND","GLEMARK",11,-.5,-.7,0.7,	/* wdiamond */
		"PLANE","PSZD",40,0.0,0.0,1.0,		/* ZapDingbats */
		"HANDPEN","PSZD",45,0.0,0.0,1.0,	/* ZapDingbats */
		"SCIRCLE","PSZD",109,0.0,0.0,1.0,	/* ZapDingbats */
		"SSQUARE","PSZD",111,0.0,0.0,1.0,	/* ZapDingbats */
		"PHONE","PSZD",37,0.0,0.0,1.0,		/* ZapDingbats */
		"LETTER","PSZD",41,0.0,0.0,1.0,		/* ZapDingbats */
		"STAR2","PSZD",69,0.0,0.0,1.0,		/* ZapDingbats */
		"STAR3","PSZD",79,0.0,0.0,1.0,		/* ZapDingbats */
		"STAR4","PSZD",98,0.0,0.0,1.0,		/* ZapDingbats */
		"FLOWER","PSZD",96,0.0,0.0,1.0,		/* ZapDingbats */
		NULL,NULL,0,0,0,0
};	/* change range check below when adding markers */

void mark_clear(void) {
	int i,fg;
	struct mark_struct *p;
	if (std_nmrk==0) {
		for (i=0; stdmark[i].name !=NULL; i++) {
			p = &stdmark[i];
			fg = false;
			if (p->rx==0) fg = true;
			g_defmarker(p->name,p->font,p->cc,p->rx,p->ry
				,p->scl,fg);
		}
		std_nmrk = nmrk;
	}
	for (i=0; i<nmark; i++) {
	  if (mark_sub[i]!=NULL) { myfree(mark_sub[i]); mark_sub[i]=NULL;}
	  if (mark_name[i]!=NULL) { myfree(mark_name[i]); mark_name[i]=NULL;}
	}
	for (i=std_nmrk; i<nmrk; i++) {
	  if (mrk_name[i]!=NULL) { myfree(mrk_name[i]); mrk_name[i]=NULL;}
	  if (mrk_fname[i]!=NULL) { myfree(mrk_fname[i]); mrk_fname[i]=NULL;}
	}
	nmrk = std_nmrk;
	nmark = 0;
}

void get_marker(TOKENS tk,int *ntok,int *curtok,int *pcode,int *plen) {
	int etype = 1;
	char vv[80];
	if (*tok(*curtok)=='(' || isdigit(*tok(*curtok))) {
		strcpy(vv,"cvtint(");
		strcat(vv,tok(*curtok)); strcat(vv,")");
		get_exps(vv);
	} else	if (strchr(tok(*curtok),'$') != NULL) {
		strcpy(vv,"cvtmarker(");
		strcat(vv,tok(*curtok)); strcat(vv,")");
		get_exps(vv);
	} else {
		*(pcode+(*plen)++) = 8;
		*(pcode+(*plen)++) = pass_marker(tok(*curtok));
	}
	(*curtok)++;
}

int pass_marker(char *s) {
	int i;
	int f=0;
	/* if 0, maybe its a user defined marker, ie a subroutine */
	/* Use -ve to signify subroutine instead of normal marker */
	for (i=0; i<nmark; i++) {
		if (str_i_equals(mark_name[i],s)) {
			f = -(++i);
			break;
		}
	}
	if (f==0)  {
		for (i=nmrk-1; i>=0; i--) {
			if (str_i_equals(mrk_name[i],s)) {
				f = ++i;
				break;
			}
		}
	}
	if (f==0) gprint("Invalid marker name {%s} \n",s);
	return f;
}

void pass_file_name(const char* name, string& file) {
	// Support operations on strings, but make sure it also works with unquoted strings containing "/" chars
	if (str_contains(name, '"') || str_contains(name, '$') || str_contains(name, '+')) {
		polish_eval_string(name, &file);
	} else {
		file = name;
	}
}

void set_global_parser(GLEParser* parser) {
	g_parser = parser;
}

GLEParser* get_global_parser() {
	return g_parser;
}

GLEPolish* get_global_polish() {
	return g_parser != NULL ? g_parser->getPolish() : NULL;
}
