
#define GRAPHDEF

#include "all.h"
#include "tokens/stokenizer.h"
#include "SourceLine.h"
#include "mem_limits.h"
#include "token.h"

#include "tokens/Tokenizer.h"
#include "glearray.h"
#include "polish.h"
#include "pass.h"
#include "var.h"
#include "mygraph.h"
#include "graph.h"
#include "begin.h"
#include "core.h"
#include "axis.h"
#include "cutils.h"
#include "gprint.h"
#include "key.h"

#define GLEG_CMD_AXIS     1
#define GLEG_CMD_LABELS   2
#define GLEG_CMD_SIDE     3
#define GLEG_CMD_SUBTICKS 4
#define GLEG_CMD_TICKS    5

#define BAR_SET_COLOR   0
#define BAR_SET_FILL    1
#define BAR_SET_TOP     2
#define BAR_SET_SIDE    3
#define BAR_SET_PATTERN 4

extern int trace_on;
static int xxgrid[GLE_AXIS_MAX+1];

char *letstr[MAX_NUMBER_OF_LET_COMMANDS];
int letline[MAX_NUMBER_OF_LET_COMMANDS];
int nlet;

int g_nbar = 0;

int freedataset(int i);
void free_temp(void);
void set_sizelength(void);
void draw_graph(KeyInfo* keyinfo) throw (ParserError);
void do_set_bar_color(const char* tk, bar_struct* bar, int type);
void do_set_bar_style(const char* tk, bar_struct* bar);
void do_axis_part_all(int xset) throw (ParserError);
bool do_remaining_entries(int ct);
void graph_freebars();

axis_struct xx[GLE_AXIS_MAX+1];

char large_inbuff[LARGE_INBUFF_SIZE];

vector<int> g_fcalls;
vector<int> g_funder;

GLEColorMap* g_colormap;

// Font sizes in a graph are based in g_fontsz, but this was set in the original
// GLE to a fixed proportion of the largest dimension of a graph. The size of an
// axis is defined in terms of its base size, which is equal to g_fontsz.
// A useful value for base is 0.25

void begin_graph(int *pln , int *pcode , int *cp) throw (ParserError) {
	static int ct,i,ncol,np,*m,t,d;
	static int iid1,iid2,d2,st;
	double *px,*py,ox,oy;
	static int xset,erflg,gosret;
	bool other_axis;
	stringstream err;

	FILE *fptr;
	g_colormap = NULL;
	nlet = 0;

	KeyInfo keyinfo;
	freeafont();  /* get some more memory, we might need it */
	g_hscale = .7;
	g_vscale = .7;
	nlet = 0;
	erflg = true;
	g_nobox = true;
	g_center = false;
	g_math = false;
	for (i = 1; i <= GLE_AXIS_MAX; i++) {
		xxgrid[i] = 0;
		vinit_axis(i);
	}
	graph_init();
	g_get_usersize(&g_xsize,&g_ysize);
	g_get_hei(&g_fontsz);
	set_sizelength();
	dp[0] = DP_CAST myallocz(sizeof(*dp[1]));  /* dataset for default settings */
	dp[0]->lwidth = -1;
	(*pln)++;
	begin_init();
	for (;;) {
		/* process next line */
do_next_line:
		srclin[0] = 0;
		st = begin_token(&pcode,cp,pln,srclin,tk,&ntk,outbuff);
		if (!st) {
			draw_graph(&keyinfo);
			return;
		}
		ct = 1;
		if (str_i_equals(tk[ct],"BAR")) goto do_bar;
		else if (str_i_equals(tk[ct],"DATA")) goto do_data;
		else kw("DATA") goto do_data;
		else kw("EXTRACT") goto do_extract;
		else kw("FILL") goto do_fill;
		else kw("HSCALE") goto do_hscale;
		else kw("LET") 	goto do_letsave;
		else kw("SIZE") goto do_size;
		else kw("KEY") goto do_key;
		else kw("VSCALE") goto do_vscale;
		else kw("SCALE") goto do_scale;
		else kw("COLORMAP") goto do_colormap;
		else kw("TITLE") goto do_main_title;
		else kw("UNDER") g_funder.push_back((*pln)-1);
		else kw("!") goto do_next_line;
		else kw(" ") goto do_next_line;
		else if (str_i_str(tk[ct],"NOTICKS") != NULL) goto do_noticks;
		else if (str_i_str(tk[ct],"AXIS") != NULL) do_axis_part_all(GLEG_CMD_AXIS);
		else if (str_i_str(tk[ct],"LABELS") != NULL) do_axis_part_all(GLEG_CMD_LABELS);
		else if (str_i_str(tk[ct],"SIDE") != NULL) do_axis_part_all(GLEG_CMD_SIDE);
		else if (str_i_str(tk[ct],"SUBTICKS") != NULL) do_axis_part_all(GLEG_CMD_SUBTICKS);
		else if (str_i_str(tk[ct],"TICKS") != NULL) do_axis_part_all(GLEG_CMD_TICKS);
		else if (str_i_str(tk[ct],"NAMES") != NULL) goto do_names;
		else if (str_i_str(tk[ct],"PLACES") != NULL) goto do_places;
		else if (str_i_str(tk[ct],"TITLE") != NULL) goto do_title;
		else if (toupper(*tk[ct])=='D') goto do_datasets;
		else if (!do_remaining_entries(ct)) {
			/* This might be a function call, find out later */
			g_fcalls.push_back((*pln)-1);
		}
	}

/* ----------------------------------------------------------------*/
do_bar:
	/* bar d1,d2,d3 from d4,d5,d6 color red,green,blue fill grey,blue,green
		dist exp, width exp, LSTYLE 3,445,1
		3dbar .5 .5 top black,black,black side red,green,blue  notop */
{
	int ng,fi,di;
	char *ss;
	g_nbar++;
	br[g_nbar] = new bar_struct();
	br[g_nbar]->ngrp = 0;
	ct = 2;
	ss = strtok(tk[ct],",");
	while (ss!=NULL) {
	  if (toupper(*ss)=='D') {
		ng = (br[g_nbar]->ngrp)++;
		d =  atoi(ss+1);
		if (d==0 || d>99) gprint("Error in bar command {%s}  token number %d \n",tk[ct],ct);
		br[g_nbar]->to[ng] = d;
 	  }
	  ss = strtok(0,",");
	}

	br[g_nbar]->horiz = false;
	for (i = 0; i <= ng; i++) {
		br[g_nbar]->color[i] = g_get_grey(0.0);
		br[g_nbar]->fill[i] = g_get_grey(i == 0 ? 0.0 : 1.0-ng/i);
		br[g_nbar]->from[i] = 0;
		br[g_nbar]->pattern[i] = -1;
		g_get_line_width(&br[g_nbar]->lwidth[i]);
		strcpy(br[g_nbar]->lstyle[i] ,"1\0");
	}

	ct++;
	while (ct<=ntk)  {
		kw("DIST") 	br[g_nbar]->dist = next_exp;
	   else kw("WIDTH") 	br[g_nbar]->width = next_exp;
	   else kw("3D") {
		br[g_nbar]->x3d = next_exp;
		br[g_nbar]->y3d = next_exp;
	   }
	   else kw("NOTOP") 	br[g_nbar]->notop = true;
	   else kw("HORIZ") 	br[g_nbar]->horiz = true;
	   else kw("LSTYLE") 	next_str((char *) br[g_nbar]->lstyle);
	   else kw("STYLE") {
	   	ct += 1;
		do_set_bar_style(tk[ct], br[g_nbar]);
	   }
	   else kw("LWIDTH") 	br[g_nbar]->lwidth[0] = next_exp;
	   else kw("FROM") {
		fi = 0;
		ct +=1;
		ss = strtok(tk[ct],",");
		while (ss!=NULL) {
		  if (toupper(*ss)=='D') {
			di =  atoi(ss+1);
			br[g_nbar]->from[fi++] = di;
		  }
		  ss = strtok(0,",");
		}
	   }
	   else kw("COLOR") {
		ct += 1;
		do_set_bar_color(tk[ct], br[g_nbar], BAR_SET_COLOR);
	   }
	   else kw("SIDE") {
		ct += 1;
		do_set_bar_color(tk[ct], br[g_nbar], BAR_SET_SIDE);
	   }
	   else kw("TOP") {
		ct += 1;
		do_set_bar_color(tk[ct], br[g_nbar], BAR_SET_TOP);
	   }
	   else kw("FILL") {
		ct++;
		do_set_bar_color(tk[ct], br[g_nbar], BAR_SET_FILL);
	   }
	   else kw("PATTERN") {
		ct++;
		do_set_bar_color(tk[ct], br[g_nbar], BAR_SET_PATTERN);
	   }
	   else gprint("Unrecognised GRAPH BAR sub command {%s} %d \n "
		,tk[ct],ct);
	  ct++;
	}
}
goto do_next_line;
/*----------------------------------------------------------------*/
/* fill x1,d2 color green xmin 1 xmax 2 ymin 1 ymax 2   */
/* fill d1,x2 color green xmin 1 xmax 2 ymin 1 ymax 2   */
/* fill d1,d2 color green xmin 1 xmax 2 ymin 1 ymax 2   */
/* fill d1 color green xmin 1 xmax 2 ymin 1 ymax 2      */
do_fill:
{
	char *ss,s1[40],s2[40];
	fd[++nfd] = FD_CAST myallocz(sizeof(*fd[1]));
	ct = 2;
	strcpy(s1,strtok(tk[ct],","));
	ss = strtok(0,",");
	if (ss==NULL) strcpy(s2,""); else {
		strcpy(s2,ss);
		strtok(0,",");
	}

	if (str_i_equals(s1,"X1")) {
		fd[nfd]->type = 1;
		d = atoi(s2+1);
		if (d==0) { gprint("Not a valid fill option {%s} wanted FILL X1,D1 \n",tk[ct]); goto end_fill_read; }
		fd[nfd]->da = d;
	} else if (str_i_equals(s2,"X2")) {
		fd[nfd]->type = 2;
		d = atoi(s1+1);
		if (d==0) { gprint("Not a valid fill option {%s} wanted FILL D2,X2 \n",tk[ct]); goto end_fill_read; }
		fd[nfd]->da = d;
	} else if (!str_i_equals(s2,"")) {
		fd[nfd]->type = 3;
		d = atoi(s1+1);
		d2 = atoi(s2+1);
		if (d==0 || d2==0) { gprint("Expecting FILL D1,D2  found {%s}\n",tk[ct]); goto end_fill_read; }
		fd[nfd]->da = d;
		fd[nfd]->db = d2;
	} else if (toupper(*s1)=='D') {
		fd[nfd]->type = 4;
		d = atoi(s1+1);
		if (d==0) { gprint("Expecting FILL d1 found {%s} \n",tk[ct]); goto end_fill_read; }
		fd[nfd]->da = d;
	} else {
		gprint("Invalid fill option, wanted d1,d2 or x1,d1 or d1,x2 or d1 \n");
		goto end_fill_read;
	}
	ct++;

	fd[nfd]->color = g_get_grey(1.0-nfd*.1);
	fd[nfd]->xmin = -LARGE_NUM;
	fd[nfd]->xmax = LARGE_NUM;
	fd[nfd]->ymin = -LARGE_NUM;
	fd[nfd]->ymax = LARGE_NUM;

	while (ct<=ntk)  {
		kw("COLOR") 	{ct++; fd[nfd]->color = pass_color(tk[ct]);}
	   else kw("XMIN") 	fd[nfd]->xmin = next_exp;
	   else kw("XMAX") 	fd[nfd]->xmax = next_exp;
	   else kw("YMIN") 	fd[nfd]->ymin = next_exp;
	   else kw("YMAX") 	fd[nfd]->ymax = next_exp;
	   else gprint("Unrecognised GRAPH FILL sub command {%s} %d (Note: CGLE syntax change\n  Expected COLOR,XMIN,XMAX,YMIN,YMAX \n "
		,tk[ct],ct);
	   ct++;
	}
}
end_fill_read:;
goto do_next_line;
/*----------------------------------------------------------------*/
do_key:
	ct = 2;
	while (ct<=ntk)  {
		kw("OFFSET") {
			keyinfo.setOffsetX(next_exp);
			keyinfo.setOffsetY(next_exp);
		}
		else kw("MARGINS") {
			double mx = next_exp;
			double my = next_exp;
			keyinfo.setMarginXY(mx, my);
		}
		else kw("ABSOLUTE") {
			keyinfo.setOffsetX(next_exp);
			keyinfo.setOffsetY(next_exp);
			keyinfo.setAbsolute(true);
		}
		else kw("ROW") keyinfo.setBase(next_exp);
		else kw("LPOS") keyinfo.setLinePos(next_exp);
		else kw("LLEN") keyinfo.setLineLen(next_exp);
		else kw("NOBOX") keyinfo.setNoBox(true);
		else kw("HEI") keyinfo.setHei(next_exp);
		else kw("POSITION") next_str(keyinfo.getJustify());
		else kw("POS") next_str(keyinfo.getJustify());
		else kw("JUSTIFY") {
			next_str(keyinfo.getJustify());
			keyinfo.setPosOrJust(false);
		}
		else kw("JUST") {
			next_str(keyinfo.getJustify());
			keyinfo.setPosOrJust(false);
		}
		else kw("DIST") keyinfo.setDist(next_exp);
		else kw("COLDIST") keyinfo.setColDist(next_exp);
		else g_throw_parser_error("unrecognised KEY sub command: '",tk[ct],"'");
		ct++;
	}
goto do_next_line;
/*----------------------------------------------------------------*/
/* data a.dat				*/
/* data a.dat d2 d5			*/
/* data a.dat d1=c1,c3  d2=c5,c1	*/
/* data a.dat IGNORE n d2 d5 */
/* data a.dat IGNORE n d1=c1,c3  d2=c5,c1	*/
do_data:
{
	static char data_d[MAX_NUMBER_OF_ITEMS],data_x[MAX_NUMBER_OF_ITEMS],data_y[MAX_NUMBER_OF_ITEMS];
	int nm=0,expect_col=0,extract=0,xi,yi,dn,j,nncol=0,allind,ignore=0;
	char *s1,buff1[30];
	string infile;
	allind = 0;
	for (i = 1; i < MAX_NUMBER_OF_ITEMS; i++) { /* default mapping if no dataset names given */
		data_d[i] = freedataset(i);
		data_x[i] = 1;
		data_y[i] = i+1;
	}
	while (ct<=ntk)  { if (str_i_str(tk[ct],"=")!=NULL) extract = true; ct++; }
	/* goto third token */
	ct = 3;
	if (extract == static_cast<int>(true)) {
	 while (ct<=ntk)  {
		if (toupper(*tk[ct])=='D') {
			nm++;
			strncpy(buff1,tk[ct],29);
			s1 = strtok(buff1,"=");
			if (s1!=NULL) data_d[nm] = atoi(s1+1);
			if (s1==NULL || data_d[nm]==0) gprint("Expecting Dn=Ca,Cb (no spaces) {%s}\n",tk[ct]);
			s1 = strtok(NULL,",");
			if (s1!=NULL) data_x[nm] = atoi(s1+1);
			s1 = strtok(NULL,",");
			if (*s1=='*') {
				allind = data_d[nm];
			} else {
			 if (s1!=NULL) data_y[nm] = atoi(s1+1);
			 if (s1==NULL || data_x[nm]==0 || data_y[nm]==0) {
				gprint("Expecting Dn=Ca,Cb   (no spaces) {%s}\n",tk[ct]);
				goto do_next_line;
			 }
			 if (data_x[nm]>expect_col) expect_col = data_x[nm];
			 if (data_y[nm]>expect_col) expect_col = data_y[nm];
		 	}
		}
		else if(str_i_equals(tk[ct],"IGNORE"))
		{	/* user wants to ignore some lines */
			ignore = -1;
			ignore = atoi(tk[++ct]);
			if(ignore < 0) gprint("Expecting IGNORE n   where n is an integer got->{%s}\n",buff1);
		}
		else
		{
			if (*tk[ct] != 0) gprint("Expecting Dn=Ca,Cb   (no spaces) {%s}\n",tk[ct]);
		}
	   	ct++;
	 }
	} else {
	 while (ct<=ntk)  {
		if (*tk[ct]==' ') ct++;
		if (toupper(*tk[ct])=='D') {
			nm++;
			data_d[nm] = atoi(tk[ct]+1);
			data_x[nm] = 1;
			data_y[nm] = nm+1;
		}
		else if(str_i_equals(tk[ct],"IGNORE"))
		{	/* user wants to ignore some lines */
			ignore = -1;
			ignore = atoi(tk[++ct]);
			if(ignore < 0) gprint("Expecting IGNORE n   where n is an integer got->{%s}\n",buff1);
		}
		else gprint("Warning, found {%s}, expecting   data a.dat d1 d2 d3 \n",tk[ct]);
	   	ct++;
	  }
	}
	if (expect_col==0 && nm>0) expect_col = nm+1;
	if (allind>0) expect_col = 0;
	ct = 2;
	pass_file_name(tk[2], infile);
	fptr = fopen(infile.c_str(),"r");
/*	printf("\n%s",infile);*/
	if (fptr==NULL) {
		gprint("Unable to open data file {%s} \n",infile.c_str());
		perror("Reason");
		do_wait_for_enter_exit(1);
//		goto do_next_line;
	}

	int line = 0;
	vector<string> temp_s;
	vector<bool> temp_miss;
	vector<double> temp_data;

	TokenizerLanguage tok_lang;
	StringTokenizer str_tok(&tok_lang);
	tok_lang.setSpaceTokens(" ,\t");
	tok_lang.setLineCommentTokens("!");
	tok_lang.setDecimalDot('.');
	tok_lang.setParseStrings(true);
	str_tok.set_fname(infile.c_str());

	/* loop through each line in the file*/
	for (;!(feof(fptr));) {
/* 		if (*tk[ct]==' ') ct++; */

		inbuff[0]=0;
		large_inbuff[0]=0;
		fgets(large_inbuff,LARGE_INBUFF_SIZE-1,fptr);
		strip_crlf(large_inbuff);

		/* ignore the lines specified by user */
		if (line >= ignore) {

			/* check to see if input line is too long */
			if (strlen(large_inbuff) > LARGE_INBUFF_SIZE-1) {
				gprint("Lines in data file too long, split file");
			}

			str_tok.set_string(large_inbuff);
			str_tok.inc_line();

			ncol = 0;
			/* count the number of columns and put in tmpf and tmpmiss */
			while (true) {
				string& token = str_tok.try_next_token();
				if (token == "") {
					break;
				}
				ncol++;
				if (token == "*" || token == "?" || token == "-" || token == ".") {
					temp_miss.push_back(true);
					temp_data.push_back(0);
					temp_s.push_back("*");
				} else {
					temp_miss.push_back(false);
					temp_data.push_back(atof(token.c_str()));
					str_remove_quote(token);
					temp_s.push_back(token);
				}
			}

			if (nncol == 0) nncol = ncol;
			if (ncol != nncol && ncol > 0) {
				if (allind==0) {
					stringstream err;
					err << "inconsistent number of columns " << ncol << " <> " << nncol;
					err << " in '" << infile << "'";
					g_throw_parser_error(err.str());
				}
			}
			if (nm==0 && ncol>1) nm = ncol-1;
		} /* end if ignore */
		line++;
	}
	fclose(fptr);
	/* file is closed now resort data in tmpf and tmpfmiss*/
	if (expect_col > 0 && expect_col>nncol)
	{
		gprint("Warning, expecting %d columns in data file {%s} found %d \n",expect_col,infile.c_str(),ncol);
	}

	np = nncol > 0 ? temp_data.size() / nncol : 0;
	if (allind>0)
	{
		/* if data file contains y values only, y1, y2 ,y3...*/
		dn = allind;
		np = temp_data.size();
		if (dn>ndata) ndata = dn;
		if (dp[dn]==NULL)
		{
			dp[dn] = DP_CAST myallocz(sizeof(*dp[dn]));
			copy_default(dn);
		}
		np++;
		dp[dn]->xv = (double*) myallocz(sizeof(*px)*np);
		dp[dn]->yv = (double*) myallocz(sizeof(*px)*np);
		dp[dn]->miss = (int*) myallocz(sizeof(*m)*np);
		dp[dn]->yv_str = NULL;
		np--;
		px = dp[dn]->xv;
		py = dp[dn]->yv;
		m = dp[dn]->miss;

	 	for (i=1;i<=np;i++)
		{
			*(px++) = i;
			*(py++) = temp_data[i-1];
			*(m++) = temp_miss[i-1];
		}
		dp[dn]->np = np;
		goto do_next_line;
	}

	for (j=1;j<=nm;j++)
	{
		dn = data_d[j];
		xi = data_x[j];
		yi = data_y[j];
		if (dn>ndata) ndata = dn;
		if (dp[dn]==NULL) {
			dp[dn] = DP_CAST myallocz(sizeof(*dp[dn]));
			if (dp[dn]==NULL) gprint("Memory allocation error, graph dataset \n");
			copy_default(dn);
		}
		np++;
		dp[dn]->xv = (double*) myallocz(sizeof(*px)*np);
		dp[dn]->yv = (double*) myallocz(sizeof(*px)*np);
		dp[dn]->miss = (int*) myallocz(sizeof(*m)*np);
		dp[dn]->yv_str = new vector<string>();
		np--;
		px = dp[dn]->xv;
		py = dp[dn]->yv;
		m = dp[dn]->miss;

		// cout << "nncol = " << nncol << " xi = " << xi << " yi = " << yi << endl;

	 	for ( i = 1 ; i <= np ; i++ )
		{
			int y_offs = i*nncol-nncol+yi-1;
			int x_offs = i*nncol-nncol+xi-1;
			*(px++) = temp_data[x_offs];
			*(py++) = temp_data[y_offs];
			// yv_str is used to use this data set as labels
			dp[dn]->yv_str->push_back(temp_s[y_offs]);
			*(m++) = (temp_miss[x_offs] == true || temp_miss[y_offs] == true);
			dbg gprint("(xym    %f %f %d \n",*(px-1),*(py-1),*(m-1));
		}
		dp[dn]->np = np;
	}

}
goto do_next_line;
/*----------------------------------------------------------------*/
do_extract:
	goto do_next_line;
/*----------------------------------------------------------------*/
do_hscale:
	g_hscale = next_exp;
	goto do_next_line;
/*----------------------------------------------------------------*/
do_letsave:
	nlet++;
	letstr[nlet] = (char*) myalloc(strlen(srclin)+1);
	letline[nlet] = g_get_error_line();
	strcpy(letstr[nlet],srclin);
	if (nlet >= MAX_NUMBER_OF_LET_COMMANDS) gprint("TOO MANY LET Commands in graph! the maximum let commands allowed is %d\n",MAX_NUMBER_OF_LET_COMMANDS);
	goto do_next_line;
/*----------------------------------------------------------------*/
do_nobox:
	g_nobox = true;
	goto do_next_line;
/*----------------------------------------------------------------*/
do_size:
	g_xsize = next_exp;
	g_ysize = next_exp;
	/* ! set up some more constants */
	set_sizelength();
	do_remaining_entries(ct+1);
	goto do_next_line;
/*----------------------------------------------------------------*/
do_scale:
	g_hscale = next_exp;
	g_vscale = next_exp;
	do_remaining_entries(ct+1);
	goto do_next_line;
do_vscale:
	g_vscale = next_exp;
	goto do_next_line;
/*----------------------------------------------------------------*/
do_colormap:
	g_colormap = new GLEColorMap();
	g_colormap->setFunction(tk[++ct]);
	g_colormap->setWidth((int)floor(next_exp+0.5));
	g_colormap->setHeight((int)floor(next_exp+0.5));
	ct++;
	while (ct <= ntk) {
		kw("COLOR") g_colormap->setColor(true);
		kw("INVERT") g_colormap->setInvert(true);
		kw("ZMIN") g_colormap->setZMin(next_exp);
		kw("ZMAX") g_colormap->setZMax(next_exp);
		kw("PALETTE") {
			string tmp;
			next_str_cpp(tmp);
			str_to_uppercase(tmp);
			g_colormap->setPalette(tmp);
		}
		ct++;
	}
	goto do_next_line;
/*----------------------------------------------------------------*/
do_names:
	t = axis_type(tk[1]);
	xx[t].label_off = false;
	if (ntk >= 3 && str_i_equals(tk[2], "FROM") && toupper(tk[3][0]) == 'D') {
		// Support syntax "xnames from d1" to retrieve names from the data set
		xx[t].setNamesDataSet(atoi(tk[3]+1));
	} else {
		ct = 1;
		while (ct<ntk)  {
			next_quote(strbuf);
			xx[t].addName(strbuf);
		}
	}
	goto do_next_line;
/*----------------------------------------------------------------*/
do_places:	/* x2places, or xplaces, or yplaces or y2places */
	t = axis_type(tk[1]);
	xx[t].label_off = false;
	ct = 1;
	while (ct<ntk)  {
		xx[t].addPlace(next_exp);
	}
	goto do_next_line;
/*----------------------------------------------------------------*/
do_noticks:
	t = axis_type(tk[1]);
	ct = 1;
	xx[t].clearNoTicks();
	if (t <= 2) xx[t+2].clearNoTicks();
	while (ct<ntk)  {
		double pos = next_exp;
		xx[t].addNoTick(pos);
		if (t <= 2) xx[t+2].addNoTick(pos);
	}
	goto do_next_line;
/*----------------------------------------------------------------*/
do_main_title:
	/* should change position and size of main title */
	t = GLE_AXIS_T;
	xx[GLE_AXIS_T].off = 0;
	ct = 1;
	next_vquote_cpp(xx[t].title);
	ct = 3;
	xx[t].title_dist = g_fontsz*.7;
	xx[t].title_hei = g_fontsz*g_get_fconst(GLEC_TITLESCALE);
	while (ct<=ntk)  {
	         kw("HEI")     xx[t].title_hei = next_exp;
	    else kw("OFF")     xx[t].title_off = true;
	    else kw("COLOR")   xx[t].title_color = next_color;
	    else kw("FONT")    xx[t].title_font = next_font;
	    else kw("DIST")    xx[t].title_dist = next_exp;
	    else gprint("Expecting TITLE sub command, found {%s} pos %d\n",tk[ct],ct);
	    ct++;
	}
	goto do_next_line;
/*----------------------------------------------------------------*/
do_title:
	t = axis_type(tk[1]);
	ct = 1;
	next_vquote_cpp(xx[t].title);
	ct = 3;
	while (ct<=ntk)  {
	         kw("HEI")     xx[t].title_hei = next_exp;
	    else kw("OFF")     xx[t].title_off = true;
	    else kw("ROT")     xx[t].title_rot = true;
	    else kw("ROTATE")  xx[t].title_rot = true;
	    else kw("COLOR")   xx[t].title_color = next_color;
	    else kw("FONT")    xx[t].title_font = next_font;
	    else kw("DIST")    xx[t].title_dist = next_exp;
	    else gprint("Expecting axis TITLE sub command, found {%s} pos %d\n",tk[ct],ct);
	    ct++;
	}
	goto do_next_line;
/*----------------------------------------------------------------*/
do_datasets:
	d = atoi(tk[1]+1); /* dataset number (right$(k$,2))  (0=dn) */
	iid1 = d;
	iid2 = d;
	if (d==0) iid2 = 100;
	if (d!=0) if (dp[d]==NULL) {
		dp[d] = DP_CAST myallocz(sizeof(*dp[ndata]));
		if (dp[d]==NULL) gprint("Memory allocation error, graph dataset \n");
		copy_default(d);
		if (ndata<d) ndata=d;
	}
	for (d=iid1;d<=iid2;d++) {
		if (dp[d]!=NULL) do_dataset(d);
		iid1=iid1;
	}
	goto do_next_line;
}

bool do_remaining_entries(int ct) {
	int nb_found = 0;
	bool found = true;
	while (found && ct <= ntk) {
		kw("NOBOX") g_nobox = true;
		else kw("BOX") g_nobox = false;
		else kw("CENTER") g_center = true;
		else kw("FULLSIZE") {
			g_vscale = 1;
			g_hscale = 1;
			g_nobox = true;
		}
		else kw("MATH") {
			g_math = true;
			xx[GLE_AXIS_Y].offset = 0.0;
			xx[GLE_AXIS_Y].has_offset = true;
			xx[GLE_AXIS_Y].ticks_both = true;
			xx[GLE_AXIS_X].offset = 0.0;
			xx[GLE_AXIS_X].has_offset = true;
			xx[GLE_AXIS_X].ticks_both = true;
			xx[GLE_AXIS_X2].off = true;
			xx[GLE_AXIS_Y2].off = true;
		}
		else found = false;
		if (found) {
			ct++;
			nb_found++;
		}
	}
	return nb_found > 0;
}

void do_axis(int axis, bool craxis) throw (ParserError) {
	int ct = 2;
	while (ct <= ntk)  {
		kw("BASE") xx[axis].base = next_exp;
		else kw("COLOR") xx[axis].color = (int) next_exp;
		else kw("DSUBTICKS") xx[axis].dsubticks = next_exp;
		else kw("DTICKS") {
			xx[axis].dticks = next_exp;
			if (craxis) xx[axis].label_off = false;
		} else kw("FTICK") {
			xx[axis].ftick = next_exp;
			xx[axis].has_ftick = true;
		}
		else kw("SYMTICKS") xx[axis].ticks_both = true;
		else kw("NOSYMTICKS") xx[axis].ticks_both = false;
		else kw("SHIFT") xx[axis].shift = next_exp;
		else kw("ANGLE") xx[axis].label_angle = next_exp;
		else kw("GRID") xxgrid[axis] = true;
		else kw("NEGATE") {
			xx[axis].negate = true;		/* a.r. */
			if (axis == 1 || axis == 2) {
				data_negate[axis] = true;
			}
		}
		else kw("FONT") xx[axis].label_font = next_font;
		else kw("LOG") xx[axis].log = true;
		else kw("LSTYLE") next_str(xx[axis].side_lstyle);
		else kw("LWIDTH") xx[axis].side_lwidth = next_exp;
		else kw("MIN") {
			xx[axis].min = next_exp;
			xx[axis].minset = true;
		}
		else kw("MAX") {
			xx[axis].max = next_exp;
			xx[axis].maxset = true;
		}
		else kw("OFFSET") {
			double offs = next_exp;
			if (craxis) {
				xx[axis].offset = offs;
				xx[axis].has_offset = true;
				if (!g_math) {
					if (axis == GLE_AXIS_X) xx[GLE_AXIS_X0].off = false;
					if (axis == GLE_AXIS_Y) xx[GLE_AXIS_Y0].off = false;
				}
			}
		}
		else kw("HEI") 	xx[axis].label_hei = next_exp;
		else kw("NOLAST") xx[axis].nolast = true;
		else kw("LAST") xx[axis].nolast = false;
		else kw("FIRST") xx[axis].nofirst = false;
		else kw("NOFIRST") xx[axis].nofirst = true;
		else kw("NSUBTICKS") xx[axis].nsubticks = (int) next_exp;
		else kw("NTICKS") {
			xx[axis].nticks = (int) next_exp;
			if (craxis) xx[axis].label_off = false;
		}
		else kw("ON") xx[axis].off = false;
		else kw("OFF") xx[axis].off = true;
		else kw("FORMAT") next_str_cpp(xx[axis].format);
		else {
			g_throw_parser_error("Expecting AXIS sub command, found '", tk[ct], "'");
		}
		ct++;
	}
}

void do_labels(int axis, bool showerr) throw (ParserError) {
	int ct = 2;
	while (ct <= ntk)  {
		if (*tk[ct]==' ') ct++;
		kw("HEI") xx[axis].label_hei = next_exp;
		else kw("OFF") xx[axis].label_off = true;
		else kw("ON") {
			if (showerr) {
				xx[axis].label_off = false;
				xx[axis].off = false;
			}
		}
		else kw("COLOR") xx[axis].label_color = next_color;
		else kw("FONT") xx[axis].label_font = next_font;
		else kw("DIST") xx[axis].label_dist = next_exp;
		else kw("LOG") {
			ct++;
			kw("OFF") xx[axis].lgset = GLE_AXIS_LOG_OFF;
			else kw("L25B") xx[axis].lgset = GLE_AXIS_LOG_25B;
			else kw("L25") xx[axis].lgset = GLE_AXIS_LOG_25;
			else kw("L1") xx[axis].lgset = GLE_AXIS_LOG_1;
			else if (showerr) g_throw_parser_error("Expecting OFF, L25 or L1, found '", tk[ct], "'");
		}
		else if (showerr) {
			g_throw_parser_error("Expecting LABELS sub command, found '", tk[ct], "'");
		}
		ct++;
	}
}

void do_side(int axis, bool showerr) throw (ParserError) {
	int ct = 2;
	while (ct <= ntk)  {
		if (*tk[ct]==' ') ct++;
		kw("OFF") xx[axis].side_off = true;
		else kw("ON") xx[axis].side_off = false;
		else kw("COLOR") xx[axis].side_color = next_color;
		else kw("LWIDTH") xx[axis].side_lwidth = next_exp;
		else kw("LSTYLE") next_str(xx[axis].side_lstyle);
		else if (showerr) {
			g_throw_parser_error("Expecting SIDE sub command, found '", tk[ct], "'");
		}
		ct++;
	}
}

void do_ticks(int axis, bool showerr) throw (ParserError) {
	int ct = 2;
	while (ct <= ntk)  {
		if (*tk[ct]==' ') ct++;
		kw("LENGTH") xx[axis].ticks_length = next_exp;
		else kw("OFF") {
			xx[axis].ticks_off = true;
			xx[axis].subticks_off = true;
		} else kw("ON") {
			xx[axis].ticks_off = false;
			xx[axis].subticks_off = false;
		} else kw("COLOR") {
			xx[axis].ticks_color = next_color;
			xx[axis].subticks_color = xx[axis].ticks_color;
		} else kw("LWIDTH") {
			xx[axis].ticks_lwidth = next_exp;
		} else kw("LSTYLE") {
			next_str(xx[axis].ticks_lstyle);
		} else if (showerr) {
			g_throw_parser_error("Expecting TICKS sub command, found '", tk[ct], "'");
		}
		ct++;
	}
}

void do_subticks(int axis, bool showerr) throw (ParserError) {
	int ct = 2;
	while (ct <= ntk)  {
		if (*tk[ct]==' ') ct++;
		kw("LENGTH") xx[axis].subticks_length = next_exp;
		else kw("OFF") xx[axis].subticks_off = true;
		else kw("ON") xx[axis].subticks_off = false;
		else kw("COLOR") xx[axis].subticks_color = next_color;
		else kw("LWIDTH") xx[axis].subticks_lwidth = next_exp;
		else kw("LSTYLE") next_str(xx[axis].subticks_lstyle);
		else if (showerr) {
			g_throw_parser_error("Expecting SUBTICKS sub command, found '", tk[ct], "'");
		}
		ct++;
	}
}

void do_axis_part(int axis, bool craxis, int xset) throw (ParserError) {
	switch (xset) {
	  case GLEG_CMD_AXIS:
		do_axis(axis, craxis);
		do_labels(axis, false);
		do_side(axis, false);
		do_ticks(axis, false);
		break;
	  case GLEG_CMD_LABELS:
		do_labels(axis, true);
		break;
	  case GLEG_CMD_SIDE:
		do_side(axis, true);
		break;
	  case GLEG_CMD_TICKS:
		do_ticks(axis, true);
		break;
	  case GLEG_CMD_SUBTICKS:
		do_subticks(axis, true);
		break;
	}
}

void do_axis_part_all(int xset) throw (ParserError) {
	int axis = axis_type(tk[1]);
	do_axis_part(axis, true, xset);
	if (axis == GLE_AXIS_X) {
		do_axis_part(GLE_AXIS_X2, false, xset);
		do_axis_part(GLE_AXIS_X0, false, xset);
		do_axis_part(GLE_AXIS_T, false, xset);
	}
	if (axis == GLE_AXIS_Y) {
		do_axis_part(GLE_AXIS_Y2, false, xset);
		do_axis_part(GLE_AXIS_Y0, false, xset);
	}
}

void do_set_bar_color(const char* tk, bar_struct* bar, int type) {
	/* Use tokenizer that jumps over () pairs to be able to parse CVTRGB(...) */
	int fi = 0;
	string tokstr = (const char*)tk;
	level_char_separator separator(",", "", "(", ")");
	tokenizer<level_char_separator> tokens(tokstr, separator);
	while (tokens.has_more()) {
		int color = pass_color_var(tokens.next_token().c_str());
		switch (type) {
			case BAR_SET_COLOR:   bar->color[fi++] = color; break;
			case BAR_SET_FILL:    bar->fill[fi++] = color; break;
			case BAR_SET_TOP:     bar->top[fi++] = color; break;
			case BAR_SET_SIDE:    bar->side[fi++] = color; break;
			case BAR_SET_PATTERN: bar->pattern[fi++] = color; break;
		}
	}
}

void do_set_bar_style(const char* tk, bar_struct* bar) {
	int fi = 0;
	string tokstr = (const char*)tk;
	level_char_separator separator(",", "", "(", ")");
	tokenizer<level_char_separator> tokens(tokstr, separator);
	while (tokens.has_more()) {
		pass_file_name(tokens.next_token().c_str(), bar->style[fi]);
		str_to_uppercase(bar->style[fi]);
		fi++;
	}
}

void set_sizelength() {
	double ox, oy;

	/* get the origin x,y */
	g_get_xy(&ox,&oy);
	if (g_hscale==0) g_hscale = .7;
	if (g_vscale==0) g_vscale = .7;

	xbl = ox + (g_xsize/2)-(g_xsize*g_hscale/2);
	ybl = oy + (g_ysize/2)-(g_ysize*g_vscale/2);
	xlength = g_xsize*g_hscale;
	ylength = g_ysize*g_vscale;

	// In more recent versions, the size of a graph font depends on the hei parameter
	if (g_get_compatibility() == GLE_COMPAT_35) {
		if (xlength<ylength) g_fontsz = xlength/23;
		else g_fontsz = ylength/23;
	}

	/* set graph globals */
	graph_x1 = xbl;
	graph_y1 = ybl;
	graph_x2 = xbl + xlength;
	graph_y2 = ybl + ylength;
	graph_xmin = wxmin;
	graph_xmax = wxmax;
	graph_ymin = wymin;
	graph_ymax = wymax;
}

void axis_init_length() {
	for (int i = 1; i <= GLE_AXIS_MAX; i++) {
		xx[i].type = i;
		if (xx[i].base==0) xx[i].base = g_fontsz;
		if (axis_horizontal(i)) xx[i].length = xlength;
		else xx[i].length = ylength;
	}
}

void axis_add_grid() {
	for (int i = 1; i <= 2; i++) {
		if (xxgrid[i]) {
			if (i == 1) {
				xx[i].ticks_length = ylength;
				xx[i+2].ticks_length = 0;
				xx[i].subticks_length = ylength;
				xx[i+2].subticks_length = 0;
				if (xx[i].log) {
					xx[i].subticks_off = false;
					xx[i+2].subticks_off = false;
				} else {
					xx[i].subticks_off = true;
					xx[i+2].subticks_off = true;
				}
			}
			if (i==2) {
				xx[i].ticks_length = xlength;
				xx[i+2].ticks_length = 0;
				xx[i].subticks_length = xlength;
				xx[i+2].subticks_length = 0;
				if (xx[i].log) {
					xx[i].subticks_off = false;
					xx[i+2].subticks_off = false;
				} else {
					xx[i].subticks_off = true;
					xx[i+2].subticks_off = true;
				}
			}
		}
	}
}

void axis_add_noticks() {
	// Disable ticks and places on some axis
	for (int i = 1; i < GLE_AXIS_MAX; i++) {
		if (!xx[i].off) {
			if (xx[i].has_offset) {
				// axis has offset:
				// - disable ticks and labels at each intersection
				for (int j = 0; j < 3; j++) {
					int orth = axis_get_orth(i, j);
					if (!xx[orth].off) {
						if (xx[orth].has_offset) {
							xx[i].insertNoTickOrLabel(xx[orth].offset);
						} else {
							if (axis_is_max(orth)) xx[i].insertNoTickOrLabel(xx[i].max);
							else xx[i].insertNoTickOrLabel(xx[i].min);
						}
					}
				}
			} else {
				// axis has no offset:
				// - disable inside ticks (ticks1) for each intersection
				for (int j = 0; j < 3; j++) {
					int orth = axis_get_orth(i, j);
					if (!xx[orth].off) {
						if (xx[orth].has_offset) {
							xx[i].insertNoTick1(xx[orth].offset);
						} else {
							if (axis_is_max(orth)) xx[i].insertNoTick1(xx[i].max);
							else xx[i].insertNoTick1(xx[i].min);
						}
					}
				}
			}
		}
	}
#ifdef AXIS_DEBUG
	cout << endl;
	for (int i = 1; i <= GLE_AXIS_MAX; i++) {
		cout << "AXIS: " << i << endl;
		xx[i].printNoTicks();
		cout << endl;
	}
#endif
}

void draw_axis_pos(int axis, double xpos, double ypos, bool xy, gbox* box) {
	if (xx[axis].has_offset) {
		if (xy) g_move(graph_xgraph(xx[axis].offset), ypos);
		else g_move(xpos, graph_ygraph(xx[axis].offset));
	} else {
		g_move(xpos,ypos);
	}
	draw_axis(&xx[axis], box);
}

void graph_draw_axis(gbox* box) {
	GLEMeasureBox measure;
	measure.measureStart();
	draw_axis_pos(GLE_AXIS_X, xbl, ybl, false, box);
	draw_axis_pos(GLE_AXIS_Y, xbl, ybl, true, box);
	draw_axis_pos(GLE_AXIS_X0, xbl, ybl, false, box);
	draw_axis_pos(GLE_AXIS_Y0, xbl, ybl, true, box);
	draw_axis_pos(GLE_AXIS_X2, xbl, ybl+ylength, false, box);
	draw_axis_pos(GLE_AXIS_Y2, xbl+xlength, ybl, true, box);
	g_set_bounds(xbl+xlength/2, ybl+ylength);
	measure.measureEnd();
	draw_axis_pos(GLE_AXIS_T, xbl, measure.getY2(), true, box);
	// otherwise it does not work if all axis are disabled!
	g_update_bounds_box(box);
}

void draw_graph(KeyInfo* keyinfo) throw (ParserError) {
	gbox box;
	int i,freedata=false;
	double ox,oy;

	g_get_bounds_box(&box);
	done_line = false;

	/* if no size given, take entire drawing */
	if (g_xsize*g_ysize==0) {
		g_xsize = 10; g_ysize = 10;
		g_get_usersize(&g_xsize,&g_ysize);
	}

	/* to handle things like "d1 y2axis" */
	set_dataset_minmax_based_on_axis();

	/* update scale based on let commands if not explicit scale given */
	if (!window_min_max_set()) {
		/* if min max not set, do_let in approximate way first */
		for (i = 1; i <= nlet; i++) {
			g_set_error_line(letline[i]);
			do_let(letstr[i], false);
		}
	}

	/* get scales from data sets */
	set_bar_axis_places();
	get_dataset_ranges();

	preview_big();
	window_set();
	store_window_bounds_to_vars();

	/* get the origin x,y */
	g_get_xy(&ox,&oy);
	g_gsave();
	set_sizelength();
	g_set_hei(g_fontsz);

	/* draw the box */
	if (!g_nobox) {
		g_line(ox+g_xsize,oy);
		g_line(ox+g_xsize,oy+g_ysize);
		g_line(ox,oy+g_ysize);
		g_line(ox,oy);
	}

	/* init axis things */
	vinit_title_axis();
	axis_add_grid();
	axis_add_noticks();
	axis_init_length();

	/* center? */
	if (g_center) {
		gbox dummy;
		g_init_bounds_box(&dummy);
		GLEMeasureBox measure;
		GLEDevice* old_device = g_set_dummy_device();
		measure.measureStart();
		graph_draw_axis(&dummy);
		measure.measureEnd();
		g_restore_device(old_device);
		ox += ox+g_xsize/2.0 - measure.getXMid();
		oy += oy+g_ysize/2.0 - measure.getYMid();
		g_move(ox,oy);
		set_sizelength();
	}

	/* do LETS now */
	for (i = 1; i <= nlet; i++) {
		g_set_error_line(letline[i]);
		do_let(letstr[i], true);
	}

	/* Throw away missing values if NOMISS on datasets */
	gr_thrownomiss();

	if (g_colormap != NULL) {
		g_colormap->setXRange(xx[1].min, xx[1].max);
		g_colormap->setYRange(xx[2].min, xx[2].max);
		g_move(graph_x1, graph_y1);
		g_colormap->draw(xlength, ylength);
		delete g_colormap;
		g_colormap = NULL;
	}

	/* Lets do any filling now. */
	draw_fills();
	g_move(ox,oy);

	/* Draw the bars */
	draw_bars();

	/* Draw the function calls under the axis */
	draw_user_function_calls(true);

	/* Draw the axis */
	g_init_bounds();
	graph_draw_axis(&box);

	/* Draw the function calls on top of the axis */
	draw_user_function_calls(false);

	/* Draw the lines and  markers */
	draw_lines();
	g_move(ox,oy);

	/* Draw the error bars */
	draw_err();
	g_move(ox,oy);

	/* Draw markers, */
	/* (after lines so symbol blanking [white markers] will work) */
	draw_markers();
	g_move(ox,oy);

	graph_freedata(); freedata = true;

	g_move(ox,oy);
	if (!keyinfo->hasHei()) keyinfo->setHei(g_fontsz);
	gdraw_key(keyinfo);
graph_abort:
	if (freedata==false) graph_freedata();
	g_move(ox,oy);
	graph_free(); /* free memory, and initialize counters for next graph */
	g_grestore();
	g_init_bounds();
	g_set_bounds_box(&box);
	return;
}

void graph_init() {
	range_x1 = range_y1 = 1e30; range_x2 = range_y2 = -1e30;
	ndata = 0; nfd = 0; g_nbar = 0;
	data_negate[1] = data_negate[2] = 0;	/* a.r. */
	xx[GLE_AXIS_X0].off = true;
	xx[GLE_AXIS_Y0].off = true;
	xx[GLE_AXIS_T].off = true;
	graph_freebars();
	g_fcalls.clear();
}

void graph_freebars() {
	for (int i = 1; i <= g_nbar; i++) {
		delete br[i];
		br[i] = NULL;
	}
	g_nbar = 0;
}

void graph_freedata() {
	int i,j;
	for (i=1;i<=nfd;i++) {myfree(fd[i]); fd[i] = 0; }
	for (i=0;i<=ndata;i++) {
		if (dp[i]!=NULL) {
			iffree(dp[i]->xv,"b");
			iffree(dp[i]->yv,"c");
			iffree(dp[i]->miss,"d");
			dp[i]->xv = 0;
			dp[i]->yv = 0;
			dp[i]->miss = 0;
		}
	}
}

void graph_free() {
	int i,j;
	for (i=0;i<=ndata;i++) {
		if (dp[i]!=NULL) {
			iffree(dp[i]->key_name,"a");
			iffree(dp[i]->bigfile,"a");
			iffree(dp[i]->xv,"b");
			iffree(dp[i]->yv,"c");
			iffree(dp[i]->miss,"d");
			myfrees(dp[i],"dp");
		}
		dp[i] = 0;
	}
}

void iffree(void *p, char *s) {
	if (p!=NULL) myfrees(p,s);
}

int freedataset(int d) {
	int i,c=0;
	for (i=1;i<=ndata;i++)
	{
		if (dp[i] == NULL) c++;
		else if (dp[i]->xv == NULL) c++;
		if (c==d) return i;
	}
	return ndata + d - c;
}

void copy_default(int dn) {
	memcpy(dp[dn],dp[0],sizeof(*dp[0]));
	dp[dn]->key_name = NULL;
	dp[dn]->bigfile = NULL;
	dp[dn]->xv = NULL;
	dp[dn]->yv = NULL;
	dp[dn]->miss = NULL;
	dp[dn]->axisscale = false;
	dp[dn]->inverted = false;
}
