
#define GLE_MAIN_CPP

#include "all.h"
#include "gprint.h"
#include "SourceLine.h"
#include "tokens/Tokenizer.h"
#include "core.h"
#include "mem_limits.h"
#include "file_io.h"
#include "cutils.h"
#include "cmdline.h"
#include "texinterface.h"
#include "config.h"
#include "drawit.h"

#ifdef __WIN32__
	#include <io.h>
#endif

/*------------ GLOBAL VARIABLES --------------*/
//
// -- these globals are needed. they are called in other source files using extern...
//
int MAX_VECTOR=MAXIMUM_PS_VECTOR;
int gle_debug;
bool control_d = true;
extern int trace_on;
bool BLACKANDWHITE = false;
bool IS_INSTALL = false;
bool GS_PREVIEW = false;

void do_find_deps(CmdLineObj& cmdline);
void gle_as_a_calculator(vector<string>* exprs);

extern string GLE_TOP_DIR;
extern string GLE_BIN_DIR;
string GLE_WORKING_DIR;

_GLESource GLESource; // need global access

class GLEOptions {
public:
	bool ASK_DEBUG;
};

/* Better to put all global options in one class? */
GLEOptions g_Options;
CmdLineObj g_CmdLine;
ConfigCollection g_Config;

void load_one_file(const char* name, CmdLineObj& cmdline, size_t* exit_code);
void init_option_args(CmdLineObj& cmdline);
void do_gen_inittex(CmdLineObj& cmdline, GLEOptions& options);
void process_option_args(CmdLineObj& cmdline, GLEOptions& options);
void do_run_other_version(ConfigCollection& coll, int argc, char **argv);
void gle_do_socket(const string& commands);
void gle_cleanup();
void tex_term();
void do_show_info();
void get_out_name(const char* name, CmdLineObj& cmdline, string& outname);

int main(int argc, char **argv) {
	g_init();

	/* Init and process command line arguments */
	init_config(&g_Config);
	init_option_args(g_CmdLine);
	do_load_config(g_CmdLine, g_Config);
	do_run_other_version(g_Config, argc, argv);
	g_CmdLine.parse(argc, argv);

	/* Error in command line arguments? */
	if (g_CmdLine.hasError()) {
		exit(-1);
	}
	/* Has calculator option? */
	if (g_CmdLine.hasOption(GLE_OPT_CALC)) {
		if (g_CmdLine.getNbMainArgs() == 0) {
			gle_as_a_calculator(NULL);
		} else {
			gle_as_a_calculator(g_CmdLine.getMainArgs());
		}
		exit(0);
	}

	/* Should regenerate inittex.ini? */
	do_gen_inittex(g_CmdLine, g_Options);

	/* Find dependencies */
	do_find_deps(g_CmdLine);

	/* Get working directory */
	GLEGetCrDir(&GLE_WORKING_DIR);

	/* This is used by the GUI */
	if (g_CmdLine.hasOption(GLE_OPT_INFO)) {
		do_show_info();
	}

	if (g_CmdLine.getNbMainArgs() == 0 || g_CmdLine.hasOption(GLE_OPT_HELP)) {
		cerr << "GLE version " << __GLEVN__ << endl;
		cerr << "Usage: gle [options] filename.gle" << endl;
		cerr << "More information: gle " << g_CmdLine.getOptionPrefix() << "help" << endl;
		if (g_CmdLine.hasOption(GLE_OPT_HELP)) g_CmdLine.showHelp(GLE_OPT_HELP);
		exit(0);
	}

	/* Process optional arguments */
	process_option_args(g_CmdLine, g_Options);

	/* iterate over "file" arguments */
	size_t exit_code = 0;
	for (int i = 0; i < g_CmdLine.getNbMainArgs(); i++) {
	  // the standard CMD shell didn't expand the wildcards
#ifdef __WIN32__
		//
		// -- globbing and wildcards
		//
		_finddata_t fileinfo;
		intptr_t File = _findfirst((char*)g_CmdLine.getMainArg(i).c_str(), &fileinfo);
		if (File != -1){
			do {
				load_one_file((char*)fileinfo.name, g_CmdLine, &exit_code);
			} while(!_findnext(File,&fileinfo));
			_findclose(File);
		} else {
			exit_code = 1;
			cout << "Can't open file: " << g_CmdLine.getMainArg(i) << endl;
		}
#else
#if defined(__OS2__) && defined(__EMX__)
		char **list;
		list = _fnexplode((char*)g_CmdLine.getMainArg(i).c_str());
		if ( list == NULL ) {
			load_one_file( g_CmdLine.getMainArg(i).c_str(), g_CmdLine, &exit_code);
		} else {
			for ( int i = 0; list[i] != NULL; i++ ) {
				load_one_file(list[i], g_CmdLine, &exit_code);
			}
		}
		_fnexplodefree(list);
#else
		// on Unix the shell expands the wildcards for us
		load_one_file(g_CmdLine.getMainArg(i).c_str(), g_CmdLine, &exit_code);
#endif
#endif
	}
	// will exit with the number of files it could NOT open
	gle_cleanup();
	if (exit_code != 0 || g_has_console_output()) {
		do_wait_for_enter_exit(exit_code);
	} else {
		exit(0);
	}
}

void gle_cleanup() {
	tex_term();
}

void do_run_other_version(ConfigCollection& coll, int argc, char **argv) {
	// Scan command line for -v option
	// Note: can not use cmdline here because older versions might have different options
	// which would result in an error message
	string version = "";
	for (int i = 1; i < argc-1; i++) {
		if (cmdline_is_option(argv[i], "v")) {
			version = argv[i+1];
		}
	}
	if (version != "") {
		ConfigSection* gle = coll.getSection(GLE_CONFIG_GLE);
		CmdLineArgSPairList* installs = (CmdLineArgSPairList*)gle->getOption(GLE_CONFIG_GLE_INSTALL)->getArg(0);
		const string* path = installs->lookup(version);
		if (path != NULL) {
			GLESetGLETop(*path);
			stringstream torun;
			torun << "\"" << (*path) << "\"";
			for (int i = 1; i < argc; i++) {
				string arg = argv[i];
				str_remove_quote(arg);
				if (cmdline_is_option(arg.c_str(), "v")) i++;
				else torun << " \"" << arg << "\"";
			}
			int res = GLESystem(torun.str());
			if (res != GLE_SYSTEM_OK) {
				cerr << "Error while running: " << (*path) << endl;
			}
		} else {
			cerr << "Don't know path for version: '" << version << "'" << endl;
		}
		exit(0);
	}
}

void do_show_info() {
	cout << "GLE version: " << __GLEVN__ << endl;
#if defined(__TIME__) && defined(__DATE__)
	string bdate = __DATE__;
	bdate += " "; bdate += __TIME__;
	str_replace_all(bdate, "  ", " ");
	cout << "Build date:  " << bdate << endl;
#endif
	cout << "GLE_TOP:     " << GLE_TOP_DIR << endl;
	cout << "GLE_BIN:     " << GLE_BIN_DIR << endl;

	/* Location of GhostScript */
	string gs_dir;
	ConfigSection* tools = g_Config.getSection(GLE_CONFIG_TOOLS);
	const string& gs_cmd = ((CmdLineArgString*)tools->getOptionValue(GLE_TOOL_GHOSTSCRIPT_CMD))->getValue();
	GetDirName(gs_cmd, gs_dir);
	if (gs_dir == "") gs_dir = "?";
	cout << "GhostScript: " << gs_dir << endl;

	do_wait_for_enter_exit(0);
}

void do_gen_inittex(CmdLineObj& cmdline, GLEOptions& options) {
	// Generate inittex.ini from init.tex
	if (cmdline.hasOption(GLE_OPT_MKINITTEX)) {
		IS_INSTALL = 1;
		size_t exit_code;
		string inittex = GLE_TOP_DIR + DIR_SEP + "init.tex";
		if (text_load(inittex, GLESource)) {
			string inittex_ini = GLE_TOP_DIR + DIR_SEP + "inittex.ini";
			TryDeleteFile(inittex_ini);
			g_select_device(GLE_DEVICE_DUMMY);
			DrawIt("output.tmp", GLESource, &cmdline);
		} else {
			cout << "<< can't read '" << inittex << "'" << endl;
		}
		exit(0);
	}
}

void process_option_args(CmdLineObj& cmdline, GLEOptions& options) {
	CmdLineArgSet* compat = (CmdLineArgSet*)cmdline.getOption(GLE_OPT_COMPAT)->getArg(0);
	g_set_compatibility(compat->getFirstValue());
	trace_on = cmdline.hasOption(GLE_OPT_TRACE);
	options.ASK_DEBUG = cmdline.hasOption(GLE_OPT_DEBUG);
	control_d = cmdline.hasOption(GLE_OPT_NO_CTRL_D);
	if (cmdline.hasOption(GLE_OPT_NO_MAXPATH)) {
		/* Huh - what does this do? */
		MAX_VECTOR = 10*MAXIMUM_PS_VECTOR;
	}
	BLACKANDWHITE = cmdline.hasOption(GLE_OPT_NO_COLOR);
	if (cmdline.hasOption(GLE_OPT_BBTWEAK)) g_psbbtweak();
	GS_PREVIEW = cmdline.hasOption(GLE_OPT_GSPREVIEW);
	// .ps output implies full page
	CmdLineArgSet* device = (CmdLineArgSet*)cmdline.getOption(GLE_OPT_DEVICE)->getArg(0);
	if (device->hasValue(GLE_DEVICE_PS)) {
		cmdline.setHasOption(GLE_OPT_FULL_PAGE, true);
	}
	// Auto detect extra arguments (after .GLE / .TEX files)
	// Note that extra arguments can also be supplied by means of -args
	if (cmdline.getMainArgSepPos() == -1) {
		int nargs = cmdline.getNbMainArgs();
		for (int i = 0; i < nargs; i++) {
			const string& arg = cmdline.getMainArg(i);
			if (!str_i_ends_with(arg, ".GLE")) {
				if (i != 0) cmdline.setMainArgSepPos(i);
				break;
			}
		}
	}
}

void init_option_args(CmdLineObj& cmdline) {
	CmdLineOption* option;
	CmdLineArgString* strarg;
	CmdLineArgInt* intarg;
	CmdLineArgSet* setarg;
	cmdline.setMainArgType("file name");
	option = new CmdLineOption("help", "h", "?");
	option->setHelp("Shows help about command line options");
	strarg = new CmdLineArgString("option");
	strarg->setHelp("show specific help about 'option'");
	strarg->setCardLimits(0, 1);
	option->addArg(strarg);
	cmdline.addOption(option, GLE_OPT_HELP);
	option = new CmdLineOption("device", "d");
	option->setHelp("Select output device(s)");
	option->setMinNbArgs(1);
	setarg = new CmdLineArgSet("device-names");
	setarg->setHelp("set output device(s)");
	setarg->setMinCard(1);
	/* Order of values is important! (must match GLE_DEVICE_ constants in core.h) */
	setarg->addPossibleValue("eps");
	setarg->addPossibleValue("ps");
	setarg->addPossibleValue("pdf");
	setarg->addPossibleValue("svg");
	setarg->addPossibleValue("jpg");
	setarg->addPossibleValue("png");
	setarg->addPossibleValue("x11");
#ifndef HAVE_X11
	setarg->setUnsupportedValue(GLE_DEVICE_X11);
#endif
	setarg->addDefaultValue(GLE_DEVICE_EPS);
	option->addArg(setarg);
	cmdline.addOption(option, GLE_OPT_DEVICE);
	option = new CmdLineOption("fullpage");
	option->setHelp("Full page output");
	cmdline.addOption(option, GLE_OPT_FULL_PAGE);
	option = new CmdLineOption("nocolor", "bw");
	option->setHelp("Grayscale output");
	cmdline.addOption(option, GLE_OPT_NO_COLOR);
	option = new CmdLineOption("noctrl-d");
	option->setHelp("Do not include CTRL-D in PostScript output");
	cmdline.addOption(option, GLE_OPT_NO_CTRL_D);
	option = new CmdLineOption("dpi");
	option->setHelp("Set DPI value for bitmap import/export and .pdf output");
	intarg = new CmdLineArgInt("dpi");
	intarg->setHelp("set DPI value to 'dpi'");
	intarg->setCardLimits(0, 1);
	intarg->setDefault(72);
	option->addArg(intarg);
	cmdline.addOption(option, GLE_OPT_DPI);
	option = new CmdLineOption("tex");
	option->setHelp("The script includes TeX expressions");
	cmdline.addOption(option, GLE_OPT_TEX);
	option = new CmdLineOption("nopdftex");
	option->setHelp("Disable PdfLaTeX for .pdf creation");
	cmdline.addOption(option, GLE_OPT_NO_PDFTEX);
	option = new CmdLineOption("inc");
	option->setHelp("Create .inc file for TeX code");
	cmdline.addOption(option, GLE_OPT_CREATE_INC);
	option = new CmdLineOption("texincprefix");
	option->setHelp("Adds given subdirectory to path in .inc file");
	strarg = new CmdLineArgString("path");
	strarg->setHelp("adds 'path' to path in .inc file");
	strarg->setCardLimits(1, 1);
	option->addArg(strarg);
	cmdline.addOption(option, GLE_OPT_TEXINCPREF);
#ifdef __WIN32__
	option = new CmdLineOption("finddeps");
	option->setHelp("Automatically finds dependencies");
	strarg = new CmdLineArgString("path");
	strarg->setHelp("find dependencies in 'path'");
	strarg->setCardLimits(0, 1);
	// Default value should also be set if no value is given?
	option->addArg(strarg);
	cmdline.addOption(option, GLE_OPT_FINDDEPS);
#endif
	option = new CmdLineOption("preview", "p");
	option->setHelp("Preview output in GLE preview window");
	cmdline.addOption(option, GLE_OPT_PREVIEW);
#ifdef ENABLE_GS_PREVIEW
	option = new CmdLineOption("gs");
	option->setHelp("Call ghostscript for previewing");
	cmdline.addOption(option, GLE_OPT_GSPREVIEW);
#endif
	option = new CmdLineOption("calc", "c");
	option->setHelp("Use GLE as a calculator");
	cmdline.addOption(option, GLE_OPT_CALC);
	option = new CmdLineOption("output", "o");
	option->setHelp("Specifies name for output file");
	strarg = new CmdLineArgString("name");
	strarg->setHelp("writes output to file 'name'");
	strarg->setCardLimits(1, 1);
	option->addArg(strarg);
	cmdline.addOption(option, GLE_OPT_OUTPUT);
	option = new CmdLineOption("cmode", "cm");
	option->setHelp("Select GLE compatibility mode");
	setarg = new CmdLineArgSet("cmode");
	setarg->setMinCard(1);
	setarg->setMaxCard(1);
	/* Order of values is important! (must match GLE_COMPAT_ constants in core.h) */
	setarg->addPossibleValue("3.0");
	setarg->addPossibleValue("4.0");
	setarg->addDefaultValue(GLE_COMPAT_40);
	option->addArg(setarg);
	cmdline.addOption(option, GLE_OPT_COMPAT);
	option = new CmdLineOption("version", "v");
	option->setHelp("Select GLE version to run");
	setarg = new CmdLineArgSet("version");
	setarg->setMinCard(1);
	setarg->setMaxCard(1);
	option->addArg(setarg);
	cmdline.addOption(option, GLE_OPT_VERSION);
	option = new CmdLineOption("trace");
	option->setHelp("Trace GLE");
	option->setExpert(true);
	cmdline.addOption(option, GLE_OPT_TRACE);
	option = new CmdLineOption("debug");
	option->setHelp("Debug GLE");
	option->setExpert(true);
	cmdline.addOption(option, GLE_OPT_DEBUG);
	option = new CmdLineOption("nomaxpath");
	option->setHelp("No max path");
	option->setExpert(true);
	cmdline.addOption(option, GLE_OPT_NO_MAXPATH);
	option = new CmdLineOption("mkinittex");
	option->setHelp("Create inittex.ini from init.tex");
	option->setExpert(true);
	cmdline.addOption(option, GLE_OPT_MKINITTEX);
	option = new CmdLineOption("info");
	option->setHelp("Output certain information");
	option->setExpert(true);
	cmdline.addOption(option, GLE_OPT_INFO);
	option = new CmdLineOption("pause");
	option->setHelp("Pause if output has been generated");
	option->setExpert(true);
	cmdline.addOption(option, GLE_OPT_PAUSE);
	cmdline.addMainArgSep("args");
	cmdline.addMainArgSep("a");
	cmdline.initOptions();
}

void gle_preview_file(const char* name, const string& out_name, CmdLineObj& cmdline) {
	string glefile, epsfile, directory, temp;
	stringstream commands;
	SplitFileName(name, temp, glefile);
	GetMainName(out_name, epsfile);
	epsfile += ".eps";
	GLEGetCrDirWin32(&directory);
	commands << "dir: \"" << directory << "\"" << endl;
	commands << "glefile: \"" << glefile << "\"" << endl;
	commands << "epsfile: \"" << epsfile << "\"" << endl;
	if (cmdline.hasOption(GLE_OPT_DPI)) {
		int dpi = ((CmdLineArgInt*)cmdline.getOption(GLE_OPT_DPI)->getArg(0))->getValue();
		commands << "dpi: \"" << dpi << "\"" << endl;
	}
	// cout << "Sending: " << commands.str() << endl;
	int result = GLESendSocket(commands.str());
	printf("\n");
	if (result == -3) {
		printf("Note: GLE is trying to launch QGLE, the GLE preview application\n");
#ifdef __WIN32__
		string qgle = "\"" + GLE_BIN_DIR + "qgle.exe\"";
#else
		string qgle = "qgle";
#endif
		int sysres = GLESystem(qgle, false);
		if (sysres == GLE_SYSTEM_OK) {
			bool done = false;
			while (!done) {
				GLESleep(1000);
				result = GLESendSocket(commands.str());
				if (result != -3) {
					done = true;
				}
			}
		} else {
			printf("Error: failed to start QGLE: '%s'\n", qgle.c_str());
			result = 0;
		}
	}
	if (result != 0) {
		printf("Error: could not connect to GLE preview application, code = %d\n", result);
	}
}

bool has_pdflatex(CmdLineObj& cmdline) {
	if (cmdline.hasOption(GLE_OPT_NO_PDFTEX)) {
		return false;
	}
	ConfigSection* tex = g_Config.getSection(GLE_CONFIG_TEX);
	CmdLineArgSet* texsys =	(CmdLineArgSet*)tex->getOptionValue(GLE_TEX_SYSTEM);
	if (texsys->hasValue(GLE_TEX_SYSTEM_VTEX)) {
		return false;
	} else {
		return true;
	}
}

bool has_eps_based_device(CmdLineArgSet* device, CmdLineObj& cmdline) {
	if (cmdline.hasOption(GLE_OPT_TEX)) return true;
	if (device->hasValue(GLE_DEVICE_EPS)) return true;
	if (device->hasValue(GLE_DEVICE_PDF)) return true;
	if (device->hasValue(GLE_DEVICE_JPEG)) return true;
	if (device->hasValue(GLE_DEVICE_PNG)) return true;
	return false;
}

bool has_tex_eps_based_device_not_inc(CmdLineArgSet* device, CmdLineObj& cmdline) {
	if (!cmdline.hasOption(GLE_OPT_CREATE_INC)) {
		if (device->hasValue(GLE_DEVICE_EPS)) return true;
		if (device->hasValue(GLE_DEVICE_PDF) && !has_pdflatex(cmdline)) return true;
	}
	if (device->hasValue(GLE_DEVICE_JPEG)) return true;
	if (device->hasValue(GLE_DEVICE_PNG)) return true;
	return false;
}

bool requires_tex(CmdLineArgSet* device, CmdLineObj& cmdline) {
	if (!cmdline.hasOption(GLE_OPT_CREATE_INC)) {
		if (device->hasValue(GLE_DEVICE_EPS)) return true;
		if (device->hasValue(GLE_DEVICE_PDF)) return true;
	}
	if (device->hasValue(GLE_DEVICE_PS)) return true;
	if (device->hasValue(GLE_DEVICE_JPEG)) return true;
	if (device->hasValue(GLE_DEVICE_PNG)) return true;
	return false;
}

bool requires_temp_eps(CmdLineArgSet* device, CmdLineObj& cmdline) {
	if (!cmdline.hasOption(GLE_OPT_CREATE_INC)) {
		if (device->hasValue(GLE_DEVICE_PDF)) return true;
	}
	if (device->hasValue(GLE_DEVICE_JPEG)) return true;
	if (device->hasValue(GLE_DEVICE_PNG)) return true;
	return false;
}

bool has_bitmap_or_pdf_device(CmdLineArgSet* device) {
	if (device->hasValue(GLE_DEVICE_JPEG)) return true;
	if (device->hasValue(GLE_DEVICE_PNG)) return true;
	if (device->hasValue(GLE_DEVICE_PDF)) return true;
	return false;
}

bool is_bitmap_device(int device) {
	if (device == GLE_DEVICE_JPEG) return true;
	if (device == GLE_DEVICE_PNG) return true;
	return false;
}

bool process_one_file_eps(const string& in_name, const string& out_name, CmdLineObj& cmdline) {
	CmdLineArgSet* device = (CmdLineArgSet*)cmdline.getOption(GLE_OPT_DEVICE)->getArg(0);
	g_select_device(GLE_DEVICE_EPS);
	/* In some cases two passes are required to measure size of TeX objects */
	int done = 1;
	TeXInterface* interface = TeXInterface::getInstance();
	interface->initialize(in_name, out_name);
	do {
		interface->reset();
		DrawIt(out_name, GLESource, &cmdline);
		done = interface->tryCreateHash();
	} while (done == 1);
	interface->checkObjectDimensions();
	if (cmdline.hasOption(GLE_OPT_CREATE_INC)) {
		CmdLineArgString* addinc = (CmdLineArgString*)cmdline.getOption(GLE_OPT_TEXINCPREF)->getArg(0);
		interface->createInc(addinc->getValue());
	}
	if (interface->hasObjects() && requires_tex(device, cmdline)) {
		bool use_geom = false;
		bool has_pdftex = has_pdflatex(cmdline);
		bool create_inc = cmdline.hasOption(GLE_OPT_CREATE_INC);
		if (device->hasValue(GLE_DEVICE_PS)) use_geom = true;
		if (has_pdftex && !create_inc) use_geom = true;
		interface->createTeX(use_geom);
	}
	return interface->hasObjects();
}

void get_out_name(const char* name, CmdLineObj& cmdline, string& outname) {
	if (cmdline.hasOption(GLE_OPT_OUTPUT)) {
		CmdLineArgString* o_file = (CmdLineArgString*)cmdline.getOption(GLE_OPT_OUTPUT)->getArg(0);
		GetMainName(o_file->getValue(), outname);
	} else {
		outname = name;
	}
}

void load_one_file(const char* name, CmdLineObj& cmdline, size_t* exit_code) {
	string out_name;
	string in_name = name;
	string dir_name = GLE_WORKING_DIR;
	if (IsAbsPath(in_name)) {
		SplitFileName(in_name, dir_name, in_name);
	}
	GLEChDir(dir_name);
	get_out_name(name, cmdline, out_name);
	if (text_load(in_name.c_str(), GLESource)) {
		g_set_console_output(false);
		g_message_first_newline(true);
		if (cmdline.hasOption(GLE_OPT_DEBUG)){
			printf("Debug options 16=do_pcode, 8=pass 4=polish, 2=eval ");
			printf("Debug "); scanf("%d",&gle_debug);
			printf("Trace "); scanf("%d",&trace_on);
		}
		bool create_inc = cmdline.hasOption(GLE_OPT_CREATE_INC);
		CmdLineArgSet* device = (CmdLineArgSet*)cmdline.getOption(GLE_OPT_DEVICE)->getArg(0);
		if (has_eps_based_device(device, cmdline)) {
			bool has_tex = process_one_file_eps(in_name, out_name, cmdline);
			/* Return if there were errors */
			if (get_nb_errors() > 0) {
				(*exit_code)++;
				return;
			}
			/* Get device information */
			PSGLEDevice* psdev = (PSGLEDevice*)g_get_device_ptr();
			int bb_wd = psdev->getBBWidth();
			int bb_hi = psdev->getBBHeight();
			int dpi = ((CmdLineArgInt*)cmdline.getOption(GLE_OPT_DPI)->getArg(0))->getValue();
			/* File has TeX code -> move it and create again */
			if (has_tex) {
				string main_name, from_name, to_name;
				/* LaTeX does not like to run with absolute paths */
				string out_name_np = out_name;
				if (IsAbsPath(out_name)) {
					string dir_name2;
					SplitFileName(out_name, dir_name2, out_name_np);
					GLEChDir(dir_name2);
				}
				GetMainName(out_name_np, main_name);
				from_name = main_name + ".eps";
				to_name = main_name + "_inc.eps";
				GLECopyFile(from_name, to_name);
				bool has_tex_file = false;
				if (has_tex_eps_based_device_not_inc(device, cmdline)) {
					has_tex_file = true;
					create_tex_eps_file(out_name_np);
				}
				if (device->hasValue(GLE_DEVICE_PS)) {
					create_ps_file_latex_dvips(out_name_np);
				}
				if (device->hasValue(GLE_DEVICE_PDF)) {
					bool has_pdftex = has_pdflatex(cmdline);
					if (has_pdftex || create_inc) {
						create_pdf_file(to_name, dpi, bb_wd, bb_hi, false);
					}
					if (!create_inc) {
						has_tex_file = true;
						if (has_pdftex) {
							create_pdf_file_pdflatex(out_name_np);
						} else {
							create_pdf_file(out_name_np, dpi, bb_wd, bb_hi, true);
						}
					}
				}
				if (has_tex_file) DeleteFileWithExt(out_name, ".tex");
				GLEChDir(dir_name);
			} else {
				if (device->hasValue(GLE_DEVICE_PDF)) {
					create_pdf_file(out_name, dpi, bb_wd, bb_hi, false);
				}
			}
			/* Create bitmap outputs */
			bool is_bw = cmdline.hasOption(GLE_OPT_NO_COLOR);
			for (int i = 0; i < device->getNbValues(); i++) {
				if (is_bitmap_device(i) && device->hasValue(i)) {
					create_bitmap_file(out_name, i, dpi, bb_wd, bb_hi, is_bw, has_tex);
				}
			}
			/* Remove .eps if not required anymore */
			if (!device->hasValue(GLE_DEVICE_EPS)) {
				/* delete original .eps file */
				DeleteFileWithExt(out_name, ".eps");
			}
			if (has_tex) {
				if (create_inc) {
					/* delete original .eps file */
					DeleteFileWithExt(out_name, ".eps");
					if (!device->hasValue(GLE_DEVICE_EPS)) {
						DeleteFileWithExt(out_name, "_inc.eps");
					}
				} else {
					/* Remove _inc.{eps,pdf} if not required anymore */
					if (requires_tex(device, cmdline)) {
						DeleteFileWithExt(out_name, "_inc.eps");
					}
					if (device->hasValue(GLE_DEVICE_PDF)) {
						DeleteFileWithExt(out_name, "_inc.pdf");
					}
				}
				/* Remove _temp.eps? (used for bounding box adjustment) */
				if (requires_temp_eps(device, cmdline)) {
					DeleteFileWithExt(out_name, "_temp.eps");
				}
			}
		}
		if (device->hasValue(GLE_DEVICE_PS) && !cmdline.hasOption(GLE_OPT_TEX)) {
			g_select_device(GLE_DEVICE_PS);
			DrawIt(out_name, GLESource, &cmdline);
			if (TeXInterface::getInstance()->hasObjects()) {
				cout << ">> To include LaTeX expressions, use \"gle -tex -d ps file.gle\"" << endl;
			}
		}
		if (device->hasValue(GLE_DEVICE_SVG)) {
			g_select_device(GLE_DEVICE_SVG);
			DrawIt(out_name, GLESource, &cmdline);
		}
		if (device->hasValue(GLE_DEVICE_X11)) {
			g_select_device(GLE_DEVICE_X11);
			DrawIt(out_name, GLESource, &cmdline);
		}
		if (cmdline.hasOption(GLE_OPT_PREVIEW)) {
			gle_preview_file(in_name.c_str(), out_name, cmdline);
		}
		GLESource.clear();
	} else {
		cout << "Can't open file: " << name << endl;
		(*exit_code)++;
	}
}
