#include <math.h>
#include <time.h>
#include <string>
#include <sstream>
#include <iostream>
#include <vector>
#include <list>
#include <map>
#include <fstream>
#include <algorithm>

#include "basicconf.h"
#include <unistd.h>
#include <sys/stat.h>

#ifdef __WIN32__
	#include <sys/types.h>
	#include <sys/stat.h>
	#include <stdlib.h>
	#include <process.h>
	#include <windows.h>
	#include <winsock.h>
#endif
#ifdef __CYGWIN__
	#include <windows.h>
#endif
#ifdef __UNIX__
	#include <sys/types.h>
	#ifdef HAVE_SOCKETS
		#include <sys/socket.h>
		#include <netinet/in.h>
		#include <arpa/inet.h>
		#include <unistd.h>
		typedef int SOCKET;
		#define SOCKET_ERROR -1
	#endif
#endif
#if defined(__OS2__) && defined(__EMX__)
	#include <sys/types.h>
	#ifdef HAVE_SOCKETS
		#include <sys/socket.h>
		#include <netinet/in.h>
		#include <arpa/inet.h>
		#include <unistd.h>
		typedef int SOCKET;
		#define SOCKET_ERROR -1
	#endif
#endif

using namespace std;

#include "tokens/stokenizer.h"
#include "errno.h"
#include "stdio.h"
#include "stdlib.h"
#include "file_io.h"
#include "cutils.h"

#ifdef __UNIX__
	/* Cygwin too */
	string PATH_SEP = ":";
	string DIR_SEP = "/";
#endif

#ifdef __WIN32__
	string PATH_SEP = ";";
	string DIR_SEP = "\\";
#endif

#if defined(__OS2__) && defined(__EMX__)
	string PATH_SEP = ";";
	string DIR_SEP = "/";
#endif

string GLE_TOP_DIR;
string GLE_BIN_DIR;

//
// -- directory finding routines
//

void FillIncludePaths(vector<string> &IP) {
	// fills containor with paths to search for
	// file.  This is called if it can't find file
	// in current directory
	//
	//  Paths searched are:
	// %GLE_TOP%
	// %GLE_TOP%/lib
	// %GLE_USRLIB%
	//  note that GLE_USERLIB can contain multiple
	// directories separated by ; (PC) or : (unix)
	// as defined in PATH_SEP
	//
	//printf("string paths for GLE_PATH\n");
	string paths = GLE_TOP_DIR + DIR_SEP + "lib";
	IP.push_back(paths);
	//printf("paths for GLE_USRLIB\n");
  	const char* usr_lib = getenv("GLE_USRLIB");
	if (usr_lib != NULL) {
		paths = getenv("GLE_USRLIB");
		// spilt on path sep ; for pc : for unix
		//printf("typedef boost::tokenizer ...\n");
		typedef tokenizer<char_separator> Tok;
		//printf("boost:: char_separator ...\n");
		char_separator sep(PATH_SEP.c_str());
		Tok tok(paths, sep);
		//printf("for(Tok::iterator ...\n");
		while (tok.has_more()) {
			IP.push_back(tok.next_token());
		}
	}
}

// Get extension of filename and convert it to lower case
void GetExtension(const string& fname, string& ext) {
	string::size_type i = fname.rfind('.');
	if (i != string::npos) {
		ext = fname.substr(i+1);
		string::size_type len = ext.length();
		for (i = 0; i < len; i++) {
			int ch = ext.at(i);
			if (ch >= 'A' && ch <= 'Z') {
				ch += 'a' - 'A';
				ext.at(i) = ch;
			}
		}
	} else {
		ext = "";
	}
}

void StripDirSep(string& fname) {
	if (str_i_ends_with(fname, DIR_SEP.c_str())) {
		int nb = DIR_SEP.length();
		fname.erase(fname.length()-nb, nb);
	}
}

void AddDirSep(string& fname) {
	if (!str_i_ends_with(fname, DIR_SEP.c_str())) {
		fname += DIR_SEP;
	}
}

bool IsAbsPath(string& path) {
	if (path.length() >= 1) {
		if (path[0] == '/') {
			// Unix style absolute path
			return true;
		}
		if (path.length() >= 3) {
			if (path[1] == ':' && (path[2] == '/' || path[2] == '\\')) {
				// Windows style absolute path
				return true;
			}
		}
	}
	return false;
}

// Get file name without extension
void GetMainName(const string& fname, string& name) {
	string::size_type i = fname.rfind('.');
	if (i != string::npos) {
		name = fname.substr(0, i);
	} else {
		name = fname;
	}
}

// Get path of file name (assume last part after DIR_SEP is file name)
void GetDirName(const string& fname, string& dir) {
	string::size_type i = fname.rfind(DIR_SEP);
	if (i != string::npos) {
		dir = fname.substr(0, i+1);
		AddDirSep(dir);
	} else {
		dir = "";
	}
}

// Split file name in path (dir) and file name (name)
void SplitFileName(const string& fname, string& dir, string& name) {
	string::size_type i = fname.rfind(DIR_SEP);
	if (i != string::npos) {
		dir = fname.substr(0, i+1);
		name = fname.substr(i+1);
		AddDirSep(dir);
	} else {
		name = fname;
		dir = "";
	}
}

// Split file name in path (dir) and file name (name) and return only file name
void SplitFileNameNoDir(const string& fname, string& name) {
	string::size_type i = fname.rfind(DIR_SEP);
	if (i != string::npos) {
		name = fname.substr(i+1);
	} else {
		name = fname;
	}
}

void StripPathComponents(string* fname, int nb) {
	while (nb > 0) {
		string::size_type i = fname->rfind(DIR_SEP);
		if (i != string::npos) {
			*fname = fname->substr(0, i);
		} else {
			break;
		}
		nb--;
	}
}

bool GLEMoveFile(const string& from, const string& to) {
#ifdef __WIN32__
#else
	if (rename(from.c_str(), to.c_str()) == -1) return false;
#endif
	return true;
}

bool GLECopyFile(const string& from, const string& to) {
	ofstream out(to.c_str());
	ifstream strm(from.c_str());
	while (!strm.eof()) {
		char ch;
		strm.read(&ch, 1);
		out << ch;
	}
	out.close();
	strm.close();
	return true;
}

// Return current directory
bool GLEGetCrDir(string* name) {
#ifdef __WIN32__
	TCHAR buffer[1024];
	if (GetCurrentDirectory(1024, buffer) != 0) {
		*name = buffer;
		return true;
	} else {
		return false;
	}
#else
	char buffer[1024];
	if (getcwd(buffer, 1024) == NULL) {
		return false;
	} else {
		*name = buffer;
		return true;
	}
#endif
}

// Return current directory
// -> Used for compatibility with QGLE.EXE
bool GLEGetCrDirWin32(string* name) {
#if defined(__WIN32__) || defined(__CYGWIN__)
	TCHAR buffer[1024];
	if (GetCurrentDirectory(1024, buffer) != 0) {
		*name = buffer;
		return true;
	} else {
		return false;
	}
#else
	char buffer[1024];
	if (getcwd(buffer, 1024) == NULL) {
		return false;
	} else {
		*name = buffer;
		return true;
	}
#endif
}

// Change directory
bool GLEChDir(const string& dir) {
#ifdef __WIN32__
	return SetCurrentDirectory(dir.c_str());
#else
	return chdir(dir.c_str()) == 0;
#endif
}

void GLESetGLETop(const string& cmdline) {
	string gle_top = cmdline;
	StripPathComponents(&gle_top, 1);
	// Newer versions have the executable in the "bin" subdir
	if (!GLEFileExists(gle_top + DIR_SEP + "inittex.ini")) {
		StripPathComponents(&gle_top, 1);
	}
	gle_top = "GLE_TOP="+gle_top;
#ifdef __WIN32__
	_putenv(gle_top.c_str());
#endif
}

int GLESystem(const string& cmd, bool wait) {
#ifdef __WIN32__
	vector<string> args;
	// Split command line arguments
	level_char_separator separator(" ", "", "\"", "\"");
	tokenizer<level_char_separator> tokens(cmd, separator);
	while (tokens.has_more()) {
		args.push_back(tokens.next_token());
	}
	// Create pointer array
	char **argv = new char*[args.size()+1];
	for (int i = 0; i < args.size(); i++) {
		argv[i] = (char*)args[i].c_str();
	}
	argv[args.size()] = NULL;
	// Command without quotes
	string cmd_arg = args[0];
	str_remove_quote(cmd_arg);
	// Run sub-process
	_flushall();
	// cout << "Running: '" << cmd_arg << "'" << endl;
	intptr_t result = _spawnvp(wait ? _P_WAIT :  _P_NOWAIT, cmd_arg.c_str(), argv);
	// Clean up
	delete[] argv;
	if (result == -1) return GLE_SYSTEM_ERROR;
#else
	if (wait) {
		system(cmd.c_str());
	} else {
		pid_t pid = fork();
		if (pid == 0) {
			execl("/bin/sh", "/bin/sh", "-c", cmd.c_str(), NULL);
			_exit(0);
		} else if (pid < 0) {
			return GLE_SYSTEM_ERROR;
		}
	}
#endif
	return GLE_SYSTEM_OK;
}

void GLESleep(int msec) {
#ifdef __WIN32__
	Sleep(msec);
#else
	sleep(msec/1000);
#endif
}

#define GLE_RUN_BUFSIZE 4096

int GLERun(const string& cmd, string& result) {
#ifdef __WIN32__
	HANDLE hChildStdoutRd, hChildStdoutWr;
	// Set the bInheritHandle flag so pipe handles are inherited.
	SECURITY_ATTRIBUTES saAttr;
 	saAttr.nLength = sizeof(SECURITY_ATTRIBUTES);
	saAttr.bInheritHandle = TRUE;
	saAttr.lpSecurityDescriptor = NULL;
	// Create a pipe for the child process's STDOUT.
	if (!CreatePipe(&hChildStdoutRd, &hChildStdoutWr, &saAttr, 0)) {
		return -1;
	}
	// Ensure the read handle to the pipe for STDOUT is not inherited.
	SetHandleInformation( hChildStdoutRd, HANDLE_FLAG_INHERIT, 0);
	// Create child process
	PROCESS_INFORMATION piProcInfo;
	STARTUPINFO siStartInfo;
	// Set up members of the PROCESS_INFORMATION structure.
	ZeroMemory( &piProcInfo, sizeof(PROCESS_INFORMATION) );
	// Set up members of the STARTUPINFO structure.
	ZeroMemory( &siStartInfo, sizeof(STARTUPINFO) );
	siStartInfo.cb = sizeof(STARTUPINFO);
	siStartInfo.hStdError = hChildStdoutWr;
	siStartInfo.hStdOutput = hChildStdoutWr;
	siStartInfo.dwFlags |= STARTF_USESTDHANDLES;
 	// Create the child process.
    	BOOL bFuncRetn = CreateProcess(NULL, (CHAR*)cmd.c_str(), NULL, NULL, TRUE, 0, NULL, NULL, &siStartInfo, &piProcInfo);
	if (bFuncRetn == 0) {
		return -2;
	}
	CloseHandle(piProcInfo.hProcess);
	CloseHandle(piProcInfo.hThread);
	DWORD dwRead, dwWritten;
	CHAR chBuf[GLE_RUN_BUFSIZE];
	CloseHandle(hChildStdoutWr);
	for (;;) {
		if (!ReadFile( hChildStdoutRd, chBuf, GLE_RUN_BUFSIZE, &dwRead, NULL) || dwRead == 0) break;
		chBuf[dwRead] = 0;
		result += chBuf;
	}
	CloseHandle(hChildStdoutRd);
#endif
	return 0;
}

bool GLEFileExists(const string& fname) {
	FILE* f = fopen(fname.c_str(), "rb");
	if (f != NULL) {
		fclose(f);
		return true;
	} else {
		return false;
	}
}

// Create new directory
void MakeDirectory(const string& dir) {
	/* Create directory user read/write/exec, group & others read/exec */
#ifdef __WIN32__
	SECURITY_ATTRIBUTES sec_attr;
	sec_attr.nLength = sizeof(SECURITY_ATTRIBUTES);
	sec_attr.lpSecurityDescriptor = NULL;
	sec_attr.bInheritHandle = FALSE;
	CreateDirectory(dir.c_str(), &sec_attr);
#else
	mkdir(dir.c_str(), S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
#endif
}

// Recursively create directory if it does not yet exist
void EnsureMkDir(const string& dir) {
	if (!IsDirectory(dir)) {
		int done = 0;
		string temp = dir;
		vector<string> comps;
		do {
			string::size_type i = temp.rfind(DIR_SEP);
			if (i != string::npos) {
				comps.push_back(temp.substr(i+1));
				temp = temp.substr(0, i);
			} else {
				comps.push_back(temp);
				done = 1;
			}
		} while (done == 0 && (!IsDirectory(temp)));
		if (done == 0) {
			temp += DIR_SEP;
		} else {
			temp = "";
		}
		for (int i = comps.size()-1; i >= 0; i--) {
			temp += comps[i];
			MakeDirectory(temp);
			if (i > 0) temp += DIR_SEP;
		}
	}
}

// Delete file with extension replaced by given one
bool TryDeleteFile(const string& fname) {
#ifdef __WIN32__
	DeleteFile(fname.c_str());
#else
	unlink(fname.c_str());
#endif
	return true;
}

// Delete file with extension replaced by given one
bool DeleteFileWithExt(const string& fname, const char* ext) {
	string main_name;
	GetMainName(fname, main_name);
	main_name += ext;
	return TryDeleteFile(main_name);
}

// Read line from file
int ReadFileLine(istream& file, string& line) {
    line = "";
    int count = 0;
    char ch = '\n';
    while ((ch == '\n' || ch == '\r') && !file.eof()) {
        file.read(&ch, 1);
    }
    while (ch != '\n' && ch != '\r' && !file.eof()) {
        count++;
        line += ch;
        file.read(&ch, 1);
    }
    return count;
}

int ReadFileLineAllowEmpty(istream& file, string& line) {
    line = "";
    char ch;
    int count = 0;
    file.read(&ch, 1);
    while (ch != '\n' && ch != '\r' && !file.eof()) {
        count++;
        line += ch;
        file.read(&ch, 1);
    }
    return count;
}

bool IsDirectory(const string& fname) {
#if defined(__UNIX__) || ( defined(__OS2__) && defined(__EMX__) )
	struct stat stat_buf;
	if (stat((const char *)fname.c_str(), &stat_buf) == 0) {
		return S_ISDIR(stat_buf.st_mode);
	} else {
		return false;
	}
#else
	struct _stat stat_buf;
	if (_stat((const char *)fname.c_str(), &stat_buf) == 0) {
		return (stat_buf.st_mode & _S_IFDIR);
	} else {
		return false;
	}
#endif
}

void GLEFindFiles(const string& dir, const vector<string>& tofind, vector<string*>& result) {
	static int find_files_progress = 0;
	vector<string> subdirs;
	if (find_files_progress++ == 10) {
		cout << "."; fflush(stdout);
		find_files_progress = 0;
	}
#ifdef __WIN32__
	WIN32_FIND_DATA FindFileData;
	string findpattern = dir + DIR_SEP + "*.*";
	HANDLE hFind = FindFirstFile(findpattern.c_str(), &FindFileData);
	if (hFind == INVALID_HANDLE_VALUE) {
		return;
	}
	do {
		const char* name = FindFileData.cFileName;
		if ((FindFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0) {
			if (!str_i_equals(name, ".") && !str_i_equals(name, "..")) {
				subdirs.push_back(name);
			}
		} else {
			for (int i = 0; i < tofind.size(); i++) {
				if (str_i_equals(name, tofind[i].c_str())) {
					string place = dir + DIR_SEP + tofind[i];
					int len_i = (*result[i]).length();
					if (len_i >= 1 && (*result[i])[len_i-1] == ';') {
						if (len_i == 1) (*result[i]) = place + ";";
						else (*result[i]) += place + ";";
					} else {
						(*result[i]) = place;
					}
				}
			}
		}
	} while (FindNextFile(hFind, &FindFileData) != 0);
	FindClose(hFind);
#endif
	for (int i = 0; i < subdirs.size(); i++) {
		string nextdir = dir + DIR_SEP + subdirs[i];
		GLEFindFiles(nextdir, tofind, result);
	}
}

string GetActualFilename(string fname) {
	// will return the filename
	// if found in ./ or in any of the include
	// paths, will return empty string if
	// file not found

	std::ifstream file;
	file.open(fname.c_str());
	if (file.is_open()){
		file.close();
		return fname;
	}

	// -- try the IncludePaths
	vector<string> IncludePaths;
	FillIncludePaths(IncludePaths);
	vector<string>::iterator vsi = IncludePaths.begin();
	while(vsi != IncludePaths.end()){
		string p = *vsi+DIR_SEP.c_str()+fname;
		file.open(p.c_str());
		if(file.is_open()){
			file.close();
			return p;
		}
		vsi++;
	}
	return "";
}

#ifdef HAVE_SOCKETS
void GLECloseSocket(SOCKET sock) {
#ifdef __WIN32__
	closesocket(sock);
	WSACleanup();
#endif
#ifdef __UNIX__
	close(sock);
#endif
}
#endif

int GLESendSocket(const string& commands) {
#ifdef HAVE_SOCKETS
#ifdef __WIN32__
	struct WSAData wsaData;
	int nCode = WSAStartup(MAKEWORD(1, 1), &wsaData);
	if (nCode != 0) {
		return -1;
	}
#endif
	struct sockaddr_in name;
	SOCKET sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (sock < 0) {
		#ifdef __WIN32__
		WSACleanup();
		#endif
		return -2;
	}
	name.sin_family = AF_INET;
	name.sin_port = htons(6667);
	name.sin_addr.s_addr = inet_addr("127.0.0.1");
	if (connect(sock, (struct sockaddr *) &name, sizeof (name)) < 0) {
		GLECloseSocket(sock);
		return -3;
	}
	int nbsend = send(sock, commands.c_str(), commands.length(), 0);
	if (nbsend != commands.length()) {
		GLECloseSocket(sock);
		return -4;
	}
	GLECloseSocket(sock);
	return 0;
#else
	return -1;
#endif
}

bool GetExeName(const char* appname, string& exe_name) {
#ifdef __WIN32__
	char name[1024];
	DWORD res = GetModuleFileName(NULL, name, 1023);
	if (res > 0) {
		name[res] = 0;
		exe_name = name;
		return true;
	}
#endif
#ifdef __UNIX__
	/* try to read location from the /proc/self/exe file */
	char path[PATH_MAX];
	struct stat stat_buf;
	string path2 = "/proc/self/exe";
	while (true) {
		int size = readlink(path2.c_str(), path, PATH_MAX - 1);
		if (size == -1) {
			break;
		}
		path[size] = '\0';
		int i = stat(path, &stat_buf);
		if (i == -1) {
			break;
		}
		if (!S_ISLNK(stat_buf.st_mode)) {
			exe_name = path;
			return true;
		}
		path2 = path;
	}
	/* try to read location from the /proc/self/maps file */
	ifstream maps("/proc/self/maps");
	if (!maps.is_open()) {
		return false;
	}
	string f1 = DIR_SEP + appname;
	string f2 = f1 + ".exe";
	while (!maps.eof()) {
		string line;
		ReadFileLine(maps, line);
		char_separator separator(" ", "");
		tokenizer<char_separator> tokens(line, separator);
		while (tokens.has_more()) {
			exe_name = tokens.next_token();
			if (str_i_ends_with(exe_name, f1.c_str()) || str_i_ends_with(exe_name, f2.c_str())) {
				return true;
			}
		}
	}
	maps.close();
#endif
	return false;
}

StreamTokenizerMax::StreamTokenizerMax(const string& fname, int sep, int max) : m_File(fname.c_str()) {
	m_Sep = sep; m_Max = max; m_IsOK = 1;
	m_LastToken = new char[m_Max+1];
	if (!m_File.is_open()) {
		m_IsOK = 0;
	}
}

StreamTokenizerMax::~StreamTokenizerMax() {
	delete[] m_LastToken;
}

bool StreamTokenizerMax::hasMoreTokens() {
	if (m_IsOK == 1) readNextToken();
	return m_IsOK == 1;
}

const char* StreamTokenizerMax::nextToken() {
	return m_LastToken;
}

void StreamTokenizerMax::close() {
	m_File.close();
}

bool StreamTokenizerMax::isSepChar(char ch) {
	return ch == m_Sep || ch == '\n' || ch == '\r' || ch == 0;
}

void StreamTokenizerMax::readNextToken() {
	char ch = m_Sep;
	while (isSepChar(ch) && !m_File.eof()) {
		m_File.read(&ch, 1);
	}
	int count = 0;
	while (count < m_Max && !isSepChar(ch) && !m_File.eof()) {
		if (ch != m_Sep) m_LastToken[count++] = ch;
		m_File.read(&ch, 1);
	}
	m_LastToken[count] = 0;
	while (!isSepChar(ch) && !m_File.eof()) {
		m_File.read(&ch, 1);
	}
	if (m_File.eof()) {
		m_IsOK = 0;
	}
}

//
// -- the only two dir finding routines needed
//
const int PATH_LENGTH = 256;

void CopyGLETop(char* buf) {
	strcpy(buf, GLE_TOP_DIR.c_str());
}

char *fontdir(char *fname) {
	// simply returns %GLE_TOP%/font
	static char fbuff[PATH_LENGTH];
	CopyGLETop(fbuff);
	strcat(fbuff,DIR_SEP.c_str());
	strcat(fbuff,"font");
	strcat(fbuff,DIR_SEP.c_str());
	strcat(fbuff,fname);
	return fbuff;
}

char *gledir(char *fname) {
	// simply returns %GLE_TOP%
	static char fbuff[PATH_LENGTH];
	CopyGLETop(fbuff);
	strcat(fbuff,DIR_SEP.c_str());
	strcat(fbuff,fname);
	return fbuff;
}
