/*--------------------------------------------------------------*/
/*	Post Script Driver, for GLE                                 */
/*--------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/

#include "all.h"
#include "SourceLine.h"
#include "tokens/Tokenizer.h"
#include "core.h"
#include "mygraph.h"
#include "d_interface.h"
#include "gprint.h"
#include "cutils.h"
#include "op_def.h"

//
// -- constants
//
#define dbg if ((gle_debug & 64)>0)

//
// elipse function only for PS
//
static char ellipse_fcn[] = "\
/ellipsedict 8 dict def\n ellipsedict /mtrx matrix put \n\
/ellipse\n\
        { ellipsedict begin\n\
          /endangle exch def\n\
          /startangle exch def\n\
          /yrad exch def\n\
          /xrad exch def\n\
          /y exch def\n\
          /x exch def\n\
          \n\
          /savematrix mtrx currentmatrix def\n\
          x y translate\n\
          xrad yrad scale\n\
          0 0 1 startangle endangle arc\n\
          savematrix setmatrix\n\
          end\n\
        } def\n\
/ellipsen\n\
        { ellipsedict begin\n\
		/endangle exch def\n\
		  /startangle exch def\n\
          /yrad exch def\n\
          /xrad exch def\n\
          /y exch def\n\
          /x exch def\n\
          \n\
          /savematrix mtrx currentmatrix def\n\
          x y translate\n\
          xrad yrad scale\n\
          0 0 1 startangle endangle arcn\n\
          savematrix setmatrix\n\
          end\n\
        } def\n\
";
//
// -- global variables can we get rid of these?
//
extern int MAX_VECTOR; /* Cant send POSTSCRIPT too complex a path */
extern bool control_d;
extern struct gmodel g;
extern bool BLACKANDWHITE;
extern bool GS_PREVIEW;
extern int gle_debug;

//
// -- function prototypes
//
char *fontdir(char *s);
char *font_getname(int i);

void d_tidyup() {
}
/*---------------------------------------------------------------------------*/
PSGLEDevice::PSGLEDevice(bool eps) : GLEDevice() {
	m_IsEps = eps;
	m_IsPageSize = false;
	first_ellipse = 1;
	ps_nvec = 0;
	g_cur_fill_color.l = GLE_COLOR_BLACK;
}
/*---------------------------------------------------------------------------*/
PSGLEDevice::~PSGLEDevice() {
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::dfont(char *c)
{
	/* only used for the DFONT driver which builds fonts */
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::message(char *s)
{
	printf("%s\n",s);
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::devcmd(char *s)
{
	fprintf(psfile,"%s",s);
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::source(char *s)
{
	dbg fprintf(psfile,"%% SOURCE, %s",s);
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::get_type(char *t)
{
	strcpy(t,"HARDCOPY, PS, FILLPATH");
	if (isEps()) strcat(t,", EPS,");
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::set_path(int onoff)
{
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::newpath()
{
	fprintf(psfile," newpath ");
	ps_nvec = 0;
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::pscomment(char* ss)
{
	comments.push_back(ss);
}
/*---------------------------------------------------------------------------*/
FILE* PSGLEDevice::get_file_pointer(void)
{
	return psfile;
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::opendev(double width, double height, const string& outputfile, const string& inputfile) {
	first_ellipse = 1;
	input_file = inputfile;
	output_file = outputfile;
	string ext = (isEps() ? ".eps":".ps");
	if( output_file.rfind('.') != output_file.npos){
		output_file.erase(output_file.rfind('.')+1,output_file.length()-1);
		output_file.replace(output_file.rfind('.'),ext.length(),ext);
	} else {
		if (output_file == "") {
			output_file = "output" + ext;
		} else {
			output_file += ext;
		}
	}
	// GS_PREVIEW a.r. 2005
	// - purpose: *FAST* previewing while writing GLE code,
	//   don't write huge eps files to disk, no hassle with resizing ghostview,
	//   gsview or gv windows.
	// - add to .Xresources: Ghostscript*geometry:   -5+0
	//   to place gs window in upper right corner like  gle -d x11
	// - UNIX: should use tool functions to populate first argument of GS_cmd,
	//   as it may be gsc (ghostscript 8.5 compiled with shared lib) !!!
        // - may use environment variable or .glerc file for additional gs options
	//   based on $DISPLAY, if you work on multiple sytems
#ifdef ENABLE_GS_PREVIEW
	if (GS_PREVIEW) {
		int gsPixelWidth, gsPixelHeight, gsPixelRes;
		displayGeometry(width, height, &gsPixelWidth, &gsPixelHeight, &gsPixelRes);
		stringstream GScmd;
		GScmd << "gs -sDEVICE=x11 -dTextAlphaBits=4 -dGraphicsAlphaBits=2 -dMaxBitmap=5000000 ";
		GScmd << "-dNOPLATFONTS -dTTYPAUSE -g" << gsPixelWidth << "x" << gsPixelHeight << " ";
		GScmd << "-r" << gsPixelRes << "x" << gsPixelRes << " -dDELAYSAFER ";
		GScmd << "-c '<< /PermitFileReading [ (/dev/tty)] >> setuserparams .locksafe' -dSAFER -q -_";
		cerr << endl << "Calling " << GScmd.str() << endl;
		// NOTE: popen may not exist (or may not work as expected) on Windows and OS/2 !
		psfile = popen(GScmd.str().c_str(), "w");
		if (psfile == NULL) {
			cerr << "GLE PS: popen ghostscript failed: " << GScmd.str() << endl;
			exit(1);
		}
	} else {
#endif
		psfile = fopen(output_file.c_str(),"w");
		if (psfile == NULL) {
			cerr << "GLE PS: Failed to open output file: " << output_file << endl;
			exit(1);
		}
#ifdef ENABLE_GS_PREVIEW
	}
#endif
	printf("[");
	if (!isEps()) {
		if (control_d) fprintf(psfile,"\x04 \n");
		fprintf(psfile,"%%!PS-Adobe-2.0 \n");
	} else {
		fprintf(psfile,"%%!PS-Adobe-2.0 EPSF-2.0 \n");
	}
	fprintf(psfile,"%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\n");
	fprintf(psfile,"%%%%                                                  %%%%\n");
	fprintf(psfile,"%%%% Created By:      GLE v%s",__GLEVN__);
	int nbspace = 26 - strlen(__GLEVN__);
	for (int i = 0; i < nbspace; i++) fprintf(psfile, " ");
	fprintf(psfile,"%%%%\n");
	fprintf(psfile,"%%%%                  www.gle-graphics.org            %%%%\n");
	fprintf(psfile,"%%%%                                                  %%%%\n");
	fprintf(psfile,"%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\n");
	time_t t;
	t = time(0);
	fprintf(psfile,"%%%%CreationDate: %s", ctime(&t));
	fprintf(psfile,"%%%%SourceFile: %s\n", input_file.c_str());
	list<string>::iterator lsi = comments.begin();
	while(lsi != comments.end())
	{
		fprintf(psfile,"%%%% %s\n",(*lsi).c_str());
		lsi++;
	}
	comments.clear();
	if (g_is_fullpage()) {
		bb_width = (int)floor(72*width/CM_PER_INCH+0.5);
		bb_height = (int)floor(72*height/CM_PER_INCH+0.5);
	} else {
		// Make bounding box a little larger (bounding box tweak)
		bb_width = (int)floor(72*width/CM_PER_INCH+2);
		bb_height = (int)floor(72*height/CM_PER_INCH+2);
	}
//	bb_xorig = -1; bb_yorig = -1;
//	fprintf(psfile,"%%%%BoundingBox: -1 -1 %d %d \n", bb_width-1, bb_height-1);
	bb_xorig = 0; bb_yorig = 0;
	fprintf(psfile,"%%%%BoundingBox: 0 0 %d %d \n", bb_width, bb_height);
	fprintf(psfile,"%%%%EndComments \n");
	fprintf(psfile,"%%%%EndProlog \n");
	if (isOutputPageSize()) {
		fprintf(psfile, "<< /PageSize [%d %d] >> setpagedevice\n", bb_width, bb_height);
	}
	fprintf(psfile,"gsave \n");
	fprintf(psfile," \n");
	fprintf(psfile,"/f {findfont exch scalefont setfont} bind def \n");
	fprintf(psfile,"/s {show} bind def \n");
	fprintf(psfile,"/ps {true charpath} bind def \n");
	fprintf(psfile,"/l {lineto} bind def \n");
	fprintf(psfile,"/m {newpath moveto} bind def \n");
	fprintf(psfile,"matrix currentmatrix /originmat exch def \n");
	fprintf(psfile,"/umatrix {originmat matrix concatmatrix setmatrix} def \n");
	fprintf(psfile," \n");
	// Measure distance in cm
	g_scale(72.0/CM_PER_INCH,72.0/CM_PER_INCH);
	if (!g_is_fullpage()) {
		// Bounding box tweak
		g_translate(1.0*CM_PER_INCH/72, 1.0*CM_PER_INCH/72);
	}
}

/*---------------------------------------------------------------------------*/

void PSGLEDevice::closedev() {
	g_flush();
	fprintf(psfile,"showpage \n");
	fprintf(psfile,"grestore \n");
	fprintf(psfile,"%%%%Trailer \n");
	if (!isEps()) if (control_d) fprintf(psfile,"\x04");
        if (GS_PREVIEW) { pclose(psfile); } else { fclose(psfile); }
	printf("%s]\n",output_file.c_str());
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::set_line_cap(int i)
{
	/*  lcap, 0= butt, 1=round, 2=projecting square */
	if (!g.inpath) g_flush();
	fprintf(psfile,"%d setlinecap \n",i);
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::set_line_join(int i)
{
	if (!g.inpath) g_flush();
	fprintf(psfile,"%d setlinejoin \n",i);
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::set_line_miterlimit(double d)
{
	if (!g.inpath) g_flush();
	fprintf(psfile,"%g setmiterlimit \n",d);
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::set_line_width(double w)
{
	if (w==0) w = 0.02;
	if (w<.0002) w = 0;
	if (!g.inpath) g_flush();
	fprintf(psfile,"%g setlinewidth \n",w);
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::set_line_styled(double dd)
{}
void PSGLEDevice::set_line_style(char *s)
{
	/* should deal with [] for solid lines */
	static char *defline[] = {"","","12","41","14","92","1282"
	,"9229","4114","54","73","7337","6261","2514"};
	static char ob[200];
	int l;

	if (!g.inpath) g_flush();
	strcpy(ob,"[");
	if (strlen(s)==1) s = defline[*s-'0'];
	l = strlen(s);
	for (i=0;i<l;i++)
		sprintf(ob+strlen(ob),"%g ",(*(s+i)-'0')*g.lstyled);
	strcat(ob,"]");
	fprintf(psfile,"%s 0 setdash \n",ob);
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::fill()
{
	fprintf(psfile,"gsave \n");
	ddfill();
	fprintf(psfile,"grestore \n");
}
void PSGLEDevice::ddfill(void)
{
	if (g_cur_fill.b[B_F] == 255) return; /* clear fill, do nothing */
	if (g_cur_fill.b[B_F] == 2) {
		shade();
		return;
	}
	set_fill();			/*because color and fill are the same*/
	fprintf(psfile,"fill \n");
	set_color();
}

void PSGLEDevice::shade(void) {
	// Implemented by using path as clip and then painting strokes over the clip
	double x,y,step1,step2;
	fprintf(psfile,"gsave \n");
	fprintf(psfile,"clip \n");
	fprintf(psfile,"newpath  \n");
	if (g_cur_fill_color.l == GLE_COLOR_BLACK) {
		fprintf(psfile,"0 setgray \n");
	} else {
		set_color(g_cur_fill_color);
	}
	step1 = g_cur_fill.b[B_B]/160.0;
	step2 = g_cur_fill.b[B_G]/160.0;
	fprintf(psfile,"%g setlinewidth\n",(double) g_cur_fill.b[B_R]/160.0);
	if (step1>0) {
	  fprintf(psfile,"%g %g %g { /x exch def \n",-40.0,step1,40.0);
	  fprintf(psfile,"x 0 moveto 40 x add 40 lineto stroke\n");
	  fprintf(psfile,"} for \n");
	}
	if (step2>0) {
	  fprintf(psfile,"%g %g %g { /x exch def \n",0.0,step2,80.0);
	  fprintf(psfile,"x 0 moveto -40 x add 40 lineto stroke\n");
	  fprintf(psfile,"} for \n");
	}
	fprintf(psfile,"grestore \n");
}

/*---------------------------------------------------------------------------*/
void PSGLEDevice::fill_ary(int nwk,double *wkx,double *wky)
{
	int i;
	fprintf(psfile,"gsave \n");
	fprintf(psfile,"newpath \n");
	fprintf(psfile,"%g %g moveto \n",wkx[0],wky[0]);
	for (i=1;i<nwk;i++)
		fprintf(psfile,"%g %g l \n",wkx[i],wky[i]);
	set_fill();
	fprintf(psfile,"fill \n");
	set_color();
	fprintf(psfile,"grestore \n");
}
void PSGLEDevice::line_ary(int nwk,double *wkx,double *wky)
{
	int i;
	fprintf(psfile,"gsave \n");
	fprintf(psfile,"newpath \n");
	fprintf(psfile,"%g %g moveto \n",wkx[0],wky[0]);
	for (i=1;i<nwk;i++)
		fprintf(psfile,"%g %g l \n",wkx[i],wky[i]);
	fprintf(psfile,"stroke \n");
	fprintf(psfile,"grestore \n");
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::stroke()
{
	fprintf(psfile,"gsave \n");
	fprintf(psfile,"stroke \n");
	fprintf(psfile,"grestore \n");
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::clip()
{
	fprintf(psfile,"clip \n");
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::set_matrix(double newmat[3][3])
{
	fprintf(psfile,"[%g %g %g %g %g %g] umatrix \n ",
		newmat[0][0],newmat[1][0],newmat[0][1],
		newmat[1][1],newmat[0][2],newmat[1][2]);
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::move(double zx,double zy)
{
	if (g.inpath==static_cast<int>(true))
		fprintf(psfile,"%g %g moveto \n",zx,zy);
	else {
		ps_nvec++;
		fprintf(psfile,"%g %g m \n",zx,zy);
	}
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::reverse() 	/* reverse the order of stuff in the current path */
{
	fprintf(psfile,"reversepath \n");
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::closepath()
{
	fprintf(psfile,"closepath \n");
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::line(double zx,double zy)
{
	dbg gprint("in d_line  g.curx,y  %g %g ",g.curx,g.cury);
	if (g.xinline==false) {
		move(g.curx,g.cury);
	}
	ps_nvec++;
	if (ps_nvec>MAX_VECTOR) {
		//gprint("Warning, complex path, if filling fails then try /nomaxpath \n");
		ps_nvec = 0; g_flush(); move(g.curx,g.cury);
	}
	if (fprintf(psfile,"%g %g l \n",zx,zy)==EOF) {
		perror("=========Unable to write to output file ");
	}
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::clear() {
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::flush()
{
	if (g.inpath) return;
	if (g.xinline) {
		fprintf(psfile,"stroke\n");
		ps_nvec = 0;
	}
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::arcto(dbl x1,dbl y1,dbl x2,dbl y2,dbl rrr)
{
	if (g.xinline==false) move(g.curx,g.cury);
	fprintf(psfile,"%g %g %g %g %g arcto clear %g %g l \n",x1,y1,x2,y2,rrr,x2,y2);
	g.xinline = true;
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::arc(dbl r,dbl t1,dbl t2,dbl cx,dbl cy)
{
	double dx,dy;
	double x,y;
	g_get_xy(&x,&y);
	polar_xy(r,t1,&dx,&dy);
	if (!g.inpath) g_move(cx+dx,cy+dy);
	fprintf(psfile,"%g %g %g %g %g arc \n",cx,cy,r,t1,t2);
	g.xinline = true;
	if (!g.inpath) g_move(x,y);
}
/*---------------------------------------------------------------------------*/
void PSGLEDevice::narc(dbl r,dbl t1,dbl t2,dbl cx,dbl cy)
{
	double dx,dy;
	double x,y;
	g_get_xy(&x,&y);
	polar_xy(r,t1,&dx,&dy);
	if (!g.inpath) g_move(cx+dx,cy+dy);
	fprintf(psfile,"%g %g %g %g %g arcn \n",cx,cy,r,t1,t2);
	g.xinline = true;
	if (!g.inpath) g_move(x,y);
}
void PSGLEDevice::elliptical_arc(dbl rx,dbl ry,dbl t1,dbl t2,dbl cx,dbl cy)
{
	double dx,dy;
	double x,y;
	if(first_ellipse)
	{
		first_ellipse = 0;
		fprintf(psfile,"\n%s\n",ellipse_fcn);
	}
	g_get_xy(&x,&y);
	polar_xy(rx,ry,t1,&dx,&dy);
	if (!g.inpath) g_move(cx+dx,cy+dy);
	fprintf(psfile,"%g %g %g %g %g %g ellipse \n",cx,cy,rx,ry,t1,t2);
	g.xinline = true;
	if (!g.inpath) g_move(x,y);
}
void PSGLEDevice::elliptical_narc(dbl rx,dbl ry,dbl t1,dbl t2,dbl cx,dbl cy)
{
	double dx,dy;
	double x,y;
	if(first_ellipse)
	{
		first_ellipse = 0;
		fprintf(psfile,"\n%s\n",ellipse_fcn);
	}
	g_get_xy(&x,&y);
	polar_xy(rx,ry,t1,&dx,&dy);
	if (!g.inpath) g_move(cx+dx,cy+dy);
	fprintf(psfile,"%g %g %g %g %g %g ellipsen\n",cx,cy,rx,ry,t1,t2);
	g.xinline = true;
	if (!g.inpath) g_move(x,y);
}

void PSGLEDevice::box_fill(dbl x1, dbl y1, dbl x2, dbl y2) {
	if (g.inpath==static_cast<int>(true)) {
		xdbox(x1,y1,x2,y2);
	} else {
		g_flush();
		fprintf(psfile,"newpath ");
		xdbox(x1,y1,x2,y2);
		ddfill();
		fprintf(psfile,"newpath\n");
	}
}

void PSGLEDevice::box_stroke(dbl x1, dbl y1, dbl x2, dbl y2, bool reverse) {
	if (g.inpath==static_cast<int>(true)) {
		if (reverse) {
			fprintf(psfile,"%g %g moveto %g %g l %g %g l %g %g l closepath\n",x1,y1,x1,y2,x2,y2,x2,y1);
		} else {
			xdbox(x1,y1,x2,y2);
		}
	} else {
		g_flush();
		fprintf(psfile,"newpath ");
		xdbox(x1,y1,x2,y2);
		fprintf(psfile,"stroke\n");
		ps_nvec = 0;
	}
}

void PSGLEDevice::xdbox(double x1, double y1, double x2, double y2) {
	fprintf(psfile,"%g %g moveto %g %g l %g %g l %g %g l closepath\n",x1,y1,x2,y1,x2,y2,x1,y2);
}

/*---------------------------------------------------------------------------*/
void PSGLEDevice::circle_stroke(double zr)
{
	double x,y;
	g_get_xy(&x,&y);
	if (g.inpath==static_cast<int>(true))
		fprintf(psfile," %g %g %g 0 360 arc \n",x,y,zr);
	else {
		g_flush();
		fprintf(psfile," newpath ");
		fprintf(psfile," %g %g %g 0 360 arc \n",x,y,zr);
		fprintf(psfile,"stroke \n");
	}
}
void PSGLEDevice::circle_fill(double zr)
{
	double x=g.curx,y=g.cury;
	if (g.inpath==static_cast<int>(true))
		fprintf(psfile," %g %g %g 0 360 arc \n",x,y,zr);
	else {
		g_flush();
		fprintf(psfile,"newpath ");
		fprintf(psfile,"%g %g %g 0 360 arc \n",x,y,zr);
		ddfill();
		fprintf(psfile,"newpath \n");
/*

		set_fill();
		fprintf(psfile,"fill \n");
		set_color();
*/
	}
}

void PSGLEDevice::ellipse_stroke(double rx, double ry)
{
	double x,y;
	if(first_ellipse)
	{
		first_ellipse = 0;
		fprintf(psfile,"\n%s\n",ellipse_fcn);
	}

	g_get_xy(&x,&y);
	if (g.inpath==static_cast<int>(true))
		fprintf(psfile," %g %g %g %g 0 360 ellipse \n",x,y,rx,ry);
	else {
		g_flush();
		fprintf(psfile," newpath ");
		fprintf(psfile," %g %g %g %g 0 360 ellipse closepath \n",x,y,rx,ry);
		fprintf(psfile,"stroke \n");
	}
}
void PSGLEDevice::ellipse_fill(double rx, double ry)
{
	double x=g.curx,y=g.cury;
	if(first_ellipse)
	{
		first_ellipse = 0;
		fprintf(psfile,"\n%s\n",ellipse_fcn);
	}
	if (g.inpath==static_cast<int>(true))
		fprintf(psfile," %g %g %g %g 0 360 ellipse \n",x,y,rx,ry);
	else {
		g_flush();
		fprintf(psfile,"newpath ");
		fprintf(psfile," %g %g %g %g 0 360 ellipse \n",x,y,rx,ry);
		ddfill();
		fprintf(psfile,"newpath \n");
/*

		set_fill();
		fprintf(psfile,"fill \n");
		set_color();
*/
	}
}

/*---------------------------------------------------------------------------*/
void PSGLEDevice::bezier(dbl x1,dbl y1,dbl x2,dbl y2,dbl x3,dbl y3)
{
	double x=g.curx,y=g.cury;
	if (g.inpath==static_cast<int>(true)) {
		if (g.xinline==false)  move(g.curx,g.cury);
		fprintf(psfile,"%g %g %g %g %g %g curveto \n"
			,x1,y1,x2,y2,x3,y3);
	} else {
		g_flush();
		if (!g.xinline) fprintf(psfile,"%g %g moveto ",x,y);
		fprintf(psfile,"%g %g %g %g %g %g curveto \n"
			,x1,y1,x2,y2,x3,y3);
	}
	g.xinline = true;
}
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
void PSGLEDevice::test_psfile()
{
	if (psfile==NULL) return;
	set_color();
}

void PSGLEDevice::set_color(colortyp& color) {
	if (BLACKANDWHITE) {
		fprintf(psfile,"%g setgray \n",((color.b[B_R]*3.0/255.0+color.b[B_G]*2.0/255.0+color.b[B_B]/255.0) / 6));
	} else {
		fprintf(psfile,"%g %g %g setrgbcolor \n",color.b[B_R]/255.0,color.b[B_G]/255.0,color.b[B_B]/255.0);
	}
}

void PSGLEDevice::set_color() {
	set_color(g_cur_color);
}

void PSGLEDevice::set_fill() {
	set_color(g_cur_fill);
}

/*---------------------------------------------------------------------------*/

void PSGLEDevice::set_color(int f) {
	g_flush();
	g_cur_color.l = f;
	set_color();
}

void PSGLEDevice::set_fill(int f) {
	g_cur_fill.l = f;
}

void PSGLEDevice::set_pattern_color(int c) {
	g_cur_fill_color.l = c;
}

void PSGLEDevice::beginclip() {
	fprintf(psfile,"gsave \n");
}

void PSGLEDevice::endclip() {
	g_flush();
	fprintf(psfile,"grestore \n");
	gmodel* state = (gmodel*) myallocz(SIZEOFSTATE);
	g_get_state(state);
	g_set_state(state);
	myfree(state);
}
/*---------------------------------------------------------------------------*/
struct psfont_struct {char *sname; char *lname;} ;
struct psfont_struct psf[70] = { /* leaves room for twenty more from PSFONT.DAT*/
	"PSTR",		"Times-Roman",
	"PSTI",		"Times-Italic",
	"PSTB",		"Times-Bold",
	"PSTBI",	"Times-BoldItalic",
	"RM",		"Times-Roman",
	"RMI",		"Times-Italic",
	"RMB",		"Times-Bold",
	"RMBI",		"Times-BoldItalic",
	"SS",		"Helvetica",
	"SSB",		"Helvetica-Bold",
	"SSI",		"Helvetica-Oblique",
	"SSBI",		"Helvetica-BoldOblique",
	"PSH",		"Helvetica",
	"PSHB",		"Helvetica-Bold",
	"PSHBO",	"Helvetica-BoldOblique",
	"PSHO",		"Helvetica-Oblique",
	"PSAGB",	"AvantGarde-Book",
	"PSAGBO",	"AvantGarde-BookOblique",
	"PSAGD",	"AvantGarde-Demi",
	"PSAGDO",	"AvantGarde-DemiOblique",
	"PSBD",		"Bookman-Demi",
	"PSBDI",	"Bookman-DemiItalic",
	"PSBL",		"Bookman-Light",
	"PSBLI",	"Bookman-LightItalic",
	"PSC",		"Courier",
	"PSCB",		"Courier-Bold",
	"PSCBO",	"Courier-BoldOblique",
	"PSCO",		"Courier-Oblique",
	"TT",		"Courier",
	"TTB",		"Courier-Bold",
	"TTBI",		"Courier-BoldOblique",
	"TTI",		"Courier-Oblique",
	"PSNCSB",	"NewCenturySchlbk-Bold",
	"PSNCSBI",	"NewCenturySchlbk-BoldItalic",
	"PSNCSI",	"NewCenturySchlbk-Italic",
	"PSNCSR",	"NewCenturySchlbk-Roman",
	"PSPB",		"Palatino-Bold",
	"PSPBI",	"Palatino-BoldItalic",
	"PSPI",		"Palatino-Italic",
	"PSPR",		"Palatino-Roman",
	"PSZCMI",	"ZapfChancery-MediumItalic",
	"PSZD",		"ZapfDingbats",
	"PSSYM",	"Symbol",
	NULL,NULL
};

static int d_ps_this_font = 0;
static double d_ps_this_size = 0;

/*---------------------------------------------------------------------------*/
void PSGLEDevice::dochar(int font, int cc)
{
	double x,y;
	char *s;

	read_psfont();
	if (font_get_encoding(font)>2) {
		my_char(font,cc);
		return;
	}
	if (d_ps_this_font!=font || d_ps_this_size!=g.fontsz) {
		if (g.fontsz<0.00001) {
			gprint("Font size is zero, error ********* \n");
			return;
		}
		s = font_getname(font);
		for (i=0;;i++) {
			if (psf[i].sname==NULL) break;
			dbg printf("font match  {%s} {%s} \n",s,psf[i].sname);
			if (str_i_equals(psf[i].sname,s)) break;
		}
		if (psf[i].sname==NULL) {
			my_char(font,cc);
			return;
		}
		d_ps_this_font = font;
		d_ps_this_size = g.fontsz;
		fprintf(psfile," %f /%s f ",g.fontsz,psf[i].lname);
	}
	if (g.inpath==static_cast<int>(true)) {
		if (isalnum(cc) && cc<127) fprintf(psfile,"(%c) ps ",cc);
		else  fprintf(psfile,"(\\%o) ps ",cc);
	} else {
		if (isalnum(cc) && cc<127) fprintf(psfile,"(%c) s ",cc);
		else  fprintf(psfile,"(\\%o) s ",cc);
	}
}

void PSGLEDevice::resetfont() {
	d_ps_this_font = 0;
	d_ps_this_size = 0;
}

void PSGLEDevice::read_psfont(void)  /* add aditional ps fonts,  e.g.  pstr = TimesRoman */
{
	static int init_done;
	FILE *fptr;
	char fname[80],*s;
	char inbuff[90];
	if (init_done) return;
	init_done = true;

	/* Find last used psf */
	for (i=0;;i++) if (psf[i].sname==NULL) break;

	strcpy(fname,fontdir("psfont.dat"));
	fptr = fopen(fname,"r");
	if (fptr==0) return; /* if not exists then don't bother */

	for (fgets(inbuff,200,fptr);!feof(fptr);fgets(inbuff,200,fptr)) {
		s = strchr(inbuff,'!');
		if (s!=NULL) *s=0;
		s = strtok(inbuff," \t,\n");
		if (s!=NULL) if (*s!='\n') {
			psf[i].sname = sdup(s);
			s = strtok(0," \t,\n");
			psf[i].lname = sdup(s);
			i++;
		}
	}
	psf[i].sname = NULL;
	psf[i].lname = NULL;
}

void PSGLEDevice::displayGeometry(double width, double height, int *gsPixelWidth, int *gsPixelHeight, int *gsPixelRes) {
#if defined(__UNIX__) && defined(ENABLE_GS_PREVIEW)
  Display *dpy = XOpenDisplay((char *) NULL);
  if (dpy == NULL){
	perror("Unable to open Display!");
	exit(1);
  }
  int screenNo = DefaultScreen(dpy);
  int dpyPixelWidth = DisplayWidth(dpy, screenNo);
  int dpyPixelHeight = DisplayHeight(dpy, screenNo);
  XCloseDisplay(dpy);
  double dpyXbyY = (double) dpyPixelWidth / (double) dpyPixelHeight;
  double gleXbyY = width / height;
  if (gleXbyY > dpyXbyY) {
	*gsPixelWidth = (int) (0.9 * dpyPixelWidth);
	*gsPixelRes = (int) (*gsPixelWidth/(width/CM_PER_INCH));
	*gsPixelHeight = (int) ( *gsPixelWidth / gleXbyY);
  } else {
	*gsPixelHeight = (int) (0.9 * dpyPixelHeight);
	*gsPixelRes = (int) (*gsPixelHeight/(height/CM_PER_INCH));
	*gsPixelWidth = (int) (*gsPixelHeight * gleXbyY);
  }
#endif
#if defined(__WIN32__) && defined(ENABLE_GS_PREVIEW)
	// Not yet implemented
#endif
#if defined(__OS2__) && defined(ENABLE_GS_PREVIEW)
	// Not yet implemented
#endif
}

int PSGLEDevice::getDeviceType() {
	return m_IsEps ? GLE_DEVICE_EPS : GLE_DEVICE_PS;
}
