/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.jaxrs.report;

import com.jaspersoft.jasperserver.api.engine.common.service.ReportExecutionStatusInformation;
import com.jaspersoft.jasperserver.api.engine.common.service.SchedulerReportExecutionStatusSearchCriteria;
import com.jaspersoft.jasperserver.remote.common.CallTemplate;
import com.jaspersoft.jasperserver.remote.common.RemoteServiceWrapper;
import com.jaspersoft.jasperserver.remote.exception.IllegalParameterValueException;
import com.jaspersoft.jasperserver.remote.exception.RemoteException;
import com.jaspersoft.jasperserver.remote.services.RunReportService;
import com.jaspersoft.jasperserver.remote.services.impl.ReportOutputResource;
import com.jaspersoft.jasperserver.war.dto.InputControlState;
import com.jaspersoft.jasperserver.war.dto.ReportInputControl;
import org.apache.commons.lang.StringUtils;
import org.codehaus.jettison.json.JSONObject;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.PathSegment;
import javax.ws.rs.core.Response;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author Yaroslav.Kovalchyk
 * @version $Id: RunReportsJaxrsService.java 23655 2012-05-08 12:18:11Z lchirita $
 */
@Service
@Path("/reports")
@CallTemplate(ReportsServiceCallTemplate.class)
public class RunReportsJaxrsService extends RemoteServiceWrapper<RunReportService> {
    @Resource(name = "runReportService")
    public void setRemoteService(RunReportService remoteService) {
        this.remoteService = remoteService;
    }

    @GET
    @Path("/{reportUnitURI: .+}/inputControls")
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    public Response getReportInputParameters(@PathParam("reportUnitURI") final String reportUnitURI, @Context final HttpServletRequest request) {
        // parameters map can be safely cast to Map<String, String[]>
        @SuppressWarnings("unchecked")
        final Map<String, String[]> rawParameters = request.getParameterMap();
        return internalGetReportInputParameters(reportUnitURI, null, rawParameters);
    }
    @POST
    @Path("/{reportUnitURI: .+}/inputControls")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response getReportInputParametersViaPost(@PathParam("reportUnitURI") String reportUnitURI, JSONObject jsonParameters){
        return internalGetReportInputParameters(reportUnitURI, null, JsonObjectParametersConverter.getParameterMapFromJson(jsonParameters));
    }
    
    protected Response internalGetReportInputParameters(final String reportUnitUri, final Set<String> inputControlIds, final Map<String, String[]> rawParameters){
        return callRemoteService(new ConcreteCaller<Response>() {
            @Override
            public Response call(RunReportService remoteService) throws RemoteException {
                List<ReportInputControl> inputControlsForReport = remoteService.getInputControlsForReport("/" + reportUnitUri, inputControlIds, rawParameters);
                if (inputControlsForReport != null && !inputControlsForReport.isEmpty())
                    return Response.ok(new ReportInputControlsListWrapper(inputControlsForReport)).build();
                else
                    return Response.status(Response.Status.NO_CONTENT).build();
            }
        });
    }

    @GET
    @Path("/{reportUnitURI: .+}/inputControls/{inputControlIds: [^;/]+(;[^;/]+)*}")
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    public Response getReportInputParametersForSpecifiedInputControls(
            @PathParam("reportUnitURI") String reportUnitURI,
            @PathParam("inputControlIds") PathSegment inputControlIds,
            @Context HttpServletRequest request) {
        // parameters map can be safely cast to Map<String, String[]>
        @SuppressWarnings("unchecked")
        final Map<String, String[]> parameterMap = request.getParameterMap();
        return internalGetReportInputParameters(reportUnitURI, getInputControlIdsFromPathSegment(inputControlIds), parameterMap);
    }

    @POST
    @Path("/{reportUnitURI: .+}/inputControls/{inputControlIds: [^;/]+(;[^;/]+)*}")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response getReportInputParametersForSpecifiedInputControlsViaPost(@PathParam("reportUnitURI") String reportUnitURI, @PathParam("inputControlIds") PathSegment inputControlIds, JSONObject jsonParameters){
        return internalGetReportInputParameters(reportUnitURI, getInputControlIdsFromPathSegment(inputControlIds), JsonObjectParametersConverter.getParameterMapFromJson(jsonParameters));
    }

    @GET
    @Path("/{reportUnitURI: .+}/inputControls/values")
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    public Response getInputControlsInitialValues(@PathParam("reportUnitURI") final String reportUnitURI, @Context final HttpServletRequest request) {
        // parameters map can be safely cast to Map<String, String[]>
        @SuppressWarnings("unchecked")
        final Map<String, String[]> parameterMap = request.getParameterMap();
        return internalGetInputControlsInitialValues(reportUnitURI, parameterMap);
    }

    @POST
    @Path("/{reportUnitURI: .+}/inputControls/values")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response getInputControlsInitialValuesViaPost(@PathParam("reportUnitURI") String reportUnitURI, JSONObject jsonParameters) {
        return internalGetInputControlsInitialValues(reportUnitURI, JsonObjectParametersConverter.getParameterMapFromJson(jsonParameters));
    }

    protected Response internalGetInputControlsInitialValues(final String reportUnitURI, final Map<String, String[]> parameters) {
        return callRemoteService(new ConcreteCaller<Response>() {
            public Response call(RunReportService remoteService) throws RemoteException {
                final String completeInputControlUri = "/" + reportUnitURI;
                List<InputControlState> values = remoteService.getValuesForInputControls(completeInputControlUri, null, parameters);
                if (values != null && !values.isEmpty())
                    return Response.ok(new InputControlStateListWrapper(values)).build();
                else
                    return Response.status(Response.Status.NO_CONTENT).entity("No input controls values found for the report /" + reportUnitURI).build();
            }
        });
    }

    @GET
    @Path("/{reportUnitURI: .+}/inputControls/{inputControlIds: [^;/]+(;[^;/]+)*}/values")
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    public Response getReportInputControlValues(
            @PathParam("reportUnitURI") final String reportUnitUri,
            @PathParam("inputControlIds") final PathSegment inputControlIds,
            @Context final HttpServletRequest request) {
        // parameters map can be safely cast to Map<String, String[]>
        @SuppressWarnings("unchecked")
        final Map<String, String[]> parameterMap = request.getParameterMap();
        return internalGetInputControlValues("/" + reportUnitUri, inputControlIds, parameterMap);
    }

    @POST
    @Path("/{reportUnitURI: .+}/inputControls/{inputControlIds: [^;/]+(;[^;/]+)*}/values")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response getReportInputControlValuesViaPost(
            @PathParam("reportUnitURI") final String reportUnitUri,
            @PathParam("inputControlIds") final PathSegment inputControlIds,
            JSONObject jsonParameters) {
        return internalGetInputControlValues("/" + reportUnitUri, inputControlIds, JsonObjectParametersConverter.getParameterMapFromJson(jsonParameters));
    }

    protected Response internalGetInputControlValues(
            final String reportUnitUri,
            final PathSegment inputControlIdsSegment,
            final Map<String, String[]> parameters) {
        return callRemoteService(new ConcreteCaller<Response>() {
            public Response call(RunReportService remoteService) throws RemoteException {
                List<InputControlState> values = remoteService.getValuesForInputControls(reportUnitUri, getInputControlIdsFromPathSegment(inputControlIdsSegment), parameters);
                if (values != null && !values.isEmpty())
                    return Response.ok(new InputControlStateListWrapper(values)).build();
                else
                    return Response.status(Response.Status.NO_CONTENT).entity("No input controls values found for the report /" + reportUnitUri).build();
            }
        });
    }

    protected Set<String> getInputControlIdsFromPathSegment(PathSegment inputControlIdsSegment){
        Set<String> inputControlIds = new HashSet<String>();
        inputControlIds.add(inputControlIdsSegment.getPath());
        for (String currentId : inputControlIdsSegment.getMatrixParameters().keySet())
            inputControlIds.add(currentId);
        return inputControlIds;
    }

    @GET
    @Path("/{reportUnitURI: .+}.{outputFormat}")
    public Response getReportOutputResource(
            @PathParam("reportUnitURI") final String reportUnitURI,
            @PathParam("outputFormat") final String outputFormat,
            @QueryParam("page") final Integer page,
            @QueryParam("transformerKey") final String transformerKey,
            @QueryParam("ignorePagination") @DefaultValue("false") final Boolean ignorePagination,
            @Context final HttpServletRequest request,
            @QueryParam("avoidCache") final Boolean avoidCache,
            @QueryParam("freshData") final Boolean freshData,
            @QueryParam("saveDataSnapshot") final Boolean saveDataSnapshot) {
        return callRemoteService(new ConcreteCaller<Response>() {
            public Response call(RunReportService remoteService) throws RemoteException {
                // parameters map can be safely cast to Map<String, String[]>
                @SuppressWarnings("unchecked")
                Map<String, String[]> parameterMap = request.getParameterMap();
                ReportOutputResource reportOutputResource = remoteService.getReportOutputFromRawParameters(
                        "/" + reportUnitURI, outputFormat, ignorePagination, page, transformerKey, parameterMap, avoidCache,
                        freshData, saveDataSnapshot);
                final Response.ResponseBuilder responseBuilder = Response.ok(reportOutputResource.getData(), reportOutputResource.getContentType());
                if(reportOutputResource.getFileName() != null)
                    responseBuilder.header("Content-Disposition", "attachment; filename=\"" + reportOutputResource.getFileName() + "\"");
                return responseBuilder.build();
            }
        });
    }

    @GET
    @Path("/{reportUnitURI: .+}/{cacheKey: \\d+}/items/{itemName}")
    public Response getReportItem(@PathParam("itemName") final String itemName, @PathParam("cacheKey") final Integer cacheKey) {
        return callRemoteService(new ConcreteCaller<Response>() {
            public Response call(RunReportService remoteService) throws RemoteException {
                ReportOutputResource reportOutputResource = remoteService.getReportItem(itemName, cacheKey);
                return Response.ok(reportOutputResource.getData(), reportOutputResource.getContentType()).build();
            }
        });
    }

    @GET
    public Response getReportsRuntimeInformation(
            @QueryParam("reportURI") final String reportURI,
            @QueryParam("jobID") final String jobID,
            @QueryParam("jobLabel") final String jobLabel,
            @QueryParam("userName") final String userName,
            @QueryParam("fireTimeFrom") final String fireTimeFrom,
            @QueryParam("fireTimeTo") final String fireTimeTo) {
        return callRemoteService(new ConcreteCaller<Response>() {
            public Response call(RunReportService remoteService) throws RemoteException {
                SchedulerReportExecutionStatusSearchCriteria criteria = null;
                if (StringUtils.isNotEmpty(reportURI)
                        || StringUtils.isNotEmpty(jobID)
                        || StringUtils.isNotEmpty(jobLabel)
                        || StringUtils.isNotEmpty(userName)
                        || StringUtils.isNotEmpty(fireTimeFrom)
                        || StringUtils.isNotEmpty(fireTimeTo)) {
                    criteria = new SchedulerReportExecutionStatusSearchCriteria();
                    criteria.setReportURI(StringUtils.isNotEmpty(reportURI) ? reportURI : null);
                    criteria.setJobLabel(StringUtils.isNotEmpty(jobLabel) ? jobLabel : null);
                    criteria.setUserName(StringUtils.isNotEmpty(userName) ? userName : null);
                    criteria.setJobID(StringUtils.isNotEmpty(jobID) ? Long.valueOf(jobID) : null);
                    final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mmZ");
                    try {
                        criteria.setFireTimeFrom(StringUtils.isNotEmpty(fireTimeFrom) ? simpleDateFormat.parse(fireTimeFrom) : null);
                    } catch (ParseException e) {
                        throw new IllegalParameterValueException("fireTimeFrom", fireTimeFrom);
                    }
                    try {
                        criteria.setFireTimeTo(StringUtils.isNotEmpty(fireTimeTo) ? simpleDateFormat.parse(fireTimeTo) : null);
                    } catch (ParseException e) {
                        throw new IllegalParameterValueException("fireTimeTo", fireTimeTo);
                    }
                }
                Set<ReportExecutionStatusInformation> currentlyRunningReports = remoteService.getCurrentlyRunningReports(criteria);
                if (currentlyRunningReports != null && !currentlyRunningReports.isEmpty())
                    return Response.ok(new ReportExecutionsSetWrapper(currentlyRunningReports)).build();
                else
                    return Response.status(Response.Status.NO_CONTENT).build();
            }
        });
    }

    @PUT
    @Path("/{executionId}/status")
    @Consumes({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    public Response cancelReportExecution(@PathParam("executionId") final String executionId, ReportExecutionCancellation cancellation) {
        Response response;
        if (cancellation != null && ReportExecutionCancellation.VALUE_CANCELLED.equals(cancellation.getValue()))
            response = callRemoteService(new ConcreteCaller<Response>() {
                public Response call(RunReportService remoteService) throws RemoteException {
                    remoteService.cancelReportExecution(executionId);
                    return remoteService.cancelReportExecution(executionId) ? Response.ok(new ReportExecutionCancellation()).build() : Response.status(Response.Status.NO_CONTENT).build();
                }
            });
        else
            response = Response.status(Response.Status.BAD_REQUEST).build();
        return response;
    }


}
