/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.jaspersoft.jasperserver.api.engine.scheduling.quartz;

import java.io.IOException;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.JobExecutionException;

import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRExporterParameter;
import net.sf.jasperreports.engine.JRExporter;
import net.sf.jasperreports.engine.export.JRHtmlExporter;
import net.sf.jasperreports.engine.export.JRHtmlExporterParameter;
import net.sf.jasperreports.engine.export.JRHyperlinkProducerFactory;

import com.jaspersoft.jasperserver.api.JSExceptionWrapper;
import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.engine.common.service.EngineService;
import com.jaspersoft.jasperserver.api.engine.jasperreports.util.HtmlExportUtil;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ContentResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.DataContainer;
import com.jaspersoft.jasperserver.api.metadata.common.domain.MemoryDataContainer;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;

/**
 * @author sanda zaharia (shertage@users.sourceforge.net)
 * @version $Id: HtmlReportOutput.java 22820 2012-03-27 13:49:36Z tdanciu $
 */
public class HtmlReportOutput extends AbstractReportOutput 
{

	private static final Log log = LogFactory.getLog(HtmlReportOutput.class);

    private boolean forceToUseHTMLExporter = false;

	public HtmlReportOutput()
	{
	}

    /*
     * set to enforce to use grid-base exporter for HTML
     * it is good for embedded report in email
     */
    public void setForceToUseHTMLExporter(boolean state) {
        forceToUseHTMLExporter = true;
    }

     /*
     * return whether it is set to use grid-base exporter for HTML
     */
    public boolean isForceToUseHTMLExporter() {
        return forceToUseHTMLExporter;
    }

	/** 
	 * @see com.jaspersoft.jasperserver.api.engine.scheduling.quartz.Output#getOutput()
	 */
	public ReportOutput getOutput(
			EngineService engineService, 
			ExecutionContext executionContext, 
			String reportUnitURI, 
			DataContainer htmlData,
			JRHyperlinkProducerFactory hyperlinkProducerFactory,
			RepositoryService repositoryService,
			JasperPrint jasperPrint, 
			String baseFilename,
			Locale locale,
			String characterEncoding) throws JobExecutionException
	{
		try {
			String filename = baseFilename + ".html";
			String childrenFolderName = null;

            if (repositoryService != null) childrenFolderName = repositoryService.getChildrenFolderName(filename);
			else childrenFolderName = "";
            JRExporter exporter = null;

            if (isForceToUseHTMLExporter()) {
                // enforce to use grid-base exporter (only use for embedded report in email)
                exporter = new JRHtmlExporter();
            } else {
                exporter = HtmlExportUtil.getHtmlExporter(null);
            }
		    exporter.setParameter(JRExporterParameter.JASPER_PRINT, jasperPrint);
			exporter.setParameter(JRExporterParameter.CHARACTER_ENCODING, characterEncoding);
			
			boolean close = true;
			OutputStream htmlDataOut = htmlData.getOutputStream();
			
			try {
				exporter.setParameter(JRExporterParameter.OUTPUT_STREAM, htmlDataOut);
				Map images = new HashMap();//TODO don't keep in memory
				exporter.setParameter(JRHtmlExporterParameter.IMAGES_MAP, images);
				if (!childrenFolderName.equals("")) {
                    exporter.setParameter(JRHtmlExporterParameter.IMAGES_URI, childrenFolderName + '/');
                } else {
                    exporter.setParameter(JRHtmlExporterParameter.IMAGES_URI, childrenFolderName);
                }
				if (hyperlinkProducerFactory != null) {
					exporter.setParameter(JRExporterParameter.HYPERLINK_PRODUCER_FACTORY, hyperlinkProducerFactory);
				}
				
				exporter.exportReport();

				close = false;
				htmlDataOut.close();

				ReportOutput htmlOutput = new ReportOutput(htmlData,
						ContentResource.TYPE_HTML, filename);

				for (Iterator it = images.entrySet().iterator(); it.hasNext();) {
					Map.Entry imageEntry = (Map.Entry) it.next();
					String imageName = (String) imageEntry.getKey();
					byte[] imageData = (byte[]) imageEntry.getValue();
					MemoryDataContainer imageDataContainer = new MemoryDataContainer(imageData);
					ReportOutput image = new ReportOutput(imageDataContainer,
							ContentResource.TYPE_IMAGE, imageName);
					htmlOutput.addChild(image);
				}

				return htmlOutput;
			} catch (IOException e) {
				throw new JSExceptionWrapper(e);
			} finally {
				if (close) {
					try {
						htmlDataOut.close();
					} catch (IOException e) {
						log.error("Error closing stream", e);
					}
				}
			}
		} catch (JRException e) {
			throw new JSExceptionWrapper(e);
		}
	}

}
