/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.war.cascade.handlers;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.util.HashMap;
import java.util.Map;

/**
 * @author Yaroslav.Kovalchyk
 * @version $Id: GenericTypeProcessorRegistry.java 24122 2012-06-11 15:57:20Z ykovalchyk $
 */
@Service
public class GenericTypeProcessorRegistry {
    @Autowired
    private ApplicationContext context;

    private volatile Map<Class<?>, Map<Class<?>, Object>> processors = new HashMap<Class<?>, Map<Class<?>, Object>>();

    public <T> T getTypeProcessor(Class<?> typeToProcess, Class<T> processorType) {
        return getTypeProcessor(typeToProcess, processorType, true);
    }

    @SuppressWarnings("unchecked")// casting to processorType is safe, see afterPropertiesSet()
    public <T> T getTypeProcessor(Class<?> typeToProcess, Class<T> processorType, Boolean exceptionIfNotConfigured) {
        Map<Class<?>, Object> concreteProcessors = processors.get(processorType);
        if (concreteProcessors == null)
            synchronized (processors) {
                concreteProcessors = processors.get(processorType);
                if (concreteProcessors == null) {
                    concreteProcessors = initializeProcessors(processorType);
                    processors.put(processorType, concreteProcessors);
                }
            }
        if (exceptionIfNotConfigured && !concreteProcessors.containsKey(typeToProcess))
            throw new IllegalStateException("Processor of type " + processorType.getName() + " for class " + typeToProcess.getName() + " not configured");
        return (T) concreteProcessors.get(typeToProcess);
    }

    protected Map<Class<?>, Object> initializeProcessors(Class<?> processorType) {
        Map<Class<?>, Object> processors = new HashMap<Class<?>, Object>();
        final String[] dataConvertersNames = context.getBeanNamesForType(processorType);
        if (dataConvertersNames != null && dataConvertersNames.length > 0)
            for (String dataConverterBeanName : dataConvertersNames) {
                Object currentDataConverter = context.getBean(dataConverterBeanName, processorType);
                Class<?> valueClass = null;
                Type[] genericInterfaces = currentDataConverter.getClass().getGenericInterfaces();
                if (genericInterfaces != null) {
                    if (genericInterfaces.length == 0) {
                        final Type genericSuperclass = currentDataConverter.getClass().getGenericSuperclass();
                        if (genericSuperclass instanceof ParameterizedType) {
                            Type genericSuperclassType = ((ParameterizedType) genericSuperclass).getActualTypeArguments()[0];
                            if (genericSuperclassType instanceof Class) {
                                valueClass = ((Class) genericSuperclassType);
                            }
                        }
                    } else {
                        for (Type currentType : genericInterfaces) {
                            if (currentType instanceof ParameterizedType && ((ParameterizedType) currentType).getRawType() == processorType)
                                valueClass = getValueClass(currentType);
                        }
                    }
                }
                if (valueClass == null)
                    throw new IllegalStateException("Unable to find generic type of bean '" + dataConverterBeanName + "' (bean class " + currentDataConverter.getClass() + ")");
                processors.put(valueClass, currentDataConverter);
            }
        return processors;
    }

    protected Class<?> getValueClass(Type type) {
        Class<?> valueClass = null;
        if (type instanceof ParameterizedType) {
            final Type nextType = ((ParameterizedType) type).getActualTypeArguments()[0];
            if (nextType instanceof WildcardType) {
                type = ((ParameterizedType) type).getRawType();
            } else {
                valueClass = getValueClass(nextType);
            }
        }
        if (type instanceof Class) {
            valueClass = (Class<?>) type;
        }
        if (type instanceof TypeVariable) {
            valueClass = getValueClass(((TypeVariable<?>) type).getBounds()[0]);
        }
        return valueClass;
    }
}
