package com.jaspersoft.jasperserver.war.cascade;

import com.jaspersoft.jasperserver.api.engine.common.service.SecurityContextProvider;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControlsContainer;
import com.jaspersoft.jasperserver.war.cascade.cache.ControlLogicCacheManager;
import com.jaspersoft.jasperserver.war.dto.InputControlState;
import com.jaspersoft.jasperserver.war.dto.ReportInputControl;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * @author Yaroslav.Kovalchyk
 * @version $Id: InputControlsLogicServiceImpl.java 22695 2012-03-21 13:55:32Z ykovalchyk $
 */
@Service("inputControlsLogicService")
public class InputControlsLogicServiceImpl implements InputControlsLogicService {
    @Resource
    private HashMap<String, ControlLogic<InputControlsContainer>> controlLogicRegestry;
    @Resource
    private HashMap<String, String> controlLogicUriMapping;
    @Resource
    private HashMap<String, String> controlLogicReportTypeMapping;
    @Resource
    private String defaultControlLogicName;
    @Resource
    private ControlLogicCacheManager controlLogicCacheManager;
    @Resource
    private CachedRepositoryService cachedRepositoryService;
    @Resource(name = "concreteSecurityContextProvider")
    protected SecurityContextProvider securityContextProvider;
    @Resource
    protected EvaluationEventsHandler defaultEvaluationEventsHandler;

    public Map<String, Object> getTypedParameters(
            String containerUri, final Map<String, String[]> requestParameters, final boolean skipValidation)
            throws CascadeResourceNotFoundException, InputControlsValidationException {

        return callControlLogic(containerUri, new ControLogicCaller<Map<String, Object>>() {
            @Override
            public Map<String, Object> callByContainerUri(ControlLogic<InputControlsContainer> controlLogic, String containerUri) throws CascadeResourceNotFoundException, InputControlsValidationException {
                return controlLogic.getTypedParameters(containerUri, requestParameters, skipValidation);
            }

            @Override
            public Map<String, Object> callByContainer(ControlLogic<InputControlsContainer> controlLogic, InputControlsContainer container) throws CascadeResourceNotFoundException, InputControlsValidationException {
                return controlLogic.getTypedParameters(container, requestParameters, skipValidation);
            }
        });
    }

    public Map<String, Object> getTypedParameters(
            String containerUri, final Map<String, String[]> requestParameters)
            throws CascadeResourceNotFoundException, InputControlsValidationException {

        return getTypedParameters(containerUri, requestParameters, false);
    }

    public Map<String, String[]> formatTypedParameters(
            String containerUri, final Map<String, Object> requestParameters) throws CascadeResourceNotFoundException, InputControlsValidationException {

        return callControlLogic(containerUri, new ControLogicCaller<Map<String, String[]>>() {
            @Override
            public Map<String, String[]> callByContainerUri(ControlLogic<InputControlsContainer> controlLogic, String containerUri) throws CascadeResourceNotFoundException, InputControlsValidationException {
                return controlLogic.formatTypedParameters(containerUri, requestParameters);
            }

            @Override
            public Map<String, String[]> callByContainer(ControlLogic<InputControlsContainer> controlLogic, InputControlsContainer container) throws CascadeResourceNotFoundException, InputControlsValidationException {
                return controlLogic.formatTypedParameters(container, requestParameters);
            }
        });
    }

    public List<ReportInputControl> getInputControlsStructure(String containerUri, final Set<String> inputControlIds) throws CascadeResourceNotFoundException {
        controlLogicCacheManager.clearCache();
        try {
            return callControlLogic(containerUri, new ControLogicCaller<List<ReportInputControl>>() {
                @Override
                public List<ReportInputControl> callByContainerUri(ControlLogic<InputControlsContainer> controlLogic, String containerUri) throws CascadeResourceNotFoundException, InputControlsValidationException {
                    return controlLogic.getInputControlsStructure(containerUri, inputControlIds);
                }

                @Override
                public List<ReportInputControl> callByContainer(ControlLogic<InputControlsContainer> controlLogic, InputControlsContainer container) throws CascadeResourceNotFoundException, InputControlsValidationException {
                    return controlLogic.getInputControlsStructure(container, inputControlIds);
                }
            });
        } catch (InputControlsValidationException e) {
            // shouldn't happen
            throw new RuntimeException("Input controls validation failed", e);
        }
    }

    public List<InputControlState> getValuesForInputControls(String containerUri, final Set<String> inputControlIds, final Map<String, String[]> parameters) throws CascadeResourceNotFoundException {
        List<InputControlState> states;
        defaultEvaluationEventsHandler.beforeEvaluation(containerUri, parameters, securityContextProvider.getContextUser());
        try {
            states = callControlLogic(containerUri, new ControLogicCaller<List<InputControlState>>() {
                @Override
                public List<InputControlState> callByContainerUri(ControlLogic<InputControlsContainer> controlLogic, String containerUri) throws CascadeResourceNotFoundException, InputControlsValidationException {
                    return controlLogic.getValuesForInputControlsFromRawData(containerUri, inputControlIds, parameters);
                }

                @Override
                public List<InputControlState> callByContainer(ControlLogic<InputControlsContainer> controlLogic, InputControlsContainer container) throws CascadeResourceNotFoundException, InputControlsValidationException {
                    return controlLogic.getValuesForInputControlsFromRawData(container, inputControlIds, parameters);
                }
            });
        } catch (InputControlsValidationException e) {
            //Should not happen
            throw new RuntimeException(e);
        }
        defaultEvaluationEventsHandler.afterEvaluation(containerUri, parameters, states, securityContextProvider.getContextUser());
        return states;
    }

    protected String getControlLogicAliasByContainerUri(String containerUri) {
        String controlLogicAlias = null;
        if (!controlLogicUriMapping.isEmpty()) {
            // try to find control logic alias in URI mapping
            for (String currentyUriMappingExpression : controlLogicUriMapping.keySet())
                if (Pattern.matches(currentyUriMappingExpression, containerUri)) {
                    controlLogicAlias = controlLogicUriMapping.get(currentyUriMappingExpression);
                    break;
                }
        }
        return controlLogicAlias;
    }

    protected <T> T callControlLogic(String containerUri, ControLogicCaller<T> caller) throws CascadeResourceNotFoundException, InputControlsValidationException {
        T result;
        String controlLogicAlias = getControlLogicAliasByContainerUri(containerUri);
        if (controlLogicAlias != null)
            result = caller.callByContainerUri(getControlLogicByAlias(controlLogicAlias), containerUri);
        else {
            final InputControlsContainer container = cachedRepositoryService.getResource(InputControlsContainer.class, containerUri);
            controlLogicAlias = controlLogicReportTypeMapping.get(container.getResourceType());
            if (controlLogicAlias == null)
                controlLogicAlias = defaultControlLogicName;
            result = caller.callByContainer(getControlLogicByAlias(controlLogicAlias), container);
        }
        return result;
    }

    protected ControlLogic<InputControlsContainer> getControlLogicByAlias(String controlLogicAlias) {
        final ControlLogic<InputControlsContainer> controlLogic = controlLogicRegestry.get(controlLogicAlias);
        if (controlLogic == null)
            throw new IllegalStateException("Control logic for alias '" + controlLogicAlias + "' not found");
        return controlLogic;
    }

    protected interface ControLogicCaller<T> {
        T callByContainerUri(ControlLogic<InputControlsContainer> controlLogic, String containerUri) throws CascadeResourceNotFoundException, InputControlsValidationException;

        T callByContainer(ControlLogic<InputControlsContainer> controlLogic, InputControlsContainer container) throws CascadeResourceNotFoundException, InputControlsValidationException;
    }
}
