/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.war.action;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.sf.jasperreports.engine.JRRuntimeException;
import net.sf.jasperreports.engine.ReportContext;
import net.sf.jasperreports.web.servlets.JasperPrintAccessor;
import net.sf.jasperreports.web.servlets.ReportExecutionStatus;
import net.sf.jasperreports.web.servlets.ReportPageStatus;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.View;
import org.springframework.web.servlet.mvc.multiaction.MultiActionController;

import com.jaspersoft.jasperserver.api.JSException;
import com.jaspersoft.jasperserver.api.JSShowOnlyErrorMessage;
import com.jaspersoft.jasperserver.api.engine.common.service.EngineService;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitResult;
import com.jaspersoft.jasperserver.api.engine.jasperreports.service.DataCacheProvider;
import com.jaspersoft.jasperserver.war.util.SessionObjectSerieAccessor;

/**
 * @author Lucian Chirita (lucianc@users.sourceforge.net)
 * @version $Id: ReportExecutionController.java 23675 2012-05-09 12:16:51Z lchirita $
 */
public class ReportExecutionController extends MultiActionController {

	private static final Log log = LogFactory.getLog(ViewReportAction.class);
	
	public static final String REPORT_EXECUTION_PREFIX = "flowReportExecution";
	public static final String REPORT_EXECUTION_ID_PREFIX = "flowReportExecutionId";
	
	private static final View NULL_VIEW = new View() {
		public String getContentType() {
			return null;
		}

		public void render(Map model, HttpServletRequest request,
				HttpServletResponse response) throws Exception {
			// NOP
		}
	};
	
	private EngineService engineService;
	private SessionObjectSerieAccessor jasperPrintAccessor;
	private DataCacheProvider dataCacheProvider; 
	
	public ModelAndView viewReportCancel(HttpServletRequest req, HttpServletResponse res) {
		String flowExecutionKey = req.getParameter("_flowExecutionKey");
		String sessionName = REPORT_EXECUTION_PREFIX + flowExecutionKey;
		ReportExecutionAttributes execution = 
			(ReportExecutionAttributes) req.getSession().getAttribute(sessionName);
		
		if (execution == null) {
			if (log.isDebugEnabled()) {
				log.debug("No report execution to cancel");
			}
		} else {
			boolean canceled = engineService.cancelExecution(execution.getRequestId());
			
			if (log.isDebugEnabled()) {
				log.debug("Report execution " + execution.getRequestId() 
						+ " cancel status: " + canceled);
			}
		}
		
		return new ModelAndView(NULL_VIEW);
	}

	public ModelAndView viewReportAsyncCancel(HttpServletRequest req, HttpServletResponse res) throws Exception {
		ReportUnitResult result = getReportResult(req);
		String requestId = result == null ? null : result.getRequestId();
		
		LinkedHashMap<String, Object> actionResult = new LinkedHashMap<String, Object>();
		if (requestId == null) {
			if (log.isDebugEnabled()) {
				log.debug("No async report execution to cancel");
			}
		} else {
			boolean canceled = engineService.cancelExecution(requestId);
			
			if (log.isDebugEnabled()) {
				log.debug("Report execution " + requestId 
						+ " cancel status: " + canceled);
			}
			
			JasperPrintAccessor resultPrintAccessor = result.getJasperPrintAccessor();
			try {
				// this will wait for the report to end
				resultPrintAccessor.getFinalJasperPrint();
			} catch (JRRuntimeException e) {
				// we don't need to handle the exception here, we're doing getReportStatus() below
			}
			
			putReportStatusResult(res, result, actionResult);
		}
		
		return new ModelAndView("json:result", Collections.singletonMap("result", actionResult));
	}

	protected ReportUnitResult getReportResult(HttpServletRequest req) {
		String jasperPrintName = req.getParameter("jasperPrintName");
		ReportUnitResult result = (ReportUnitResult) getJasperPrintAccessor().getObject(req, jasperPrintName);
		return result;
	}

	public ModelAndView viewReportPageUpdateCheck(HttpServletRequest req, HttpServletResponse res) throws Exception {
		ReportUnitResult reportResult = getReportResult(req);
		JasperPrintAccessor printAccessor = reportResult == null ? null : reportResult.getJasperPrintAccessor();
		if (printAccessor == null) {
			return null;
		}
		
		String pageIdxParam = req.getParameter("pageIndex");
		Integer pageIndex = pageIdxParam == null ? null : Integer.valueOf(pageIdxParam);
		String pageTimestampParam = req.getParameter("pageTimestamp");
		Long pageTimestamp = pageTimestampParam == null ? null : Long.valueOf(pageTimestampParam);
		
		if (log.isDebugEnabled()) {
			log.debug("report page update check for " + reportResult.getRequestId()
					+ ", pageIndex: " + pageIndex + ", pageTimestamp: " + pageTimestamp);
		}
		
		LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
		putReportStatusResult(res, reportResult, result);
		
		if (pageIndex != null && pageTimestamp != null) {
			ReportPageStatus pageStatus = printAccessor.pageStatus(pageIndex, pageTimestamp);
			boolean modified = pageStatus.hasModified();
			result.put("pageModified", modified);
			
			if (log.isDebugEnabled()) {
				log.debug("page modified " + modified);
			}
		}
		
		return new ModelAndView("json:result", Collections.singletonMap("result", result));
	}

	protected void putReportStatusResult(HttpServletResponse res,
			ReportUnitResult reportResult, LinkedHashMap<String, Object> result) throws Exception {
		JasperPrintAccessor printAccessor = reportResult.getJasperPrintAccessor();
		ReportExecutionStatus reportStatus = printAccessor.getReportStatus();
		result.put("lastPartialPageIndex", reportStatus.getCurrentPageCount() - 1);
		
		String status;
		switch (reportStatus.getStatus()) {
		case FINISHED:
			status = "finished";
			Integer totalPageCount = reportStatus.getTotalPageCount();
			result.put("lastPageIndex", totalPageCount - 1);
			
			ReportContext reportContext = reportResult.getReportContext();
			DataCacheProvider.SnapshotSaveStatus snapshotSaveStatus = 
					dataCacheProvider.getSnapshotSaveStatus(reportContext);
			if (snapshotSaveStatus != null) {
				result.put("snapshotSaveStatus", snapshotSaveStatus.toString());
			}
			
			if (log.isDebugEnabled()) {
				log.debug("report finished " + totalPageCount + " pages; snapshot status " + snapshotSaveStatus);
			}
			break;
		case ERROR:
			status = "error";
			handleReportUpdateError(res, reportStatus);
			break;
		case CANCELED:
			status = "canceled";
			
			if (log.isDebugEnabled()) {
				log.debug("report canceled");
			}
			break;
		case RUNNING:
		default:
			status = "running";
			
			if (log.isDebugEnabled()) {
				log.debug("report running");
			}
			break;
		}
		
		result.put("status", status);
	}

	protected void handleReportUpdateError(HttpServletResponse res, ReportExecutionStatus reportStatus) throws Exception {
		Throwable error = reportStatus.getError();
		if (log.isDebugEnabled()) {
			log.debug("report error " + error);// only message
		}
		// set a header so that the UI knows it's a report execution error
		res.setHeader("reportError", "true");
		// set as a header because we don't have other way to pass it
		res.setHeader("lastPartialPageIndex", Integer.toString(reportStatus.getCurrentPageCount() - 1));
		
		// throw an exception to get to the error page
		if (error instanceof Exception) {
			// copied from ViewReportAction.executeReport
			// note that the message is not localized
            int indexIO = ExceptionUtils.indexOfThrowable(error, IOException.class);
            if (indexIO != -1) {
                Exception sourceException = (Exception) ExceptionUtils.getThrowableList(error).get(indexIO);
				throw new JSShowOnlyErrorMessage(sourceException.getMessage());
            }
            
			throw (Exception) error;
		}
		
		throw new JSException("jsexception.view.report.error", error);
	}

	public EngineService getEngineService() {
		return engineService;
	}

	public void setEngineService(EngineService engineService) {
		this.engineService = engineService;
	}

	public SessionObjectSerieAccessor getJasperPrintAccessor() {
		return jasperPrintAccessor;
	}

	public void setJasperPrintAccessor(
			SessionObjectSerieAccessor jasperPrintAccessor) {
		this.jasperPrintAccessor = jasperPrintAccessor;
	}

	public DataCacheProvider getDataCacheProvider() {
		return dataCacheProvider;
	}

	public void setDataCacheProvider(DataCacheProvider dataCacheProvider) {
		this.dataCacheProvider = dataCacheProvider;
	}
	
}
