#include "VertexProgramNV.h"

#include <stdio.h>
#include <string.h>

#include "vpext.h"

// GPU Vertex Program - Based on algo by Erik Lindholm (nVidia) -------------------

GLuint vpid_; // Vertex Program id

const unsigned char vertexProgram0_[] = {
	// Standard vertex program header
	"!!VP1.0\n"
  // Transform stuff
  "DP4   o[HPOS].x, c[4], v[OPOS];"
  "DP4   o[HPOS].y, c[5], v[OPOS];"
  "DP4   o[HPOS].z, c[6], v[OPOS];"
  "DP4   o[HPOS].w, c[7], v[OPOS];"
  // Init stuff
  "MUL R0, v[0].xyzy, c[0].xxxy;"
  "MUL R4, R0.xzyw, c[0].xxww;"
};

const unsigned char vertexProgram1_[] = {
  "MAD R1, R0.xyxx, R0.xyyw, R4;"
  "ADD R0.xyw, R1.xzww, -R1.ywwz;"
};

const unsigned char vertexProgram2_[] = {
  "MOV o[COL0].xyz, R1.xyzw;"
	"END"
};

unsigned char vertexProgram_[sizeof(vertexProgram0_) + sizeof(vertexProgram1_)*128000 + sizeof(vertexProgram2_)];


VertexProgramNV::VertexProgramNV(int iters, int w, int h, double ax, double ay, double ex, double ey) : VertexProgram(iters, w, h, ax, ay, ex, ey) {
  isValid_ = initialize(iters, w, h, ax, ay, ex, ey);
}

VertexProgramNV::~VertexProgramNV(void) {
#if !defined(sgi)
  if (isValid_) glDisable(GL_VERTEX_PROGRAM_NV);
#endif
}

bool VertexProgramNV::isValid(void) {
  return isValid_;
}

bool VertexProgramNV::initialize(int iters, int w, int h, double ax, double ay, double ex, double ey) {
  return initializeGPU_VP(iters, w, h, ax, ay, ex, ey);
}

bool VertexProgramNV::initializeGPU_VP(int iters, int w, int h, double ax, double ay, double ex, double ey) {
#if defined(sgi)
  return false;
#else
  //Initialize NV_Vertex_Program
  if (!InitializeVertexProgramNV()) {
    return false;
  }
  else {
    // Generate Vertex Program for required number of iters
    char* vpPrt = (char*) vertexProgram_;
    int len = sprintf(vpPrt, "%s", vertexProgram0_);
    vpPrt += len;
    for (unsigned int i=0; i<iters; i++) {
      len = sprintf(vpPrt, "%s", vertexProgram1_);
      vpPrt += len;
    }
    len = sprintf(vpPrt, "%s", vertexProgram2_);
    vpPrt += len;
    //Generate a new vertex program
    //Note the similarity between vertex programs and textures.
    //They are essentially handled in the same manner, you bind the
    //vertex programs.  The gpu then loads it into resident memory.
    glGenProgramsNV(1, &vpid_);
    
    //Bind our program
    glBindProgramNV(GL_VERTEX_PROGRAM_NV, vpid_);
    
    //Load our program(enum, vertex program id, length of program, program text)
    glLoadProgramNV(GL_VERTEX_PROGRAM_NV, vpid_, strlen((char*)vertexProgram_), (unsigned char*)vertexProgram_);
    
    //Error checking for vertex programs this will return
    //GL_INVALID_OPERATION if the vertex program has any syntax errors.
    if (glGetError() == GL_INVALID_OPERATION) {
      printf("ERROR: glLoadProgramNV has returned an invalid operation!\n");
      return false;
    }

    // Place constants into GPU Constant Memory
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 0, 1.0f, -1.0f, 0.0f, 0.5f);
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 1, 0.2f, 1.0f, 0.4f, 0.0f);
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 2, 10.0f, 10.0f, 10.0f, 10.0f);
    this->prepareWorldSpace(w, h, ax, ay, ex, ey);
   
    // Enable Vertex Programs
    glEnable(GL_VERTEX_PROGRAM_NV);
  }
  return true;
#endif // sgi
}

void VertexProgramNV::prepareWorldSpace(int w, int h, double ax, double ay, double ex, double ey) {
  // Recalc real plane parameters
  double sx = (ex - ax) / ((double) w);
  //sy_ = (ey_ - ay_) / ((double) h_);
  double sy = sx;

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  setOrtho2D(ax, ex, ay+sy*(double)h, ay);
}

bool VertexProgramNV::setOrtho2D(GLdouble left, GLdouble right, GLdouble bottom, GLdouble top) {
  gluOrtho2D(left, right, bottom, top);
#if defined(sgi)
  return false;
#else
  glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 4, GL_PROJECTION, GL_IDENTITY_NV);
#endif
  return true;
};

bool VertexProgramNV::checkRequiredExtensions(void) {
  // TODO
  return true;
}
