// File: DTD2RELAXParser.java - last edit: 
// ʥХ祦 2000-05-07

// Copyright (c) 2000 by MURATA Makoto, NANBA Ryosuke.

package com.horobi.relax;

//// imports
import com.ibm.xml.parser.DTD;
import com.ibm.xml.parser.*;
import org.xml.sax.DocumentHandler;
import org.xml.sax.SAXException;
import org.xml.sax.AttributeList;
import org.xml.sax.helpers.AttributeListImpl;
import java.io.*;
import java.util.*;

/* DTD2RELAXParser convert DTD to RELAX and send SAX event for the
 * RELAX module.
 * @author MURATA Makoto, NANBA Ryosuke
 * @version $Id: DTD2RELAXParser.java,v 1.1 2000/05/07 15:05:38 rnanba Exp $ */
public class DTD2RELAXParser {
	public static final String RELAX_XMLNS =
		"http://www.xml.gr.jp/xmlns/relaxCore";

	DocumentHandler handler;
    Vector elementTypes; 
    Hashtable attLists; 
    DTD dtd;
	String finalElementType;

	AttributeListImpl attBuffer = new AttributeListImpl();
    boolean any;

	DTD2RELAXParser() {
		elementTypes = new Vector();
		attLists = new Hashtable();
		any = false;
    }

	void setFinalElementType(String aType) {
		finalElementType = aType;
	}
	public void setDocumentHandler(DocumentHandler aHandler) {
		handler = aHandler;
	}
	public void parse(String inputFilename) 
		throws SAXException, IOException, FileNotFoundException {
		FileInputStream is = new FileInputStream(inputFilename);
		dtd = new Parser(inputFilename).readDTDStream(is);
		if(dtd == null)
			throw new RuntimeException("no DTD.");
		startDocument();
		// <module moduleVersion="1.0" relaxCoreVersion="1.0"
		//         xmlns="$RELAX_XMLNS">
		addAttribute("moduleVersion", "CDATA", "1.0");
		addAttribute("relaxCoreVersion", "CDATA", "1.0");
		addAttribute("xmlns", "CDATA", RELAX_XMLNS);
		startElement("module");
		//   <interface>
		startElement("interface");
		// <export label="$final_etype"/>
		addAttribute("label", "CDATA", finalElementType);
		startElement("export");
		endElement("export");
		// </interface>
		endElement("interface");
		// <elementRule ...>...</elementRule> *
		Enumeration en = dtd.externalElements();
		while(en.hasMoreElements()) {
			Child cld = (Child)en.nextElement();
			switch (cld.getNodeType()) {
			case Child.ELEMENT_DECL:
				ElementDecl ed = (ElementDecl)cld;
				elementTypes.addElement(ed.getName());
				printRule(ed);
				break;
			case Child.ATTLIST:
				Attlist al = (Attlist)cld;
				Vector v = (Vector)attLists.get(al.getName());
				if(v == null)
					attLists.put(al.getName(), v = new Vector());
				v.addElement(al);
				break;
			}
		}
		printImplicitTags(); // <tag .../> *
		printTags(); // <tag ...><attribute ...>...</attribute>*</tag> *
		printAny(); // <hedgeRule label="any">...</hedgeRule>
		// </module>
		endElement("module");
		endDocument();
    }

    void printAny() throws SAXException {
		if(any){
			// <hedgeRule label="any">
			addAttribute("label", "CDATA", "any");
			startElement("hedgeRule");
			// <choice>
			startElement("choice");
			for(Enumeration e=elementTypes.elements(); e.hasMoreElements();){
				String elem_name = (String)e.nextElement();
				// <ref label="$elm_name"/>
				addAttribute("label", "CDATA", elem_name);
				startElement("ref");
				endElement("ref");
			}
			// </choice>
			endElement("choice");
			// </hedgeRule>
			endElement("hedgeRule");
		}
    }

    void printImplicitTags() throws SAXException {
		for(Enumeration e=elementTypes.elements(); e.hasMoreElements(); ) {
			String elem_name = (String)e.nextElement();
			if (!(attLists.containsKey(elem_name))) {
				// <tag name="$elem_name"/>
				addAttribute("name", "CDATA", elem_name);
				startElement("tag");
				endElement("tag");
			}
		}
    }

    void printContentModel(CMNode cmnd, String occurrenceIndicator)
		throws SAXException {
		if(cmnd instanceof CMLeaf){
			CMLeaf cmlf = (CMLeaf)cmnd;
			printContentModel(cmlf, occurrenceIndicator);
		}else if (isMixedContent(cmnd)){
			// <mixed>
			startElement("mixed");
			// <choice occurs="*">
			addAttribute("occurs", "CDATA", "*");
			startElement("choice");
			// ...
			printMixedContent(cmnd);
			// </choice>
			endElement("choice");
			// </mixed>
			endElement("mixed");
		}else if (cmnd instanceof CM1op){
			CM1op cm1 = (CM1op) cmnd;
			printContentModel(cm1);
		}
		else if (cmnd instanceof CM2op) {
			CM2op cm2 = (CM2op) cmnd;
			printContentModel(cm2, occurrenceIndicator);
		}else
			throw new SAXException("invalid content model");
    }

    boolean isMixedContent(CMNode cmnd) {
		if(cmnd instanceof CMLeaf){
			CMLeaf cmlf = (CMLeaf)cmnd;
			if(cmlf.getName() == "#PCDATA"){
				return true;
			}else
				return false;
		}else if(cmnd instanceof CM2op){
			CM2op cm2 = (CM2op) cmnd;
			if(cm2.getType() == '|'){
				return isMixedContent(cm2.getLeft());
			}else
				return false;
		}else
			return false;
    }

    void printContentModelInChoice(CMNode cmnd, String occurrenceIndicator)
		throws SAXException {
		if(cmnd instanceof CM2op){
			CM2op cm2 = (CM2op) cmnd;
			printContentModelInChoice(cm2, occurrenceIndicator);
			return;
		}
		printContentModel(cmnd, occurrenceIndicator);
    }

    void printContentModelInSequence(CMNode cmnd, String occurrenceIndicator)
		throws SAXException {
		if(cmnd instanceof CM2op){
			CM2op cm2 = (CM2op) cmnd;
			printContentModelInSequence(cm2, occurrenceIndicator);
			return;
		}
		printContentModel(cmnd, occurrenceIndicator);
    }

    void printContentModel(CM1op cm1) throws SAXException {
		String occurrenceIndicator = "";
		if(cm1.getType() == '*'){
			occurrenceIndicator = "*";
		}else if(cm1.getType() == '+'){
			occurrenceIndicator = "+";
		}else if(cm1.getType() == '?'){
			occurrenceIndicator = "?";
		}else{
			throw new SAXException("henda printContentModel(CM1op");
		}
		printContentModel(cm1.getNode(), occurrenceIndicator);
    }

    void printContentModel(CMLeaf cmlf, String occurrenceIndicator) 
		throws SAXException {
		String cmlfname = cmlf.getName();
		// <ref label="$cmlfname" occurs="$occurrenceIndicator"/>
		addAttribute("label", "CDATA", cmlfname);
		if(occurrenceIndicator.length() > 0)
			addAttribute("occurs", "CDATA", occurrenceIndicator);
		startElement("ref");
		endElement("ref");
    }

    void printContentModel(CM2op cm2, String occurrenceIndicator)
		throws SAXException {
		if(cm2.getType() == '|'){
			// <choice occurs="$occurrenceIndicator">
			if(occurrenceIndicator.length() > 0)
				addAttribute("occurs", "CDATA", occurrenceIndicator);
			startElement("choice");
			// ...
			printContentModelInChoice(cm2.getLeft(), "");
			printContentModel(cm2.getRight(), "");
			// </choice>
			endElement("choice");
		}else if(cm2.getType() == ','){
			// <sequence occurs="$occurrenceIndicator">
			if(occurrenceIndicator.length() > 0)
				addAttribute("occurs", "CDATA", occurrenceIndicator);
			startElement("sequence");
			// ...
			printContentModelInSequence(cm2.getLeft(), "");
			printContentModel(cm2.getRight(), "");
			// </sequence>
			endElement("sequence");
		}else
			throw new SAXException("henda printContentModel(CM2op");
    }

    void printContentModelInChoice(CM2op cm2, String occurrenceIndicator)
		throws SAXException {
		if(cm2.getType() == '|'){
			printContentModelInChoice(cm2.getLeft(), "");
			printContentModelInChoice(cm2.getRight(), "");
		}else
			printContentModel(cm2, occurrenceIndicator);
    }

    void printMixedContent(CMNode cmnd) throws SAXException {
		if(cmnd instanceof CM2op){
			CM2op cm2 = (CM2op) cmnd;
			printMixedContent(cm2.getLeft());
			printMixedContent(cm2.getRight());
		}else if(cmnd instanceof CMLeaf){
			CMLeaf cmlf = (CMLeaf) cmnd;
			if(cmlf.getName() != "#PCDATA")
				printContentModel(cmlf, "");
		}
    }

    void printContentModelInSequence(CM2op cm2, String occurrenceIndicator)
		throws SAXException {
		if(cm2.getType() == ','){
			printContentModelInSequence(cm2.getLeft(), "");
			printContentModelInSequence(cm2.getRight(), "");
		}else
			printContentModel(cm2, occurrenceIndicator);
    }

    void printRule(ElementDecl ed) throws SAXException {
		String edname = ed.getName();
		// <elementRule role="$edname">
		addAttribute("role", "CDATA", edname);
		//startElement("elementRule");
		//printContentModel(ed);
	    switch (dtd.getContentType(edname)) {
	    case 0: 
			System.err.println("???");
			return; //This should not happen!
	    case ElementDecl.EMPTY:
			startElement("elementRule");
			// <empty/>
			startElement("empty");
			endElement("empty");
			break;
	    case ElementDecl.ANY:
			any = true;
			startElement("elementRule");
			// <mixed><hedgeRef label="any" occurs="*"/></mixed>
			startElement("mixed");
			addAttribute("label", "CDATA", "any");
			addAttribute("occurs", "CDATA", "*");
			startElement("hedgeRef");
			endElement("hedgeRef");
			endElement("mixed");
			break;
	    case ElementDecl.MODEL_GROUP:
			CMNode cmnd = dtd.getContentModel(edname).getContentModelNode();
			if(isPCDATA(cmnd)){
				addAttribute("type", "NMTOKEN", "string");
				startElement("elementRule");
			}else{
				startElement("elementRule");
				printContentModel(cmnd, "");
			}
	    }
		// </elementRule>
		endElement("elementRule");
    }

	void printTags() throws SAXException {
		int n = elementTypes.size();
		for(int i=0; i<n; i++){
			String edname = (String)elementTypes.elementAt(i);
			Vector v = (Vector)attLists.get(elementTypes.elementAt(i));
			if(v != null)
				printATag(edname, v);
		}
	}

    void printATag(String edname, Vector atts) throws SAXException {
		// <tag name="$edname">
		addAttribute("name", "CDATA", edname);
		startElement("tag");
		// ...
		int n = atts.size();
		for(int i=0; i<n; i++){
			Attlist al = (Attlist)atts.elementAt(i);
			int m = al.size();
			for(int j=0; j<m; j++){
				AttDef attdef = al.elementAt(j);
				printAttDef(attdef);
			}
		}
		// </tag>
		endElement("tag");
    }

    String getDatatype(AttDef attdef) {
		switch (attdef.getDeclaredType()) {
		case AttDef.CDATA:
			return "string";
		case AttDef.ENTITIES:
			return "ENTITIES";
		case AttDef.ENTITY:
			return "ENTITY";
		case AttDef.ID:
			return "ID";
		case AttDef.IDREF:
			return "IDREF";
		case AttDef.IDREFS:
			return "IDREFS";
		case AttDef.NAME_TOKEN_GROUP:
			return "NMTOKEN";
		case AttDef.NMTOKEN:
			return "NMTOKEN";
		case AttDef.NMTOKENS:
			return "NMTOKENS";
		case AttDef.NOTATION:
			return "NOTATION";
		}
		return "";
    }

    void printEnumeration(AttDef attdef) throws SAXException {
		Enumeration en = attdef.elements();
		while(en.hasMoreElements()) {
			String val = (String)en.nextElement();
			// <enumeration value="$val"/>
			addAttribute("value", "CDATE", val); // ?
			startElement("enumeration");
			endElement("enumeration");
		}
    }

    void printAttDef(AttDef attdef) throws SAXException {
		// <attribute name="$attName" [required="true"] [type="..."]>
		addAttribute("name", "CDATA", attdef.getName());
		switch(attdef.getDefaultType()){
		case AttDef.REQUIRED:
			addAttribute("required", "NMTOKEN", "true");
			break;
		case AttDef.IMPLIED:
		case AttDef.NOFIXED:
		case AttDef.FIXED:
			break;
		}
		String type = getDatatype(attdef);
		if(type.length() > 0)
			addAttribute("type", "NMTOKEN", type);
		startElement("attribute");
		if(attdef.getDeclaredType() == AttDef.NAME_TOKEN_GROUP){
			// <enumeration ...> *
			printEnumeration(attdef);
		}
		endElement("attribute");
    }
	
	boolean isPCDATA(CMNode cmnd) {
		String pcdataname = "#PCDATA";
		if(cmnd instanceof CMLeaf){
			String cmlf_name = ((CMLeaf)cmnd).getName();
			if(pcdataname.equals(cmlf_name))
				return true;
		}else if(cmnd instanceof CM1op){
			CM1op cm1op = (CM1op)cmnd;
			CMNode cmnd_child = cm1op.getNode();
			if(cm1op.getType() == '*' && cmnd_child instanceof CMLeaf){
				String cmlf_child_name = ((CMLeaf)cmnd_child).getName();
				if(pcdataname.equals(cmlf_child_name))
					return true;
			}
		}
		return false;
    }
	
	protected void addAttribute(String aname, String type, String value) {
		attBuffer.addAttribute(aname, type, value);
	}
	protected void startElement(String ename) throws SAXException {
		if(handler != null)
			handler.startElement(ename, attBuffer);
		attBuffer.clear();
	}
	protected void endElement(String ename) throws SAXException {
		if(handler != null)
			handler.endElement(ename);
	}
	protected void startDocument() throws SAXException {
		if(handler != null)
			handler.startDocument();
	}
	protected void endDocument() throws SAXException {
		if(handler != null)
			handler.endDocument();
	}

}
