/*
 * Created on 6-Sept-2003
 * Created by Olivier
 * Copyright (C) 2004 Aelitis, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * AELITIS, SARL au capital de 30,000 euros
 * 8 Allee Lenotre, La Grille Royale, 78600 Le Mesnil le Roi, France.
 *
 */

package org.gudy.azureus2.ui.swt.views;

import java.text.Collator;
import java.util.Locale;
import java.util.StringTokenizer;

import org.eclipse.swt.SWT; 
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Text;

import java.io.*;
import java.util.Calendar;
import java.util.GregorianCalendar;

import org.gudy.azureus2.irc.IrcClient;
import org.gudy.azureus2.irc.IrcListener;

import org.gudy.azureus2.plugins.*;
import org.gudy.azureus2.plugins.utils.*;
import org.gudy.azureus2.ui.swt.plugins.UISWTView;

/**
 * @author Olivier
 * 
 */
public class 
SWTIrcView 
	implements IrcListener 
{
	PluginInterface	plugin_interface;
	PluginConfig plugin_config;
	LocaleUtilities locale_utils;
	
	Display display;
	Composite cIrc;
	ConsoleText consoleText;
	List users;
	Label userSumUp;
	Text inputField;
	Color[] colors;
	FileOutputStream logOut;

	IrcClient client;
	boolean newMessage;
	private boolean	blink;
	
	private String lastPrivate;

	public 
	SWTIrcView(
		PluginInterface _plugin_interface)
	{
		plugin_interface	= _plugin_interface;
				
		plugin_config	= _plugin_interface.getPluginconfig();
				
		locale_utils = plugin_interface.getUtilities().getLocaleUtilities();				
	}

  public void initialize(Composite composite) {
    display = composite.getDisplay();
    cIrc = new Composite(composite,SWT.NULL);
    GridLayout layout = new GridLayout();
    layout.numColumns = 2;
    layout.makeColumnsEqualWidth = false;
    cIrc.setLayout(layout);
	GridData gridData = new GridData(GridData.FILL_BOTH);
	cIrc.setLayoutData(gridData);
	
    consoleText = new ConsoleText(cIrc) {
    	
	      public void hyperlinkSelected(String link)
	      {
	    	  if (link.startsWith(" #")) {
	    		  
	    		  sendMessage("/join " + link);
	    		  
	    	  } else if (link.startsWith("magnet") || (link.indexOf(".torrent")!= -1 )) {
	    		  
	    		  new URLBrowser( plugin_interface, link);
	    		  
	    	  } else {
	    		  
	    		  new URLBrowser( link );
	
	    	  }
	      }
      
    };
    
    consoleText.addHyperlinkStyle();
    gridData = new GridData(GridData.FILL_BOTH | GridData.CENTER);
    gridData.grabExcessHorizontalSpace = true;
    consoleText.setLayoutData(gridData);
    users = new List(cIrc, SWT.MULTI | SWT.BORDER | SWT.V_SCROLL );
    gridData = new GridData(GridData.FILL_VERTICAL | GridData.HORIZONTAL_ALIGN_END | GridData.END);
    gridData.widthHint = 120;
    users.setLayoutData(gridData);
    inputField = new Text(cIrc, SWT.BORDER);
    gridData = new GridData(GridData.FILL_HORIZONTAL);
    //gridData.horizontalSpan = 2;
    inputField.setLayoutData(gridData);
    inputField.addKeyListener(new KeyAdapter() {
      /* (non-Javadoc)
       * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
       */
      public void keyReleased(KeyEvent event) {
        if (event.keyCode == 13) {
          String text = inputField.getText();
          inputField.setText("");
          sendMessage(text);
        }
      }
    });
    
    userSumUp = new Label(cIrc, SWT.NONE);
    gridData = new GridData(GridData.HORIZONTAL_ALIGN_END);
    gridData.widthHint = 120;
    userSumUp.setLayoutData(gridData);
    colors = new Color[5];
    colors[0] = new Color(display, new RGB(169,212,254 ));
    colors[1] = new Color(display, new RGB(198,226,255 ));
    colors[2] = new Color(display, new RGB(226,240,255 ));
    colors[3] = new Color(display, new RGB(255,192,192));
    colors[4] = new Color(display, new RGB(255,170,170));

    client = new IrcClient(plugin_interface,this);
    consoleText.addStyle(nickHighlight(client.getUserName()), null, colors[3],
      0);
//client.setVerbose(true);
  }

  protected void
  refresh(
	UISWTView		view )
  {
	   String title = "";
	   	   
	   if ( newMessage ){
		   
		  blink = !blink;
		   
		  title += blink?"!":" " ;
	   }
	   
	   title += locale_utils.getLocalisedMessageText("IrcView.title.short");
	  
	   if ( client != null){
		   
		   title += " " + client.getChannel() + " on " + client.getSrvName();
	   }     
	   
	   view.setTitle( title );
  }

  public void 
  focusGained() 
  {
    newMessage = false;
  
    inputField.setFocus();
  }


  public void delete() {
    Thread t = new Thread() {
      public void run() {
         client.close();
      }
    };
    t.setDaemon(true);
    t.start();
    
    if ( colors != null ){
    	for (int i=0;i<colors.length;i++){
    		if ( !colors[i].isDisposed()){
    			colors[i].dispose();
    		}
    	}
    }
  }

  public void messageReceived(String sender, String message) {
	doLog(1, "<" + sender + "> " + message);
    newMessage = true;
  }

  public void systemMessage(String message) {
    doLog(2, message);
  }

  private void doLog(int color, String text)
  {
	consoleText.append(color, text);
    if( plugin_config.getPluginBooleanParameter(IrcClient.CONFIG_IRC_LOG) ) {
		try
		{
			File log = new File("IRC_Log.htm");
			if (log.exists() == false) {
				logOut = new FileOutputStream ("IRC_log.htm", true);
				new PrintStream(logOut).print("<span style=\"font-family:Verdana,sans-serif;font-size:small;\"><h4>Azureus IRC log</h4><br>");
				logOut.close();
	    	}
	    	logOut = new FileOutputStream ("IRC_log.htm", true);
        	Calendar now = GregorianCalendar.getInstance();
        	String endText = "  " + text;
        	endText = endText.replace("<", "&lt;");
        	endText = endText.replace(">", "&gt;");
        	endText = endText + "<br>";
        	if (endText.contains(client.getUserName())) {
        		endText = "<strong>" + endText + "</strong>";
        		//endText = endText.replace(client.getUserName(), "<strong>" + client.getUserName() + "</strong>");
        	}
        	if ((endText.contains(locale_utils.getLocalisedMessageText("IrcClient.hasleft"))) || endText.contains(locale_utils.getLocalisedMessageText("IrcClient.hasjoined")))
        	{
        		endText = "<span style=\"color:#999999;\"><em>" + endText + "</em></span>";
        	}
        	if (endText.contains("  " + locale_utils.getLocalisedMessageText("IrcView.noticefrom") + " -") || endText.contains("  " + locale_utils.getLocalisedMessageText("IrcView.privatefrom") + " *") || endText.contains("  " + locale_utils.getLocalisedMessageText("IrcView.privateto") + " *")) {
        		endText = "<span style=\"color:#FF0000;\">" + endText + "</span>";
        	}
        	if (endText.contains("  " + locale_utils.getLocalisedMessageText("IrcClient.topicforchannel") + " #")) {
        		endText = "<span style=\"color:#008000;\">" + endText + "</span>";
        	}
        	endText = "[" + format(now.get(Calendar.HOUR_OF_DAY)) + ":" + format(now.get(Calendar.MINUTE)) + ":" + format(now.get(Calendar.SECOND)) + "]" + endText;
        	new PrintStream(logOut).print(endText);
	    	logOut.close();
		}
		catch (IOException e)
		{
			System.err.println ("Unable to write to file");
		}
	 }
  }
  private String format(int n)
  {
	return (n < 10) ? "0" + n : "" + n;
  }

  /* (non-Javadoc)
   * @see org.gudy.azureus2.irc.IrcListener#action(java.lang.String, java.lang.String)
   */
  public void action(String sender, String action) {
    doLog(0, sender + " " + action);
  }

  /* (non-Javadoc)
   * @see org.gudy.azureus2.irc.IrcListener#clientEntered(java.lang.String)
   */
  public void clientEntered(final String client) {
    if (display == null || display.isDisposed())
      return;
    display.asyncExec(new Runnable() {
      /* (non-Javadoc)
       * @see java.lang.Runnable#run()
       */
      public void run() {
        if (users != null && !users.isDisposed()) {
          int index = users.indexOf(client);
          if (index == -1) {
            Collator collator = Collator.getInstance(Locale.getDefault());
            String items[] = users.getItems();
            int i = 0;
            for (; i < items.length; i++) {
              if (collator.compare(client, items[i]) < 0) {
                users.add(client, i);
                break;
              }
            }
            if (i == items.length) {
              users.add(client);
            }
            int nbUsers = users.getItemCount();
            if (userSumUp != null && !userSumUp.isDisposed()) {
              userSumUp.setText(nbUsers + " " + locale_utils.getLocalisedMessageText("IrcView.clientsconnected"));
            }
          }
        }
      }
    });
  }

  /* (non-Javadoc)
   * @see org.gudy.azureus2.irc.IrcListener#clientExited(java.lang.String)
   */
  public void clientExited(final String client) {
    if (display == null || display.isDisposed())
      return;
    display.asyncExec(new Runnable() {
      /* (non-Javadoc)
       * @see java.lang.Runnable#run()
       */
      public void run() {
        if (users != null && !users.isDisposed()) {
          int index = users.indexOf(client);
          if (index != -1) {
            users.remove(index);
          }
          int nbUsers = users.getItemCount();
          if (userSumUp != null && !userSumUp.isDisposed()) {
            userSumUp.setText(nbUsers + " " + locale_utils.getLocalisedMessageText("IrcView.clientsconnected"));
          }
        }
      }
    });
  }
  public void allExited() {
	    if (display == null || display.isDisposed())
	      return;
	    display.asyncExec(new Runnable() {
	      /* (non-Javadoc)
	       * @see java.lang.Runnable#run()
	       */
	      public void run() {
	        if (users != null && !users.isDisposed()) {
	            users.removeAll();
	          int nbUsers = 0;
	          if (userSumUp != null && !userSumUp.isDisposed()) {
	            userSumUp.setText(nbUsers + " " + locale_utils.getLocalisedMessageText("IrcView.clientsconnected"));
	          }
	        }
	      }
	    });
	  }
  /**
   * Regular expression for matching the nick for highlighting. This can be a
   * partial nick.
   */
  private String nickHighlight(String nick)
  {
    return "(?i)(?<!<)\\b" + nick + "\\w*";
  }

  private void sendMessage(String text) {
    if (text.equals(""))
      return;  
    if (text.startsWith("/")) {
      if(text.equals("/help")) {
        doLog(0,locale_utils.getLocalisedMessageText("IrcView.help"));
      } else if (text.startsWith("/nick ") || text.startsWith("/name ")) {
        String newNick = text.substring(6).trim();
	consoleText.addStyle(nickHighlight(client.getUserName()), null, null,
	  0);
	consoleText.addStyle(nickHighlight(newNick), null, colors[3], 0);
        client.setUserName(newNick);
      } else if (text.startsWith("/me ")) {
        String action = text.substring(4).trim();
        client.sendAction(action);
        action(client.getUserName(), action);
      } else if(text.startsWith("/msg ") || text.startsWith("/to ")) {
        StringTokenizer st = new StringTokenizer(text," ");
        st.nextToken();
        try {
        String target = st.nextToken();
        String message = "";
        while(st.hasMoreElements()) {
          message += st.nextElement() + " ";
        }
        client.sendMessage(target,message);
        doLog(4,locale_utils.getLocalisedMessageText("IrcView.privateto") + " *" + target + "* " + message);
        } catch(Exception e) {
          doLog(0,locale_utils.getLocalisedMessageText("IrcView.errormsg"));
        } 
      } else if(text.startsWith("/r ")) {
        if(lastPrivate != null) {
          String message = text.substring(3).trim();
          client.sendMessage(lastPrivate,message);
          doLog(4,locale_utils.getLocalisedMessageText("IrcView.privateto") + " *" + lastPrivate + "* " + message);
        }
      } 
      else if(text.startsWith("/join "))
        {
          String channel = text.substring(6).trim();
          client.changeChannel(channel);
        }
     
      else {
        systemMessage(locale_utils.getLocalisedMessageText("IrcView.actionnotsupported"));
      }
    }
    else {
      client.sendMessage(text);
    }
  }

  /* (non-Javadoc)
   * @see org.gudy.azureus2.irc.IrcListener#notice(java.lang.String, java.lang.String)
   */
  public void notice(String sender, String message) {
    doLog(3,locale_utils.getLocalisedMessageText("IrcView.noticefrom") + " -" + sender + "- " + message);
    newMessage = true;
    lastPrivate = sender;
  }

  /* (non-Javadoc)
   * @see org.gudy.azureus2.irc.IrcListener#privateMessage(java.lang.String, java.lang.String)
   */
  public void privateMessage(String sender, String message) {
    if (message.startsWith("/shove") && client.isChanelOp(sender))
    {
      String st[] = message.split(" ");
      sendMessage("/join " + st[1]);
      message = message.substring(st[1].length() + 7);
    }
    else
    {
      lastPrivate = sender;
      newMessage = true;
    }
    doLog(4,locale_utils.getLocalisedMessageText("IrcView.privatefrom") + " *" + sender + "* " + message);
  }

}
