#!/usr/bin/python -O

from os.path import join
from unittest import TestLoader, TestCase, TestSuite, TextTestRunner

from unittestplus.mock import Mock, patch
from unittestplus.run import (
    combine, get_all_tests, run,
    _get_testcases, _get_modules, _path_to_module, 
)
from unittestplus.testcaseplus import TestCasePlus


class CombineTest(TestCasePlus):
    # pylint: disable-msg=R0904
    # Too many public methods: we're a subclass of TestCase

    def setUp(self):
        self.test_ids = []
        outer_self = self

        class ATest(TestCase):
            def testOne(self):
                outer_self.test_ids.append(1)
            def testTwo(self):
                outer_self.test_ids.append(2)

        class BTest(TestCase):
            def testThree(self):
                outer_self.test_ids.append(3)

        self.a_test = ATest
        self.b_test = BTest


    def assert_combine_behaviour(self, tests, exp_test_count, exp_test_ids):

        combined = combine(*tests)

        self.assertTrue(isinstance(combined, TestSuite),
            "should return a TestSuite")
        num_test_cases = combined.countTestCases()
        self.assertEquals(num_test_cases, exp_test_count,
            "should have %d test cases, had %s" % (
                exp_test_count, num_test_cases))

        class NullStream(object):
            write = lambda *_: None
            flush = lambda *_: None

        TextTestRunner(stream=NullStream()).run(combined)
        self.assertEquals(self.test_ids, exp_test_ids,
            "combined suite was %s, should be %s" % (
                self.test_ids, exp_test_ids))


    def test_combine_nothing(self):
        self.assert_combine_behaviour([], 0, [])


    def test_combine_classes(self):
        self.assert_combine_behaviour([self.a_test, self.b_test], 3, [1, 2, 3])


    def test_combine_suites(self):
        loader = TestLoader()
        suite1 = loader.loadTestsFromTestCase(self.a_test)
        suite2 = loader.loadTestsFromTestCase(self.b_test)
        self.assert_combine_behaviour([suite1, suite2], 3, [1, 2, 3])


    def test_combine_classes_and_suites(self):
        loader = TestLoader()
        suite1 = loader.loadTestsFromTestCase(self.a_test)
        suite2 = loader.loadTestsFromTestCase(self.b_test)
        many_tests = [self.a_test, suite1, self.b_test, suite2]
        self.assert_combine_behaviour(many_tests, 6, [1, 2, 1, 2, 3, 3])


    def test_combine_raises_on_nontests(self):
        self.assertRaises(lambda: combine(object()), TypeError)
        self.assertRaises(lambda: combine(object), TypeError)


class RunFuncTest(TestCasePlus):
    # pylint: disable-msg=R0904
    # Too many public methods: we're a subclass of TestCase

    class ClassUnderTest(TestCase):
        def testAlwaysPass(self):
            pass

    @patch('unittestplus.run.TextTestRunner')
    def test_run_creates_testrunner(self, mock_runner):

        run(self.ClassUnderTest, verbosity='verbosity')
        self.assertEquals(
            mock_runner.call_args,
            ((), dict(verbosity='verbosity', descriptions=1)))
        
    def test_run_raises_on_junk(self):
        self.assertRaises(
            lambda: run(object()),
            TypeError,
        )


class FindTest(TestCasePlus):
    # pylint: disable-msg=R0904
    # Too many public methods: we're a subclass of TestCase

    def test_get_modules(self):
        path = join('unittestplus', 'tests', 'testdata', 'walk')
        actual = _get_modules(path)
        actual_names = [module.__name__[len(path)+1:] for module in actual]
        expected = [
            r'module',
            r'package.module1',
            r'package.module2',
            r'package.module3',
            r'package.package1.module1',
            r'package.package2.module2',
            r'package.package3.module3',
        ]

        self.assertEquals(set(actual_names), set(expected))


    def test_path_to_module(self):
        actual = _path_to_module(join(
            'unittestplus', 'tests', 'testdata', 'walk', 'module.py')
        )
        from unittestplus.tests.testdata.walk import module
        self.assertTrue(actual is module)
        

    def test_get_testcases(self):
        from unittestplus.tests.testdata.get_tests import testmodule
        from unittestplus.tests.testdata.get_tests.testmodule import (
            Test1, Test2)
        actual = _get_testcases(testmodule)
        self.assertEquals(actual, [Test1, Test2])


    @patch('unittestplus.run._get_modules')
    def test_get_all_tests_calls_get_modules(self, mock_get_modules):
        mock_get_modules.return_value = []

        root = object()
        get_all_tests(root)

        self.assertEquals(mock_get_modules.call_args[0], (root,))


    @patch('unittestplus.run._get_modules')
    @patch('unittestplus.run._get_testcases')
    def test_get_all_tests_passes_modules_to_get_testcases(
        self, mock_get_testcases, mock_get_modules):

        module1 = Mock()
        module2 = Mock()
        mock_get_modules.return_value = [module1, module2]
        mock_get_testcases.return_value = []

        get_all_tests(None)

        expected = [
            ((module1,), {}),
            ((module2,), {}),
        ]
        self.assertEquals(mock_get_testcases.call_args_list, expected)


    @patch('unittestplus.run._get_modules')
    @patch('unittestplus.run._get_testcases')
    @patch('unittestplus.run.combine')
    def test_get_all_tests_combines_testcases(
        self, mock_combine, mock_get_testcases, mock_get_modules):

        mock_get_modules.return_value = [None]
        case1 = Mock()
        case2 = Mock()
        mock_get_testcases.return_value = [case1, case2]
        mock_suite = object()
        mock_combine.return_value = mock_suite

        actual = get_all_tests(None)

        self.assertEquals(mock_combine.call_args[0], (case1, case2))
        self.assertTrue(actual is mock_suite)
    

RunTest = combine(
    CombineTest,
    RunFuncTest,
    FindTest,
)

if __name__ == "__main__":
    run(RunTest)

