from ctypes import Array
from types import FloatType
from unittest import TestCase

def _is_int_indexable(item):
    return (
        type(item) is not dict and
        hasattr(item, "__len__") and
        hasattr(item, "__getitem__")
    )


def _tostr(item):
    if isinstance(item, Array):
        items = ', '.join(str(x) for x in item)
        return "Array(%s)" % items
    else:
        return str(item)

def _compare_lengths(actual, expected, message):
    if len(actual) != len(expected):
        actual_str = _tostr(actual)
        expected_str = _tostr(expected)
        msg = (
            "not equal, lengths differ: %d != %d\n"
            "  %s\n"
            "  %s\n"
            "%s" %
            (len(actual), len(expected), actual_str, expected_str, message)
        )
        raise AssertionError(msg)


def _compare_scalars(actual, expected, epsilon):
    if isinstance(actual, FloatType):
        return abs(actual - expected) <= epsilon
    else:
        return actual == expected


def _compare_indexables(actual, expected, message):
    _compare_lengths(actual, expected, message)
    for index in range(len(actual)):
        if actual[index] != expected[index]:
            actual_str = _tostr(actual)
            expected_str = _tostr(expected)
            msg = (
                "%s != %s at index %d\n"
                "  %s\n"
                "  %s\n"
                "%s" %
                (actual[index], expected[index], index,
                    actual_str, expected_str, message)
            )
            raise AssertionError(msg)


def _compare_types(actual, expected, message):
    if type(actual) == type(expected):
        return

    # ctypes arrays of different length are actually different types
    # but reporting them as such is just confusing
    if isinstance(actual, Array) and isinstance(expected, Array):
        return

    msg = "not equal. types differ:\n  %s %s\n  %s %s\n%s" % \
        (type(actual), actual, type(expected), expected, message)
    raise AssertionError(msg)


class TestCasePlus(TestCase):
    # pylint: disable-msg=C0103
    #   Invalid method names: This class uses unittest.TestCase conventions
    # pylint: disable-msg=R0904
    #   Too many public methods: we are a subclass of unittest.TestCase
    # pylint: disable-msg=R0201
    #   Method could be a function: acknowledged.

    def assertNone(self, item, message=''):
        if not item is None:
            raise AssertionError("not None: %s\n  %s" % (item, message))


    def assertNotNone(self, item, message=''):
        if item is None:
            raise AssertionError("is None\n  %s" % (message))


    def assertTrue(self, value, message=''):
        if not isinstance(value, bool):
            raise AssertionError('%s is not boolean. %s' % (value, message))
        if value is not True:
            raise AssertionError('is False. %s' % (message,))


    def assertFalse(self, value, message=''):
        if not isinstance(value, bool):
            raise AssertionError('%s is not boolean. %s' % (value, message))
        if value is not False:
            raise AssertionError('is True. %s' % (message,))


    def assertEquals(self, actual, expected, message='', epsilon=0):
        _compare_types(actual, expected, message)

        if _is_int_indexable(actual):
            _compare_indexables(actual, expected, message)
        else:

            passed = _compare_scalars(actual, expected, epsilon)
            if not passed:
                msg = "%s != %s\n  %s" % (actual, expected, message)
                raise AssertionError(msg)

    assertEqual = assertEquals


    def _assertRaises_test_args(self, func, excClass):

        def is_exception(obj):
            return isinstance(obj, type) and issubclass(obj, Exception)

        msg = ""
        arg1IsCallable = hasattr(func, '__call__') and not is_exception(func)
        if not arg1IsCallable:
            msg += "1st arg is not callable\n"
        arg2IsExcClass = is_exception(excClass)
        if not arg2IsExcClass:
            msg += "2nd arg is not exception class\n"

        if not arg1IsCallable or not arg2IsExcClass:
            msg += (
            "WARNING: TestCasePlus.assertRaises() has a new signature:\n"
            "  assertRaises(self, func, expectedType, expected_msg=None)")
            raise TypeError(msg)


    def assertRaises(
        self, func, expectedException, expected_msg=None):

        self._assertRaises_test_args(func, expectedException)

        try:
            func()
        except expectedException, actual:
            messageWrong = (
                expected_msg is not None and \
                str(actual) != expected_msg)
            if messageWrong:
                msg = "raised exception with wrong message:\n  %s\n  %s\n" % \
                    (str(actual), expected_msg)
                self.fail(msg)
        except Exception, actual:
            msg = 'raised wrong exception type:\n  %s("%s")\n  %s' % \
                (type(actual), str(actual), expectedException)
            self.fail(msg)
        else:
            msg = "didn't raise.\n  Expected %s(\"%s\")" % \
                (expectedException.__name__, expected_msg)
            self.fail(msg)
        return actual


    def assertValidColor(self, color, message=''):
        if len(color) != 3:
            self.fail("bad color: %s\n%s" % (color, message))
        for i in [0, 1, 2]:
            cpt = color[i]
            if not isinstance(cpt, int) or not 0 <= cpt <= 255:
                self.fail("bad color: %s\n%s" % (color, message))


    def assertVertsEqual(self, actual, expected, message=''):
        if len(actual) != len(expected):
            self.fail('verts differ in len: %d, %d\n%s'
                % (len(actual), len(expected), message))
                     
        for index in xrange(len(actual)):
            a = actual[index]
            e = expected[index]

            if len(a) != 2:
                self.fail('actual verts badly formed at vert %d: %s\n%s'
                    % (index, a, message))
            if len(e) != 2:
                self.fail('expected verts badly formed at vert %d: %s\n%s'
                    % (index, e, message))

            if not (
                _compare_scalars(a[0], e[0], epsilon=10**-6)
                and
                _compare_scalars(a[1], e[1], epsilon=10**-6)
                ):

                self.fail('verts differ at vert %d: %s, %s\n%s'
                    % (index, a, e, message))


    def shortDescription(self):
        desc = '.'.join([
            self.__class__.__module__,
            self.__class__.__name__,
            self._testMethodName,
        ])
        desc = desc.replace('.tests.', '.')

        prefix = __name__.split('.')[0] + '.'
        if desc.startswith(prefix):
            desc = desc[len(prefix):]
        return desc

