/*
Copyright (c) 2006-2007, Tom Thielicke IT Solutions

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.
*/

/****************************************************************
**
** Implementation of the StartWidget class
** File name: startwidget.cpp
**
****************************************************************/

#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QIcon>
#include <QSqlQuery>
#include <QSettings>
#include <QApplication>
#include <QMessageBox>

#include "startwidget.h"
#include "sql/startsql.h"
#include "updatedialog.h"
#include "def/defines.h"
#include "def/errordefines.h"
#include "errormessage.h"
#include "lessondialog.h"
#include "illustrationdialog.h"

StartWidget::StartWidget(QWidget *parent) : QWidget(parent) {
	// Set the logo
	companyLogo = new CompanyLogo(this);

	// Create group boxes with user settings
    createGroupLesson();
    createGroupLimit();
    createGroupError();
    txtIntelligenceTrue = (QObject::tr("Deaktivieren Sie diese Option, "
		"wenn die Texte der Lektion in ihrer vorgesehenen Reihenfolge diktiert "
		"und nicht an die Tippfehler angepasst werden sollen."));
    txtIntelligenceFalse = (QObject::tr("Aktivieren Sie diese Option, "
		"wenn die Texte der Lektion nicht in ihrer vorgesehenen Reihenfolge diktiert, "
		"sondern an die Tippfehler angepasst werden sollen."));
    createGroupIntelligence();
    createGroupSupport();

    // Create buttons
    createButtons();

	// Set the layout of all widgets created above
	createLayout();

	// Read user settings
	readSettings();

	// Widget connections
	createConnections();

    buttonTraining->setFocus();
}

StartWidget::~StartWidget() {
	// Save settings before closing the startwidget
	writeSettings();
}

void StartWidget::createGroupLesson() {

	tabLessons = new QTabWidget();
	//tabLessons->setUsesScrollButtons(false);
	tabLessons->setElideMode(Qt::ElideRight);

	// List widget with training lessons
	listLesson = new QListWidget();
	//fillLessonList();

	tabTrainingLessons = new QWidget();
	QVBoxLayout *tabVLayout = new QVBoxLayout;
	tabVLayout->addWidget(listLesson);
	tabTrainingLessons->setLayout(tabVLayout);

	tabLessons->addTab(tabTrainingLessons, QIcon(":/img/tab_training.png"), QObject::tr("bungslektionen"));

	// List widget with training lessons
	listOpen = new QListWidget();

	comboTheme = new QComboBox();

	labelTheme = new QLabel(QObject::tr("Thema:"));

	tabOpenLessons = new QWidget();
	QHBoxLayout *tabPLayout = new QHBoxLayout;
	tabPLayout->addStretch(1);
	tabPLayout->addWidget(labelTheme);
	tabPLayout->addWidget(comboTheme);
	QVBoxLayout *tabOLayout = new QVBoxLayout;
	tabOLayout->addLayout(tabPLayout);
	tabOLayout->addWidget(listOpen);
	tabOpenLessons->setLayout(tabOLayout);

	tabLessons->addTab(tabOpenLessons, QIcon(":/img/tab_open.png"), QObject::tr(""));


	// List widget with training lessons
	listOwn = new QListWidget();

	lessonEditMenu = new QMenu();
	buttonEditLesson = new QPushButton(QObject::tr("&Bearbeiten"));

	lessonNew = new QAction(QIcon(":/img/menu_new.png"),
		QObject::tr("&Neue Lektion"), this);
	lessonEdit = new QAction(QIcon(":/img/menu_edit.png"),
		QObject::tr("Lektion &editieren"), this);
	lessonDel = new QAction(QIcon(":/img/menu_del.png"),
		QObject::tr("Lektion &lschen"), this);
	//lessonPublish = new QAction(QIcon(":/img/menu_help.png"),
	//	QObject::tr("Hilfe"), this);

	lessonEditMenu->addAction(lessonNew);
	lessonEditMenu->addAction(lessonEdit);
	lessonEditMenu->addAction(lessonDel);
	//lessonEditMenu->addAction(lessonPublish);

	buttonEditLesson->setMenu(lessonEditMenu);
	tabOwnLessons = new QWidget();
	QHBoxLayout *tabHLayout = new QHBoxLayout;
	tabHLayout->addStretch(1);
	tabHLayout->addWidget(buttonEditLesson);
	QVBoxLayout *tabLayout = new QVBoxLayout;
	tabLayout->addLayout(tabHLayout);
	tabLayout->addWidget(listOwn);
	tabOwnLessons->setLayout(tabLayout);

	tabLessons->addTab(tabOwnLessons, QIcon(":/img/tab_own.png"), QObject::tr(""));
}

void StartWidget::createGroupLimit() {
	// Group "Duration of lesson"
	groupLimit = new QGroupBox(QObject::tr("Dauer der Lektion"));

	// Radiobutton "Time limit"
	radioLimitTime = new QRadioButton(QObject::tr("Zeitlimit:"));
	radioLimitTime->setToolTip(QObject::tr("Das Diktat wird nach Ablauf "
		"einer\nfestgelegten Zeit beendet"));

	// Spin textbox "Time limit" with values from 2 to 20 minutes
	spinLimitTime = new QSpinBox();
	spinLimitTime->setMinimum(2);
	spinLimitTime->setMaximum(20);
	spinLimitTime->setSuffix(QObject::tr(" Minuten"));
	spinLimitTime->setToolTip(QObject::tr("Das Diktat wird nach Ablauf einer\n"
		"festgelegten Zeit beendet"));

	//Radiobutton "Number of tokens"
	radioLimitToken = new QRadioButton(QObject::tr("Zeichenlimit:"));
	radioLimitToken->setToolTip(QObject::tr("Das Diktat wird nach einer "
		"festgelegten Anzahl\nkorrekt getippter Zeichen beendet"));

	// Spin testbox "Number of tokens" with values from 200 to 2000 tokens
	spinLimitToken = new QSpinBox();
	spinLimitToken->setMinimum(200);
	spinLimitToken->setMaximum(2000);
	spinLimitToken->setSingleStep(100);
	spinLimitToken->setSuffix(QObject::tr(" Zeichen"));
	spinLimitToken->setToolTip(QObject::tr("Das Diktat wird nach einer "
		"bestimmten Anzahl\nkorrekt getippter Zeichen beendet"));

	//Radiobutton "To end of lesson"
	radioLimitLesson = new QRadioButton(QObject::tr("Gesamte\nLektion"));
	radioLimitLesson->setToolTip(QObject::tr("Es wird die gesamte (freie) Lektion\n"
		"von Anfang bis Ende diktiert"));
	labelLimitLesson = new QLabel(QObject::tr("(gesamte Lektion)"));


	// Layout of group box
	QVBoxLayout *timeLayout = new QVBoxLayout;
	timeLayout->addWidget(radioLimitTime);
	timeLayout->addWidget(spinLimitTime);
    //timeLayout->addSpacing(10);
	QVBoxLayout *tokenLayout = new QVBoxLayout;
	tokenLayout->addWidget(radioLimitToken);
	tokenLayout->addWidget(spinLimitToken);
    //tokenLayout->addSpacing(10);
	QVBoxLayout *lessonLayout = new QVBoxLayout;
	lessonLayout->addStretch(1);
	lessonLayout->addWidget(radioLimitLesson);
	//lessonLayout->addStretch(1);
	//lessonLayout->addWidget(labelLimitLesson);
	lessonLayout->addStretch(1);
    //lessonLayout->addSpacing(10);
	QHBoxLayout *groupLayout = new QHBoxLayout;
	groupLayout->addLayout(timeLayout);
    groupLayout->addSpacing(20);
	groupLayout->addLayout(tokenLayout);
    groupLayout->addSpacing(20);
	groupLayout->addLayout(lessonLayout);
	groupLayout->setMargin(10);
	groupLimit->setLayout(groupLayout);
}

void StartWidget::createGroupIntelligence() {
	// Group "Type error handling"
	groupIntelligence = new QGroupBox(QObject::tr("Intelligenz"));

	labelIntelligence = new QLabel();
	labelIntelligence->setWordWrap(true);
	// Checkbox "Block type errors"
	checkIntelligence = new QCheckBox(QObject::tr("Diktat in Echtzeit an meine Tippfehler anpassen"));
	checkIntelligence->setToolTip(QObject::tr("Anhand der aktuellen Fehlerquoten aller Schriftzeichen "
		"werden\ndie Worte und Stze des Dikats dynamisch ausgewhlt.\nIst die Intelligenz deaktiviert, "
		"werden die Texte der Lektion\nstets in der gleichen Reihenfolge diktiert."));

	// Layout of group box
	QVBoxLayout *layout = new QVBoxLayout;
	layout->addWidget(checkIntelligence);
	layout->addWidget(labelIntelligence);
    //layout->addSpacing(10);
	layout->setMargin(10);
    // Pass layout to parent widget (group box)
	groupIntelligence->setLayout(layout);
}

void StartWidget::createGroupError() {
	// Group "Type error handling"
	groupError = new QGroupBox(QObject::tr("Reaktion auf Tippfehler"));

	// Checkbox "Block type errors"
	checkErrorStop = new QCheckBox(QObject::tr("Diktat blockieren"));
	checkErrorStop->setToolTip(QObject::tr("Das Diktat wird erst fortgesetzt, "
		"wenn die\nrichtige Taste gedrckt wurde"));

	// Checkbox "Correct type errors"
	checkErrorCorrect = new QCheckBox(QObject::tr("Fehler korrigieren"));
	checkErrorCorrect->setToolTip(QObject::tr("Tippfehler mssen zustzlich ber die\n"
		"Rcklauftaste entfernt werden"));

	// Checkbox "Beep on type errors"
	checkErrorBeep = new QCheckBox(QObject::tr("Akustisches Signal"));
	checkErrorBeep->setToolTip(QObject::tr("Bei jedem Tippfehler ertnt ein "
		"Beepton"));

	// Layout of group box
	QHBoxLayout *hlayout = new QHBoxLayout;
    hlayout->addSpacing(12);
	hlayout->addWidget(checkErrorCorrect);
	QVBoxLayout *layout = new QVBoxLayout;
	layout->addWidget(checkErrorStop);
	layout->addLayout(hlayout);
    layout->addSpacing(16);
	layout->addWidget(checkErrorBeep);
    //layout->addSpacing(10);
	layout->setMargin(10);
    // Pass layout to parent widget (group box)
	groupError->setLayout(layout);
}

void StartWidget::createGroupSupport() {
	// Group "Other user support"
	groupKeyboard = new QGroupBox(QObject::tr("Hilfestellungen"));

	checkHelpers = new QCheckBox(QObject::tr("Hilfestellungen anzeigen"));
	checkHelpers->setToolTip(QObject::tr("Zur visuellen Untersttzung "
			"werden die virtuelle\nTastatur und Statusinformationen angezeigt"));

	// Checkbox "Enable color selection of keys"
	checkKeySelection = new QCheckBox(QObject::tr("Tasten farbig markieren"));
	checkKeySelection->setToolTip(QObject::tr("Zur visuellen Untersttzung "
		"werden die zu\ndrckenden Tasten farbig markiert"));

	// Checkbox "Enable color selection of start keys"
	checkKeySelectionStart = new QCheckBox(QObject::tr("Grundstellung markieren"));
	checkKeySelectionStart->setToolTip(QObject::tr("Zur visuellen Untersttzung "
		"werden die\nverbleibenden Finger der Grundstellung\nfarbig markiert"));

	// Checkbox "Enable status information"
	checkKeyBorder = new QCheckBox(QObject::tr("Trennlinie anzeigen"));
	checkKeyBorder->setToolTip(QObject::tr("Zur visuellen Untersttzung "
		"wird eine Trennlinie\nfr die zu Unterscheidung von linker und rechter Hand\n"
		"angezeigt"));

	// Checkbox "Enable status information"
	checkStatusInformation = new QCheckBox(QObject::tr("Hilfetext anzeigen"));
	checkStatusInformation->setToolTip(QObject::tr("Zur Untersttzung werden "
		"die zu verwendenden\nFinger in der Statusleiste angezeigt"));

	// Layout of group box
	QHBoxLayout *helpLayout = new QHBoxLayout;
	helpLayout->addWidget(checkHelpers);
	QHBoxLayout *sLayout = new QHBoxLayout;
    sLayout->addSpacing(12);
	sLayout->addWidget(checkKeySelection);
	QHBoxLayout *ssLayout = new QHBoxLayout;
    ssLayout->addSpacing(12);
	ssLayout->addWidget(checkKeySelectionStart);
	QHBoxLayout *bLayout = new QHBoxLayout;
    bLayout->addSpacing(12);
	bLayout->addWidget(checkKeyBorder);
	QHBoxLayout *siLayout = new QHBoxLayout;
    siLayout->addSpacing(12);
	siLayout->addWidget(checkStatusInformation);
	QVBoxLayout *selectionLayout = new QVBoxLayout;
	selectionLayout->addLayout(helpLayout);
	selectionLayout->addLayout(sLayout);
	selectionLayout->addLayout(ssLayout);
	selectionLayout->addLayout(bLayout);
	selectionLayout->addLayout(siLayout);
	selectionLayout->setMargin(10);
    // Pass layout to parent widget (group box)
	groupKeyboard->setLayout(selectionLayout);
}

void StartWidget::createButtons() {
	//Button "Training starten"
	buttonHelp = new QPushButton(QObject::tr("&Hilfe"));
	buttonExit = new QPushButton(QObject::tr("&Beenden"));
	buttonTraining = new QPushButton(QObject::tr("&Schreibtraining starten"));
    buttonTraining->setDefault(true);
}

void StartWidget::createLayout() {
	// Button layout horizontal
	QHBoxLayout *buttonLayoutHorizontal = new QHBoxLayout;
    buttonLayoutHorizontal->addWidget(buttonExit);
    buttonLayoutHorizontal->addSpacing(10);
    buttonLayoutHorizontal->addWidget(buttonHelp);
    buttonLayoutHorizontal->addWidget(buttonTraining);
	// Button layout vertical
	QVBoxLayout *buttonLayoutVertical = new QVBoxLayout;
    buttonLayoutVertical->addSpacing(20);
    buttonLayoutVertical->addLayout(buttonLayoutHorizontal);
	// Logo layout vertical
	QVBoxLayout *logoLayout = new QVBoxLayout;
    logoLayout->addSpacing(10);
    logoLayout->addWidget(companyLogo);
	// Logo layout + button layout horizontal
	QHBoxLayout *bottomLayout = new QHBoxLayout;
    bottomLayout->addLayout(logoLayout);
    bottomLayout->addStretch(1);
    bottomLayout->addLayout(buttonLayoutVertical);
    // Group error handling and other user support vertical
	/*QVBoxLayout *bottomboxesLayoutV = new QVBoxLayout;
    bottomboxesLayoutV->addWidget(groupError);
    bottomboxesLayoutV->addWidget(groupIntelligence);*/
    // Group error handling and other user support horizontal
	QHBoxLayout *bottomboxesLayout = new QHBoxLayout;
    bottomboxesLayout->addWidget(groupError);
    bottomboxesLayout->addWidget(groupKeyboard);
    // Group duration and layout above vertical
	QVBoxLayout *boxesLayout = new QVBoxLayout;
    boxesLayout->addWidget(groupLimit);
    boxesLayout->addLayout(bottomboxesLayout);
    boxesLayout->addWidget(groupIntelligence);
    // Layout all groups, buttons and logo horizontal
	QHBoxLayout *centerLayout = new QHBoxLayout;
    centerLayout->addWidget(tabLessons);
    centerLayout->addLayout(boxesLayout);
	// Full layout of all widgets vertical
	QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addLayout(centerLayout);
    mainLayout->addSpacing(1);
    mainLayout->addLayout(bottomLayout);
    mainLayout->setMargin(15);
    mainLayout->setSpacing(15);
    // Pass layout to parent widget (this)
	this->setLayout(mainLayout);
}

void StartWidget::createConnections() {
	// Widget connections
    connect(buttonTraining, SIGNAL(clicked()), this, SLOT(clickTraining()));
    connect(buttonExit, SIGNAL(clicked()), qApp, SLOT(quit()));
    connect(buttonHelp, SIGNAL(clicked()), this, SLOT(showHelp()));

	connect(radioLimitTime, SIGNAL(clicked()), this, SLOT(toggleLimit()));
	connect(radioLimitToken, SIGNAL(clicked()), this, SLOT(toggleLimit()));
	connect(radioLimitLesson, SIGNAL(clicked()), this, SLOT(toggleLimit()));
	connect(checkErrorStop, SIGNAL(clicked()), this, SLOT(toggleError()));
	connect(checkHelpers, SIGNAL(clicked()), this, SLOT(toggleHelpers()));
	connect(checkIntelligence, SIGNAL(clicked()), this, SLOT(toggleIntelligence()));

	connect(listLesson, SIGNAL(itemDoubleClicked(QListWidgetItem*)), this,
		SLOT(doubleClickLesson(QListWidgetItem*)));
	connect(listOpen, SIGNAL(itemDoubleClicked(QListWidgetItem*)), this,
		SLOT(doubleClickLesson(QListWidgetItem*)));
	connect(listOwn, SIGNAL(itemDoubleClicked(QListWidgetItem*)), this,
		SLOT(doubleClickLesson(QListWidgetItem*)));

	connect(lessonNew, SIGNAL(triggered()), this, SLOT(clickNewLesson()));
	connect(lessonDel, SIGNAL(triggered()), this, SLOT(clickDeleteLesson()));
	connect(lessonEdit, SIGNAL(triggered()), this, SLOT(clickEditLesson()));

	connect(tabLessons, SIGNAL(currentChanged(int)), this, SLOT(toggleTabs(int)));
	connect(comboTheme, SIGNAL(activated(int)), this, SLOT(toggleThemes(int)));
}

void StartWidget::fillLessonList(bool themeChanged = false) {
	int trainingNum = 0;
	int openNum = 0;
	int ownNum = 0;
	int themesNum = 0;
	QSettings settings;
	StartSql *lessonSql = new StartSql();
	// Training lesson list
	// --------------------
	if ((trainingNum = lessonSql->fillLessonList(listLesson, &arrayTraining)) == -1) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_LESSONS_EXIST, TYPE_CRITICAL,
			CANCEL_OPERATION);
		return;
	}
	// Preselection
	settings.beginGroup("lesson");
	int selectTraining = arrayTraining.indexOf(settings.value("list_training","").toString());
	if (selectTraining == -1) {
		listLesson->setCurrentRow(0);
	} else {
		listLesson->setCurrentRow(selectTraining);
	}
	settings.endGroup();
	// Open themes list
	// --------------------
	if ((themesNum = lessonSql->fillThemes(comboTheme, &arrayThemes)) == -1) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_LESSONS_EXIST, TYPE_CRITICAL,
			CANCEL_OPERATION);
		return;
	}
	// Preselection
	settings.beginGroup("lesson");
	int selectTheme = arrayThemes.indexOf(settings.value("combo_theme","").toString());
	if (selectTheme == -1) {
		comboTheme->setCurrentIndex(0);
	} else {
		comboTheme->setCurrentIndex(selectTheme);
	}
	// Hide Theme-Combo if only one ("all") theme exist
	if (comboTheme->count() <= 1) {
		labelTheme->setVisible(false);
		comboTheme->setVisible(false);
	} else {
		labelTheme->setVisible(true);
		comboTheme->setVisible(true);
	}
	settings.endGroup();
	// Open lesson list
	// --------------------
	if ((openNum = lessonSql->fillOpenList(listOpen, &arrayOpen,
		arrayThemes.at(comboTheme->currentIndex()))) == -1) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_LESSONS_EXIST, TYPE_CRITICAL,
			CANCEL_OPERATION);
		return;
	}
	// Preselection
	settings.beginGroup("lesson");
	int selectOpen = arrayOpen.indexOf(settings.value("list_open","").toString());
	if (selectOpen == -1) {
		listOpen->setCurrentRow(0);
	} else {
		listOpen->setCurrentRow(selectOpen);
	}
	settings.endGroup();
	// Own lesson list
	// --------------------
	if ((ownNum = lessonSql->fillOwnList(listOwn, &arrayOwn)) == -1) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_LESSONS_EXIST, TYPE_CRITICAL,
			CANCEL_OPERATION);
		return;
	}
	// Preselection
	settings.beginGroup("lesson");
	int selectOwn = arrayOwn.indexOf(settings.value("list_own","").toString());
	if (selectOwn == -1) {
		listOwn->setCurrentRow(0);
	} else {
		listOwn->setCurrentRow(selectOwn);
	}
	settings.endGroup();
	if (listOwn->count() == 0) {
		lessonEdit->setEnabled(false);
		lessonDel->setEnabled(false);
	} else {
		buttonTraining->setEnabled(true);
		lessonEdit->setEnabled(true);
		lessonDel->setEnabled(true);
	}
}

void StartWidget::toggleLimit() {
	// Check radio buttons in group "Duration of lesson"
	if (radioLimitTime->isChecked()) {
		// "Time limit" selected
		spinLimitTime->setEnabled(true);
		spinLimitToken->setEnabled(false);
	} else {
		if (radioLimitToken->isChecked()) {
			// "Token limit" selected
			spinLimitTime->setEnabled(false);
			spinLimitToken->setEnabled(true);
		} else {
			spinLimitTime->setEnabled(false);
			spinLimitToken->setEnabled(false);
		}
	}
}

void StartWidget::toggleError() {
	// Check check box in group "Type error handling"
	if (checkErrorStop->isChecked()) {
		checkErrorCorrect->setEnabled(true);
	} else {
		checkErrorCorrect->setChecked(false);
		checkErrorCorrect->setEnabled(false);
	}
}

void StartWidget::toggleHelpers() {
	// Check check box in group "Type error handling"
	if (checkHelpers->isChecked()) {
		checkKeySelection->setEnabled(true);
		checkKeySelection->setChecked(true);
		checkKeySelectionStart->setEnabled(true);
		checkKeySelectionStart->setChecked(true);
		checkKeyBorder->setEnabled(true);
		checkKeyBorder->setChecked(true);
		checkStatusInformation->setEnabled(true);
		checkStatusInformation->setChecked(true);
	} else {
		checkKeySelection->setEnabled(false);
		checkKeySelection->setChecked(false);
		checkKeySelectionStart->setEnabled(false);
		checkKeySelectionStart->setChecked(false);
		checkKeyBorder->setEnabled(false);
		checkKeyBorder->setChecked(false);
		checkStatusInformation->setEnabled(false);
		checkStatusInformation->setChecked(false);
	}
}

void StartWidget::toggleIntelligence() {
	QSettings settings;
	// Check check box in group "Type error handling"
	if (checkIntelligence->isChecked()) {
		radioLimitLesson->setVisible(false);
		if (radioLimitLesson->isChecked()) {
			radioLimitTime->setChecked(true);
			spinLimitTime->setEnabled(true);
		}
		labelIntelligence->setText(txtIntelligenceTrue);
	} else {
		radioLimitLesson->setVisible(true);
		if (toggleLimitLesson) {
			radioLimitLesson->setChecked(true);
			spinLimitTime->setEnabled(false);
			spinLimitToken->setEnabled(false);
		}
		labelIntelligence->setText(txtIntelligenceFalse);
	}
	settings.beginGroup("intelligence");
	settings.setValue("check_intelligence", checkIntelligence->isChecked());
	settings.endGroup();
}

void StartWidget::toggleThemes(int index){
	StartSql *lessonSql = new StartSql();
	// Open lesson list
	// --------------------
	if (lessonSql->fillOpenList(listOpen, &arrayOpen,
		arrayThemes.at(index)) == -1) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_LESSONS_EXIST, TYPE_CRITICAL,
			CANCEL_OPERATION);
		return;
	}
	// Preselection
	listOpen->setCurrentRow(0);
}

void StartWidget::toggleTabs(int index) {
	QSettings settings;
	tabLessons->setTabText(0, QObject::tr(""));
	tabLessons->setTabText(1, QObject::tr(""));
	tabLessons->setTabText(2, QObject::tr(""));
	switch (index) {
		case 0:
			tabLessons->setTabText(0, QObject::tr("bungslektionen"));
			buttonTraining->setEnabled(true);
			checkIntelligence->setChecked(true);
			groupIntelligence->setEnabled(false);
			radioLimitLesson->setVisible(false);
			if (radioLimitLesson->isChecked()) {
				radioLimitTime->setChecked(true);
				spinLimitTime->setEnabled(true);
			}
			labelIntelligence->setText(txtIntelligenceTrue);
			break;
		case 1:
			tabLessons->setTabText(1, QObject::tr("Freie Lektionen"));
			buttonTraining->setEnabled(true);
			groupIntelligence->setEnabled(true);
			settings.beginGroup("intelligence");
			checkIntelligence->setChecked(settings.value("check_intelligence", false).toBool());
			settings.endGroup();
			// Enable/disable radioLimitLesson
			if (checkIntelligence->isChecked() == true) {
				radioLimitLesson->setVisible(false);
				if (radioLimitLesson->isChecked()) {
					radioLimitTime->setChecked(true);
					spinLimitTime->setEnabled(true);
				}
				labelIntelligence->setText(txtIntelligenceTrue);
			} else {
				radioLimitLesson->setVisible(true);
				if (toggleLimitLesson) {
					radioLimitLesson->setChecked(true);
					spinLimitTime->setEnabled(false);
					spinLimitToken->setEnabled(false);
				}
				labelIntelligence->setText(txtIntelligenceFalse);
			}
			break;
		case 2:
			tabLessons->setTabText(2, QObject::tr("Eigene Lektionen"));
			if (listOwn->count() == 0) {
				buttonTraining->setEnabled(false);
				lessonEdit->setEnabled(false);
				lessonDel->setEnabled(false);
			} else {
				buttonTraining->setEnabled(true);
				lessonEdit->setEnabled(true);
				lessonDel->setEnabled(true);
			}
			groupIntelligence->setEnabled(true);
			settings.beginGroup("intelligence");
			checkIntelligence->setChecked(settings.value("check_intelligence", false).toBool());
			settings.endGroup();
			// Enable/disable radioLimitLesson
			if (checkIntelligence->isChecked() == true) {
				radioLimitLesson->setVisible(false);
				if (radioLimitLesson->isChecked()) {
					radioLimitTime->setChecked(true);
					spinLimitTime->setEnabled(true);
				}
				labelIntelligence->setText(txtIntelligenceTrue);
			} else {
				radioLimitLesson->setVisible(true);
				if (toggleLimitLesson) {
					radioLimitLesson->setChecked(true);
					spinLimitTime->setEnabled(false);
					spinLimitToken->setEnabled(false);
				}
				labelIntelligence->setText(txtIntelligenceFalse);
			}
			break;
	}
}

void StartWidget::clickTraining() {
	// User finished setting the properties and wants to start the
	// training lesson
	int selectedLesson = -1;
	QString selectedName = "";
	int listSize = 0;
	int infoResult = 1;
	switch (tabLessons->currentIndex()) {
		case 0:
			selectedLesson = arrayTraining.at(listLesson->currentRow()).toInt();
			listSize = arrayTraining.size();
			selectedName = listLesson->currentItem()->text();
			break;
		case 1:
			selectedLesson = arrayOpen.at(listOpen->currentRow()).toInt();
			listSize = arrayOpen.size();
			selectedName = listOpen->currentItem()->text();
			break;
		case 2:
			selectedLesson = arrayOwn.at(listOwn->currentRow()).toInt();
			listSize = arrayOwn.size();
			selectedName = listOwn->currentItem()->text();
			break;
	}
	if (selectedLesson == -1 || listSize == 0) {
		// No selected lesson found in combo box
		// -> error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_LESSONS_SELECTED, TYPE_INFO,
			CANCEL_OPERATION);
		return;
	}
	// Infowindow about intelligence
	if (checkToggleIntelligence && tabLessons->currentIndex() != 0 &&
		checkIntelligence->isChecked()) {
		IllustrationDialog illustrationDialog("intelligence.png", 540, 420,
			"Hinweis zur Intelligenz", "", "check_toggle_intelligence", 3, this);
		infoResult = illustrationDialog.exec();
		QSettings settings;
		settings.beginGroup("general");
		checkToggleIntelligence = settings.value("check_toggle_intelligence",
			true).toBool();
		settings.endGroup();
		if (infoResult == 0) {
			return;
		}
	}
	// -> emit the signal that mainwindow knows user wants to start training
	emit trainingClicked(selectedLesson, tabLessons->currentIndex(), selectedName);
}

void StartWidget::doubleClickLesson(QListWidgetItem *item) {
	clickTraining();
}

void StartWidget::clickNewLesson() {
	// Show lesson publishing info
	if (checkLessonPublish) {
		IllustrationDialog illustrationDialog("publish.png", 540, 400,
			"Lektionen verffentlichen", "", "check_lesson_publish", 2, this);
		illustrationDialog.exec();
		QSettings settings;
		settings.beginGroup("general");
		checkLessonPublish = settings.value("check_lesson_publish", true).toBool();
		settings.endGroup();
	}
	LessonDialog lessonDialog("-1", this);
	lessonDialog.exec();
	// Fill lesson list after changing lessons
	StartSql *lessonSql = new StartSql();
	// Own lesson list
	// ---------------
	if (lessonSql->fillOwnList(listOwn, &arrayOwn) == -1) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_LESSONS_EXIST, TYPE_CRITICAL,
			CANCEL_OPERATION);
		return;
	}
	// Preselection
	listOwn->setCurrentRow(0);
	if (listOwn->count() == 0) {
		buttonTraining->setEnabled(false);
		lessonEdit->setEnabled(false);
		lessonDel->setEnabled(false);
	} else {
		buttonTraining->setEnabled(true);
		lessonEdit->setEnabled(true);
		lessonDel->setEnabled(true);
	}
}

void StartWidget::clickEditLesson() {
	int tempLesson = listOwn->currentRow();
	LessonDialog lessonDialog(arrayOwn.at(tempLesson), this);
	lessonDialog.exec();
	// Fill lesson list after changing lessons
	StartSql *lessonSql = new StartSql();
	// Own lesson list
	// ---------------
	if (lessonSql->fillOwnList(listOwn, &arrayOwn) == -1) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_LESSONS_EXIST, TYPE_CRITICAL,
			CANCEL_OPERATION);
		return;
	}
	// Preselection
	listOwn->setCurrentRow(tempLesson);
}

void StartWidget::clickDeleteLesson() {
	switch (QMessageBox::question(this, APP_NAME,
		QObject::tr("Wollen Sie die Lektion wirklich lschen und damit auch\n"
		"alle aufgezeichneten Daten, die im Zusammenhang\nmit dieser Lektion stehen? "
		"\n\n"), QObject::tr("&Ja"), QObject::tr("&Abbrechen"),
		0, 1)) {
		case 0:
			StartSql *lessonSql = new StartSql();
			if (!lessonSql->deleteOwnLesson(arrayOwn.at(listOwn->currentRow()))) {
				// Error message
				ErrorMessage *errorMessage = new ErrorMessage(this);
				errorMessage->showMessage(ERR_LESSONS_EXIST, TYPE_CRITICAL,
					CANCEL_OPERATION);
				return;
			}
			if (!lessonSql->analyzeOwnLessons()) {
				// No selected lesson found in combo box
				// -> error message
				ErrorMessage *errorMessage = new ErrorMessage(this);
				errorMessage->showMessage(ERR_USER_LESSON_ANALYZE, TYPE_INFO,
					CANCEL_OPERATION);
				return;
			}
			// Fill lesson list after changing lessons
			if (lessonSql->fillOwnList(listOwn, &arrayOwn) == -1) {
				// Error message
				ErrorMessage *errorMessage = new ErrorMessage(this);
				errorMessage->showMessage(ERR_LESSONS_EXIST, TYPE_CRITICAL,
					CANCEL_OPERATION);
				return;
			}
			// Preselection
			listOwn->setCurrentRow(0);
			break;
	}
	if (listOwn->count() == 0) {
		buttonTraining->setEnabled(false);
		lessonEdit->setEnabled(false);
		lessonDel->setEnabled(false);
	} else {
		buttonTraining->setEnabled(true);
		lessonEdit->setEnabled(true);
		lessonDel->setEnabled(true);
	}
}

void StartWidget::showHelp() {
	helpBrowser = new HelpBrowser("", 0);
	helpBrowser->show();
}

void StartWidget::readSettings() {
	// Restores settings of the startwiget
	// (uses the default constructor of QSettings, passing
	// the application and company name see main function)
	QSettings settings;

	// Set current lesson tab
	settings.beginGroup("lesson");
	tabLessons->setCurrentIndex(settings.value("tab_current", 0).toInt());
	settings.endGroup();
	tabLessons->setTabText(0, QObject::tr(""));
	tabLessons->setTabText(1, QObject::tr(""));
	tabLessons->setTabText(2, QObject::tr(""));
	switch(tabLessons->currentIndex()) {
		case 0:
			tabLessons->setTabText(0, QObject::tr("bungslektionen"));
			break;
		case 1:
			tabLessons->setTabText(1, QObject::tr("Freie Lektionen"));
			break;
		case 2:
			tabLessons->setTabText(2, QObject::tr("Eigene Lektionen"));
			break;
	}

	settings.beginGroup("duration");
	radioLimitTime->setChecked(settings.value("radio_time", true).toBool());
	spinLimitTime->setValue(settings.value("spin_time",
		LESSON_TIMELEN_STANDARD).toInt());
	spinLimitTime->setEnabled(settings.value("radio_time", true).toBool());
	radioLimitToken->setChecked(settings.value("radio_token", false).toBool());
	spinLimitToken->setValue(settings.value("spin_token",
		LESSON_TOKENLEN_STANDARD).toInt());
	spinLimitToken->setEnabled(settings.value("radio_token", false).toBool());
	radioLimitLesson->setChecked(settings.value("radio_lesson", false).toBool());
	settings.endGroup();
	settings.beginGroup("error");
	checkErrorStop->setChecked(settings.value("check_stop", true).toBool());
	checkErrorCorrect->setChecked(settings.value("check_correct", false).toBool());
	checkErrorCorrect->setEnabled(settings.value("check_stop", true).toBool());
	checkErrorBeep->setChecked(settings.value("check_beep", false).toBool());
	settings.endGroup();
	settings.beginGroup("support");
	checkHelpers->setChecked(settings.value("check_helpers", true).toBool());
	checkKeySelection->setChecked(settings.value("check_selection", true).toBool());
	checkKeySelectionStart->setChecked(settings.value("check_selection_start", true).toBool());
	checkKeyBorder->setChecked(settings.value("check_border", true).toBool());
	checkStatusInformation->setChecked(settings.value("check_status", true).toBool());
	settings.endGroup();

	if (tabLessons->currentIndex() != 0) {
		settings.beginGroup("intelligence");
		checkIntelligence->setChecked(settings.value("check_intelligence", false).toBool());
		settings.endGroup();
	} else {
		checkIntelligence->setChecked(true);
		groupIntelligence->setEnabled(false);
	}

	settings.beginGroup("general");
	toggleLimitLesson = settings.value("check_limit_lesson", true).toBool();
	checkLessonPublish = settings.value("check_lesson_publish", true).toBool();
	checkToggleIntelligence = settings.value("check_toggle_intelligence", true).toBool();
	settings.endGroup();

	// Enable/disable radioLimitLesson
	if (tabLessons->currentIndex() == 0) {
		labelIntelligence->setText(txtIntelligenceTrue);
		radioLimitLesson->setVisible(false);
	} else {
		if (checkIntelligence->isChecked() == true) {
			radioLimitLesson->setVisible(false);
			labelIntelligence->setText(txtIntelligenceTrue);
		} else {
			radioLimitLesson->setVisible(true);
			labelIntelligence->setText(txtIntelligenceFalse);
		}
	}
}

void StartWidget::writeSettings() {
	// Saves settings of the startwiget
	// (uses the default constructor of QSettings, passing
	// the application and company name see main function)
	QSettings settings;
	settings.beginGroup("lesson");
	settings.setValue("tab_current", tabLessons->currentIndex());
	if (listLesson->count() == 0) {
		settings.setValue("list_training", -1);
	} else {
		settings.setValue("list_training", arrayTraining.at(listLesson->currentRow()));
	}
	if (listOpen->count() == 0) {
		settings.setValue("list_open", -1);
	} else {
		settings.setValue("list_open", arrayOpen.at(listOpen->currentRow()));
	}
	if (comboTheme->count() == 0) {
		settings.setValue("combo_theme", -1);
	} else {
		settings.setValue("combo_theme", arrayThemes.at(comboTheme->currentIndex()));
	}
	if (listOwn->count() == 0) {
		settings.setValue("list_own", -1);
	} else {
		settings.setValue("list_own", arrayOwn.at(listOwn->currentRow()));
	}
	settings.endGroup();
	settings.beginGroup("duration");
	settings.setValue("radio_time", radioLimitTime->isChecked());
	settings.setValue("spin_time", spinLimitTime->value());
	settings.setValue("radio_token", radioLimitToken->isChecked());
	settings.setValue("spin_token", spinLimitToken->value());
	settings.setValue("radio_lesson", radioLimitLesson->isChecked());
	settings.endGroup();
	settings.beginGroup("error");
	settings.setValue("check_stop", checkErrorStop->isChecked());
	settings.setValue("check_correct", checkErrorCorrect->isChecked());
	settings.setValue("check_beep", checkErrorBeep->isChecked());
	settings.endGroup();
	settings.beginGroup("support");
	settings.setValue("check_helpers", checkHelpers->isChecked());
	settings.setValue("check_selection", checkKeySelection->isChecked());
	settings.setValue("check_selection_start", checkKeySelectionStart->isChecked());
	settings.setValue("check_border", checkKeyBorder->isChecked());
	settings.setValue("check_status", checkStatusInformation->isChecked());
	settings.endGroup();
	if (tabLessons->currentIndex() != 0) {
		settings.beginGroup("intelligence");
		settings.setValue("check_intelligence", checkIntelligence->isChecked());
		settings.endGroup();
	}
}
