/*
Copyright (c) 2006-2007, Tom Thielicke IT Solutions

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.
*/

/****************************************************************
**
** Implementation of the KeyBoard class
** File name: keyboard.cpp
**
****************************************************************/

#include <QPainter>
#include <QSettings>

#include "keyboard.h"
#include "../def/defines.h"
#include "../def/errordefines.h"
#include "errormessage.h"

KeyBoard::KeyBoard(QWidget *parent) : QWidget(parent) {
	// Read settings (keyboard layout and key coloring)
	readSettings();

    // Load all images used in the keyboard
    loadKeyImages();

    // Start flag
    isStarted = false;
    // Pause flag
    isPaused = false;
    // Flag whether the key properties where found
    keyLayoutFound = false;

	// Default values
    activeKey = '0';
    activeX = 0;
	activeY = 0;
    activeColor = 0;
    activeForm = 0;
    activeModifier = 0;
    activeFinger = 8;
    activeModifierX = 0;
	activeModifierY = 0;
    activeModifierColor = 0;
    activeModifierForm = 0;
    activeModifierFinger = 8;

    // Sql class to get the key layout of a given char
    keyboardSql = new KeyboardSql(tableName);
    // Fix the size of this class because of using fix sized images
    setFixedSize(610, 228);
}

void KeyBoard::startBoard() {
	// Turn start flag true
	isStarted = true;
	// Turn pause flag false
	isPaused = false;
	// Call the paint event (refresh the keyboard)
	repaint();
}

void KeyBoard::pauseBoard() {
	// Turn start flag false
	isStarted = false;
	// Turn pause flag true
	isPaused = true;
	// Call the paint event (refresh the keyboard)
	update();
}

void KeyBoard::stopBoard() {
	// Turn start flag false
	isStarted = false;
	// Call the paint event (refresh the keyboard)
	update();
}

void KeyBoard::paintEvent(QPaintEvent *event) {
	// Draw empty grey keyboard
	cleanKeyboard();

	if (isStarted) {
		// Color current key and modifier
		colorKey();
	} else {
		if (keyColoringStart) {
			// Color user's start position
			colorStart();
		}
	}
	// Draw keyboard labeling
	drawLayout();
}

void KeyBoard::loadKeyImages() {
	// Load background
	if (!loadImage(background, ":/img/keyboardbg.png")) {
		return;
	}
	// Keys
	// -> four different key colors
	for (int x = 0; x < 5; x++) {
		// -> six different key forms
		for (int y = 0; y < 7; y++) {
			// Load key
			if (!loadImage(keyImage[x][y], ":/img/key" + QString::number(x)
				+ "_" + QString::number(y) + ".png")) {
				return;
			}
		}
	}
	// -> five different start key colors
	for (int y = 0; y < 5; y++) {
		// Load key
		if (!loadImage(keyImageStart[y], ":/img/key" + QString::number(y)
			+ ".png")) {
			return;
		}
	}
	// Load key labeling
	if (!loadImage(keyLayout, ":/img/keylayout_" + imageName + ".png")) {
		return;
	}
	// Load keyboard border
	if (!loadImage(keyBorder, ":/img/keyboard_border.png")) {
		return;
	}
}

bool KeyBoard::loadImage(QImage &img, QString filename) {
	// Check whether loading the image is correct
	if (!img.load(filename)) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_KEY_PIC, TYPE_WARNING, CANCEL_OPERATION);
		return false;
	}
	return true;
}

void KeyBoard::cleanKeyboard() {
	int i;
	QPainter painter(this);
	// Background image
	painter.drawImage(0, 0, background);

    // Keys first row
    for (i = 0; i < 13; i++) {
    	painter.drawImage(10 + (40 * i), 15, keyImage[0][0]);
	}
	painter.drawImage(530, 15, keyImage[0][5]);

    // Enter key
    painter.drawImage(550, 55, keyImage[0][4]);

    // Keys second row
    painter.drawImage(10, 55, keyImage[0][6]);
    for (i = 0; i < 12; i++) {
    	painter.drawImage(70 + (40 * i), 55, keyImage[0][0]);
	}

    // Keys third row
    painter.drawImage(10, 95, keyImage[0][5]);
    if (keyColoringStart) {
		// Four keys left hand
		for (i = 3; i >= 0; i--) {
			if (i != activeFinger && i != activeModifierFinger) {
				painter.drawImage(80 + (40 * i), 95, keyImageStart[4 - i]);
			} else {
				painter.drawImage(80 + (40 * i), 95, keyImageStart[0]);
			}
		}
		painter.drawImage(80 + (40 * 4), 95, keyImage[0][0]);
		painter.drawImage(80 + (40 * 5), 95, keyImage[0][0]);
		// Four keys right hand
		for (i = 6; i < 10; i++) {
			if ((i - 2) != activeFinger && (i - 2) != activeModifierFinger) {
				painter.drawImage(80 + (40 * i), 95, keyImageStart[i - 5]);
			} else {
				painter.drawImage(80 + (40 * i), 95, keyImageStart[0]);
			}
		}
		painter.drawImage(80 + (40 * 10), 95, keyImage[0][0]);
		painter.drawImage(80 + (40 * 11), 95, keyImage[0][0]);
	} else {
    	for (i = 0; i < 12; i++) {
    		painter.drawImage(80 + (40 * i), 95, keyImage[0][0]);
		}
	}

    // Keys fourth row
    painter.drawImage(10, 135, keyImage[0][1]);
    for (i = 0; i < 11; i++) {
    	painter.drawImage(60 + (40 * i), 135, keyImage[0][0]);
	}
	painter.drawImage(500, 135, keyImage[0][2]);

	// Keys fifth row
    painter.drawImage(10, 175, keyImage[0][6]);
    painter.drawImage(70, 175, keyImage[0][0]);
	painter.drawImage(110, 175, keyImage[0][6]);
	painter.drawImage(170, 175, keyImage[0][3]);
	painter.drawImage(440, 175, keyImage[0][6]);
	painter.drawImage(500, 175, keyImage[0][0]);
	painter.drawImage(540, 175, keyImage[0][6]);
}

void KeyBoard::drawLayout() {
	QPainter painter(this);
	// Draw the labeling of the keyboard
	painter.drawImage(10, 15, keyLayout);
	// Draw the border of the keyboard
	if (keyDrawBorder) {
		painter.drawImage(270, 15, keyBorder);
	}
}

void KeyBoard::colorStart() {
	QPainter painter(this);

	if (keyColoring && !isPaused) {
		// User's startposition
		int i;
		// Four keys left hand
		for (i = 3; i >= 0; i--) {
			painter.drawImage(80 + (40 * i), 95, keyImageStart[4 - i]);
		}
		// Four keys right hand
		for (i = 6; i < 10; i++) {
			painter.drawImage(80 + (40 * i), 95, keyImageStart[i - 5]);
		}
	}
	// Colored space key
	painter.drawImage(170, 175, keyImage[1][3]);
}

// Places the current colored keys
void KeyBoard::colorKey() {
	QPainter painter(this);

	if (keyColoring && keyLayoutFound) {
		// Color current key
		painter.drawImage(activeX, activeY, keyImage[activeColor][activeForm]);
		if (activeModifier != 0) {
			// Color current modifier
			painter.drawImage(activeModifierX, activeModifierY,
				keyImage[activeModifierColor][activeModifierForm]);
		}
	}
}

void KeyBoard::setKey(QChar key) {
	QString statusText = "0";

	// Check whether key is already colored and whether keyboard is started
	if (activeKey != key && isStarted) {
		activeModifierFinger = 8;
		activeFinger = 8;
		// New current key
		activeKey = key;
		// Get the properties of the key
		if (keyboardSql->getKeyLayout(activeKey, &activeX, &activeY,
			&activeColor, &activeForm, &activeModifier, &activeFinger, &statusText)) {
			keyLayoutFound = true;
			if (activeModifier != 0) {
				// If there is also a modifier get the properties, too
				if (!keyboardSql->getModifierLayout(activeModifier,
					&activeModifierX, &activeModifierY, &activeModifierColor,
					&activeModifierForm, &activeModifierFinger, &statusText)) {
					activeModifier = 0;
				}
			}
		} else {
			// No properties were found
			keyLayoutFound = false;
		}
		// Emit the help text of current key(s)
		emit statusRefreshed(statusText);
    	// Call the paint event (refresh the keyboard)
		repaint();
	}
}

void KeyBoard::readSettings() {
	QSettings settings;

	// Check user's keyboard layout (win or mac)
	settings.beginGroup("settings");
	opSystem = "win";
	opLayout = "de";
	chLayout = "sg";
	switch (settings.value("combo_layout_ch", 0).toInt()) {
		case 0:
			chLayout = "sg";
			break;
		case 1:
			chLayout = "sf";
			break;
	}
	switch (settings.value("combo_layout_os",
		KEYBOARD_LAYOUT_MAC).toInt()) {
		case 0:
			opSystem = "win";
			break;
		case 1:
			opSystem = "mac";
			break;
	}
	switch (settings.value("combo_layout",
		KEYBOARD_LAYOUT_STANDARD).toInt()) {
		case 0:
			opLayout = "de";
			chLayout = "";
			break;
		case 1:
			opLayout = "ch";
			break;
		case 2:
			opLayout = "neo";
			opSystem = "";
			chLayout = "";
			break;
		case 3:
			opLayout = "dvorak";
			opSystem = "";
			chLayout = "";
			break;
		case 4:
			opLayout = "ristome";
			opSystem = "";
			chLayout = "";
			break;
	}
	imageName = opLayout;
	if (opSystem != "") {
		imageName.append("_" + opSystem);
	}
	tableName = imageName;
	if (opLayout == "ch") {
		tableName.append("_" + chLayout);
	}
	settings.endGroup();
	// Check if key coloring is turned on
	settings.beginGroup("support");
	keyColoring = settings.value("check_selection", true).toBool();
	keyColoringStart = settings.value("check_selection_start", true).toBool();
	keyDrawBorder = settings.value("check_border", true).toBool();
	settings.endGroup();
}
