/*
Copyright (c) 2006-2007, Tom Thielicke IT Solutions

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.
*/

/****************************************************************
**
** Implementation of the LessonTableSql class
** File name: lessontablesql.h
**
****************************************************************/

#include <QObject>
#include <QItemSelectionModel>
#include <QAbstractItemView>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QColor>
#include <QFont>
#include <QDateTime>
#include <math.h>

#include "lessontablesql.h"
#include "def/defines.h"

LessonSqlModel::LessonSqlModel(int row, int type, QWidget *parent) : QSqlQueryModel(parent) {
	parentWidget = parent;
	lastIdInserted = row;
	lastTypeInserted = type;
}

QVariant LessonSqlModel::data(const QModelIndex &index, int role) const {
	QVariant value = QSqlQueryModel::data(index, role);
	QDateTime timeStamp;
	double lessonGrade;
	int timeSec;
	double timeMin;
	static int coloredRow = -1;
	QString lessonName;
	if (value.isValid() && role == Qt::DisplayRole) {
		if (index.column() == 0) {
			// Expand lesson name
			lessonName = value.toString();
			return lessonName; //.prepend(QObject::tr("bungslektion "));
		}
		if (index.column() == 1) {
			// Convert time stamp into a readable format
			timeStamp = QDateTime::fromString(value.toString(), "yyyyMMddhhmmss");
			return timeStamp.toString(DATE_TIME_FORMAT) + QObject::tr(" Uhr");
		}
		if (index.column() == 2) {
			// Show time length in seconds or minutes
			if ((timeSec = value.toInt()) < 60) {
				return QString::number(timeSec) + QObject::tr(" sek.");
			} else {
				timeMin = floor((timeSec / 60.0) / 0.1 + 0.5) * 0.1;
				return QString::number(timeMin) + QObject::tr(" min.");
			}
		}
		if (index.column() == 6) {
			// There is never grade smaller than zero
			if ((lessonGrade = value.toDouble()) < 0) {
				lessonGrade = 0;
			}
			return QString::number(lessonGrade) + QObject::tr(" Punkte");
		}
		if (index.column() == 7 && value.toInt() == lastIdInserted) {
			// Current row has to be colored
			coloredRow = index.row();
			return QString::number(coloredRow);
		}
		if (lastIdInserted == 0) {
			// No row is colored
			coloredRow = -1;
		}
	}
	if (role == Qt::FontRole && (index.column() == 0 || index.column() == 6)) {
		// Show the lesson number bold
		QFont font;
		font = parentWidget->font();
		font.setBold(true);
		return qVariantFromValue(font);
	}
	/*if (role == Qt::BackgroundColorRole) {
		if (index.row() == coloredRow) {
			// Show the row with background color
			return qVariantFromValue(QColor(249, 126, 50));
		}
	}*/
	//if (role == Qt::TextColorRole
	return value;
}

LessonTableSql::LessonTableSql(int row, int type, QWidget *parent) : QWidget(parent) {
	previousColumnIndex = -1;
	whereClausel = "";

	// Create filter headline
	labelFilter = new QLabel(QObject::tr("Zeige: "));
	comboFilter = new QComboBox();
	comboFilter->insertItem(0, QObject::tr("Alle Lektionen"));
	comboFilter->insertItem(1, QObject::tr("bungslektionen"));
	comboFilter->insertItem(2, QObject::tr("Freie Lektionen"));
	comboFilter->insertItem(3, QObject::tr("Eigene Lektionen"));

	// Create QModel object
	model = new LessonSqlModel(row, type, this);

	// Column headers (see sql query)
	model->setHeaderData(0, Qt::Horizontal, QObject::tr("Lektion"));
	model->setHeaderData(1, Qt::Horizontal, QObject::tr("Zeitpunkt"));
	model->setHeaderData(2, Qt::Horizontal, QObject::tr("Dauer"));
	model->setHeaderData(3, Qt::Horizontal, QObject::tr("Zeichen"));
	model->setHeaderData(4, Qt::Horizontal, QObject::tr("Fehler"));
	model->setHeaderData(5, Qt::Horizontal, QObject::tr("A/min"));
	model->setHeaderData(6, Qt::Horizontal, QObject::tr("Bewertung"));

	//model->setSort(0, Qt::AscendingOrder);

	// Create the table view
	view = new QTableView;
	view->setModel(model);
	// User should not be able to select a row
	view->setSelectionMode(QAbstractItemView::NoSelection);
	// Hide the lesson id column

	headerview = view->horizontalHeader();

	// Set the sql query (every lesson, it's properties and rating)
	sortColumn(-1);

	headerview->setStretchLastSection(true);
	headerview->setResizeMode(QHeaderView::Interactive);
	headerview->setSortIndicatorShown(true);

	// Resize the columns
	view->resizeColumnsToContents();
	view->setColumnHidden(7, true);
	view->resizeColumnsToContents();
	//view->setColumnHidden(8, true);

	// Set a horizonal layout
	QHBoxLayout *filterLayout = new QHBoxLayout;
    filterLayout->addStretch(1);
	filterLayout->addWidget(labelFilter);
	filterLayout->addWidget(comboFilter);
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addLayout(filterLayout);
	mainLayout->addWidget(view);
	// Pass layout to parent widget (this)
	this->setLayout(mainLayout);

	connect(headerview, SIGNAL(sectionClicked(int)), this, SLOT(sortColumn(int)));
	connect(comboFilter, SIGNAL(activated(int)), this, SLOT(changeFilter(int)));
}

void LessonTableSql::sortColumn(int columnindex) {
	// Select columnname from columnindex
	switch (columnindex) {
		case 0:
			columnName = "user_lesson_lesson";
			break;
		case 1:
		default:
			columnName = "user_lesson_timestamp";
			columnindex = 1;
			break;
		case 2:
			columnName = "user_lesson_timelen";
			break;
		case 3:
			columnName = "user_lesson_tokenlen";
			break;
		case 4:
			columnName = "user_lesson_errornum";
			break;
		case 5:
			columnName = "user_lesson_cpm";
			break;
		case 6:
			columnName = "user_lesson_grade";
			break;
	}
	if (previousColumnIndex != columnindex) {
		isDesc = 1;
		headerview->setSortIndicator(columnindex, Qt::DescendingOrder);
	} else {
		if (headerview->sortIndicatorOrder() == Qt::AscendingOrder) {
			isDesc = 0;
		} else {
			isDesc = 1;
		}
	}
	previousColumnIndex = columnindex;
	if (columnindex != -1) {
		model->lastIdInserted = 0;
	}
	// Call new query
	setQueryOrder(columnName, isDesc);
}

void LessonTableSql::changeFilter(int rowindex) {
	// Select columnname from columnindex
	switch (rowindex) {
		case 0:
			whereClausel = "";
			break;
		case 1:
		default:
			whereClausel = "WHERE user_lesson_type = 0 ";
			break;
		case 2:
			whereClausel = "WHERE user_lesson_type = 1 ";
			break;
		case 3:
			whereClausel = "WHERE user_lesson_type = 2 ";
			break;
	}
	model->lastIdInserted = 0;
	// Call new query
	setQueryOrder(columnName, isDesc);
}

void LessonTableSql::setQueryOrder(QString columnname, int isdesc) {
	QString descText;
	if (isdesc == 1) {
		descText = " DESC";
	} else {
		descText = " ASC";
	}

	// Set the sql query (every lesson, it's properties and rating)
	model->setQuery("SELECT user_lesson_name, user_lesson_timestamp, "
		"user_lesson_timelen, user_lesson_tokenlen, "
		"user_lesson_errornum, ROUND(user_lesson_strokesnum / "
		"(user_lesson_timelen / 60.0), 0) AS user_lesson_cpm, "
		"ROUND(((user_lesson_strokesnum - (20 * user_lesson_errornum)) / "
		"(user_lesson_timelen / 60.0)) * 0.4, 0) AS user_lesson_grade, "
		"user_lesson_id FROM user_lesson_list " +
		whereClausel + "ORDER BY " + columnname + descText + ";");
}
