/*
Copyright (c) 2006-2007, Tom Thielicke IT Solutions

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.
*/

/****************************************************************
**
** Implementation of the CharTableSql class
** File name: chartablesql.cpp
**
****************************************************************/

#include <QTableView>
#include <QObject>
#include <QItemSelectionModel>
#include <QAbstractItemView>
#include <QHBoxLayout>
#include <QColor>
#include <QFont>

#include "chartablesql.h"

CharSqlModel::CharSqlModel(QWidget *parent) : QSqlQueryModel(parent) {
	// Remember the parent widget
	parentWidget = parent;
}

QVariant CharSqlModel::data(const QModelIndex &index, int role) const {
	QVariant value = QSqlQueryModel::data(index, role);
	int unicode;
	QString unicodeToChar;
	QString errorRatio;
	if (value.isValid() && role == Qt::DisplayRole) {
		// First column
		if (index.column() == 0) {
			// Read the unicode value
			unicode = value.toInt();
			// Convert unicode to a char
			unicodeToChar = QString::QString(QChar(unicode)); //"\'" + QString::QString(QChar(unicode)) + "\'";
			return unicodeToChar;
		} else {
			// Last column (error weight)
			if (index.column() == 4) {
				// If there is no error ratio (this happens if the denominator
				// of error weight calculation is zero
				if ((errorRatio = value.toString()) == "") {
					errorRatio = QObject::tr("0");
				}
				// Append a percent sign
				return errorRatio.append(QObject::tr(" %"));
			}
		}
	}
	// Change font of first column to bold
	if (role == Qt::FontRole && index.column() == 0) {
		QFont font;
		font = parentWidget->font();
		font.setFamily("Courier New");
		font.setPointSize(font.pointSize() + 2);
		font.setBold(true);
		return qVariantFromValue(font);
	}
	// Change font of first column to bold
	if (role == Qt::FontRole && index.column() == 4) {
		QFont font;
		font = parentWidget->font();
		font.setBold(true);
		return qVariantFromValue(font);
	}
	return value;
}

CharTableSql::CharTableSql(QWidget *parent) : QWidget(parent) {
	previousColumnIndex = -1;
	// Create QModel object
	model = new CharSqlModel(this);

	// Column headers (see sql query)
	model->setHeaderData(0, Qt::Horizontal, QObject::tr("Schriftzeichen"));
	model->setHeaderData(1, Qt::Horizontal, QObject::tr("Soll-Fehler"));
	model->setHeaderData(2, Qt::Horizontal, QObject::tr("Ist-Fehler"));
	model->setHeaderData(3, Qt::Horizontal, QObject::tr("Vorkommen"));
	model->setHeaderData(4, Qt::Horizontal, QObject::tr("Fehlerquote"));


	// Create the table view
	QTableView *view = new QTableView;
	view->setModel(model);
	// User should not be able to select a row
	view->setSelectionMode(QAbstractItemView::NoSelection);

	headerview = view->horizontalHeader();

	// Set the sql query (every unicode, it's errors and an error weight)
	sortColumn(4);

	headerview->setStretchLastSection(true);
	headerview->setResizeMode(QHeaderView::Interactive);
	headerview->setSortIndicatorShown(true);

	// Resize the columns
	view->resizeColumnsToContents();

	// Set a horizonal layout
	QHBoxLayout *mainLayout = new QHBoxLayout;
	mainLayout->addWidget(view);
	// Pass layout to parent widget (this)
	this->setLayout(mainLayout);

	connect (headerview, SIGNAL(sectionClicked(int)), this, SLOT(sortColumn(int)));
}

void CharTableSql::sortColumn(int columnindex) {
	// Select columnname from columnindex
	QString columnName;
	int isDesc;
	switch (columnindex) {
		case 0:
			columnName = "user_char_unicode";
			break;
		case 1:
			columnName = "user_char_target_errornum";
			break;
		case 2:
			columnName = "user_char_mistake_errornum";
			break;
		case 3:
			columnName = "user_char_occur_num";
			break;
		case 4:
		default:
			columnName = "(user_char_target_errornum * 100.0) / user_char_occur_num";
			break;
	}
	if (previousColumnIndex != columnindex) {
		isDesc = 1;
		headerview->setSortIndicator(columnindex, Qt::DescendingOrder);
	} else {
		if (headerview->sortIndicatorOrder() == Qt::AscendingOrder) {
			isDesc = 0;
		} else {
			isDesc = 1;
		}
	}
	previousColumnIndex = columnindex;
	// Call new query
	setQueryOrder(columnName, isDesc);
}

void CharTableSql::setQueryOrder(QString columnname, int isdesc) {
	QString descText;
	if (isdesc == 1) {
		descText = " DESC";
	} else {
		descText = " ASC";
	}
	// Set the sql query (every unicode, it's errors and an error weight)
	model->setQuery("SELECT user_char_unicode, user_char_target_errornum, "
		"user_char_mistake_errornum, user_char_occur_num, "
		"ROUND((user_char_target_errornum * 100.0) / user_char_occur_num, 0) "
		"AS user_char_weighted FROM user_chars ORDER BY "
		+ columnname + descText + ";");

}
