//Please read license.txt for licensing and copyright information

#include "headers.h"
#include "db.h"
#include "rpc.h"
#include "net.h"
#include "init.h"
#include "strlcpy.h"
#include <boost/filesystem.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/interprocess/sync/file_lock.hpp>

#ifdef __WXMSW__
#include <boost/filesystem/detail/utf8_codecvt_facet.hpp>
#endif

using namespace std;
using namespace boost;

void SolidCoinMineController(void* parg);

void ExitTimeout(void* parg)
{
#ifdef __WXMSW__
    Sleep(5000);
    ExitProcess(0);
#endif
}



void Shutdown(void* parg)
{
    static CCriticalSection cs_Shutdown;
    static bool fTaken;
    bool fFirstThread;
    {
        MUTEX_LOCK(cs_Shutdown);
        fFirstThread = !fTaken;
        fTaken = true;
    }
    static bool fExit;
    if (fFirstThread)
    {
        fShutdown = true;
        nTransactionsUpdated++;
        DBFlush(false);
        StopNode();
        DBFlush(true);
        #ifndef __WXMSW__
        boost::filesystem::remove(GetPidFile());
        #endif
        Wallet_Cleanup();
        SaveMiningPools();
        SaveConfigFile();
        BlockHash_DeInit();

        CreateThread(ExitTimeout, NULL);
        Sleep(50);
        debugprintf(ALWAYS, "SolidCoin exiting\n\n");
        fExit = true;
        exit(0);
    }
    else
    {
        while (!fExit)
            Sleep(500);
        Sleep(100);
        ExitThread(0);
    }
}

void HandleSIGTERM(int)
{
    fRequestShutdown = true;
}

#ifndef GUI
int main(int argc, char* argv[])
{
    bool fRet = false;
    std::string *str = 0;
    if(argc) str=new std::string[argc];
    for(int i=0;i<argc;i++) str[i]=argv[i];
    fRet = ServiceInit(argc,str);
    delete[] str;

    if (fRet && fDaemon)
        return 0;

    return 1;
}
#endif

bool AppInit(int argc, std::string *argv)
{
    bool fRet = false;
    try
    {
        fRet = AppInit2(argc, argv);
    }
    catch (std::exception& e) {
        PrintException(&e, "AppInit()");
    } catch (...) {
        PrintException(NULL, "AppInit()");
    }
    if (!fRet)
        Shutdown(NULL);
    return fRet;
}

void DoWallet(const string &name, string &errors, bool bClean)
{
    CWallet *pwallet = new CWallet(name);
    int nLoadWalletRet = pwallet->LoadWallet(bClean);
    if (nLoadWalletRet != DB_LOAD_OK)
    {
        if (nLoadWalletRet == DB_CORRUPT)           errors += str(boost::format("Error loading %s due to wallet corruption\n") % pwallet->GetWalletFileName());
        else if (nLoadWalletRet == DB_TOO_NEW)      errors += str(boost::format("Error loading %s Wallet requires newer version of SolidCoin\n") % pwallet->GetWalletFileName());
        else                                        errors += str(boost::format("Error loading %s\n") % pwallet->GetWalletFileName());
    }
    RegisterWallet(pwallet);
    debugprintf(INFO, "wallet %s loaded\n",pwallet->GetWalletFileName().c_str());
}
bool AppInit2(int argc, std::string *argv)
{
#ifdef _MSC_VER
    // Turn off microsoft heap dump noise
    _CrtSetReportMode(_CRT_WARN, _CRTDBG_MODE_FILE);
    _CrtSetReportFile(_CRT_WARN, CreateFileA("NUL", GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, 0));
#endif
#if _MSC_VER >= 1400
    // Disable confusing "helpful" text message on abort, ctrl-c
    _set_abort_behavior(0, _WRITE_ABORT_MSG | _CALL_REPORTFAULT);
#endif
#ifndef __WXMSW__
    umask(077);
#endif
#ifndef __WXMSW__
    // Clean shutdown on SIGTERM
    struct sigaction sa;
    sa.sa_handler = HandleSIGTERM;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = 0;
    sigaction(SIGTERM, &sa, NULL);
    sigaction(SIGINT, &sa, NULL);
    sigaction(SIGHUP, &sa, NULL);
#endif

#ifdef __WXMSW__
    std::locale global_loc = std::locale();
    std::locale loc(global_loc, new boost::filesystem::detail::utf8_codecvt_facet);
    std::locale old_loc = boost::filesystem::path::imbue(loc);
#endif

    // Parameters
    ParseParameters(argc, argv);

    //some value tests on blocks
    int64 nRetDiff1_0=Block_GetCoinBaseValue(0x1E7FFFFF, 180001+0);                 //diff 1 at 0 years
    int64 nRetDiff1_1=Block_GetCoinBaseValue(0x1E7FFFFF, 180001+525600);            //diff 1 at 1 years
    int64 nRetDiff20697_0=Block_GetCoinBaseValue(0x1D01954D, 180001+0);             //diff 20697 at 0 years
    int64 nRetDiff20697_1=Block_GetCoinBaseValue(0x1D01954D, 180001+525600);        //diff 20697 at 1 years
    int64 nRetDiff20697_5=Block_GetCoinBaseValue(0x1D01954D, 180001+(525600*5));    //diff 20697 at 5 years
    int64 nRetDiff20697_10=Block_GetCoinBaseValue(0x1D01954D, 180001+(525600*10));  //diff 20697 at 10 years

    if(nRetDiff1_0!=501 || nRetDiff1_1!=501 || nRetDiff20697_0!=700 || nRetDiff20697_1!=647  || nRetDiff20697_5!=542 || nRetDiff20697_10!=509)
    {
        fprintf(stderr, "Error: Block Rewards aren't correct\n");
        Shutdown(NULL);
    }


    if (CMDSetting_Exist("datadir"))
    {
        filesystem::path pathDataDir = filesystem::system_complete(CMDSetting_Get("datadir"));
        std::string datadir=CMDSetting_Get("datadir");
        if (filesystem::is_directory(datadir))
        {

            strlcpy(pszSetDataDir, pathDataDir.string().c_str(), sizeof(pszSetDataDir));
        }
        else
        {
            fprintf(stderr, "Error: Specified directory does not exist\n");
            Shutdown(NULL);
        }
    }
    BlockHash_Init();
    ReadConfigFile(); // Must be done after processing datadir
    InitMiningPools();
    if(!SolidTime_Init()) return false;

    extern std::string g_TrustedNodePublicKey;
    g_TrustedNodePublicKey = Setting_Get("trustednode");

    if (CMDSetting_Exist("?") || CMDSetting_Exist("-help"))
    {
        string strUsage = string() +
          _("SolidCoin version") + " " + FormatFullVersion() + "\n\n" +
          _("Usage:") + "\t\t\t\t\t\t\t\t\t\t\n" +
            "  solidcoin [options]                   \t  " + "\n" +
            "  solidcoin [options] <command> [params]\t  " + _("Send command to -server or solidcoind\n") +
            "  solidcoin [options] --help              \t\t  " + _("List commands\n") +
            "  solidcoin [options] help <command>    \t\t  " + _("Get help for a command\n") +
          _("Options:\n") +
            "  -conf=<file>     \t\t  " + _("Specify configuration file (default: solidcoin.conf)\n") +
            "  -pid=<file>      \t\t  " + _("Specify pid file (default: solidcoin.pid)\n") +
            "  -min             \t\t  " + _("Start minimized\n") +
            "  -datadir=<dir>   \t\t  " + _("Specify data directory\n") +
            "  -timeout=<n>     \t  "   + _("Specify connection timeout (in milliseconds)\n") +
            "  -proxy=<ip:port> \t  "   + _("Connect through socks4 proxy\n") +
            "  -dns             \t  "   + _("Allow DNS lookups for addnode and connect\n") +
            "  -addnode=<ip>    \t  "   + _("Add a node to connect to\n") +
            "  -connect=<ip>    \t\t  " + _("Connect only to the specified node\n") +
            "  -nolisten        \t  "   + _("Don't accept connections from outside\n") +
#ifdef USE_UPNP
#if USE_UPNP
            "  -upnp=0          \t  "   + _("Don't attempt to use UPnP to map the listening port\n") +
#else
            "  -upnp=1            \t  "   + _("Attempt to use UPnP to map the listening port\n") +
#endif
#endif
#ifdef GUI
            "  -server          \t\t  " + _("Accept command line and JSON-RPC commands\n") +
#endif
#ifndef __WXMSW__
            "  -daemon          \t\t  " + _("Run in the background as a daemon and accept commands\n") +
#elif NOGUI
            "  -serviceinstall  \t\t  "   + _("Installs a service on Windows\n") +
            "  -serviceremove  \t\t  "   + _("Installs a service on Windows\n") +
#endif
            "  -testnet         \t\t  " + _("Use the test network\n") +
            "  -rpcuser=<user>  \t  "   + _("Username for JSON-RPC connections\n") +
            "  -rpcpassword=<pw>\t  "   + _("Password for JSON-RPC connections\n") +
            "  -rpcport=<port>  \t\t  " + _("Listen for JSON-RPC connections on <port> (default: 8555)\n") +
            "  -rpcallowip=<ip> \t\t  " + _("Allow JSON-RPC connections from specified IP address\n") +
            "  -rpcconnect=<ip> \t  "   + _("Send commands to node running on <ip> (default: 127.0.0.1)\n") +
            "  -keypool=<n>     \t  "   + _("Set key pool size to <n> (default: 100)\n") +
            "  -rescan          \t  "   + _("Rescan the block chain for missing wallet transactions\n")+
            "  -cleanwallet     \t  "   + _("Cleans wallet of transactions and rescans blockchain\n");



#ifdef USE_SSL
        strUsage += string() +
            _("\nSSL options: (see the SolidCoin Wiki for SSL setup instructions)\n") +
            "  -rpcssl                                \t  " + _("Use OpenSSL (https) for JSON-RPC connections\n") +
            "  -rpcsslcertificatechainfile=<file.cert>\t  " + _("Server certificate file (default: server.cert)\n") +
            "  -rpcsslprivatekeyfile=<file.pem>       \t  " + _("Server private key (default: server.pem)\n") +
            "  -rpcsslciphers=<ciphers>               \t  " + _("Acceptable ciphers (default: TLSv1+HIGH:!SSLv2:!aNULL:!eNULL:!AH:!3DES:@STRENGTH)\n");
#endif

        strUsage += string() +
            "  -?               \t\t  " + _("This help message\n");

#if defined(__WXMSW__) && defined(GUI)
        // Tabs make the columns line up in the message box
        wxMessageBox(strUsage, "SolidCoin", wxOK);
#else
        // Remove tabs
        strUsage.erase(std::remove(strUsage.begin(), strUsage.end(), '\t'), strUsage.end());
        fprintf(stderr, "%s", strUsage.c_str());
#endif
        return false;
    }

    fDebug = Setting_GetBOOL("debug");
    fAllowDNS = Setting_GetBOOL("dns");
#ifndef __WXMSW__
    fDaemon = Setting_GetBOOL("daemon");
#elif NOGUI
    if(Setting_Exist("serviceinstall"))
    {
        ServiceCreate();
        return true;
    }
    if(Setting_Exist("serviceremove"))
    {
        ServiceDelete();
        return true;
    }
    fDaemon = false;
#endif

    if (fDaemon)    fServer = true;
    else            fServer = Setting_GetBOOL("server");

    /* force fServer when running without GUI */
#ifndef GUI
    fServer = true;
#endif

    fPrintToConsole = Setting_GetBOOL("printtoconsole");
    fPrintToDebugger = Setting_GetBOOL("printtodebugger");

    fTestNet = Setting_GetBOOL("testnet");
    bool fTOR = (g_bUseProxy && addrProxy.port == htons(9050));
    fNoListen = Setting_GetBOOL("nolisten") || fTOR;
    fLogTimestamps = Setting_GetBOOL("logtimestamps");
	fLogDebugLevel = Setting_GetINT64("debuglevel");

    for (int i = 1; i < argc; i++)
        if (!IsSwitchChar(argv[i][0]))
            fCommandLine = true;

    if (fCommandLine)
    {
        int ret = CommandLineRPC(argc, argv);
        exit(ret);
    }

#ifndef __WXMSW__
    if (fDaemon)
    {
        // Daemonize
        pid_t pid = fork();
        if (pid < 0)
        {
            fprintf(stderr, "Error: fork() returned %d errno %d\n", pid, errno);
            return false;
        }
        if (pid > 0)
        {
            CreatePidFile(GetPidFile(), pid);
            return true;
        }

        pid_t sid = setsid();
        if (sid < 0)
            fprintf(stderr, "Error: setsid() returned %d errno %d\n", sid, errno);
    }
#endif

    if (!fDebug && !pszSetDataDir[0])
        ShrinkDebugFile();
    debugprintf(ALWAYS, "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n");
    debugprintf(ALWAYS, "SolidCoin version %s\n", FormatFullVersion().c_str());
#ifdef GUI
    debugprintf(ALWAYS, "OS version %s\n", ((string)wxGetOsDescription()).c_str());
    debugprintf(ALWAYS, "System default language is %d %s\n", g_locale.GetSystemLanguage(), ((string)g_locale.GetSysName()).c_str());
    debugprintf(ALWAYS, "Language file %s (%s)\n", (string("locale/") + (string)g_locale.GetCanonicalName() + "/LC_MESSAGES/solidcoin.mo").c_str(), ((string)g_locale.GetLocale()).c_str());
#endif
    debugprintf(ALWAYS, "Default data directory %s\n", GetDefaultDataDir().c_str());

    if (CMDSetting_Exist("loadblockindextest"))
    {
        CTxDB txdb("r");
        txdb.LoadBlockIndex();

        Block_PrintTree();
        return false;
    }

    //
    // Limit to single instance per user
    // Required to protect the database files if we're going to keep deleting log.*
    //
#if defined(__WXMSW__) && defined(GUI)
    // wxSingleInstanceChecker doesn't work on Linux
    wxString strMutexName = wxString("solidcoin_running.") + getenv("HOMEPATH");
    for (int i = 0; i < strMutexName.size(); i++)
        if (!isalnum(strMutexName[i]))
            strMutexName[i] = '.';
    wxSingleInstanceChecker* psingleinstancechecker = new wxSingleInstanceChecker(strMutexName);
    if (psingleinstancechecker->IsAnotherRunning())
    {
        debugprintf(ERR, "Existing instance found\n");
        unsigned int nStart = SolidTime_Get();
        loop
        {
            // Show the previous instance and exit
            HWND hwndPrev = FindWindowA("wxWindowNR", "SolidCoin 2.041");
            if (hwndPrev)
            {
                if (IsIconic(hwndPrev))
                    ShowWindow(hwndPrev, SW_RESTORE);
                SetForegroundWindow(hwndPrev);
                return false;
            }

            if (SolidTime_Get() > nStart + 60)
                return false;

            // Resume this instance if the other exits
            delete psingleinstancechecker;
            Sleep(1000);
            psingleinstancechecker = new wxSingleInstanceChecker(strMutexName);
            if (!psingleinstancechecker->IsAnotherRunning())
                break;
        }
    }
#endif

    // Make sure only a single solidcoin process is using the data directory.
    string strLockFile;
#ifndef __WXMSW__
    strLockFile = GetDataDir() + "/.lock";
    {
        //filesystem::ofstream streamLock(strLockFile);
        //streamLock.close();
    }
    FILE* file = fopenSOLID(strLockFile.c_str(), "a"); // empty lock file; created if it doesn't exist.
    if (file) fclose(file);
    static boost::interprocess::file_lock lock(strLockFile.c_str());
    if (!lock.try_lock())
    {
        wxMessageBox(strprintf(_("Cannot obtain a lock on data directory %s.  SolidCoin is probably already running."), GetDataDir().c_str()), "SolidCoin");
        return false;
    }
    #endif

    // Bind to the port early so we can tell if another instance is already running.
    string strErrors;
    if (!fNoListen)
    {
        if (!BindListenPort(strErrors))
        {
            wxMessageBox(strErrors, "SolidCoin");
            return false;
        }
    }

    // Load data files
    if (fDaemon)    fprintf(stdout, "solidcoin server starting\n");
    strErrors = "";
    int64 nStart;

    debugprintf(ALWAYS, "Loading addresses...\n");
    nStart = GetTimeMillis();
    if (!LoadAddresses())   strErrors += _("Error loading addr.dat      \n");
    debugprintf(ALWAYS, " addresses   %15"PRI64d"ms\n", GetTimeMillis() - nStart);

    debugprintf(ALWAYS, "Loading block index...\n");
    nStart = GetTimeMillis();

    if (!Block_LoadIndex())
        strErrors += _("Error loading blkindex.dat      \n");
    debugprintf(ALWAYS, " block index %15"PRI64d"ms\n", GetTimeMillis() - nStart);

    debugprintf(ALWAYS, "Loading wallet...\n");
    nStart = GetTimeMillis();
    bool fFirstRun=false;

    //detect old wallet name format and migrate it to new format, only do so if new doesnt exist and old does
    bool bCleanWallet=false;
    if (CMDSetting_Exist("cleanwallet"))    bCleanWallet=true;

    boost::filesystem::path oldWallet= GetDataDir()+"/wallet.dat";
    boost::filesystem::path newWallet= GetDataDir()+"/wallet_main.dat";
    if(boost::filesystem::exists(oldWallet) && !boost::filesystem::exists(newWallet))
    {
        wxMessageBox("Moving wallet.dat to new filename wallet_main.dat and cleaning it","Upgrading wallet");
        boost::filesystem::rename(oldWallet,newWallet);
        bCleanWallet=true;
    }
    {
        LOCK_WALLET_ACCESS();

        boost::filesystem::path p= GetDataDir();
        boost::filesystem::directory_iterator dir_iter(p), dir_end;
        for(;dir_iter != dir_end; ++dir_iter)
        {
            if(boost::filesystem::is_directory(*dir_iter)) continue;
            std::string file = dir_iter->path().filename().string();
            string name=boost::filesystem::basename(file);
            if(name.find("wallet_")==0 && boost::filesystem::extension(file) == ".dat")
            {
                name=name.substr(7);
                DoWallet(name, strErrors,bCleanWallet);
            }
        }

        CWallet *pFirstWallet = Wallet_Get(0);
        if(!pFirstWallet)   //no wallets
        {
            DoWallet("main",strErrors,bCleanWallet);
        }

        CWallet*pWallet = Wallet_Get(0);
        if(pWallet==0)
        {
            //we might as well shit ourselves at this point because no wallet exists and its obvious that it failed to create one
            wxMessageBox("Couldn't find a wallet or create one");
            return false;
        }

        //set
        string defaultWalletName=Setting_Get("wallet_default");
        pWallet=Wallet_Get(defaultWalletName);
        if(!pWallet) pWallet=Wallet_Get(0);     //couldnt find default wallet, use first

        Wallet_SetDefault(pWallet);
        Wallet_SetSelected(pWallet);

        debugprintf(ALWAYS, "g_BlockIndexMap.size() = %d\n",   g_BlockIndexMap.size());
        debugprintf(ALWAYS, "g_qBlockBestHeight = %"PRI64d"\n",     g_qBlockBestHeight);

        int nWalletID=0;
        while((pWallet=Wallet_Get(nWalletID++)) )
        {
            CBlockIndex *pindexRescan = g_pBlockBestIndex;
            if (CMDSetting_Exist("rescan") || bCleanWallet)  pindexRescan = g_pBlockGenesisIndex;
            else
            {
                CWalletDB walletdb(pWallet->GetWalletFileName());
                CBlockLocator locator;
                if (walletdb.ReadBestBlock(locator))    pindexRescan = locator.GetBlockIndex();
            }
            if (g_pBlockBestIndex != pindexRescan)
            {
                debugprintf(ALWAYS, "Rescanning last %"PRI64d" blocks (from block %"PRI64d")...\n", g_pBlockBestIndex->blk.nBlockNum - pindexRescan->blk.nBlockNum, pindexRescan->blk.nBlockNum);
                nStart = GetTimeMillis();
                pWallet->ScanForWalletTransactions(pindexRescan, true);
                debugprintf(ALWAYS, " rescan      %15"PRI64d"ms\n", GetTimeMillis() - nStart);
            }

            //// debug print
            debugprintf(ALWAYS, "setKeyPool.size() = %d\n",      pWallet->setKeyPool.size());
            debugprintf(ALWAYS, "mapWallet.size() = %d\n",       pWallet->mapWallet.size());
            debugprintf(ALWAYS, "mapAddressBook.size() = %d\n",  pWallet->mapAddressBook.size());
            pWallet->ReacceptWalletTransactions();
        }
    }

    // Add wallet transactions that aren't already in a block to mapTransactions
    if (!strErrors.empty())
    {
        wxMessageBox(strErrors, "SolidCoin", wxOK | wxICON_ERROR);
        return false;
    }

    // Parameters
    if (CMDSetting_Exist("printblockindex") || CMDSetting_Exist("printblocktree"))
    {
        Block_PrintTree();
        return false;
    }

    if (Setting_Exist("timeout"))
    {
        int nNewTimeout = Setting_GetINT64("timeout");
        if (nNewTimeout > 0 && nNewTimeout < 600000)    nConnectTimeout = nNewTimeout;
    }

    if (CMDSetting_Exist("printblock"))
    {
        string strMatch = CMDSetting_Get("printblock");
        int nFound = 0;
        for (map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.begin(); mi != g_BlockIndexMap.end(); ++mi)
        {
            uint256 hash = (*mi).first;
            if (strncmp(hash.ToString().c_str(), strMatch.c_str(), strMatch.size()) == 0)
            {
                CBlockIndex* pindex = (*mi).second;
                CBlock block;
                block.ReadFromDisk(pindex);
                block.BuildMerkleTree();
                block.print();
                debugprintf(INFO, "\n");
                nFound++;
            }
        }
        if (nFound == 0)    debugprintf(WARN, "No blocks matching %s were found\n", strMatch.c_str());
        return false;
    }

    if (Setting_GetBOOL("proxy_use"))
    {
        g_bUseProxy = true;
        addrProxy = CAddress(Setting_Get("proxy_address")+":"+Setting_Get("proxy_port"));
        if (!addrProxy.IsValid())
        {
            wxMessageBox(_("Invalid -proxy address"), "SolidCoin");
            return false;
        }
    }

    if (Setting_Exist("addnode"))
    {
        BOOST_FOREACH(string strAddr, Setting_GetVector("addnode"))
        {
            CAddress addr(strAddr, fAllowDNS);
            addr.nTime = 0; // so it won't relay unless successfully connected
            if (addr.IsValid()) AddAddress(addr);
        }
    }

    //if (GetBoolArg("-nodnsseed"))
        //debugprintf(WARN, "DNS seeding disabled\n");
    //else
        //DNSAddressSeed(); //disabled until we have a dns seed
    //
    // Create the main window and start the node
    //
#ifdef GUI
    if (!fDaemon)
        CreateMainWindow();
#endif

    if (!CheckDiskSpace())
        return false;

    RandAddSeedPerfmon();

    if (!CreateThread(StartNode, NULL))
        wxMessageBox("Error: CreateThread(StartNode) failed", "SolidCoin");

    if (fServer)
        CreateThread(ThreadRPCServer, NULL);



    CreateThread(SolidCoinMineController, NULL);

#if defined(__WXMSW__) && defined(GUI)
    if (fFirstRun)
        SetStartOnSystemStartup(true);
#endif

#if defined(NOGUI)
    while (1)
    {
        if(ServiceQueryExit()) break;
        Sleep(500);
    }
    Shutdown(NULL);
#endif

    return true;
}
