//Please read license.txt for licensing and copyright information

//CHECKPOINTS
//Block 40000, check for correct ntimes in generates before allowing them to be spent
//Block 42000, do correct 5% CPF payments. Before was 5% of base only
//Block 43000, fixed millionaire work calculation
//block 50000, move to 5SC block value, powerblocks added (1 in 32 chance to double block value)
//block 91500, move to signing trust blocks so that they cannot be changed

#include "headers.h"
#include "db.h"
#include "net.h"
#include "init.h"
#include "cryptopp/sha.h"
#include <boost/filesystem.hpp>
#include <boost/filesystem/fstream.hpp>
#include "wallet_func.h"

using namespace std;
using namespace boost;

CCriticalSection cs_main;
map<uint256, CTransaction> mapTransactions;
CCriticalSection cs_mapTransactions;
unsigned int nTransactionsUpdated = 0;
map<COutPoint, CInPoint> mapNextTx;
map<uint256, CDataStream*> mapOrphanTransactions;
multimap<uint256, CDataStream*> mapOrphanTransactionsByPrev;

double dHashesPerSec;
int64 nHPSTimerStart;
const int64 COIN = 10000;
const int64 CENT = 100;
const int64 MIN_TX_FEE = 500;
const int COINBASE_MATURITY = 10;
const int VERSION = 203000;
const bool VERSION_IS_BETA = true;
// Threshold for nLockTime: below this value it is interpreted as block number, otherwise as UNIX timestamp.
const int LOCKTIME_THRESHOLD = 500000000; // Tue Nov  5 00:53:20 1985 UTC
#ifdef USE_UPNP
const int fHaveUPnP = true;
#else
const int fHaveUPnP = false;
#endif

//////////////////////////////////////////////////////////////////////////////
//
// mapOrphanTransactions
//

void static AddOrphanTx(const CDataStream& vMsg)
{
    CTransaction tx;
    CDataStream(vMsg) >> tx;
    uint256 hash = tx.GetHash();
    if (mapOrphanTransactions.count(hash))
        return;
    CDataStream* pvMsg = mapOrphanTransactions[hash] = new CDataStream(vMsg);
    BOOST_FOREACH(const CTxIn& txin, tx.vin)
        mapOrphanTransactionsByPrev.insert(make_pair(txin.prevout.hash, pvMsg));
}

void static EraseOrphanTx(uint256 hash)
{
    if (!mapOrphanTransactions.count(hash))
        return;
    const CDataStream* pvMsg = mapOrphanTransactions[hash];
    CTransaction tx;
    CDataStream(*pvMsg) >> tx;
    BOOST_FOREACH(const CTxIn& txin, tx.vin)
    {
        for (multimap<uint256, CDataStream*>::iterator mi = mapOrphanTransactionsByPrev.lower_bound(txin.prevout.hash);
             mi != mapOrphanTransactionsByPrev.upper_bound(txin.prevout.hash);)
        {
            if ((*mi).second == pvMsg)
                mapOrphanTransactionsByPrev.erase(mi++);
            else
                mi++;
        }
    }
    delete pvMsg;
    mapOrphanTransactions.erase(hash);
}













bool CheckDiskSpace(uint64 nAdditionalBytes)
{
    uint64 nFreeBytesAvailable = filesystem::space(GetDataDir()).available;

    // Check for 15MB because database could create another 10MB log file at any time
    if (nFreeBytesAvailable < (uint64)15000000 + nAdditionalBytes)
    {
        fShutdown = true;
        string strMessage = _("Warning: Disk space is low  ");
        strMiscWarning = strMessage;
        debugprintf(ALWAYS, "*** %s\n", strMessage.c_str());
        ThreadSafeMessageBox(strMessage, "SolidCoin", wxOK | wxICON_EXCLAMATION);
        CreateThread(Shutdown, NULL);
        return false;
    }
    return true;
}

FILE* OpenBlockFile(unsigned int nFile, unsigned int nBlockPos, const char* pszMode)
{
    if (nFile == -1)    return NULL;
    FILE* file = fopenSOLID(strprintf("%s/blk%04d.dat", GetDataDir().c_str(), nFile).c_str(), pszMode);
    if (!file)  return NULL;
    if (nBlockPos != 0 && !strchr(pszMode, 'a') && !strchr(pszMode, 'w'))
    {
        if (fseek(file, nBlockPos, SEEK_SET) != 0)
        {
            fclose(file);
            return NULL;
        }
    }
    return file;
}

static unsigned int nCurrentBlockFile = 1;

FILE* AppendBlockFile(unsigned int& nFileRet)
{
    nFileRet = 0;
    loop
    {
        FILE* file = OpenBlockFile(nCurrentBlockFile, 0, "ab");
        if (!file)
            return NULL;
        if (fseek(file, 0, SEEK_END) != 0)
            return NULL;
        // FAT32 filesize max 4GB, fseek and ftell max 2GB, so we must stay under 2GB
        if (ftell(file) < 0x7F000000 - MAX_SIZE)
        {
            nFileRet = nCurrentBlockFile;
            return file;
        }
        fclose(file);
        nCurrentBlockFile++;
    }
}

// CAlert

map<uint256, CAlert> mapAlerts;
CCriticalSection cs_mapAlerts;

string GetWarnings(string strFor)
{
    int nPriority = 0;
    string strStatusBar;
    string strRPC;
    if (Setting_GetBOOL("testsafemode"))
        strRPC = "test";

    // Misc warnings like out of disk space and clock is wrong
    if (strMiscWarning != "")
    {
        nPriority = 1000;
        strStatusBar = strMiscWarning;
    }

    // Longer invalid proof-of-work chain
    //if (pindexBest && bnBestInvalidWork > bnBestChainWork + pindexBest->GetBlockWork() * 6)
    {
        //nPriority = 2000;
        //strStatusBar = strRPC = "WARNING: Displayed transactions may not be correct!  You may need to upgrade, or other nodes may need to upgrade.";
    }

    // Alerts
    {
        MUTEX_LOCK(cs_mapAlerts);

        BOOST_FOREACH(PAIRTYPE(const uint256, CAlert)& item, mapAlerts)
        {
            const CAlert& alert = item.second;
            if (alert.AppliesToMe() && alert.nPriority > nPriority)
            {
                nPriority = alert.nPriority;
                strStatusBar = alert.strStatusBar;
            }
        }
    }

    if (strFor == "statusbar")  return strStatusBar;
    else if (strFor == "rpc")   return strRPC;
    assert(!"GetWarnings() : invalid parameter");
    return "error";
}

bool CAlert::ProcessAlert()
{
    if (!CheckSignature())  return false;
    if (!IsInEffect())      return false;

    {
        MUTEX_LOCK(cs_mapAlerts);

        // Cancel previous alerts
        for (map<uint256, CAlert>::iterator mi = mapAlerts.begin(); mi != mapAlerts.end();)
        {
            const CAlert& alert = (*mi).second;
            if (Cancels(alert))
            {
                debugprintf(ALWAYS, "cancelling alert %d\n", alert.nID);
                mapAlerts.erase(mi++);
            }
            else if (!alert.IsInEffect())
            {
                debugprintf(ALWAYS, "expiring alert %d\n", alert.nID);
                mapAlerts.erase(mi++);
            }
            else
                mi++;
        }

        // Check if this alert has been cancelled
        BOOST_FOREACH(PAIRTYPE(const uint256, CAlert)& item, mapAlerts)
        {
            const CAlert& alert = item.second;
            if (alert.Cancels(*this))
            {
                debugprintf(ALWAYS, "alert already cancelled by %d\n", alert.nID);
                return false;
            }
        }

        // Add to mapAlerts
        mapAlerts.insert(make_pair(GetHash(), *this));
    }

    debugprintf(ALWAYS, "accepted alert %d, AppliesToMe()=%d\n", nID, AppliesToMe());
    MainFrameRepaint();
    return true;
}








//////////////////////////////////////////////////////////////////////////////
//
// Messages
//


bool static AlreadyHave(CTxDB& txdb, const CInv& inv)
{
    switch (inv.type)
    {
    case MSG_TX:    return mapTransactions.count(inv.hash) || mapOrphanTransactions.count(inv.hash) || txdb.ContainsTx(inv.hash);
    case MSG_BLOCK: return g_BlockIndexMap.count(inv.hash) || g_BlockOrphanMap.count(inv.hash);
    }
    // Don't know what it is, just say we already got one
    return true;
}


bool IsHiddenAddress(const string& strAddress)
{
	//checks if the passed address is in the list of addresses to not distribute (hide)
	const vector<string>& vHide = Setting_GetVector("hidenode");
	BOOST_FOREACH(string strHide, vHide)
		if (WildcardMatch(strAddress, strHide))
			return true;
	return false;
}


// The message start string is designed to be unlikely to occur in normal data.
// The characters are rarely used upper ascii, not valid as UTF-8, and produce
// a large 4-byte int at any alignment.
//unsigned char pchMessageStart[4] = { 0xf9, 0xbe, 0xb4, 0xd9 };
unsigned char pchMessageStart[4] = { 0xde, 0x00, 0xba, 0xd9 };


bool static ProcessMessage(CNode* pfrom, string strCommand, CDataStream& vRecv)
{
    static map<unsigned int, vector<unsigned char> > mapReuseKey;
    RandAddSeedPerfmon();
    if (fDebug) debugprintf(INFO, "%s ", DateTimeStrFormat("%x %H:%M:%S", SolidTime_Get()).c_str());
    debugprintf(INFO, "received: %s (%d bytes)\n", strCommand.c_str(), vRecv.size());
    if (CMDSetting_Exist("dropmessagestest") && GetRand(atoi(CMDSetting_Get("dropmessagestest"))) == 0)
    {
        debugprintf(ALWAYS, "dropmessagestest DROPPING RECV MESSAGE\n");
        return true;
    }

    if (strCommand == "version")
    {
        // Each connection can only send one version message
        if (pfrom->nVersion != 0)
        {
            debugprintf(INFO, "version=0 from client1\n");
            return false;
        }

        int64 nTime;
        CAddress addrMe;
        CAddress addrFrom;
        uint64 nNonce = 1;
        vRecv >> pfrom->nVersion >> pfrom->nServices >> nTime >> addrMe;
        vRecv >> addrFrom >> nNonce;
        vRecv >> pfrom->strSubVer;
        vRecv >> pfrom->qStartingHeight;
        if (pfrom->nVersion < 202000)
        {
            debugprintf(WARN, "version<2.02 from client2 (saw version %d from %s), disconnecting \n", pfrom->nVersion, pfrom->addr.ToString().c_str());
            //disconnect from clients that are too old
            pfrom->fDisconnect = true;
            return false;
        }

        // Disconnect if we connected to ourself
        if (nNonce == nLocalHostNonce && nNonce > 1)
        {
            debugprintf(WARN, "connected to self at %s, disconnecting\n", pfrom->addr.ToString().c_str());
            pfrom->fDisconnect = true;
            return true;
        }

        // Be shy and don't send version until we hear
        if (pfrom->fInbound)
            pfrom->PushVersion();

        pfrom->fClient = !(pfrom->nServices & NODE_NETWORK);

        AddTimeData(pfrom->addr.ip, nTime);

        // Change version
        pfrom->PushMessage("verack");
        pfrom->vSend.SetVersion(min(pfrom->nVersion, VERSION));
        if (!pfrom->fInbound)
        {
            // Advertise our address
            if (addrLocalHost.IsRoutable() && !g_bUseProxy)
            {
                CAddress addr(addrLocalHost);
                addr.nTime = SolidTime_Get();
                pfrom->PushAddress(addr);
            }
            pfrom->PushMessage("getaddr");  // Get recent addresses
            pfrom->fGetAddr = true;
        }

        // Ask the first 2 connected nodes for block updates
        static int nAskedForBlocks=0;
        if (!pfrom->fClient && (nAskedForBlocks < 2 || vNodes.size() <= 2))
        {
            nAskedForBlocks++;
            pfrom->PushGetBlocks(g_pBlockBestIndex, uint256(0));
        }

        // Relay alerts
        {
            MUTEX_LOCK(cs_mapAlerts);
            BOOST_FOREACH(PAIRTYPE(const uint256, CAlert)& item, mapAlerts)
            {
                item.second.RelayTo(pfrom);
            }
        }
        pfrom->fSuccessfullyConnected = true;
        debugprintf(INFO, "version message: version %d, blocks=%"PRI64d"\n", pfrom->nVersion, pfrom->qStartingHeight);
    }
    else if (pfrom->nVersion == 0)
    {
        // Must have a version message before anything else
        return false;
    }


    else if (strCommand == "verack")
    {
        pfrom->vRecv.SetVersion(min(pfrom->nVersion, VERSION));
    }
    else if (strCommand == "addr")
    {
        vector<CAddress> vAddr;
        vRecv >> vAddr;

        if (vAddr.size() > 1000)    return error("message addr size() = %d", vAddr.size());

        // Store the new addresses
        CAddrDB addrDB;
        addrDB.TxnBegin();
        int64 nNow = SolidTime_Get();
        int64 nSince = nNow - 10 * 60;
        BOOST_FOREACH(CAddress& addr, vAddr)
        {
            if (fShutdown)  return true;
            // ignore IPv6 for now, since it isn't implemented anyway
            if (!addr.IsIPv4()) continue;
            if (addr.nTime <= 100000000 || addr.nTime > nNow + 10 * 60) addr.nTime = nNow - 5 * 24 * 60 * 60;
            AddAddress(addr, 2 * 60 * 60, &addrDB);
            pfrom->AddAddressKnown(addr);
            if (addr.nTime > nSince && !pfrom->fGetAddr && vAddr.size() <= 10 && addr.IsRoutable())
            {
                // Relay to a limited number of other nodes
                MUTEX_LOCK(cs_vNodes);

                // Use deterministic randomness to send to the same nodes for 24 hours at a time so the setAddrKnowns of the chosen nodes prevent repeats
                static uint256 hashSalt;
                if (hashSalt == 0)  RAND_bytes((unsigned char*)&hashSalt, sizeof(hashSalt));
                uint256 hashRand = hashSalt ^ (((int64)addr.ip)<<32) ^ ((SolidTime_Get()+addr.ip)/(24*60*60));
                hashRand = Hash(BEGIN(hashRand), END(hashRand));
                multimap<uint256, CNode*> mapMix;
                BOOST_FOREACH(CNode* pnode, vNodes)
                {
                    unsigned int nPointer;
                    memcpy(&nPointer, &pnode, sizeof(nPointer));
                    uint256 hashKey = hashRand ^ nPointer;
                    hashKey = Hash(BEGIN(hashKey), END(hashKey));
                    mapMix.insert(make_pair(hashKey, pnode));
                }
                int nRelayNodes = 2;
                for (multimap<uint256, CNode*>::iterator mi = mapMix.begin(); mi != mapMix.end() && nRelayNodes-- > 0; ++mi)
                    ((*mi).second)->PushAddress(addr);
            }
        }
        addrDB.TxnCommit();  // Save addresses (it's ok if this fails)
        if (vAddr.size() < 1000)
            pfrom->fGetAddr = false;
    }


    else if (strCommand == "inv")
    {
        vector<CInv> vInv;
        vRecv >> vInv;
        if (vInv.size() > 50000)
            return error("message inv size() = %d", vInv.size());

        CTxDB txdb("r");
        BOOST_FOREACH(const CInv& inv, vInv)
        {
            if (fShutdown)
                return true;
            pfrom->AddInventoryKnown(inv);

            bool fAlreadyHave = AlreadyHave(txdb, inv);
            debugprintf(INFO, "  got inventory: %s  %s\n", inv.ToString().c_str(), fAlreadyHave ? "have" : "new");

            if (!fAlreadyHave)
                pfrom->AskFor(inv);
            else if (inv.type == MSG_BLOCK && g_BlockOrphanMap.count(inv.hash))
                pfrom->PushGetBlocks(g_pBlockBestIndex, Block_GetOrphanRoot(g_BlockOrphanMap[inv.hash]));

            // Track requests for our stuff
            {
                LOCK_WALLET_ACCESS();
                Wallet_Inventory(inv.hash);
            }
        }
    }


    else if (strCommand == "getdata")
    {
        vector<CInv> vInv;
        vRecv >> vInv;
        if (vInv.size() > 1000)
            return error("message getdata size() = %d", vInv.size());

        BOOST_FOREACH(const CInv& inv, vInv)
        {
            if (fShutdown)
                return true;
            debugprintf(INFO, "received getdata for [%s]: %s\n", pfrom->addr.ToString().c_str(), inv.ToString().c_str());

            if (inv.type == MSG_BLOCK)
            {
                // Send block from disk
                map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(inv.hash);
                if (mi != g_BlockIndexMap.end())
                {
                    CBlock block;
                    block.ReadFromDisk((*mi).second);
                    pfrom->PushMessage("block", block);

                    // Trigger them to send a getblocks request for the next batch of inventory
                    if (inv.hash == pfrom->hashContinue)
                    {
                        // Bypass PushInventory, this must send even if redundant,
                        // and we want it right after the last block so they don't
                        // wait for other stuff first.
                        vector<CInv> vInv;
                        vInv.push_back(CInv(MSG_BLOCK, g_BlockBestChainHash));
                        pfrom->PushMessage("inv", vInv);
                        pfrom->hashContinue = 0;
                    }
                }
            }
            else if (inv.IsKnownType())
            {
                // Send stream from relay memory
                MUTEX_LOCK(cs_mapRelay);
                map<CInv, CDataStream>::iterator mi = mapRelay.find(inv);
                if (mi != mapRelay.end())   pfrom->PushMessage(inv.GetCommand(), (*mi).second);
            }

            // Track requests for our stuff
            {
                LOCK_WALLET_ACCESS();
                Wallet_Inventory(inv.hash);
            }
        }
    }


    else if (strCommand == "getblocks")
    {
        CBlockLocator locator;
        uint256 hashStop;
        vRecv >> locator >> hashStop;

        // Find the last block the caller has in the main chain
        CBlockIndex* pindex = locator.GetBlockIndex();

        // Send the rest of the chain
        if (pindex)
            pindex = pindex->pnext;
        //int nLimit = 500 + locator.GetDistanceBack();
        unsigned int nBytes = 0;
        int nLimit=0;
        //printf("getblocks %d to %s limit %d bytes\n", (pindex ? pindex->nHeight : -1), hashStop.ToString().substr(0,20).c_str(), (SendBufferSize()/3));
        for (; pindex; pindex = pindex->pnext)
        {
            if (pindex->GetBlockHash() == hashStop)
            {
                debugprintf(INFO, "  getblocks stopping at %"PRI64d" %s (%u bytes)\n", pindex->blk.nBlockNum, pindex->GetBlockHash().ToString().substr(0,20).c_str(), nBytes);
                break;
            }
            pfrom->PushInventory(CInv(MSG_BLOCK, pindex->GetBlockHash()));
            CBlock block;
            block.ReadFromDisk(pindex, true);
            nBytes += block.GetSerializeSize(SER_NETWORK);
            nLimit++;
            if(nLimit>=500 || nBytes >= SendBufferSize()/5)
            {
                // When this block is requested, we'll send an inv that'll make them getblocks the next batch of inventory.
                debugprintf(WARN, "  getblocks stopping at limit %"PRI64d" %s (%u bytes) for %s (v%d)\n", pindex->blk.nBlockNum, pindex->GetBlockHash().ToString().substr(0,20).c_str(), nBytes, pfrom->addr.ToString().c_str(), pfrom->nVersion);
                pfrom->hashContinue = pindex->GetBlockHash();
                break;
            }
        }
    }
    else if (strCommand == "getheaders")
    {
        CBlockLocator locator;
        uint256 hashStop;
        vRecv >> locator >> hashStop;

        CBlockIndex* pindex = NULL;
        if (locator.IsNull())
        {
            // If locator is null, return the hashStop block
            map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(hashStop);
            if (mi == g_BlockIndexMap.end())  return true;
            pindex = (*mi).second;
        }
        else
        {
            // Find the last block the caller has in the main chain
            pindex = locator.GetBlockIndex();
            if (pindex)
                pindex = pindex->pnext;
        }

        vector<CBlock> vHeaders;
        int nLimit = 2000 + locator.GetDistanceBack();
        debugprintf(INFO, "getheaders %"PRI64d" to %s limit %d\n", (pindex ? pindex->blk.nBlockNum : (int64)-1), hashStop.ToString().substr(0,20).c_str(), nLimit);
        for (; pindex; pindex = pindex->pnext)
        {
            vHeaders.push_back(pindex->GetBlockHeader());
            if (--nLimit <= 0 || pindex->GetBlockHash() == hashStop)
                break;
        }
        pfrom->PushMessage("headers", vHeaders);
    }


    else if (strCommand == "tx")
    {
        vector<uint256> vWorkQueue;
        CDataStream vMsg(vRecv);
        CTransaction tx;
        vRecv >> tx;

        CInv inv(MSG_TX, tx.GetHash());
        pfrom->AddInventoryKnown(inv);

        bool fMissingInputs = false;
        if (tx.AcceptToMemoryPool(true, &fMissingInputs))
        {
            {
                LOCK_WALLET_ACCESS();
                Wallet_Sync(tx, NULL, true);
            }
            RelayMessage(inv, vMsg);
            mapAlreadyAskedFor.erase(inv);
            vWorkQueue.push_back(inv.hash);

            // Recursively process any orphan transactions that depended on this one
            for (int i = 0; i < vWorkQueue.size(); i++)
            {
                uint256 hashPrev = vWorkQueue[i];
                for (multimap<uint256, CDataStream*>::iterator mi = mapOrphanTransactionsByPrev.lower_bound(hashPrev);
                     mi != mapOrphanTransactionsByPrev.upper_bound(hashPrev);
                     ++mi)
                {
                    const CDataStream& vMsg = *((*mi).second);
                    CTransaction tx;
                    CDataStream(vMsg) >> tx;
                    CInv inv(MSG_TX, tx.GetHash());

                    if (tx.AcceptToMemoryPool(true))
                    {
                        debugprintf(INFO, "   accepted orphan tx %s\n", inv.hash.ToString().substr(0,10).c_str());
                        {
                            LOCK_WALLET_ACCESS();
                            Wallet_Sync(tx, NULL, true);
                        }
                        RelayMessage(inv, vMsg);
                        mapAlreadyAskedFor.erase(inv);
                        vWorkQueue.push_back(inv.hash);
                    }
                }
            }

            BOOST_FOREACH(uint256 hash, vWorkQueue)
                EraseOrphanTx(hash);
        }
        else if (fMissingInputs)
        {
            //pfrom->nShitSent++;
            //printf("storing orphan tx %s\n", inv.hash.ToString().substr(0,10).c_str());
            AddOrphanTx(vMsg);

        }
        else
        {
            debugprintf(WARN, "Received junk/bad tx from %s\n",pfrom->addr.ToString().c_str());
            pfrom->nShitSent+=5;
        }
    }


    else if (strCommand == "block")
    {
        CBlock block;
        vRecv >> block;
        debugprintf(INFO, "received block %"PRI64d"\n", block.blk.nBlockNum);
        CInv inv(MSG_BLOCK, block.GetHash());
        pfrom->AddInventoryKnown(inv);
        if (Block_Process(pfrom, &block))    mapAlreadyAskedFor.erase(inv);
    }


    else if (strCommand == "getaddr")
    {
        // Nodes rebroadcast an addr every 24 hours
        pfrom->vAddrToSend.clear();
        int64 nSince = SolidTime_Get() - 3 * 60 * 60; // in the last 3 hours

        MUTEX_LOCK(cs_mapAddresses);
        unsigned int nCount = 0;
        BOOST_FOREACH(const PAIRTYPE(vector<unsigned char>, CAddress)& item, mapAddresses)
        {
            const CAddress& addr = item.second;
			if (addr.nTime > nSince && !IsHiddenAddress(addr.ToStringIP()))
				nCount++;
		}
        BOOST_FOREACH(const PAIRTYPE(vector<unsigned char>, CAddress)& item, mapAddresses)
        {
            const CAddress& addr = item.second;
            if (addr.nTime > nSince && GetRand(nCount) < 2500 && !IsHiddenAddress(addr.ToStringIP()))
			{
				debugprintf(VERBOSE, "  getaddr sending node IP %s (ntime: %d) to peer at %s\n", addr.ToStringIP().c_str(), addr.nTime, pfrom->addr.ToString().c_str());
				pfrom->PushAddress(addr);
			}
        }
    }
    else if (strCommand == "checkorder")
    {
        uint256 hashReply;
        vRecv >> hashReply;

        if (!Setting_GetBOOL("allowreceivebyip"))
        {
            pfrom->PushMessage("reply", hashReply, (int)2, string(""));
            return true;
        }

        CWalletTx order;
        vRecv >> order;

        /// we have a chance to check the order here

        // Keep giving the same key to the same ip until they use it
        if (!mapReuseKey.count(pfrom->addr.ip))
        {
            LOCK_WALLET_ACCESS();
            mapReuseKey[pfrom->addr.ip] = Wallet_GetSelected()->GetOrReuseKeyFromPool();
        }

        // Send back approval of order and pubkey to use
        CScript scriptPubKey;
        scriptPubKey << mapReuseKey[pfrom->addr.ip] << OP_CHECKSIG;
        pfrom->PushMessage("reply", hashReply, (int)0, scriptPubKey);
    }


    else if (strCommand == "reply")
    {
        uint256 hashReply;
        vRecv >> hashReply;

        CRequestTracker tracker;

        {
            MUTEX_LOCK(pfrom->cs_mapRequests);
            map<uint256, CRequestTracker>::iterator mi = pfrom->mapRequests.find(hashReply);
            if (mi != pfrom->mapRequests.end())
            {
                tracker = (*mi).second;
                pfrom->mapRequests.erase(mi);
            }
        }
        if (!tracker.IsNull())  tracker.fn(tracker.param1, vRecv);
    }


    else if (strCommand == "ping")
    {
    }


    else if (strCommand == "alert")
    {
        CAlert alert;
        vRecv >> alert;

        if (alert.ProcessAlert())
        {
            pfrom->setKnown.insert(alert.GetHash());    // Relay
            MUTEX_LOCK(cs_vNodes);
            BOOST_FOREACH(CNode* pnode, vNodes)
            {
                alert.RelayTo(pnode);
            }
        }
    }


    else
    {
        // Ignore unknown commands for extensibility
    }


    // Update the last seen time for this node's address
    if (pfrom->fNetworkNode)
        if (strCommand == "version" || strCommand == "addr" || strCommand == "inv" || strCommand == "getdata" || strCommand == "ping")
            AddressCurrentlyConnected(pfrom->addr);


    return true;
}

bool ProcessMessages(CNode* pfrom)
{
    CDataStream& vRecv = pfrom->vRecv;
    if (vRecv.empty())
        return true;
    //if (fDebug)
    //    debugprintf(VERBOSE, "ProcessMessages(%u bytes)\n", vRecv.size());

    //
    // Message format
    //  (4) message start
    //  (12) command
    //  (4) size
    //  (4) checksum
    //  (x) data
    //

    loop
    {
        // Scan for message start
        CDataStream::iterator pstart = search(vRecv.begin(), vRecv.end(), BEGIN(pchMessageStart), END(pchMessageStart));
        int nHeaderSize = vRecv.GetSerializeSize(CMessageHeader());
        if (vRecv.end() - pstart < nHeaderSize)
        {
            if (vRecv.size() > nHeaderSize)
            {
                debugprintf(WARN, "\n\nPROCESSMESSAGE MESSAGESTART NOT FOUND\n\n");
                vRecv.erase(vRecv.begin(), vRecv.end() - nHeaderSize);
            }
            break;
        }
        if (pstart - vRecv.begin() > 0)
            debugprintf(WARN, "\n\nPROCESSMESSAGE SKIPPED %d BYTES\n\n", pstart - vRecv.begin());
        vRecv.erase(vRecv.begin(), pstart);

        // Read header
        vector<char> vHeaderSave(vRecv.begin(), vRecv.begin() + nHeaderSize);
        CMessageHeader hdr;
        vRecv >> hdr;
        if (!hdr.IsValid())
        {
            debugprintf(WARN, "\n\nPROCESSMESSAGE: ERRORS IN HEADER %s\n\n\n", hdr.GetCommand().c_str());
            continue;
        }
        string strCommand = hdr.GetCommand();

        // Message size
        unsigned int nMessageSize = hdr.nMessageSize;
        if (nMessageSize > MAX_SIZE)
        {
            debugprintf(WARN, "ProcessMessage(%s, %u bytes) : nMessageSize > MAX_SIZE\n", strCommand.c_str(), nMessageSize);
            continue;
        }
        if (nMessageSize > vRecv.size())
        {
            // Rewind and wait for rest of message
            vRecv.insert(vRecv.begin(), vHeaderSave.begin(), vHeaderSave.end());
            break;
        }

        // Checksum
        if (vRecv.GetVersion() >= 209)
        {
            uint256 hash = Hash(vRecv.begin(), vRecv.begin() + nMessageSize);
            unsigned int nChecksum = 0;
            memcpy(&nChecksum, &hash, sizeof(nChecksum));
            if (nChecksum != hdr.nChecksum)
            {
                debugprintf(WARN, "ProcessMessage(%s, %u bytes) : CHECKSUM ERROR nChecksum=%08x hdr.nChecksum=%08x\n",
                       strCommand.c_str(), nMessageSize, nChecksum, hdr.nChecksum);
                continue;
            }
        }

        // Copy message to its own buffer
        CDataStream vMsg(vRecv.begin(), vRecv.begin() + nMessageSize, vRecv.nType, vRecv.nVersion);
        vRecv.ignore(nMessageSize);

        // Process message
        bool fRet = false;
        try
        {
            {
                MUTEX_LOCK(cs_main);
                fRet = ProcessMessage(pfrom, strCommand, vMsg);
            }
            if (fShutdown)  return true;
        }
        catch (std::ios_base::failure& e)
        {
            if (strstr(e.what(), "end of data"))
            {
                // Allow exceptions from underlength message on vRecv
                debugprintf(WARN, "ProcessMessage(%s, %u bytes) : Exception '%s' caught, normally caused by a message being shorter than its stated length\n", strCommand.c_str(), nMessageSize, e.what());
            }
            else if (strstr(e.what(), "size too large"))
            {
                // Allow exceptions from overlong size
                debugprintf(WARN, "ProcessMessage(%s, %u bytes) : Exception '%s' caught\n", strCommand.c_str(), nMessageSize, e.what());
            }
            else
            {
                PrintExceptionContinue(&e, "ProcessMessage()");
            }
        }
        catch (std::exception& e) {
            PrintExceptionContinue(&e, "ProcessMessage()");
        } catch (...) {
            PrintExceptionContinue(NULL, "ProcessMessage()");
        }

        if (!fRet)
            debugprintf(WARN, "ProcessMessage(%s, %u bytes) FAILED\n", strCommand.c_str(), nMessageSize);
    }

    vRecv.Compact();
    return true;
}


bool SendMessages(CNode* pto, bool fSendTrickle)
{
    MUTEX_LOCK(cs_main);

    // Don't send anything until we get their version message
    if (pto->nVersion == 0)
        return true;

    // Keep-alive ping
    if (pto->nLastSend && SolidTime_Get() - pto->nLastSend > 30 * 60 && pto->vSend.empty())
        pto->PushMessage("ping");

    // Resend wallet transactions that haven't gotten in a block yet
    {
        LOCK_WALLET_ACCESS();
        Wallet_ResendWalletTransactions();
    }

    // Address refresh broadcast
    static int64 nLastRebroadcast;
    if (SolidTime_Get() - nLastRebroadcast > 24 * 60 * 60)
    {
        nLastRebroadcast = SolidTime_Get();
        MUTEX_LOCK(cs_vNodes);
        BOOST_FOREACH(CNode* pnode, vNodes)
        {
            // Periodically clear setAddrKnown to allow refresh broadcasts
            pnode->setAddrKnown.clear();

            // Rebroadcast our address
            if (addrLocalHost.IsRoutable() && !g_bUseProxy)
            {
                CAddress addr(addrLocalHost);
                addr.nTime = SolidTime_Get();
                pnode->PushAddress(addr);
            }
        }
    }

    // Clear out old addresses periodically so it's not too much work at once
    static int64 nLastClear;
    if (nLastClear == 0)
        nLastClear = SolidTime_Get();
    if (SolidTime_Get() - nLastClear > 10 * 60 && vNodes.size() >= 3)
    {
        nLastClear = SolidTime_Get();

        MUTEX_LOCK(cs_mapAddresses);
        CAddrDB addrdb;
        int64 nSince = SolidTime_Get() - 14 * 24 * 60 * 60;
        for (map<vector<unsigned char>, CAddress>::iterator mi = mapAddresses.begin();mi != mapAddresses.end();)
        {
            const CAddress& addr = (*mi).second;
            if (addr.nTime < nSince)
            {
                if (mapAddresses.size() < 1000 || SolidTime_Get() > nLastClear + 20)
                    break;
                addrdb.EraseAddress(addr);
                mapAddresses.erase(mi++);
            }
            else    mi++;
        }
    }


    //
    // Message: addr
    //
    if (fSendTrickle)
    {
        vector<CAddress> vAddr;
        vAddr.reserve(pto->vAddrToSend.size());
        BOOST_FOREACH(const CAddress& addr, pto->vAddrToSend)
        {
            // returns true if wasn't already contained in the set
            if (pto->setAddrKnown.insert(addr).second)
            {
                vAddr.push_back(addr);
                // receiver rejects addr messages larger than 1000
                if (vAddr.size() >= 1000)
                {
                    pto->PushMessage("addr", vAddr);
                    vAddr.clear();
                }
            }
        }
        pto->vAddrToSend.clear();
        if (!vAddr.empty())
            pto->PushMessage("addr", vAddr);
    }


    //
    // Message: inventory
    //
    vector<CInv> vInv;
    vector<CInv> vInvWait;

    {
        MUTEX_LOCK(pto->cs_inventory);
        vInv.reserve(pto->vInventoryToSend.size());
        vInvWait.reserve(pto->vInventoryToSend.size());
        BOOST_FOREACH(const CInv& inv, pto->vInventoryToSend)
        {
            if (pto->setInventoryKnown.count(inv))
                continue;

            // trickle out tx inv to protect privacy
            if (inv.type == MSG_TX && !fSendTrickle)
            {
                // 1/4 of tx invs blast to all immediately
                static uint256 hashSalt;
                if (hashSalt == 0)
                    RAND_bytes((unsigned char*)&hashSalt, sizeof(hashSalt));
                uint256 hashRand = inv.hash ^ hashSalt;
                hashRand = Hash(BEGIN(hashRand), END(hashRand));
                bool fTrickleWait = ((hashRand & 3) != 0);

                // always trickle our own transactions
                if (!fTrickleWait)
                {
                    CWalletTx wtx;
                    LOCK_WALLET_ACCESS();
                    if (Wallet_GetTransaction(inv.hash, wtx) && wtx.fFromMe)    fTrickleWait = true;
                }

                if (fTrickleWait)
                {
                    vInvWait.push_back(inv);
                    continue;
                }
            }

            // returns true if wasn't already contained in the set
            if (pto->setInventoryKnown.insert(inv).second)
            {
                vInv.push_back(inv);
                if (vInv.size() >= 1000)
                {
                    pto->PushMessage("inv", vInv);
                    vInv.clear();
                }
            }
        }
        pto->vInventoryToSend = vInvWait;
    }
    if (!vInv.empty())
        pto->PushMessage("inv", vInv);


    //
    // Message: getdata
    //
    vector<CInv> vGetData;
    int64 nNow = SolidTime_Get() * 1000000;
    CTxDB txdb("r");
    while (!pto->mapAskFor.empty() && (*pto->mapAskFor.begin()).first <= nNow)
    {
        const CInv& inv = (*pto->mapAskFor.begin()).second;
        if (!AlreadyHave(txdb, inv))
        {
            debugprintf(INFO, "sending getdata: %s\n", inv.ToString().c_str());
            vGetData.push_back(inv);
            if (vGetData.size() >= 100)
            {
                pto->PushMessage("getdata", vGetData);
                vGetData.clear();
            }
        }
        mapAlreadyAskedFor[inv] = nNow;
        pto->mapAskFor.erase(pto->mapAskFor.begin());
    }
    if (!vGetData.empty())  pto->PushMessage("getdata", vGetData);

    return true;
}














//////////////////////////////////////////////////////////////////////////////
//
// SolidCoinMiner
//

int static FormatHashBlocks(void* pbuffer, unsigned int len)
{
    unsigned char* pdata = (unsigned char*)pbuffer;
    unsigned int blocks = 1 + ((len + 8) / 64);
    unsigned char* pend = pdata + 64 * blocks;
    memset(pdata + len, 0, 64 * blocks - len);
    pdata[len] = 0x80;
    unsigned int bits = len * 8;
    pend[-1] = (bits >> 0) & 0xff;
    pend[-2] = (bits >> 8) & 0xff;
    pend[-3] = (bits >> 16) & 0xff;
    pend[-4] = (bits >> 24) & 0xff;
    return blocks;
}

using CryptoPP::ByteReverse;

static const unsigned int pSHA256InitState[8] =
{0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a, 0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19};

inline void SHA256Transform(void* pstate, void* pinput, const void* pinit)
{
    memcpy(pstate, pinit, 32);
    CryptoPP::SHA256::Transform((CryptoPP::word32*)pstate, (CryptoPP::word32*)pinput);
}

//
// ScanHash scans nonces looking for a hash with at least some zero bits.
// It operates on big endian data.  Caller does the byte reversing.
// All input buffers are 16-byte aligned.  nNonce is usually preserved
// between calls, but periodically or if nNonce is 0xffff0000 or above,
// the block is rebuilt and nNonce starts over at zero.
//
unsigned int static ScanHash_CryptoPP(char* pmidstate, char* pdata, char* phash1, char* phash, unsigned int& nHashesDone)
{
    unsigned int& nNonce = *(unsigned int*)(pdata + 12);
    for (;;)
    {
        // Crypto++ SHA-256
        // Hash pdata using pmidstate as the starting state into
        // preformatted buffer phash1, then hash phash1 into phash
        nNonce++;
        SHA256Transform(phash1, pdata, pmidstate);
        SHA256Transform(phash, phash1, pSHA256InitState);

        // Return the nonce if the hash has at least some zero bits,
        // caller will check if it has enough to reach the target
        if (((unsigned short*)phash)[14] == 0)
            return nNonce;

        // If nothing found after trying for a while, return -1
        if ((nNonce & 0xffff) == 0)
        {
            nHashesDone = 0xffff+1;
            return -1;
        }
    }
}



